/* JLdL 17Feb08.
 *
 * Copyright (C) 2007-2008 by Jorge L. deLyra.
 * Mail: <delyra@latt.if.usp.br>. Web: "http://latt.if.usp.br".
 * This program may be copied and/or distributed freely. See the
 * _ terms and conditions in the files in the doc/ subdirectory.
 *
 * This is a C interface to be called from Fortran programs.
 *
 * This interface allows the use from Fortran programs of the g2
 * _ function g2_filled_spline(), which draws a filled spline
 * _ curve through NP given points, with NI interpolated points
 * _ per data point; the larger NI is, the more fluent the curve
 * _ will be; this version uses Young's method of successive
 * _ over-relaxation.
 *
 * Note that the array of positions to be passed as an argument
 * _ should be defined within the calling Fortran program as a
 * _ two-index array, such as
 * _
 * _    dimension pnt_pos(2,NP)
 * _
 * _ in the order shown, where the first index corresponds to
 * _ the x and y coordinates and the second index corresponds
 * _ to the NP points.
 *
 * This is to be called from Fortran as a subroutine, that is,
 * _ as "call g2_filled_spline_ctof(<args>)".
 *
 * Include the g2 headers.
 */
#include <g2.h>
/*
 * Include the declaration of the calloc() function.
 */
#include <stdlib.h>
/*
 * Include the local g2-ctof headers.
 */
#include "g2-ctof.h"
/*
 * Define the interface; the arguments are:
 * _ id_ad:   the id of the device;
 * _ n_pnt:   the number of points;
 * _ pnt_pos: pointer to an array of 2*NP float numbers,
 * _          containing the coordinates of the points
 * _          in the order x1, y1, x2, y2, ...;
 * _ n_int:   the number of interpolation points per data point.
 */
void g2_filled_spline_ctof__(int *id_ad,
			     int *n_pnt,
			     float *pnt_pos,
			     int *n_int)
{
  /*
   * Local variables:
   * _ i_id_ad, i_n_pnt: int variables to acquire the arguments;
   * _ ic: an int counter for the "for" loop;
   * _ d_pnt_pos: a pointer to an array of 2*NP double numbers; since
   * _            we do not want to dimension it statically here,
   * _            we declare it simply as a pointer to double;
   * _ i_n_int: int variable to acquire the argument.
   */
  int    i_id_ad, i_n_pnt, i_n_int, ic;
  double *d_pnt_pos;
  /*
   * Pass the data from the pointer arguments to the local variables.
   */
  i_id_ad = *id_ad;
  i_n_pnt = *n_pnt;
  i_n_int = *n_int;
  /*
   * Dimensioning the array dynamically: now that we have the number
   * _ NP of points available, we call the calloc() function in order
   * _ to dimension the d_pnt_pos array, as an array of 2*NP double
   * _ numbers; note how calloc() returns the pointer to the array.
   */
  d_pnt_pos = calloc(2*i_n_pnt, 8);
  /*
   * Pass the data from the array argument
   * _ to the local double array.
   */
  for (ic = 0; ic < 2*i_n_pnt; ic++)
    {
      d_pnt_pos[ic] = pnt_pos[ic];
    }
  /*
   * Call the C function with the required arguments.
   */
  g2_filled_spline(i_id_ad,
		   i_n_pnt,
		   d_pnt_pos,
		   i_n_int);
}
/*
 * Compatibility function for gfortran.
 */
void g2_filled_spline_ctof_(int *id_ad,
			    int *n_pnt,
			    float *pnt_pos,
			    int *n_int)
{
  g2_filled_spline_ctof__(id_ad,
			  n_pnt,
			  pnt_pos,
			  n_int);
}
