/*!
 * \file
 * \brief Vector class test program
 * \author Tony Ottosson and Adam Piatyszek
 *
 * -------------------------------------------------------------------------
 *
 * Copyright (C) 1995-2010  (see AUTHORS file for a list of contributors)
 *
 * This file is part of IT++ - a C++ library of mathematical, signal
 * processing, speech processing, and communications classes and functions.
 *
 * IT++ is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * IT++ is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with IT++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * -------------------------------------------------------------------------
 */

#include <itpp/itbase.h>
#include <typeinfo>
#include "gtest/gtest.h"

using namespace itpp;
using namespace std;

static
const double tol = 1e-4;

static
void assert_vec_p(const vec &ref, const vec &act, int line)
{
  ASSERT_EQ(ref.length(), act.length()) << line;
  for(int n = 0; n < ref.length(); ++n) {
    ASSERT_NEAR(ref(n), act(n), tol) << line;
  }
}
#define assert_vec(ref, act) assert_vec_p(ref, act, __LINE__)

static
void assert_cvec_p(const cvec &ref, const cvec &act, int line)
{
  ASSERT_EQ(ref.length(), act.length()) << line;
  for(int n = 0; n < ref.length(); ++n) {
    ASSERT_NEAR(ref(n).real(), act(n).real(), tol) << line;
    ASSERT_NEAR(ref(n).imag(), act(n).imag(), tol) << line;
  }
}
#define assert_cvec(ref, act) assert_cvec_p(ref, act, __LINE__)

static
void assert_mat_p(const mat &ref, const mat &act, int line)
{
  ASSERT_EQ(ref.rows(), act.rows()) << line;
  ASSERT_EQ(ref.cols(), act.cols()) << line;
  for(int n = 0; n < ref.rows(); ++n) {
    for(int k = 0; k < ref.cols(); ++k) {
      ASSERT_NEAR(ref(n, k), act(n, k), tol) << line;
    }
  }
}
#define assert_mat(ref, act) assert_mat_p(ref, act, __LINE__)

static
void assert_cmat_p(const cmat &ref, const cmat &act, int line)
{
  ASSERT_EQ(ref.rows(), act.rows()) << line;
  ASSERT_EQ(ref.cols(), act.cols()) << line;
  for(int n = 0; n < ref.rows(); ++n) {
    for(int k = 0; k < ref.cols(); ++k) {
      ASSERT_NEAR(ref(n, k).real(), act(n, k).real(), tol) << line;
      ASSERT_NEAR(ref(n, k).imag(), act(n, k).imag(), tol) << line;
    }
  }
}
#define assert_cmat(ref, act) assert_cmat_p(ref, act, __LINE__)

void common_operators(const bvec &a, const bvec &b, bin c)
{
  bvec a2 = a;
  bvec x1;
  bvec b2 = b;
  bvec x2;
  bvec ref = "1 1 1 0 1 0 1 1 1 1";
  ASSERT_TRUE(ref == a + b);
  ASSERT_TRUE((a + c) == (c + a));
  ref = "1 1 1 0 1 0 1 1 1 1";
  a2 += b;
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2 += c;
  ref = "1 1 1 1 0 1 1 0 0 1";
  ASSERT_TRUE(ref == a2);
  ref = "1 1 1 0 1 0 1 1 1 1";
  ASSERT_TRUE(ref == a - b);
  ref = "1 1 1 1 0 1 1 0 0 1";
  ASSERT_TRUE(ref == a - c);
  ASSERT_TRUE(ref == c - a);
  a2 = a;
  a2 -= b;
  ref = "1 1 1 0 1 0 1 1 1 1";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2 -= c;
  ref = "1 1 1 1 0 1 1 0 0 1";
  ASSERT_TRUE(ref == a2);
  ref = "1 1 1 1 0 1 1 0 0 1";
  ASSERT_TRUE(ref == -a);

  ASSERT_TRUE(bin(0) == a * b);
  ASSERT_TRUE(bin(0) == dot(a, b));
  bmat ref_m = "0 0 0 1 1 1 0 1 1 0;"
               "0 0 0 1 1 1 0 1 1 0;"
               "0 0 0 1 1 1 0 1 1 0;"
               "0 0 0 1 1 1 0 1 1 0;"
               "0 0 0 0 0 0 0 0 0 0;"
               "0 0 0 1 1 1 0 1 1 0;"
               "0 0 0 1 1 1 0 1 1 0;"
               "0 0 0 0 0 0 0 0 0 0;"
               "0 0 0 0 0 0 0 0 0 0;"
               "0 0 0 1 1 1 0 1 1 0";
  ASSERT_TRUE(ref_m == outer_product(a, b));
  ASSERT_TRUE(a * c == c * a);
  a2 = a;
  a2 *= c;
  ASSERT_TRUE(a * c == a2);
  ref = "0 0 0 1 0 1 0 0 0 0";
  ASSERT_TRUE(ref == elem_mult(a, b));
  elem_mult_out(a, b, x1);
  ref = "0 0 0 1 0 1 0 0 0 0";
  ASSERT_TRUE(ref == x1);
  elem_mult_inplace(a, b2);
  ref = "0 0 0 1 0 1 0 0 0 0";
  ASSERT_TRUE(ref == b2);
  ASSERT_TRUE(bin(0) == elem_mult_sum(a, b));

  ref = "1 1 1 1 0 1 1 0 0 1";
  ASSERT_TRUE(ref == a / c);
  ASSERT_TRUE(ref == c / a);
  a2 = a;
  a2 /= c;
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2 /= b;
  ref = "1 1 1 1 1 1 1 1 1 1";
  ASSERT_TRUE(ref == a2);
  ASSERT_TRUE(ref == elem_div(a, b));
  elem_div_out(a, b, x2);
  ASSERT_TRUE(ref == x2);
  ASSERT_TRUE(bin(0) == elem_div_sum(a, b));

  ref = "1 1 1 1 0 1 1 0 0 1 0 0 0 1 1 1 0 1 1 0";
  ASSERT_TRUE(ref == concat(a, b));
  ref = "1 1 1 1 0 1 1 0 0 1 0";
  ASSERT_TRUE(ref == concat(a, c));
  ref = "0 1 1 1 1 0 1 1 0 0 1";
  ASSERT_TRUE(ref == concat(c, a));
  ref = "1 1 1 1 0 1 1 0 0 1 0 0 0 1 1 1 0 1 1 0 1 1 1 1 0 1 1 0 0 1";
  ASSERT_TRUE(ref == concat(a, b, a));
  ref = "1 1 1 1 0 1 1 0 0 1 0 0 0 1 1 1 0 1 1 0 1 1 1 1 0 1 1 0 0 1 0 0 0 1 1 1 0 1 1 0";
  ASSERT_TRUE(ref == concat(a, b, a, b));
  ref = "1 1 1 1 0 1 1 0 0 1 0 0 0 1 1 1 0 1 1 0 1 1 1 1 0 1 1 0 0 1 0 0 0 1 1 1 0 1 1 0 1 1 1 1 0 1 1 0 0 1";
  ASSERT_TRUE(ref == concat(a, b, a, b, a));

  ref_m = "1 1 1 1 0 1 1 0 0 1";
  ASSERT_TRUE(ref_m == a.T());
  ref_m = "1 1 1 1 0 1 1 0 0 1";
  ASSERT_TRUE(ref_m == a.H());

  ASSERT_TRUE(10 == a.size());
  a2 = a;
  a2.set_size(a2.size() + 3, true);
  ref = "1 1 1 1 0 1 1 0 0 1 0 0 0";
  ASSERT_TRUE(ref == a2);
  a2.set_size(a2.size() - 6, true);
  ref = "1 1 1 1 0 1 1";
  ASSERT_TRUE(ref == a2);

  ASSERT_TRUE(bin(1) == a(5));
  ASSERT_TRUE(bin(1) == a.get(5));
  ref = "1 1 1 1 0 1";
  ASSERT_TRUE(ref == a(0, 5));
  ASSERT_TRUE(ref == a.get(0, 5));
  ref = "1 0 0 1";
  ASSERT_TRUE(ref == a(6, -1));
  ivec idx_list = "0 5 6 7";
  ref = "1 1 1 0";
  ASSERT_TRUE(ref == a(idx_list));
  ASSERT_TRUE(ref == a.get(idx_list));
  bvec bin_list = "1 0 0 0 0 1 1 1 0 0";
  ref = "1 1 1 0";
  ASSERT_TRUE(ref == a(bin_list));
  ASSERT_TRUE(ref == a.get(bin_list));
  ref = "0 0 1";
  ASSERT_TRUE(ref == a.right(3));
  ref = "1 1 1 1";
  ASSERT_TRUE(ref == a.left(4));
  ref = "1 0";
  ASSERT_TRUE(ref == a.mid(3, 2));
  a2 = a;
  ASSERT_TRUE(0 == a2.split(0).length());
  a2 = a;
  ASSERT_TRUE(a == a2.split(a2.size()));
  ASSERT_TRUE(0 == a2.length());
  a2 = a;
  ref = "1 1 1 1";
  ASSERT_TRUE(ref == a2.split(4));
  ref = "0 1 1 0 0 1";
  ASSERT_TRUE(ref == a2);
  a2(5) = a(6);
  ref = "0 1 1 0 0 1";
  ASSERT_TRUE(ref == a2);

  a2 = a;
  a2.shift_left(c, 2);
  ref = "1 1 0 1 1 0 0 1 0 0";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2.shift_right(c);
  ref = "0 1 1 1 1 0 1 1 0 0";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2.shift_left(b.mid(0, 2));
  ref = "1 1 0 1 1 0 0 1 0 0";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2.shift_right(b.right(5));
  ref = "1 0 1 1 0 1 1 1 1 0";
  ASSERT_TRUE(ref == a2);

  a2 = a;
  a2.set_subvector(0, b);
  ref = "0 0 0 1 1 1 0 1 1 0";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2.set_subvector(4, b(3, 5));
  ref = "1 1 1 1 1 1 1 0 0 1";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2.replace_mid(4, b(3, 5));
  ref = "1 1 1 1 1 1 1 0 0 1";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2.del(6);
  ref = "1 1 1 1 0 1 0 0 1";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2.del(3, 9);
  ref = "1 1 1";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2.ins(0, c);
  ref = "0 1 1 1 1 0 1 1 0 0 1";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2.ins(2, c);
  ref = "1 1 0 1 1 0 1 1 0 0 1";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2.ins(10, c);
  ref = "1 1 1 1 0 1 1 0 0 1 0";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2.ins(3, b(0, 2));
  ref = "1 1 1 0 0 0 1 0 1 1 0 0 1";
  ASSERT_TRUE(ref == a2);

  a2 = a;
  a2.zeros();
  ASSERT_FALSE(any(a2));
  a2 = a;
  a2.ones();
  ASSERT_TRUE(all(a2));
  a2 = a;
  a2 = c;
  ASSERT_FALSE(any(a2));
  a2 = a;
  a2 = b(0, 4);
  ref = "0 0 0 1 1";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2 = b.T();
  ref = "0 0 0 1 1 1 0 1 1 0";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2 = b.T().T();
  ref = "0 0 0 1 1 1 0 1 1 0";
  ASSERT_TRUE(ref == a2);
}

void common_operators(const ivec &a, const ivec &b, int c)
{
  ivec a2 = a;
  ivec x1;
  ivec b2 = b;
  ivec x2;
  ivec ref = "15 7 11 10 15 5 9 13 12 7";
  ivec act = a + b;
  ASSERT_TRUE(ref == act);
  ASSERT_TRUE((a + c) == (c + a));
  ref = "15 7 11 10 15 5 9 13 12 7";
  a2 += b;
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2 += c;
  ref = "14 10 12 13 15 9 7 11 14 11";
  ASSERT_TRUE(ref == a2);
  ref = "1 1 1 4 3 1 -7 -3 4 3";
  ASSERT_TRUE(ref == a - b);
  ref = "2 -2 0 1 3 -3 -5 -1 2 -1";
  ASSERT_TRUE(ref == a - c);
  ref = "-2 2 0 -1 -3 3 5 1 -2 1";
  ASSERT_TRUE(ref == c - a);
  a2 = a;
  a2 -= b;
  ref = "1 1 1 4 3 1 -7 -3 4 3";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2 -= c;
  ref = "2 -2 0 1 3 -3 -5 -1 2 -1";
  ASSERT_TRUE(ref == a2);
  ref = "-8 -4 -6 -7 -9 -3 -1 -5 -8 -5";
  ASSERT_TRUE(ref == -a);

  ASSERT_TRUE(269 == a * b);
  ASSERT_TRUE(269 == dot(a, b));
  imat ref_m = "56 24 40 24 48 16 64 64 32 16;"
               "28 12 20 12 24 8 32 32 16 8;"
               "42 18 30 18 36 12 48 48 24 12;"
               "49 21 35 21 42 14 56 56 28 14;"
               "63 27 45 27 54 18 72 72 36 18;"
               "21 9 15 9 18 6 24 24 12 6;"
               "7 3 5 3 6 2 8 8 4 2;"
               "35 15 25 15 30 10 40 40 20 10;"
               "56 24 40 24 48 16 64 64 32 16;"
               "35 15 25 15 30 10 40 40 20 10";
  ASSERT_TRUE(ref_m == outer_product(a, b));
  ASSERT_TRUE(a * c == c * a);
  a2 = a;
  a2 *= c;
  ASSERT_TRUE(a * c == a2);
  ref = "56 12 30 21 54 6 8 40 32 10";
  ASSERT_TRUE(ref == elem_mult(a, b));
  elem_mult_out(a, b, x1);
  ref = "56 12 30 21 54 6 8 40 32 10";
  ASSERT_TRUE(ref == x1);
  elem_mult_inplace(a, b2);
  ref = "56 12 30 21 54 6 8 40 32 10";
  ASSERT_TRUE(ref == b2);
  ASSERT_TRUE(269 == elem_mult_sum(a, b));

  ref = "1 0 1 1 1 0 0 0 1 0";
  ASSERT_TRUE(ref == a / c);
  ref = "0 1 1 0 0 2 6 1 0 1";
  ASSERT_TRUE(ref == c / a);
  a2 = a;
  a2 /= c;
  ref = "1 0 1 1 1 0 0 0 1 0";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2 /= b;
  ref = "1 1 1 2 1 1 0 0 2 2";
  ASSERT_TRUE(ref == a2);
  ASSERT_TRUE(ref == elem_div(a, b));
  elem_div_out(a, b, x2);
  ASSERT_TRUE(ref == x2);
  ASSERT_TRUE(11 == elem_div_sum(a, b));

  ref = "8 4 6 7 9 3 1 5 8 5 7 3 5 3 6 2 8 8 4 2";
  ASSERT_TRUE(ref == concat(a, b));
  ref = "8 4 6 7 9 3 1 5 8 5 6";
  ASSERT_TRUE(ref == concat(a, c));
  ref = "6 8 4 6 7 9 3 1 5 8 5";
  ASSERT_TRUE(ref == concat(c, a));
  ref = "8 4 6 7 9 3 1 5 8 5 7 3 5 3 6 2 8 8 4 2 8 4 6 7 9 3 1 5 8 5";
  ASSERT_TRUE(ref == concat(a, b, a));
  ref = "8 4 6 7 9 3 1 5 8 5 7 3 5 3 6 2 8 8 4 2 8 4 6 7 9 3 1 5 8 5 7 3 5 3 6 2 8 8 4 2";
  ASSERT_TRUE(ref == concat(a, b, a, b));
  ref = "8 4 6 7 9 3 1 5 8 5 7 3 5 3 6 2 8 8 4 2 8 4 6 7 9 3 1 5 8 5 7 3 5 3 6 2 8 8 4 2 8 4 6 7 9 3 1 5 8 5";
  ASSERT_TRUE(ref == concat(a, b, a, b, a));

  ref_m = "8 4 6 7 9 3 1 5 8 5";
  ASSERT_TRUE(ref_m == a.T());
  ref_m = "8 4 6 7 9 3 1 5 8 5";
  ASSERT_TRUE(ref_m == a.H());

  ASSERT_TRUE(10 == a.size());
  a2 = a;
  a2.set_size(a2.size() + 3, true);
  ref = "8 4 6 7 9 3 1 5 8 5 0 0 0";
  ASSERT_TRUE(ref == a2);
  a2.set_size(a2.size() - 6, true);
  ref = "8 4 6 7 9 3 1";
  ASSERT_TRUE(ref == a2);

  ASSERT_TRUE(3 == a(5));
  ASSERT_TRUE(3 == a.get(5));
  ref = "8 4 6 7 9 3";
  ASSERT_TRUE(ref == a(0, 5));
  ASSERT_TRUE(ref == a.get(0, 5));
  ref = "1 5 8 5";
  ASSERT_TRUE(ref == a(6, -1));
  ivec idx_list = "0 5 6 7";
  ref = "8 3 1 5";
  ASSERT_TRUE(ref == a(idx_list));
  ASSERT_TRUE(ref == a.get(idx_list));
  bvec bin_list = "1 0 0 0 0 1 1 1 0 0";
  ref = "8 3 1 5";
  ASSERT_TRUE(ref == a(bin_list));
  ASSERT_TRUE(ref == a.get(bin_list));
  ref = "5 8 5";
  ASSERT_TRUE(ref == a.right(3));
  ref = "8 4 6 7";
  ASSERT_TRUE(ref == a.left(4));
  ref = "7 9";
  ASSERT_TRUE(ref == a.mid(3, 2));
  a2 = a;
  ASSERT_TRUE(0 == a2.split(0).length());
  a2 = a;
  ASSERT_TRUE(a == a2.split(a2.size()));
  ASSERT_TRUE(0 == a2.length());
  a2 = a;
  ref = "8 4 6 7";
  ASSERT_TRUE(ref == a2.split(4));
  ref = "9 3 1 5 8 5";
  ASSERT_TRUE(ref == a2);
  a2(5) = a(6);
  ref = "9 3 1 5 8 1";
  ASSERT_TRUE(ref == a2);

  a2 = a;
  a2.shift_left(c, 2);
  ref = "6 7 9 3 1 5 8 5 6 6";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2.shift_right(c);
  ref = "6 8 4 6 7 9 3 1 5 8";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2.shift_left(b.mid(0, 2));
  ref = "6 7 9 3 1 5 8 5 7 3";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2.shift_right(b.right(5));
  ref = "2 8 8 4 2 8 4 6 7 9";
  ASSERT_TRUE(ref == a2);

  a2 = a;
  a2.set_subvector(0, b);
  ref = "7 3 5 3 6 2 8 8 4 2";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2.set_subvector(4, b(3, 5));
  ref = "8 4 6 7 3 6 2 5 8 5";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2.replace_mid(4, b(3, 5));
  ref = "8 4 6 7 3 6 2 5 8 5";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2.del(6);
  ref = "8 4 6 7 9 3 5 8 5";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2.del(3, 9);
  ref = "8 4 6";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2.ins(0, c);
  ref = "6 8 4 6 7 9 3 1 5 8 5";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2.ins(2, c);
  ref = "8 4 6 6 7 9 3 1 5 8 5";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2.ins(10, c);
  ref = "8 4 6 7 9 3 1 5 8 5 6";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2.ins(3, b(0, 2));
  ref = "8 4 6 7 3 5 7 9 3 1 5 8 5";
  ASSERT_TRUE(ref == a2);

  a2 = a;
  a2.zeros();
  ref = "0 0 0 0 0 0 0 0 0 0";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2.ones();
  ref = "1 1 1 1 1 1 1 1 1 1";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2 = c;
  ref = "6 6 6 6 6 6 6 6 6 6";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2 = b(0, 4);
  ref = "7 3 5 3 6";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2 = b.T();
  ref = "7 3 5 3 6 2 8 8 4 2";
  ASSERT_TRUE(ref == a2);
  a2 = a;
  a2 = b.T().T();
  ref = "7 3 5 3 6 2 8 8 4 2";
  ASSERT_TRUE(ref == a2);
}

void common_operators(const vec &a, const vec &b, double c)
{
  vec a2 = a;
  vec x1;
  vec b2 = b;
  vec x2;
  vec ref = "0.444997 1.38884 1.04291 0.566956 0.729646 0.74925 0.732926 0.331245 0.85397 1.83142";
  assert_vec(ref, a + b);
  assert_vec((a + c), (c + a));
  a2 += b;
  assert_vec(ref, a2);
  a2 = a;
  a2 += c;
  ref = "1.10195 1.41587 1.20924 1.01898 1.45767 1.44645 1.52906 1.11209 1.22129 1.76434";
  assert_vec(ref, a2);
  ref = "0.0776177 -0.238389 -0.305723 -0.210277 0.504406 0.462364 0.643916 0.211661 -0.0926674 0.0159702";
  assert_vec(ref, a - b);
  ref = "-0.579334 -0.265414 -0.472048 -0.662302 -0.223616 -0.234835 -0.152221 -0.569189 -0.459991 0.0830525";
  assert_vec(ref, a - c);
  ref = -ref;
  assert_vec(ref, c - a);
  a2 = a;
  a2 -= b;
  ref = "0.0776177 -0.238389 -0.305723 -0.210277 0.504406 0.462364 0.643916 0.211661 -0.0926674 0.0159702";
  assert_vec(ref, a2);
  a2 = a;
  a2 -= c;
  ref = "-0.579334 -0.265414 -0.472048 -0.662302 -0.223616 -0.234835 -0.152221 -0.569189 -0.459991 0.0830525";
  assert_vec(ref, a2);
  ref = "-0.261307 -0.575228 -0.368593 -0.17834 -0.617026 -0.605807 -0.688421 -0.271453 -0.380651 -0.923694";
  assert_vec(ref, -a);

  ASSERT_NEAR(2.05575, a * b, tol);
  ASSERT_NEAR(2.05575, dot(a, b), tol);
  mat ref_m = "0.0479994 0.212604 0.176204 0.101548 0.0294284 0.0374828 0.0116294 0.0156241 0.123682 0.237195;"
              "0.105663 0.468015 0.387886 0.223543 0.0647821 0.0825125 0.0256003 0.0343942 0.272266 0.522148;"
              "0.0677068 0.299894 0.248549 0.143241 0.041511 0.0528722 0.0164041 0.022039 0.174462 0.334581;"
              "0.0327592 0.1451 0.120258 0.0693058 0.0200846 0.0255816 0.00793693 0.0106633 0.0844115 0.161883;"
              "0.113341 0.502023 0.416071 0.239787 0.0694895 0.0885082 0.0274605 0.0368934 0.29205 0.56009;"
              "0.11128 0.492895 0.408506 0.235427 0.068226 0.0868989 0.0269612 0.0362225 0.28674 0.549906;"
              "0.126456 0.560111 0.464214 0.267532 0.0775299 0.0987493 0.0306379 0.0411622 0.325842 0.624896;"
              "0.0498631 0.220859 0.183045 0.105491 0.030571 0.0389381 0.0120809 0.0162308 0.128484 0.246404;"
              "0.0699217 0.309704 0.256679 0.147927 0.0428689 0.0546018 0.0169407 0.02276 0.180169 0.345526;"
              "0.169673 0.751533 0.622863 0.358963 0.104026 0.132498 0.0411086 0.0552297 0.437202 0.838459";
  assert_mat(ref_m, outer_product(a, b));
  ASSERT_TRUE(a * c == c * a);
  a2 = a;
  a2 *= c;
  ASSERT_TRUE(a * c == a2);
  ref = "0.0479994 0.468015 0.248549 0.0693058 0.0694895 0.0868989 0.0306379 0.0162308 0.180169 0.838459";
  assert_vec(ref, elem_mult(a, b));
  elem_mult_out(a, b, x1);
  ref = "0.0479994 0.468015 0.248549 0.0693058 0.0694895 0.0868989 0.0306379 0.0162308 0.180169 0.838459";
  assert_vec(ref, x1);
  elem_mult_inplace(a, b2);
  ref = "0.0479994 0.468015 0.248549 0.0693058 0.0694895 0.0868989 0.0306379 0.0162308 0.180169 0.838459";
  assert_vec(ref, b2);
  ASSERT_NEAR(2.05575, elem_mult_sum(a, b), tol);

  ref = "0.310843 0.684272 0.438467 0.212147 0.733994 0.720648 0.818923 0.322911 0.45281 1.0988";
  assert_vec(ref, a / c);
  ref = "3.21706 1.46141 2.28067 4.71371 1.36241 1.38764 1.22112 3.09683 2.20843 0.910087";
  assert_vec(ref, c / a);
  a2 = a;
  a2 /= c;
  ref = "0.310843 0.684272 0.438467 0.212147 0.733994 0.720648 0.818923 0.322911 0.45281 1.0988";
  assert_vec(ref, a2);
  a2 = a;
  a2 /= b;
  ref = "1.42255 0.707001 0.546618 0.45891 5.47884 4.22332 15.4686 4.53993 0.804218 1.01759";
  assert_vec(ref, a2);
  assert_vec(ref, elem_div(a, b));
  ref = "1.42255 0.707001 0.546618 0.45891 5.47884 4.22332 15.4686 4.53993 0.804218 1.01759";
  elem_div_out(a, b, x2);
  assert_vec(ref, x2);
  ASSERT_NEAR(34.6675, elem_div_sum(a, b), tol);

  ref = "0.261307 0.575228 0.368593 0.17834 0.617026 0.605807 0.688421 0.271453 0.380651 0.923694 0.18369 "
        "0.813617 0.674317 0.388616 0.11262 0.143443 0.0445045 0.0597922 0.473318 0.907724";
  assert_vec(ref, concat(a, b));
  ref = "0.261307 0.575228 0.368593 0.17834 0.617026 0.605807 0.688421 0.271453 0.380651 0.923694 0.840642";
  assert_vec(ref, concat(a, c));
  ref = "0.840642 0.261307 0.575228 0.368593 0.17834 0.617026 0.605807 0.688421 0.271453 0.380651 0.923694";
  assert_vec(ref, concat(c, a));
  ref = "0.261307 0.575228 0.368593 0.17834 0.617026 0.605807 0.688421 0.271453 0.380651 0.923694 0.18369 "
        "0.813617 0.674317 0.388616 0.11262 0.143443 0.0445045 0.0597922 0.473318 0.907724 0.261307 0.575228 "
        "0.368593 0.17834 0.617026 0.605807 0.688421 0.271453 0.380651 0.923694";
  assert_vec(ref, concat(a, b, a));
  ref = "0.261307 0.575228 0.368593 0.17834 0.617026 0.605807 0.688421 0.271453 0.380651 0.923694 0.18369 "
        "0.813617 0.674317 0.388616 0.11262 0.143443 0.0445045 0.0597922 0.473318 0.907724 0.261307 0.575228 "
        "0.368593 0.17834 0.617026 0.605807 0.688421 0.271453 0.380651 0.923694 0.18369 0.813617 0.674317 0.388616 "
        "0.11262 0.143443 0.0445045 0.0597922 0.473318 0.907724";
  assert_vec(ref, concat(a, b, a, b));
  ref = "0.261307 0.575228 0.368593 0.17834 0.617026 0.605807 0.688421 0.271453 0.380651 0.923694 0.18369 "
        "0.813617 0.674317 0.388616 0.11262 0.143443 0.0445045 0.0597922 0.473318 0.907724 0.261307 0.575228 "
        "0.368593 0.17834 0.617026 0.605807 0.688421 0.271453 0.380651 0.923694 0.18369 0.813617 0.674317 0.388616 "
        "0.11262 0.143443 0.0445045 0.0597922 0.473318 0.907724 0.261307 0.575228 0.368593 0.17834 0.617026 "
        "0.605807 0.688421 0.271453 0.380651 0.923694";
  assert_vec(ref, concat(a, b, a, b, a));

  ref_m = "0.261307 0.575228 0.368593 0.17834 0.617026 0.605807 0.688421 0.271453 0.380651 0.923694";
  assert_mat(ref_m, a.T());
  assert_mat(ref_m, a.H());

  ASSERT_TRUE(10 == a.size());
  a2 = a;
  a2.set_size(a2.size() + 3, true);
  ref = "0.261307 0.575228 0.368593 0.17834 0.617026 0.605807 0.688421 0.271453 0.380651 0.923694 0 0 0";
  assert_vec(ref, a2);
  a2.set_size(a2.size() - 6, true);
  ref = "0.261307 0.575228 0.368593 0.17834 0.617026 0.605807 0.688421";
  assert_vec(ref, a2);

  ASSERT_NEAR(0.605807, a(5), tol);
  ASSERT_NEAR(0.605807, a.get(5), tol);
  ref = "0.261307 0.575228 0.368593 0.17834 0.617026 0.605807";
  assert_vec(ref, a(0, 5));
  assert_vec(ref, a.get(0, 5));
  ref = "0.688421 0.271453 0.380651 0.923694";
  assert_vec(ref, a(6, -1));
  ivec idx_list = "0 5 6 7";
  ref = "0.261307 0.605807 0.688421 0.271453";
  assert_vec(ref, a(idx_list));
  assert_vec(ref, a.get(idx_list));
  bvec bin_list = "1 0 0 0 0 1 1 1 0 0";
  ref = "0.261307 0.605807 0.688421 0.271453";
  assert_vec(ref, a(bin_list));
  assert_vec(ref, a.get(bin_list));
  ref = "0.271453 0.380651 0.923694";
  assert_vec(ref, a.right(3));
  ref = "0.261307 0.575228 0.368593 0.17834";
  assert_vec(ref, a.left(4));
  ref = "0.17834 0.617026";
  assert_vec(ref, a.mid(3, 2));
  a2 = a;
  ASSERT_TRUE(0 == a2.split(0).length());
  a2 = a;
  assert_vec(a, a2.split(a2.size()));
  ASSERT_TRUE(0 == a2.length());
  a2 = a;
  ref = "0.261307 0.575228 0.368593 0.17834";
  assert_vec(ref, a2.split(4));
  ref = "0.617026 0.605807 0.688421 0.271453 0.380651 0.923694";
  assert_vec(ref, a2);
  a2(5) = a(6);
  ref = "0.617026 0.605807 0.688421 0.271453 0.380651 0.688421";
  assert_vec(ref, a2);

  a2 = a;
  a2.shift_left(c, 2);
  ref = "0.368593 0.17834 0.617026 0.605807 0.688421 0.271453 0.380651 0.923694 0.840642 0.840642";
  assert_vec(ref, a2);
  a2 = a;
  a2.shift_right(c);
  ref = "0.840642 0.261307 0.575228 0.368593 0.17834 0.617026 0.605807 0.688421 0.271453 0.380651";
  assert_vec(ref, a2);
  a2 = a;
  a2.shift_left(b.mid(0, 2));
  ref = "0.368593 0.17834 0.617026 0.605807 0.688421 0.271453 0.380651 0.923694 0.18369 0.813617";
  assert_vec(ref, a2);
  a2 = a;
  a2.shift_right(b.right(5));
  ref = "0.143443 0.0445045 0.0597922 0.473318 0.907724 0.261307 0.575228 0.368593 0.17834 0.617026";
  assert_vec(ref, a2);

  a2 = a;
  a2.set_subvector(0, b);
  ref = "0.18369 0.813617 0.674317 0.388616 0.11262 0.143443 0.0445045 0.0597922 0.473318 0.907724";
  assert_vec(ref, a2);
  a2 = a;
  a2.set_subvector(4, b(3, 5));
  ref = "0.261307 0.575228 0.368593 0.17834 0.388616 0.11262 0.143443 0.271453 0.380651 0.923694";
  assert_vec(ref, a2);
  a2 = a;
  a2.replace_mid(4, b(3, 5));
  ref = "0.261307 0.575228 0.368593 0.17834 0.388616 0.11262 0.143443 0.271453 0.380651 0.923694";
  assert_vec(ref, a2);
  a2 = a;
  a2.del(6);
  ref = "0.261307 0.575228 0.368593 0.17834 0.617026 0.605807 0.271453 0.380651 0.923694";
  assert_vec(ref, a2);
  a2 = a;
  a2.del(3, 9);
  ref = "0.261307 0.575228 0.368593";
  assert_vec(ref, a2);
  a2 = a;
  a2.ins(0, c);
  ref = "0.840642 0.261307 0.575228 0.368593 0.17834 0.617026 0.605807 0.688421 0.271453 0.380651 0.923694";
  assert_vec(ref, a2);
  a2 = a;
  a2.ins(2, c);
  ref = "0.261307 0.575228 0.840642 0.368593 0.17834 0.617026 0.605807 0.688421 0.271453 0.380651 0.923694";
  assert_vec(ref, a2);
  a2 = a;
  a2.ins(10, c);
  ref = "0.261307 0.575228 0.368593 0.17834 0.617026 0.605807 0.688421 0.271453 0.380651 0.923694 0.840642";
  assert_vec(ref, a2);
  a2 = a;
  a2.ins(3, b(0, 2));
  ref = "0.261307 0.575228 0.368593 0.18369 0.813617 0.674317 0.17834 0.617026 0.605807 0.688421 0.271453 0.380651 0.923694";
  assert_vec(ref, a2);

  a2 = a;
  a2.zeros();
  ref = "0 0 0 0 0 0 0 0 0 0";
  assert_vec(ref, a2);
  a2 = a;
  a2.ones();
  ref = "1 1 1 1 1 1 1 1 1 1";
  assert_vec(ref, a2);
  a2 = a;
  a2 = c;
  ref = "0.840642 0.840642 0.840642 0.840642 0.840642 0.840642 0.840642 0.840642 0.840642 0.840642";
  assert_vec(ref, a2);
  a2 = a;
  a2 = b(0, 4);
  ref = "0.18369 0.813617 0.674317 0.388616 0.11262";
  assert_vec(ref, a2);
  a2 = a;
  a2 = b.T();
  ref = "0.18369 0.813617 0.674317 0.388616 0.11262 0.143443 0.0445045 0.0597922 0.473318 0.907724";
  assert_vec(ref, a2);
  a2 = a;
  a2 = b.T().T();
  assert_vec(ref, a2);
}

void common_operators(const cvec &a, const cvec &b, complex<double> c)
{
  cvec a2 = a;
  cvec x1;
  cvec b2 = b;
  cvec x2;
  cvec ref = "-0.868942-0.457227i 0.449875-0.969414i 0.0252512+1.74683i -0.8595-2.69579i 0.96306-0.65313i "
             "0.505363-0.555045i -0.388457-0.827127i -0.804482+0.173017i -1.72448+0.367115i 0.837401+0.282386i";
  assert_cvec(ref, a + b);
  assert_cvec((a + c), (c + a));
  a2 += b;
  assert_cvec(ref, a2);
  a2 = a;
  a2 += c;
  ref = "-0.779698+0.528737i -0.421426+0.0486936i -1.43411+2.73273i -0.0208198-0.188316i -0.367156+1.54651i "
        "0.198414+1.18689i -1.02644+0.93711i -1.23285+1.59869i -1.23005+1.19476i -0.016846+1.03072i";
  assert_cvec(ref, a2);
  ref = "0.250684-0.7905i -0.351589-1.2384i -1.95234+1.41343i 1.759+0.0139543i -0.756234+1.44096i "
        "0.832603+0.623633i -0.723294+0.396147i -0.720078+0.719172i 0.205516-0.282798i 0.0700463-0.526148i";
  assert_cvec(ref, a - b);
  ref = "0.16144-1.77646i 0.519713-2.25651i -0.492974+0.427534i 0.920319-2.49352i 0.573983-0.758688i "
        "1.13955-1.11831i -0.085306-1.36809i -0.29171-0.706505i -0.28891-1.11044i 0.924293-1.27448i";
  assert_cvec(ref, a - c);
  ref = -ref;
  assert_cvec(ref, c - a);
  a2 = a;
  a2 -= b;
  ref = "0.250684-0.7905i -0.351589-1.2384i -1.95234+1.41343i 1.759+0.0139543i -0.756234+1.44096i "
        "0.832603+0.623633i -0.723294+0.396147i -0.720078+0.719172i 0.205516-0.282798i 0.0700463-0.526148i";
  assert_cvec(ref, a2);
  a2 = a;
  a2 -= c;
  ref = "0.16144-1.77646i 0.519713-2.25651i -0.492974+0.427534i 0.920319-2.49352i 0.573983-0.758688i "
        "1.13955-1.11831i -0.085306-1.36809i -0.29171-0.706505i -0.28891-1.11044i 0.924293-1.27448i";
  assert_cvec(ref, a2);
  ref = "0.309129+0.623863i -0.0491432+1.10391i 0.963544-1.58013i -0.44975+1.34092i -0.103413-0.393913i "
        "-0.668983-0.034294i 0.555875+0.21549i 0.76228-0.446095i 0.75948-0.0421585i -0.453723+0.121881i";
  assert_cvec(ref, -a);

  complex<double> act = a * b;
  ASSERT_NEAR(-1.89282, act.real(), tol);
  ASSERT_NEAR(2.58293, act.imag(), tol);
  act = dot(a, b);
  ASSERT_NEAR(-1.89282, act.real(), tol);
  ASSERT_NEAR(2.58293, act.imag(), tol);
  cmat ref_m = "0.277013+0.297734i -0.0399727-0.291577i -0.201667-0.668405i -0.440527+1.23562i "
               "-0.918953-0.212631i -0.317087+0.284258i -0.433332+0.0846286i -0.157317+0.110744i 0.501037+0.501572i 0.133602-0.364333i;"
               "0.15644+0.62617i 0.168161-0.435761i 0.232614-1.08334i -1.55999+1.37871i -1.11359-1.00042i "
               "-0.658616+0.151659i -0.666962-0.214872i -0.303526+0.0331671i 0.311299+1.08123i 0.465128-0.403677i;"
               "0.276096-1.04514i -0.598639+0.50362i -1.21616+1.4018i 3.4024-0.763313i 0.82616+2.36723i 1.08889+0.309313i "
               "0.805153+0.853882i 0.472162+0.196437i 0.416341-1.83793i -1.00849+0.216732i;"
               "-0.0283301+0.825607i 0.360573-0.47686i 0.668242-1.25092i -2.4056+1.14624i -1.01737-1.62362i "
               "-0.863842-0.0456547i -0.744858-0.499577i -0.385154-0.0662272i 0.001733+1.44013i 0.714647-0.33266i;"
               "-0.123532-0.203285i -0.0115374+0.171762i 0.0365891+0.406738i 0.398307-0.655842i 0.501342+0.230348i "
               "0.215228-0.125397i 0.258245+0.00269673i 0.103204-0.0448637i -0.227798-0.346519i -0.119569+0.192942i;"
               "-0.38022+0.0922787i 0.26347+0.103716i 0.65577+0.14543i -0.829402-0.951285i 0.610996-0.670973i "
               "-0.089248-0.399869i 0.132975-0.403433i -0.0188674-0.184131i -0.65671+0.184297i 0.24281+0.283606i;"
               "0.347095+0.0280051i -0.193775-0.161115i -0.513724-0.30574i 0.435819+1.03527i -0.703484+0.39678i "
               "-0.0360446+0.362858i -0.224865+0.303917i -0.0353865+0.160891i 0.606442+0.0273118i -0.126161-0.307401i;"
               "0.352398-0.376753i -0.365466+0.0762432i -0.828102+0.314024i 1.60242+0.448741i -0.188211+1.18162i "
               "0.387625+0.376251i 0.145229+0.540923i 0.153988+0.189335i 0.590635-0.678187i -0.472811-0.137008i;"
               "0.418141-0.150158i -0.310018-0.0852504i -0.757998-0.0849196i 1.05147+0.973801i -0.608743+0.831449i "
               "0.149112+0.440693i -0.101365+0.471584i 0.043564+0.205618i 0.719195-0.287481i -0.308438-0.290857i;"
               "-0.23369+0.143838i 0.198214+0.0121808i 0.468957-0.0448797i -0.759171-0.455163i 0.262427-0.579843i "
               "-0.146068-0.247455i 0.00141441-0.297919i -0.0524311-0.118758i -0.398235+0.265055i 0.223356+0.136662i";
  assert_cmat(ref_m, outer_product(a, b));
  ASSERT_TRUE((a * c) == (c * a));
  a2 = a;
  a2 *= c;
  assert_cvec(a*c, a2);
  ref = "0.277013+0.297734i 0.168161-0.435761i -1.21616+1.4018i -2.4056+1.14624i 0.501342+0.230348i "
        "-0.089248-0.399869i -0.224865+0.303917i 0.153988+0.189335i 0.719195-0.287481i 0.223356+0.136662i";
  assert_cvec(ref, elem_mult(a, b));
  elem_mult_out(a, b, x1);
  ref = "0.277013+0.297734i 0.168161-0.435761i -1.21616+1.4018i -2.4056+1.14624i 0.501342+0.230348i "
        "-0.089248-0.399869i -0.224865+0.303917i 0.153988+0.189335i 0.719195-0.287481i 0.223356+0.136662i";
  assert_cvec(ref, x1);
  elem_mult_inplace(a, b2);
  ref = "0.277013+0.297734i 0.168161-0.435761i -1.21616+1.4018i -2.4056+1.14624i 0.501342+0.230348i "
        "-0.089248-0.399869i -0.224865+0.303917i 0.153988+0.189335i 0.719195-0.287481i 0.223356+0.136662i";
  assert_cvec(ref, b2);
  act = elem_mult_sum(a, b);
  ASSERT_NEAR(-1.89282, act.real(), tol);
  ASSERT_NEAR(2.58293, act.imag(), tol);

  ref = "-0.370082+0.419294i -0.83584+0.29861i 1.46761+0.236797i -1.13372+0.072657i 0.261536-0.196498i "
        "-0.177606-0.507901i 0.00851914+0.478801i 0.563173+0.431431i 0.261936+0.551988i -0.228391-0.300407i";
  assert_cvec(ref, a / c);
  ref = "-1.18325-1.34059i -1.06098-0.379044i 0.664093-0.107151i -0.878444-0.0562971i 2.44397+1.83622i "
        "-0.613476+1.75436i 0.037149-2.08789i 1.11897-0.857211i 0.701675-1.47867i -1.60379+2.1095i";
  assert_cvec(ref, c / a);
  a2 = a;
  a2 /= c;
  ref = "-0.370082+0.419294i -0.83584+0.29861i 1.46761+0.236797i -1.13372+0.072657i 0.261536-0.196498i "
        "-0.177606-0.507901i 0.00851914+0.478801i 0.563173+0.431431i 0.261936+0.551988i -0.228391-0.300407i";
  assert_cvec(ref, a2);
  a2 = a;
  a2 /= b;
  ref = "0.202534+1.1747i -0.720721-2.51284i -0.685564+1.71362i 0.345915+0.666218i -0.17629+0.243506i "
        "-0.346625+1.03891i 0.0963326-0.935201i -1.17415-2.9729i 0.720086+0.198797i 0.401786-0.741014i";
  assert_cvec(ref, a2);
  assert_cvec(ref, elem_div(a, b));
  ref = "0.202534+1.1747i -0.720721-2.51284i -0.685564+1.71362i 0.345915+0.666218i -0.17629+0.243506i "
        "-0.346625+1.03891i 0.0963326-0.935201i -1.17415-2.9729i 0.720086+0.198797i 0.401786-0.741014i";
  elem_div_out(a, b, x2);
  assert_cvec(ref, x2);
  act = elem_div_sum(a, b);
  ASSERT_NEAR(-1.33669, act.real(), tol);
  ASSERT_NEAR(-2.1262, act.imag(), tol);

  ref = "-0.309129-0.623863i 0.0491432-1.10391i -0.963544+1.58013i 0.44975-1.34092i 0.103413+0.393913i "
        "0.668983+0.034294i -0.555875-0.21549i -0.76228+0.446095i -0.75948+0.0421585i 0.453723-0.121881i "
        "-0.559813+0.166636i 0.400732+0.134493i 0.988795+0.166701i -1.30925-1.35487i 0.859647-1.04704i "
        "-0.16362-0.589339i 0.167418-0.611637i -0.0422019-0.273077i -0.964996+0.324957i 0.383677+0.404267i";
  assert_cvec(ref, concat(a, b));
  ref = "-0.309129-0.623863i 0.0491432-1.10391i -0.963544+1.58013i 0.44975-1.34092i 0.103413+0.393913i "
        "0.668983+0.034294i -0.555875-0.21549i -0.76228+0.446095i -0.75948+0.0421585i 0.453723-0.121881i -0.470569+1.1526i";
  assert_cvec(ref, concat(a, c));
  ref = "-0.470569+1.1526i -0.309129-0.623863i 0.0491432-1.10391i -0.963544+1.58013i 0.44975-1.34092i "
        "0.103413+0.393913i 0.668983+0.034294i -0.555875-0.21549i -0.76228+0.446095i -0.75948+0.0421585i 0.453723-0.121881i";
  assert_cvec(ref, concat(c, a));
  ref = "-0.309129-0.623863i 0.0491432-1.10391i -0.963544+1.58013i 0.44975-1.34092i 0.103413+0.393913i "
        "0.668983+0.034294i -0.555875-0.21549i -0.76228+0.446095i -0.75948+0.0421585i 0.453723-0.121881i "
        "-0.559813+0.166636i 0.400732+0.134493i 0.988795+0.166701i -1.30925-1.35487i 0.859647-1.04704i "
        "-0.16362-0.589339i 0.167418-0.611637i -0.0422019-0.273077i -0.964996+0.324957i 0.383677+0.404267i "
        "-0.309129-0.623863i 0.0491432-1.10391i -0.963544+1.58013i 0.44975-1.34092i 0.103413+0.393913i "
        "0.668983+0.034294i -0.555875-0.21549i -0.76228+0.446095i -0.75948+0.0421585i 0.453723-0.121881i";
  assert_cvec(ref, concat(a, b, a));
  ref = "-0.309129-0.623863i 0.0491432-1.10391i -0.963544+1.58013i 0.44975-1.34092i 0.103413+0.393913i "
        "0.668983+0.034294i -0.555875-0.21549i -0.76228+0.446095i -0.75948+0.0421585i 0.453723-0.121881i "
        "-0.559813+0.166636i 0.400732+0.134493i 0.988795+0.166701i -1.30925-1.35487i 0.859647-1.04704i "
        "-0.16362-0.589339i 0.167418-0.611637i -0.0422019-0.273077i -0.964996+0.324957i 0.383677+0.404267i "
        "-0.309129-0.623863i 0.0491432-1.10391i -0.963544+1.58013i 0.44975-1.34092i 0.103413+0.393913i "
        "0.668983+0.034294i -0.555875-0.21549i -0.76228+0.446095i -0.75948+0.0421585i 0.453723-0.121881i "
        "-0.559813+0.166636i 0.400732+0.134493i 0.988795+0.166701i -1.30925-1.35487i 0.859647-1.04704i "
        "-0.16362-0.589339i 0.167418-0.611637i -0.0422019-0.273077i -0.964996+0.324957i 0.383677+0.404267i";
  assert_cvec(ref, concat(a, b, a, b));
  ref = "-0.309129-0.623863i 0.0491432-1.10391i -0.963544+1.58013i 0.44975-1.34092i 0.103413+0.393913i "
        "0.668983+0.034294i -0.555875-0.21549i -0.76228+0.446095i -0.75948+0.0421585i 0.453723-0.121881i "
        "-0.559813+0.166636i 0.400732+0.134493i 0.988795+0.166701i -1.30925-1.35487i 0.859647-1.04704i "
        "-0.16362-0.589339i 0.167418-0.611637i -0.0422019-0.273077i -0.964996+0.324957i 0.383677+0.404267i "
        "-0.309129-0.623863i 0.0491432-1.10391i -0.963544+1.58013i 0.44975-1.34092i 0.103413+0.393913i "
        "0.668983+0.034294i -0.555875-0.21549i -0.76228+0.446095i -0.75948+0.0421585i 0.453723-0.121881i "
        "-0.559813+0.166636i 0.400732+0.134493i 0.988795+0.166701i -1.30925-1.35487i 0.859647-1.04704i "
        "-0.16362-0.589339i 0.167418-0.611637i -0.0422019-0.273077i -0.964996+0.324957i 0.383677+0.404267i "
        "-0.309129-0.623863i 0.0491432-1.10391i -0.963544+1.58013i 0.44975-1.34092i 0.103413+0.393913i "
        "0.668983+0.034294i -0.555875-0.21549i -0.76228+0.446095i -0.75948+0.0421585i 0.453723-0.121881i";
  assert_cvec(ref, concat(a, b, a, b, a));

  ref_m = "-0.309129-0.623863i 0.0491432-1.10391i -0.963544+1.58013i 0.44975-1.34092i 0.103413+0.393913i "
          "0.668983+0.034294i -0.555875-0.21549i -0.76228+0.446095i -0.75948+0.0421585i 0.453723-0.121881i";
  assert_cmat(ref_m, a.T());
  ref_m = "-0.309129+0.623863i 0.0491432+1.10391i -0.963544-1.58013i 0.44975+1.34092i 0.103413-0.393913i "
          "0.668983-0.034294i -0.555875+0.21549i -0.76228-0.446095i -0.75948-0.0421585i 0.453723+0.121881i";
  assert_cmat(ref_m, a.H());

  ASSERT_TRUE(10 == a.size());
  a2 = a;
  a2.set_size(a2.size() + 3, true);
  ref = "-0.309129-0.623863i 0.0491432-1.10391i -0.963544+1.58013i 0.44975-1.34092i 0.103413+0.393913i "
        "0.668983+0.034294i -0.555875-0.21549i -0.76228+0.446095i -0.75948+0.0421585i 0.453723-0.121881i 0+0i 0+0i 0+0i";
  assert_cvec(ref, a2);
  a2.set_size(a2.size() - 6, true);
  ref = "-0.309129-0.623863i 0.0491432-1.10391i -0.963544+1.58013i 0.44975-1.34092i 0.103413+0.393913i "
        "0.668983+0.034294i -0.555875-0.21549i";
  assert_cvec(ref, a2);

  ASSERT_NEAR(0.668983, a(5).real(), tol);
  ASSERT_NEAR(0.034294, a(5).imag(), tol);
  ASSERT_NEAR(0.668983, a.get(5).real(), tol);
  ASSERT_NEAR(0.034294, a.get(5).imag(), tol);
  ref = "-0.309129-0.623863i 0.0491432-1.10391i -0.963544+1.58013i 0.44975-1.34092i 0.103413+0.393913i 0.668983+0.034294i";
  assert_cvec(ref, a(0, 5));
  assert_cvec(ref, a.get(0, 5));
  ref = "-0.555875-0.21549i -0.76228+0.446095i -0.75948+0.0421585i 0.453723-0.121881i";
  assert_cvec(ref, a(6, -1));
  ivec idx_list = "0 5 6 7";
  ref = "-0.309129-0.623863i 0.668983+0.034294i -0.555875-0.21549i -0.76228+0.446095i";
  assert_cvec(ref, a(idx_list));
  assert_cvec(ref, a.get(idx_list));
  bvec bin_list = "1 0 0 0 0 1 1 1 0 0";
  ref = "-0.309129-0.623863i 0.668983+0.034294i -0.555875-0.21549i -0.76228+0.446095i";
  assert_cvec(ref, a(bin_list));
  assert_cvec(ref, a.get(bin_list));
  ref = "-0.76228+0.446095i -0.75948+0.0421585i 0.453723-0.121881i";
  assert_cvec(ref, a.right(3));
  ref = "-0.309129-0.623863i 0.0491432-1.10391i -0.963544+1.58013i 0.44975-1.34092i";
  assert_cvec(ref, a.left(4));
  ref = "0.44975-1.34092i 0.103413+0.393913i";
  assert_cvec(ref, a.mid(3, 2));
  a2 = a;
  ASSERT_TRUE(0 == a2.split(0).length());
  a2 = a;
  assert_cvec(a, a2.split(a2.size()));
  ASSERT_TRUE(0 == a2.length());
  a2 = a;
  ref = "-0.309129-0.623863i 0.0491432-1.10391i -0.963544+1.58013i 0.44975-1.34092i";
  assert_cvec(ref, a2.split(4));
  ref = "0.103413+0.393913i 0.668983+0.034294i -0.555875-0.21549i -0.76228+0.446095i -0.75948+0.0421585i 0.453723-0.121881i";
  assert_cvec(ref, a2);
  a2(5) = a(6);
  ref = "0.103413+0.393913i 0.668983+0.034294i -0.555875-0.21549i -0.76228+0.446095i -0.75948+0.0421585i -0.555875-0.21549i";
  assert_cvec(ref, a2);

  a2 = a;
  a2.shift_left(c, 2);
  ref = "-0.963544+1.58013i 0.44975-1.34092i 0.103413+0.393913i 0.668983+0.034294i -0.555875-0.21549i "
        "-0.76228+0.446095i -0.75948+0.0421585i 0.453723-0.121881i -0.470569+1.1526i -0.470569+1.1526i";
  assert_cvec(ref, a2);
  a2 = a;
  a2.shift_right(c);
  ref = "-0.470569+1.1526i -0.309129-0.623863i 0.0491432-1.10391i -0.963544+1.58013i 0.44975-1.34092i "
        "0.103413+0.393913i 0.668983+0.034294i -0.555875-0.21549i -0.76228+0.446095i -0.75948+0.0421585i";
  assert_cvec(ref, a2);
  a2 = a;
  a2.shift_left(b.mid(0, 2));
  ref = "-0.963544+1.58013i 0.44975-1.34092i 0.103413+0.393913i 0.668983+0.034294i -0.555875-0.21549i "
        "-0.76228+0.446095i -0.75948+0.0421585i 0.453723-0.121881i -0.559813+0.166636i 0.400732+0.134493i";
  assert_cvec(ref, a2);
  a2 = a;
  a2.shift_right(b.right(5));
  ref = "-0.16362-0.589339i 0.167418-0.611637i -0.0422019-0.273077i -0.964996+0.324957i 0.383677+0.404267i "
        "-0.309129-0.623863i 0.0491432-1.10391i -0.963544+1.58013i 0.44975-1.34092i 0.103413+0.393913i";
  assert_cvec(ref, a2);

  a2 = a;
  a2.set_subvector(0, b);
  ref = "-0.559813+0.166636i 0.400732+0.134493i 0.988795+0.166701i -1.30925-1.35487i 0.859647-1.04704i "
        "-0.16362-0.589339i 0.167418-0.611637i -0.0422019-0.273077i -0.964996+0.324957i 0.383677+0.404267i";
  assert_cvec(ref, a2);
  a2 = a;
  a2.set_subvector(4, b(3, 5));
  ref = "-0.309129-0.623863i 0.0491432-1.10391i -0.963544+1.58013i 0.44975-1.34092i -1.30925-1.35487i "
        "0.859647-1.04704i -0.16362-0.589339i -0.76228+0.446095i -0.75948+0.0421585i 0.453723-0.121881i";
  assert_cvec(ref, a2);
  a2 = a;
  a2.replace_mid(4, b(3, 5));
  ref = "-0.309129-0.623863i 0.0491432-1.10391i -0.963544+1.58013i 0.44975-1.34092i -1.30925-1.35487i "
        "0.859647-1.04704i -0.16362-0.589339i -0.76228+0.446095i -0.75948+0.0421585i 0.453723-0.121881i";
  assert_cvec(ref, a2);
  a2 = a;
  a2.del(6);
  ref = "-0.309129-0.623863i 0.0491432-1.10391i -0.963544+1.58013i 0.44975-1.34092i 0.103413+0.393913i "
        "0.668983+0.034294i -0.76228+0.446095i -0.75948+0.0421585i 0.453723-0.121881i";
  assert_cvec(ref, a2);
  a2 = a;
  a2.del(3, 9);
  ref = "-0.309129-0.623863i 0.0491432-1.10391i -0.963544+1.58013i";
  assert_cvec(ref, a2);
  a2 = a;
  a2.ins(0, c);
  ref = "-0.470569+1.1526i -0.309129-0.623863i 0.0491432-1.10391i -0.963544+1.58013i 0.44975-1.34092i "
        "0.103413+0.393913i 0.668983+0.034294i -0.555875-0.21549i -0.76228+0.446095i -0.75948+0.0421585i 0.453723-0.121881i";
  assert_cvec(ref, a2);
  a2 = a;
  a2.ins(2, c);
  ref = "-0.309129-0.623863i 0.0491432-1.10391i -0.470569+1.1526i -0.963544+1.58013i 0.44975-1.34092i "
        "0.103413+0.393913i 0.668983+0.034294i -0.555875-0.21549i -0.76228+0.446095i -0.75948+0.0421585i 0.453723-0.121881i";
  assert_cvec(ref, a2);
  a2 = a;
  a2.ins(10, c);
  ref = "-0.309129-0.623863i 0.0491432-1.10391i -0.963544+1.58013i 0.44975-1.34092i 0.103413+0.393913i "
        "0.668983+0.034294i -0.555875-0.21549i -0.76228+0.446095i -0.75948+0.0421585i 0.453723-0.121881i -0.470569+1.1526i";
  assert_cvec(ref, a2);
  a2 = a;
  a2.ins(3, b(0, 2));
  ref = "-0.309129-0.623863i 0.0491432-1.10391i -0.963544+1.58013i -0.559813+0.166636i 0.400732+0.134493i "
        "0.988795+0.166701i 0.44975-1.34092i 0.103413+0.393913i 0.668983+0.034294i -0.555875-0.21549i -0.76228+0.446095i "
        "-0.75948+0.0421585i 0.453723-0.121881i";
  assert_cvec(ref, a2);

  a2 = a;
  a2.zeros();
  ref = "0+0i 0+0i 0+0i 0+0i 0+0i 0+0i 0+0i 0+0i 0+0i 0+0i";
  assert_cvec(ref, a2);
  a2 = a;
  a2.ones();
  ref = "1+0i 1+0i 1+0i 1+0i 1+0i 1+0i 1+0i 1+0i 1+0i 1+0i";
  assert_cvec(ref, a2);
  a2 = a;
  a2 = c;
  ref = "-0.470569+1.1526i -0.470569+1.1526i -0.470569+1.1526i -0.470569+1.1526i -0.470569+1.1526i "
        "-0.470569+1.1526i -0.470569+1.1526i -0.470569+1.1526i -0.470569+1.1526i -0.470569+1.1526i";
  assert_cvec(ref, a2);
  a2 = a;
  a2 = b(0, 4);
  ref = "-0.559813+0.166636i 0.400732+0.134493i 0.988795+0.166701i -1.30925-1.35487i 0.859647-1.04704i";
  assert_cvec(ref, a2);
  a2 = a;
  a2 = b.T();
  ref = "-0.559813+0.166636i 0.400732+0.134493i 0.988795+0.166701i -1.30925-1.35487i 0.859647-1.04704i "
        "-0.16362-0.589339i 0.167418-0.611637i -0.0422019-0.273077i -0.964996+0.324957i 0.383677+0.404267i";
  assert_cvec(ref, a2);
  a2 = a;
  a2 = b.T().T();
  assert_cvec(ref, a2);
}

void logical_operators(const bvec &a, const bvec &b, bin c)
{
  bvec a2 = a;
  bvec ref = "0 0 0 0 1 0 0 1 1 0";
  ASSERT_TRUE(ref == (a == c));
  ref = "1 1 1 1 0 1 1 0 0 1";
  ASSERT_TRUE(ref == (a != c));
  ref = "0 0 0 0 1 0 0 1 1 0";
  ASSERT_TRUE(ref == (a <= c));
  ref = "1 1 1 1 1 1 1 1 1 1";
  ASSERT_TRUE(ref == (a >= c));
  ref = "0 0 0 0 0 0 0 0 0 0";
  ASSERT_TRUE(ref == (a < c));
  ref = "1 1 1 1 0 1 1 0 0 1";
  ASSERT_TRUE(ref == (a > c));
  ASSERT_FALSE(a == b);
  ASSERT_TRUE(a != b);
  ASSERT_TRUE(a2 == a);
  ASSERT_FALSE(a2 != a);
}

void logical_operators(const ivec &a, const ivec &b, int c)
{
  ivec a2 = a;
  bvec ref = "0 0 1 0 0 0 0 0 0 0";
  ASSERT_TRUE(ref == (a == c));
  ref = "1 1 0 1 1 1 1 1 1 1";
  ASSERT_TRUE(ref == (a != c));
  ref = "0 1 1 0 0 1 1 1 0 1";
  ASSERT_TRUE(ref == (a <= c));
  ref = "1 0 1 1 1 0 0 0 1 0";
  ASSERT_TRUE(ref == (a >= c));
  ref = "0 1 0 0 0 1 1 1 0 1";
  ASSERT_TRUE(ref == (a < c));
  ref = "1 0 0 1 1 0 0 0 1 0";
  ASSERT_TRUE(ref == (a > c));
  ASSERT_FALSE(a == b);
  ASSERT_TRUE(a != b);
  ASSERT_TRUE(a2 == a);
  ASSERT_FALSE(a2 != a);
}

void logical_operators(const vec &a, const vec &b, double c)
{
  vec a2 = a;
  bvec ref = "0 0 0 0 0 0 0 0 0 0";
  ASSERT_TRUE(ref == (a == c));
  ref = "1 1 1 1 1 1 1 1 1 1";
  ASSERT_TRUE(ref == (a != c));
  ref = "1 1 1 1 1 1 1 1 1 0";
  ASSERT_TRUE(ref == (a <= c));
  ref = "0 0 0 0 0 0 0 0 0 1";
  ASSERT_TRUE(ref == (a >= c));
  ref = "1 1 1 1 1 1 1 1 1 0";
  ASSERT_TRUE(ref == (a < c));
  ref = "0 0 0 0 0 0 0 0 0 1";
  ASSERT_TRUE(ref == (a > c));
  ASSERT_FALSE(a == b);
  ASSERT_TRUE(a != b);
  ASSERT_TRUE(a2 == a);
  ASSERT_FALSE(a2 != a);
}

TEST(Vec, All)
{
  RNG_reset(0);

  // Testing Vec<bin> (bvec)
  bvec bv1 = randb(10);
  bvec bv2 = randb(10);
  bin bx = randb();
  common_operators(bv1, bv2, bx);
  logical_operators(bv1, bv2, bx);

  // Testing Vec<int> (ivec)
  ivec iv1 = randi(10, 1, 9);
  ivec iv2 = randi(10, 1, 9);
  int ix = randi(1, 9);
  common_operators(iv1, iv2, ix);
  logical_operators(iv1, iv2, ix);

  // Testing Vec<double> (vec)
  vec dv1 = randu(10);
  vec dv2 = randu(10);
  double dx = randu();
  common_operators(dv1, dv2, dx);
  logical_operators(dv1, dv2, dx);

  // Testing Vec<std::complex<double> > (cvec)
  cvec cv1 = randn_c(10);
  cvec cv2 = randn_c(10);
  complex<double> cx = randn_c();
  common_operators(cv1, cv2, cx);
  ASSERT_FALSE(any(cv1 == cx));
  ASSERT_TRUE(all(cv1 != cx));


  // Test vectror initialisation with string
  vec v = "-10.000 :.5:-4.5  1.33e+1, -.9, 1e0:1.5:1E+1";
  vec ref = "-10.0000 -9.5000 -9.0000 -8.5000 -8.0000 -7.5000 -7.0000 -6.5000 -6.0000 -5.5000 -5.0000 "
            "-4.5000 13.3000 -0.9000 1.0000 2.5000 4.0000 5.5000 7.0000 8.5000 10.0000";
  assert_vec(ref, v);

  ivec iv = "0xA :-0x1: -010";
  ivec ref_i = "10 9 8 7 6 5 4 3 2 1 0 -1 -2 -3 -4 -5 -6 -7 -8";
  ASSERT_TRUE(ref_i == iv);

  iv = "-5:3:9, 7, 1:10";
  ref_i = "-5 -2 1 4 7 7 1 2 3 4 5 6 7 8 9 10";
  ASSERT_TRUE(ref_i == iv);

  svec sv = "3 0xF -10, 0133 0177, 0x0 ";
  svec ref_s = "3 15 -10 91 127 0";
  ASSERT_TRUE(ref_s == sv);

  cvec cv = " (0.3, 0.4)  .2-.01i, 1e-3+0.25i";
  cvec ref_c = "0.3000+0.4000i 0.2000-0.0100i 0.0010+0.2500i";
  assert_cvec(ref_c, cv);

  bvec bv = "1 1 0,1  1  ,  0 ,1  ";
  bvec ref_b = "1 1 0 1 1 0 1";
  ASSERT_TRUE(ref_b == bv);

  // Test of rem:
  v = "1.0 2.0 3.4 -4.5 6.7";
  double y = 0.76;
  ref = "0.2400 0.4800 0.3600 -0.7000 0.6200";
  assert_vec(ref, rem(v, y));
  ref = "0.0000 0.0000 3.2000 1.0000 3.3000";
  assert_vec(ref, rem(10, v));
  mat M = "1.0 2.3; 4.5 -6.7";
  mat ref_m = "0.2400 0.0200; 0.7000 -0.6200";
  assert_mat(ref_m, rem(M, y));
  ref_m = "0.0000 0.8000; 1.0000 3.3000";
  assert_mat(ref_m, rem(10, M));

  // Test of all and any:
  bvec b1 = "0 0 0 0 0 0 0 1 0 0";
  bvec b2 = "0 0 0 0 0 0 0 0 0 0";
  bvec b3 = "1 1 1 1 1 1 1 1 1 1 1 1 1";
  bvec b4 = "1 1 1 1 1 1 1 1 1 1 1 0 1";
  ASSERT_TRUE(any(b1));
  ASSERT_FALSE(any(b2));
  ASSERT_TRUE(all(b3));
  ASSERT_FALSE(all(b4));
}
