// -------------------------------------------------------------------------
//
//    Copyright (C) 2010 Fons Adriaensen <fons@linuxaudio.org>
//    
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// -------------------------------------------------------------------------


#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include <stdio.h>
#include <math.h>
#include "impdata.h"
#include "abprocess.h"


enum { HELP, HPF, INVX, INVY, INVZ, ENDF, FUMA, SN3D, N3D };
enum { BUFFSIZE = 256 };


static float hpfr = 0;
static int   form = Bformat::FM_FUMA;
static int   invb = 0;
static int   endf = 0;


static void help (void)
{
    fprintf (stderr, "\ntetra_ald %s\n", VERSION);
    fprintf (stderr, "(C) 2010 Fons Adriaensen  <fons@kokkinizita.net>\n");
    fprintf (stderr, "Usage: tetra_ald <options> <config file> <input file> <output file>.\n");
    fprintf (stderr, "Options:\n");
    fprintf (stderr, "  Display this text:     --help\n");
    fprintf (stderr, "  Highpass filter:       --hpf <frequency>\n");
    fprintf (stderr, "  B format standard:     --fuma, --sn3d, --n3d\n");
    fprintf (stderr, "  B format inversion:    --invx, --invy, --invz\n");
    fprintf (stderr, "  Endfire mode:          --endf\n");
    fprintf (stderr, "The default options are: --fuma.\n");
    exit (1);
}


static struct option options [] = 
{
    { "help",  0, 0, HELP  },
    { "hpf",   1, 0, HPF   },
    { "invx",  0, 0, INVX  },
    { "invy",  0, 0, INVY  },
    { "invz",  0, 0, INVZ  },
    { "endf",  0, 0, ENDF  },
    { "fuma",  0, 0, FUMA  },
    { "sn3d",  0, 0, SN3D  },
    { "n3d",   0, 0, N3D   },
    { 0, 0, 0, 0 }
};


static void procoptions (int ac, char *av [])
{
    int k;

    while ((k = getopt_long (ac, av, "", options, 0)) != -1)
    {
	switch (k)
	{
        case '?':
	case HELP:
	    help ();
	    break;
	case HPF:
	    if ((sscanf (optarg, "%f", &hpfr) != 1) || (hpfr < 0) || (hpfr > 160))
	    {
		fprintf (stderr, "Illegal value for --hpf option: '%s'.\n", optarg);
		exit (1);
	    }
	    break;
	case INVX:
	    invb |= 2;
	    break;
	case INVY:
	    invb |= 4;
	    break;
	case INVZ:
	    invb |= 8;
	    break;
	case ENDF:
	    endf = 1;
	    break;
	case FUMA:
	    form = Bformat::FM_FUMA;
	    break;
	case SN3D:
	    form = Bformat::FM_SN3D;
	    break;
	case N3D:
	    form = Bformat::FM_N3D;
	    break;
 	}
    }
}


int main (int ac, char *av [])
{
    Impdata       Ainp;
    Impdata       Aout;
    ABprocess     abproc;
    ABconfig      config;
    unsigned int  i, s, n, k, rate;
    float         *p, *q, *inp [4], *out [4];

    procoptions (ac, av);
    if (ac - optind < 3)
    {
        fprintf (stderr, "Missing arguments, try --help.\n");
	return 1;
    }
    if (ac - optind > 3 )
    {
        fprintf (stderr, "Too many arguments, try --help.\n");
	return 1;
    }

    if (config.load (av [optind]))
    {
	fprintf (stderr, "Can't open config file '%s'.\n", av [optind]);
	return 1;
    }

    optind++;
    if (Ainp.open_read (av [optind]))
    {
	fprintf (stderr, "Can't open input file '%s'.\n", av [optind]);
	return 1;
    }
    if (Ainp.n_chan () != 4)
    {
	fprintf (stderr, "Input file '%s' does not have 4 channels.\n", av [optind]);
	Ainp.close ();
	return 1;
    }
    rate = (int)(floor (Ainp.drate () + 0.5));

    Aout.set_type (Impdata::TYPE_AMB_B4);
    Aout.set_rate (Ainp.rate_n (), Ainp.rate_d ());
    Aout.set_tref (Ainp.tref_i (), Ainp.tref_n (), Ainp.tref_d ());
    Aout.set_n_sect (Ainp.n_sect ());
    Aout.set_n_fram (Ainp.n_fram ());

    optind++;
    if (Aout.open_write (av [optind]))
    {
	fprintf (stderr, "Can't open output file '%s'.\n", av [optind]);
	Ainp.close ();
	return 1;
    }

    Ainp.alloc ();
    Aout.alloc ();

    abproc.init (rate, BUFFSIZE, BUFFSIZE);
    abproc.set_lffilt (&config);
    abproc.set_matrix (&config);
    abproc.set_convol (&config);
    abproc.set_hffilt (&config); 
    abproc.set_hpfil ((hpfr < 1) ? 1 : hpfr);
    abproc.set_invb (invb);
    abproc.set_endf (endf);
    abproc.set_form (form);

    for (i = 0; i < 4; i++)
    {
	inp [i] = new float [BUFFSIZE];
	out [i] = new float [BUFFSIZE];
    }

    for (s = 0; s < Ainp.n_sect (); s++)
    {
	abproc.reset ();
	Ainp.read_sect (s);
	n = Ainp.n_fram ();
	p = Ainp.data (0);
	q = Aout.data (0);
	while (n)
	{
	    k = (n > BUFFSIZE) ?  BUFFSIZE : n;
	    if (k)
	    {
		for (i = 0; i < k; i++)
		{
		    inp [0][i] = p [0];
		    inp [1][i] = p [1];
		    inp [2][i] = p [2];
		    inp [3][i] = p [3];
		    p += 4;
		}
		while (i < BUFFSIZE)
		{
		    inp [0][i] = 0;
		    inp [1][i] = 0;
		    inp [2][i] = 0;
		    inp [3][i] = 0;
		    i++;
		}
		abproc.process (BUFFSIZE, inp, out); 
		for (i = 0; i < k; i++)
		{
		    q [0] = out [0][i];
		    q [1] = out [1][i];
		    q [2] = out [2][i];
		    q [3] = out [3][i];
		    q += 4;
		}
		n -= k;
	    }
	    else break;
	}
	Aout.write_sect (s);
    }

    Ainp.close ();
    Aout.close ();
    for (i = 0; i < 4; i++)
    {
	delete[] inp [i];
	delete[] out [i];
    }

    return 0;
}
