#include "aria2cinterface.h"

#include <QProcess>
#include <QDir>
#include <QStandardPaths>

//#ln /usr/bin/aria2c /usr/share/uos-downloadmanager/nfs-aria2c
const QString Aria2cInterface::basePath = "/usr/share/uos-downloadmanager/";// /usr/bin/
const QString Aria2cInterface::aria2cCmd = "nfs-aria2c";// aria2c

Aria2cInterface::Aria2cInterface(QObject *parent) : QObject(parent)
{
    //this->defaultDownloadPath = QDir::homePath() + "/Downloads";
}

//static
Aria2cBtInfo Aria2cInterface::getBtInfo(QString torrentPath)
{
    QProcess *proc = new QProcess;
    QStringList opt;
    opt << "--show-files=true";
    opt << torrentPath;
    proc->start(Aria2cInterface::basePath + Aria2cInterface::aria2cCmd, opt);
    proc->waitForFinished();
    QByteArray rt = proc->readAllStandardOutput();
    proc->close();

    QStringList lt = QString(rt).split("\n");
    Aria2cBtInfo info;
    bool flag = false;
    QString temp = "";
    foreach(QString line, lt) {
        if(line.startsWith("Mode: ")) {
            info.mode = line.mid(6);
        }
        else if(line.startsWith("Announce:")) {
            continue;
        }
        else if(!flag && line.startsWith(" ")) {
            info.announceList.append(line.mid(1));
        }
        else if(line.startsWith("Info Hash: ")) {
            info.infoHash = line.mid(11);
        }
        else if(line.startsWith("Piece Length: ")) {
            info.pieceLength = line.mid(14).remove('i');
        }
        else if(line.startsWith("The Number of Pieces: ")) {
            info.pieceNumber = line.mid(22).toInt();
        }
        else if(line.startsWith("Total Length: ")) {
            QString tpl = line.mid(14);
            QStringList sp = tpl.split(" ");
            info.totalLength = sp[0].remove('i');//MiB==>MB

            QString len = sp[1].remove(',');
            QString len2 = len.mid(1, len.length() - 2);
            info.totalLengthByets = len2.toLong();
        }
        else if(line.startsWith("Name: ")) {
            info.name = line.mid(6);
        }
        else if(line.startsWith("Magnet URI: ")) {
            info.magnet = line.mid(12);
        }
        else if(line.startsWith("Files:")) {
            continue;
        }
        else if(line.startsWith("idx")) {
            continue;
        }
        else if(line.startsWith("===+===")) {
            flag = true;
            temp = "";
            continue;
        }
        else if(flag && line.startsWith("---+---")) {//
            QStringList stl = temp.split("|");
            Aria2cBtFile f;
            f.index = stl[0].toInt();
            f.path = stl[1];
            QStringList spl = stl[2].split(" ");
            f.length = spl[0].remove('i');

            QString len = spl[1].remove(',');
            QString len2 = len.mid(1, len.length() - 2);
            f.lengthBytes = len2.toLong();
            info.files.append(f);
            temp = "";
        }
        else if(flag) {//idx|fileName
            temp += line.trimmed();
        }
    }
    return info;
}

//static
QString Aria2cInterface::bytesFormat(qint64 size) {
    if(!size) {
        return "0B";
    }
    QStringList sl;
    if(sl.empty()) {
        sl << "B" <<"KB" << "MB" << "GB" << "TB" << "PB";
    }
    int i = qFloor(qLn(size) / qLn(1024));
    return QString::number(size * 1.0 / qPow(1024, qFloor(i)), 'f', (i > 1) ? 2 : 0) + sl.at(i);
}

//static
QString Aria2cInterface::getCapacityFree(QString path)
{
    QProcess *proc = new QProcess;
    QStringList opt;
    opt << "-c";
    opt << "df -h " + path;
    proc->start("/bin/bash", opt);
    proc->waitForFinished();
    QByteArray rt = proc->readAllStandardOutput();
    proc->close();
    delete proc;

    QString free = "0B";
    QStringList lt = QString(rt).split("\n");
    if(lt.length() >= 2) {
        QString line = lt.at(1);
        QString temp;
        QStringList tpl;
        for(int i = 0;i <line.length();i++) {
            if(line[i] != ' ') {
                temp.append(line[i]);
            }
            else {
                if(temp != "") {
                    tpl.append(temp);
                    temp = "";
                }
            }
        }
        free = tpl[3];
    }
    return free + "B";
}

long Aria2cInterface::getCapacityFreeByte(QString path)
{
    QProcess *proc = new QProcess;
    QStringList opt;
    opt << "-c";
    opt << "df " + path;
    proc->start("/bin/bash", opt);
    proc->waitForFinished();
    QByteArray rt = proc->readAllStandardOutput();
    proc->close();
    delete proc;

    QString free = "0";
    QStringList lt = QString(rt).split("\n");
    if(lt.length() >= 2) {
        QString line = lt.at(1);
        QString temp;
        QStringList tpl;
        for(int i = 0;i <line.length();i++) {
            if(line[i] != ' ') {
                temp.append(line[i]);
            }
            else {
                if(temp != "") {
                    tpl.append(temp);
                    temp = "";
                }
            }
        }
        free = tpl[3];
    }
    return free.toLong();
}

bool Aria2cInterface::checkAria2cFile()
{
    QFile file(Aria2cInterface::basePath + Aria2cInterface::aria2cCmd);
    return file.exists();
}

int Aria2cInterface::checkAria2cProc()//有必要检查吗?
{
    QProcess *proc = new QProcess;
    QStringList opt;
    opt << "-c";
    //opt << "ps aux | grep aria2c";
    opt << "ps aux|grep " + Aria2cInterface::aria2cCmd;
    proc->start("/bin/bash", opt);
    proc->waitForFinished();
    QString output = QString::fromLocal8Bit(proc->readAllStandardOutput());
    QStringList lineList = output.split("\n");
    int cnt = 0;
    foreach(QString t, lineList) {
        if(t == "") {
            continue;
        }
        if(t.indexOf("grep " + Aria2cInterface::aria2cCmd) >= 0) {
            continue;
        }
        if(t.indexOf(Aria2cInterface::aria2cCmd) >= 0) {
            cnt++;
            //break;
        }
    }
    return cnt;
}

void Aria2cInterface::sendMessage(QJsonObject jsonObj, const QString &method)
{
    QNetworkAccessManager *manager = new QNetworkAccessManager;
    if(!jsonObj.isEmpty())
    {
        //qDebug() << "METHOD: " << jsonObj.value("method").toString();
       // qDebug() << ">>>SendMessage : " << QString(QJsonDocument(jsonObj).toJson());
        QNetworkRequest *request = new QNetworkRequest;
        request->setUrl(QUrl(this->rpcServer));
        request->setHeader(QNetworkRequest::ContentTypeHeader, "application/json" );
        manager->post(*request, QJsonDocument(jsonObj).toJson());

        QObject::connect(manager,
                 &QNetworkAccessManager::finished,
                 this,
                 [=](QNetworkReply* reply){


                     this->rpcRequestReply(reply, method, jsonObj.value("id").toString());
                     manager->deleteLater();
                     manager->destroyed();
                 });
    }
}

void Aria2cInterface::rpcRequestReply(QNetworkReply *reply, const QString &method, const QString id)
{
    //qDebug() << ">>>RequestReply";
    int code = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();
    if(code == 200) {
        QByteArray buf = reply->readAll();
//        qDebug() << "BUFFER: " << QString(buf);
       // qDebug() << "=========================";

        QJsonDocument doc = QJsonDocument::fromJson(buf);
       // qDebug() << "JSON: " << QString(doc.toJson());
       // qDebug() << "=========================";
        QJsonObject obj = doc.object();

        emit signal_success(method, obj);
    }
    else {
        emit signal_error(method, id, code);
    }
    /*if(reply->error() != QNetworkReply::NoError) {
        qDebug() << "Error:" << reply->errorString();
        emit signal_error(method, code);
    }
    else {
        QByteArray buf = reply->readAll();
        QJsonDocument doc = QJsonDocument::fromJson(buf);
        qDebug() << QString(doc.toJson());
        QJsonObject obj = doc.object();

        emit signal_success(method, obj);

    }*/
    reply->deleteLater();
    reply->destroyed();
}

void Aria2cInterface::callRPC(QString method, QJsonArray params, QString id)
{
    QJsonObject json;
    json.insert("jsonrpc", "2.0");
    if(id == "") {
        json.insert("id", method);
    }
    else {
        json.insert("id", id);
    }
    json.insert("method", method);
    if(!params.isEmpty()) {
        json.insert("params", params);
    }
    this->sendMessage(json, method);
}

void Aria2cInterface::callRPC(QString method, QString id)
{
    callRPC(method, QJsonArray(), id);
}

QString Aria2cInterface::getConfigPath() const
{
    return configPath;
}

void Aria2cInterface::setConfigPath(const QString &value)
{
    configPath = value;
}

QString Aria2cInterface::fileToBase64(QString filePath)
{
    QFile file(filePath);
    file.open(QIODevice::ReadOnly);
    QByteArray ba = file.readAll();
    QString b64Str = ba.toBase64();
    return b64Str;
}

QString Aria2cInterface::processThunderUri(QString thunder) {
    QString uri = thunder;
    if(thunder.startsWith("thunder://")) {
        QString oUir = thunder.mid(thunder.indexOf("thunder://") + 9+1);
        uri = QString(QByteArray::fromBase64(oUir.toLatin1()));
        uri = uri.mid(2, uri.length() - 4);//AA[URI]ZZ
    }
    return uri;
}


int Aria2cInterface::killAria2cProc()
{
    QStringList opt;
    opt << "-c";
    opt << "ps -ef|grep " + Aria2cInterface::aria2cCmd + "|grep -v grep|awk '{print $2}'|xargs kill -9";
    return QProcess::execute("/bin/bash", opt);
}

bool Aria2cInterface::startup()
{
    if(!this->checkAria2cFile()) {
        qDebug() << "未发现" << Aria2cInterface::basePath + Aria2cInterface::aria2cCmd ;
        return false;
    }

    int st = checkAria2cProc();
    if(st > 0) {
        qDebug() << Aria2cInterface::aria2cCmd + "进程已存在,killAria2cProc()";
        killAria2cProc();
    }

    QString sessionCacheFile = QDir::homePath() + "/.cache/nfs-aria2c.session";
    QString inputFile = QDir::homePath() + "/.cache/nfs-aria2c.input";
    QString saveSessionInterval = "30";//秒
    //创建session缓存文件
    qDebug() << "创建session缓存文件: " << sessionCacheFile;
    QProcess::execute("/usr/bin/touch", QStringList() << sessionCacheFile);
    //
    QStringList opt;
    opt << "--enable-rpc=true";//启动RPC
    opt << "--rpc-listen-port=" + this->rpcPort;//RPC监听的端口
    opt << "--check-certificate=false";//停用rpc身份验证
    opt << "--rpc-allow-origin-all=true";//
    opt << "--rpc-save-upload-metadata=true";//

    //opt << "--not-conf=true";//不使用配置文件
    if(this->configPath != "") {
        opt << "--conf-path=" + this->configPath;//加载指定的配置文件
    }
    if(this->defaultDownloadPath != "") {
      opt << "--dir=" + this->defaultDownloadPath;//配置默认下载路径。优先级高于配置文件，已移动到配置文件中
    }
    opt << "--continue=true";//http续传配置
    opt << "--disable-ipv6";//禁用ipv6
    //opt << "--seed-time=0";//bt完成不做种
    opt << "--bt-metadata-only=true";//仅下载bt metadata，不自动发起follow下载
    opt << "--bt-save-metadata=true";//保存magnet metadata到同目录下.torrent文件
    opt << "--follow-torrent=false";//当下载的文件是以.torrent结尾的，是否继续下载。true，是；false，否，只下载torrent文件；mem,不写文件保存在内存
    //opt << "--follow-metalink=false";//类似torrent
    opt << "--bt-remove-unselected-file=true";
    //opt << "--input-file=" + inputFile;
    opt << "--save-session=" + sessionCacheFile;
    opt << "--save-session-interval=" + saveSessionInterval;

    qDebug() << Aria2cInterface::basePath + Aria2cInterface::aria2cCmd << opt.join(' ');

    QProcess *proc = new QProcess;
    proc->start(Aria2cInterface::basePath + Aria2cInterface::aria2cCmd, opt);
    proc->waitForStarted();
    st = checkAria2cProc();
    qDebug() << "启动aria2c完成！ " << proc->state() << st;
    return st > 0;
}

void Aria2cInterface::shutdown(QString id)
{
    callRPC(ARIA2C_METHOD_SHUTDOWN, id);
}

void Aria2cInterface::forceShutdown(QString id)
{
    callRPC(ARIA2C_METHOD_FORCE_SHUTDOWN, id);
}

void Aria2cInterface::setDefaultDownloadPath(QString path)
{
    this->defaultDownloadPath = path;
}

void Aria2cInterface::addUri(QString uri, QMap<QString, QVariant> opt, QString id)
{
    uri = processThunderUri(uri);//处理迅雷链接
    QJsonArray ja, inner;
    inner.append(uri);//可支持多个URI
    ja.append(inner);

    QJsonDocument doc = QJsonDocument::fromVariant(QVariant(opt));
    QJsonObject optJson = doc.object();
    ja.append(optJson);

    callRPC(ARIA2C_METHOD_ADD_URI, ja, id);
}

void Aria2cInterface::addTorrent(QString torrentFile, QMap<QString, QVariant> opt, QString id)
{
    QString torrentB64Str = fileToBase64(torrentFile);
    QJsonArray ja;
    ja.append(torrentB64Str);
    ja.append(QJsonArray());

    QJsonDocument doc = QJsonDocument::fromVariant(QVariant(opt));
    QJsonObject optJson = doc.object();
    ja.append(optJson);

    callRPC(ARIA2C_METHOD_ADD_TORRENT, ja, id);
}

void Aria2cInterface::addMetalink(QString metalinkFile, QMap<QString, QVariant> opt, QString id)
{
    QString metalinkB64Str = fileToBase64(metalinkFile);
    QJsonArray ja;
    ja.append(metalinkB64Str);
    ja.append(QJsonArray());

    QJsonDocument doc = QJsonDocument::fromVariant(QVariant(opt));
    QJsonObject optJson = doc.object();
    ja.append(optJson);

    callRPC(ARIA2C_METHOD_ADD_METALINK, ja, id);
}

void Aria2cInterface::pause(QString gId, QString id)
{
    QJsonArray ja;
    ja.append(gId);
    callRPC(ARIA2C_METHOD_PAUSE, ja, id);
}

void Aria2cInterface::forcePause(QString gId, QString id)
{
    QJsonArray ja;
    ja.append(gId);
    callRPC(ARIA2C_METHOD_FORCE_PAUSE, ja, id);
}

void Aria2cInterface::pauseAll(QString id)
{
    callRPC(ARIA2C_METHOD_PAUSE_ALL, id);
}

void Aria2cInterface::forcePauseAll(QString id)
{
    callRPC(ARIA2C_METHOD_FORCE_PAUSE_ALL, id);
}

void Aria2cInterface::unpause(QString gId, QString id)
{
    QJsonArray ja;
    ja.append(gId);
    callRPC(ARIA2C_METHOD_UNPAUSE, ja, id);
}

void Aria2cInterface::unpauseAll(QString id)
{
    callRPC(ARIA2C_METHOD_UNPAUSE_ALL, id);
}

void Aria2cInterface::remove(QString gId, QString id)
{
    QJsonArray ja;
    ja.append(gId);
    callRPC(ARIA2C_METHOD_REMOVE, ja, id);
}

void Aria2cInterface::forceRemove(QString gId, QString id)
{
    QJsonArray ja;
    ja.append(gId);
    callRPC(ARIA2C_METHOD_FORCE_REMOVE, ja, id);
}

void Aria2cInterface::tellStatus(QString gId, QString id)
{
    QJsonArray ja;
    ja.append(gId);
    callRPC(ARIA2C_METHOD_TELL_STATUS, ja, id);
}

void Aria2cInterface::tellStatus(QString gId, QStringList keys, QString id)
{
    QJsonArray ja;
    ja.append(gId);
    QJsonArray ka;
    foreach(QString k, keys) {
        ka.append(k);
    }
    ja.append(ka);
    callRPC(ARIA2C_METHOD_TELL_STATUS, ja, id);
}

void Aria2cInterface::getUris(QString gId, QString id)
{
    QJsonArray ja;
    ja.append(gId);
    callRPC(ARIA2C_METHOD_GET_URIS, ja, id);
}

void Aria2cInterface::getFiles(QString gId, QString id)
{
    QJsonArray ja;
    ja.append(gId);
    callRPC(ARIA2C_METHOD_GET_FILES, ja, id);
}

void Aria2cInterface::getPeers(QString gId, QString id)
{
    QJsonArray ja;
    ja.append(gId);
    callRPC(ARIA2C_METHOD_GET_PEERS, ja, id);
}

void Aria2cInterface::getServers(QString gId, QString id)
{
    QJsonArray ja;
    ja.append(gId);
    callRPC(ARIA2C_METHOD_GET_SERVERS, ja, id);
}

void Aria2cInterface::tellActive(QStringList keys, QString id)
{
    QJsonArray ja;
    QJsonArray ka;
    foreach(QString k, keys) {
        ka.append(k);
    }
    ja.append(ka);
    callRPC(ARIA2C_METHOD_TELL_ACTIVE, ja, id);
}

void Aria2cInterface::tellWaiting(int offset, int num, QStringList keys, QString id)
{
    QJsonArray ja;
    ja.append(offset);
    ja.append(num);
    QJsonArray ka;
    foreach(QString k, keys) {
        ka.append(k);
    }
    ja.append(ka);
    callRPC(ARIA2C_METHOD_TELL_WAITING, ja, id);
}

void Aria2cInterface::tellStopped(int offset, int num, QStringList keys, QString id)
{
    QJsonArray ja;
    ja.append(offset);
    ja.append(num);
    QJsonArray ka;
    foreach(QString k, keys) {
        ka.append(k);
    }
    ja.append(ka);
    callRPC(ARIA2C_METHOD_TELL_STOPPED, ja, id);
}

void Aria2cInterface::changePosition(QString gId, int pos, QString how, QString id)
{
    QJsonArray ja;
    ja.append(gId);
    ja.append(pos);
    ja.append(how);
    callRPC(ARIA2C_METHOD_CHANGE_POSITION, ja, id);
}

void Aria2cInterface::changeUri(QString gId, int fileIndex, QStringList delUris, QStringList addUris, QString id)
{
    QJsonArray ja, du, au;
    ja.append(gId);
    ja.append(fileIndex);
    foreach(QString d, delUris) {
        du.append(d);
    }
    ja.append(du);
    foreach(QString a, addUris) {
        au.append(a);
    }
    ja.append(au);
    callRPC(ARIA2C_METHOD_CHANGE_URI, ja, id);

}

void Aria2cInterface::getOption(QString gId, QString id)
{
    QJsonArray ja;
    ja.append(gId);
    callRPC(ARIA2C_METHOD_GET_OPTION, ja, id);
}

void Aria2cInterface::changeOption(QString gId, QMap<QString, QVariant> options, QString id)
{
    QJsonArray ja;
    ja.append(gId);
    QJsonDocument doc = QJsonDocument::fromVariant(QVariant(options));
    QJsonObject optJson = doc.object();
    ja.append(optJson);
    callRPC(ARIA2C_METHOD_CHANGE_OPTION, ja, id);
}

void Aria2cInterface::getGlobalOption(QString id)
{
    callRPC(ARIA2C_METHOD_GET_GLOBAL_OPTION, id);
}

void Aria2cInterface::changeGlobalOption(QMap<QString, QVariant> options, QString id)
{
    QJsonArray ja;
    QJsonDocument doc = QJsonDocument::fromVariant(QVariant(options));
    QJsonObject optJson = doc.object();
    ja.append(optJson);
    callRPC(ARIA2C_METHOD_CHANGE_GLOBAL_OPTION, ja, id);
}

void Aria2cInterface::getGlobalStat(QString id)
{
    callRPC(ARIA2C_METHOD_GET_GLOBAL_STAT, id);
}

void Aria2cInterface::purgeDownloadResult(QString id)
{
    callRPC(ARIA2C_METHOD_PURGE_DOWNLOAD_RESULT, id);
}

void Aria2cInterface::removeDownloadResult(QString gId, QString id)
{
    QJsonArray ja;
    ja.append(gId);
    callRPC(ARIA2C_METHOD_REMOVE_DOWNLOAD_RESULT, ja, id);
}

void Aria2cInterface::getVersion(QString id)
{
    callRPC(ARIA2C_METHOD_GET_VERSION, id);
}

void Aria2cInterface::getSessionInfo(QString id)
{
    callRPC(ARIA2C_METHOD_GET_SESSION_INFO, id);
}

void Aria2cInterface::saveSession()
{
    callRPC(ARIA2C_METHOD_SAVE_SESSION);
}

void Aria2cInterface::multicall(QJsonArray params, QString id)
{
    callRPC(ARIA2C_METHOD_MULTICALL, params, id);
}

void Aria2cInterface::listMethods()
{
    callRPC(ARIA2C_METHOD_LIST_METHODS);
}

void Aria2cInterface::listNotifications()
{
    callRPC(ARIA2C_METHOD_LIST_NOTIFICATIONS);
}

