
/*
 * ===========================
 * VDK Visual Develeopment Kit
 * Version 2.0.0
 * March 2002
 * ===========================
 *
 * Copyright (C) 2002, Mario Motta
 * Developed by Mario Motta <mmotta@guest.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public 
 * License as published by the Free Software Foundation; either 
 * version 2 of the License, or (at your option) any later version. 
 * 
 * This library is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
 * Library General Public License for more details. 
 * 
 * You should have received a copy of the GNU Library General Public 
 * License along with this library; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA. 
 */ 
 
#ifndef  _VDKDRAWAREA_H 
#define  _VDKDRAWAREA_H  
 
#include <vdk/vdkobj.h> 
#include <vdk/colors.h> 
#include <vdk/rawpixbuf.h> 

class VDKDrawingArea;

/*!
  \class VDKDrawTool
  \brief This class and subclasses provid customized drawing tools
  for VDKDrawingArea widget
  */

class VDKDrawTool: public VDKNotCopyAble
{
  protected:
  VDKDrawingArea* owner;
  GdkPixmap *tile;
  public:
  /*!
  Constructor
  \arg owner VDKDrawingArea that owns the tool
  */
  VDKDrawTool (VDKDrawingArea* owner);
  /*!
  Destructor
  */
  virtual ~VDKDrawTool ();
  /*!
  Set how your drawing is filled,  following modes are supported:
  - GDK_SOLID draw with the foreground color
  - GDK_TILED draw with a tiled pixmap
  */
  VDKReadWriteValueProp <VDKDrawTool, GdkFill> Fill;
  /*!
    Determines how the bit values for the source pixels are combined with the bit values 
    for destination pixels to produce the final result. 
    Only a few of these values are usually useful: 
    for colored images, only GDK_COPY, GDK_XOR and GDK_INVERT are generally useful. 
    For bitmaps, GDK_AND and GDK_OR are also useful.
    Supported modes:
    GDK_COPY,
    GDK_INVERT,
    GDK_XOR,
    GDK_CLEAR,
    GDK_AND,
    GDK_AND_REVERSE,
    GDK_AND_INVERT,
    GDK_NOOP,
    GDK_OR,
    GDK_EQUIV,
    GDK_OR_REVERSE,
    GDK_COPY_INVERT,
    GDK_OR_INVERT,
    GDK_NAND,
    GDK_NOR,
    GDK_SET
   */
  VDKReadWriteValueProp <VDKDrawTool, GdkFunction> Function;

  void SetFill (GdkFill fill);
  /*!
  Sets tile pixmap,  valid only on GDK_TILED mode
  \arg tfile a file containing a pixmap
  */
  void SetTile (const char* tfile);
  /*!
  Sets tile pixmap,  valid only on GDK_TILED fill mode
  \arg tdata a string array of pixmap data
  */
  void SetTile (char** tdata);
  /*!
  Sets tile pixmap,  valid only on GDK_TILED fill mode
  \arg tile a valid GdkPixmap, cam be obtained using VDKDrawTool::Tile()
  */
  void SetTile (GdkPixmap* tile);
  /*!
    Returns a tile
  \arg tdata a string array of pixmap data
  */
  GdkPixmap *Tile(char** tdata);
  /*!
    Returns a tile
    \arg tfile a file containing a pixmap
  */
  GdkPixmap *Tile(const char* tfile);

  void SetFunction(GdkFunction f);
  };

/*!
  \class VDKPen
  \brief This class provides a Pen to customize drawing.
  */

class VDKPen: public VDKDrawTool
{
  public:
  /*!
  Constructor
  \arg owner VDKDrawingArea that owns the tool
  */

  VDKPen (VDKDrawingArea* owner);
    /*!
  Destructor
  */
  virtual ~VDKPen () {}
  /*!
    Sets/gets pen drawing color,  valid only on GDK_SOLID fill mode
  */
  VDKReadWriteValueProp <VDKPen, VDKRgb> Color;
  /*!
    Sets/gets pen thickness
  */
  VDKReadWriteValueProp <VDKPen, gint> Thickness;
  /*!
    Sets/gets pen line style,  supported modes:
  - GDK_LINE_SOLID
  
  - GDK_LINE_ON_OFF_DASH
  
  - GDK_LINE_DOUBLE_DASH
  */
  VDKReadWriteValueProp <VDKPen, GdkLineStyle> Style;
  /*!
    Sets/gets pen cap style,  supported modes:

  - GDK_CAP_NOT_LAST the same as GDK_CAP_BUTT for lines of non-zero width.
    for zero width lines, the final point on the line will not be drawn.
  
  - GDK_CAP_BUTT the ends of the lines are drawn squared off and extending 
  to the coordinates of the end point.
  
  - GDK_CAP_ROUND the ends of the lines are drawn as semicircles with the 
  diameter equal to the line width and centered at the end point.
  
  - GDK_CAP_PROJECTING the ends of the lines are drawn squared off and 
  extending half the width of the line beyond the end point.
  */
  VDKReadWriteValueProp <VDKPen, GdkCapStyle> Cap;
  /*!
    Sets/gets pen join style,  supported modes:
  - GDK_JOIN_MITER the sides of each line are extended to meet at an angle.
  
  - GDK_JOIN_ROUND the sides of the two lines are joined by a circular arc.
  
  - GDK_JOIN_BEVEL the sides of the two lines are joined by a straight line 
    which makes an equal angle with each line.
  */
  VDKReadWriteValueProp <VDKPen, GdkJoinStyle> Join;
//
  void SetColor (VDKRgb color);
  void SetThickness (int t);
  void SetStyle (GdkLineStyle style);
  void SetCap (GdkCapStyle cap);
  void SetJoin (GdkJoinStyle join);
};

/*!
  \class VDKBrush
  \brief This class provides a brush to customize drawing.
  Actually unused,  left for future improuvments
*/

class VDKBrush: public VDKDrawTool
{
  public:
  VDKBrush (VDKDrawingArea* owner): VDKDrawTool (owner){}
  virtual ~VDKBrush () {}
};

/*!
  \class VDKDrawingArea
  \brief This class provides a drawing area widget. 
  This widget should substitute VDKCanvas which is now deprecated but
  will continue to be supported even if not enhanced more.
  However VDKCanvas is not really deprecated, simply VDKDrawingArea is a 
  better widget,  much powerfull and easy to use.
  */
class VDKDrawingArea: public VDKObject 
{ 
 private:
 static void RealizeEvent (GtkWidget* wid,  gpointer gp );
  
 protected:
 GdkGC*   _gc;
 public:
  /*!
    The pen drawing tool
  */ 
  VDKPen* Pen; 
  /*!
    The brush drawing tool (actually unused)
  */
  VDKBrush *Brush; 
  /*!
  Constructor
  \arg owner VDKForm
  */
  VDKDrawingArea(VDKForm* owner); 
  /*!
  Destructor
  */
  virtual ~VDKDrawingArea();
  /*!
  Clears canvas area
  */
  void Clear(); 

  void DrawString(int x, int y, char* text); 
  void DrawText(int x, int y, char* text, int n); 
  void DrawPoint(int x, int y); 
  void DrawLine(int x, int y, int x1, int y1); 
  void DrawRect(gint filled, int x, int y, int w, int h); 
  void DrawArc(int filled,int x,int y, int width,  
	       int height,int angle1, int angle2); 
  void DrawPolygon(int filled,GdkPoint *points, gint npoints); 
  void DrawPoints(GdkPoint *points, int npoints); 
  void DrawSegments(GdkSegment *segs, int nsegs); 
  void DrawLines(GdkPoint *points, int npoints); 
  void DrawPixbuf(int x, int y, const char *pixfile);
  void DrawPixbuf(int x, int y, VDKPixbuf* pix); 
  void DrawPixbuf(int x, int y, const char ** data); 

  void SetForeground(VDKRgb color, GtkStateType state = GTK_STATE_NORMAL);
  void SetBackground(VDKRgb color, GtkStateType state = GTK_STATE_NORMAL);
  GdkGC* GC () { return _gc; }
  GdkDrawable* Drawable () { return widget->window; }
  virtual void SetFont(VDKFont* f);
}; 
 
#endif 



