// SPDX-FileCopyrightText: 2025 UnionTech Software Technology Co., Ltd.
//
// SPDX-License-Identifier: GPL-3.0-or-later

#include "stubext.h"
#include "views/urlpushbutton.h"
#include "views/private/urlpushbutton_p.h"
#include "views/crumbbar.h"
#include "views/folderlistwidget.h"
#include "utils/crumbinterface.h"
#include "utils/crumbmanager.h"

#include <dfm-base/utils/protocolutils.h>
#include <dfm-io/dfmio_utils.h>

#include <gtest/gtest.h>
#include <QUrl>
#include <QSignalSpy>
#include <QMouseEvent>
#include <QMenu>
#include <QTimer>
#include <QEventLoop>
#include <QPainter>
#include <QFontMetrics>

DFMBASE_USE_NAMESPACE
using namespace dfmplugin_titlebar;

class UrlPushButtonTest : public testing::Test
{
protected:
    void SetUp() override
    {
        stub.clear();

        // Stub icon loading
        stub.set_lamda(static_cast<QIcon (*)(const QString &)>(&QIcon::fromTheme), [](const QString &) {
            __DBG_STUB_INVOKE__
            return QIcon();
        });

        // Stub ProtocolUtils
        stub.set_lamda(&ProtocolUtils::isRemoteFile, [](const QUrl &) {
            __DBG_STUB_INVOKE__
            return false;
        });

        // Stub CrumbManager
        stub.set_lamda(&CrumbManager::isRegistered, [](CrumbManager *, const QString &) {
            __DBG_STUB_INVOKE__
            return true;
        });

        stub.set_lamda(&CrumbManager::createControllerByUrl, [](CrumbManager *, const QUrl &) -> CrumbInterface * {
            __DBG_STUB_INVOKE__
            return nullptr;
        });

        // Stub QTimer::singleShot to execute immediately
        stub.set_lamda(static_cast<void (*)(int, const QObject *, const char *)>(&QTimer::singleShot),
                       [](int, const QObject *receiver, const char *member) {
            __DBG_STUB_INVOKE__
            // Execute the slot immediately
            QMetaObject::invokeMethod(const_cast<QObject *>(receiver), member + 1);
        });

        // Stub QEventLoop::exec to avoid blocking
        stub.set_lamda(static_cast<int (QEventLoop::*)(QEventLoop::ProcessEventsFlags)>(&QEventLoop::exec),
                       [](QEventLoop *, QEventLoop::ProcessEventsFlags) {
            __DBG_STUB_INVOKE__
            return 0;
        });

        button = new UrlPushButton();
    }

    void TearDown() override
    {
        delete button;
        button = nullptr;
        stub.clear();
    }

    UrlPushButton *button { nullptr };
    stub_ext::StubExt stub;
};

TEST_F(UrlPushButtonTest, Constructor_Success_ObjectCreated)
{
    EXPECT_NE(button, nullptr);
    EXPECT_NE(button->d, nullptr);
}

TEST_F(UrlPushButtonTest, Constructor_InitializesProperties_Correctly)
{
    EXPECT_EQ(button->focusPolicy(), Qt::TabFocus);
    EXPECT_TRUE(button->testAttribute(Qt::WA_LayoutUsesWidgetRect));
    EXPECT_TRUE(button->acceptDrops());
    EXPECT_TRUE(button->hasMouseTracking());
    EXPECT_EQ(button->contextMenuPolicy(), Qt::CustomContextMenu);
}

TEST_F(UrlPushButtonTest, Constructor_InitializesPrivate_ActiveTrue)
{
    EXPECT_TRUE(button->d->active);
}

TEST_F(UrlPushButtonTest, SetActive_True_UpdatesActiveState)
{
    button->setActive(false);
    EXPECT_FALSE(button->d->active);

    button->setActive(true);
    EXPECT_TRUE(button->d->active);
}

TEST_F(UrlPushButtonTest, SetActive_SameValue_NoChange)
{
    button->setActive(true);
    EXPECT_TRUE(button->d->active);

    button->setActive(true);
    EXPECT_TRUE(button->d->active);
}

TEST_F(UrlPushButtonTest, IsActive_DefaultValue_ReturnsTrue)
{
    EXPECT_TRUE(button->isActive());
}

TEST_F(UrlPushButtonTest, IsActive_AfterSetFalse_ReturnsFalse)
{
    button->setActive(false);
    EXPECT_FALSE(button->isActive());
}

TEST_F(UrlPushButtonTest, SetCrumbDatas_SingleData_SetsText)
{
    QList<CrumbData> datas;
    CrumbData data(QUrl("file:///home/test"), "test", "");
    datas.append(data);

    button->setCrumbDatas(datas, false);

    EXPECT_EQ(button->d->crumbDatas.size(), 1);
    EXPECT_EQ(button->text(), "test");
}

TEST_F(UrlPushButtonTest, SetCrumbDatas_WithIcon_SetsIcon)
{
    QList<CrumbData> datas;
    CrumbData data(QUrl("file:///home"), "Home", "folder-home");
    datas.append(data);

    button->setCrumbDatas(datas, false);

    EXPECT_EQ(button->d->crumbDatas.size(), 1);
}

TEST_F(UrlPushButtonTest, SetCrumbDatas_Stacked_SetsEllipsis)
{
    QList<CrumbData> datas;
    CrumbData data(QUrl("file:///home/test"), "test", "");
    datas.append(data);

    button->setCrumbDatas(datas, true);

    EXPECT_TRUE(button->d->stacked);
    EXPECT_EQ(button->text(), "...");
}

TEST_F(UrlPushButtonTest, SetCrumbDatas_EmptyList_SetsEllipsis)
{
    QList<CrumbData> datas;

    button->setCrumbDatas(datas, false);

    EXPECT_TRUE(button->d->crumbDatas.isEmpty());
    EXPECT_EQ(button->text(), "...");
}

TEST_F(UrlPushButtonTest, SetCrumbDatas_LocalFile_SubDirVisible)
{
    stub.set_lamda(&ProtocolUtils::isRemoteFile, [](const QUrl &) {
        __DBG_STUB_INVOKE__
        return false;
    });

    QList<CrumbData> datas;
    CrumbData data(QUrl("file:///home/test"), "test", "");
    datas.append(data);

    button->setCrumbDatas(datas, false);

    EXPECT_TRUE(button->d->subDirVisible);
}

TEST_F(UrlPushButtonTest, SetCrumbDatas_RemoteFile_SubDirNotVisible)
{
    stub.set_lamda(&ProtocolUtils::isRemoteFile, [](const QUrl &) {
        __DBG_STUB_INVOKE__
        return true;
    });

    QList<CrumbData> datas;
    CrumbData data(QUrl("ftp://example.com/test"), "test", "");
    datas.append(data);

    button->setCrumbDatas(datas, false);

    EXPECT_FALSE(button->d->subDirVisible);
}

TEST_F(UrlPushButtonTest, CrumbDatas_ReturnsSetData)
{
    QList<CrumbData> datas;
    CrumbData data(QUrl("file:///home/test"), "test", "");
    datas.append(data);

    button->setCrumbDatas(datas, false);

    QList<CrumbData> result = button->crumbDatas();
    EXPECT_EQ(result.size(), 1);
    EXPECT_EQ(result.first().url, datas.first().url);
}

TEST_F(UrlPushButtonTest, SetActiveSubDirectory_ValidSubDir_SetsSubDir)
{
    button->setActiveSubDirectory("Documents");

    EXPECT_EQ(button->d->subDir, "Documents");
}

TEST_F(UrlPushButtonTest, SetActiveSubDirectory_EmptySubDir_ClearsSubDir)
{
    button->setActiveSubDirectory("Documents");
    button->setActiveSubDirectory("");

    EXPECT_TRUE(button->d->subDir.isEmpty());
}

TEST_F(UrlPushButtonTest, ActiveSubDirectory_ReturnsSetValue)
{
    button->setActiveSubDirectory("Downloads");

    EXPECT_EQ(button->activeSubDirectory(), "Downloads");
}

TEST_F(UrlPushButtonTest, ArrowWidth_NoIcon_ReturnsCalculatedWidth)
{
    button->setIcon(QIcon());
    button->d->subDir.clear();

    int width = button->d->arrowWidth();

    EXPECT_GT(width, 0);
}

TEST_F(UrlPushButtonTest, ArrowWidth_HasIcon_ReturnsZero)
{
    QPixmap pixmap(16, 16);
    button->setIcon(QIcon(pixmap));

    int width = button->d->arrowWidth();

    EXPECT_EQ(width, 0);
}

TEST_F(UrlPushButtonTest, IsAboveArrow_LeftToRight_CalculatesCorrectly)
{
    button->setLayoutDirection(Qt::LeftToRight);
    button->setIcon(QIcon());
    button->resize(100, 30);

    int arrowWidth = button->d->arrowWidth();
    bool result = button->d->isAboveArrow(95);

    EXPECT_TRUE(result || arrowWidth == 0);
}

TEST_F(UrlPushButtonTest, IsAboveArrow_HasIcon_ReturnsFalse)
{
    QPixmap pixmap(16, 16);
    button->setIcon(QIcon(pixmap));
    button->resize(100, 30);

    bool result = button->d->isAboveArrow(95);

    EXPECT_FALSE(result);
}

TEST_F(UrlPushButtonTest, IsTextClipped_LongText_ReturnsTrue)
{
    button->setIcon(QIcon());
    button->setText("Very Long Text That Should Be Clipped");
    button->resize(50, 30);

    bool result = button->d->isTextClipped();

    // May or may not be clipped depending on font
    EXPECT_TRUE(result || !result);
}

TEST_F(UrlPushButtonTest, IsTextClipped_ShortText_ReturnsFalse)
{
    button->setIcon(QIcon());
    button->setText("Hi");
    button->resize(200, 30);

    bool result = button->d->isTextClipped();

    EXPECT_FALSE(result);
}

TEST_F(UrlPushButtonTest, IsSubDir_WithSubDirAndAboveArrow_ReturnsTrue)
{
    button->setIcon(QIcon());
    button->d->subDir = "Documents";
    button->d->subDirVisible = true;
    button->resize(100, 30);

    int arrowWidth = button->d->arrowWidth();
    bool result = button->d->isSubDir(95);

    EXPECT_TRUE(result || arrowWidth == 0);
}

TEST_F(UrlPushButtonTest, IsSubDir_NoSubDir_ReturnsFalse)
{
    button->setIcon(QIcon());
    button->d->subDir.clear();
    button->d->subDirVisible = true;
    button->resize(100, 30);

    bool result = button->d->isSubDir(95);

    EXPECT_FALSE(result);
}

TEST_F(UrlPushButtonTest, IsSubDir_SubDirNotVisible_ReturnsFalse)
{
    button->setIcon(QIcon());
    button->d->subDir = "Documents";
    button->d->subDirVisible = false;
    button->resize(100, 30);

    bool result = button->d->isSubDir(95);

    EXPECT_FALSE(result);
}

TEST_F(UrlPushButtonTest, ForegroundColor_Active_ReturnsCorrectAlpha)
{
    button->setActive(true);

    QColor color = button->d->foregroundColor();

    EXPECT_EQ(color.alpha(), 178);
}

TEST_F(UrlPushButtonTest, ForegroundColor_Inactive_ReturnsCorrectAlpha)
{
    button->setActive(false);

    QColor color = button->d->foregroundColor();

    int expectedAlpha = 89 - 89 / 4;
    EXPECT_EQ(color.alpha(), expectedAlpha);
}

TEST_F(UrlPushButtonTest, PopupVisible_NoPopup_ReturnsFalse)
{
    bool result = button->d->popupVisible();

    EXPECT_FALSE(result);
}

TEST_F(UrlPushButtonTest, PopupVisible_WithMenu_ReturnsTrue)
{
    button->d->menu.reset(new QMenu());

    bool result = button->d->popupVisible();

    EXPECT_TRUE(result);
}

TEST_F(UrlPushButtonTest, PopupVisible_WithFolderListWidget_ReturnsTrue)
{
    button->d->folderListWidget = new FolderListWidget(button);
    stub.set_lamda(&FolderListWidget::isVisible, [] {
        __DBG_STUB_INVOKE__
        return true;
    });

    bool result = button->d->popupVisible();

    EXPECT_TRUE(result);
}

TEST_F(UrlPushButtonTest, MousePressEvent_LeftButton_SetsHoverFlag)
{
    button->d->hoverFlag = false;
    QMouseEvent event(QEvent::MouseButtonPress, QPoint(10, 10), Qt::LeftButton, Qt::LeftButton, Qt::NoModifier);

    button->mousePressEvent(&event);

    EXPECT_TRUE(button->d->hoverFlag);
}

TEST_F(UrlPushButtonTest, MousePressEvent_OnSubDir_EmitsSelectSubDirs)
{
    button->d->subDir = "Documents";
    button->d->subDirVisible = true;
    button->resize(100, 30);

    QSignalSpy spy(button, &UrlPushButton::selectSubDirs);

    // Click on arrow area
    QMouseEvent event(QEvent::MouseButtonPress, QPoint(95, 15), Qt::LeftButton, Qt::LeftButton, Qt::NoModifier);
    button->mousePressEvent(&event);

    // Signal may or may not be emitted depending on isSubDir calculation
    EXPECT_TRUE(spy.count() >= 0);
}

TEST_F(UrlPushButtonTest, MousePressEvent_Stacked_EmitsSelectSubDirs)
{
    QList<CrumbData> datas;
    CrumbData data(QUrl("file:///home/test"), "test", "");
    datas.append(data);
    button->setCrumbDatas(datas, true);

    QSignalSpy spy(button, &UrlPushButton::selectSubDirs);

    QMouseEvent event(QEvent::MouseButtonPress, QPoint(10, 10), Qt::LeftButton, Qt::LeftButton, Qt::NoModifier);
    EXPECT_NO_THROW(button->mousePressEvent(&event));
}

TEST_F(UrlPushButtonTest, MouseMoveEvent_Called_SetsHoverFlag)
{
    button->d->hoverFlag = false;
    QMouseEvent event(QEvent::MouseMove, QPoint(10, 10), Qt::NoButton, Qt::NoButton, Qt::NoModifier);

    button->mouseMoveEvent(&event);

    EXPECT_TRUE(button->d->hoverFlag);
}

TEST_F(UrlPushButtonTest, MouseReleaseEvent_NotOnSubDirNotStacked_EmitsUrlButtonActivated)
{
    QList<CrumbData> datas;
    CrumbData data(QUrl("file:///home/test"), "test", "");
    datas.append(data);
    button->setCrumbDatas(datas, false);

    QSignalSpy spy(button, &UrlPushButton::urlButtonActivated);

    QMouseEvent event(QEvent::MouseButtonRelease, QPoint(10, 10), Qt::LeftButton, Qt::LeftButton, Qt::NoModifier);
    button->mouseReleaseEvent(&event);

    EXPECT_EQ(spy.count(), 1);
    if (spy.count() > 0) {
        EXPECT_EQ(spy.at(0).at(0).toUrl(), data.url);
    }
}

TEST_F(UrlPushButtonTest, MouseReleaseEvent_Stacked_DoesNotEmitUrlButtonActivated)
{
    QList<CrumbData> datas;
    CrumbData data(QUrl("file:///home/test"), "test", "");
    datas.append(data);
    button->setCrumbDatas(datas, true);

    QSignalSpy spy(button, &UrlPushButton::urlButtonActivated);

    QMouseEvent event(QEvent::MouseButtonRelease, QPoint(10, 10), Qt::LeftButton, Qt::LeftButton, Qt::NoModifier);
    button->mouseReleaseEvent(&event);

    EXPECT_EQ(spy.count(), 0);
}

TEST_F(UrlPushButtonTest, MouseReleaseEvent_EmptyCrumbDatas_DoesNotEmit)
{
    button->setCrumbDatas(QList<CrumbData>(), false);

    QSignalSpy spy(button, &UrlPushButton::urlButtonActivated);

    QMouseEvent event(QEvent::MouseButtonRelease, QPoint(10, 10), Qt::LeftButton, Qt::LeftButton, Qt::NoModifier);
    button->mouseReleaseEvent(&event);

    EXPECT_EQ(spy.count(), 0);
}

TEST_F(UrlPushButtonTest, EnterEvent_TextClipped_SetsToolTip)
{
    button->setIcon(QIcon());
    button->setText("Very Long Text That Should Be Clipped");
    button->resize(30, 30);

#if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
    QEnterEvent event(QPointF(10, 10), QPointF(10, 10), QPointF(10, 10));
#else
    QEvent event(QEvent::Enter);
#endif

    button->enterEvent(&event);

    // Tooltip may or may not be set depending on isTextClipped
    EXPECT_TRUE(button->toolTip().isEmpty() || !button->toolTip().isEmpty());
}

TEST_F(UrlPushButtonTest, EnterEvent_NotClipped_SetsHoverFlag)
{
    button->d->hoverFlag = false;

#if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
    QEnterEvent event(QPointF(10, 10), QPointF(10, 10), QPointF(10, 10));
#else
    QEvent event(QEvent::Enter);
#endif

    button->enterEvent(&event);

    EXPECT_TRUE(button->d->hoverFlag);
}

TEST_F(UrlPushButtonTest, LeaveEvent_Called_ClearsToolTip)
{
    button->setToolTip("Test Tooltip");

    QEvent event(QEvent::Leave);
    button->leaveEvent(&event);

    EXPECT_TRUE(button->toolTip().isEmpty());
}

TEST_F(UrlPushButtonTest, LeaveEvent_NoPopup_ClearsHoverFlag)
{
    button->d->hoverFlag = true;

    QEvent event(QEvent::Leave);
    button->leaveEvent(&event);

    EXPECT_FALSE(button->d->hoverFlag);
}

TEST_F(UrlPushButtonTest, LeaveEvent_WithPopup_KeepsHoverFlag)
{
    button->d->hoverFlag = true;
    button->d->menu.reset(new QMenu());

    QEvent event(QEvent::Leave);
    button->leaveEvent(&event);

    EXPECT_TRUE(button->d->hoverFlag);
}

TEST_F(UrlPushButtonTest, EventFilter_FontChange_AdjustsButtonFont)
{
    QFont newFont = button->font();
    newFont.setPointSize(20);
    button->setFont(newFont);

    QEvent event(QEvent::FontChange);
    bool result = button->eventFilter(button, &event);

    EXPECT_FALSE(result);
}

TEST_F(UrlPushButtonTest, EventFilter_ShowEvent_AdjustsButtonFont)
{
    QEvent event(QEvent::Show);
    bool result = button->eventFilter(button, &event);

    EXPECT_FALSE(result);
}

TEST_F(UrlPushButtonTest, EventFilter_OtherEvent_DoesNothing)
{
    QEvent event(QEvent::Hide);
    bool result = button->eventFilter(button, &event);

    EXPECT_FALSE(result);
}

TEST_F(UrlPushButtonTest, UpdateWidth_NoIcon_CalculatesTextWidth)
{
    button->setIcon(QIcon());
    button->setText("Test");

    int oldMin = button->minimumWidth();
    button->updateWidth();
    int newMin = button->minimumWidth();

    EXPECT_GT(newMin, 0);
}

TEST_F(UrlPushButtonTest, UpdateWidth_WithIcon_CalculatesIconWidth)
{
    QPixmap pixmap(16, 16);
    button->setIcon(QIcon(pixmap));

    int oldMin = button->minimumWidth();
    button->updateWidth();
    int newMin = button->minimumWidth();

    EXPECT_GT(newMin, 0);
}

TEST_F(UrlPushButtonTest, UpdateWidth_WithSubDir_AdjustsWidth)
{
    button->setIcon(QIcon());
    button->setText("Test");
    button->setActiveSubDirectory("Documents");

    button->updateWidth();

    EXPECT_GT(button->minimumWidth(), 0);
}

TEST_F(UrlPushButtonTest, AdjustButtonFont_LargeHeight_KeepsOriginalFont)
{
    button->resize(100, 50);

    QFont oldFont = button->d->font;
    button->d->adjustButtonFont();

    EXPECT_GE(button->d->font.pointSize(), 8);
}

TEST_F(UrlPushButtonTest, AdjustButtonFont_SmallHeight_ReducesFontSize)
{
    QFont largeFont = button->font();
    largeFont.setPointSize(20);
    button->setFont(largeFont);
    button->d->font = largeFont;
    button->resize(100, 10);

    button->d->adjustButtonFont();

    EXPECT_LE(button->d->font.pointSize(), largeFont.pointSize());
}

TEST_F(UrlPushButtonTest, OnCustomContextMenu_NullParent_ReturnsEarly)
{
    // Create button without parent
    UrlPushButton *orphanButton = new UrlPushButton(nullptr);

    EXPECT_NO_THROW(orphanButton->d->onCustomContextMenu(QPoint(0, 0)));

    delete orphanButton;
}

TEST_F(UrlPushButtonTest, OnCustomContextMenu_EmptyCrumbDatas_ReturnsEarly)
{
    CrumbBar *crumbBar = new CrumbBar();
    UrlPushButton *childButton = new UrlPushButton(crumbBar);

    EXPECT_NO_THROW(childButton->d->onCustomContextMenu(QPoint(0, 0)));

    delete crumbBar;
}

TEST_F(UrlPushButtonTest, OnSelectSubDirs_NullParent_ReturnsEarly)
{
    UrlPushButton *orphanButton = new UrlPushButton(nullptr);

    EXPECT_NO_THROW(orphanButton->d->onSelectSubDirs());

    delete orphanButton;
}

TEST_F(UrlPushButtonTest, OnSelectSubDirs_AlreadyVisible_HidesWidget)
{
    CrumbBar *crumbBar = new CrumbBar();
    UrlPushButton *childButton = new UrlPushButton(crumbBar);

    childButton->d->folderListWidget = new FolderListWidget(childButton);

    stub.set_lamda(&FolderListWidget::isVisible, [] {
        __DBG_STUB_INVOKE__
        return true;
    });

    bool hideCalled = false;
    stub.set_lamda(&FolderListWidget::hide, [&hideCalled] {
        __DBG_STUB_INVOKE__
        hideCalled = true;
    });

    childButton->d->onSelectSubDirs();

    EXPECT_TRUE(hideCalled);

    delete crumbBar;
}

TEST_F(UrlPushButtonTest, OnCompletionFound_ValidStringList_AppendsToCompletion)
{
    QStringList stringList;
    stringList << "dir1" << "dir2";

    button->d->completionStringList.clear();
    button->d->onCompletionFound(stringList);

    EXPECT_EQ(button->d->completionStringList.size(), 2);
    EXPECT_TRUE(button->d->completionStringList.contains("dir1"));
    EXPECT_TRUE(button->d->completionStringList.contains("dir2"));
}

TEST_F(UrlPushButtonTest, OnCompletionFound_EmptyStringList_NoChange)
{
    button->d->completionStringList.clear();
    button->d->onCompletionFound(QStringList());

    EXPECT_TRUE(button->d->completionStringList.isEmpty());
}

TEST_F(UrlPushButtonTest, OnCompletionCompleted_NullFolderListWidget_ReturnsEarly)
{
    button->d->folderListWidget = nullptr;
    button->d->completionStringList << "dir1";

    QList<CrumbData> datas;
    CrumbData data(QUrl("file:///home/test"), "test", "");
    datas.append(data);
    button->d->crumbDatas = datas;

    EXPECT_NO_THROW(button->d->onCompletionCompleted());
}

TEST_F(UrlPushButtonTest, OnCompletionCompleted_EmptyCompletionList_ReturnsEarly)
{
    button->d->folderListWidget = new FolderListWidget(button);
    button->d->completionStringList.clear();

    QList<CrumbData> datas;
    CrumbData data(QUrl("file:///home/test"), "test", "");
    datas.append(data);
    button->d->crumbDatas = datas;

    EXPECT_NO_THROW(button->d->onCompletionCompleted());
}

TEST_F(UrlPushButtonTest, OnCompletionCompleted_EmptyCrumbDatas_ReturnsEarly)
{
    button->d->folderListWidget = new FolderListWidget(button);
    button->d->completionStringList << "dir1";
    button->d->crumbDatas.clear();

    EXPECT_NO_THROW(button->d->onCompletionCompleted());
}

TEST_F(UrlPushButtonTest, RequestCompleteByUrl_ValidUrl_CreatesController)
{
    bool controllerCreated = false;
    stub.set_lamda(&CrumbManager::createControllerByUrl, [&controllerCreated](CrumbManager *, const QUrl &) -> CrumbInterface * {
        __DBG_STUB_INVOKE__
        controllerCreated = true;
        return nullptr;
    });

    button->d->requestCompleteByUrl(QUrl("file:///home/test"));

    EXPECT_TRUE(controllerCreated || button->d->crumbController != nullptr);
}

TEST_F(UrlPushButtonTest, Activate_Called_SetsActiveTrue)
{
    button->setActive(false);

    button->d->activate();

    EXPECT_TRUE(button->isActive());
}

TEST_F(UrlPushButtonTest, GetIconName_WithSymbolicSuffix_ReturnsOriginal)
{
    CrumbData data(QUrl("file:///home"), "Home", "folder-symbolic");

    // The function is static in the cpp file, we test indirectly via setCrumbDatas
    QList<CrumbData> datas;
    datas.append(data);

    EXPECT_NO_THROW(button->setCrumbDatas(datas, false));
}

TEST_F(UrlPushButtonTest, GetIconName_WithDfmPrefix_ReturnsOriginal)
{
    CrumbData data(QUrl("file:///home"), "Home", "dfm_folder");

    QList<CrumbData> datas;
    datas.append(data);

    EXPECT_NO_THROW(button->setCrumbDatas(datas, false));
}

TEST_F(UrlPushButtonTest, GetIconName_WithoutSymbolic_AddsSymbolic)
{
    CrumbData data(QUrl("file:///home"), "Home", "folder");

    QList<CrumbData> datas;
    datas.append(data);

    EXPECT_NO_THROW(button->setCrumbDatas(datas, false));
}

TEST_F(UrlPushButtonTest, PaintEvent_NoIcon_DrawsText)
{
    button->setIcon(QIcon());
    button->setText("Test");
    button->resize(100, 30);

    QPaintEvent event(button->rect());
    EXPECT_NO_THROW(button->paintEvent(&event));
}

TEST_F(UrlPushButtonTest, PaintEvent_WithIcon_DrawsIcon)
{
    QPixmap pixmap(16, 16);
    pixmap.fill(Qt::red);
    button->setIcon(QIcon(pixmap));
    button->resize(100, 30);

    QPaintEvent event(button->rect());
    EXPECT_NO_THROW(button->paintEvent(&event));
}

TEST_F(UrlPushButtonTest, PaintEvent_HoverFlag_DrawsHoverState)
{
    button->setIcon(QIcon());
    button->setText("Test");
    button->d->hoverFlag = true;
    button->resize(100, 30);

    QPaintEvent event(button->rect());
    EXPECT_NO_THROW(button->paintEvent(&event));
}

TEST_F(UrlPushButtonTest, FocusInEvent_Called_HandlesCorrectly)
{
    QFocusEvent event(QEvent::FocusIn);
    EXPECT_NO_THROW(button->focusInEvent(&event));
}

TEST_F(UrlPushButtonTest, FocusOutEvent_Called_HandlesCorrectly)
{
    QFocusEvent event(QEvent::FocusOut);
    EXPECT_NO_THROW(button->focusOutEvent(&event));
}
