#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <dirent.h>
#include <sys/types.h>
#include <vector>
#include <syslog.h>
#include <cstdlib>
#include <regex>
#include "boost/algorithm/string.hpp"
#include "cachedata.h"
#include "logger.h"
#include "common.h"

using std::chrono::system_clock;

static const uint32_t EVENT_STEP = 100;
static const uint64_t MAX_SIZE = 1024 * 1024 * 2;  //2M
static const uint MAX_FILENUM = 24;
static const uint MIN_FILENUM = 7;
static const long ARCHIVE_SIZE = 1024 * 50;  //50K

CacheData::CacheData(): m_archivetimer(m_io), m_scantimer(m_io), m_quit(false), m_cachedir(LOG_PATH), m_fileindex(1)
{
    Logger::get_mutable_instance().Print("cache data this:%p", this);
}

CacheData::~CacheData()
{
    m_io.stop();
    m_quit = true;
    m_cond.notify_all();
    if(m_thrd.joinable())
    {
        m_thrd.join();
    }
}

void CacheData::Init()
{
    // 获取 xxx.log 日志文件
    m_logfile = FindLogFile();

    uint64_t totalsize = 0;
    ScanMap mapscan;

    // 扫描日志目录（保存文件名和大小）
    ScanLogDir(totalsize, mapscan);

    // 删除旧文件
    RemoveOldFiles(totalsize, mapscan);

    // 获取事件码文件和事件码
    LoadEventCode();

    // 更新事件码（+100）
    m_nextevcode = m_eventcode + EVENT_STEP;
    UpdateEventCode();

    // 获取 .bk 文件中索引最大的值，并计算下一个文件的索引
    if(!mapscan.empty())
    {
        uint64_t maxnum = 0;
        for (auto &file: mapscan)
        {
            if(boost::ends_with(file.first, ".bk"))
            {
                string::size_type beginpos = file.first.rfind("_");
                string::size_type endpos = file.first.rfind(".");
                string strindex = file.first.substr(beginpos + 1, endpos - beginpos - 1);
                uint64_t currindex = strtoul(strindex.c_str(), nullptr, 10);
                maxnum = std::max(maxnum, currindex);
            }
        }

        m_fileindex = maxnum + 1;
    }

    bool bCreatenew = true;
    if(!m_logfile.empty())
    {
        Logger::get_mutable_instance().Print("find old log:%s", m_logfile.c_str());
        struct stat fileinfo;
        stat(m_logfile.c_str(), &fileinfo);

        // 计算日志文件中的时间（event_1676445975.log 中的 1676445975）
        string::size_type beginpos = m_logfile.rfind("_");
        string::size_type endpos = m_logfile.rfind(".");
        string strtime = m_logfile.substr(beginpos + 1, endpos - beginpos - 1);
        system_clock::time_point ctime = system_clock::from_time_t(strtol(strtime.c_str(), nullptr, 10));
        // 存档超过一小时且文件 > 6 个字节，或者文件 >= 50K
        if(((system_clock::now() - ctime) > std::chrono::hours(ARCHIVE_PERIOD) &&
            fileinfo.st_size > 6) ||
                fileinfo.st_size >= ARCHIVE_SIZE)
        {
            // 移动日志文件 .log -> .bk
            MoveLogFile();
        }
        else
        {
            Logger::get_mutable_instance().Print("open old log file");
            bCreatenew = false;
            m_ofscache.open(m_logfile.c_str(), ios::app | ios::out | ios::binary);
            // 异步定时器，超时后添加一条任务（类型：归档）
            m_archivetimer.expires_after((ctime + std::chrono::hours(ARCHIVE_PERIOD)) - system_clock::now());
            m_archivetimer.async_wait(std::bind(&CacheData::OnArchiveHandle, this, m_logfile));
        }
    }

    if(bCreatenew)
    {
        // 创建新日志
        CreateNewLog();
        m_archivetimer.expires_after(std::chrono::hours(ARCHIVE_PERIOD));
        m_archivetimer.async_wait(std::bind(&CacheData::OnArchiveHandle, this, m_logfile));
    }

    // 10 分钟后，扫描日志目录，并删除旧文件
    m_scantimer.expires_after(std::chrono::minutes(SCAN_PERIOD));
    m_scantimer.async_wait(std::bind(&CacheData::OnScanHandle, this, SCAN_PERIOD));
    std::thread([this]{this->m_io.run();}).detach();
    m_thrd = std::thread(&CacheData::LoopWriteData, this);
}

void CacheData::AddData(const string &package, const string &strdata)
{
    json evdata;
    evdata["package"] = package;
    evdata["serial"] = GetNextEventCode();
    evdata["data"] = strdata;

    AddTaskInfo(WRITE_CACHE, evdata.dump());
}

void CacheData::AddTaskInfo(TASKTYPE type, const string &data)
{
    std::unique_lock<std::mutex> lock(m_mutex);
    if(m_taskqueue.size() < MAX_QUEUESIZE)
    {
        m_taskqueue.emplace_back(type, data);
        m_cond.notify_all();
    }
}

string CacheData::FindLogFile()
{
    // 打开日志目录 /var/lib/deepin/event-log
    string strevlog = "";
    DIR *pdir = opendir(m_cachedir.c_str());
    if(pdir == nullptr)
    {
        Logger::get_mutable_instance().Print("open cache dir failed");
        syslog(LOG_ERR, "open cache dir failed");
        return strevlog;
    }

    // 获取日志文件 xxx.log
    struct dirent *pinfo = nullptr;
    while ((pinfo = readdir(pdir)) != nullptr)
    {
        if(!strcmp(pinfo->d_name, ".") ||
                !strcmp(pinfo->d_name, ".."))
        {
            continue;
        }

        if(pinfo->d_type == DT_REG)
        {
            string filename = m_cachedir;
            filename.append(pinfo->d_name);
            if(boost::ends_with(filename, ".log"))
            {
                strevlog = filename;
                break;
            }
        }
    }

    // 关闭目录
    closedir(pdir);

    return strevlog;
}

void CacheData::CreateNewLog()
{
    unsigned short major = MAJOR_VERSION;
    unsigned short minor = MINOR_VERSION;
    unsigned short micro = MICRO_VERSION;

    m_logfile = m_cachedir;
    m_logfile.append(LOG_NAME);
    m_logfile.append("_");
    m_logfile.append(std::to_string(system_clock::to_time_t(system_clock::now())));
    m_logfile.append(".log");
    m_ofscache.open(m_logfile.c_str(), ios::out | ios::trunc | ios::binary);
    if(m_ofscache.is_open())
    {
        m_ofscache.write((char*)&major, sizeof (major));
        m_ofscache.write((char*)&minor, sizeof (minor));
        m_ofscache.write((char*)&micro, sizeof (micro));
        m_ofscache.flush();
    }
}

void CacheData::ScanLogDir(uint64_t &totalsize, ScanMap &mapscan)
{
    // 打开日志目录
    DIR *pdir = opendir(m_cachedir.c_str());
    if(pdir == nullptr)
    {
        Logger::get_mutable_instance().Print("open cache dir failed");
        syslog(LOG_ERR, "open cache dir failed");
        return;
    }

    // 保存文件及其大小
    struct dirent *pinfo = nullptr;
    while ((pinfo = readdir(pdir)) != nullptr)
    {
        if(!strcmp(pinfo->d_name, ".") ||
                !strcmp(pinfo->d_name, ".."))
        {
            continue;
        }

        if(pinfo->d_type == DT_REG)
        {
            struct stat fileinfo;
            string filename = m_cachedir;
            filename.append(pinfo->d_name);
            stat(filename.c_str(), &fileinfo);

            totalsize += fileinfo.st_size;
            mapscan.insert(std::make_pair(filename, fileinfo.st_size));
        }
    }

    // 关闭目录
    closedir(pdir);
}

void CacheData::ArchiveLog(string strlog)
{
    system_clock::duration extime;
    if(m_logfile == strlog)
    {
        long filesize = 0;
        Logger::get_mutable_instance().Print("period archive log");
        if(m_ofscache.is_open())
        {
            filesize = m_ofscache.tellp();
        }
        else
        {
            struct stat fileinfo;
            stat(strlog.c_str(), &fileinfo);
            filesize = fileinfo.st_size;
        }

        if(filesize > 6) //6 is the head version size
        {
            if(m_ofscache.is_open())
            {
                m_ofscache.close();
            }

            // 移动日志为 .bk，并创建新日志 .log
            MoveLogFile();
            CreateNewLog();
        }

        extime = std::chrono::hours(ARCHIVE_PERIOD);
    }
    else
    {
        Logger::get_mutable_instance().Print("log has been archived");
        string::size_type beginpos = m_logfile.rfind("_");
        string::size_type endpos = m_logfile.rfind(".");
        time_t ctime = strtol(m_logfile.substr(beginpos + 1, endpos - beginpos - 1).c_str(), nullptr, 10);
        system_clock::time_point savetime = system_clock::from_time_t(ctime);
        extime = (savetime + std::chrono::hours(ARCHIVE_PERIOD)) - system_clock::now();
    }

    m_archivetimer.expires_after(extime);
    m_archivetimer.async_wait(std::bind(&CacheData::OnArchiveHandle, this, m_logfile));
}

void CacheData::OnArchiveHandle(string logname)
{
    AddTaskInfo(ARCHIVE_LOG, logname);
}

void CacheData::OnScanHandle(int period)
{
    Logger::get_mutable_instance().Print("run scan log dir");
    ScanForRemove();
    m_scantimer.expires_after(std::chrono::minutes(period));
    m_scantimer.async_wait(std::bind(&CacheData::OnScanHandle, this, period));
}

void CacheData::ScanForRemove()
{
    uint64_t totalsize = 0;
    ScanMap mapscan;
    ScanLogDir(totalsize, mapscan);
    RemoveOldFiles(totalsize, mapscan);
}

void CacheData::RemoveOldFiles(uint64_t totalsize, ScanMap &mapscan)
{
    // 当文件数 > 24 个时，删除多余的文件
    while(mapscan.size() > MAX_FILENUM)
    {
        Logger::get_mutable_instance().Print("remove old files:%s", mapscan.begin()->first.c_str());
        totalsize -= mapscan.begin()->second;
        unlink(mapscan.begin()->first.c_str());
        mapscan.erase(mapscan.begin());
    }

    // 当文件数 > 7 个，且文件大小 >= 2M 时，删除一个文件
    if(mapscan.size() > MIN_FILENUM &&
            totalsize >= MAX_SIZE)
    {
        Logger::get_mutable_instance().Print("directory size too large:%l, remove one old files", totalsize);
        totalsize -= mapscan.begin()->second;
        unlink(mapscan.begin()->first.c_str());
        mapscan.erase(mapscan.begin());
    }
}

void CacheData::MoveLogFile()
{
    string strarchive = m_logfile.substr(0, m_logfile.rfind("."));
    strarchive.append("_");
    strarchive.append(std::to_string(m_fileindex++));
    strarchive.append(".bk");
    rename(m_logfile.c_str(), strarchive.c_str());
}

void CacheData::LoadEventCode()
{
    // 若目录 /var/lib/deepin/event-log/increment/ 不存在，则创建
    string strname = "";
    string evdir(INCREASE_CODE);
    if(access(evdir.c_str(), F_OK) == -1)
    {
        syslog(LOG_DEBUG, "increase code file is not exists");
    }
    else
    {
        DIR *pevdir = opendir(evdir.c_str());
        if(pevdir == nullptr)
        {
            Logger::get_mutable_instance().Print("%s", strerror(errno));
            syslog(LOG_ERR, "open increase code dir failed");
            return;
        }

        // 获取事件码文件
        string filename;
        struct dirent *pinfo = nullptr;
        while ((pinfo = readdir(pevdir)) != nullptr)
        {
            if(!strcmp(pinfo->d_name, ".") ||
                    !strcmp(pinfo->d_name, ".."))
            {
                continue;
            }

            filename.assign(pinfo->d_name);
            if(pinfo->d_type == DT_REG &&
                    std::regex_match(filename, std::regex("^\\d*$")))
            {
                strname.assign(pinfo->d_name);
                break;
            }
        }

        closedir(pevdir);
    }

    // 若文件不存在，则创建
    if(strname.empty())
    {
        strname = "1";
        string filename = evdir + strname;
        CreateFile(filename);
    }

    m_codefile = evdir + strname;
    m_eventcode = strtoull(strname.c_str(), nullptr, 10);
}

bool CacheData::CreateFile(const std::string &filename)
{
    std::ofstream file(filename);

    if (!file) {
        Logger::get_mutable_instance().Print("create file failed:%s", filename.c_str());
        return false;
    }

    file.close();
    return true;
}

uint64_t CacheData::GetNextEventCode()
{
    if(m_eventcode + 1 > m_nextevcode)
    {
        m_nextevcode += EVENT_STEP;
        UpdateEventCode();
    }

    return ++m_eventcode;
}

void CacheData::UpdateEventCode()
{
    string nextfile(INCREASE_CODE);
    nextfile.append(std::to_string(m_nextevcode));
    rename(m_codefile.c_str(), nextfile.c_str());
    m_codefile = nextfile;
}

bool CacheData::CheckCache()
{
    // 若本地文件不存在
    if (access(m_logfile.c_str(), F_OK) == -1)
    {
        // 不存在，但是被打开了（即：本地文件被删除）
        if (m_ofscache.is_open())
            m_ofscache.close();

        Logger::get_mutable_instance().Print("create new log file");
        CreateNewLog();
    } else {
        // 文件存在，但打开失败了
        if (!m_ofscache.is_open())
        {
            Logger::get_mutable_instance().Print("create new log file");
            CreateNewLog();
        }
    }

    return m_ofscache.is_open();
}

void CacheData::WriteCacheData(const string &strdata)
{
    syslog(LOG_DEBUG, "write data to cache");
    Logger::get_mutable_instance().Print("write cache data:%s", strdata.c_str());

    // 若果本地缓存校验失败
    if (!CheckCache())
    {
        syslog(LOG_ERR, "open cache file failed");
        Logger::get_mutable_instance().Print("open cache file failed");
        return;
    }

    uint32_t length = strdata.size();
    m_ofscache.write((char*)&length, sizeof (length));
    m_ofscache.write(strdata.c_str(), strdata.size());
    m_ofscache.flush();
    // 若文件大于 50K，则移动归档，并创建新日志
    long filelength = m_ofscache.tellp();
    if(filelength >= ARCHIVE_SIZE)
    {
        Logger::get_mutable_instance().Print("log file is too large, need to archive");
        m_ofscache.close();
        MoveLogFile();
        CreateNewLog();
    }
}

void CacheData::LoopWriteData(void *p)
{
    CacheData *pcache = reinterpret_cast<CacheData*>(p);
    if(pcache)
    {
        while (!pcache->m_quit)
        {
            TaskInfo taskdata;
            {
                // 若队列为空，则阻塞线程 3 秒
                std::unique_lock<std::mutex> lock(pcache->m_mutex);
                while (!pcache->m_quit && pcache->m_taskqueue.size() == 0)
                {
                    pcache->m_cond.wait_for(lock, std::chrono::seconds(3));
                }

                // 取出并删除第一个任务
                if(!pcache->m_quit)
                {
                    taskdata = pcache->m_taskqueue.front();
                    pcache->m_taskqueue.pop_front();
                }
            }

            // 根据任务类型，进行归档或者缓存
            switch (taskdata.type)
            {
            case ARCHIVE_LOG:
                pcache->ArchiveLog(taskdata.data);
                break;
            case WRITE_CACHE:
                if(!taskdata.data.empty())
                {
                    pcache->WriteCacheData(taskdata.data);
                }
                break;
            default:
                break;
            }
        }

        Logger::get_mutable_instance().Print("loop write exit");
    }
}
