/*
 * Copyright (c) 2010-2020 Nathan Rajlich
 *
 *  Permission is hereby granted, free of charge, to any person
 *  obtaining a copy of this software and associated documentation
 *  files (the "Software"), to deal in the Software without
 *  restriction, including without limitation the rights to use,
 *  copy, modify, merge, publish, distribute, sublicense, and/or sell
 *  copies of the Software, and to permit persons to whom the
 *  Software is furnished to do so, subject to the following
 *  conditions:
 *
 *  The above copyright notice and this permission notice shall be
 *  included in all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 *  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 *  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 *  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 *  HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 *  WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 *  OTHER DEALINGS IN THE SOFTWARE.
 */

package org.java_websocket.example;

import java.io.File;
import java.io.FileInputStream;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import java.nio.file.Paths;
import java.security.KeyStore;
import java.util.Collections;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManagerFactory;
import org.java_websocket.WebSocket;
import org.java_websocket.drafts.Draft;
import org.java_websocket.drafts.Draft_6455;
import org.java_websocket.handshake.ClientHandshake;
import org.java_websocket.server.DefaultSSLWebSocketServerFactory;
import org.java_websocket.server.WebSocketServer;

public class AutobahnSSLServerTest extends WebSocketServer {

  private static int counter = 0;

  public AutobahnSSLServerTest(int port, Draft d) throws UnknownHostException {
    super(new InetSocketAddress(port), Collections.singletonList(d));
  }

  public AutobahnSSLServerTest(InetSocketAddress address, Draft d) {
    super(address, Collections.singletonList(d));
  }

  @Override
  public void onOpen(WebSocket conn, ClientHandshake handshake) {
    counter++;
    System.out.println("///////////Opened connection number" + counter);
  }

  @Override
  public void onClose(WebSocket conn, int code, String reason, boolean remote) {
    System.out.println("closed");
  }

  @Override
  public void onError(WebSocket conn, Exception ex) {
    System.out.println("Error:");
    ex.printStackTrace();
  }

  @Override
  public void onStart() {
    System.out.println("Server started!");
  }

  @Override
  public void onMessage(WebSocket conn, String message) {
    conn.send(message);
  }

  @Override
  public void onMessage(WebSocket conn, ByteBuffer blob) {
    conn.send(blob);
  }

  public static void main(String[] args) throws UnknownHostException {
    int port;
    try {
      port = Integer.parseInt(args[0]);
    } catch (Exception e) {
      System.out.println("No port specified. Defaulting to 9003");
      port = 9003;
    }
    AutobahnSSLServerTest test = new AutobahnSSLServerTest(port, new Draft_6455());
    try {
      // load up the key store
      String STORETYPE = "JKS";
      String KEYSTORE = Paths.get("src", "test", "java", "org", "java_websocket", "keystore.jks")
          .toString();
      String STOREPASSWORD = "storepassword";
      String KEYPASSWORD = "keypassword";

      KeyStore ks = KeyStore.getInstance(STORETYPE);
      File kf = new File(KEYSTORE);
      ks.load(new FileInputStream(kf), STOREPASSWORD.toCharArray());

      KeyManagerFactory kmf = KeyManagerFactory.getInstance("SunX509");
      kmf.init(ks, KEYPASSWORD.toCharArray());
      TrustManagerFactory tmf = TrustManagerFactory.getInstance("SunX509");
      tmf.init(ks);

      SSLContext sslContext = null;
      sslContext = SSLContext.getInstance("TLS");
      sslContext.init(kmf.getKeyManagers(), tmf.getTrustManagers(), null);

      test.setWebSocketFactory(new DefaultSSLWebSocketServerFactory(sslContext));
    } catch (Exception e) {
      e.printStackTrace();
    }
    test.setConnectionLostTimeout(0);
    test.start();
  }

}
