// SPDX-FileCopyrightText: 2018 - 2023 UnionTech Software Technology Co., Ltd.
//
// SPDX-License-Identifier: GPL-3.0-or-later

#include "connectivityprocesser.h"
#include "connectivitychecker.h"
#include "netinterface.h"
#include "wirelessdevice.h"

#include <DConfig>

#include <QTimer>
#include <QThread>

using namespace dde::network;

ConnectivityProcesser::ConnectivityProcesser(NetworkProcesser *networkProcesser, QObject *parent)
    : QObject(parent)
    , m_networkProcesser(networkProcesser)
    , m_connectivityChecker(nullptr)
    , m_connectivity(networkProcesser->connectivity())
    , m_checkThread(nullptr)
    , m_checkTimer(nullptr)
    , m_timer(nullptr)
    , m_dConfig(Dtk::Core::DConfig::create("org.deepin.dde.network", "org.deepin.dde.network", QString(), this))
{
    connect(m_dConfig, &Dtk::Core::DConfig::valueChanged, this, &ConnectivityProcesser::onConfigChanged);
    QStringList keyList = m_dConfig->keyList();
    if (keyList.contains("enableConnectivity") && m_dConfig->value("enableConnectivity").toBool()) {
        createConnectivityChecker();
        initConnectivityChecker(keyList.contains("ConnectivityCheckInterval") ? m_dConfig->value("ConnectivityCheckInterval").toInt() * 1000 : 30000);
    } else {
        m_nmConnectivityConnection = connect(m_networkProcesser, &NetworkProcesser::connectivityChanged, this, &ConnectivityProcesser::onConnectivityChanged, Qt::UniqueConnection);
    }
}

ConnectivityProcesser::~ConnectivityProcesser()
{
    stopChecker();
}

Connectivity ConnectivityProcesser::connectivity() const
{
    return m_connectivity;
}

void ConnectivityProcesser::createConnectivityChecker()
{
    m_connectivityChecker = new ConnectivityChecker;
    m_checkerConnection << connect(m_connectivityChecker, &ConnectivityChecker::checkFinished, this, &ConnectivityProcesser::onCheckStatusChanged);

    // 检查网络的线程
    m_checkThread = new QThread;
    m_connectivityChecker->moveToThread(m_checkThread);
    m_checkThread->start();
}

void ConnectivityProcesser::initConnectivityChecker(int interval)
{
    // 定期检查网络的状态
    m_checkTimer = new QTimer;
    m_checkTimer->setInterval(interval);
    m_checkerConnection << connect(m_checkTimer, &QTimer::timeout, m_connectivityChecker, &ConnectivityChecker::startCheck, Qt::UniqueConnection);
    m_checkTimer->start();

    // 每次执行一次
    m_timer = new QTimer;
    m_timer->setSingleShot(true);
    m_timer->setInterval(1000);
    m_checkerConnection << connect(m_timer, &QTimer::timeout, m_connectivityChecker, &ConnectivityChecker::startCheck, Qt::UniqueConnection);
    m_checkerConnection << connect(m_connectivityChecker, &ConnectivityChecker::runFinished, this, [ this ] {
        if (m_timer->isActive())
            m_timer->stop();
    }, Qt::UniqueConnection);

    // 当网络设备状态发生变化的时候需要检查网络状态
    initDeviceConnect(m_networkProcesser->devices());

    m_checkerConnection << connect(m_networkProcesser, &NetworkProcesser::deviceAdded, this, [ this ](QList<NetworkDeviceBase*> newDevices) {
        initDeviceConnect(newDevices);
    }, Qt::UniqueConnection);
}

void ConnectivityProcesser::updateAccessPointConnect(const QList<AccessPoints *> &accessPoints)
{
    for (dde::network::AccessPoints *accessPoint : accessPoints) {
        m_checkerConnection << connect(accessPoint, &dde::network::AccessPoints::connectionStatusChanged, this, [ this ] {
            if (!m_timer->isActive()) {
                m_timer->start();
            }
        }, Qt::UniqueConnection);
    }
}

void ConnectivityProcesser::initDeviceConnect(const QList<NetworkDeviceBase *> &devices)
{
    for (NetworkDeviceBase *device : devices) {
        m_checkerConnection << connect(device, &NetworkDeviceBase::deviceStatusChanged, m_connectivityChecker, &ConnectivityChecker::startCheck, Qt::UniqueConnection);
        m_checkerConnection << connect(device, &NetworkDeviceBase::activeConnectionChanged, this, [ this ] {
            if (!m_timer->isActive())
                m_timer->start();
        }, Qt::UniqueConnection);
        if (device->deviceType() == DeviceType::Wireless) {
            dde::network::WirelessDevice *wirelessDevice = static_cast<dde::network::WirelessDevice *>(device);
            m_checkerConnection << connect(wirelessDevice, &dde::network::WirelessDevice::networkAdded, this, [ this ](QList<AccessPoints*> accessPoints) {
                updateAccessPointConnect(accessPoints);
            });
            updateAccessPointConnect(wirelessDevice->accessPointItems());
        }
    }
}

void ConnectivityProcesser::stopChecker()
{
    for (QMetaObject::Connection connection : m_checkerConnection) {
        disconnect(connection);
    }
    m_checkerConnection.clear();

    if (m_checkTimer) {
        m_checkTimer->stop();
        m_checkTimer->deleteLater();
        m_checkTimer = nullptr;
    }

    if (m_timer) {
        m_timer->stop();
        m_timer->deleteLater();
        m_timer = nullptr;
    }

    if (m_checkThread) {
        m_checkThread->quit();
        m_checkThread->wait();
        m_checkThread->deleteLater();
        m_checkThread = nullptr;
    }

    if (m_connectivityChecker) {
        delete m_connectivityChecker;
        m_connectivityChecker = nullptr;
    }

    if (m_nmConnectivityConnection) {
        disconnect(m_nmConnectivityConnection);
    }
}

void ConnectivityProcesser::onCheckStatusChanged(bool isConnected)
{
    Connectivity connectivity = m_connectivity;
    if (isConnected) {
        // 如果检测到了网络是联通的，则认为当前网络为联通
        connectivity = Connectivity::Full;
    } else {
        // 如果检测到当前的网络不通，则做如下处理
        // 1、NM的网络返回为已经连接，则以当前的检测连接为准（未连接）
        // 2、NM的网络返回为其他内容，则以NM返回的结果为准
        if (m_networkProcesser->connectivity() == Connectivity::Full)
            connectivity = Connectivity::Noconnectivity;
        else
            connectivity = m_networkProcesser->connectivity();
    }

    if (connectivity == m_connectivity)
        return;

    onConnectivityChanged(connectivity);
}

void ConnectivityProcesser::onConnectivityChanged(const Connectivity &connectivity)
{
    m_connectivity = connectivity;
    // 更新每个设备的联通性
    QList<NetworkDeviceBase *> devices = m_networkProcesser->devices();
    for (NetworkDeviceBase *device : devices) {
        device->deviceRealize()->setConnectivity(m_connectivity);
    }
    emit connectivityChanged(m_connectivity);
}

void ConnectivityProcesser::onConfigChanged(const QString &key)
{
    if (key != QString("enableConnectivity"))
        return;

    stopChecker();
    if (m_dConfig->value("enableConnectivity").toBool()) {
        createConnectivityChecker();
        initConnectivityChecker(m_dConfig->keyList().contains("ConnectivityCheckInterval") ? m_dConfig->value("ConnectivityCheckInterval").toInt() * 1000 : 30000);
    } else {
        m_nmConnectivityConnection = connect(m_networkProcesser, &NetworkProcesser::connectivityChanged, this, &ConnectivityProcesser::onConnectivityChanged, Qt::UniqueConnection);
    }

}
