/*
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "glue-setup-interactive.h"
#include "simsetup.h"
#include "system.h"

static int old_token = -1;
static char old_buf[1024];

static void
syntax_error(void)
{
	fprintf(stderr, "%s: syntax error.\n", simsetup.vhdl_model);
	exit(1);
}

static int
token_get(FILE *fp, char *buf)
{
	int token;
	int c;

	if (old_token != -1) {
		token = old_token;
		strcpy(buf, old_buf);

		old_token = -1;
		return token;
	}

	for (;;) {
		c = fgetc(fp);
		if (c == EOF) {
			/* End-of-File */
			return EOF;

		} else if (c == '-') {
			c = fgetc(fp);
			if (c != '-') {
				ungetc(c, fp);
				c = '-';
				goto integer;
			}
			/* Comment */
			do {
				c = fgetc(fp);
			} while (c != EOF
			      && c != '\n');

		} else if (c == '\t'
			|| c == '\n'
			|| c == '\r'
			|| c == ' ') {
			/* Skip white space. */

		} else if (c == ':'
			|| c == ';'
			|| c == ','
			|| c == '='
			|| c == '>'
			|| c == '<'
			|| c == '('
			|| c == ')') {
			/* Punctuation */
			return c;

		} else if (c == '-'
			|| ('0' <= c && c <= '9')) {
			/* Integer */
		integer:;
			do {
				*buf++ = c;
				c = fgetc(fp);
			} while ('0' <= c && c <= '9');
			*buf = '\0';
			ungetc(c, fp);
			return '0';

		} else if (('A' <= c && c <= 'Z')
			|| ('a' <= c && c <= 'z')) {
			/* Identifier */
			do {
				*buf++ = c;
				c = fgetc(fp);
			} while (('0' <= c && c <= '9')
			      || ('A' <= c && c <= 'Z')
			      || ('a' <= c && c <= 'z')
			      || c == '_');
			*buf = '\0';
			ungetc(c, fp);
			return 'A';

		} else if (c == '"') {
			/* String */
			c = fgetc(fp);
			while (c != '"') {
				*buf++ = c;
				c = fgetc(fp);
			}
			*buf = '\0';
			return '"';

		} else {
			syntax_error();
		}
	}
}

static void
token_unget(int token, char *buf)
{
	assert(old_token == -1);

	old_token = token;
	strcpy(old_buf, buf);
}

void
glue_setup_interactive_create(void)
{
	FILE *fp;
	char string[1024];
	char name[1024];
	char signame[1024];
	char sigtype[1024];
	char compname[1024];
	char comptype[1024];
	char portname[1024];
	char genericname[1024];
	char value[1024];
	int comp_id;
	int manage_id;
	int token;
	int ret;

	/* Read system.svhdl. */
	fp = fopen(simsetup.vhdl_model, "r");
	assert(fp);

	/*
	 * Skip ENTITY.
	 */
	token = token_get(fp, string);
	assert(token == 'A'
	    && strcasecmp(string, "ENTITY") == 0);

	token = token_get(fp, name);
	assert(token == 'A');

	token = token_get(fp, string);
	assert(token == 'A'
	    && strcasecmp(string, "IS") == 0);

	token = token_get(fp, string);
	assert(token == 'A'
	    && strcasecmp(string, "END") == 0);

	token = token_get(fp, string);
	assert(token == 'A'
	    && strcasecmp(string, name) == 0);

	token = token_get(fp, string);
	assert(token == ';');

	/*
	 * Read ARCHITECTURE.
	 */
	/* Read header. */
	token = token_get(fp, string);
	assert(token == 'A'
	    && strcasecmp(string, "ARCHITECTURE") == 0);

	token = token_get(fp, string);
	assert(token == 'A'
	    && strcasecmp(string, "STRUCTURAL") == 0);

	token = token_get(fp, string);
	assert(token == 'A'
	    && strcasecmp(string, "OF") == 0);

	token = token_get(fp, string);
	assert(token == 'A'
	    && strcasecmp(string, name) == 0);

	token = token_get(fp, string);
	assert(token == 'A'
	    && strcasecmp(string, "IS") == 0);

	/* Read signal list. */
	for (;;) {
		token = token_get(fp, string);
		if (token != 'A'
		 || strcasecmp(string, "BEGIN") == 0) {
			token_unget(token, string);
			break;
		}

		signame[0] = ':';
		token = token_get(fp, signame + 1);
		assert(token == 'A');

		token = token_get(fp, string);
		assert(token == ':');

		token = token_get(fp, sigtype);
		assert(token == 'A');

		token = token_get(fp, string);
		assert(token == ';');

		ret = system_sig_create(sigtype, signame);
		assert(0 <= ret);
	}

	/* Skip BEGIN. */
	token = token_get(fp, string);
	assert(token == 'A'
	    && strcasecmp(string, "BEGIN") == 0);

	ret = system_page_create("<default>");
	assert(0 <= ret);

	/* Read component list. */
	for (;;) {
		compname[0] = ':';
		token = token_get(fp, compname + 1);
		if (token != 'A'
		 || strcasecmp(compname + 1, "END") == 0) {
			token_unget(token, compname + 1);
			break;
		}
		assert(token == 'A');

		token = token_get(fp, string);
		assert(token == ':');

		token = token_get(fp, comptype);
		assert(token == 'A');

		comp_id = system_comp_create(comptype, compname, 0, 0);
		assert(0 <= comp_id);
		manage_id = system_sig_create("manage", compname);
		assert(0 <= manage_id);
		system_comp_port_connect(comp_id, "manage", manage_id);

		/* Read generic map. */
		token = token_get(fp, string);
		if (token != 'A'
		    || strcasecmp(string, "GENERIC") != 0) {
			token_unget(token, string);
			goto port_map;
		}

		token = token_get(fp, string);
		assert(token == 'A'
		    && strcasecmp(string, "MAP") == 0);

		token = token_get(fp, string);
		assert(token == '(');

		/* Read generic map list. */
		do {
			token = token_get(fp, genericname);
			assert(token == 'A');

			token = token_get(fp, string);
			assert(token == '=');
			token = token_get(fp, string);
			assert(token == '>');

			token = token_get(fp, value);
			assert(token == '0'
			    || token == '"');

			if (token == '0') {
				system_comp_generic_set(comp_id,
						"integer", genericname, value);
			} else if (token == '"') {
				system_comp_generic_set(comp_id,
						"string", genericname, value);
			}

			token = token_get(fp, string);
			assert(token == ','
			    || token == ')');
		} while (token != ')');

		/* Read port map. */
	port_map:;
		token = token_get(fp, string);
		assert(token == 'A'
		    && strcasecmp(string, "PORT") == 0);

		token = token_get(fp, string);
		assert(token == 'A'
		    && strcasecmp(string, "MAP") == 0);

		token = token_get(fp, string);
		assert(token == '(');

		/* Read port map list. */
		do {
			token = token_get(fp, portname);
			assert(token == 'A');

			token = token_get(fp, string);
			assert(token == '=');
			token = token_get(fp, string);
			assert(token == '>');

			signame[0] = ':';
			token = token_get(fp, signame + 1);
			assert(token == 'A');

			token = token_get(fp, string);
			assert(token == ','
			    || token == ')');
			
			system_comp_port_connect(comp_id,
					portname, system_sig_lookup(signame));
		} while (token != ')');

		token = token_get(fp, string);
		assert(token == ';');

		system_comp_init(comp_id);
	}
	
	/* Read trailer. */
	token = token_get(fp, string);
	assert(token == 'A'
	    && strcasecmp(string, "END") == 0);
	
	token = token_get(fp, string);
	assert(token == 'A'
	    && strcasecmp(string, "STRUCTURAL") == 0);

	token = token_get(fp, string);
	assert(token == ';');

	token = token_get(fp, string);
	assert(token == EOF);

	/* Close file. */
	ret = fclose(fp);
	assert(ret == 0);
}

void
glue_setup_interactive_destroy(void)
{
	char path[1024];
	FILE *fp;
	unsigned int sigid;
	unsigned int compid;
	int ret;

	/* Write temporary system.svhdl. */
	strcpy(path, simsetup.vhdl_model);
	strcat(path, ".tmp");

	fp = fopen(path, "w");
	assert(fp);

	fprintf(fp, "--\n");
	fprintf(fp, "-- FAUmachine interactive setup file\n");
	fprintf(fp, "--\n");

	fprintf(fp, "\n");

	fprintf(fp, "entity system is\n");
	fprintf(fp, "end system;\n");

	fprintf(fp, "\n");

	fprintf(fp, "architecture structural of system is\n");

	for (sigid = 0; sigid < 100000; sigid++) { /* Ugly -- FIXME */
		char sigtype[1024];
		char signame[1024];

		if (system_sig_info(sigid, sigtype, signame) < 0) {
			continue;
		}

		if (strcmp(sigtype, "manage") == 0) {
			continue;
		}

		fprintf(fp, "\tsignal %s : %s;\n", signame + 1, sigtype);
	}

	fprintf(fp, "begin\n");

	for (compid = 0; compid < 100000; compid++) { /* Ugly -- FIXME */
		char comptype[1024];
		char compname[1024];
		unsigned int node;
		unsigned int page;
		unsigned int genericid;
		char gentype[1024];
		char genname[1024];
		char genvalue[1024];
		unsigned int portid;
		char portname[1024];
		unsigned int sigid;
		char sigtype[1024];
		char signame[1024];
		int first;

		if (system_comp_info(compid, comptype, compname,
				&node, &page) < 0) {
			continue;
		}

		fprintf(fp, "\t%s : %s\n", compname + 1, comptype);

		if (0 <= system_comp_generic_info(compid, 0,
				gentype, genname, genvalue)) {
			fprintf(fp, "\t\tgeneric map(\n");
			
			if (strcmp(gentype, "integer") == 0) {
				fprintf(fp, "\t\t\t%s => %s",
						genname, genvalue);
			} else if (strcmp(gentype, "string") == 0) {
				fprintf(fp, "\t\t\t%s => \"%s\"",
						genname, genvalue);
			} else {
				assert(0);
			}

			for (genericid = 1; genericid < 1000; genericid++) {
				if (system_comp_generic_info(compid, genericid,
						gentype, genname, genvalue) < 0) {
					continue;
				}
				fprintf(fp, ",\n");

				if (strcmp(gentype, "integer") == 0) {
					fprintf(fp, "\t\t\t%s => %s",
							genname, genvalue);
				} else if (strcmp(gentype, "string") == 0) {
					fprintf(fp, "\t\t\t%s => \"%s\"",
							genname, genvalue);
				} else {
					assert(0);
				}
			}

			fprintf(fp, "\n");
			fprintf(fp, "\t\t)\n");
		}

		fprintf(fp, "\t\tport map(\n");
		
		first = 1;
		for (portid = 0; portid < 1000; portid++) { /* Ugly -- FIXME */
			if (system_comp_port_info(compid, portid,
					portname, &sigid) < 0) {
				continue;
			}
			ret = system_sig_info(sigid,
					sigtype, signame);
			assert(0 <= ret);
			if (strcmp(sigtype, "manage") == 0) {
				continue;
			}

			if (first) {
				first = 0;
			} else {
				fprintf(fp, ",\n");
			}

			fprintf(fp, "\t\t\t%s => %s", portname, signame + 1);
		}
		fprintf(fp, "\n");

		fprintf(fp, "\t\t);\n");
	}

	fprintf(fp, "end structural;\n");

	ret = fclose(fp);
	assert(ret == 0);

	/* Rename atomically. */
	ret = rename(path, simsetup.vhdl_model);
	assert(0 <= ret);

	/* Destroy components. */
	for (compid = 0; compid < 100000; compid++) { /* Ugly -- FIXME */
		system_comp_exit(compid);
		system_comp_destroy(compid);
		/* Should destroy "manage" signal, too! FIXME */
	}

	/* Destroy signals. */
	for (sigid = 0; sigid < 100000; sigid++) { /* Ugly -- FIXME */
		system_sig_destroy(sigid);
	}
}
