// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


//
// stararray.cc - The Star Array class.  Not much more than a container class
//  for stars, with the ability to read and filter a text data file on the fly.
//

#include "stararray.h"
#include "greek.h"

#define NEED_FULL_NAMES
#include "constellations.h"

using std::string;
using std::vector;

// StarArray private functions -------------------------------------

// The usual functions to switch between coordinate systems; as always,
//  they don't check to see what coordinate system things are already
//  in before blindly applying the transform.  These are private, so the
//  check will be done by StarArray::SetRules().

void StarArray::toCelestial()
{ iterate (vector<Star>, Array, i) (*i).toCelestial(); }

void StarArray::toGalactic()
{ iterate (vector<Star>, Array, i) (*i).toGalactic(); }


// Function to read an input file (assumes input is a valid opened ifstream)
//  and APPEND onto Array all the stars which pass the filters specified
//  in ArrayRules.
//  Data files are in the following format, with records separated by
//  the character RECORD_DELIMITER defined in "stararray.h":
//
// File header: may not contain the character RECORD_DELIMITER
// RECORD_DELIMITER first record (may contain line breaks)
// RECORD_DELIMITER second record (likewise)
// RECORD_DELIMITER etc.
//
// [See comment above Star::Star(const char *) in star.cc, or example
// StarPlot data file "sample.stars", for detailed structure of records.]

void StarArray::Read(std::ifstream &input)
{
  double chartdist = ArrayRules.ChartLocation.magnitude();
  string garbage;
  char record[RECORD_MAX_LENGTH + 1];
  vector<Star> companions = vector<Star>();
  companions.reserve(100);
  Star tempstar, faststar;

  // First, discard all characters before the first record
  std::getline(input, garbage, RECORD_DELIMITER);

  // Then loop over records.  std::getline(istream, string, char) would be
  //  much simpler, but it's _slow_.
  while (input.get(record, RECORD_MAX_LENGTH + 1, RECORD_DELIMITER)) {

    // Purge the record delimiter from input and check to see if we've
    //  exceeded the size of the record string buffer.  If so, ignore
    //  the rest of that record.
    if (input.get() != RECORD_DELIMITER) 
      std::getline(input, garbage, RECORD_DELIMITER);

    // keep track of total # of records in all files we've scanned in,
    //  and beware of memory overflows
    TotalStars++;

    // in the following we want "continue" instead of "return" so that
    //  TotalStars will be tallied correctly.
    if (Array.size() >= MAX_STARS) continue;

    // Filter out stars by distance and magnitude.  This is the fastest
    //  filter because it doesn't create any temporary objects.  So it 
    //  comes first.
    {
      double starmag, stardist;
      size_t posn = 0, field = 0, len = strlen(record);

      while (posn < len && field < 3)
	if (record[posn++] == FIELD_DELIMITER) field++;
      stardist = starmath::atof(record + posn);
      if (stardist > chartdist + ArrayRules.ChartRadius ||
	  stardist < chartdist - ArrayRules.ChartRadius) continue;

      while (posn < len && field < 6)
	if (record[posn++] == FIELD_DELIMITER) field++;
      starmag = starmath::atof(record + posn);
      if (starmag > ArrayRules.ChartDimmestMagnitude ||
	  starmag < ArrayRules.ChartBrightestMagnitude) continue;
    }

    // Convert each record which passes the fast filter into a Star and see
    //  whether it should go into the STL vector of Stars.
    faststar = Star(record, true /* fast conversion */);

    if (! ArrayRules.CelestialCoords)  // we assume the data file is in
      faststar.toGalactic();           // celestial, not galactic, coords

    if (faststar.PassesFilter(ArrayRules)) {
      tempstar = Star(record);
      if (! ArrayRules.CelestialCoords)	tempstar.toGalactic();

      // If it passes all the filters and is a singleton, primary, or
      //  sufficiently far from its primary, put it into the STL vector.
      if (! ArrayRules.ChartHideCompanions
	  || (tempstar.GetStarPrimaryDistance() == 0.0)
	  || ((tempstar.GetStarPrimaryDistance() * (200 /*pixels*/) 
	       / ArrayRules.ChartRadius) > MIN_SECONDARY_DISTANCE))
	Array.push_back(tempstar);

      // If it is a companion star, but no membership is given, we have
      //  no way of telling whether its primary has been filtered out,
      //  so toss it.
      else if (tempstar.GetStarMembership()[0].size() == 0) continue;

      // Put everything else into a separate STL vector to examine later.
      else companions.push_back(tempstar);
    }
  }

  // Now, go through all the stars in the array of companions.
  //  Put stars in this array into the main array only if their primaries
  //  are not already in the main array.

  iterate (vector<Star>, companions, comp_ptr) {
    vector<Star>::iterator Array_ptr = Array.begin();

    if (Array.size() >= MAX_STARS) return;
    while ((Array_ptr != Array.end())
	   && (Array_ptr->GetStarPrimaryDistance() != 0.0
	       || ((*comp_ptr).GetStarMembership()[0] !=
			  Array_ptr->GetStarMembership()[0])))
      Array_ptr++;
    if (Array_ptr == Array.end()) /* no match */ Array.push_back(*comp_ptr);
  }
  return;
}


// Functions to sort stars in order of increasing "local" x-coordinate
// XXX: rewrite this with C++ sort() algorithm
// First, a typedef to hold all the information the compare function will need

typedef struct {
  double xposition;
  Star   star;
} sortable;

// Next, the function to compare for qsort().

int compare_function(const void *p, const void *q)
{
  double x1 = ((const sortable *)p)->xposition;
  double x2 = ((const sortable *)q)->xposition;
  return (x1 - x2 >= 0.0) ? 1 : -1;
}

// Finally, the main function which calls qsort()

void StarArray::Sort()
{
  size_t size = Array.size();
  Vector3 relativeLocation;
  SolidAngle orientation = ArrayRules.ChartOrientation;
  sortable *temparray = new sortable[size];

  // Make a temporary array for qsort(), consisting of "sortable" structs
  //  which each contain a Star and a position in local coordinates.
  for (size_t i = 0; i < size; i++) {
    relativeLocation = Array[i].GetStarXYZ() - ArrayRules.ChartLocation;
    temparray[i].xposition =
      relativeLocation.getX() * cos(orientation.getPhi())
      + relativeLocation.getY() * sin(orientation.getPhi());
    temparray[i].star = Array[i];
  }

  qsort(temparray, size, sizeof(sortable), compare_function);

  // Put the sorted temporary array back into the vector
  Array.clear();
  for (size_t i = 0; i < size; i++) {
    temparray[i].star.SetPlace(i+1);    // label stars starting at 1
    Array.push_back(temparray[i].star);
  }

  delete [] temparray;
  return;
}


// StarArray public functions --------------------------------------


// Search(): function to search for a given string in the names of all the
//  stars in a set of files.  For safety, this function refuses to work unless
//  the array is currently empty.  Stars containing the substring are added 
//  to Array.  This function uses the "rules" argument to set the correct
//  coordinate system (celestial / galactic), but does not bother to set 
//  its own Rules member.

// Options:
//
//  If `casesensitive' is false, the function ignores the distinction
// between upper and lower case.  Otherwise all cases must match.
//
//  If `exactmatch' is true, `searchstring' must be identical to a star name
// (possibly excluding case) for a match after some substitutions have been
// performed.  If `exactmatch' is false, the function looks for `searchstring'
// as a substring of star names.  If, in addition, `searchstring' contains
// spaces, it is tokenized and all of the tokens must be substrings of the
// same star name for a match to occur.  No substrings may overlap.
//  In either case, some strings and characters are interpreted in special
// ways.  See the StarPlot documentation, sections 3.2.1 and 4.3.2, for
// details.
//
//  If `exitonmatch' is true, the function returns only the first match found.
// Otherwise all matches are returned.

void StarArray::Search(const std::string &search, StringList filelist, 
		       const Rules &rules, bool casesensitive,
		       bool exactmatch /* i.e. not a substring */,
		       bool exitonmatch /* i.e. return at most one result */)
{
  if (Array.size() || starstrings::isempty(search)) return;

  string searchstring = search;
  starstrings::stripspace(searchstring);

  // Interpret enclosing quote marks
  unsigned size = searchstring.size();
  if (size > 2 && searchstring[0] == '"' && searchstring[size - 1] == '"') {
    searchstring = searchstring.substr(1, size - 2);
    exactmatch = true;
  }

  // Replace real (UTF-8) lowercase Greek letters with their spelled-out names.
  for (int i = 1; i < NUM_GREEK_LETTERS; i++) {
    string name;
    if (! exactmatch) name += '^';
    name += Greek[i].name;
 
    // Make a maximum of one replacement (it would make no sense for
    // someone to include more than one Greek letter in a search string).
    if (starstrings::find_and_replace(searchstring, Greek[i].utf8, name))
      break;
  }
  
  if (! casesensitive) starstrings::toupper(searchstring);

  // Abbreviate constellation names in the search string, if any.
  for (int i = 0; i < NUM_CONSTELLATIONS; i++) {
    string name(constelnames[i]);
    string abbr;
    if (!exactmatch) abbr += '[';
    abbr += constellations[i];
    if (!exactmatch) abbr += ']';
    if (! casesensitive) {
      starstrings::toupper(name);
      starstrings::toupper(abbr);
    }
  
    // Make a maximum of one replacement (it would make no sense for
    // someone to include more than one constellation name in a search string).
    if (starstrings::find_and_replace(searchstring, name, abbr))
      break;
  }

  // Replace asterisks with degree symbols.
  while (starstrings::find_and_replace(searchstring, "*", DEGREE_UTF8))
    /* empty body */ ;

  const StringList temp_tokens(searchstring);
  StringList start_tokens, end_tokens, tokens;

  if (! exactmatch) {
    int last = 0; // index of currently last element of tokens array
    for (unsigned i = 0; i < temp_tokens.size(); i++, last++) {
      unsigned s = temp_tokens[i].size();
      if (s > 1 && temp_tokens[i][0] == '^') {
	last--;
	start_tokens.push_back(temp_tokens[i].substr(1, s - 1));
      }
      else if (s > 2 && temp_tokens[i][0] == '['
		     && temp_tokens[i][s - 1] == ']') {
	// Replace [ ] with spaces so that a search for "[token]" must turn
	// up only instances where "token" appears surrounded by whitespace.
	tokens.push_back(temp_tokens[i]);
	tokens[last][0] = tokens[last][s - 1] = ' ';

	// Also construct an alternate set of strings like " token" to
	// be searched for only at the end of each name field.
	end_tokens.push_back(tokens[last].substr(0, s - 1));
      }
      else {
	tokens.push_back(temp_tokens[i]);
	end_tokens.push_back(temp_tokens[i]);
      }
    }

    if (start_tokens.size() > 1) // invalid syntax: more than one "^token"
      return;
  }
  
  // loop over data files
  iterate (StringList, filelist, filename_ptr) {
    std::ifstream input;
    string garbage;
    char record[RECORD_MAX_LENGTH + 1];
    int nameposn = 0;

    input.open((*filename_ptr).c_str());
    if (!input.good()) continue;
    
    // First, discard all characters before the first record
    std::getline(input, garbage, RECORD_DELIMITER);

    // Then loop over records.  std::getline(istream, string, char) would be
    //  much simpler, but it's _slow_.
    while (input.get(record, RECORD_MAX_LENGTH + 1, RECORD_DELIMITER)) {
      bool result = false;
      
      // Purge the record delimiter from input and check to see if we've
      //  exceeded the size of the record string buffer.  If so, ignore
      //  the rest of that record.
      if (input.get() != RECORD_DELIMITER) 
	std::getline(input, garbage, RECORD_DELIMITER);
      
      StringList namelist(StringList(record, FIELD_DELIMITER)[0],
		          SUBFIELD_DELIMITER);
      namelist.stripspace();
      if (! casesensitive) namelist.toupper();

      iterate (StringList, namelist, name_ptr) {
	if (exactmatch)
	  result = (searchstring == *name_ptr);
	else {
	  // Search on substrings; all substrings must match within the
	  // same star name.  Replace matching substrings with *'s as we go
	  // so that found substrings don't overlap (e.g. if someone searches
	  // for "Tau Tauri").  We don't delete them completely since that
	  // could screw up word matching.
	  result = true;
	  string temp = *name_ptr;
	  if (start_tokens.size()) { // start_tokens.size() == 0 or 1
	    unsigned s = start_tokens[0].size();
	    if (start_tokens[0] == temp.substr(0, s))
	      starstrings::find_and_replace(temp, start_tokens[0], "*");
	    else
	      result = false;
	  }
	  if (result)
	  for (unsigned i = 0; i < tokens.size(); i++) {
	    bool found = false;
	    
	    // If this is a word we can safely replace it with a space;
	    // otherwise it must be replaced with a '*'
	    string replacement = ((tokens[i][0] == ' ') ? " " : "*");
	    if (starstrings::find_and_replace(temp, end_tokens[i],
				    replacement,
				    temp.size() - end_tokens[i].size()))
	      found = true;
	    else if (starstrings::find_and_replace(temp, tokens[i],
				    replacement))
	      found = true;

	    result = (result && found);
	    if (!result) break;
	  }
	}
	
	if (result) { nameposn = name_ptr - namelist.begin(); break; }

      } // closes "iterate (StringList, namelist, name_ptr)"

      if (result) {
	Star tempstar = Star(record);
	// use Star.sPlace to hold the position of the name in which a match
	//  was found:
	tempstar.SetPlace(nameposn);
	if (! rules.CelestialCoords) tempstar.toGalactic();
	Array.push_back(tempstar);

	if (exitonmatch || Array.size() >= MAX_STARS) {
	  input.close(); 
	  return; 
	}
      }

    } // closes "while (input.get(record, ...))"

    input.close();
  } // closes "iterate (StringList, filelist, filename_ptr)"

  return;
}


// function to set the rules for the ArrayRules member of StarArray.
//  Will also update the Array itself by re-reading the files specified
//  in the rules (if necessary) and/or re-sorting the array (if necessary).
//  This does not, however, call StarArray::Display(); that's up to the
//  main program.

bool StarArray::SetRules(const Rules &rules, star_changetype_t ruleschange)
{
  ArrayRules = rules;

 beginning: // if ruleschange is not a valid Changetype, set it to be
            //  type FILE_CHANGE and jump back here from default: label below

  switch (ruleschange) {
    case FILE_CHANGE:
    case LOCATION_CHANGE: case RADIUS_CHANGE:
    case COORDINATE_CHANGE: case FILTER_CHANGE:
      // reload and refilter the file(s) in ArrayRules.ChartFileNames
      //  (return with an error if none of the file(s) can be read)
      {
	size_t numValidFiles = 0;

	TotalStars = 0;
	Array.clear();
	// will probably not usually display more stars than this at once:
	Array.reserve(100);

	citerate (StringList, ArrayRules.ChartFileNames, i) {
	  std::ifstream data;
	  data.open((*i).c_str());
	  if (data.good()) {
	    Read(data);
	    data.close();
	    numValidFiles++;
	  }
	}

	// Ideally, we would have a check here to make sure there are not
	//  duplicate entries if more than one file was opened.

	if (numValidFiles == 0)
	  return false;
      }
      // fall through...

    case ORIENTATION_CHANGE: // re-sort Array
      Sort();
      // fall through...

    case DECORATIONS_CHANGE: // search for brightest 8 stars, if necessary
      if (ArrayRules.StarLabels == LANDMARK_LABEL) {

	if (Array.size() <= DISPLAY_N_STARS)
	  for (size_t i = 0; i <DISPLAY_N_STARS && i < Array.size(); i++)
	    Array[i].SetLabel(true);

	else {
	  Star *brightarray[DISPLAY_N_STARS];
	  double dimmag = -10000;

	  // The following variable is set "volatile" to work around a gcc
	  //  loop optimization bug which sometimes occurs when compiling
	  //  at -O2 or higher.  See one of these URLs for details:
	  //  http://bugs.debian.org/cgi-bin/bugreport.cgi?bug=83363
	  //  http://www.winehq.com/hypermail/wine-patches/2000/11/0076.html
	  //
	  //  Supposedly it's fixed in the latest CVS versions.  However,
	  //  I still observe it in the Debian-packaged version of the
	  //  CVS of g++-3.0.2 of 2001-09-22.
	  //
	  //  Instead of declaring this "volatile", you can instead compile
	  //  this file with the g++ flag -fno-strength-reduce if you want.
#if defined __GNUC__ // && (__GNUC__ < 3)
	  volatile
#endif
	  unsigned int dimptr = 0;

	  // find the brightest DISPLAY_N_STARS stars and put pointers to them
	  //  in brightarray

	  for (size_t i = 0; i < DISPLAY_N_STARS; i++) {
	    Array[i].SetLabel(false);
	    brightarray[i] = &(Array[i]);
	    if (brightarray[i]->GetStarMagnitude() > dimmag) {
	      dimmag = brightarray[i]->GetStarMagnitude();
	      dimptr = i;
	    }
	  }

	  for (size_t i = DISPLAY_N_STARS; i < Array.size(); i++) {
	    Array[i].SetLabel(false);

	    if (Array[i].GetStarMagnitude() < dimmag) {
	      brightarray[dimptr] = &(Array[i]);
	    
	      // now find new dimmest star in brightarray
	      dimmag = -10000;
	      for (size_t j = 0; j < DISPLAY_N_STARS; j++) {
		if (brightarray[j]->GetStarMagnitude() > dimmag) {
		  dimmag = brightarray[j]->GetStarMagnitude();
		  dimptr = j;
		}
	      }
	    }
	  }
	  
	  // now set those DISPLAY_N_STARS stars to be labeled, as well as
	  //  any other stars which have the same magnitude as the dimmest
	  //  star in brightarray.
	  iterate (StarArray, Array, star_ptr)
	    if ((*star_ptr).GetStarMagnitude() <= dimmag)
	      (*star_ptr).SetLabel(true);
	}
      }
    case UNITS_CHANGE: case NO_CHANGE:
      // do nothing; these changes are taken care of in StarArray::Display()
      break;

    default:  // assume the most radical type of change, to be safe
      ruleschange = FILE_CHANGE;
      goto beginning;            // yes, a goto statement.  I'm sorry, ok?
  }
  return true;
}


// Drawing the StarArray display -----------------------------------

// private functions to draw the chart grid

void StarArray::drawgridnorth(StarViewer *sv, unsigned int wincenterX,
			      unsigned int wincenterY, 
			      unsigned int pixelradius) const
{
  sv->setfill(false);
  sv->setcolor(POSITIVE);
  sv->drawarc(wincenterX, wincenterY, pixelradius, pixelradius, 0, M_PI);
  return;
}

void StarArray::drawgridsouth(StarViewer *sv, unsigned int wincenterX,
			      unsigned int wincenterY, 
			      unsigned int pixelradius) const
{
  sv->setfill(false);
  sv->setcolor(NEGATIVE);
  sv->drawarc(wincenterX, wincenterY, pixelradius, pixelradius,
	      M_PI, 2 * M_PI);
  return;
}

void StarArray::drawgridequator(StarViewer *sv, unsigned int wincenterX,
				unsigned int wincenterY, 
				unsigned int pixelradius) const
{
  double p_sinRA = sin(ArrayRules.ChartOrientation.getPhi()) * pixelradius;
  double p_cosRA = cos(ArrayRules.ChartOrientation.getPhi()) * pixelradius;
  double sinDec = sin(ArrayRules.ChartOrientation.getTheta());
  double p_sinRA_sinDec = p_sinRA * sinDec;
  double p_cosRA_sinDec = p_cosRA * sinDec;
  
  // equator
  sv->setfill(false);
  if (sinDec >= 0.0) sv->setcolor(POSITIVE);
  else sv->setcolor(NEGATIVE);
  sv->drawellipse(wincenterX, wincenterY,
		  pixelradius, ROUND(pixelradius * sinDec));

  // 0 - 12 hr line
  sv->drawline(wincenterX - ROUND(p_sinRA),
	       wincenterY + ROUND(p_cosRA_sinDec),
	       wincenterX + ROUND(p_sinRA),
	       wincenterY - ROUND(p_cosRA_sinDec));

  // 6 - 18 hr line
  sv->drawline(wincenterX + ROUND(p_cosRA),
	       wincenterY + ROUND(p_sinRA_sinDec),
	       wincenterX - ROUND(p_cosRA),
	       wincenterY - ROUND(p_sinRA_sinDec));

  // RA / GalLong labels
  vector<string> gridlabels;
  sv->setcolor(POSITIVE);
  if (ArrayRules.CelestialCoords)
    for (unsigned int i = 0; i < 4; i++)
      gridlabels.push_back(starstrings::itoa(6 * i) + 
	  /* TRANSLATORS: This is the abbreviation for
	     "hours of right ascension". */
	  _("h"));
  else
    for (unsigned int i = 0; i < 4; i++)
      gridlabels.push_back(starstrings::itoa(90 * i) + DEGREE_UTF8);

  sv->drawtext(gridlabels[0].c_str(),
	       wincenterX - ROUND(1.1 * p_sinRA) - 5,
	       wincenterY + ROUND(1.1 * p_cosRA_sinDec) + 5);
  sv->drawtext(gridlabels[1].c_str(),
	       wincenterX + ROUND(1.1 * p_cosRA) - 5,
	       wincenterY + ROUND(1.1 * p_sinRA_sinDec) + 5);
  sv->drawtext(gridlabels[2].c_str(),
	       wincenterX + ROUND(1.1 * p_sinRA) - 5,
	       wincenterY - ROUND(1.1 * p_cosRA_sinDec) + 5);
  sv->drawtext(gridlabels[3].c_str(),
	       wincenterX - ROUND(1.1 * p_cosRA) - 5,
	       wincenterY - ROUND(1.1 * p_sinRA_sinDec) + 5);

  return;
}


// private function to draw a single chart legend item
void StarArray::drawlegenditem(StarViewer *sv, color_t color, 
			       unsigned int number, unsigned int x,
			       unsigned int y, unsigned int r,
			       const string &text, int relx, int rely) const
{
  sv->setcolor(color);
  if (ArrayRules.StarClasses[number])
    sv->drawstar(x, y, r);
  else {
    sv->setfill(false);
    sv->drawcircle(x, y, r);
  }

  sv->setcolor(LABEL_COLOR);
  sv->drawtext(text, x + relx, y + rely);
}


// private function to draw the legend
void StarArray::drawlegend(StarViewer *sv) const
{
  unsigned int xbase, ybase = 0;
  string sra, sdec, sdist, srad, sdmag, sbmag, temp;
    
  // first draw the chart information in the upper left
  if (ArrayRules.CelestialCoords) {
    sra = 
      /* TRANSLATORS: This is the abbreviation for "right ascension." */
      string(_("R.A.")) + " ";
    sdec =
      /* TRANSLATORS: This is the abbreviation for "declination." */
      string(_("Dec.")) + " ";
  }
  else {
    sra =
      /* TRANSLATORS: This is the abbreviation for "longitude". */
      string(_("Long.")) + " ";
    sdec =
      /* TRANSLATORS: This is the abbreviation for "latitude". */
      string(_("Lat.")) + " ";
  }
  SolidAngle gridposn = ArrayRules.ChartLocation.toSpherical();
  sra += starstrings::ra_to_str(gridposn.getPhi(), ' ',
				ArrayRules.CelestialCoords, true);
  sdec += starstrings::dec_to_str(gridposn.getTheta(), ' ', true);
  sdist = starstrings::distance_to_str(ArrayRules.ChartLocation.magnitude(),
				       ArrayRules.ChartUnits)
	  + " " + _("from Earth");
  srad = string(_("Chart Radius")) + ": " +
	  starstrings::distance_to_str(ArrayRules.ChartRadius,
				       ArrayRules.ChartUnits);
  sdmag = 
	  /* TRANSLATORS: "Mag" is the abbreviation for "magnitude",
	     the astronomical brightness of an object. */
	  string(_("Dim Mag")) + ": " + starstrings::ftoa(
	      starmath::roundoff(ArrayRules.ChartDimmestMagnitude, 2), 4);
  sbmag =
	  /* TRANSLATORS: "Mag" is the abbreviation for "magnitude",
	     the astronomical brightness of an object. */
	  string(_("Bright Mag")) + ": " + starstrings::ftoa(
	      starmath::roundoff(ArrayRules.ChartBrightestMagnitude, 2), 4);

  // Calculate dimmest apparent magnitude possible for any star permitted
  // to appear on the chart, and display it.  Useful information in case
  // the user knows s/he is using a star catalog with a hard lower limit
  // based on apparent magnitude.
  double max_distance = ArrayRules.ChartLocation.magnitude()
			+ ArrayRules.ChartRadius;
  float dimmest_apparent_mag = starmath::roundoff(
    starmath::get_appmag(ArrayRules.ChartDimmestMagnitude, max_distance), 2);
  sdmag += " (";
  sdmag += starstrings::ftoa(dimmest_apparent_mag, 4);
  sdmag += ")";
  
  sv->setcolor(BOLD_COLOR); sv->drawtext(_("CHART STATUS"), 15, ybase += 15);
  sv->setcolor(LABEL_COLOR);
  sv->drawtext(_("Location of Chart Origin:"), 5, ybase += 15);
  if (ArrayRules.ChartLocation.magnitude() > 0.0) {
    sv->drawtext(sra, 10, ybase += 15);
    sv->drawtext(sdec, 10, ybase += 15);
  }
  sv->drawtext(sdist, 10, ybase += 15);
  
  if (ArrayRules.ChartRadius <= 0.0) sv->setcolor(ERROR_COLOR);
  sv->drawtext(srad, 5, ybase += 15);
  
  if (ArrayRules.ChartDimmestMagnitude - ArrayRules.ChartBrightestMagnitude
      <= 0.0)
    sv->setcolor(ERROR_COLOR);
  else sv->setcolor(LABEL_COLOR);
  sv->drawtext(sdmag, 5, ybase += 15);
  // Show bright mag only if it's been changed from the default
  if (ArrayRules.ChartBrightestMagnitude > -25)
    sv->drawtext(sbmag, 5, ybase += 15);

  // then draw the legend (which doubles as a display of which spectral
  //  classes are currently turned on) in the upper right
  
  xbase = (sv->width() > 100) ? sv->width() - 95 : 5;
  sv->setcolor(BOLD_COLOR); sv->drawtext(_("LEGEND"), xbase + 5, 15);

#define LO LEGEND_OFFSETS

  drawlegenditem(sv, WOLF_RAYET_COLOR, 0, 
		 xbase + LO[10][0], LO[10][1], 5,
	    /* TRANSLATORS: "Wolf-Rayet" is a proper name of a type of star. */
		 _("Wolf-Rayet"));
  for (unsigned int i = 0; i < 7; i++)
    drawlegenditem(sv, CLASS_COLORS[i], i, xbase + LO[i][0], LO[i][1], 5,
		   string() + SpecClass::int_to_class(i));
  drawlegenditem(sv, D_COLOR, 7, xbase + LO[7][0], LO[7][1], 3, "wd");
  drawlegenditem(sv, DEFAULT_COLOR, 8, xbase + LO[8][0], LO[8][1], 5,
		 _("Unknown"), -85, 5);
  drawlegenditem(sv, NON_STELLAR_COLOR, 9, xbase + LO[9][0], LO[9][1], 5,
		 _("Non-stellar"), -85, 5);
  return;
}


// public function to display all the stars in Array onto the pixmap,
//  as well as a grid outline and some other decorations

void StarArray::Display(StarViewer *sv) const
{
  int windowsize, pixelradius, wincenterX, wincenterY;
  double ChartDec = ArrayRules.ChartOrientation.getTheta();
  double starZ;

  // Determine radius and center of chart, in pixels
  windowsize = (sv->width() > sv->height()) ? sv->height() : sv->width();
  pixelradius = ROUND(0.4 * windowsize);
  wincenterX = sv->width() / 2;
  wincenterY = sv->height() / 2;

  // clear pixmap with the background color (defined in "specclass.h")
  sv->fill(BACKGROUND);

  // draw the hemisphere (N/S) of the grid facing AWAY from the screen
  if (ArrayRules.ChartGrid) {
    if (ChartDec >= 0.0)
      drawgridsouth(sv, wincenterX, wincenterY, pixelradius);
    else
      drawgridnorth(sv, wincenterX, wincenterY, pixelradius);
  }

  // Draw all the stars which are in the hemisphere of the chart facing
  //  away from the screen (e.g. the southern hemisphere for
  //  ArrayRules.ChartOrientation.getTheta() > 0)
  citerate (StarArray, Array, star_ptr) {
    starZ = (*star_ptr).GetStarXYZ().getZ() - ArrayRules.ChartLocation.getZ();
    if (starZ * ChartDec < 0.0)
      (*star_ptr).Display(ArrayRules, sv);
  }

  // Draw the chart equatorial plane and crosshairs
  if (ArrayRules.ChartGrid) 
    drawgridequator(sv, wincenterX, wincenterY, pixelradius);

  // Draw all the stars which are in the hemisphere of the chart facing
  //  toward the screen (e.g. the northern hemisphere for
  //  ArrayRules.ChartOrientation.getTheta() > 0)
  citerate (StarArray, Array, star_ptr) {
    starZ = (*star_ptr).GetStarXYZ().getZ() - ArrayRules.ChartLocation.getZ();
    if (starZ * ChartDec >= 0.0)
      (*star_ptr).Display(ArrayRules, sv);
  }

  // draw the hemisphere (N/S) of the grid facing TOWARD the screen
  if (ArrayRules.ChartGrid) {
    if (ChartDec < 0.0)
      drawgridsouth(sv, wincenterX, wincenterY, pixelradius);
    else
      drawgridnorth(sv, wincenterX, wincenterY, pixelradius);
  }

  // put in a legend, if desired
  if (ArrayRules.ChartLegend) drawlegend(sv);

  return;
}


// public function to plot the stars in the StarArray onto a
//  Hertzsprung-Russell diagram

void StarArray::Diagram(StarViewer *sv, double brightmag, double dimmag) const
{
  int width, height;
  double magscale = dimmag - brightmag;
  int scaling = (magscale < 0) ? -1 : 1;

  // if dimmag < brightmag, this will flip the vertical axis
  if (magscale < 0) 
    { double tmp = brightmag; brightmag = dimmag; dimmag = tmp; }

  // Determine size of HR diagram, in pixels, leaving a margin for axis labels
  width = sv->width() - 80;
  height = sv->height() - 60;

  // clear pixmap with the background color (defined in "star.h")
  sv->fill(BACKGROUND);

  // draw lines indicating the bright and dim magnitude cutoffs
  if (ArrayRules.ChartBrightestMagnitude > brightmag
      && ArrayRules.ChartBrightestMagnitude < dimmag) {
    int y = (magscale != 0.0) ? ROUND((ArrayRules.ChartBrightestMagnitude - 
				       brightmag) * height / magscale) : 0;
    if (magscale < 0) y += height;
    sv->setcolor(NEGATIVE);
    sv->drawline(60, y, width + 80, y);
    sv->drawline(60, y + 5 * scaling, 64, y);
    sv->drawline(60, y + 4 * scaling, 63, y);
  }
  if (ArrayRules.ChartDimmestMagnitude < dimmag
      && ArrayRules.ChartDimmestMagnitude > brightmag) {
    int y = (magscale != 0.0) ? ROUND((ArrayRules.ChartDimmestMagnitude - 
				       brightmag) * height / magscale) : 0;
    if (magscale < 0) y += height;
    sv->setcolor(POSITIVE);
    sv->drawline(60, y, width + 80, y);
    sv->drawline(60, y - 5 * scaling, 64, y);
    sv->drawline(60, y - 4 * scaling, 63, y);
  }

  // draw the chart axes
  sv->setcolor(LABEL_COLOR);
  sv->drawline(60, 0, 60, height);
  sv->drawline(60, height, width + 80, height);

  // Plot all the stars
  citerate (StarArray, Array, star_ptr) {
    int xposn, yposn;
    double mag = (*star_ptr).GetStarMagnitude();
    double numclass = SpecHash((*star_ptr).GetStarClass().classmajor());
    double numsubclass = (*star_ptr).GetStarClass().classminor();

    // don't plot stars which would be off-scale:
    if (mag > dimmag || mag < brightmag) continue;

    // don't plot non-stellar objects, white dwarfs, or objects without a
    //  known spectrum subclass (e.g. the '8' in "M8 V"):
    if (numclass >= 7.0 || numsubclass >= 10.0) continue; 

    numclass += 0.1 * numsubclass;
    xposn = ROUND(numclass / 7.0 * width);
    yposn = (magscale != 0.0) ? ROUND((mag - brightmag) * height / magscale):0;
    if (magscale < 0) yposn += height;

    // leave a margin for axis labels, hence the "+ 60"
    (*star_ptr).Draw(ArrayRules, sv, xposn + 60, yposn);
  }

  // Draw horizontal axis labels.  Start at x position 56 instead of 60 to 
  //  account for width of letters, so they look better centered.
  for (unsigned int i = 0; i < 7; i++) {
    sv->setcolor(ArrayRules.StarClasses[i] ? LABEL_COLOR : DIM_COLOR);
    sv->drawtext(SpecClass::int_to_class(i),
		 56 + width * (2 * i + 1) / 14, height + 20);
  }

  // Draw horizontal axis tickmarks.
  sv->setcolor(LABEL_COLOR);
  for (unsigned int i = 1; i <= 7; i++)
    sv->drawline(60 + i * width / 7, height - 2,
		 60 + i * width / 7, height + 2);

  // Draw vertical axis labels and tickmarks.
  for (int i = FLOOR(brightmag) + 1; i < dimmag; i++) {
    int y = (magscale != 0.0) ? ROUND((i - brightmag) * height / magscale) : 0;
    if (magscale < 0) y += height;
    if (i % 5)
      sv->drawline(58, y, 62, y);
    else
      sv->drawline(56, y, 64, y);
    if (magscale * scaling < 10 || ! (i % 5)) {
      string magtext = starstrings::itoa(i);
      sv->drawtext(magtext, 30, y + 4);
    }
  }

  // Draw axis titles.
  sv->setcolor(BOLD_COLOR);
  sv->drawtext(_("SPECTRAL CLASS"), 60 + width / 2 - 50, height + 50);

  if (magscale == 0.0) sv->setcolor(ERROR_COLOR);
  sv->drawtext_vertical(_(
    /* TRANSLATORS: "MAGNITUDE" refers to the astronomical brightness
       of an object. */
    "MAGNITUDE"), 10, height / 2 - 60);
      
  //for (unsigned int posn = 0; posn < strlen(maglabel); posn++) {
  //  int y = height / 2 - 60 + 16 * posn;
  //  sv->drawtext(maglabel[posn], 10, y);
  //}

  return;
}
