/*
 * Copyright (C) 2019 ~ 2019 Deepin Technology Co., Ltd.
 *
 * Author:
 *
 * Maintainer:
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef ARIA2CINTERFACE_H
#define ARIA2CINTERFACE_H

#include "aria2cconst.h"
#include "aria2cbtinfo.h"

#include <QObject>
#include <QMap>
#include <QVariant>
#include <QtNetwork/QNetworkAccessManager>
#include <QtNetwork/QNetworkRequest>
#include <QtNetwork/QNetworkReply>
#include <QJsonObject>
#include <QJsonDocument>
#include <QJsonArray>
#include <QtMath>

class Aria2cInterface : public QObject
{
    Q_OBJECT
public:
    explicit Aria2cInterface(QObject *parent = nullptr);

    /**
     * @brief killAria2cProc 强制停止aria2c进程
     * @return 返回kill exit code
     */
    int killAria2cProc();

    /**
     * @brief checkAria2cProc 检查nfs-aria2c进程
     * @return 返回值大于0表示进程已启动，等于0为未启动
     */
    int checkAria2cProc();

    /**
     * @brief startup
     * 启动aria2c。启动是会先检查是否存在进程，存在会kill掉之后启动新的aria2c进程
     * @return
     */
    bool startup();//启动

    /**
     * @brief shutdown 停止aria2c程序
     * 调用后会给aria2c发送shutdown消息。
     * signal_success信号中异步返回 "OK"
     * {id:"", jsonrpc:"2.0", result:"ok"}
     * QString ok = json.value("result").toString();
     *
     * @param id 可选，该参数用来唯一标示一次请求，异步返回的结果里会包含请求发出是指定的id，用以区分匹配多次请求。
     * 默认为method名，详见aria2cconst.h常量
     * QString id = json.value("id").toString();
     */
    void shutdown(QString id = "");

    /**
     * @brief forceShutdown 强制停止aria2c程序
     * @param id
     * 同shutdown，
     */
    void forceShutdown(QString id = "");

    /**
     * @brief setDefaultDownloadPath 设置默认下载路径，startup前调用有效
     * @param path
     */
    void setDefaultDownloadPath(QString path);

    /**
     * @brief getDefaultDownloadPath 获取默认下载链接
     * @return
     */
    QString getDefaultDownloadPath() {return this->defaultDownloadPath;}

    QString getConfigPath() const;

    /**
     * @brief setConfigPath 指定aria2c加载的配置文件，不设置使用默认配置文件
     * @param value
     */
    void setConfigPath(const QString &value);

public:
    /**
     * @brief addUri 添加下载 HTTP(S)/FTP/BitTorrent Magnet 链接 ，追加迅雷链接的支持
     * @param uri 链接
     * @param opt QMap<QString, QVariant> 下载参数，配置仅应用于本次下载
     * 参数详见aria2c文档
     * 示例代码
     * QMap<QString, QVariant> opt;
     * //dir 指定文件存储的路径
     * opt.insert("dir", dir);
     * //max-download-limit 指定最大下载速度（字节/秒）速度值可以追加K或者M
     * opt.insert("max-download-limit", "100K");
     *
     * @param id 可选，该参数用来唯一标示一次请求，异步返回的结果里会包含请求发出是指定的id，用以区分匹配多次请求。
     * 默认为method名，详见aria2cconst.h常量
     *
     * signal_success信号中异步返回本次下载的GID
     * {id:"", jsonrpc:"2.0", result:"gid"}
     * QString gId = json.value("result").toString();
     */
    void addUri(QString uri, QMap<QString, QVariant> opt, QString id = "");

    /**
     * @brief addTorrent 添加下载Torrent
     * @param torrentFile torrent种子文件路径
     * @param opt QMap<QString, QVariant> 下载参数，配置仅应用于本次下载
     * 参数详见aria2c文档
     * 示例代码
     * QMap<QString, QVariant> opt;
     * //dir 指定文件存储的路径
     * opt.insert("dir", dir);
     * //seed-time bt下载完成后做种上传的时间。默认为0不做种。如果不为0，下载会一直处于active状态儿不会完成。
     * opt.insert("seed-time", 0);
     * //max-upload-limit 指定最大上传速度（字节/秒）速度值可以追加K或者M
     * opt.insert("max-upload-limit", "100K");
     * //max-download-limit 指定最大下载速度（字节/秒）速度值可以追加K或者M
     *
     * @param id 可选，该参数用来唯一标示一次请求，异步返回的结果里会包含请求发出是指定的id，用以区分匹配多次请求。
     * 默认为method名，详见aria2cconst.h常量
     *
     * signal_success信号中异步返回本次下载的GID
     * {id:"", jsonrpc:"2.0", result:"gid"}
     * QString gId = json.value("result").toString();
     */
    void addTorrent(QString torrentFile, QMap<QString, QVariant> opt, QString id = "");

    /**
     * @brief addMetalink 添加Metalink
     * @param metalinkFile metalink种子文件路径
     * @param opt 同addTorrent
     * @param id 可选，该参数用来唯一标示一次请求，异步返回的结果里会包含请求发出是指定的id，用以区分匹配多次请求。
     * 默认为method名，详见aria2cconst.h常量
     *
     * signal_success信号中异步返回本次下载的GID
     * {id:"", jsonrpc:"2.0", result:"gid"}
     * QString gId = json.value("result").toString();
     */
    void addMetalink(QString metalinkFile, QMap<QString, QVariant> opt, QString id = "");

    /**
     * @brief pause 暂停指定下载 active/waiting状态将变为paused
     * @param gId GID
     * @param id 可选，该参数用来唯一标示一次请求，异步返回的结果里会包含请求发出是指定的id，用以区分匹配多次请求。
     * 默认为method名，详见aria2cconst.h常量
     *
     * signal_success信号中异步返回本次下载的GID
     * {id:"", jsonrpc:"2.0", result:"gid"}
     * QString gId = json.value("result").toString();
     */
    void pause(QString gId, QString id = "");

    /**
     * @brief forcePause 强制暂停 active/waiting状态将变为paused
     * @param gId GID aria2c生成的下载唯一标示
     * @param id 可选，该参数用来唯一标示一次请求，异步返回的结果里会包含请求发出是指定的id，用以区分匹配多次请求。
     * 默认为method名，详见aria2cconst.h常量
     *
     * signal_success信号中异步返回本次下载的GID
     * {id:"", jsonrpc:"2.0", result:"gid"}
     * QString gId = json.value("result").toString();
     */
    void forcePause(QString gId, QString id = "");

    /**
     * @brief pauseAll 全部暂停 active/waiting状态将变为paused
     * @param id 同其它方法参数
     *
     * signal_success信号中异步返回 "OK"
     * {id:"", jsonrpc:"2.0", result:"ok"}
     * QString ok = json.value("result").toString();
     */
    void pauseAll(QString id = "");//

    /**
     * @brief forcePauseAll 强制全部暂停
     * @param id 同其它方法参数
     *
     * signal_success信号中异步返回 "OK"
     * {id:"", jsonrpc:"2.0", result:"ok"}
     * QString ok = json.value("result").toString();
     */
    void forcePauseAll(QString id = "");

    /**
     * @brief unpause 恢复指定GID下载 paused状态变为waiting
     * @param gId GID aria2c生成的下载唯一标示
     * @param id 同其它方法参数
     *
     * signal_success信号中异步返回本次下载的GID
     * {id:"", jsonrpc:"2.0", result:"gid"}
     * QString gId = json.value("result").toString();
     */
    void unpause(QString gId, QString id = "");

    /**
     * @brief unpauseAll 全部恢复下载
     * @param id 同其它方法参数
     *
     * signal_success信号中异步返回 "OK"
     * {id:"", jsonrpc:"2.0", result:"ok"}
     * QString ok = json.value("result").toString();
     */
    void unpauseAll(QString id = "");

    /**
     * @brief remove 移除指定下载。下载状态会变为removed。
     * 注意，实验表明并不会删除下载的文件。
     * @param gId GID aria2c生成的下载唯一标示
     * @param id 同其它方法参数
     *
     * signal_success信号中异步返回本次下载的GID
     * {id:"", jsonrpc:"2.0", result:"gid"}
     * QString gId = json.value("result").toString();
     */
    void remove(QString gId, QString id = "");

    /**
     * @brief forceRemove 强制移除,不会做后处理
     * @param gId GID aria2c生成的下载唯一标示
     * @param id 同其它方法参数
     *
     * signal_success信号中异步返回本次下载的GID
     * {id:"", jsonrpc:"2.0", result:"gid"}
     * QString gId = json.value("result").toString();
     */
    void forceRemove(QString gId, QString id = "");

    /**
     * @brief tellStatus 获取下载状态
     * @param gId GID aria2c生成的下载唯一标示
     * @param id 同其它方法参数
     *
     * signal_success信号中异步返回指定下载的当前状态
     * result为一个JsonObject，包含类似map键值，详见aria2文档
     */
    void tellStatus(QString gId, QString id = "");

    /**
     * @brief tellStatus 获取下载状态 可指定需要的返回字段
     * @param gId GID aria2c生成的下载唯一标示
     * @param keys 指定需要的返回字段列表，参数详见aria2文档
     * [status|totalLength|completedLength|uploadLength|bitfield|downloadSpeed|uploadSpeed|...]
     * 示例代码：
     * QStringList keys;
     * keys<<"gid"<<"status"<<"totalLength"<<"completedLength"<<"downloadSpeed";
     * @param id 同其它方法参数
     *
     * signal_success信号中异步返回指定下载的当前状态
     * result为一个JsonObject，包含类似map键值，详见aria2文档
     */
    void tellStatus(QString gId, QStringList keys, QString id = "");
    /**
     * @brief getUris 获取指定下载的链接
     * @param gId
     * @param id
     *
     * signal_success信号中异步返回,参数详见aria2文档
     */
    void getUris(QString gId, QString id = "");

    /**
     * @brief getFiles 获取文件列表
     * @param gId
     * @param id
     *
     * signal_success信号中异步返回,参数详见aria2文档
     */
    void getFiles(QString gId, QString id = "");

    /**
     * @brief getPeers 获取torrent下载的Peers
     * @param gId
     * @param id
     *
     * signal_success信号中异步返回,参数详见aria2文档
     */
    void getPeers(QString gId, QString id = "");

    /**
     * @brief getServers 获取指定下载服务器信息
     * @param gId
     * @param id
     *
     * signal_success信号中异步返回,参数详见aria2文档
     */
    void getServers(QString gId, QString id = "");

    /**
     * @brief tellActive 查询下载中在项 （active状态）
     * @param keys 参见tellStatus参数
     * @param id
     *
     * signal_success信号中异步返回,参数详见aria2文档
     */
    void tellActive(QStringList keys, QString id = "");

    /**
     * @brief tellWaiting 查询等待下载项（包括paused和waiting状态）
     * @param offset 下载列表偏移索引
     * @param num 读取数量
     * @param keys 参见tellStatus参数
     * @param id
     *
     * signal_success信号中异步返回,参数详见aria2文档
     */
    void tellWaiting(int offset, int num, QStringList keys, QString id = "");

    /**
     * @brief tellStopped 查询已停止在下载项
     * @param offset 下载列表偏移索引
     * @param num 读取数量
     * @param keys 参见tellStatus参数
     * @param id
     *
     * signal_success信号中异步返回,参数详见aria2文档
     */
    void tellStopped(int offset, int num, QStringList keys, QString id = "");

    /**
     * @brief changePosition 变更下载列表的位置
     * @param gId
     * @param pos 位置索引
     * @param how [RIA2C_PARAM_POS_SET|RIA2C_PARAM_POS_CUR|RIA2C_PARAM_POS_END]
     * @param id
     *
     * signal_success信号中异步返回变更后的索引位置
     */
    void changePosition(QString gId, int pos, QString how, QString id = "");

    /**
     * @brief changeUri 变更指定下载项下载地址
     * @param gId
     * @param fileIndex 文件索引
     * @param delUris 要移除的uri列表
     * @param addUris 要添加的uri列表
     * @param id
     *
     * 详见aria2c文档
     */
    void changeUri(QString gId, int fileIndex, QStringList delUris, QStringList addUris, QString id = "");

    /**
     * @brief getOption 获取指定下载项在配置
     * @param gId
     * @param id
     *
     * signal_success信号中异步返回该下载的所有配置项，参数结构见aria2文档
     */
    void getOption(QString gId, QString id = "");

    /**
     * @brief changeOption 变更指定下载项在配置
     * @param gId
     * @param options 配置项键值map
     * @param id
     *
     * signal_success信号中异步返回，参数结构见aria2文档
     */
    void changeOption(QString gId, QMap<QString, QVariant> options, QString id = "");

    /**
     * @brief getGlobalOption 获取全局配置
     * @param id
     *
     * signal_success信号中异步返回，参数结构见aria2文档
     */
    void getGlobalOption(QString id = "");

    /**
     * @brief changeGlobalOption 变更全局配置
     * @param options 配置项键值map
     * @param id
     *
     * signal_success信号中异步返回，参数结构见aria2文档
     */
    void changeGlobalOption(QMap<QString, QVariant> options, QString id = "");

    /**
     * @brief getGlobalStat 获取全局下载状态
     * @param id
     *
     * signal_success信号中异步返回，参数结构见aria2文档
     */
    void getGlobalStat(QString id = "");

    /**
     * @brief purgeDownloadResult 清除已下载结果（包括已完成/下载错误/已移出）,不会删除文件
     * @param id
     *
     * signal_success信号中异步返回 OK
     */
    void purgeDownloadResult(QString id = "");

    /**
     * @brief removeDownloadResult 移出指定下载项下载结果（包括已完成/下载错误/已移出）,不会删除文件
     * @param gId
     * @param id
     *
     * signal_success信号中异步返回 OK
     */
    void removeDownloadResult(QString gId, QString id = "");

    /**
     * @brief getVersion 获取aria2程序版本信息
     * @param id
     *
     * signal_success信号中异步返回
     */
    void getVersion(QString id = "");

    /**
     * @brief getSessionInfo 获取SessionID（每次调用都会产生一个Session ID？）
     * @param id
     *
     * signal_success信号中异步返回
     */
    void getSessionInfo(QString id = "");

    /**
     * @brief saveSession 将当前session信息写入到--save-session指定的文件中
     */
    void saveSession();

    /**
     * @brief multicall 多方法调用
     * @param params
     * [
     *   [{
     *      methodName:'aria2.addUri',
     *      params:[
     *          [...]
     *      ]
     *    },
     *    {
     *
     *    },
     *    ....
     *   ]
     * ]
     * @param id
     */
    void multicall(QJsonArray params, QString id = "");//批量调用多个方法

    /**
     * @brief listMethods 列出所有RPC支持的方法
     */
    void listMethods();

    /**
     * @brief listNotifications 列出所有有效的RPC通知名称
     */
    void listNotifications();

private:
    QString fileToBase64(QString filePath);//文件转base64
    bool checkAria2cFile();//检查aria2c程序

    void sendMessage(QJsonObject jsonObj, const QString &method);//发送请求

    void callRPC(QString method, QJsonArray params, QString id = "");//
    void callRPC(QString method, QString id = "");//

    QString processThunderUri(QString thunder);//如果是迅雷链接会解密处理，否则原样返回

private:
    //QString cmd = "/usr/bin/aria2c";//aria2c程序路径 -> 已改成public static变量
    QString rpcPort = "16800";//端口
    QString rpcServer = "http://localhost:" + rpcPort + "/jsonrpc";//rpc地址
    QString defaultDownloadPath;//默认下载路径
    QString configPath = "";

signals:
    /**
     * @brief signal_success 请求成功的信号
     * @param method 请求的方法名 见aria2cconst.h
     * @param json
     */
    void signal_success(QString method, QJsonObject json);
    /**
     * @brief signal_error 请求失败的信号
     * @param method 请求的方法名 见aria2cconst.h
     * @param errCode 错误代码
     * 400 一般为参数错误，比如GID不存在
     * 0   aria2c进程没有启动
     */
    void signal_error(QString method, QString id, int errCode);

public slots:
    /**
     * @brief rpcRequestReply 请求返回的回调槽
     * @param reply
     * @param method
     */
    void rpcRequestReply(QNetworkReply *reply, const QString &method, const QString id);

public:
    static const QString aria2cCmd;//aria2c程序路径
    static const QString basePath;//下载器安装目录

    /**
     * @brief getBtInfo  获取torrent信息
     * @param torrentPath
     * @return Aria2cBtInfo
     */
    static Aria2cBtInfo getBtInfo(QString torrentPath);

    /**
     * @brief bytesFormat 格式化字节
     * @param size
     * @return
     */
    static QString bytesFormat(qint64 size);

    /**
     * @brief getCapacityFree 获取指定路径所在分区的磁盘剩余容量
     * @param path
     * @return
     */
    static QString getCapacityFree(QString path);

    /**
     * @brief getCapacityFreeByte 获取指定路径所在分区的磁盘剩余容量(字节)
     * @param path
     * @return
     */
    static long getCapacityFreeByte(QString path);
};

#endif // ARIA2CINTERFACE_H
