/***************************************************************
 * Name:      engine.cpp
 * Author:    David Vachulka (arch_dvx@users.sourceforge.net)
 * Copyright: 2013
 * License:   GPL3
 **************************************************************/

#include <wx/datetime.h>
#include <wx/stdpaths.h>
#include <wx/tokenzr.h>
#include <map>
#include "engine.h"
#include "sqlite3.h"
#include "dxsettings.h"
#include "dxutils.h"

wxDEFINE_EVENT(ENGINE_UPDATED, wxCommandEvent);
wxDEFINE_EVENT(ENGINE_MSG, wxCommandEvent);

Engine *Engine::m_this = nullptr;

bool sortEvents(Event i, Event j)
{
    if(i.realEventDate().IsSameDate(j.realEventDate()))
    {
        if((i.recurrence()==R_ONCE||i.recurrence()==R_ONCENOTDELETE) && j.recurrence()!=R_ONCE && j.recurrence()!=R_ONCENOTDELETE)
            return true;
    }
    return i.realEventDate().IsEarlierThan(j.realEventDate());
}

bool sortEventsView(EventView i, EventView j)
{
    if((i.recurrencetype()==R_NONE||i.recurrencetype()==R_ONCENOTDELETE) && (j.recurrencetype()==R_NONE||j.recurrencetype()==R_ONCENOTDELETE))
    {
        if(dxutils::textToInt(i.name()) && dxutils::textToInt(j.name()))
            return dxutils::textToInt(i.name()) < dxutils::textToInt(j.name());
        if(i.name().CmpNoCase(j.name()) < 0) return true;
        else return false;
    }
    if((i.recurrencetype()==R_NONE||i.recurrencetype()==R_ONCENOTDELETE) && !(j.recurrencetype()==R_NONE||j.recurrencetype()==R_ONCENOTDELETE))
    {
        return true;
    }
    return i.date().IsEarlierThan(j.date());
}

bool sortHoliday(Holiday i, Holiday j)
{
    return i.month()*100+i.day()<j.month()*100+j.day();
}

bool sortCalendarday(Calendarday i, Calendarday j)
{
    return  i.day() < j.day();
}

int rdn(const wxDateTime &date)  /* Rata Die day one is 0001-01-01 */
{
    int y = date.GetYear();
    int m = date.GetMonth()+1;
    int d = date.GetDay();
    if(m < 3)
        y--, m += 12;
    return 365*y + y/4 - y/100 + y/400 + (153*m - 457)/5 + d - 306;
}

Engine::~Engine()
{
    try
    {
        m_db.Close();
    }
    catch(const wxSQLite3Exception& e)
    {
        wxLogError(wxString::Format("Close database error: %s", e.GetMessage().c_str()));
    }
    m_this = nullptr;
}

Engine *Engine::instance()
{
    return m_this;
}

bool Engine::loadData()
{
    if(m_db.IsOpen())
    {
        try
        {
            m_db.Close();
        }
        catch(const wxSQLite3Exception& e)
        {
            wxLogError(wxString::Format("Close database error: %s", e.GetMessage().c_str()));
            return false;
        }
    }
    try
    {
        if(dxsettings.encrypt()) m_db.Open(wxStandardPaths::Get().GetUserDataDir()+wxFILE_SEP_PATH+"reminders.db", dxsettings.pk());
        else m_db.Open(wxStandardPaths::Get().GetUserDataDir()+wxFILE_SEP_PATH+"reminders.db");
    }
    catch(const wxSQLite3Exception &e)
    {
        wxLogError(wxString::Format("Open database error: %s", e.GetMessage()));
        return false;
    }
    //Try if db selectable
    try
    {
        m_db.ExecuteScalar("select count(*) from  sqlite_master where type='table'");
    }
    catch(const wxSQLite3Exception &e)
    {
        wxLogError(wxString::Format("Open database error: %s", e.GetMessage()));
        return false;
    }
    createEventsTable();
    createNoteTable();
    createHolidaysTable();
    try
    {
        if(!hasTableColumn(&m_db,"EVENTS","EVENTID") || !hasTableColumn(&m_db,"EVENTS","EVENT") || !hasTableColumn(&m_db,"EVENTS","DESCRIPTION") || !hasTableColumn(&m_db,"EVENTS","EVENTDATE")
                || !hasTableColumn(&m_db,"EVENTS","REMINDER") || !hasTableColumn(&m_db,"EVENTS","ACTUALREMINDER") || !hasTableColumn(&m_db,"EVENTS","REMINDED"))
            //1.10.1 db schema
        {
            fireMsg(_("Couldn't open database file,\nsome columns are missing at database"));
            return false;
        }
        if(!hasTableColumn(&m_db,"EVENTS", "DAY"))
        {
            tryExecuteUpdate(&m_db, "alter table EVENTS add column DAY integer not null default 0");
            tryExecuteUpdate(&m_db, "update EVENTS set DAY=cast(strftime('%d',EVENTDATE) as integer)");
        }
        if(!hasTableColumn(&m_db,"EVENTS", "MONTH"))
        {
            tryExecuteUpdate(&m_db, "alter table EVENTS add column MONTH integer not null default 0");
            tryExecuteUpdate(&m_db, "update EVENTS set MONTH=cast(strftime('%m',EVENTDATE) as integer)");
        }
        if(!hasTableColumn(&m_db,"EVENTS", "HIDDEN"))
        {
            tryExecuteUpdate(&m_db, "alter table EVENTS add column HIDDEN integer not null default 0");
            tryExecuteUpdate(&m_db, "update EVENTS set HIDDEN=0");
        }
        if(!hasTableColumn(&m_db,"EVENTS", "ALWAYS"))
        {
            tryExecuteUpdate(&m_db, "alter table EVENTS add column ALWAYS integer not null default 0");
            tryExecuteUpdate(&m_db, "update EVENTS set ALWAYS=0");
        }
        if(!hasTableColumn(&m_db,"EVENTS", "COLOR"))
        {
            tryExecuteUpdate(&m_db, "alter table EVENTS add column COLOR integer not null default 0");
            tryExecuteUpdate(&m_db, "update EVENTS set COLOR=0");
        }
        if(!hasTableColumn(&m_db,"EVENTS", "RECURRENCE"))
        {
            tryExecuteUpdate(&m_db, "alter table EVENTS add column RECURRENCE integer not null default 0");
            tryExecuteUpdate(&m_db, "update EVENTS set RECURRENCE=0");
        }
        if(!hasTableColumn(&m_db,"EVENTS", "MINUTES"))
        {
            tryExecuteUpdate(&m_db, "alter table EVENTS add column MINUTES integer not null default 0");
            tryExecuteUpdate(&m_db, "update EVENTS set MINUTES=0");
        }
        if(!hasTableColumn(&m_db,"EVENTS", "HOURS"))
        {
            tryExecuteUpdate(&m_db, "alter table EVENTS add column HOURS integer not null default 0");
            tryExecuteUpdate(&m_db, "update EVENTS set HOURS=0");
        }
        if(!hasTableColumn(&m_db,"EVENTS", "DAYS"))
        {
            tryExecuteUpdate(&m_db, "alter table EVENTS add column DAYS integer not null default 0");
            tryExecuteUpdate(&m_db, "update EVENTS set DAYS=0");
        }
        if(!hasTableColumn(&m_db,"EVENTS", "MONTHS"))
        {
            tryExecuteUpdate(&m_db, "alter table EVENTS add column MONTHS integer not null default 0");
            tryExecuteUpdate(&m_db, "update EVENTS set MONTHS=0");
        }
        if(!hasTableColumn(&m_db,"EVENTS", "MDAY"))
        {
            tryExecuteUpdate(&m_db, "alter table EVENTS add column MDAY integer not null default 0");
            tryExecuteUpdate(&m_db, "update EVENTS set MDAY=0");
        }
        if(!hasTableColumn(&m_db,"EVENTS","MWEEK"))
        {
            tryExecuteUpdate(&m_db, "alter table EVENTS add column MWEEK integer not null default 0");
            tryExecuteUpdate(&m_db, "update EVENTS set MWEEK=0");
        }
        if(!hasTableColumn(&m_db,"EVENTS","UNTIL"))
        {
            tryExecuteUpdate(&m_db, wxString::Format("alter table EVENTS add column UNTIL text not null default '%s'", INFINITY_DATE.FormatISODate()));
        }
        if(m_db.TableExists("RECURRENCE")) //1.10.1 db schema
        {
            if(hasTableColumn(&m_db,"RECURRENCE","EVENTID") && hasTableColumn(&m_db,"RECURRENCE","TYPE") && hasTableColumn(&m_db,"RECURRENCE","DAYS") && hasTableColumn(&m_db,"RECURRENCE","MONTHS"))
            {
                wxSQLite3ResultSet q = m_db.ExecuteQuery("select EVENTID,TYPE,DAYS,MONTHS from RECURRENCE");
                while(q.NextRow())
                {
                    tryExecuteUpdate(&m_db, wxString::Format("UPDATE EVENTS set RECURRENCE=%d,DAYS=%d,MONTHS=%d,MDAY=%d,MWEEK=%d WHERE EVENTID=%lld", q.GetInt(1),q.GetInt(2),q.GetInt(3),
                                                             0,0,q.GetInt64(0).GetValue()));
                }
                q.Finalize();
            }
            tryExecuteUpdate(&m_db, "DROP TABLE RECURRENCE");
        }
        wxSQLite3ResultSet q = m_db.ExecuteQuery("select * from EVENTS");
        while(q.NextRow())
        {
            Event ev(q.GetInt64("EVENTID").GetValue(), q.GetString("EVENT"), q.GetString("DESCRIPTION"),
                     q.GetDateTime("EVENTDATE"), q.GetInt64("REMINDER").GetValue(),
                     q.GetBool("REMINDED"), q.GetInt("DAY"), q.GetInt("MONTH"), q.GetBool("HIDDEN"), q.GetInt("RECURRENCE"), q.GetInt("MINUTES"),
                     q.GetInt("HOURS"), q.GetInt("DAYS"), q.GetInt("MONTHS"), q.GetInt("MDAY"), q.GetInt("MWEEK"), q.GetInt64("ACTUALREMINDER").GetValue(),
                     q.GetDate("UNTIL"), q.GetBool("ALWAYS"), q.GetInt("COLOR"));
            m_events.push_back(ev);
        }
        q.Finalize();
        if(!hasTableColumn(&m_db,"HOLIDAYS", "REMINDED"))
        {
            tryExecuteUpdate(&m_db, "alter table HOLIDAYS add column REMINDED integer not null default 0");
            tryExecuteUpdate(&m_db, "update HOLIDAYS set REMINDED=0");
        }
    }
    catch(const wxSQLite3Exception& e)
    {
        wxLogError(wxString::Format("Error in loadData: %s", e.GetMessage().c_str()));
        return false;
    }
    m_lastid = -1;
    fireUpdated();
    return true;
}

void Engine::loadCalendarData(bool showhidden)
{
    updateCalendarDays(wxDateTime::GetCurrentMonth(), wxDateTime::GetCurrentYear(), showhidden);
}

bool Engine::overwriteEvents(const wxString &filename, bool encypted, const wxString &key)
{
    wxSQLite3Database od;
    try
    {
        if(encypted) od.Open(filename, key);
        else od.Open(filename);
    }
    catch(const wxSQLite3Exception &e)
    {
        wxLogError(wxString::Format("Open database error: %s", e.GetMessage()));
        return false;
    }
    //Try if db selectable
    try
    {
        od.ExecuteScalar("select count(*) from  sqlite_master where type='table'");
    }
    catch(const wxSQLite3Exception &e)
    {
        wxLogError(wxString::Format("Open database error: %s", e.GetMessage()));
        return false;
    }
    if(!checkTableEvents(&od))
    {
        od.Close();
        return false;
    }
    std::vector<Event> events;
    try
    {
        wxString msg = wxEmptyString;
        wxSQLite3ResultSet q;
        q = od.ExecuteQuery("select * from EVENTS");
        while(q.NextRow())
        {
            if(!q.GetDateTime("EVENTDATE").IsValid())
            {
                msg << wxString::Format(_("Event %s (ID=%lld) has invalid date and time"),q.GetString("EVENT"),q.GetInt64("EVENTID").GetValue()) << "\n";
                continue;
            }
            if(!q.GetDate("UNTIL").IsValid())
            {
                msg << wxString::Format(_("Event %s (ID=%lld) has invalid date for until"),q.GetString("EVENT"),q.GetInt64("EVENTID").GetValue()) << "\n";
                continue;
            }
            Event ev(q.GetInt64("EVENTID").GetValue(), q.GetString("EVENT"), q.GetString("DESCRIPTION"),
                     q.GetDateTime("EVENTDATE"), q.GetInt64("REMINDER").GetValue(),
                     q.GetInt("RECURRENCE")==R_NONE?true:q.GetBool("REMINDED"), q.GetInt("DAY"), q.GetInt("MONTH"), q.GetBool("HIDDEN"), q.GetInt("RECURRENCE"), q.GetInt("MINUTES"),
                     q.GetInt("HOURS"), q.GetInt("DAYS"), q.GetInt("MONTHS"), q.GetInt("MDAY"), q.GetInt("MWEEK"), q.GetInt64("ACTUALREMINDER").GetValue(),
                     q.GetDate("UNTIL"),
                     q.GetBool("ALWAYS"),
                     q.GetInt("COLOR"));
            events.push_back(ev);
        }
        q.Finalize();
        if(!msg.IsEmpty()) fireMsg(msg);
        clearDB();
        m_events.clear();
        for(size_t i=0; i<events.size(); i++)
        {
            wxSQLite3Statement st = m_db.PrepareStatement("insert into EVENTS (EVENTID, EVENT, DESCRIPTION, EVENTDATE, REMINDER, ACTUALREMINDER, REMINDED, DAY, HIDDEN, MONTH,"
                                                          "RECURRENCE, MINUTES, HOURS,"
                                                          "DAYS, MONTHS, MDAY, MWEEK, UNTIL, ALWAYS, COLOR) "
                                                          "values (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)");
            st.Bind(1, wxLongLong(events[i].id()));
            st.Bind(2, events[i].event());
            st.Bind(3, events[i].descr());
            st.BindDateTime(4, events[i].date());
            st.Bind(5, wxLongLong(events[i].reminder()));
            st.Bind(6, wxLongLong(events[i].reminder()));
            st.BindBool(7, events[i].reminded());
            st.Bind(8, events[i].day());
            st.BindBool(9, events[i].hidden());
            st.Bind(10, events[i].month());
            st.Bind(11, events[i].recurrence());
            st.Bind(12, events[i].minutes());
            st.Bind(13, events[i].hours());
            st.Bind(14, events[i].days());
            st.Bind(15, events[i].months());
            st.Bind(16, events[i].monthlyday());
            st.Bind(17, events[i].monthlyweek());
            st.BindDate(18, events[i].until());
            st.BindBool(19, events[i].always());
            st.Bind(20, events[i].color());
            st.ExecuteUpdate();
            st.Finalize();
            m_events.push_back(events[i]);
        }
        od.Close();
    }
    catch(const wxSQLite3Exception& e)
    {
        wxLogError(wxString::Format("Error in overwriteEvents: %s", e.GetMessage().c_str()));
        return false;
    }
    m_lastid = -1;
    fireUpdated();
    return true;
}

bool Engine::mergeEvents(const wxString &filename, bool encypted, const wxString &key)
{
    wxSQLite3Database md;
    try
    {
        if(encypted) md.Open(filename, key);
        else md.Open(filename);
    }
    catch(const wxSQLite3Exception &e)
    {
        wxLogError(wxString::Format("Open database error: %s", e.GetMessage()));
        return false;
    }
    //Try if db selectable
    try
    {
        md.ExecuteScalar("select count(*) from  sqlite_master where type='table'");
    }
    catch(const wxSQLite3Exception &e)
    {
        wxLogError(wxString::Format("Open database error: %s", e.GetMessage()));
        return false;
    }
    if(!checkTableEvents(&md))
    {
        md.Close();
        return false;
    }
    m_mergeEvents.clear();
    try
    {
        wxString msg = wxEmptyString;
        wxSQLite3ResultSet q = md.ExecuteQuery("select * from EVENTS");
        while(q.NextRow())
        {
            if(!q.GetDateTime("EVENTDATE").IsValid())
            {
                msg << wxString::Format(_("Event %s (ID=%lld) has invalid date and time"),q.GetString("EVENT"),q.GetInt64("EVENTID").GetValue()) << "\n";
                continue;
            }
            if(!q.GetDate("UNTIL").IsValid())
            {
                msg << wxString::Format(_("Event %s (ID=%lld) has invalid date for until"),q.GetString("EVENT"),q.GetInt64("EVENTID").GetValue()) << "\n";
                continue;
            }
            Event ev(q.GetInt64("EVENTID").GetValue(), q.GetString("EVENT"), q.GetString("DESCRIPTION"),
                     q.GetDateTime("EVENTDATE"), q.GetInt64("REMINDER").GetValue(),
                     q.GetInt("RECURRENCE")==R_NONE?true:q.GetBool("REMINDED"), q.GetInt("DAY"), q.GetInt("MONTH"), q.GetBool("HIDDEN"), q.GetInt("RECURRENCE"), q.GetInt("MINUTES"),
                     q.GetInt("HOURS"), q.GetInt("DAYS"), q.GetInt("MONTHS"), q.GetInt("MDAY"), q.GetInt("MWEEK"), q.GetInt64("ACTUALREMINDER").GetValue(),
                     q.GetDate("UNTIL"),
                     q.GetBool("ALWAYS"),
                     q.GetInt("COLOR"));
            m_mergeEvents.push_back(ev);
        }
        if(!msg.IsEmpty()) fireMsg(msg);
        q.Finalize();
        std::sort(m_mergeEvents.begin(), m_mergeEvents.end(), sortEvents);
    }
    catch(const wxSQLite3Exception& e)
    {
        wxLogError(wxString::Format("Error in mergeEvents: %s", e.GetMessage().c_str()));
        return false;
    }
    return true;
}

bool Engine::overwriteHolidays(const wxString &filename, bool encypted, const wxString &key)
{
    wxSQLite3Database od;
    try
    {
        if(encypted) od.Open(filename, key);
        else od.Open(filename);
    }
    catch(const wxSQLite3Exception &e)
    {
        wxLogError(wxString::Format("Open database error: %s", e.GetMessage()));
        return false;
    }
    //Try if db selectable
    try
    {
        od.ExecuteScalar("select count(*) from  sqlite_master where type='table'");
    }
    catch(const wxSQLite3Exception &e)
    {
        wxLogError(wxString::Format("Open database error: %s", e.GetMessage()));
        return false;
    }
    if(!checkTableHolidays(&od))
    {
        od.Close();
        return false;
    }
    std::vector<Holiday> holidays;
    try
    {
        wxSQLite3ResultSet q = od.ExecuteQuery("select * from HOLIDAYS");
        while(q.NextRow())
        {
            Holiday ho(q.GetInt64("HOLIDAYID").GetValue(), q.GetString("NAME"), q.GetInt("DAY"), static_cast<wxDateTime::Month>(q.GetInt("MONTH")), q.GetInt("REMINDED"));
            holidays.push_back(ho);
        }
        q.Finalize();
        od.Close();
    }
    catch(const wxSQLite3Exception& e)
    {
        wxLogError(wxString::Format("Error in overwriteHolidays: %s", e.GetMessage().c_str()));
        return false;
    }
    setHolidays(holidays);
    return true;
}

bool Engine::overwriteNote(const wxString &filename, bool encypted, const wxString &key)
{
    wxSQLite3Database od;
    try
    {
        if(encypted) od.Open(filename, key);
        else od.Open(filename);
    }
    catch(const wxSQLite3Exception &e)
    {
        wxLogError(wxString::Format("Open database error: %s", e.GetMessage()));
        return false;
    }
    //Try if db selectable
    try
    {
        od.ExecuteScalar("select count(*) from  sqlite_master where type='table'");
    }
    catch(const wxSQLite3Exception &e)
    {
        wxLogError(wxString::Format("Open database error: %s", e.GetMessage()));
        return false;
    }
    if(!checkTableNote(&od))
    {
        od.Close();
        return false;
    }
    wxString note;
    try
    {
        static const char sql[] = "select NTEXT from NOTE";
        wxSQLite3ResultSet q = od.ExecuteQuery(sql);
        for( ; q.NextRow(); )
        {
            note = q.GetString(0);
        }
        q.Finalize();
        od.Close();
    }
    catch(const wxSQLite3Exception& e)
    {
        wxLogError(wxString::Format("Error in overwriteNote: %s", e.GetMessage().c_str()));
        return false;
    }
    updateNote(note);
    return true;
}

bool Engine::mergeAllEvents(const wxString &filename, bool encypted, const wxString &key)
{
    wxSQLite3Database md;
    try
    {
        if(encypted) md.Open(filename, key);
        else md.Open(filename);
    }
    catch(const wxSQLite3Exception &e)
    {
        wxLogError(wxString::Format("Open database error: %s", e.GetMessage()));
        return false;
    }
    //Try if db selectable
    try
    {
        md.ExecuteScalar("select count(*) from  sqlite_master where type='table'");
    }
    catch(const wxSQLite3Exception &e)
    {
        wxLogError(wxString::Format("Open database error: %s", e.GetMessage()));
        return false;
    }
    if(!checkTableEvents(&md))
    {
        md.Close();
        return false;
    }
    std::vector<Event> mergeEvents;
    try
    {
        wxString msg = wxEmptyString;
        wxSQLite3ResultSet q = md.ExecuteQuery("select * from EVENTS");
        while(q.NextRow())
        {
            if(!q.GetDateTime("EVENTDATE").IsValid())
            {
                msg << wxString::Format(_("Event %s (ID=%lld) has invalid date and time"),q.GetString("EVENT"),q.GetInt64("EVENTID").GetValue()) << "\n";
                continue;
            }
            if(!q.GetDate("UNTIL").IsValid())
            {
                msg << wxString::Format(_("Event %s (ID=%lld) has invalid date for until"),q.GetString("EVENT"),q.GetInt64("EVENTID").GetValue()) << "\n";
                continue;
            }
            Event ev(q.GetInt64("EVENTID").GetValue(), q.GetString("EVENT"), q.GetString("DESCRIPTION"),
                     q.GetDateTime("EVENTDATE"), q.GetInt64("REMINDER").GetValue(),
                     q.GetInt("RECURRENCE")==R_NONE?true:q.GetBool("REMINDED"), q.GetInt("DAY"), q.GetInt("MONTH"), q.GetBool("HIDDEN"), q.GetInt("RECURRENCE"), q.GetInt("MINUTES"),
                     q.GetInt("HOURS"), q.GetInt("DAYS"), q.GetInt("MONTHS"), q.GetInt("MDAY"), q.GetInt("MWEEK"), q.GetInt64("ACTUALREMINDER").GetValue(),
                     q.GetDate("UNTIL"),
                     q.GetBool("ALWAYS"),
                     q.GetInt("COLOR"));
            mergeEvents.push_back(ev);
        }
        q.Finalize();
        if(!msg.IsEmpty()) fireMsg(msg);
        for(size_t i=0; i<mergeEvents.size(); i++)
        {
            addEvent(mergeEvents[i], false);
        }
    }
    catch(const wxSQLite3Exception& e)
    {
        wxLogError(wxString::Format("Error in mergeAllEvents: %s", e.GetMessage().c_str()));
        return false;
    }
    m_lastid = -1;
    fireUpdated();
    return true;
}

std::vector<Holiday> Engine::holidays()
{
    std::vector<Holiday> holidays;
    wxSQLite3ResultSet rs = m_db.ExecuteQuery("select * from HOLIDAYS");
    while(rs.NextRow())
    {
        holidays.push_back(Holiday(rs.GetInt64("HOLIDAYID").GetValue(), rs.GetString("NAME"), rs.GetInt("DAY"), static_cast<wxDateTime::Month>(rs.GetInt("MONTH")), rs.GetInt("REMINDED")));
    }
    rs.Finalize();
    return holidays;
}

void Engine::setHolidays(const std::vector<Holiday> &holidays)
{
    tryExecuteUpdate(&m_db, "delete from HOLIDAYS");
    std::vector<Holiday>::const_iterator it;
    for(it = holidays.begin(); it != holidays.end(); ++it)
    {
        try
        {
            static const char sql[] =
                "insert into HOLIDAYS ("
                "HOLIDAYID, NAME, DAY, MONTH, REMINDED)"
                "values (?, ?, ?, ?, ?)";
            wxSQLite3Statement st = m_db.PrepareStatement(sql);
            st.Bind(1, wxLongLong((*it).id()));
            st.Bind(2, (*it).name());
            st.Bind(3, (*it).day());
            st.Bind(4, (*it).month());
            st.Bind(5, (*it).reminded());
            st.ExecuteUpdate();
            st.Finalize();
        }
        catch(const wxSQLite3Exception& e)
        {
            wxLogError(wxString::Format("Error in setHolidays: %s", e.GetMessage().c_str()));
        }
    }
    m_lastid = -1;
    fireUpdated();
}

Holiday Engine::addHoliday(const wxString &name, int day, wxDateTime::Month month)
{
    try
    {
        static const char sql[] =
            "insert into HOLIDAYS ("
            "NAME, DAY, MONTH, REMINDED)"
            "values (?, ?, ?, ?)";
        wxSQLite3Statement st = m_db.PrepareStatement(sql);
        st.Bind(1, name);
        st.Bind(2, day);
        st.Bind(3, month);
        st.Bind(4, wxDateTime::GetCurrentYear()-1);
        st.ExecuteUpdate();
        wxInt64 id = m_db.GetLastRowId().GetValue();
        st.Finalize();
        return Holiday(id, name, day, month, wxDateTime::GetCurrentYear()-1);
    }
    catch(const wxSQLite3Exception& e)
    {
        wxLogError(wxString::Format("Error in addHoliday: %s", e.GetMessage().c_str()));
        return Holiday();
    }
}

bool Engine::isHoliday(const wxDateTime &date)
{
    wxSQLite3ResultSet rs = m_db.ExecuteQuery("select * from HOLIDAYS");
    while(rs.NextRow())
    {
        if(rs.GetInt("DAY")==date.GetDay() && static_cast<wxDateTime::Month>(rs.GetInt("MONTH"))==date.GetMonth()) return true;
    }
    rs.Finalize();
    if(dxsettings.easternMonday() && date.IsSameDate(dxutils::easternSunday(date.GetYear())+wxDateSpan::Day())) return true;
    if(dxsettings.easternFriday() && date.IsSameDate(dxutils::easternSunday(date.GetYear())-wxDateSpan::Days(2))) return true;
    return false;
}

bool Engine::someHolidayToFire()
{
    wxDateTime date = wxDateTime::Today();
    for(int day=0; day<=dxsettings.remindHolidayDays(); day++)
    {
        wxSQLite3ResultSet rs = m_db.ExecuteQuery("select * from HOLIDAYS order by MONTH,DAY");
        while(rs.NextRow())
        {
            if(rs.GetInt("DAY")==date.GetDay() && static_cast<wxDateTime::Month>(rs.GetInt("MONTH"))==date.GetMonth() && date.GetYear()>rs.GetInt("REMINDED")) return true;
        }
        rs.Finalize();
        if(dxsettings.easternMonday() && date.IsSameDate(dxutils::easternSunday(date.GetYear())+wxDateSpan::Day()) && date.GetYear()>dxsettings.easternMondayReminded()) return true;
        if(dxsettings.easternFriday() && date.IsSameDate(dxutils::easternSunday(date.GetYear())-wxDateSpan::Days(2)) && date.GetYear()>dxsettings.easternFridayReminded()) return true;
        date += wxDateSpan::Day();
    }
    return false;
}

wxString Engine::holidayTextsForDate()
{
    wxDateTime hdate = wxDateTime::Today();
    wxString text;
    std::vector<Holiday> days;
    wxSQLite3ResultSet rs = m_db.ExecuteQuery("select * from HOLIDAYS order by MONTH,DAY");
    while(rs.NextRow())
    {
        days.push_back(Holiday(rs.GetInt64("HOLIDAYID").GetValue(),rs.GetString("NAME"),rs.GetInt("DAY"),static_cast<wxDateTime::Month>(rs.GetInt("MONTH")),rs.GetInt("REMINDED")));
    }
    rs.Finalize();
    if(dxsettings.easternMonday())
    {
        wxDateTime monday = dxutils::easternSunday(hdate.GetYear())+wxDateSpan::Day();
        days.push_back(Holiday(-2,_("Easter Monday"),monday.GetDay(),monday.GetMonth(),dxsettings.easternMondayReminded(),true));
    }
    if(dxsettings.easternFriday())
    {
        wxDateTime friday = dxutils::easternSunday(hdate.GetYear())-wxDateSpan::Days(2);
        days.push_back(Holiday(-3,_("Good Friday"),friday.GetDay(),friday.GetMonth(),dxsettings.easternFridayReminded(),false,true));
    }
    std::sort(days.begin(),days.end(),sortHoliday);
    for(int day=0; day<=dxsettings.remindHolidayDays(); day++)
    {
        for(const Holiday &holiday : days)
        {
            if(holiday.day()==hdate.GetDay() && holiday.month()==hdate.GetMonth() && hdate.GetYear()>holiday.reminded())
            {
                text << wxString::Format("%s\n%s\n\n",holiday.name(),dxutils::formatDate(wxDateTime(holiday.day(),holiday.month(),hdate.GetYear()),dxsettings.dateFormat(),true));
                if(holiday.easternMonday()) dxsettings.setEasternMondayReminded(hdate.GetYear());
                else if(holiday.easternFriday()) dxsettings.setEasternFridayReminded(hdate.GetYear());
                else setHolidayReminded(holiday.id());
            }
        }
        hdate += wxDateSpan::Day();
    }
    return text;
}

wxString Engine::holidayTipTexts(const wxDateTime &date)
{
    wxString text = wxEmptyString;
    std::vector<Holiday> days;
    wxSQLite3ResultSet rs = m_db.ExecuteQuery("select * from HOLIDAYS order by MONTH,DAY");
    while(rs.NextRow())
    {
        days.push_back(Holiday(rs.GetInt64("HOLIDAYID").GetValue(),rs.GetString("NAME"),rs.GetInt("DAY"),static_cast<wxDateTime::Month>(rs.GetInt("MONTH")),rs.GetInt("REMINDED")));
    }
    rs.Finalize();
    if(dxsettings.easternMonday())
    {
        wxDateTime monday = dxutils::easternSunday(date.GetYear())+wxDateSpan::Day();
        days.push_back(Holiday(-2,_("Easter Monday"),monday.GetDay(),monday.GetMonth(),dxsettings.easternMondayReminded(),true));
    }
    if(dxsettings.easternFriday())
    {
        wxDateTime friday = dxutils::easternSunday(date.GetYear())-wxDateSpan::Days(2);
        days.push_back(Holiday(-3,_("Good Friday"),friday.GetDay(),friday.GetMonth(),dxsettings.easternFridayReminded(),false,true));
    }
    std::sort(days.begin(),days.end(),sortHoliday);
    if(days.size())
    {
        text << _("Public holiday") << ":";
    }
    for(const Holiday &holiday : days)
    {
        if(holiday.day()==date.GetDay() && holiday.month()==date.GetMonth())
        {
            text << wxString::Format("\n%s",holiday.name());
        }
    }
    return text;
}

void Engine::setHolidayReminded(wxInt64 id)
{
    try
    {
        wxSQLite3Statement st = m_db.PrepareStatement("update HOLIDAYS set REMINDED = ? where HOLIDAYID = ?");
        st.Bind(1, wxDateTime::GetCurrentYear());
        st.Bind(2, wxLongLong(id));
        st.ExecuteUpdate();
        st.Finalize();
    }
    catch(const wxSQLite3Exception& e)
    {
        wxLogError(wxString::Format("Error in setHolidayReminded: %s", e.GetMessage().c_str()));
    }
}

std::vector<EventView> Engine::eventViews(bool onlyhidden, bool showhidden)
{
    std::vector<EventView> events;
    if(onlyhidden)
    {
        for(auto it=m_events.begin(); it!=m_events.end(); ++it)
        {
            if((*it).hidden())
            {
                events.push_back(EventView((*it).id(), (*it).event(), (*it).realEventDate(), (*it).reminderText(), (*it).recurrenceText(),
                                           (*it).color(), (*it).recurrence(), (*it).always()));
            }
        }
        std::sort(events.begin(), events.end(), sortEventsView);
        return events;
    }
    for(auto it=m_events.begin(); it!=m_events.end(); ++it)
    {
        if(!showhidden && (*it).hidden())
        {
            continue;
        }
        events.push_back(EventView((*it).id(), (*it).event(), (*it).realEventDate(), (*it).reminderText(), (*it).recurrenceText(),
                                   (*it).color(), (*it).recurrence(), (*it).always()));
    }
    std::sort(events.begin(), events.end(), sortEventsView);
    return events;
}

void Engine::reEncrypt()
{
    if(dxsettings.encrypt()) m_db.ReKey(dxsettings.pk());
    else m_db.ReKey("");
}

void Engine::createEventsTable()
{
    try
    {
        bool exists = m_db.TableExists("EVENTS");
        if(!exists)
        {
            m_db.ExecuteUpdate("create table EVENTS(EVENTID integer primary key, \
                                EVENT text not null, DESCRIPTION text, EVENTDATE text not null, \
                                REMINDER integer not null, ACTUALREMINDER integer not null, \
                                REMINDED integer not null, DAY integer not null default 0, HIDDEN integer not null default 0, MONTH integer not null default 0, \
                                RECURRENCE integer not null default 0, MINUTES integer not null default 0, HOURS integer not null default 0, \
                                DAYS integer not null default 0, MONTHS integer not null default 0, MDAY integer not null default 0, MWEEK integer not null default 0, \
                                UNTIL text not null, ALWAYS integer not null default 0, COLOR integer not null default 0);");
            exists = m_db.TableExists("EVENTS");
            wxASSERT(exists);
        }
    }
    catch(const wxSQLite3Exception& e)
    {
        wxLogError(wxString::Format("Error in createEventsTable: %s", e.GetMessage().c_str()));
    }
}

void Engine::createNoteTable()
{
    try
    {
        bool exists = m_db.TableExists("NOTE");
        if(!exists)
        {
            m_db.ExecuteUpdate("create table NOTE(NTEXT text);");
            m_db.ExecuteUpdate("insert into NOTE(NTEXT) values('')");
            exists = m_db.TableExists("NOTE");
            wxASSERT(exists);
        }
    }
    catch(const wxSQLite3Exception& e)
    {
        wxLogError(wxString::Format("Error in createNoteTable: %s", e.GetMessage().c_str()));
    }
}

void Engine::createHolidaysTable()
{
    try
    {
        bool exists = m_db.TableExists("HOLIDAYS");
        if(!exists)
        {
            m_db.ExecuteUpdate("create table HOLIDAYS(HOLIDAYID integer primary key, \
                                NAME text, DAY integer not null, MONTH integer not null, REMINDED integer not null default 0);");
            exists = m_db.TableExists("HOLIDAYS");
            wxASSERT(exists);
        }
    }
    catch(const wxSQLite3Exception& e)
    {
        wxLogError(wxString::Format("Error in createHolidaysTable: %s", e.GetMessage().c_str()));
    }
}

bool Engine::isNonworkingDay(const wxDateTime &date)
{
    if(isHoliday(date)) return true;
    if(date.GetWeekDay() == wxDateTime::Sat) return true;
    if(date.GetWeekDay() == wxDateTime::Sun) return true;
    return false;
}

wxDateTime Engine::nextWorkingDay(const wxDateTime &date)
{
    wxDateTime nextdate = date+wxDateSpan::Day();
    while(isHoliday(nextdate) || nextdate.GetWeekDay() == wxDateTime::Sat || nextdate.GetWeekDay() == wxDateTime::Sun)
    {
        nextdate += wxDateSpan::Day();
    }
    return nextdate;
}

void Engine::fireUpdated(bool updateToolTip)
{
    tooltipPluginForDate();
    wxCommandEvent event(ENGINE_UPDATED);
    if(updateToolTip) event.SetId(1);
    else event.SetId(0);
    wxPostEvent(m_parent, event);
}

void Engine::fireMsg(const wxString &msg)
{
    wxCommandEvent event(ENGINE_MSG);
    event.SetString(msg);
    wxPostEvent(m_parent, event);
}

wxDateTime Engine::plusBusinessDay(const wxDateTime &eventday, wxDateTime::wxDateTime_t days)
{
    if(days == 0)
    {
        wxDateTime date = eventday;
        if(isNonworkingDay(date)) date = nextWorkingDay(date);
        return date;
    }
    wxDateTime date = eventday;
    wxDateTime::wxDateTime_t wdays=0;
    while(1)
    {
        if(wdays < days)
        {
            date = date+wxDateSpan::Day();
            wdays++;
            if(isNonworkingDay(date)) date = nextWorkingDay(date);
        }
        else
        {
            break;
        }
    }
    return date;
}

wxDateTime Engine::montlyAtGivenBusinessDay(const wxDateTime &eventday, wxDateTime::wxDateTime_t day)
{
    if(day < 2)
    {
        wxDateTime date = eventday;
        if(date.GetDay() != 1) date.SetDay(1);
        if(isNonworkingDay(date)) date = nextWorkingDay(date);
        return date;
    }
    wxDateTime date = eventday;
    if(date.GetDay() != 1) date.SetDay(1);
    if(isNonworkingDay(date)) date = nextWorkingDay(date);
    wxDateTime::wxDateTime_t wdays=1;
    while(1)
    {
        if(wdays < day)
        {
            date = date+wxDateSpan::Day();
            wdays++;
            if(isNonworkingDay(date)) date = nextWorkingDay(date);
        }
        else
        {
            break;
        }
    }
    return date;
}

wxDateTime Engine::weeklyAtGivenBusinessDay(const wxDateTime &eventday, wxDateTime::wxDateTime_t day)
{
    if(day < 2)
    {
        wxDateTime date = eventday;
        if(date.GetWeekDay() != wxDateTime::Mon) date.SetToWeekDayInSameWeek(wxDateTime::Mon);
        if(isNonworkingDay(date)) date = nextWorkingDay(date);
        return date;
    }
    wxDateTime date = eventday;
    if(date.GetWeekDay() != wxDateTime::Mon) date.SetToWeekDayInSameWeek(wxDateTime::Mon);
    if(isNonworkingDay(date)) date = nextWorkingDay(date);
    wxDateTime::wxDateTime_t wdays=1;
    while(1)
    {
        if(wdays < day)
        {
            date = date+wxDateSpan::Day();
            wdays++;
            if(isNonworkingDay(date)) date = nextWorkingDay(date);
        }
        else
        {
            break;
        }
    }
    return date;
}

wxString Engine::getTooltipPlugin() const
{
    return m_tooltipPlugin+wxString("\r\n");
}

bool Engine::someEventOnHiddenList()
{
    return std::any_of(m_events.begin(), m_events.end(), [](const Event &ev){ return ev.hidden(); });
}

bool Engine::isEventOnHiddenList(wxInt64 id)
{
    return std::any_of(m_events.begin(), m_events.end(), [id](const Event &ev){ return ev.id()==id && ev.hidden(); });
}

void Engine::removeEventFromHiddenList(wxInt64 id)
{
    for(auto it=m_events.begin(); it!=m_events.end(); ++it)
    {
        if((*it).id() == id)
        {
            (*it).setHidden(false);
            tryExecuteUpdate(&m_db, wxString::Format("update EVENTS set HIDDEN = 0 where EVENTID=%lld", id));
            break;
        }
    }
    m_lastid = -1;
    fireUpdated(false);
}

void Engine::clearListOfHiddenEvents()
{
    tryExecuteUpdate(&m_db, "update EVENTS set HIDDEN = 0");
    for(auto it=m_events.begin(); it!=m_events.end(); ++it)
    {
        (*it).setHidden(false);
    }
    m_lastid = -1;
    fireUpdated(false);
}

bool Engine::someEventOnHighlightList()
{
    return std::any_of(m_events.begin(), m_events.end(), [](const Event &ev){ return ev.color(); });
}

int Engine::eventHighlightColor(wxInt64 id)
{
    auto it = std::find_if(m_events.begin(), m_events.end(), [id](const Event &ev)->bool{return ev.id() == id;});
    if(it != m_events.end()) return (*it).color();
    return 0;
}

void Engine::removeEventFromHighlightList(wxInt64 id)
{
    for(auto it=m_events.begin(); it!=m_events.end(); ++it)
    {
        if((*it).id() == id)
        {
            (*it).setColor(0);
            tryExecuteUpdate(&m_db, wxString::Format("update EVENTS set COLOR = 0 where EVENTID=%lld", id));
            break;
        }
    }
    m_lastid = -1;
    fireUpdated(false);
}

void Engine::clearListOfHighlightEvents()
{
    tryExecuteUpdate(&m_db, "update EVENTS set COLOR = 0");
    for(auto it=m_events.begin(); it!=m_events.end(); ++it)
    {
        (*it).setColor(0);
    }
    m_lastid = -1;
    fireUpdated(false);
}

void Engine::highlightEvent(wxInt64 id, int color, bool fireUpdate)
{
    for(auto it=m_events.begin(); it!=m_events.end(); ++it)
    {
        if((*it).id() == id)
        {
            (*it).setColor(color);
            tryExecuteUpdate(&m_db, wxString::Format("update EVENTS set COLOR = %d where EVENTID=%lld", color, id));
            break;
        }
    }
    m_lastid = -1;
    if(fireUpdate) fireUpdated(false);
}

void Engine::alwaysshowEvent(wxInt64 id, bool always, bool fireUpdate)
{
    try
    {
        wxSQLite3Statement st = m_db.PrepareStatement("update EVENTS set ALWAYS=? where EVENTID=?");
        st.BindBool(1, always);
        st.Bind(2, wxLongLong(id));
        st.ExecuteUpdate();
        st.Finalize();
        for(auto it=m_events.begin(); it!=m_events.end(); ++it)
        {
            if((*it).id() == id)
            {
                (*it).setAlways(always);
                break;
            }
        }
    }
    catch(const wxSQLite3Exception& e)
    {
        wxLogError(wxString::Format("Error in alwaysshowEvent: %s", e.GetMessage().c_str()));
    }
    m_lastid = -1;
    if(fireUpdate) fireUpdated(false);
}

wxInt64 Engine::addEvent(const Event &event, bool fireUpdate)
{
    wxInt64 id = -1;
    try
    {
        wxSQLite3Statement st = m_db.PrepareStatement("insert into EVENTS (EVENT, DESCRIPTION, EVENTDATE, REMINDER, ACTUALREMINDER, REMINDED, DAY, HIDDEN, MONTH,"
                                                      "RECURRENCE, MINUTES, HOURS,"
                                                      "DAYS, MONTHS, MDAY, MWEEK, UNTIL, ALWAYS, COLOR) "
                                                      "values (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)");
        st.Bind(1, event.event());
        st.Bind(2, event.descr());
        st.BindDateTime(3, event.date());
        st.Bind(4, wxLongLong(event.reminder()));
        st.Bind(5, wxLongLong(event.actualreminder()));
        st.BindBool(6, event.recurrence()==R_NONE?true:false);
        st.Bind(7, event.day());
        st.BindBool(8, event.hidden());
        st.Bind(9, event.month());
        st.Bind(10, event.recurrence());
        st.Bind(11, event.minutes());
        st.Bind(12, event.hours());
        st.Bind(13, event.days());
        st.Bind(14, event.months());
        st.Bind(15, event.monthlyday());
        st.Bind(16, event.monthlyweek());
        st.BindDate(17, event.until());
        st.BindBool(18, event.always());
        st.Bind(19, event.color());
        st.ExecuteUpdate();
        id = m_db.GetLastRowId().GetValue();
        st.Finalize();
        m_events.push_back(Event(id,event.event(),event.descr(),event.date(),event.reminder(),event.recurrence()==R_NONE?true:false,event.day(),event.month(),event.hidden(),
                                 event.recurrence(),event.minutes(),event.hours(),event.days(),event.months(),event.monthlyday(),event.monthlyweek(),event.actualreminder(),
                                 event.until(),event.always(),event.color()));
    }
    catch(const wxSQLite3Exception& e)
    {
        wxLogError(wxString::Format("Error in addEvent: %s", e.GetMessage().c_str()));
    }
    m_lastid = id;
    if(fireUpdate) fireUpdated();
    return id;
}

void Engine::editEvent(const Event &event)
{
    try
    {
        wxSQLite3Statement st = m_db.PrepareStatement("update EVENTS set EVENT=?, DESCRIPTION=?, EVENTDATE=?, REMINDER=?, ACTUALREMINDER=?, REMINDED=?, DAY=? , HIDDEN=?, MONTH=?, "
                                                      "RECURRENCE=?, MINUTES=?, HOURS=?, DAYS=?, MONTHS=?, MDAY=?, MWEEK=?, UNTIL=?, ALWAYS=?, COLOR=? "
                                                      "where EVENTID=?");
        st.Bind(1, event.event());
        st.Bind(2, event.descr());
        st.BindDateTime(3, event.date());
        st.Bind(4, wxLongLong(event.reminder()));
        st.Bind(5, wxLongLong(event.reminder()));
        st.BindBool(6, event.reminded());
        st.Bind(7, event.day());
        st.BindBool(8, event.hidden());
        st.Bind(9, event.month());
        st.Bind(10, event.recurrence());
        st.Bind(11, event.minutes());
        st.Bind(12, event.hours());
        st.Bind(13, event.days());
        st.Bind(14, event.months());
        st.Bind(15, event.monthlyday());
        st.Bind(16, event.monthlyweek());
        st.BindDate(17, event.until());
        st.BindBool(18, event.always());
        st.Bind(19, event.color());
        st.Bind(20, wxLongLong(event.id()));
        st.ExecuteUpdate();
        st.Finalize();
        for(auto it=m_events.begin(); it!=m_events.end(); ++it)
        {
            if((*it).id() == event.id())
            {
                (*it) = event;
                break;
            }
        }
    }
    catch(const wxSQLite3Exception& e)
    {
        wxLogError(wxString::Format("Error in editEvent: %s", e.GetMessage().c_str()));
    }
    m_lastid = -1;
    fireUpdated();
}

void Engine::editEvent(wxInt64 id, const wxDateTime &date)
{
    try
    {
        int day = date.GetDay();
        int type = getEvent(id).recurrence();
        if(type == R_WEEKLYBUSINESS || type == R_WEEKLYPLUSBUSINESSDAY || type == R_WEEKLYGIVENBUSINESSDAY
                || type == R_2WEEKLYBUSINESS || type == R_2WEEKLYPLUSBUSINESSDAY || type == R_2WEEKLYGIVENBUSINESSDAY)
        {
            day = date.GetWeekDay();
        }
        wxSQLite3Statement st = m_db.PrepareStatement("update EVENTS set EVENTDATE=?, DAY=? , MONTH=? where EVENTID=?");
        st.BindDateTime(1, date);
        st.Bind(2, day);
        st.Bind(3, static_cast<int>(date.GetMonth())+1);
        st.Bind(4, wxLongLong(id));
        st.ExecuteUpdate();
        st.Finalize();
        for(auto it=m_events.begin(); it!=m_events.end(); ++it)
        {
            if((*it).id() == id)
            {
                (*it).setDate(date);
                (*it).setDay(day);
                (*it).setMonth(static_cast<int>(date.GetMonth())+1);
                break;
            }
        }
    }
    catch(const wxSQLite3Exception& e)
    {
        wxLogError(wxString::Format("Error in editEvent: %s", e.GetMessage().c_str()));
    }
    m_lastid = -1;
    fireUpdated();
}

void Engine::editEvent(wxInt64 id, wxInt64 reminder)
{
    try
    {
        wxSQLite3Statement st = m_db.PrepareStatement("update EVENTS set ACTUALREMINDER = ?, REMINDED = ? where EVENTID = ?");
        st.Bind(1, wxLongLong(reminder));
        st.BindBool(2, false);
        st.Bind(3, wxLongLong(id));
        st.ExecuteUpdate();
        st.Finalize();
        for(auto it=m_events.begin(); it!=m_events.end(); ++it)
        {
            if((*it).id() == id)
            {
                (*it).setActualreminder(reminder);
                (*it).setReminded(false);
                break;
            }
        }
    }
    catch(const wxSQLite3Exception& e)
    {
        wxLogError(wxString::Format("Error in editEvent: %s", e.GetMessage().c_str()));
    }
    m_lastid = -1;
    fireUpdated();
}

void Engine::editEvent(wxInt64 id, const wxString &name)
{
    try
    {
        wxSQLite3Statement st = m_db.PrepareStatement("update EVENTS set EVENT = ? where EVENTID = ?");
        st.Bind(1, name);
        st.Bind(2, wxLongLong(id));
        st.ExecuteUpdate();
        st.Finalize();
        for(auto it=m_events.begin(); it!=m_events.end(); ++it)
        {
            if((*it).id() == id)
            {
                (*it).setEvent(name);
                break;
            }
        }
    }
    catch(const wxSQLite3Exception& e)
    {
        wxLogError(wxString::Format("Error in editEvent: %s", e.GetMessage().c_str()));
    }
    m_lastid = -1;
    fireUpdated();
}

void Engine::removeEvent(wxInt64 id, bool fireUpdate)
{
    try
    {
        static const char* sql[] =
        {
            "delete from EVENTS where EVENTID = ?",
            nullptr
        };
        for(int i = 0; sql[i]; ++i)
        {
            wxSQLite3Statement st = m_db.PrepareStatement(sql[i]);
            st.Bind(1, wxLongLong(id));
            st.ExecuteUpdate();
            st.Finalize();
        }
        for(auto it=m_events.begin(); it!=m_events.end(); ++it)
        {
            if((*it).id() == id)
            {
                m_events.erase(it);
                break;
            }
        }
    }
    catch(const wxSQLite3Exception& e)
    {
        wxLogError(wxString::Format("Error in removeEvent: %s", e.GetMessage().c_str()));
    }
    m_lastid = -1;
    if(fireUpdate) fireUpdated();
}

void Engine::hideEvent(wxInt64 id, bool fireUpdate)
{
    try
    {
        wxSQLite3Statement st = m_db.PrepareStatement("update EVENTS set HIDDEN = ? where EVENTID = ?");
        st.BindBool(1, true);
        st.Bind(2, wxLongLong(id));
        st.ExecuteUpdate();
        st.Finalize();
        for(auto it=m_events.begin(); it!=m_events.end(); ++it)
        {
            if((*it).id() == id)
            {
                (*it).setHidden(true);
                break;
            }
        }
    }
    catch(const wxSQLite3Exception& e)
    {
       wxLogError(wxString::Format("Error in hideEvent: %s", e.GetMessage().c_str()));
    }
    m_lastid = -1;
    if(fireUpdate) fireUpdated(false);
}

void Engine::unhideEvent(wxInt64 id, bool fireUpdate)
{
    try
    {
        wxSQLite3Statement st = m_db.PrepareStatement("update EVENTS set HIDDEN = ? where EVENTID = ?");
        st.BindBool(1, false);
        st.Bind(2, wxLongLong(id));
        st.ExecuteUpdate();
        st.Finalize();
        for(auto it=m_events.begin(); it!=m_events.end(); ++it)
        {
            if((*it).id() == id)
            {
                (*it).setHidden(false);
                break;
            }
        }
    }
    catch(const wxSQLite3Exception& e)
    {
       wxLogError(wxString::Format("Error in unhideEvent: %s", e.GetMessage().c_str()));
    }
    m_lastid = -1;
    if(fireUpdate) fireUpdated(false);
}

bool Engine::someEvents()
{
    return m_events.size();
}

bool Engine::someDonotdeleteEvent(bool showhidden)
{
    if(showhidden)
    {
        return std::any_of(m_events.begin(), m_events.end(), [](const Event &ev){ return ev.recurrence()==R_NONE || ev.recurrence()==R_ONCENOTDELETE; });
    }
    else
    {
        return std::any_of(m_events.begin(), m_events.end(), [](const Event &ev){ return !ev.hidden() && (ev.recurrence()==R_NONE || ev.recurrence()==R_ONCENOTDELETE); });
    }
}

void Engine::removeEvents()
{
    tryExecuteUpdate(&m_db, "delete from EVENTS");
    m_events.clear();
    m_lastid = -1;
    fireUpdated();
}

void Engine::recurrentEvent(wxInt64 id, bool fireUpdate)
{
    Event event = getEvent(id);
    if(event.isValid())
    {
        wxDateTime date = recurrentEventDate(event,wxDateTime::Now());
        if(date.IsSameDate(INFINITY_DATE))
        {
            removeEvent(id, fireUpdate);
            return;
        }
        if(date.IsLaterThan(event.until()) || date.IsSameDate(event.until()))
        {
            removeEvent(id, fireUpdate);
            return;
        }
        try
        {
            wxInt64 reminder = 0;
            wxSQLite3ResultSet q = m_db.ExecuteQuery(wxString::Format("select REMINDER from EVENTS where EVENTID = %lld;", id));
            if(q.NextRow())
            {
                reminder = q.GetInt64(0).GetValue();
            }
            q.Finalize();
            wxSQLite3Statement st = m_db.PrepareStatement("update EVENTS set EVENTDATE = ?, REMINDER = ?, ACTUALREMINDER = ?, REMINDED = ? where EVENTID = ?");
            st.BindDateTime(1, date);
            st.Bind(2, wxLongLong(reminder));
            st.Bind(3, wxLongLong(reminder));
            st.BindBool(4, false);
            st.Bind(5, wxLongLong(id));
            st.ExecuteUpdate();
            st.Finalize();
            for(auto it=m_events.begin(); it!=m_events.end(); ++it)
            {
                if((*it).id() == id)
                {
                    (*it).setDate(date);
                    (*it).setReminder(reminder);
                    (*it).setActualreminder(reminder);
                    (*it).setReminded(false);
                    break;
                }
            }
        }
        catch(const wxSQLite3Exception& e)
        {
            wxLogError(wxString::Format("Error in recurrentEvent: %s", e.GetMessage().c_str()));
        }
        m_lastid = -1;
        if(fireUpdate) fireUpdated();
    }
}

wxDateTime Engine::recurrentEventDate(const Event &event, const wxDateTime &rdate)
{
    if(event.isValid())
    {
        wxDateTime date = INFINITY_DATE;
        switch (event.recurrence()) {
        case R_MINUTES:
        {
            date = event.date()+wxTimeSpan::Minutes(event.minutes());
            while(date.IsEarlierThan(rdate))
            {
                date += wxTimeSpan::Minutes(event.minutes());
            }
            break;
        }
        case R_HOURLY:
        {
            date = event.date()+wxTimeSpan::Hours(event.hours());
            while(date.IsEarlierThan(rdate))
            {
                date += wxTimeSpan::Hours(event.hours());
            }
            break;
        }
        case R_DAILY:
        {
            date = rdate+wxDateSpan::Day();
            date.SetHour(event.date().GetHour());
            date.SetMinute(event.date().GetMinute());
            date.SetSecond(event.date().GetSecond());
            break;
        }
        case R_DAILYBUSINESS:
        {
            date = rdate+wxDateSpan::Day();
            if(isNonworkingDay(date)) date = nextWorkingDay(date);
            date.SetHour(event.date().GetHour());
            date.SetMinute(event.date().GetMinute());
            date.SetSecond(event.date().GetSecond());
            break;
        }
        case R_WEEKLY:
        {
            date = event.date()+wxDateSpan::Days(7);
            while(date.IsEarlierThan(rdate))
            {
                date += wxDateSpan::Days(7);
            }
            break;
        }
        case R_WEEKLYBUSINESS:
        {
            date = event.date()+wxDateSpan::Days(7);
            while(date.IsEarlierThan(rdate))
            {
                date += wxDateSpan::Days(7);
            }
            if(isNonworkingDay(date))
            {
                wxDateTime::wxDateTime_t week = date.GetWeekOfYear();
                while(week != nextWorkingDay(date).GetWeekOfYear())
                {
                    date += wxDateSpan::Days(7);
                    week = date.GetWeekOfYear();
                    if(!isNonworkingDay(date)) break;
                }
            }
            break;
        }
        case R_WEEKLYPLUSBUSINESSDAY:
        {
            date = event.date()+wxDateSpan::Days(7);
            while(date.IsEarlierThan(rdate))
            {
                date += wxDateSpan::Days(7);
            }
            wxDateTime::wxDateTime_t week = date.GetWeekOfYear();
            while(week != plusBusinessDay(date, event.monthlyday()).GetWeekOfYear())
            {
                date += wxDateSpan::Days(7);
                week = date.GetWeekOfYear();
            }
            break;
        }
        case R_WEEKLYGIVENBUSINESSDAY:
        {
            wxDateTime::WeekDay wd = event.date().GetWeekDay();
            date = event.date().SetToWeekDayInSameWeek(wxDateTime::Mon);
            while(date.IsEarlierThan(rdate))
            {
                date += wxDateSpan::Days(7);
            }
            wxDateTime::wxDateTime_t week = date.GetWeekOfYear();
            while(week != weeklyAtGivenBusinessDay(date, event.monthlyday()).GetWeekOfYear() || weeklyAtGivenBusinessDay(date, event.monthlyday()).IsEarlierThan(rdate))
            {
                date += wxDateSpan::Days(7);
                week = date.GetWeekOfYear();
            }
            date.SetToWeekDayInSameWeek(wd);
            break;
        }
        case R_2WEEKLY:
        {
            date = event.date()+wxDateSpan::Days(14);
            while(date.IsEarlierThan(rdate))
            {
                date = date+wxDateSpan::Days(14);
            }
            break;
        }
        case R_2WEEKLYBUSINESS:
        {
            date = event.date()+wxDateSpan::Days(14);
            while(date.IsEarlierThan(rdate))
            {
                date += wxDateSpan::Days(14);
            }
            if(isNonworkingDay(date))
            {
                wxDateTime::wxDateTime_t week = rdate.GetWeekOfYear();
                while(week != nextWorkingDay(date).GetWeekOfYear())
                {
                    date += wxDateSpan::Days(14);
                    week = date.GetWeekOfYear();
                    if(!isNonworkingDay(date)) break;
                }
            }
            break;
        }
        case R_2WEEKLYPLUSBUSINESSDAY:
        {
            date = event.date()+wxDateSpan::Days(14);
            while(date.IsEarlierThan(rdate))
            {
                date += wxDateSpan::Days(14);
            }
            wxDateTime::wxDateTime_t week = date.GetWeekOfYear();
            while(week != plusBusinessDay(date, event.monthlyday()).GetWeekOfYear())
            {
                date += wxDateSpan::Days(14);
                week = date.GetWeekOfYear();
            }
            break;
        }
        case R_2WEEKLYGIVENBUSINESSDAY:
        {
            wxDateTime::WeekDay wd = event.date().GetWeekDay();
            date = event.date().SetToWeekDayInSameWeek(wxDateTime::Mon);
            while(date.IsEarlierThan(rdate))
            {
                date += wxDateSpan::Days(14);
            }
            wxDateTime::wxDateTime_t week = date.GetWeekOfYear();
            while(week != weeklyAtGivenBusinessDay(date, event.monthlyday()).GetWeekOfYear() || weeklyAtGivenBusinessDay(date, event.monthlyday()).IsEarlierThan(rdate))
            {
                date += wxDateSpan::Days(14);
                week = date.GetWeekOfYear();
            }
            date.SetToWeekDayInSameWeek(wd);
            break;
        }
        case R_MONTHLY:
        {
            if(event.day() > rdate.GetDay())
            {
                if(dxutils::lastdayofmonth(rdate.GetMonth(), rdate.GetYear()) >= event.day())
                {
                    date = wxDateTime(event.day(), rdate.GetMonth(), rdate.GetYear(), event.date().GetHour(), event.date().GetMinute(), event.date().GetSecond());
                    break;
                }
                else
                {
                    date = wxDateTime(1, rdate.GetMonth(), rdate.GetYear(), event.date().GetHour(), event.date().GetMinute(), event.date().GetSecond())+wxDateSpan::Month();
                    while(event.day() > dxutils::lastdayofmonth(date.GetMonth(), date.GetYear()))
                    {
                        date += wxDateSpan::Month();
                    }
                    date.SetDay(event.day());
                    break;
                }
            }
            else
            {
                date = wxDateTime(1, rdate.GetMonth(), rdate.GetYear(), event.date().GetHour(), event.date().GetMinute(), event.date().GetSecond())+wxDateSpan::Month();
                while(event.day() > dxutils::lastdayofmonth(date.GetMonth(), date.GetYear()))
                {
                    date += wxDateSpan::Month();
                }
                date.SetDay(event.day());
                break;
            }
        }
        case R_OWN:
        {
            date = event.date()+wxDateSpan(0, event.months(), 0, event.days())+wxTimeSpan(event.hours(), event.minutes());
            while(date.IsEarlierThan(rdate))
            {
                date = date+wxDateSpan(0, event.months(), 0, event.days())+wxTimeSpan(event.hours(), event.minutes());
            }
            break;
        }
        case R_MONTHLYATDAY:
        {
            date = wxDateTime(1, rdate.GetMonth(), rdate.GetYear(), event.date().GetHour(), event.date().GetMinute(), event.date().GetSecond());
            while(!date.SetToWeekDay(static_cast<wxDateTime::WeekDay>(event.monthlyday()), event.monthlyweek(), date.GetMonth(), date.GetYear()))
            {
                date += wxDateSpan::Month();
            }
            if(date < rdate)
            {
                date += wxDateSpan::Month();
                date.SetDay(1);
                while(!date.SetToWeekDay(static_cast<wxDateTime::WeekDay>(event.monthlyday()), event.monthlyweek(), date.GetMonth(), date.GetYear()))
                {
                    date += wxDateSpan::Month();
                }
            }
            date.SetHour(event.date().GetHour());
            date.SetMinute(event.date().GetMinute());
            date.SetSecond(event.date().GetSecond());
            break;
        }
        case R_MONTHLYBUSINESS:
        {
            date = wxDateTime(wxMin(event.day(),dxutils::lastdayofmonth(rdate.GetMonth(), rdate.GetYear())), rdate.GetMonth(),
                              rdate.GetYear(), event.date().GetHour(), event.date().GetMinute(), event.date().GetSecond());
            if(isNonworkingDay(date)) date = nextWorkingDay(date);
            if(date < rdate)
            {
                date += wxDateSpan::Month();
                if(isNonworkingDay(date)) date = nextWorkingDay(date);
            }
            break;
        }
        case R_MONTHLYPLUSBUSINESSDAY:
        {
            if(event.day() > rdate.GetDay())
            {
                if(dxutils::lastdayofmonth(rdate.GetMonth(), rdate.GetYear()) >= event.day())
                {
                    date = wxDateTime(event.day(), rdate.GetMonth(), rdate.GetYear(), event.date().GetHour(), event.date().GetMinute(), event.date().GetSecond());
                    break;
                }
                else
                {
                    date = wxDateTime(1, rdate.GetMonth(), rdate.GetYear(), event.date().GetHour(), event.date().GetMinute(), event.date().GetSecond())+wxDateSpan::Month();
                    while(event.day() > dxutils::lastdayofmonth(date.GetMonth(), date.GetYear()))
                    {
                        date += wxDateSpan::Month();
                    }
                    date.SetDay(event.day());
                    break;
                }
            }
            else
            {
                date = wxDateTime(1, rdate.GetMonth(), rdate.GetYear(), event.date().GetHour(), event.date().GetMinute(), event.date().GetSecond())+wxDateSpan::Month();
                while(event.day() > dxutils::lastdayofmonth(date.GetMonth(), date.GetYear()))
                {
                    date += wxDateSpan::Month();
                }
                date.SetDay(event.day());
                break;
            }
        }
        case R_MONTHLYGIVENBUSINESSDAY:
        {
            date = wxDateTime(1, rdate.GetMonth(), rdate.GetYear(), event.date().GetHour(), event.date().GetMinute(), event.date().GetSecond());
            if(date < rdate)
            {
                date += wxDateSpan::Month();
            }
            break;
        }
        case R_QUARTERLY:
        {
            date = wxDateTime(1, rdate.GetMonth(), rdate.GetYear(), event.date().GetHour(), event.date().GetMinute(), event.date().GetSecond());
            if(date.GetMonth() == event.date().GetMonth())
            {
                date += wxDateSpan::Month();
            }
            while(!isMonthQuarterly(date.GetMonth(), static_cast<wxDateTime::Month>(event.month()-1)))
            {
                date += wxDateSpan::Month();
                if(event.day() > dxutils::lastdayofmonth(date.GetMonth(), date.GetYear())) date += wxDateSpan::Month();
            }
            date.SetDay(event.day());
            break;
        }
        case R_QUARTERLYBUSINESS:
        {
            date = wxDateTime(wxMin(event.day(),dxutils::lastdayofmonth(rdate.GetMonth(), rdate.GetYear())), rdate.GetMonth(),
                              rdate.GetYear(), event.date().GetHour(), event.date().GetMinute(), event.date().GetSecond());
            if(date.GetMonth() == event.date().GetMonth())
            {
                date += wxDateSpan::Month();
            }
            while(!isMonthQuarterly(date.GetMonth(), static_cast<wxDateTime::Month>(event.month()-1)))
            {
                date += wxDateSpan::Month();
            }
            if(isNonworkingDay(date)) date = nextWorkingDay(date);
            break;
        }
        case R_QUARTERLYPLUSBUSINESSDAY:
        {
            date = nextQuarterDay(event.day(), event.date().GetMonth());
            date.SetHour(event.date().GetHour());
            date.SetMinute(event.date().GetMinute());
            date.SetSecond(event.date().GetSecond());
            break;
        }
        case R_QUARTERLYGIVENBUSINESSDAY:
        {
            date = wxDateTime(1, rdate.GetMonth(), rdate.GetYear(), event.date().GetHour(), event.date().GetMinute(), event.date().GetSecond());
            if(date.GetMonth() == event.date().GetMonth())
            {
                date += wxDateSpan::Month();
            }
            while(!isMonthQuarterly(date.GetMonth(), static_cast<wxDateTime::Month>(event.month()-1)))
            {
                date += wxDateSpan::Month();
            }
            break;
        }
        case R_YEARLY:
        {
            int year = rdate.GetYear()+1;
            while(event.day() > dxutils::lastdayofmonth(static_cast<wxDateTime::Month>(event.month()-1), year))
            {
                year++;
            }
            date = wxDateTime(event.day(), static_cast<wxDateTime::Month>(event.month()-1), year, event.date().GetHour(), event.date().GetMinute(), event.date().GetSecond());
            break;
        }
        default: break;
        }
        return date;
    }
    return INFINITY_DATE;
}

Event Engine::getEvent(wxInt64 id)
{
    auto it = std::find_if(m_events.begin(), m_events.end(), [id](const Event &ev)->bool{return ev.id() == id;});
    if(it != m_events.end()) return (*it);
    return Event();
}

wxInt64 Engine::getEventReminder(wxInt64 id)
{
    auto it = std::find_if(m_events.begin(), m_events.end(), [id](const Event &ev)->bool{return ev.id() == id;});
    if(it != m_events.end()) return (*it).actualreminder();
    return 0;
}

int Engine::getEventRecurrence(wxInt64 id)
{
    auto it = std::find_if(m_events.begin(), m_events.end(), [id](const Event &ev)->bool{return ev.id() == id;});
    if(it != m_events.end()) return (*it).recurrence();
    return R_ONCE;
}

std::vector<Event> Engine::eventsByReminder(wxDateTime time)
{
    std::vector<Event> events;
    for(auto it=m_events.begin(); it!=m_events.end(); ++it)
    {
        if(!(*it).reminded())
        {
            if((*it).realEventDate() <= time)
            {
                events.push_back((*it));
                continue;
            }
            if((*it).actualreminder() > 0)
            {
                if(wxDateTime((*it).date()-wxTimeSpan::Minutes((*it).actualreminder())) <= time)
                {
                    events.push_back((*it));
                    continue;
                }
            }
        }
    }
    std::sort(events.begin(), events.end(), sortEvents);
    return events;
}

bool Engine::someEventsByReminder(wxDateTime time)
{
    bool result = false;
    for(auto it=m_events.begin(); it!=m_events.end(); ++it)
    {
        if(!(*it).reminded())
        {
            if((*it).realEventDate() <= time)
            {
                result = true;
                break;
            }
            if((*it).actualreminder() > 0)
            {
                if(wxDateTime((*it).date()-wxTimeSpan::Minutes((*it).actualreminder())) <= time)
                {
                    result = true;
                    break;
                }
            }
        }
    }
    return result;
}

void Engine::setEventReminded(wxInt64 id, bool reminded)
{
    try
    {
        wxSQLite3Statement st = m_db.PrepareStatement("update EVENTS SET REMINDED = ? where EVENTID = ?");
        st.Bind(1, reminded);
        st.Bind(2, wxLongLong(id));
        st.ExecuteUpdate();
        st.Finalize();
        for(auto it=m_events.begin(); it!=m_events.end(); ++it)
        {
            if((*it).id() == id)
            {
                (*it).setReminded(reminded);
                break;
            }
        }
    }
    catch(const wxSQLite3Exception& e)
    {
        wxLogError(wxString::Format("Error in setEventReminded: %s", e.GetMessage().c_str()));
    }
}

std::vector<Event> Engine::eventsForCalendarDate(const wxDateTime &date)
{
    std::vector<Event> events;
    for(Event event : m_events)
    {
        if(event.realEventDate().IsSameDate(date))
        {
            events.push_back(event);
            continue;
        }
        else
        {
            if(date.IsEarlierThan(event.date())) continue;
            switch(event.recurrence()) {
            case R_ONCE:
            case R_ONCENOTDELETE:
            case R_NONE:
            {
                continue;
            }
            case R_MINUTES:
            case R_HOURLY:
            case R_DAILY:
            {
                events.push_back(event);
                break;
            }
            case R_DAILYBUSINESS:
            {
                if(!isNonworkingDay(date)) events.push_back(event);
                break;
            }
            case R_WEEKLY:
            {
                if(isWeekly(event.date().GetWeekDay(), date.GetWeekDay()))
                {
                    events.push_back(event);
                }
                break;
            }
            case R_WEEKLYBUSINESS:
            {
                if(isWeeklyBusiness(static_cast<wxDateTime::WeekDay>(event.day()), date))
                {
                    events.push_back(event);
                }
                break;
            }
            case R_WEEKLYPLUSBUSINESSDAY:
            {
                if(isWeeklyPlusBusinessDay(event.date(), date, static_cast<wxDateTime::WeekDay>(event.day()), event.monthlyday()))
                {
                    events.push_back(event);
                }
                break;
            }
            case R_WEEKLYGIVENBUSINESSDAY:
            {
                if(isWeeklyGivenBusinessDay(event.date(), date, event.monthlyday()))
                {
                    events.push_back(event);
                }
                break;
            }
            case R_2WEEKLY:
            {
                if(is2Weekly(event.date(), date))
                {
                    events.push_back(event);
                }
                break;
            }
            case R_2WEEKLYBUSINESS:
            {
                if(is2WeeklyBusiness(event.date(), static_cast<wxDateTime::WeekDay>(event.day()), date))
                {
                    events.push_back(event);
                }
                break;
            }
            case R_2WEEKLYPLUSBUSINESSDAY:
            {
                if(is2WeeklyPlusBusinessDay(event.date(), date, static_cast<wxDateTime::WeekDay>(event.day()), event.monthlyday()))
                {
                    events.push_back(event);
                }
                break;
            }
            case R_2WEEKLYGIVENBUSINESSDAY:
            {
                if(is2WeeklyGivenBusinessDay(event.date(), date, event.monthlyday()))
                {
                    events.push_back(event);
                }
                break;
            }
            case R_MONTHLY:
            {
                if(isMontly(event.date().GetDay(), date.GetDay()))
                {
                    events.push_back(event);
                }
                break;
            }
            case R_OWN:
            {
                if(isOwn(event.date(), date, event.days(), event.months(), event.hours(), event.minutes(), true))
                {
                    events.push_back(event);
                }
                break;
            }
            case R_MONTHLYATDAY:
            {
                if(isMonthlyAtDay(event.date(), date, event.monthlyday(), event.monthlyweek()))
                {
                    events.push_back(event);
                }
                break;
            }
            case R_MONTHLYBUSINESS:
            {
                if(isMontlyBusiness(event.date(), date, event.day()))
                {
                    events.push_back(event);
                }
                break;
            }
            case R_MONTHLYPLUSBUSINESSDAY:
            {
                if(isMontlyPlusBusinessDay(event.date(), date, event.day(), event.monthlyday()))
                {
                    events.push_back(event);
                    break;
                }
                break;
            }
            case R_MONTHLYGIVENBUSINESSDAY:
            {
                if(isMontlyGivenBusinessDay(event.date(), date, event.monthlyday()))
                {
                    events.push_back(event);
                }
                break;
            }
            case R_QUARTERLY:
            {
                if(isQuarterly(event.date(), date, event.day(), static_cast<wxDateTime::Month>(event.month()-1)))
                {
                    events.push_back(event);
                }
                break;
            }
            case R_QUARTERLYBUSINESS:
            {
                if(isQuarterlyBusiness(event.date(), date, event.day(), static_cast<wxDateTime::Month>(event.month()-1)))
                {
                    events.push_back(event);
                }
                break;
            }
            case R_QUARTERLYPLUSBUSINESSDAY:
            {
                if(isQuarterlyPlusBusinessDay(event.date(), date, event.day(), event.monthlyday(), static_cast<wxDateTime::Month>(event.month()-1)))
                {
                    events.push_back(event);
                }
                break;
            }
            case R_QUARTERLYGIVENBUSINESSDAY:
            {
                if(isQuarterlyGivenBusinessDay(event.date(), date, event.monthlyday(),static_cast<wxDateTime::Month>(event.month()-1)))
                {
                    events.push_back(event);
                }
                break;
            }
            case R_YEARLY:
            {
                if(isYearly(event.date(), date, event.day()))
                {
                    events.push_back(event);
                }
                break;
            }
            }
        }
    }
    std::sort(events.begin(), events.end(), sortEvents);
    return events;
}

bool Engine::hasEventsForCalendarDate(const wxDateTime &date)
{
    for(Event event : m_events)
    {
        if(event.realEventDate().IsSameDate(date))
        {
            return true;
        }
        else
        {
            if(date.IsEarlierThan(event.date())) continue;
            switch(event.recurrence()) {
            case R_ONCE:
            case R_ONCENOTDELETE:
            case R_NONE:
            {
                continue;
            }
            case R_MINUTES:
            case R_HOURLY:
            case R_DAILY:
            {
                return true;
            }
            case R_DAILYBUSINESS:
            {
                if(!isNonworkingDay(date)) return true;
                break;
            }
            case R_WEEKLY:
            {
                if(isWeekly(event.date().GetWeekDay(), date.GetWeekDay()))
                {
                    return true;
                }
                break;
            }
            case R_WEEKLYBUSINESS:
            {
                if(isWeeklyBusiness(static_cast<wxDateTime::WeekDay>(event.day()), date))
                {
                    return true;
                }
                break;
            }
            case R_WEEKLYPLUSBUSINESSDAY:
            {
                if(isWeeklyPlusBusinessDay(event.date(), date, static_cast<wxDateTime::WeekDay>(event.day()), event.monthlyday()))
                {
                    return true;
                }
                break;
            }
            case R_WEEKLYGIVENBUSINESSDAY:
            {
                if(isWeeklyGivenBusinessDay(event.date(), date, event.monthlyday()))
                {
                    return true;
                }
                break;
            }
            case R_2WEEKLY:
            {
                if(is2Weekly(event.date(), date))
                {
                    return true;
                }
                break;
            }
            case R_2WEEKLYBUSINESS:
            {
                if(is2WeeklyBusiness(event.date(), static_cast<wxDateTime::WeekDay>(event.day()), date))
                {
                    return true;
                }
                break;
            }
            case R_2WEEKLYPLUSBUSINESSDAY:
            {
                if(is2WeeklyPlusBusinessDay(event.date(), date, static_cast<wxDateTime::WeekDay>(event.day()), event.monthlyday()))
                {
                    return true;
                }
                break;
            }
            case R_2WEEKLYGIVENBUSINESSDAY:
            {
                if(is2WeeklyGivenBusinessDay(event.date(), date, event.monthlyday()))
                {
                    return true;
                }
                break;
            }
            case R_MONTHLY:
            {
                if(isMontly(event.date().GetDay(), date.GetDay()))
                {
                    return true;
                }
                break;
            }
            case R_OWN:
            {
                if(isOwn(event.date(), date, event.days(), event.months(), event.hours(), event.minutes(), true))
                {
                    return true;
                }
                break;
            }
            case R_MONTHLYATDAY:
            {
                if(isMonthlyAtDay(event.date(), date, event.monthlyday(), event.monthlyweek()))
                {
                    return true;
                }
                break;
            }
            case R_MONTHLYBUSINESS:
            {
                if(isMontlyBusiness(event.date(), date, event.day()))
                {
                    return true;
                }
                break;
            }
            case R_MONTHLYPLUSBUSINESSDAY:
            {
                if(isMontlyPlusBusinessDay(event.date(), date, event.day(), event.monthlyday()))
                {
                    return true;
                }
                break;
            }
            case R_MONTHLYGIVENBUSINESSDAY:
            {
                if(isMontlyGivenBusinessDay(event.date(), date, event.monthlyday()))
                {
                    return true;
                }
                break;
            }
            case R_QUARTERLY:
            {
                if(isQuarterly(event.date(), date, event.day(), static_cast<wxDateTime::Month>(event.month()-1)))
                {
                    return true;
                }
                break;
            }
            case R_QUARTERLYBUSINESS:
            {
                if(isQuarterlyBusiness(event.date(), date, event.day(), static_cast<wxDateTime::Month>(event.month()-1)))
                {
                    return true;
                }
                break;
            }
            case R_QUARTERLYPLUSBUSINESSDAY:
            {
                if(isQuarterlyPlusBusinessDay(event.date(), date, event.day(), event.monthlyday(), static_cast<wxDateTime::Month>(event.month()-1)))
                {
                    return true;
                }
                break;
            }
            case R_QUARTERLYGIVENBUSINESSDAY:
            {
                if(isQuarterlyGivenBusinessDay(event.date(), date, event.monthlyday(),static_cast<wxDateTime::Month>(event.month()-1)))
                {
                    return true;
                }
                break;
            }
            case R_YEARLY:
            {
                if(isYearly(event.date(), date, event.day()))
                {
                    return true;
                }
                break;
            }
            }
        }
    }
    return false;
}

bool Engine::isRecurrentedEventForDate(const wxDateTime &date, const Event &event)
{
    if(date.IsEarlierThan(event.date())) return false;
    if(date.IsLaterThan(event.until()) || date.IsSameDate(event.until()))
    {
        return false;
    }
    switch(event.recurrence()) {
    case R_ONCE:
    case R_ONCENOTDELETE:
    case R_NONE:
    case R_MINUTES:
    case R_HOURLY:
    case R_DAILY:
    case R_DAILYBUSINESS:
    {
        return false;
    }
    case R_WEEKLY:
    {
        if(isWeekly(event.date().GetWeekDay(), date.GetWeekDay()))
        {
            return true;
        }
        break;
    }
    case R_WEEKLYBUSINESS:
    {
        if(isWeeklyBusiness(static_cast<wxDateTime::WeekDay>(event.day()), date))
        {
            return true;
        }
        break;
    }
    case R_WEEKLYPLUSBUSINESSDAY:
    {
        if(isWeeklyPlusBusinessDay(event.date(), date, static_cast<wxDateTime::WeekDay>(event.day()), event.monthlyday()))
        {
            return true;
        }
        break;
    }
    case R_WEEKLYGIVENBUSINESSDAY:
    {
        if(isWeeklyGivenBusinessDay(event.date(), date, event.monthlyday()))
        {
            return true;
        }
        break;
    }
    case R_2WEEKLY:
    {
        if(is2Weekly(event.date(), date))
        {
            return true;
        }
        break;
    }
    case R_2WEEKLYBUSINESS:
    {
        if(is2WeeklyBusiness(event.date(), static_cast<wxDateTime::WeekDay>(event.day()), date))
        {
            return true;
        }
        break;
    }
    case R_2WEEKLYPLUSBUSINESSDAY:
    {
        if(is2WeeklyPlusBusinessDay(event.date(), date, static_cast<wxDateTime::WeekDay>(event.day()), event.monthlyday()))
        {
            return true;
        }
        break;
    }
    case R_2WEEKLYGIVENBUSINESSDAY:
    {
        if(is2WeeklyGivenBusinessDay(event.date(), date, event.monthlyday()))
        {
            return true;
        }
        break;
    }
    case R_MONTHLY:
    {
        if(isMontly(event.date().GetDay(), date.GetDay()))
        {
            return true;
        }
        break;
    }
    case R_OWN:
    {
        if(event.days()==0 && event.months()==0)
        {
            return false;
        }
        if(isOwn(event.date(), date, event.days(), event.months(), event.hours(), event.minutes()))
        {
            return true;
        }
        break;
    }
    case R_MONTHLYATDAY:
    {
        if(isMonthlyAtDay(event.date(), date, event.monthlyday(), event.monthlyweek()))
        {
            return true;
        }
        break;
    }
    case R_MONTHLYBUSINESS:
    {
        if(isMontlyBusiness(event.date(), date, event.day()))
        {
            return true;
        }
        break;
    }
    case R_MONTHLYPLUSBUSINESSDAY:
    {
        if(isMontlyPlusBusinessDay(event.date(), date, event.day(), event.monthlyday()))
        {
            return true;
        }
        break;
    }
    case R_MONTHLYGIVENBUSINESSDAY:
    {
        if(isMontlyGivenBusinessDay(event.date(), date, event.monthlyday()))
        {
            return true;
        }
        break;
    }
    case R_QUARTERLY:
    {
        if(isQuarterly(event.date(), date, event.day(), static_cast<wxDateTime::Month>(event.month()-1)))
        {
            return true;
        }
        break;
    }
    case R_QUARTERLYBUSINESS:
    {
        if(isQuarterlyBusiness(event.date(), date, event.day(), static_cast<wxDateTime::Month>(event.month()-1)))
        {
            return true;
        }
        break;
    }
    case R_QUARTERLYPLUSBUSINESSDAY:
    {
        if(isQuarterlyPlusBusinessDay(event.date(), date, event.day(), event.monthlyday(), static_cast<wxDateTime::Month>(event.month()-1)))
        {
            return true;
        }
        break;
    }
    case R_QUARTERLYGIVENBUSINESSDAY:
    {
        if(isQuarterlyGivenBusinessDay(event.date(), date, event.monthlyday(), static_cast<wxDateTime::Month>(event.month()-1)))
        {
            return true;
        }
        break;
    }
    case R_YEARLY:
    {
        if(isYearly(event.date(), date, event.day()))
        {
            return true;
        }
        break;
    }
    }
    return false;
}

wxInt64 Engine::getLastid() const
{
    return m_lastid;
}

void Engine::setLastid(const wxInt64 &lastid)
{
    m_lastid = lastid;
}

bool Engine::hasRecurrence(wxInt64 id)
{
    return std::any_of(m_events.begin(), m_events.end(), [id](const Event &ev){ return ev.id()==id && ev.recurrence()!=R_ONCE && ev.recurrence()!=R_ONCENOTDELETE && ev.recurrence()!=R_NONE; });
}

void Engine::tooltipPluginForDate()
{
    m_tooltipPlugin = "<tooltip>";
    wxString events = wxEmptyString;
    size_t i = 0;
    for(Event ev : m_events)
    {
        if(ev.realEventDate().IsSameDate(wxDateTime::Now()))
        {
            events << "\n";
            events << ev.event();
            i++;
        }
    }
    if(i == 0) m_tooltipPlugin << _("No reminders today");
    else
    {
        m_tooltipPlugin << wxString::Format(wxPLURAL("Today %zu event:","Today %zu events:",i),i);
        m_tooltipPlugin << events;
    }
}

wxDateTime Engine::nextQuarterDay(wxDateTime::wxDateTime_t eventDay, wxDateTime::Month eventMonth)
{
    wxDateTime date(1, wxDateTime::GetCurrentMonth(), wxDateTime::GetCurrentYear());
    if(eventDay > wxDateTime::Today().GetDay()) date += wxDateSpan::Month();
    else
    {
        if(eventMonth == wxDateTime::GetCurrentMonth()) date += wxDateSpan::Month();
    }
    while(!isMonthQuarterly(date.GetMonth(), eventMonth))
    {
        date += wxDateSpan::Month();
        if(eventDay > dxutils::lastdayofmonth(date.GetMonth(), date.GetYear())) date += wxDateSpan::Month();
    }
    date.SetDay(eventDay);
    return date;
}

bool Engine::isWeekly(wxDateTime::WeekDay eventWeekDay, wxDateTime::WeekDay dateWeekDay)
{
    if(eventWeekDay == dateWeekDay) return true;
    return false;
}

bool Engine::isWeeklyBusiness(wxDateTime::WeekDay eventWeekDay, const wxDateTime &date)
{
    if(!isNonworkingDay(date))
    {
        if(eventWeekDay == date.GetWeekDay()) return true;
        wxDateTime edate = date-wxDateSpan::Day();
        while(edate.GetWeekOfYear() == date.GetWeekOfYear())
        {
            if(isNonworkingDay(edate))
            {
                if(eventWeekDay == edate.GetWeekDay()) return true;
            }
            else
            {
                break;
            }
            edate -= wxDateSpan::Day();
        }
    }
    return false;
}

bool Engine::isWeeklyPlusBusinessDay(const wxDateTime &event, const wxDateTime &date, wxDateTime::WeekDay dateWeekDay, int monthlyday)
{
    if(!isNonworkingDay(date))
    {
        wxDateTime edate = date.GetWeekDayInSameWeek(dateWeekDay);
        edate = plusBusinessDay(edate, monthlyday);
        if(edate.GetWeekOfYear() == date.GetWeekOfYear() && edate.IsSameDate(date) && event <= date)
        {
            return true;
        }
    }
    return false;
}

bool Engine::isWeeklyGivenBusinessDay(const wxDateTime &event, const wxDateTime &date, int monthlyday)
{
    if(!isNonworkingDay(date))
    {
        wxDateTime edate = weeklyAtGivenBusinessDay(date, monthlyday);
        if(edate.GetWeekOfYear() == date.GetWeekOfYear() && edate.IsSameDate(date) && event <= date)
        {
            return true;
        }
    }
    return false;
}

bool Engine::is2Weekly(const wxDateTime &event, const wxDateTime &date)
{
    if((rdn(date)-rdn(event))%14==0)
        return true;
    return false;
}

bool Engine::is2WeeklyBusiness(const wxDateTime &event, wxDateTime::WeekDay eventWeekDay, const wxDateTime &date)
{
    if(!isNonworkingDay(date))
    {
        if((rdn(date.GetWeekDayInSameWeek(wxDateTime::Mon))-rdn(event.GetWeekDayInSameWeek(wxDateTime::Mon)))%14==0)
        {
            if(eventWeekDay == date.GetWeekDay()) return true;
            wxDateTime edate = date-wxDateSpan::Day();
            while(edate.GetWeekOfYear() == date.GetWeekOfYear())
            {
                if(isNonworkingDay(edate))
                {
                    if(eventWeekDay == edate.GetWeekDay()) return true;
                }
                else
                {
                    break;
                }
                edate -= wxDateSpan::Day();
            }
        }
    }
    return false;
}

bool Engine::is2WeeklyPlusBusinessDay(const wxDateTime &event, const wxDateTime &date, wxDateTime::WeekDay dateWeekDay, int monthlyday)
{
    if(!isNonworkingDay(date))
    {
        if((rdn(date.GetWeekDayInSameWeek(wxDateTime::Mon))-rdn(event.GetWeekDayInSameWeek(wxDateTime::Mon)))%14==0)
        {
            wxDateTime edate = date.GetWeekDayInSameWeek(dateWeekDay);
            edate = plusBusinessDay(edate, monthlyday);
            if(edate.GetWeekOfYear() == date.GetWeekOfYear() && edate.IsSameDate(date) && event <= date)
            {
                return true;
            }
        }
    }
    return false;
}

bool Engine::is2WeeklyGivenBusinessDay(const wxDateTime &event, const wxDateTime &date, int monthlyday)
{
    if(!isNonworkingDay(date))
    {
        if((rdn(date.GetWeekDayInSameWeek(wxDateTime::Mon))-rdn(event.GetWeekDayInSameWeek(wxDateTime::Mon)))%14==0)
        {
            wxDateTime edate = weeklyAtGivenBusinessDay(date, monthlyday);
            if(edate.GetWeekOfYear() == date.GetWeekOfYear() && edate.IsSameDate(date) && event <= date)
            {
                return true;
            }
        }
    }
    return false;
}

bool Engine::isMontly(wxDateTime::wxDateTime_t eventDay, wxDateTime::wxDateTime_t dateDay)
{
    if(eventDay == dateDay) return true;
    return false;
}

bool Engine::isOwn(const wxDateTime &event, const wxDateTime &date, int days, int months, int hours, int minutes, bool all)
{
    if(!all && days<2 && months<1) return false;
    wxDateTime edate = event+wxDateSpan(0, months, 0, days)+wxTimeSpan(hours, minutes);
    while(edate < date)
    {
        edate = edate+wxDateSpan(0, months, 0, days)+wxTimeSpan(hours, minutes);
    }
    if(edate.IsSameDate(date))
    {
        return true;
    }
    return false;
}

bool Engine::isMonthlyAtDay(const wxDateTime &event, const wxDateTime &date, int monthlyday, int monthlyweek)
{
    if(date.GetWeekDay() != static_cast<wxDateTime::WeekDay>(monthlyday))
        return false;
    if(event <= date && date.GetWeekDay(static_cast<wxDateTime::WeekDay>(monthlyday), monthlyweek, date.GetMonth(), date.GetYear()).IsValid() &&
            date.GetWeekDay(static_cast<wxDateTime::WeekDay>(monthlyday), monthlyweek, date.GetMonth(), date.GetYear()).IsSameDate(date))
    {
        return true;
    }
    return false;
}

bool Engine::isMontlyBusiness(const wxDateTime &event, const wxDateTime &date, int eventDay)
{
    if(!isNonworkingDay(date))
    {
        wxDateTime edate = wxDateTime(wxMin(eventDay,dxutils::lastdayofmonth(date.GetMonth(), date.GetYear())), date.GetMonth(), date.GetYear());
        if(isNonworkingDay(edate)) edate = nextWorkingDay(edate);
        if(edate.IsSameDate(date) && event <= date)
        {
            return true;
        }
    }
    return false;
}

bool Engine::isMontlyPlusBusinessDay(const wxDateTime &event, const wxDateTime &date, int eventDay, int monthlyday)
{
    if(!isNonworkingDay(date))
    {
        wxDateTime edate = wxDateTime(1, date.GetMonth(), date.GetYear()) - wxDateSpan::Month();
        if(eventDay <= dxutils::lastdayofmonth(edate.GetMonth(), edate.GetYear()))
        {
            edate.SetDay(eventDay);
            edate = plusBusinessDay(edate, monthlyday);
            if(edate.IsSameDate(date) && event <= date)
            {
                return true;
            }
        }
        edate = wxDateTime(1, date.GetMonth(), date.GetYear());
        if(eventDay <= dxutils::lastdayofmonth(edate.GetMonth(), edate.GetYear()))
        {
            edate.SetDay(eventDay);
            edate = plusBusinessDay(edate, monthlyday);
            if(edate.IsSameDate(date) && event <= date)
            {
                return true;
            }
        }
    }
    return false;
}

bool Engine::isMontlyGivenBusinessDay(const wxDateTime &event, const wxDateTime &date, int monthlyday)
{
    if(!isNonworkingDay(date))
    {
        wxDateTime edate = wxDateTime(1, date.GetMonth(), date.GetYear());
        edate = montlyAtGivenBusinessDay(edate, monthlyday);
        if(edate.IsSameDate(date) && event <= date)
        {
            return true;
        }
    }
    return false;
}

bool Engine::isQuarterly(const wxDateTime &event, const wxDateTime &date, int eventDay, wxDateTime::Month eventMonth)
{
    if(isMonthQuarterly(date.GetMonth(),eventMonth))
    {
        wxDateTime edate = wxDateTime(wxMin(eventDay,dxutils::lastdayofmonth(date.GetMonth(), date.GetYear())), date.GetMonth(), date.GetYear());
        if(eventDay==date.GetDay() && edate.IsSameDate(date) && event <= date)
        {
            return true;
        }
    }
    return false;
}

wxArrayString Engine::tableColumns(wxSQLite3Database *db, const wxString &name)
{
    wxArrayString arr;
    wxSQLite3ResultSet rs = db->ExecuteQuery(wxString::Format("PRAGMA table_info(%s)", name));
    while(rs.NextRow())
    {
        arr.Add(rs.GetString(1));
    }
    rs.Finalize();
    return arr;
}

bool Engine::checkTableEvents(wxSQLite3Database *db)
{
    bool result = true;
    if(db->TableExists("EVENTS"))
    {
        wxString msg = _("Missing columns at table EVENTS:")+"\n";
        wxArrayString arr = tableColumns(db,"EVENTS");
        if(arr.Index("EVENTID")==wxNOT_FOUND)
        {
            result = false;
            msg << "EVENTID\n";
        }
        if(arr.Index("EVENT")==wxNOT_FOUND)
        {
            result = false;
            msg << "EVENT\n";
        }
        if(arr.Index("DESCRIPTION")==wxNOT_FOUND)
        {
            result = false;
            msg << "DESCRIPTION\n";
        }
        if(arr.Index("EVENTDATE")==wxNOT_FOUND)
        {
            result = false;
            msg << "EVENTDATE\n";
        }
        if(arr.Index("REMINDER")==wxNOT_FOUND)
        {
            result = false;
            msg << "REMINDER\n";
        }
        if(arr.Index("ACTUALREMINDER")==wxNOT_FOUND)
        {
            result = false;
            msg << "ACTUALREMINDER\n";
        }
        if(arr.Index("REMINDED")==wxNOT_FOUND)
        {
            result = false;
            msg << "REMINDED\n";
        }
        if(arr.Index("DAY")==wxNOT_FOUND)
        {
            result = false;
            msg << "DAY\n";
        }
        if(arr.Index("HIDDEN")==wxNOT_FOUND)
        {
            result = false;
            msg << "HIDDEN\n";
        }
        if(arr.Index("MONTH")==wxNOT_FOUND)
        {
            result = false;
            msg << "MONTH\n";
        }
        if(arr.Index("RECURRENCE")==wxNOT_FOUND)
        {
            result = false;
            msg << "RECURRENCE\n";
        }
        if(arr.Index("MINUTES")==wxNOT_FOUND)
        {
            result = false;
            msg << "MINUTES\n";
        }
        if(arr.Index("HOURS")==wxNOT_FOUND)
        {
            result = false;
            msg << "HOURS\n";
        }
        if(arr.Index("DAYS")==wxNOT_FOUND)
        {
            result = false;
            msg << "DAYS\n";
        }
        if(arr.Index("MONTHS")==wxNOT_FOUND)
        {
            result = false;
            msg << "MONTHS\n";
        }
        if(arr.Index("MDAY")==wxNOT_FOUND)
        {
            result = false;
            msg << "MDAY\n";
        }
        if(arr.Index("MWEEK")==wxNOT_FOUND)
        {
            result = false;
            msg << "MWEEK\n";
        }
        if(arr.Index("UNTIL")==wxNOT_FOUND)
        {
            result = false;
            msg << "UNTIL\n";
        }
        if(arr.Index("ALWAYS")==wxNOT_FOUND)
        {
            result = false;
            msg << "ALWAYS\n";
        }
        if(arr.Index("COLOR")==wxNOT_FOUND)
        {
            result = false;
            msg << "COLOR\n";
        }
        if(!result) fireMsg(msg);
    }
    else
    {
        result = false;
        fireMsg(_("Database table EVENT doesn't exist"));
    }
    return result;
}

bool Engine::checkTableHolidays(wxSQLite3Database *db)
{
    bool result = true;
    if(db->TableExists("HOLIDAYS"))
    {
        wxString msg = _("Missing columns at table HOLIDAYS:")+"\n";
        wxArrayString arr = tableColumns(db,"HOLIDAYS");
        if(arr.Index("HOLIDAYID")==wxNOT_FOUND)
        {
            result = false;
            msg << "HOLIDAYID\n";
        }
        if(arr.Index("NAME")==wxNOT_FOUND)
        {
            result = false;
            msg << "NAME\n";
        }
        if(arr.Index("DAY")==wxNOT_FOUND)
        {
            result = false;
            msg << "DAY\n";
        }
        if(arr.Index("MONTH")==wxNOT_FOUND)
        {
            result = false;
            msg << "MONTH\n";
        }
        if(arr.Index("REMINDED")==wxNOT_FOUND)
        {
            result = false;
            msg << "REMINDED\n";
        }
        if(!result) fireMsg(msg);
    }
    else
    {
        result = false;
        fireMsg(_("Database table HOLIDAYS doesn't exist"));
    }
    return result;
}

bool Engine::checkTableNote(wxSQLite3Database *db)
{
    bool result = true;
    if(db->TableExists("NOTE"))
    {
        wxString msg = _("Missing column at table NOTE:")+"\n";
        wxArrayString arr = tableColumns(db,"NOTE");
        if(arr.Index("NTEXT")==wxNOT_FOUND)
        {
            result = false;
            msg << "NTEXT\n";
        }
        if(!result) fireMsg(msg);
    }
    else
    {
        result = false;
        fireMsg(_("Database table NOTE doesn't exist"));
    }
    return result;
}

std::vector<Calendarday> Engine::updateCalendarDays(wxDateTime::Month month, int year, bool showhidden)
{
    std::vector<Calendarday> days;
    wxArrayString tips;
    wxDateTime::wxDateTime_t lastday = dxutils::lastdayofmonth(month,year);
    for(wxDateTime::wxDateTime_t day=1; day<=lastday; day++)
    {
        tips.Clear();
        wxString ids = "0|";
        wxUint8 daytype = 0;
        for(Event ev : m_events)
        {
            if(!showhidden && ev.hidden()) continue;
            if(m_lastid == ev.id() && ev.hidden()) continue;
            if(ev.recurrence()==R_NONE) continue;
            if(ev.realEventDate().IsSameDate(wxDateTime(day,month,year)))
            {
                if(ev.recurrence()==R_ONCE || ev.recurrence()==R_ONCENOTDELETE)
                {
                    daytype = 1;
                    tips.Add(ev.tipText());
                }
                else
                {
                    daytype = 2;
                    tips.Add(ev.tipTextRecurrent());
                }
                ids << wxString::Format("%lld|",ev.id());
            }
            else
            {
                if(isRecurrentedEventForDate(wxDateTime(day,month,year), ev))
                {
                    daytype = 2;
                    ids << wxString::Format("%lld|",ev.id());
                    tips.Add(ev.tipTextRecurrent());
                }
            }
        }
        wxString tip = wxEmptyString;
        if(isHoliday(wxDateTime(day,month,year)))
        {
            tip = holidayTipTexts(wxDateTime(day,month,year));
        }
        if(tips.size())
        {
            for(size_t i=0; i<tips.GetCount(); i++)
            {
                if(tip.Len()) tip << "\n\n";
                tip << tips[i];
            }
        }
        days.push_back(Calendarday(day,daytype,tip,ids));
    }
    return days;
}

bool Engine::isQuarterlyBusiness(const wxDateTime &event, const wxDateTime &date, int eventDay, wxDateTime::Month eventMonth)
{
    if(!isNonworkingDay(date) && isMonthQuarterly(date.GetMonth(),eventMonth))
    {
        wxDateTime edate = wxDateTime(wxMin(eventDay,dxutils::lastdayofmonth(date.GetMonth(), date.GetYear())), date.GetMonth(), date.GetYear());
        if(isNonworkingDay(edate)) edate = nextWorkingDay(edate);
        if(edate.IsSameDate(date) && event <= date)
        {
            return true;
        }
    }
    return false;
}

bool Engine::isQuarterlyPlusBusinessDay(const wxDateTime &event, const wxDateTime &date, int eventDay, int monthlyday, wxDateTime::Month eventMonth)
{
    if(!isNonworkingDay(date))
    {
        wxDateTime edate = wxDateTime(1, date.GetMonth(), date.GetYear()) - wxDateSpan::Month();
        if(isMonthQuarterly(edate.GetMonth(),eventMonth) && eventDay <= dxutils::lastdayofmonth(edate.GetMonth(), edate.GetYear()))
        {
            edate.SetDay(eventDay);
            edate = plusBusinessDay(edate, monthlyday);
            if(edate.IsSameDate(date) && event <= date)
            {
                return true;
            }
        }
        if(isMonthQuarterly(date.GetMonth(),eventMonth) && eventDay <= dxutils::lastdayofmonth(date.GetMonth(), date.GetYear()))
        {
            edate = wxDateTime(eventDay, date.GetMonth(), date.GetYear());
            edate = plusBusinessDay(edate, monthlyday);
            if(edate.IsSameDate(date) && event <= date)
            {
                return true;
            }
        }
    }
    return false;
}

bool Engine::isQuarterlyGivenBusinessDay(const wxDateTime &event, const wxDateTime &date, int monthlyday, wxDateTime::Month eventMonth)
{
    if(!isNonworkingDay(date) && isMonthQuarterly(date.GetMonth(),eventMonth))
    {
        wxDateTime edate = wxDateTime(1, date.GetMonth(), date.GetYear());
        edate = montlyAtGivenBusinessDay(edate, monthlyday);
        if(edate.IsSameDate(date) && event <= date)
        {
            return true;
        }
    }
    return false;
}

bool Engine::isYearly(const wxDateTime &event, const wxDateTime &date, int eventDay)
{
    if(date.GetMonth() != event.GetMonth()) return false;
    if(date.GetDay() != event.GetDay()) return false;
    wxDateTime edate = wxDateTime(wxMin(eventDay,dxutils::lastdayofmonth(date.GetMonth(), date.GetYear())), date.GetMonth(), date.GetYear());
    if(edate.IsSameDate(date) && event < date)
    {
        return true;
    }
    return false;
}

std::vector<Event> Engine::mergeEvents() const
{
    return m_mergeEvents;
}

wxString Engine::noteText()
{
    return m_db.ExecuteQuery("select NTEXT from NOTE").GetString(0);
}

void Engine::updateNote(const wxString &text)
{
    try
    {
        if(m_db.ExecuteScalar("select count(*) from NOTE"))
        {
            wxSQLite3Statement st = m_db.PrepareStatement("update NOTE set NTEXT=?");
            st.Bind(1, text);
            st.ExecuteUpdate();
            st.Finalize();
        }
        else
        {
            wxSQLite3Statement st = m_db.PrepareStatement("insert into NOTE(NTEXT) values(?)");
            st.Bind(1, text);
            st.ExecuteUpdate();
            st.Finalize();
        }
    }
    catch(const wxSQLite3Exception& e)
    {
        wxLogError(wxString::Format("Error in updateNote: %s", e.GetMessage().c_str()));
    }
}

bool Engine::hasTableColumn(wxSQLite3Database *db, const wxString &name, const wxString &column)
{
    wxSQLite3ResultSet rs = db->ExecuteQuery(wxString::Format("PRAGMA table_info(%s)", name));
    while(rs.NextRow())
    {
        if(rs.GetString(1) == column)
        {
            rs.Finalize();
            return true;
        }
    }
    rs.Finalize();
    return false;
}

void Engine::tryExecuteUpdate(wxSQLite3Database *db, const wxString &sql)
{
    if(db->IsOpen())
    {
        try
        {
            db->ExecuteUpdate(sql);
        }
        catch(const wxSQLite3Exception& e)
        {
            wxLogError(wxString::Format("Error in tryExecuteUpdate \"%s\": %s", sql, e.GetMessage()));
        }
    }
    else
    {
        wxLogError("Database isn't open");
    }
}

void Engine::clearDB()
{
    tryExecuteUpdate(&m_db, "delete from EVENTS");
    m_lastid = -1;
    fireUpdated();
}

bool Engine::isMonthQuarterly(wxDateTime::Month month, wxDateTime::Month eventMonth)
{
    switch (eventMonth) {
    case wxDateTime::Jan:
    case wxDateTime::Apr:
    case wxDateTime::Jul:
    case wxDateTime::Oct:
    {
        if(month == wxDateTime::Jan || month == wxDateTime::Apr || month == wxDateTime::Jul || month == wxDateTime::Oct)
            return true;
        return false;
    }
    case wxDateTime::Feb:
    case wxDateTime::May:
    case wxDateTime::Aug:
    case wxDateTime::Nov:
    {
        if(month == wxDateTime::Feb || month == wxDateTime::May || month == wxDateTime::Aug || month == wxDateTime::Nov)
            return true;
        return false;
    }
    case wxDateTime::Mar:
    case wxDateTime::Jun:
    case wxDateTime::Sep:
    case wxDateTime::Dec:
    {
        if(month == wxDateTime::Mar || month == wxDateTime::Jun || month == wxDateTime::Sep || month == wxDateTime::Dec)
            return true;
        return false;
    }
    default: return false;
    }
}
