/***************************************************************
 * Name:      daydialog.cpp
 * Author:    David Vachulka (arch_dvx@users.sourceforge.net)
 * Copyright: 2020
 * License:   GPL3
 **************************************************************/

#include "dxdefs.h"
#include "daydialog.h"
#include "eventdialog.h"
#include "dxutils.h"
#include "dxicons.h"
#include "dxsettings.h"

IMPLEMENT_DYNAMIC_CLASS(DayDialog, wxDialog)

BEGIN_EVENT_TABLE(DayDialog, wxDialog)
    EVT_BUTTON(wxID_OK, DayDialog::OnOk)
    EVT_BUTTON(wxID_CANCEL, DayDialog::OnCancel)
    EVT_MENU(ID_DAYNEW, DayDialog::OnAdd)
    EVT_MENU(ID_DAYEDIT, DayDialog::OnEdit)
    EVT_MENU(ID_DAYREMOVE, DayDialog::OnRemove)
    EVT_LIST_ITEM_RIGHT_CLICK(ID_DAYLIST, DayDialog::OnListRightClick)
    EVT_LIST_ITEM_ACTIVATED(ID_DAYLIST, DayDialog::OnListActivated)
    EVT_LIST_ITEM_SELECTED(ID_DAYLIST, DayDialog::OnListSelected)
    EVT_LIST_ITEM_DESELECTED(ID_DAYLIST, DayDialog::OnListDeselected)
END_EVENT_TABLE()

DayDialog::DayDialog(wxWindow *parent, const wxDateTime &date)
    : wxDialog(parent, wxID_ANY, wxString::Format(_("Events for %s"), dxutils::formatDate(date, dxsettings.dateFormat())),
               wxDefaultPosition, wxDefaultSize, wxDEFAULT_FRAME_STYLE), m_date(date)
{
    m_events = iengine->eventsForCalendarDate(date.GetDateOnly());

    m_index = -1;

    wxBoxSizer *mainSizer = new wxBoxSizer(wxVERTICAL);
#if defined (__WXGTK__) || defined (__WXMAC__)
    m_panel = new wxPanel(this, wxID_ANY, wxDefaultPosition, wxDefaultSize, wxBORDER_SUNKEN);
#else
    m_panel = new wxPanel(this);
#endif
    wxBoxSizer *sizer1 = new wxBoxSizer(wxVERTICAL);
    m_list = new dxListCtrl(m_panel, ID_DAYLIST, false);
    m_list->InsertColumn(0, _("Events"));
    m_list->setResizeColumn(0);
    m_list->setResizeColumnMinWidth(450);
    sizer1->Add(m_list, 1, wxALL|wxEXPAND, 0);
    m_panel->SetSizer(sizer1);
    sizer1->Fit(m_panel);
    sizer1->SetSizeHints(m_panel);
    mainSizer->Add(m_panel, 1, wxALL|wxEXPAND, 5);

    m_bar = new dxToolBar(this);
    m_bar->AddTool(ID_DAYNEW, wxEmptyString, ICO_EADD, ICO_GEADD, wxITEM_NORMAL, _("New event"));
    m_bar->AddTool(ID_DAYEDIT, wxEmptyString, ICO_EEDIT, ICO_GEEDIT, wxITEM_NORMAL, _("Edit event"));
    m_bar->AddTool(ID_DAYREMOVE, wxEmptyString,ICO_EDELETE, ICO_GEDELETE, wxITEM_NORMAL, _("Remove event"));
    m_bar->Realize();
    mainSizer->Add(m_bar, 0, wxALL, 5);

    wxStdDialogButtonSizer *btnSizer = new wxStdDialogButtonSizer();
    wxButton* okButton = new wxButton(this, wxID_OK, "", wxDefaultPosition, wxDefaultSize, 0);
    btnSizer->AddButton(okButton);
    wxButton* cancelButton = new wxButton(this, wxID_CANCEL, "", wxDefaultPosition, wxDefaultSize, 0);
    btnSizer->AddButton(cancelButton);
    btnSizer->Realize();
    mainSizer->Add(btnSizer, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 5);

    this->SetSizer(mainSizer);
    mainSizer->Fit(this);
    mainSizer->SetSizeHints(this);

    fillList();
    enableTools();

    SetSize(dxsettings.daydialogW(), dxsettings.daydialogH());
}

void DayDialog::OnOk(wxCommandEvent& /*event*/)
{
    dxsettings.setDaydialogW(GetSize().x);
    dxsettings.setDaydialogH(GetSize().y);
    EndModal(wxID_OK);
    return;
}

void DayDialog::OnCancel(wxCommandEvent &/*event*/)
{
    dxsettings.setDaydialogW(GetSize().x);
    dxsettings.setDaydialogH(GetSize().y);
    EndModal(wxID_CANCEL);
    return;
}

void DayDialog::OnAdd(wxCommandEvent &/*event*/)
{
    if(dxsettings.closeDayDialog()) EndModal(wxID_OK);
    EventDialog dialog(this->GetParent(), m_date);
    dialog.ShowModal();
    m_events = iengine->eventsForCalendarDate(m_date.GetDateOnly());
    fillList();
    if(m_events.size())
    {
        m_list->SetItemState(0, wxLIST_STATE_SELECTED, wxLIST_STATE_SELECTED);
        m_index = 0;
    }
    else
    {
        m_index = -1;
    }
    enableTools();
}

void DayDialog::OnEdit(wxCommandEvent &/*event*/)
{
    if(dxsettings.closeDayDialog()) EndModal(wxID_OK);
    EventDialog dialog(this, m_events[static_cast<size_t>(m_index)]);
    dialog.ShowModal();
    m_events = iengine->eventsForCalendarDate(m_date.GetDateOnly());
    fillList();
    if(m_events.size())
    {
        wxInt64 id =  m_events[m_index].id();
        m_index = 0;
        std::vector<Event>::iterator i;
        for(i = m_events.begin(); i != m_events.end(); ++i, ++m_index)
        {
            if((*i).id() == id)
            {
                break;
            }
        }
        m_list->SetItemState(m_index, wxLIST_STATE_SELECTED, wxLIST_STATE_SELECTED);
    }
    else
    {
        m_index = -1;
        enableTools();
    }
}

void DayDialog::OnRemove(wxCommandEvent &/*event*/)
{
    if(m_index < 0) return;
    wxMessageDialog dialog(this, _("Do you really want to delete the Event?")+"\n"+m_events[static_cast<size_t>(m_index)].tipText(true), _("Confirm Event Deletion"), wxYES_NO | wxNO_DEFAULT | wxICON_QUESTION);
    if(dialog.ShowModal() == wxID_YES)
    {
        iengine->removeEvent(m_events[static_cast<size_t>(m_index)].id());
    }
    m_events = iengine->eventsForCalendarDate(m_date.GetDateOnly());
    fillList();
    if(m_events.size())
    {
        m_list->SetItemState(0, wxLIST_STATE_SELECTED, wxLIST_STATE_SELECTED);
        m_index = 0;
    }
    else
    {
        m_index = -1;
    }
    enableTools();
}

void DayDialog::OnListRightClick(wxListEvent &event)
{
    m_index = event.GetIndex();
    wxMenu popup;
    popup.Append(ID_DAYNEW, _("New event"));
    if(m_index != -1)
    {
        popup.AppendSeparator();
        popup.Append(ID_DAYEDIT, _("Edit event"));
        popup.Append(ID_DAYREMOVE, _("Remove event"));

    }
    PopupMenu(&popup, event.GetPoint()+m_panel->GetPosition());
}

void DayDialog::OnListActivated(wxListEvent &event)
{
    m_index = event.GetIndex();
    if(dxsettings.closeDayDialog()) EndModal(wxID_OK);
    EventDialog dialog(this, m_events[static_cast<size_t>(m_index)]);
    dialog.ShowModal();
    m_events = iengine->eventsForCalendarDate(m_date.GetDateOnly());
    fillList();
    if(m_events.size())
    {
        wxInt64 id =  m_events[m_index].id();
        m_index = 0;
        std::vector<Event>::iterator i;
        for(i = m_events.begin(); i != m_events.end(); ++i, ++m_index)
        {
            if((*i).id() == id)
            {
                break;
            }
        }
        m_list->SetItemState(m_index, wxLIST_STATE_SELECTED, wxLIST_STATE_SELECTED);
    }
    else
    {
        m_index = -1;
        enableTools();
    }
}

void DayDialog::OnListSelected(wxListEvent &event)
{
    m_index = event.GetIndex();
    enableTools();
}

void DayDialog::OnListDeselected(wxListEvent &/*event*/)
{
    m_index = -1;
    enableTools();
}

void DayDialog::fillList()
{
    m_list->DeleteAllItems();
    m_list->Hide();
    long row = 0;
    std::vector<Event>::iterator i;
    for(i = m_events.begin(); i != m_events.end(); ++i, row++)
    {
        wxListItem item;
        item.SetId(row);
        if((*i).recurrence() == R_ONCE || (*i).recurrence() == R_ONCENOTDELETE || (*i).recurrence() == R_NONE)
        {
            item.SetText(wxString::Format(_("Edit event %s %s"), (*i).event(),
                                          dxutils::formatDateTime((*i).realEventDate(), dxsettings.dateFormat(), dxsettings.timeFormat(), dxsettings.dateLayout())));
        }
        else
        {
            item.SetText(wxString::Format(_("Edit recurring event %s %s"), (*i).event(),
                                          dxutils::formatDateTime((*i).realEventDate(), dxsettings.dateFormat(), dxsettings.timeFormat(), dxsettings.dateLayout())));
        }
        m_list->InsertItem(item);
    }
    m_list->Show();
}

void DayDialog::enableTools()
{
    if(m_index == -1)
    {
        m_bar->EnableTool(ID_DAYNEW, true);
        m_bar->EnableTool(ID_DAYEDIT, false);
        m_bar->EnableTool(ID_DAYREMOVE, false);
    }
    else
    {
        m_bar->EnableTool(ID_DAYNEW, true);
        m_bar->EnableTool(ID_DAYEDIT, true);
        m_bar->EnableTool(ID_DAYREMOVE, true);
    }
}
