/***************************************************************
 * Name:      filter.cpp
 * Author:    David Vachulka (arch_dvx@users.sourceforge.net)
 * Copyright: 2018
 * License:   GPL3
 **************************************************************/

#include "filter.h"
#include <wx/arrimpl.cpp>
#include <wx/tokenzr.h>
#include "dxutils.h"

WX_DEFINE_OBJARRAY(dxFilterArray)

Filter::Filter(const FilterAction &action, bool active, const wxDateTime &datum)
    : m_type(FT_DATUM), m_action(action), m_active(active), m_datum(datum), m_intvalue(0), m_reminder(0), m_datespan(0), m_timespan(0)
{
}

Filter::Filter(const FilterAction &action, bool active, const wxString &text)
    : m_type(FT_TEXT), m_action(action), m_active(active), m_text(text), m_intvalue(0), m_reminder(0), m_datespan(0), m_timespan(0)
{
}

Filter::Filter(const FilterAction &action, bool active,  int value)
    : m_type(FT_INTEGER), m_action(action), m_active(active), m_intvalue(value), m_reminder(0), m_datespan(0), m_timespan(0)
{
}

Filter::Filter(const FilterAction &action, bool active, Reminder value) :
    m_type(FT_REMINDER), m_action(action), m_active(active), m_intvalue(0), m_reminder(value), m_datespan(0), m_timespan(0)
{
}

Filter::Filter(const FilterAction &action, bool active, const wxDateSpan &datespan, const wxTimeSpan &timespan)
    : m_type(FT_TIME), m_action(action), m_active(active), m_intvalue(0), m_reminder(0), m_datespan(datespan), m_timespan(timespan)
{

}

bool Filter::matchFilter(const wxDateTime &datum)
{

    if(!m_active) return true;
    if(m_type == FT_DATUM)
    {
    switch (m_action) {
    case FA_LESS:
        return datum.IsEarlierThan(m_datum);
    case FA_LESSEQUAL:
        return datum.IsEarlierThan(m_datum)||datum.IsEqualTo(m_datum);
    case FA_EQUAL:
        return datum.IsEqualTo(m_datum);
    case FA_GREATEREQUAL:
        return datum.IsLaterThan(m_datum)||datum.IsEqualTo(m_datum);
    case FA_GREATER:
        return datum.IsLaterThan(m_datum);
    default:
        break;
    }
    }
    else //FT_TIME
    {
        wxDateTime span = wxDateTime::Now()+m_datespan+m_timespan;
        return datum.IsEarlierThan(span) || datum.IsEqualTo(span);
    }
    return false;
}

bool Filter::matchFilter(const wxString &text)
{
    if(!m_active) return true;
    switch (m_action) {
    case FA_EQUAL:
        return text.Lower() == m_text.Lower();
    case FA_CONTAINS:
        return text.Lower().Contains(m_text.Lower());
    default:
        break;
    }
    return false;
}

bool Filter::matchFilter(int value)
{
    if(!m_active) return true;
    switch (m_action) {
    case FA_LESS:
        return value < m_intvalue;
    case FA_LESSEQUAL:
        return value <= m_intvalue;
    case FA_EQUAL:
        return value == m_intvalue;
    case FA_GREATEREQUAL:
        return value >= m_intvalue;
    case FA_GREATER:
        return value > m_intvalue;
    default:
        break;
    }
    return false;
}

bool Filter::matchFilter(const Reminder &reminder)
{
    if(!m_active) return true;
    switch (m_action) {
    case FA_LESS:
        return reminder.reminder() < m_reminder.reminder();
    case FA_LESSEQUAL:
        return reminder.reminder() <= m_reminder.reminder();
    case FA_EQUAL:
        return reminder.reminder() == m_reminder.reminder();
    case FA_GREATEREQUAL:
        return reminder.reminder() >= m_reminder.reminder();
    case FA_GREATER:
        return reminder.reminder() > m_reminder.reminder();
    default:
        break;
    }
    return false;
}

FilterType Filter::type() const
{
    return m_type;
}

void Filter::setType(const FilterType &type)
{
    m_type = type;
}

FilterAction Filter::action() const
{
    return m_action;
}

void Filter::setAction(const FilterAction &action)
{
    m_action = action;
}

wxDateTime Filter::datum() const
{
    return m_datum;
}

void Filter::setDatum(const wxDateTime &datum)
{
    m_datum = datum;
}

wxString Filter::text() const
{
    return m_text;
}

void Filter::setText(const wxString &text)
{
    m_text = text;
}

int Filter::intvalue() const
{
    return m_intvalue;
}

void Filter::setIntvalue(int intvalue)
{
    m_intvalue = intvalue;
}

bool Filter::active() const
{
    return m_active;
}

void Filter::setActive(bool active)
{
    m_active = active;
}

Reminder Filter::reminder() const
{
    return m_reminder;
}

void Filter::setReminder(const Reminder &reminder)
{
    m_reminder = reminder;
}

wxDateSpan Filter::datespan() const
{
    return m_datespan;
}

void Filter::setDatespan(const wxDateSpan &datespan)
{
    m_datespan = datespan;
}

wxTimeSpan Filter::timespan() const
{
    return m_timespan;
}

void Filter::setTimespan(const wxTimeSpan &timespan)
{
    m_timespan = timespan;
}


Reminder::Reminder(wxInt64 reminder)
{
    if(reminder < 0) m_reminder = 0;
    else m_reminder = reminder;
}

Reminder::Reminder(wxInt64 days, wxInt64 hours, wxInt64 mins)
{
    m_reminder = days*60*24 + hours*60 + mins;
}

Reminder::Reminder(const wxString &text)
{
    fromwxString(text);
}

wxString Reminder::towxString() const
{
    return wxString::Format("%lld|%lld|%lld",days(),hours(),mins());
}

void Reminder::fromwxString(const wxString &text)
{
    if(dxutils::numberOfChars(text,'|') == 2)
    {
        wxStringTokenizer tok(text,'|');
        wxInt64 days = 0, hours = 0, mins = 0;
        int i = 0;
        while (tok.HasMoreTokens()) {
            wxString token = tok.GetNextToken();
            switch(i) {
            case 0: token.ToLongLong(&days); break;
            case 1: token.ToLongLong(&hours); break;
            case 2: token.ToLongLong(&mins); break;
            default: break;
            }
            i++;
        }
        m_reminder = days*60*24 + hours*60 + mins;
        return;
    }
    m_reminder = 0;
}

wxInt64 Reminder::days() const
{
    return m_reminder/(60*24);
}

wxInt64 Reminder::hours() const
{
    wxInt64 mins = m_reminder;
    wxInt64 days = mins/(60*24);
    mins -= days*60*24;
    return mins/60;
}

wxInt64 Reminder::mins() const
{
    wxInt64 mins = m_reminder;
    wxInt64 days = mins/(60*24);
    mins -= days*60*24;
    wxInt64 hours = mins/60;
    mins -= hours*60;
    return mins;
}

wxInt64 Reminder::reminder() const
{
    return m_reminder;
}

void Reminder::setReminder(const wxInt64 &reminder)
{
    m_reminder = reminder;
}
