// license:BSD-3-Clause
// copyright-holders:Aaron Giles
/***************************************************************************

    Incredible Technologies/Strata system
    (8-bit blitter variant)

    driver by Aaron Giles

    Games supported:
        * Strata Bowling [3 sets]
        * Super Strike Bowling
        * Wheel of Fortune [2 sets]
        * Grudge Match
        * Golden Tee Golf [4 sets]
        * Golden Tee Golf II [3 sets]
        * Golden Par Golf [2 sets]
        * Slick Shot [3 sets]
        * Dyno-Bop
        * Arlington Horse Racing [2 sets]
        * Neck & Neck
        * Peggle [2 sets]
        * Poker Dice
        * Hot Shots Tennis [2 sets]
        * Rim Rockin' Basketball [5 sets]
        * Ninja Clowns

    Known issues:
        * None

****************************************************************************

    Memory map

****************************************************************************

    ========================================================================
    CPU #1 (6809 games)
    ========================================================================
    Note that many games have the regions 0000-0FFF and 1000-1FFF swapped.
    Also, Golden Tee Golf II (V2.2) has most of the addresses in the
    0000-0FFF range swizzled randomly.
    ========================================================================
    0100          W   xxxxxxxx    Unknown
    0120          W   xxxxxxxx    Sound data
    0140        R     xxxxxxxx    Input port
    0140          W   xxxxxxxx    GROM blitter bank select
    0160        R     xxxxxxxx    Input port
    0160          W   xx------    Display page select
                  W   x-------       (upper page select)
                  W   -x------       (unknown)
    0180        R     xxxxxxxx    Input port/DIP switches
    0180          W   xxxxxxxx    Video RAM upper data latch
    01A0          W   --------    NMI acknowledge (on some boards)
    01C0-01C1   R/W   xxxxxxxx    Blitter chip address high
    01C2-01C3   R/W   xxxxxxxx    Blitter chip address low
    01C4-01C5   R/W   ---xxxxx    Blitter chip flags
                R/W   ---x----       (render transparent)
                R/W   ----x---       (expand RLE compressed data)
                R/W   -----x--       (flip vertically)
                R/W   ------x-       (flip horizontally)
                R/W   -------x       (shift data left 4 bits)
    01C6-01C7   R     x-------    Blitter chip status (1=busy)
    01C6-01C7     W   --------    Blitter chip trigger
    01C8-01C9   R/W   xxxxxxxx    Blitter chip source width
    01CA-01CB   R/W   xxxxxxxx    Blitter chip source height
    01CC-01CD   R/W   xxxxxxxx    Blitter chip source data mask
    01CE-01CF   R/W   xxxxxxxx    Blitter chip output port
                  W   --x-----       (main program ROM bank select)
    01D0-01D1   R/W   xxxxxxxx    Blitter chip X starting coordinate
    01D2-01D3   R/W   xxxxxxxx    Blitter chip maximum scanline count
    01D4-01D5   R/W   xxxxxxxx    Blitter chip X ending coordinate
    01D6-01D7   R/W   xxxxxxxx    Blitter chip initial scanline skip
    01D8-01DF   R     xxxxxxxx    Blitter chip input ports
    01E0          W   xxxxxxxx    Palette address latch
    01E2-01E3     W   --xxxxxx    Palette data latches
    1000-11FF   R/W   xxxxxxxx    TMS34061 register access
    1200-13FF   R/W   xxxxxxxx    TMS34061 XY addressing access
    1400-15FF   R/W   xxxxxxxx    TMS34061 register access
    1600-17FF   R/W   xxxxxxxx    TMS34061 direct access
    1800-19FF   R/W   xxxxxxxx    TMS34061 shift register read
    1A00-1BFF   R/W   xxxxxxxx    TMS34061 shift register write
    2000-3FFF   R/W   xxxxxxxx    RAM (battery-backed)
    4000-7FFF   R     xxxxxxxx    Banked program ROM
    8000-FFFF   R     xxxxxxxx    Fixed program ROM
    ========================================================================
    Interrupts:
        NMI generated by VBLANK
        IRQ generated by TMS34061 scanline interrupt
        FIRQ generated by blitter when finished
    ========================================================================

    ========================================================================
    CPU #1 (68000 games)
    ========================================================================
    000000-003FFF   R/W   xxxxxxxx xxxxxxxx   RAM (battery-backed)
    004000-03FFFF   R     xxxxxxxx xxxxxxxx   Main program ROM
    100080          W     xxxxxxxx --------   Sound data
    100100          R     xxxxxxxx --------   Input port
    100100            W   xxxxxxxx --------   GROM blitter bank select
    100180          R     xxxxxxxx --------   Input port
    100100            W   xx------ --------   Display page select
                      W   x------- --------       (upper page select)
                      W   -x------ --------       (unknown)
    100240            W   xxxxxxxx --------   Video RAM upper data latch
    100280          R     xxxxxxxx --------   Input port/DIP switches
    100280            W   -------- --------   Unknown
    100300-10031F   R/W   xxxxxxxx xxxxxxxx   Blitter chip (as above)
    100380            W   xxxxxxxx --------   Palette address latch
    1003a0            W   --xxxxxx --------   Palette data latches
    110000-1101FF   R/W   xxxxxxxx xxxxxxxx   TMS34061 register access
    110200-1103FF   R/W   xxxxxxxx xxxxxxxx   TMS34061 XY addressing access
    110400-1105FF   R/W   xxxxxxxx xxxxxxxx   TMS34061 register access
    110600-1107FF   R/W   xxxxxxxx xxxxxxxx   TMS34061 direct access
    110800-1109FF   R/W   xxxxxxxx xxxxxxxx   TMS34061 shift register read
    110A00-110BFF   R/W   xxxxxxxx xxxxxxxx   TMS34061 shift register write
    ========================================================================
    Interrupts:
        INT 3 generated by VBLANK
        INT 2 generated by blitter when finished
        INT 1 generated by ???
    ========================================================================

    ========================================================================
    CPU #2
    ========================================================================
    0000          W   xxxxxxxx    Unknown
    1000        R     xxxxxxxx    Sound command input
    2000-2003   R/W   xxxxxxxx    Yamaha chip I/O
    3000-37FF   R/W   xxxxxxxx    RAM
    4000        R/W   xxxxxxxx    OKI 6295 I/O
    5000-5003   R/W   xxxxxxxx    6521 PIA chip (on early YM3812 boards)
    5000-500F   R/W   xxxxxxxx    6522 VIA chip (on later YM3812 boards)
    8000-FFFF   R     xxxxxxxx    Program ROM
    ========================================================================
    Interrupts:
        NMI not connected
        IRQ generated by write to sound command input
        FIRQ generated by Yamaha chip, or by 6522 VIA (if present)
    ========================================================================

****************************************************************************

    PCB layouts

****************************************************************************

    Summary:

        There are 8 known variants of PCBs for these games. All the PCBs
        have the following features in common:

            68B09 @ 2 MHz for the main CPU
            8k RAM for the main CPU

            68B09 @ 2 MHz for the sound CPU
            2k RAM for the sound CPU
            YM2203C, YM2608B or YM3812 for music
            OKI M6295 for speech (except for YM2608B-based system)

            TMS34061 for the video controller
            ITV4400 custom blitter for rendering
            6-bit RAMDAC for palette, or 2xTMS34070NL (Grudge Match only)
            From 2-8 64k x 4-bit VRAM chips for frame buffers

        An overview of each style PCB is given below:

        Wheel Watcher (Wheel of Fortune)-style
            * Single board
            * YM2203C for music
            * 2 VRAM chips for a single 8-bit 256x256 video page

        Grudge Match-style
            * Single board
            * YM2608B for music
            * 4 VRAM chips for two 4-bit 512x256 layers

        Strata Bowling-style
            * Single board
            * YM2203C for music
            * 3 VRAM chips for an 8-bit 256x256 video background
              plus a 4-bit 256x256 video foreground layer

        Slick Shot-style
            * Single board
            * YM2203C for music
            * 4 VRAM chips for two 8-bit 256x256 video pages
            * additional Z80 and program ROM for sensor reading

        Hot Shots Tennis-style
            * Single board
            * YM3812 for music
            * 8 VRAM chips for two 8-bit 512x256 video pages

        Rim Rockin' Basketball-style
            * Separate main and sound boards
            * HD63C09 @ 3MHz replaces 68B09 for main CPU
            * YM3812 for music
            * 8 VRAM chips for two 8-bit 512x256 video pages

        Ninja Clowns-style
            * Separate main and sound boards
            * 68000 @ 12MHz replaces 68B09 for main CPU
            * YM3812 for music
            * 8 VRAM chips for two 8-bit 512x256 video pages

        Golden Tee Golf II-style
            * Separate main and sound boards
            * YM3812 for music
            * I haven't see one yet, so I can't provide additional details

****************************************************************************

    --------------------------------
    Wheel Watcher (Wheel of Fortune)
    --------------------------------

        +------------------------------------+-+
        |    Bt476KP35                       | |
        |      12MHz  DIPSW4                 | |
        |(MT42C4064Z-10)                     | |
        |(MT42C4064Z-10)                     | |
        | MT42C4064Z-10                      | |
        | MT42C4064Z-10                      | |
        |                                    | |
        |       TMS34061                     +-
        |                                      |
        |          ITV4400   ITVS              |
        |                          YM2203C     |
        |                              OKI6295 |
        | (GROM4)  Battery                     |
        |  GROM3    CDM6264LE12      SROM0     |
        |  GROM2    WOF-PGM   HY6116           |
        |  GROM1              WOFSND      8MHz |
        |  GROM0                               |
        |        MC68B09         MC68B09       |
        +--------------------------------------+

        GROM0-3 = AM27C256
        WOF-PGM = ???
        WOFSND = AM27C256
        SROM0 = AM27C010
        ITVS = ???
        MT42C4064Z = 64k x 4 VRAM (2 populated on Wheel of Fortune)
        CDM6464 = 8k x 8 RAM
        HY6116 = 2k x 8 RAM
        Bt476 = 6-bit DAC
        TMS34061 = video controller
        ITV4400 = IT custom blitter


    -------------
    Grudge Match
    ------------

        +------------------------------------+-+
        | TMS34070NL                         | |
        | TMS34070NL  DIPSW4                 | |
        | TMS4461-15   8MHz                  | |
        | TMS4461-15                         | |
        | TMS4461-15                         | |
        | TMS4461-15                         | |
        |                                    | |
        |       TMS34061                     +-
        |                                      |
        |          ITV4400                     |
        |  GROM6                     SROM0     |
        |  GROM5                               |
        |  GROM4                               |
        |  GROM3    MK40702B-20      YM2608B   |
        |  GROM2    U5-PGM    MK6116           |
        |  GROM1              U27-SND          |
        |  GROM0                               |
        |        MC68B09         MC68B09       |
        +--------------------------------------+

        GROM0-6 = AM27C010-25
        U5-PGM = D27512-25
        U27-SND = 27C256-20
        SROM0 = AM27C010-25
        ITVS = ???
        MT42C4064Z = 64k x 4 VRAM (2 populated on Wheel of Fortune)
        MK40702B = zero-power RAM
        MK6116 = 2k x 8 RAM
        TMS34070NL = 16-color RAMDAC
        TMS34061 = video controller
        ITV4400 = IT custom blitter


    ------------------------------
    Strata Bowling/Golden Tee Golf
    ------------------------------

        +---------------------------------+-+
        |                       IMSG176   | |
        | MT42C4064Z-10                   | |
        | MT42C4064Z-10                   | |
        | MT42C4064Z-10    12MHz          | |
        |                                 | |
        |       TMS34061         DIPSW4   | |
        |                                 | |
        |    ITV4400                      +-
        |                                   |
        |        Battery        ITVS        |
        | (GROM5)                           |
        | (GROM4)                           |
        | (GROM3)   P5164                   |
        |  GROM2   SBPROG   MS6516   SROM0  |
        |  GROM1            SBSNDS          |
        |  GROM0                      8MHz  |
        |        MC68B09          MC68B09   |
        +-----------------------------------+

        GROM0-3 = AM27C010
        SBPROG = 27C256
        SBSNDS = 27C256
        SROM0 = AM27C010
        ITVS = ???
        MT42C4064Z = 64k x 4 VRAM
        P5164 = 8k x 8 RAM
        MS6516 = 2k x 8 RAM
        IMSG176 = 6-bit DAC
        TMS34061 = video controller
        ITV4400 = IT custom blitter


    ------------------------------
    Slick Shot/Poker Dice/Dyno Bop
    ------------------------------

        +------------------------------------+-+
        |   Z80PGM   4Mhz      itvs5         | |
        |            Z80       rev.1         | |
        |                                    | |
        | MT42C4064Z-10   MS176-50           | |
        | MT42C4064Z-10                      | |
        | MT42C4064Z-10   12MHz              | |
        | MT42C4064Z-10           DIPSW4     | |
        |                                    +-
        |      TMS34061                        |
        |                                      |
        |         ITV4400    ITVS              |
        |                          YM2203C     |
        |                              OKI6295 |
        | (GROM4)  Battery                     |
        | (GROM3)   P5164S           SROM0     |
        | (GROM2)   POOLPGM   P5116            |
        |  GROM1             POOLSND      8MHz |
        |  GROM0                               |
        |        MC68B09         MC68B09       |
        +--------------------------------------+

        (lower half is almost identical to Wheel of Fortune)
        (top contains extra Z80 and logic)
        (Z80 and ROM not populated on Poker Dice)

        GROM0-1 = AM27C010
        POOLPGM = ???
        POOLSND = AM27C256
        SROM0 = 27C512
        ITVS = ???
        MT42C4064Z = 64k x 4 VRAM
        P5164S = 8k x 8 RAM
        P5116 = 2k x 8 RAM
        MS176 = 6-bit DAC
        TMS34061 = video controller
        ITV4400 = IT custom blitter


    -----------------------------------
    Hot Shots Tennis/Peggle/Neck & Neck
    -----------------------------------
    (Likely Arlington Horse Racing as well)

        +------------------------------------------------+-+
        | MT42C4064Z-10    8MHz     MS176-50             | |
        | MT42C4064Z-10                                  | |
        | MT42C4064Z-10                                  | |
        | MT42C4064Z-10                                  | |
        | MT42C4064Z-10                                  | |
        | MT42C4064Z-10   TMS34061    DIPSW4             | |
        | MT42C4064Z-10           12MHz                  +-
        | MT42C4064Z-10                                    |
        |                 ITV4400                          |
        |  (GROM5)                        YM3812  OKI6295  |
        |  (GROM4) Battery  ITVS-3        MS6516           |
        |  (GROM3)   P5164              PEG-SND    SROM0   |
        |   GROM2    PEGGLE.BIN                    6821    |
        |   GROM1                                          |
        |   GROM0                                          |
        |         EF68B09     8MHz      EF68B09            |
        +--------------------------------------------------+

        GROM0-5 = AM27C010
        POOLPGM = ???
        POOLSND = AM27C256
        SROM0 = 27C512
        ITVS = ???
        MT42C4064Z = 64k x 4 VRAM
        P5164S = 8k x 8 RAM
        P5116 = 2k x 8 RAM
        MS176 = 6-bit DAC
        TMS34061 = video controller
        ITV4400 = IT custom blitter


    ----------------------
    Rim Rockin' Basketball
    ----------------------

        Lower board

        +--------------- --------------------------------+
        |               |                                |
        +---------------+   MT42C4064Z-10  MT42C4064Z-10 |
        |                   MT42C4064Z-10  MT42C4064Z-10 |
        |      IMSG176      MT42C4064Z-10  MT42C4064Z-10 |
        |                   MT42C4064Z-10  MT42C4064Z-10 |
        |  DIPSW4 8MHz ITBP-1                            |
        |                                                |
        |    MS6264-10                                   |
        |     RRB.BIM          TM34061       ITV4400     |
        |         ITVS-2                                 |
        |                      GROM02        GROM00      |
        |  Battery                                       |
        |  HD63C09   12MHz     GROM03        GROM01      |
        +------------------------------------------------+

        GROM0-3 =
        RRB.BIM =
        ITVS-2 = ???
        MT42C4064Z-10 = 64k x 4 VRAM
        MS6264 = 8k x 8 RAM
        IMSG176 = 6-bit DAC
        TMS34061 = video controller
        ITV4400 = IT custom blitter


        Sound board

        +---------------------------------+
        |         YM3812  OKI6295  SROM0  |
        | ITVS-4 MS6516-10                |
        |         NCSND                   |
        |                                 |
        |  EF68B09                        |
        |                                 |
        |   6522                          |
        | P/N 1038 REV2                   |
        +---------------------------------+

        SROM0 = M27C2001
        NCSND = AM27C256
        MS6516-10 = 2k x 8 RAM
        6522 = VIA


    ------------
    Ninja Clowns
    ------------

        Lower board

        +-----------------------------------------------------+-+-
        |  MS6264   PROM1  (PROM3)   DIPSW4                   | |
        |  MS6264   PROM0  (PROM2)                            | |
        |                                                     | |
        |  Battery        MC68000P12                          | |
        |                                                     | |
        |           12MHz   ITVS-5  ITMP-1                    | |
        |                                                     | |
        |  (GROM7)                                            +-
        |  (GROM6)          8MHz                                |
        |   GROM5                                      MS176-50 |
        |   GROM4                                               |
        |   GROM3   TMS34061   MT42C4064Z-10  MT42C4064Z-10     |
        |   GROM2              MT42C4064Z-10  MT42C4064Z-10     |
        |   GROM1   ITV4400    MT42C4064Z-10  MT42C4064Z-10     |
        |   GROM0              MT42C4064Z-10  MT42C4064Z-10     |
        | P/N 1029 REV3A                                        |
        +-------------------------------------------------------+

        GROM0-5 = M27C2001
        PROM0-1 = AM27C010
        ITVS-5 = PAL16L8
        ITMP-1 = PAL20R4
        MT42C4064Z = 64k x 4 VRAM
        MS6264 = 8k x 8 RAM
        MS176 = 6-bit DAC
        TMS34061 = video controller
        ITV4400 = IT custom blitter


        Sound board - same as Rim Rockin' Basketball

***************************************************************************/


#include "emu.h"
#include "itech8.h"

#include "cpu/m68000/m68000.h"
#include "cpu/m6809/hd6309.h"
#include "cpu/m6809/m6809.h"
#include "cpu/z80/z80.h"
#include "machine/6522via.h"
#include "machine/6821pia.h"
#include "sound/okim6295.h"
#include "sound/ymopn.h"
#include "sound/ymopl.h"

#include "speaker.h"


#define FULL_LOGGING    0

#define CLOCK_8MHz      (8000000)
#define CLOCK_12MHz     (12000000)



/*************************************
 *
 *  Interrupt handling
 *
 *************************************/

void itech8_state::update_interrupts(int periodic, int tms34061, int blitter)
{
	device_type main_cpu_type = m_maincpu->type();

	/* update the states */
	if (periodic != -1) m_periodic_int = periodic;
	if (tms34061 != -1) m_tms34061_int = tms34061;
	if (blitter != -1) m_blitter_int = blitter;

	/* handle the 6809 case */
	if (main_cpu_type == MC6809 || main_cpu_type == HD6309)
	{
		/* just modify lines that have changed */
		if (periodic != -1) m_maincpu->set_input_line(INPUT_LINE_NMI, periodic ? ASSERT_LINE : CLEAR_LINE);
		if (tms34061 != -1) m_maincpu->set_input_line(M6809_IRQ_LINE, tms34061 ? ASSERT_LINE : CLEAR_LINE);
		if (blitter != -1) m_maincpu->set_input_line(M6809_FIRQ_LINE, blitter ? ASSERT_LINE : CLEAR_LINE);
	}

	/* handle the 68000 case */
	else
	{
		m_maincpu->set_input_line(2, m_blitter_int ? ASSERT_LINE : CLEAR_LINE);
		m_maincpu->set_input_line(3, m_periodic_int ? ASSERT_LINE : CLEAR_LINE);
	}
}



/*************************************
 *
 *  Interrupt generation
 *
 *************************************/

TIMER_CALLBACK_MEMBER(itech8_state::irq_off)
{
	update_interrupts(0, -1, -1);
}


void itech8_state::generate_nmi(int state)
{
	if (state)
	{
		/* signal the NMI */
		update_interrupts(1, -1, -1);
		m_irq_off_timer->adjust(attotime::from_usec(1));

		if (FULL_LOGGING) logerror("------------ VBLANK (%d) --------------\n", m_screen->vpos());
	}
}

void itech8_state::ninclown_irq(int state)
{
	// definitely doesn't like the generate_nmi code, so we just generate VBlank irq here instead
	if (state)
		m_maincpu->set_input_line(3, HOLD_LINE);
}


void itech8_state::nmi_ack_w(uint8_t data)
{
/* doesn't seem to hold for every game (e.g., hstennis) */
/*  m_maincpu->set_input_line(INPUT_LINE_NMI, CLEAR_LINE);*/
}



/*************************************
 *
 *  Machine initialization
 *
 *************************************/




MACHINE_START_MEMBER(itech8_state,sstrike)
{
	/* we need to update behind the beam as well */
	m_behind_beam_update_timer = timer_alloc(FUNC(itech8_state::behind_the_beam_update), this);
	m_behind_beam_update_timer->adjust(m_screen->time_until_pos(0), 32);

	itech8_state::machine_start();
}

void itech8_state::machine_start()
{
	if (m_mainbank)
	{
		if (memregion("maincpu")->bytes() > 0x10000)
		{
			// rimrockn uses different banking address and more banks
			m_mainbank->configure_entries(0, 4, memregion("maincpu")->base() + 0x0000, 0x4000);
		}
		else
		{
			m_mainbank->configure_entries(0, 2, memregion("maincpu")->base() + 0x0000, 0x4000);
		}
	}

	if (m_fixed)
	{
		uint8_t* fixedstart = memregion("maincpu")->base() + memregion("maincpu")->bytes() - 0x8000; // last 0x8000 bytes of the ROM
		m_fixed->configure_entry(0, fixedstart);
		m_fixed->set_entry(0);
	}

	m_irq_off_timer = timer_alloc(FUNC(itech8_state::irq_off), this);
	m_blitter_done_timer = timer_alloc(FUNC(itech8_state::blitter_done), this);

	save_item(NAME(m_grom_bank));
	save_item(NAME(m_blitter_int));
	save_item(NAME(m_tms34061_int));
	save_item(NAME(m_periodic_int));
	save_item(NAME(m_pia_porta_data));
	save_item(NAME(m_pia_portb_data));
}

void grmatch_state::machine_start()
{
	itech8_state::machine_start();

	save_item(NAME(m_palcontrol));
	save_item(NAME(m_xscroll));
	save_item(NAME(m_palette));

	m_palette_timer = timer_alloc(FUNC(grmatch_state::palette_update), this);
}

void itech8_state::machine_reset()
{
	/* make sure bank 0 is selected */
	if (m_mainbank)
	{
		m_mainbank->set_entry(0 ^ m_bankxor);
		m_maincpu->reset();
	}

	/* set the visible area */
	if (m_visarea.width() > 1)
	{
		m_screen->set_visible_area(m_visarea.min_x, m_visarea.max_x, m_visarea.min_y, m_visarea.max_y);
		m_visarea.set(0, 0, 0, 0);
	}
}

void grmatch_state::machine_reset()
{
	itech8_state::machine_reset();
	m_palette_timer->adjust(m_screen->time_until_pos(m_screen->vpos()+1));
}


/*************************************
 *
 *  Bank switching
 *
 *************************************/

TIMER_CALLBACK_MEMBER(itech8_state::behind_the_beam_update)
{
	int scanline = param >> 8;
	int interval = param & 0xff;

	/* force a partial update to the current scanline */
	m_screen->update_partial(scanline);

	/* advance by the interval, and wrap to 0 */
	scanline += interval;
	if (scanline >= 256) scanline = 0;

	/* set a new timer */
	m_behind_beam_update_timer->adjust(m_screen->time_until_pos(scanline), (scanline << 8) + interval);
}



/*************************************
 *
 *  Bank switching
 *
 *************************************/

void itech8_state::blitter_bank_w(offs_t offset, uint8_t data)
{
	/* bit 0x20 on address 7 controls CPU banking */
	if (offset / 2 == 7)
		m_mainbank->set_entry(((data >> 5) & 1) ^ m_bankxor);

	/* the rest is handled by the video hardware */
	blitter_w(offset, data);
}


void itech8_state::rimrockn_bank_w(uint8_t data)
{
	/* banking is controlled here instead of by the blitter output */
	m_mainbank->set_entry(data & 3);
}



/*************************************
 *
 *  Input handling
 *
 *************************************/

int itech8_state::special_r()
{
	return m_pia_portb_data & 0x01;
}


/*************************************
 *
 *  6821 PIA handling
 *
 *************************************/

void itech8_state::pia_porta_out(uint8_t data)
{
	logerror("PIA port A write = %02x\n", data);
	m_pia_porta_data = data;
}


void itech8_state::pia_portb_out(uint8_t data)
{
	logerror("PIA port B write = %02x\n", data);

	/* bit 0 provides feedback to the main CPU */
	/* bit 4 controls the ticket dispenser */
	/* bit 5 controls the coin counter */
	/* bit 6 controls the diagnostic sound LED */
	m_pia_portb_data = data;
	m_ticket->motor_w(BIT(data, 4));
	machine().bookkeeping().coin_counter_w(0, BIT(data, 5));
}


void itech8_state::ym2203_portb_out(uint8_t data)
{
	logerror("YM2203 port B write = %02x\n", data);

	/* bit 0 provides feedback to the main CPU */
	/* bit 5 controls the coin counter */
	/* bit 6 controls the diagnostic sound LED */
	/* bit 7 controls the ticket dispenser */
	m_pia_portb_data = data;
	m_ticket->motor_w(BIT(data, 7));
	machine().bookkeeping().coin_counter_w(0, BIT(data, 5));
}



/*************************************
 *
 *  Sound communication
 *
 *************************************/


void itech8_state::gtg2_sound_data_w(uint8_t data)
{
	/* on the later GTG2 board, they swizzle the data lines */
	data = ((data & 0x80) >> 7) |
			((data & 0x5d) << 1) |
			((data & 0x20) >> 3) |
			((data & 0x02) << 5);
	m_soundlatch->write(data);
}


void itech8_state::grom_bank_w(uint8_t data)
{
	m_grom_bank = data;
}



/*************************************
 *
 *  16-bit-specific handlers
 *
 *************************************/

uint16_t itech8_state::rom_constant_r(offs_t offset)
{
//  Ninja Clowns reads this area for program ROM checksum
	logerror("Read ROM constant area %04x\n",offset*2+0x40000);
	return 0xd840;
}

uint8_t itech8_state::ninclown_palette_r(offs_t offset)
{
	return m_tlc34076->read(offset / 16);
}

void itech8_state::ninclown_palette_w(offs_t offset, uint8_t data)
{
	m_tlc34076->write(offset / 16, data);
}



/*************************************
 *
 *  Main CPU memory handlers
 *
 *************************************/

/*------ common layout with TMS34061 at 0000 ------*/
void itech8_state::common_lo_map(address_map &map)
{
	map(0x0000, 0x0fff).rw(FUNC(itech8_state::tms34061_r), FUNC(itech8_state::tms34061_w));
	map(0x1100, 0x1100).nopw();
	map(0x1120, 0x1120).w(m_soundlatch, FUNC(generic_latch_8_device::write));
	map(0x1140, 0x1140).portr("40").w(FUNC(itech8_state::grom_bank_w));
	map(0x1160, 0x1160).portr("60").w(FUNC(itech8_state::page_w));
	map(0x1180, 0x1180).portr("80").w(m_tms34061, FUNC(tms34061_device::latch_w));
	map(0x11a0, 0x11a0).w(FUNC(itech8_state::nmi_ack_w));
	map(0x11c0, 0x11df).rw(FUNC(itech8_state::blitter_r), FUNC(itech8_state::blitter_bank_w));
	map(0x11e0, 0x11ff).w(FUNC(itech8_state::palette_w));
	map(0x2000, 0x3fff).ram().share("nvram");
	map(0x4000, 0x7fff).bankr("mainbank");
	map(0x8000, 0xffff).bankr("fixed"); // non-banked area
}


/*------ common layout with TMS34061 at 1000 ------*/
void itech8_state::common_hi_map(address_map &map)
{
	map(0x0100, 0x0100).nopw();
	map(0x0120, 0x0120).w(m_soundlatch, FUNC(generic_latch_8_device::write));
	map(0x0140, 0x0140).portr("40").w(FUNC(itech8_state::grom_bank_w));
	map(0x0160, 0x0160).portr("60").w(FUNC(itech8_state::page_w));
	map(0x0180, 0x0180).portr("80").w(m_tms34061, FUNC(tms34061_device::latch_w));
	map(0x01a0, 0x01a0).w(FUNC(itech8_state::nmi_ack_w));
	map(0x01c0, 0x01df).rw(FUNC(itech8_state::blitter_r), FUNC(itech8_state::blitter_bank_w));
	map(0x01e0, 0x01ff).w(FUNC(itech8_state::palette_w));
	map(0x1000, 0x1fff).rw(FUNC(itech8_state::tms34061_r), FUNC(itech8_state::tms34061_w));
	map(0x2000, 0x3fff).ram().share("nvram");
	map(0x4000, 0x7fff).bankr("mainbank");
	map(0x8000, 0xffff).bankr("fixed"); // non-banked area
}


/*------ Grudge Match layout ------*/
void grmatch_state::grmatch_map(address_map &map)
{
	itech8_state::common_hi_map(map);
	map(0x0160, 0x0160).w(FUNC(grmatch_state::palette_w));
	map(0x0180, 0x0180).w(FUNC(grmatch_state::xscroll_w));
	map(0x01e0, 0x01ff).nopw();
}


/*------ Slick Shot layout ------*/
void itech8_state::slikshot_map(address_map &map)
{
	common_hi_map(map);
	map(0x0180, 0x0180).r(FUNC(itech8_state::slikshot_z80_r));
	map(0x01cf, 0x01cf).rw(FUNC(itech8_state::slikshot_z80_control_r), FUNC(itech8_state::slikshot_z80_control_w));
}


/*------ Super Strike Bowling layout ------*/
void itech8_state::sstrike_map(address_map &map)
{
	common_lo_map(map);
	map(0x1180, 0x1180).r(FUNC(itech8_state::slikshot_z80_r));
	map(0x11cf, 0x11cf).rw(FUNC(itech8_state::slikshot_z80_control_r), FUNC(itech8_state::slikshot_z80_control_w));
}


/*------ Rim Rockin' Basketball layout ------*/
void itech8_state::rimrockn_map(address_map &map)
{
	common_hi_map(map);
	map(0x0161, 0x0161).portr("161");
	map(0x0162, 0x0162).portr("162");
	map(0x0163, 0x0163).portr("163");
	map(0x0164, 0x0164).portr("164");
	map(0x0165, 0x0165).portr("165");
	map(0x01a0, 0x01a0).w(FUNC(itech8_state::rimrockn_bank_w));
	map(0x01c0, 0x01df).w(FUNC(itech8_state::blitter_w));
}


/*------ Golden Tee Golf II 1992 layout ------*/
void itech8_state::gtg2_map(address_map &map)
{
	map(0x0100, 0x0100).portr("40").w(FUNC(itech8_state::nmi_ack_w));
	map(0x0120, 0x0120).portr("60").w(FUNC(itech8_state::page_w));
	map(0x0140, 0x015f).w(FUNC(itech8_state::palette_w));
	map(0x0140, 0x0140).portr("80");
	map(0x0160, 0x0160).w(FUNC(itech8_state::grom_bank_w));
	map(0x0180, 0x019f).rw(FUNC(itech8_state::blitter_r), FUNC(itech8_state::blitter_bank_w));
	map(0x01c0, 0x01c0).w(FUNC(itech8_state::gtg2_sound_data_w));
	map(0x01e0, 0x01e0).w(m_tms34061, FUNC(tms34061_device::latch_w));
	map(0x1000, 0x1fff).rw(FUNC(itech8_state::tms34061_r), FUNC(itech8_state::tms34061_w));
	map(0x2000, 0x3fff).ram().share("nvram");
	map(0x4000, 0x7fff).bankr("mainbank");
	map(0x8000, 0xffff).bankr("fixed"); // non-banked area
}

/*------ Ninja Clowns layout ------*/
void itech8_state::ninclown_map(address_map &map)
{
	map(0x000000, 0x003fff).ram().share("nvram");
	map(0x000000, 0x000007).rom();
	map(0x004000, 0x03ffff).rom();
	map(0x040000, 0x07ffff).r(FUNC(itech8_state::rom_constant_r));
	map(0x100080, 0x100080).w(m_soundlatch, FUNC(generic_latch_8_device::write));
	map(0x100100, 0x100100).w(FUNC(itech8_state::grom_bank_w));
	map(0x100100, 0x100101).portr("40");
	map(0x100180, 0x100180).lw8(NAME([this](u8 data){ page_w(~data); }));
	map(0x100180, 0x100181).portr("60");
	map(0x100240, 0x100240).w(m_tms34061, FUNC(tms34061_device::latch_w));
	map(0x100280, 0x100281).portr("80").nopw();
	map(0x100300, 0x10031f).rw(FUNC(itech8_state::blitter_r), FUNC(itech8_state::blitter_w));
	map(0x100380, 0x1003ff).rw(FUNC(itech8_state::ninclown_palette_r), FUNC(itech8_state::ninclown_palette_w)).umask16(0xff00);
	map(0x110000, 0x110fff).rw(FUNC(itech8_state::tms34061_r), FUNC(itech8_state::tms34061_w));
}



/*************************************
 *
 *  Sound CPU memory handlers
 *
 *************************************/

/*------ YM2203-based sound ------*/
void itech8_state::sound2203_map(address_map &map)
{
	map(0x0000, 0x0000).nopw();
	map(0x1000, 0x1000).r(m_soundlatch, FUNC(generic_latch_8_device::read));
	map(0x2000, 0x2001).mirror(0x0002).rw("ymsnd", FUNC(ym2203_device::read), FUNC(ym2203_device::write));
	map(0x3000, 0x37ff).ram();
	map(0x4000, 0x4000).rw("oki", FUNC(okim6295_device::read), FUNC(okim6295_device::write));
	map(0x8000, 0xffff).rom();
}


/*------ YM2608B-based sound ------*/
void itech8_state::sound2608b_map(address_map &map)
{
	map(0x1000, 0x1000).nopw();
	map(0x2000, 0x2000).r(m_soundlatch, FUNC(generic_latch_8_device::read));
	map(0x4000, 0x4003).rw("ymsnd", FUNC(ym2608_device::read), FUNC(ym2608_device::write));
	map(0x6000, 0x67ff).ram();
	map(0x8000, 0xffff).rom();
}


/*------ YM3812-based sound ------*/
void itech8_state::sound3812_map(address_map &map)
{
	map(0x0000, 0x0000).nopw();
	map(0x1000, 0x1000).r(m_soundlatch, FUNC(generic_latch_8_device::read));
	map(0x2000, 0x2001).rw("ymsnd", FUNC(ym3812_device::read), FUNC(ym3812_device::write));
	map(0x3000, 0x37ff).ram();
	map(0x4000, 0x4000).rw("oki", FUNC(okim6295_device::read), FUNC(okim6295_device::write));
	map(0x5000, 0x5003).rw("pia", FUNC(pia6821_device::read), FUNC(pia6821_device::write));
	map(0x8000, 0xffff).rom();
}


/*------ external YM3812-based sound board ------*/
void itech8_state::sound3812_external_map(address_map &map)
{
	map(0x0000, 0x0000).nopw();
	map(0x1000, 0x1000).r(m_soundlatch, FUNC(generic_latch_8_device::read));
	map(0x2000, 0x2001).rw("ymsnd", FUNC(ym3812_device::read), FUNC(ym3812_device::write));
	map(0x3000, 0x37ff).ram();
	map(0x4000, 0x4000).rw("oki", FUNC(okim6295_device::read), FUNC(okim6295_device::write));
	map(0x5000, 0x500f).m("via6522_0", FUNC(via6522_device::map));
	map(0x8000, 0xffff).rom();
}



/*************************************
 *
 *  Other CPU memory handlers
 *
 *************************************/

void itech8_state::slikz80_mem_map(address_map &map)
{
	map(0x0000, 0x7ff).rom();
}


void itech8_state::slikz80_io_map(address_map &map)
{
	map.global_mask(0xff);
	map(0x00, 0x00).rw(FUNC(itech8_state::slikz80_port_r), FUNC(itech8_state::slikz80_port_w));
}



/*************************************
 *
 *  Port definitions
 *
 *************************************/

static INPUT_PORTS_START( wfortune )
	PORT_START("40")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_CUSTOM ) PORT_READ_LINE_MEMBER(itech8_state, special_r) /* input from sound board */
	PORT_BIT( 0x06, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )
	PORT_BIT( 0x70, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START("60")
	PORT_BIT( 0x07, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("Blue Player") PORT_PLAYER(3)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("Yellow Player") PORT_PLAYER(2)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("Red Player") PORT_PLAYER(1)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN2 )

	PORT_START("80")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("AN_D")  /* analog D */
	PORT_BIT( 0xff, 0x00, IPT_DIAL ) PORT_SENSITIVITY(75) PORT_KEYDELTA(10) PORT_PLAYER(1)

	PORT_START("AN_F")  /* analog F */
	PORT_BIT( 0xff, 0x00, IPT_DIAL ) PORT_SENSITIVITY(75) PORT_KEYDELTA(10) PORT_COCKTAIL PORT_PLAYER(2)
INPUT_PORTS_END


static INPUT_PORTS_START( grmatch )
	PORT_START("40")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_CUSTOM ) PORT_READ_LINE_MEMBER(itech8_state, special_r) /* input from sound board */
	PORT_BIT( 0x06, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x08, 0x08, "Adjustments Lockout" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x10, DEF_STR( 1C_1C ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x20, "1" )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START("60")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(1)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_PLAYER(1)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_PLAYER(1)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START("80")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(2)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_PLAYER(2)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_PLAYER(2)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_PLAYER(2)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN2 )
INPUT_PORTS_END


static INPUT_PORTS_START( stratab )
	PORT_START("40")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_CUSTOM ) PORT_READ_LINE_MEMBER(itech8_state, special_r) /* input from sound board */
	PORT_BIT( 0x06, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )
	PORT_BIT( 0x70, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START("60")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P2 Right Hook") PORT_COCKTAIL PORT_PLAYER(2)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P2 Left Hook") PORT_COCKTAIL PORT_PLAYER(2)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P1 Right Hook") PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P1 Left Hook") PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START("80")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("AN_C")  /* analog C */
	PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_Y ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_PLAYER(1)

	PORT_START("AN_D")  /* analog D */
	PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_X ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_REVERSE PORT_PLAYER(1)

	PORT_START("AN_E")  /* analog E */
	PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_Y ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_COCKTAIL PORT_PLAYER(2)

	PORT_START("AN_F")  /* analog F */
	PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_X ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_REVERSE PORT_COCKTAIL PORT_PLAYER(2)
INPUT_PORTS_END

static INPUT_PORTS_START( stratabs )
	PORT_INCLUDE( stratab )

	PORT_START("FAKEX") /* fake */
	PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_X ) PORT_SENSITIVITY(50) PORT_KEYDELTA(10) PORT_RESET PORT_PLAYER(1)

	PORT_START("FAKEY") /* fake */
	PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_Y ) PORT_SENSITIVITY(50) PORT_KEYDELTA(100) PORT_RESET PORT_REVERSE PORT_PLAYER(1)
INPUT_PORTS_END



CUSTOM_INPUT_MEMBER(itech8_state::gtg_mux)
{
	return m_p1->read() & m_p2->read();
}

static INPUT_PORTS_START( gtg )
	PORT_START("40")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_CUSTOM ) PORT_READ_LINE_MEMBER(itech8_state, special_r) /* input from sound board */
	PORT_BIT( 0x06, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )
	PORT_BIT( 0x70, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START("60")
	/* it is still unknown how the second player inputs are muxed in */
	/* currently we map both sets of controls to the same inputs */
	PORT_BIT( 0x1f, IP_ACTIVE_HIGH, IPT_CUSTOM ) PORT_CUSTOM_MEMBER(itech8_state, gtg_mux)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START("80")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("P1")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_PLAYER(1)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_PLAYER(1)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P1 Swing") PORT_PLAYER(1)

	PORT_START("P2")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_COCKTAIL PORT_PLAYER(2)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_COCKTAIL PORT_PLAYER(2)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_COCKTAIL PORT_PLAYER(2)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_COCKTAIL PORT_PLAYER(2)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P2 Swing") PORT_COCKTAIL PORT_PLAYER(2)
INPUT_PORTS_END


static INPUT_PORTS_START( gtgt )
	PORT_START("40")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_CUSTOM ) PORT_READ_LINE_MEMBER(itech8_state, special_r) /* input from sound board */
	PORT_BIT( 0x7e, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START("60")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("Face Right") PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("Face Left") PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START("80")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("AN_C")  /* analog C */
	PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_X ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_PLAYER(1)

	PORT_START("AN_D")  /* analog D */
	PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_Y ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_REVERSE PORT_PLAYER(1)
INPUT_PORTS_END


static INPUT_PORTS_START( gtg2t )
	PORT_START("40")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_CUSTOM ) PORT_READ_LINE_MEMBER(itech8_state, special_r) /* input from sound board */
	PORT_BIT( 0x06, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )
	PORT_BIT( 0x70, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START("60")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P2 Face Right") PORT_COCKTAIL PORT_PLAYER(2)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P2 Face Left") PORT_COCKTAIL PORT_PLAYER(2)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P1 Face Right") PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P1 Face Left") PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START("80")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("AN_C")  /* analog C */
	PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_X ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_PLAYER(1)

	PORT_START("AN_D")  /* analog D */
	PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_Y ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_REVERSE PORT_PLAYER(1)

	PORT_START("AN_E")  /* analog E */
	PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_X ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_COCKTAIL PORT_PLAYER(2)

	PORT_START("AN_F")  /* analog F */
	PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_Y ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_REVERSE PORT_COCKTAIL PORT_PLAYER(2)
INPUT_PORTS_END


static INPUT_PORTS_START( slikshot )
	PORT_START("40")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_CUSTOM ) PORT_READ_LINE_MEMBER(itech8_state, special_r) /* input from sound board */
	PORT_BIT( 0x7e, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START("60")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_CUSTOM )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("Yellow")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_NAME("Red")
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("Green")
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START("80")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("FAKEX") /* fake */
	PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_X ) PORT_SENSITIVITY(50) PORT_KEYDELTA(10) PORT_RESET PORT_PLAYER(1)

	PORT_START("FAKEY") /* fake */
	PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_Y ) PORT_SENSITIVITY(50) PORT_KEYDELTA(100) PORT_RESET PORT_REVERSE PORT_PLAYER(1)
INPUT_PORTS_END


static INPUT_PORTS_START( dynobop )
	PORT_START("40")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_CUSTOM ) PORT_READ_LINE_MEMBER(itech8_state, special_r) /* input from sound board */
	PORT_BIT( 0x7e, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START("60")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_CUSTOM )   /* ball gate */
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_CUSTOM )   /* ball detect */
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START("80")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("FAKEX") /* fake */
	PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_X ) PORT_SENSITIVITY(50) PORT_KEYDELTA(10) PORT_RESET PORT_PLAYER(1)

	PORT_START("FAKEY") /* fake */
	PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_Y ) PORT_SENSITIVITY(50) PORT_KEYDELTA(100) PORT_RESET PORT_REVERSE PORT_PLAYER(1)
INPUT_PORTS_END


static INPUT_PORTS_START( sstrike )
	PORT_START("40")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_CUSTOM ) PORT_READ_LINE_MEMBER(itech8_state, special_r) /* input from sound board */
	PORT_BIT( 0x7e, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START("60")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_CUSTOM )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("Left Hook")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("Right Hook")
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START("80")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("FAKEX") /* fake */
	PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_X ) PORT_SENSITIVITY(50) PORT_KEYDELTA(10) PORT_RESET PORT_PLAYER(1)

	PORT_START("FAKEY") /* fake */
	PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_Y ) PORT_SENSITIVITY(50) PORT_KEYDELTA(100) PORT_RESET PORT_REVERSE PORT_PLAYER(1)
INPUT_PORTS_END


static INPUT_PORTS_START( pokrdice )
	PORT_START("40")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_CUSTOM ) PORT_READ_LINE_MEMBER(itech8_state, special_r) /* input from sound board */
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_NAME("Lower Right") PORT_CODE(KEYCODE_3_PAD)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_BIT( 0x60, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START("60")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_NAME("Upper Right") PORT_CODE(KEYCODE_9_PAD)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON5 ) PORT_NAME("Middle") PORT_CODE(KEYCODE_5_PAD)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON6 ) PORT_NAME("Lower Left") PORT_CODE(KEYCODE_1_PAD)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("Raise")
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_NAME("Upper Left") PORT_CODE(KEYCODE_7_PAD)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("Play")
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START("80")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END


static INPUT_PORTS_START( hstennis )
	PORT_START("40")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_CUSTOM ) PORT_READ_LINE_MEMBER(itech8_state, special_r) /* input from sound board */
	PORT_BIT( 0x06, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )  /* see code at fbb5 */
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_BIT( 0x60, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START("60")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P1 Soft") PORT_PLAYER(1)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_PLAYER(1)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_PLAYER(1)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P1 Hard") PORT_PLAYER(1)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START("80")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P2 Soft") PORT_PLAYER(2)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(2)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_PLAYER(2)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_PLAYER(2)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_PLAYER(2)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P2 Hard") PORT_PLAYER(2)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN2 )
INPUT_PORTS_END


static INPUT_PORTS_START( arlingtn )
	PORT_START("40")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_CUSTOM ) PORT_READ_LINE_MEMBER(itech8_state, special_r) /* input from sound board */
	PORT_BIT( 0x06, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )  /* see code at e23c */
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_BIT( 0x70, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START("60")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("Place")
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("Win")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_PLAYER(1)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_PLAYER(1)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START("80")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x06, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_NAME("Show")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START1 ) PORT_NAME("Start Race")
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_NAME("Collect")
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN2 )
INPUT_PORTS_END


static INPUT_PORTS_START( peggle )
	PORT_START("40")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_CUSTOM ) PORT_READ_LINE_MEMBER(itech8_state, special_r) /* input from sound board */
	PORT_BIT( 0x7e, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START("60")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_PLAYER(1)
	PORT_BIT( 0x30, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START("80")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x7e, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN2 )
INPUT_PORTS_END


static INPUT_PORTS_START( pegglet )
	PORT_START("40")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_CUSTOM ) PORT_READ_LINE_MEMBER(itech8_state, special_r) /* input from sound board */
	PORT_BIT( 0x7e, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START("60")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x3e, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START("80")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x7e, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN2 )

	PORT_START("AN_D")      /* analog D */
	PORT_BIT( 0xff, 0x00, IPT_DIAL ) PORT_SENSITIVITY(50) PORT_KEYDELTA(10) PORT_RESET PORT_PLAYER(1)
INPUT_PORTS_END


static INPUT_PORTS_START( neckneck )
	PORT_START("40")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_CUSTOM ) PORT_READ_LINE_MEMBER(itech8_state, special_r) /* input from sound board */
	PORT_BIT( 0x06, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x08, 0x00, DEF_STR( Unknown ) )  /* see code at e23c */
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_BIT( 0x70, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE( 0x80, IP_ACTIVE_LOW )

	PORT_START("60")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_NAME("Horse 3")
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("Horse 2")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("Horse 1")
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START("80")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x06, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_NAME("Horse 4")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON6 ) PORT_NAME("Horse 6")
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON5 ) PORT_NAME("Horse 5")
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN2 )
INPUT_PORTS_END


static INPUT_PORTS_START( rimrockn )
	PORT_START("40")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("60")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_SERVICE_NO_TOGGLE( 0x02, IP_ACTIVE_LOW )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_CUSTOM ) PORT_READ_LINE_MEMBER(itech8_state, special_r) /* input from sound board */

	PORT_START("80")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("161")   /* special 161 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P1 Pass") PORT_PLAYER(1)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P1 Shoot") PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_PLAYER(1)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_PLAYER(1)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_PLAYER(1)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START1 )

	PORT_START("162")   /* special 162 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P2 Pass") PORT_PLAYER(2)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P2 Shoot") PORT_PLAYER(2)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(2)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_PLAYER(2)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_PLAYER(2)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_PLAYER(2)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START2 )

	PORT_START("163")   /* special 163 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P3 Shoot") PORT_PLAYER(3)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P3 Pass") PORT_PLAYER(3)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(3)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_PLAYER(3)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_PLAYER(3)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_PLAYER(3)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START3 )

	PORT_START("164")   /* special 164 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P4 Shoot") PORT_PLAYER(4)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P4 Pass") PORT_PLAYER(4)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(4)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_PLAYER(4)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_PLAYER(4)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_PLAYER(4)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START4 )

	PORT_START("165")   /* special 165 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_SERVICE2 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_SERVICE3 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_SERVICE4 )
	PORT_DIPNAME( 0x18, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x18, "1 player" )
	PORT_DIPSETTING(    0x10, "2 players" )
	PORT_DIPSETTING(    0x08, "3 players" )
	PORT_DIPSETTING(    0x00, "4 players" )
	PORT_DIPNAME( 0x20, 0x00, "Coin Slots" )
	PORT_DIPSETTING(    0x04, "Common" )
	PORT_DIPSETTING(    0x00, "Individual" )
	PORT_DIPNAME( 0x40, 0x00, "Video Sync" )
	PORT_DIPSETTING(    0x02, "Positive" )
	PORT_DIPSETTING(    0x00, "Negative" )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


static INPUT_PORTS_START( ninclown )
	PORT_START("40")
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_SERVICE_NO_TOGGLE( 0x0200, IP_ACTIVE_LOW )
	PORT_BIT( 0x0c00, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0xc000, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("60")
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_NAME("P1 Throw") PORT_PLAYER(1)
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_PLAYER(1)
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_PLAYER(1)
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_PLAYER(1)
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(1)
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P1 Kick") PORT_PLAYER(1)
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P1 Punch") PORT_PLAYER(1)

	PORT_START("80")
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_NAME("P2 Throw") PORT_PLAYER(2)
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_PLAYER(2)
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_PLAYER(2)
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_PLAYER(2)
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(2)
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P2 Kick") PORT_PLAYER(2)
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P2 Punch") PORT_PLAYER(2)
INPUT_PORTS_END


static INPUT_PORTS_START( gtg2 )
	PORT_START("40")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_SERVICE_NO_TOGGLE( 0x02, IP_ACTIVE_LOW )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x30, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("60")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P1 Face Right") PORT_PLAYER(1)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P1 Face Left") PORT_PLAYER(1)
	PORT_BIT( 0x78, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START1 )

	PORT_START("80")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_NAME("P2 Face Right") PORT_COCKTAIL PORT_PLAYER(2)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P2 Face Left") PORT_COCKTAIL PORT_PLAYER(2)
	PORT_BIT( 0x78, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START2 )

	PORT_START("AN_C")  /* analog C */
	PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_X ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_PLAYER(1)

	PORT_START("AN_D")  /* analog D */
	PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_Y ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_REVERSE PORT_PLAYER(1)

	PORT_START("AN_E")  /* analog E */
	PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_X ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_COCKTAIL PORT_PLAYER(2)

	PORT_START("AN_F")  /* analog F */
	PORT_BIT( 0xff, 0x00, IPT_TRACKBALL_Y ) PORT_SENSITIVITY(25) PORT_KEYDELTA(32) PORT_RESET PORT_REVERSE PORT_COCKTAIL PORT_PLAYER(2)
INPUT_PORTS_END


static INPUT_PORTS_START( gpgolf )
	PORT_START("40")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_SERVICE_NO_TOGGLE( 0x02, IP_ACTIVE_LOW )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x30, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("60")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_NAME("P1 Swing") PORT_PLAYER(1)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_PLAYER(1)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_PLAYER(1)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_PLAYER(1)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_PLAYER(1)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START1 )

	PORT_START("80")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END

/*************************************
 *
 *  TMS34061 interfacing
 *
 *************************************/

void itech8_state::generate_tms34061_interrupt(int state)
{
	update_interrupts(-1, state, -1);

	if (FULL_LOGGING && state) logerror("------------ DISPLAY INT (%d) --------------\n", m_screen->vpos());
}

/*************************************
 *
 *  Machine driver
 *
 *************************************/

/************* core pieces ******************/

void itech8_state::itech8_core_devices(machine_config &config)
{
	NVRAM(config, m_nvram, nvram_device::DEFAULT_RANDOM);

	TICKET_DISPENSER(config, m_ticket, attotime::from_msec(200), TICKET_MOTOR_ACTIVE_HIGH, TICKET_STATUS_ACTIVE_LOW);

	TLC34076(config, m_tlc34076, tlc34076_device::TLC34076_6_BIT);

	SCREEN(config, m_screen, SCREEN_TYPE_RASTER);
	m_screen->set_video_attributes(VIDEO_UPDATE_BEFORE_VBLANK);
	m_screen->set_refresh_hz(60);
	m_screen->set_size(512, 263);
	m_screen->screen_vblank().set(FUNC(itech8_state::generate_nmi));

	TMS34061(config, m_tms34061, 0);
	m_tms34061->set_rowshift(8);  /* VRAM address is (row << rowshift) | col */
	m_tms34061->set_vram_size(itech8_state::VRAM_SIZE);
	m_tms34061->int_callback().set(FUNC(itech8_state::generate_tms34061_interrupt));      /* interrupt gen callback */

	SPEAKER(config, "mono").front_center();

	GENERIC_LATCH_8(config, m_soundlatch, 0);
	m_soundlatch->data_pending_callback().set_inputline(m_soundcpu, M6809_IRQ_LINE);

	via6522_device &via(MOS6522(config, "via6522_0", CLOCK_8MHz/4));
	via.writepb_handler().set(FUNC(itech8_state::pia_portb_out));
	via.irq_handler().set_inputline(m_soundcpu, M6809_FIRQ_LINE);
}

void itech8_state::itech8_core_lo(machine_config &config)
{
	MC6809(config, m_maincpu, CLOCK_8MHz);
	m_maincpu->set_addrmap(AS_PROGRAM, &itech8_state::common_lo_map);

	itech8_core_devices(config);
}

void itech8_state::itech8_core_hi(machine_config &config)
{
	itech8_core_lo(config);
	m_maincpu->set_addrmap(AS_PROGRAM, &itech8_state::common_hi_map);
}

void itech8_state::itech8_sound_ym2203(machine_config &config)
{
	/* basic machine hardware */
	MC6809(config, m_soundcpu, CLOCK_8MHz);
	m_soundcpu->set_addrmap(AS_PROGRAM, &itech8_state::sound2203_map);

	/* sound hardware */
	ym2203_device &ymsnd(YM2203(config, "ymsnd", CLOCK_8MHz/2));
	ymsnd.irq_handler().set_inputline(m_soundcpu, M6809_FIRQ_LINE);
	ymsnd.port_b_write_callback().set(FUNC(itech8_state::ym2203_portb_out));
	ymsnd.add_route(0, "mono", 0.07);
	ymsnd.add_route(1, "mono", 0.07);
	ymsnd.add_route(2, "mono", 0.07);
	ymsnd.add_route(3, "mono", 0.75);

	okim6295_device &oki(OKIM6295(config, "oki", CLOCK_8MHz/8, okim6295_device::PIN7_HIGH)); // was /128, not /132, so unsure so pin 7 not verified
	oki.add_route(ALL_OUTPUTS, "mono", 0.75);
}

void itech8_state::itech8_sound_ym2608b(machine_config &config)
{
	/* basic machine hardware */
	MC6809(config, m_soundcpu, CLOCK_8MHz);
	m_soundcpu->set_addrmap(AS_PROGRAM, &itech8_state::sound2608b_map);

	/* sound hardware */
	ym2608_device &ymsnd(YM2608(config, "ymsnd", CLOCK_8MHz));
	ymsnd.irq_handler().set_inputline(m_soundcpu, M6809_FIRQ_LINE);
	ymsnd.port_b_write_callback().set(FUNC(itech8_state::ym2203_portb_out));
	ymsnd.add_route(ALL_OUTPUTS, "mono", 0.75);
}

void itech8_state::itech8_sound_ym3812(machine_config &config)
{
	/* basic machine hardware */
	MC6809(config, m_soundcpu, CLOCK_8MHz);
	m_soundcpu->set_addrmap(AS_PROGRAM, &itech8_state::sound3812_map);

	pia6821_device &pia(PIA6821(config, "pia", 0));
	pia.readpb_handler().set("ticket", FUNC(ticket_dispenser_device::line_r));
	pia.writepa_handler().set(FUNC(itech8_state::pia_porta_out));
	pia.writepb_handler().set(FUNC(itech8_state::pia_portb_out));

	/* sound hardware */
	ym3812_device &ymsnd(YM3812(config, "ymsnd", CLOCK_8MHz/2));
	ymsnd.irq_handler().set_inputline(m_soundcpu, M6809_FIRQ_LINE);
	ymsnd.add_route(ALL_OUTPUTS, "mono", 0.75);

	okim6295_device &oki(OKIM6295(config, "oki", CLOCK_8MHz/8, okim6295_device::PIN7_HIGH)); // was /128, not /132, so unsure so pin 7 not verified
	oki.add_route(ALL_OUTPUTS, "mono", 0.75);
}

void itech8_state::itech8_sound_ym3812_external(machine_config &config)
{
	/* basic machine hardware */
	MC6809(config, m_soundcpu, CLOCK_8MHz);
	m_soundcpu->set_addrmap(AS_PROGRAM, &itech8_state::sound3812_external_map);

	/* sound hardware */
	ym3812_device &ymsnd(YM3812(config, "ymsnd", CLOCK_8MHz/2));
	ymsnd.irq_handler().set_inputline(m_soundcpu, M6809_FIRQ_LINE);
	ymsnd.add_route(ALL_OUTPUTS, "mono", 0.75);

	okim6295_device &oki(OKIM6295(config, "oki", CLOCK_8MHz/8, okim6295_device::PIN7_HIGH)); // was /128, not /132, so unsure so pin 7 not verified
	oki.add_route(ALL_OUTPUTS, "mono", 0.75);
}


/************* full drivers ******************/

void itech8_state::wfortune(machine_config &config)
{
	itech8_core_hi(config);
	itech8_sound_ym2203(config);

	m_screen->set_visarea(0, 255, 0, 239);
	m_screen->set_screen_update(FUNC(itech8_state::screen_update_2layer));
}

void grmatch_state::grmatch(machine_config &config)
{
	itech8_core_hi(config);
	itech8_sound_ym2608b(config);

	m_maincpu->set_addrmap(AS_PROGRAM, &grmatch_state::grmatch_map);

	m_screen->set_visarea(0, 399, 0, 239);
	m_screen->set_screen_update(FUNC(grmatch_state::screen_update));
}

void itech8_state::stratab_hi(machine_config &config)
{
	itech8_core_hi(config);
	itech8_sound_ym2203(config);

	m_screen->set_visarea(0, 255, 0, 239);
	m_screen->set_screen_update(FUNC(itech8_state::screen_update_2layer));
}

void itech8_state::stratab_lo(machine_config &config)
{
	itech8_core_lo(config);
	itech8_sound_ym2203(config);

	m_screen->set_visarea(0, 255, 0, 239);
	m_screen->set_screen_update(FUNC(itech8_state::screen_update_2layer));
}

void itech8_state::slikshot_hi(machine_config &config)
{
	itech8_core_hi(config);
	itech8_sound_ym2203(config);

	m_maincpu->set_addrmap(AS_PROGRAM, &itech8_state::slikshot_map);

	Z80(config, m_subcpu, CLOCK_8MHz/2);
	m_subcpu->set_addrmap(AS_PROGRAM, &itech8_state::slikz80_mem_map);
	m_subcpu->set_addrmap(AS_IO, &itech8_state::slikz80_io_map);

	m_screen->set_visarea(0, 255, 0, 239);
	m_screen->set_screen_update(FUNC(itech8_state::screen_update_slikshot));
	MCFG_VIDEO_START_OVERRIDE(itech8_state, slikshot)
}

void itech8_state::slikshot_lo(machine_config &config)
{
	itech8_core_lo(config);
	itech8_sound_ym2203(config);

	m_maincpu->set_addrmap(AS_PROGRAM, &itech8_state::sstrike_map);

	Z80(config, m_subcpu, CLOCK_8MHz/2);
	m_subcpu->set_addrmap(AS_PROGRAM, &itech8_state::slikz80_mem_map);
	m_subcpu->set_addrmap(AS_IO, &itech8_state::slikz80_io_map);

	m_screen->set_visarea(0, 255, 0, 239);
	m_screen->set_screen_update(FUNC(itech8_state::screen_update_slikshot));
	MCFG_VIDEO_START_OVERRIDE(itech8_state, slikshot)
}

void itech8_state::slikshot_lo_noz80(machine_config &config)
{
	itech8_core_lo(config);
	itech8_sound_ym2203(config);

	m_screen->set_visarea(0, 255, 0, 239);
	m_screen->set_screen_update(FUNC(itech8_state::screen_update_2page));
}

void itech8_state::sstrike(machine_config &config)
{
	slikshot_lo(config);
	MCFG_MACHINE_START_OVERRIDE(itech8_state,sstrike)
}

void itech8_state::hstennis_hi(machine_config &config)
{
	itech8_core_hi(config);
	itech8_sound_ym3812(config);

	m_screen->set_visarea(0, 399, 0, 239);
	m_screen->set_screen_update(FUNC(itech8_state::screen_update_2page_large));
}

void itech8_state::hstennis_lo(machine_config &config)
{
	itech8_core_lo(config);
	itech8_sound_ym3812(config);

	m_screen->set_visarea(0, 399, 0, 239);
	m_screen->set_screen_update(FUNC(itech8_state::screen_update_2page_large));
}

void itech8_state::rimrockn(machine_config &config)
{
	itech8_core_devices(config);
	itech8_sound_ym3812_external(config);

	HD6309(config, m_maincpu, CLOCK_12MHz);
	m_maincpu->set_addrmap(AS_PROGRAM, &itech8_state::rimrockn_map);

	m_screen->set_visarea(24, 375, 0, 239);
	m_screen->set_screen_update(FUNC(itech8_state::screen_update_2page_large));
}

void itech8_state::ninclown(machine_config &config)
{
	itech8_core_devices(config);
	itech8_sound_ym3812_external(config);

	//  m_nvram->set_custom_handler([this](nvram_device &, void *p, size_t s) { memcpy(p, memregion("maincpu")->base(), s); }, "vectors");

	M68000(config, m_maincpu, CLOCK_12MHz);
	m_maincpu->set_addrmap(AS_PROGRAM, &itech8_state::ninclown_map);

	m_screen->set_visarea(64, 423, 0, 239);
	m_screen->set_screen_update(FUNC(itech8_state::screen_update_2page_large));
	m_screen->screen_vblank().set(FUNC(itech8_state::ninclown_irq));
}

void itech8_state::gtg2(machine_config &config)
{
	itech8_core_lo(config);
	itech8_sound_ym3812_external(config);
	m_maincpu->set_addrmap(AS_PROGRAM, &itech8_state::gtg2_map);

	m_screen->set_visarea(0, 255, 0, 239);
	m_screen->set_screen_update(FUNC(itech8_state::screen_update_2layer));
}



/*************************************
 *
 *  ROM definitions
 *
 *************************************/

ROM_START( wfortune )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "wofpgm.u5", 0x00000, 0x10000, CRC(bd984654) SHA1(8e16d2feb26e9a6f86c4a36bf0f03db80ded03f6) )

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "wof_snd-wof.u27", 0x08000, 0x8000, CRC(0a6aa5dc) SHA1(42eef40a4300d6d16d9e2af678432a02be05f104) )

	ROM_REGION( 0xc0000, "grom", 0 )
	ROM_LOAD( "grom0-wof.grom0", 0x00000, 0x10000, CRC(9a157b2c) SHA1(c349b41ba00cf6e2fec32872627c8cfdd8b5c1b9) )
	ROM_LOAD( "grom1-wof.grom1", 0x10000, 0x10000, CRC(5064739b) SHA1(424e3f94333f8ca21ac39b64b684cf6b487164d3) )
	ROM_LOAD( "grom2-wof.grom2", 0x20000, 0x10000, CRC(3d393b2b) SHA1(2c94d2dab7369c099c470cf96391b033f39add78) )
	ROM_LOAD( "grom3-wof.grom3", 0x30000, 0x10000, CRC(117a2ce9) SHA1(8d601c1cf9f783a42617f13c6862a5835553ac4f) )

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "wof_vr-sbom0.srom0", 0x00000, 0x20000, CRC(5c28c3fe) SHA1(eba64ede749fb26f9926f644d66860b54b4c76e7) ) /* Labeled as WOF VR-SBOM0  (yes it's actually "SBOM0") */
ROM_END


ROM_START( wfortunea )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "wof-pgm_r1.u5", 0x00000, 0x10000, CRC(c3d3eb21) SHA1(21137663afd19fba875e188640f0347fc8c5dcf0) ) /* Labeled as WOF-PGM R1 */

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "wof_snd-wof.u27", 0x08000, 0x8000, CRC(0a6aa5dc) SHA1(42eef40a4300d6d16d9e2af678432a02be05f104) )

	ROM_REGION( 0xc0000, "grom", 0 )
	ROM_LOAD( "grom0-wof.grom0", 0x00000, 0x10000, CRC(9a157b2c) SHA1(c349b41ba00cf6e2fec32872627c8cfdd8b5c1b9) )
	ROM_LOAD( "grom1-wof.grom1", 0x10000, 0x10000, CRC(5064739b) SHA1(424e3f94333f8ca21ac39b64b684cf6b487164d3) )
	ROM_LOAD( "grom2-wof.grom2", 0x20000, 0x10000, CRC(3d393b2b) SHA1(2c94d2dab7369c099c470cf96391b033f39add78) )
	ROM_LOAD( "grom3-wof.grom3", 0x30000, 0x10000, CRC(117a2ce9) SHA1(8d601c1cf9f783a42617f13c6862a5835553ac4f) )

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "wof_vr-sbom0.srom0", 0x00000, 0x20000, CRC(5c28c3fe) SHA1(eba64ede749fb26f9926f644d66860b54b4c76e7) ) /* Labeled as WOF VR-SBOM0  (yes it's actually "SBOM0") */
ROM_END


ROM_START( grmatch )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "grudgematch.u5", 0x00000, 0x10000, CRC(11cadec9) SHA1(e21df623d1311ea63bafa2d6d0d94eb7d13232da) )

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "grudgematch.u27", 0x08000, 0x8000, CRC(59c18e63) SHA1(0d00c9cc683ff17e3213ba343ae65d533b57a243) )

	ROM_REGION( 0xe0000, "grom", 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, CRC(9064eff9) SHA1(4c80432ef359266c43b4691b503f529ed018a655) )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, CRC(6919c314) SHA1(e4a7ca77ee27ea8e41a236806e689c964d981210) )
	ROM_LOAD( "grom2.bin", 0x40000, 0x20000, CRC(36b1682c) SHA1(6098b8aa576522c3cea66b5ac8f9406bf4222008) )
	ROM_LOAD( "grom3.bin", 0x60000, 0x20000, CRC(7bf05f61) SHA1(269e1cacf319d9f2a2b865ac6573e690f36827a1) )
	ROM_LOAD( "grom4.bin", 0x80000, 0x20000, CRC(55bded89) SHA1(61e0c60e391327a5fdda704c7756fcd6e11253c5) )
	ROM_LOAD( "grom5.bin", 0xa0000, 0x20000, CRC(37b47b2e) SHA1(352204d3e95e6db556aacf053c42d0d5871245a7) )
	ROM_LOAD( "grom6.bin", 0xc0000, 0x20000, CRC(860ee822) SHA1(2ca821c2fa220065b99b99b7487fe9666f338c75) )

	ROM_REGION( 0x20000, "ymsnd", 0 )
	ROM_LOAD( "srom0.bin", 0x00000, 0x20000, CRC(49bce954) SHA1(68a8b11c03722349d673f7383288c63054f0d6f6) )
ROM_END


ROM_START( stratab )
	ROM_REGION( 0x8000, "maincpu", 0 )
	ROM_LOAD( "sb_prog_v3_u5.u5", 0x00000, 0x8000, CRC(a5ae728f) SHA1(85098eef1614d5148e8082df4c936883662292ee) ) /* Labeled as SB PROG V3 (U5) */

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "sb_snds_u27.u27", 0x08000, 0x8000, CRC(b36c8f0a) SHA1(c4c3edf3352d95561f76705087338c1946137447) ) /* Labeled as SB SNDS (U27) */

	ROM_REGION( 0xc0000, "grom", 0 )
	ROM_LOAD( "sb_grom0_0.grom0", 0x00000, 0x20000, CRC(a915b0bd) SHA1(0955c7ebb48e97ccffc18c5deec6eccce1d68de8) )
	ROM_LOAD( "sb_grom0_1.grom1", 0x20000, 0x20000, CRC(340c661f) SHA1(150f9158fa6d956d08051c67c17723b1d8c66867) )
	ROM_LOAD( "sb_grom0_2.grom2", 0x40000, 0x20000, CRC(5df9f1cf) SHA1(cad87f63ac0e902dffeeaa42538fc73f792d87d9) )

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "sb_srom0.srom0", 0x00000, 0x20000, CRC(6ff390b9) SHA1(f31dae9e31f3fc83b9253e49fd4204820db3587e) )
ROM_END


ROM_START( stratab1 )
	ROM_REGION( 0x8000, "maincpu", 0 )
	ROM_LOAD( "sb_prog_v1_u5.u5",  0x00000, 0x8000, CRC(46d51604) SHA1(de7b6306fdcee4907b07667baf874bd195822e6a) ) /* Labeled as SB PROG V1 (U5) */

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "sb_snds_u27.u27", 0x08000, 0x8000, CRC(b36c8f0a) SHA1(c4c3edf3352d95561f76705087338c1946137447) ) /* Labeled as SB SNDS (U27) */

	ROM_REGION( 0xc0000, "grom", 0 )
	ROM_LOAD( "sb_grom0_0.grom0", 0x00000, 0x20000, CRC(a915b0bd) SHA1(0955c7ebb48e97ccffc18c5deec6eccce1d68de8) )
	ROM_LOAD( "sb_grom0_1.grom1", 0x20000, 0x20000, CRC(340c661f) SHA1(150f9158fa6d956d08051c67c17723b1d8c66867) )
	ROM_LOAD( "sb_grom0_2.grom2", 0x40000, 0x20000, CRC(5df9f1cf) SHA1(cad87f63ac0e902dffeeaa42538fc73f792d87d9) )

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "sb_srom0.srom0", 0x00000, 0x20000, CRC(6ff390b9) SHA1(f31dae9e31f3fc83b9253e49fd4204820db3587e) )
ROM_END


/*
Fun fact:
 With the earlier versions of Golden Tee Golf, GROM5 changes with every set but the label doesn't change or give any indication
   the data would be different between sets.
 The sound CPU code also changes between most sets and finally stabilized with v1.1 with a revised SROM0. However earlier changes
   weren't documented and the CPU code ROM was simply labeled GOLF SND (U27) with the sample ROM always labeled GOLF SROM0
*/
ROM_START( gtg )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "gtg_joy_v3.3_u5.u5", 0x00000, 0x10000, CRC(983a5c0c) SHA1(245fd6b86e96ef57ea9a85c7a501d846e135cfc6) ) /* Joystick version - Labeled GTG JOY V3.3 (U5) */

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "golf_snd-v1.1_u27.u27", 0x08000, 0x8000, CRC(358d2440) SHA1(7b09350c89f9d2c86dc187d8812bbf26b576a38f) ) /* Labeled GOLF SND-V1.1 (U27) */

	ROM_REGION( 0xc0000, "grom", 0 )
	ROM_LOAD( "golf_grom00.grom0", 0x00000, 0x20000, CRC(a29c688a) SHA1(32dbb996a5e4c23cfd44b79312ac4a767658f20a) )
	ROM_LOAD( "golf_grom01.grom1", 0x20000, 0x20000, CRC(b52a23f6) SHA1(092961acf47875179b44342e2dd8955670e67ea2) )
	ROM_LOAD( "golf_grom02.grom2", 0x40000, 0x20000, CRC(9b8e3a61) SHA1(1b5682b1328d6c97b604fb71512e8f72322a688f) )
	ROM_LOAD( "golf_grom03.grom3", 0x60000, 0x20000, CRC(b6e9fb15) SHA1(c1b28ea911696cb4ed56bfba212848693530b59f) )
	ROM_LOAD( "golf_grom04.grom4", 0x80000, 0x20000, CRC(faa16729) SHA1(5d46cddda66b6d23c9ebdf2fb4cebce15586b4ad) )
	ROM_LOAD( "golf_grom05.grom5", 0xa0000, 0x20000, CRC(c108c56c) SHA1(5c67b6479e093c34e7ee2b68d93eba07a96f72e4) )

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "golf_srom0.srom0", 0x00000, 0x20000, CRC(1cccbfdf) SHA1(546059fea2e7cd5627a666d80b1fc3ed8fcc0762) )
ROM_END


ROM_START( gtgj31 )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "gtg_joy_v3.1_u5.u5", 0x00000, 0x10000, CRC(61984272) SHA1(be735f8576fb2cccc0e9e6ea6f2fd54b6c0b3bb3) ) /* Joystick version - Labeled GTG JOY V3.1 (U5) */

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "golf_snd-v1.1_u27.u27", 0x08000, 0x8000, CRC(358d2440) SHA1(7b09350c89f9d2c86dc187d8812bbf26b576a38f) ) /* Labeled GOLF SND-V1.1 (U27) */

	ROM_REGION( 0xc0000, "grom", 0 )
	ROM_LOAD( "golf_grom00.grom0", 0x00000, 0x20000, CRC(a29c688a) SHA1(32dbb996a5e4c23cfd44b79312ac4a767658f20a) )
	ROM_LOAD( "golf_grom01.grom1", 0x20000, 0x20000, CRC(b52a23f6) SHA1(092961acf47875179b44342e2dd8955670e67ea2) )
	ROM_LOAD( "golf_grom02.grom2", 0x40000, 0x20000, CRC(9b8e3a61) SHA1(1b5682b1328d6c97b604fb71512e8f72322a688f) )
	ROM_LOAD( "golf_grom03.grom3", 0x60000, 0x20000, CRC(b6e9fb15) SHA1(c1b28ea911696cb4ed56bfba212848693530b59f) )
	ROM_LOAD( "golf_grom04.grom4", 0x80000, 0x20000, CRC(faa16729) SHA1(5d46cddda66b6d23c9ebdf2fb4cebce15586b4ad) )
	ROM_LOAD( "golf_grom05.grom5", 0xa0000, 0x20000, CRC(5b393314) SHA1(9e314a75ea52373369904915ec786f09eee725a9) ) /* sldh */

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "golf_srom0.srom0", 0x00000, 0x20000, CRC(1cccbfdf) SHA1(546059fea2e7cd5627a666d80b1fc3ed8fcc0762) )
ROM_END


ROM_START( gtgt21 )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "gtg_bim_2.1.u5", 0x0000, 0x10000, CRC(25f626d9) SHA1(19d3e01f388ac4d430cda6e46b4dda8cfceaf325) ) /* Trackball version */

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "golf_snd_u27.u27", 0x08000, 0x8000, CRC(ac6d3f32) SHA1(82b98df915cd794037ad863558e777f3ad10145b) ) /* Labeled GOLF SND (U27) - slds */

	ROM_REGION( 0xc0000, "grom", 0 )
	ROM_LOAD( "golf_grom00.grom0", 0x00000, 0x20000, CRC(a29c688a) SHA1(32dbb996a5e4c23cfd44b79312ac4a767658f20a) )
	ROM_LOAD( "golf_grom01.grom1", 0x20000, 0x20000, CRC(b52a23f6) SHA1(092961acf47875179b44342e2dd8955670e67ea2) )
	ROM_LOAD( "golf_grom02.grom2", 0x40000, 0x20000, CRC(9b8e3a61) SHA1(1b5682b1328d6c97b604fb71512e8f72322a688f) )
	ROM_LOAD( "golf_grom03.grom3", 0x60000, 0x20000, CRC(b6e9fb15) SHA1(c1b28ea911696cb4ed56bfba212848693530b59f) )
	ROM_LOAD( "golf_grom04.grom4", 0x80000, 0x20000, CRC(faa16729) SHA1(5d46cddda66b6d23c9ebdf2fb4cebce15586b4ad) )
	ROM_LOAD( "golf_grom05.grom5", 0xa0000, 0x10000, CRC(dab92dfb) SHA1(f99f553d48fb0600d971959efb4d0410d7606131) ) /* sldh */

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "golf_srom0.srom0", 0x00000, 0x20000, CRC(d041e0c9) SHA1(1d90f37071d92c714ff69ab2b0337c7c66147995) ) /* sldh */
ROM_END


ROM_START( gtgt20 )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "gtg_bim_2.0.u5", 0x0000, 0x10000, CRC(4c907166) SHA1(338a599645fa49c9fcbfbe5ba3431dafffddacc7) ) /* Trackball version */

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "golf_snd_u27.u27", 0x08000, 0x8000, CRC(f6a7429b) SHA1(0fb378606c12c3543aa1ff603101e262acb9c692) ) /* Labeled GOLF SND (U27) - slds */

	ROM_REGION( 0xc0000, "grom", 0 )
	ROM_LOAD( "golf_grom00.grom0", 0x00000, 0x20000, CRC(a29c688a) SHA1(32dbb996a5e4c23cfd44b79312ac4a767658f20a) )
	ROM_LOAD( "golf_grom01.grom1", 0x20000, 0x20000, CRC(b52a23f6) SHA1(092961acf47875179b44342e2dd8955670e67ea2) )
	ROM_LOAD( "golf_grom02.grom2", 0x40000, 0x20000, CRC(9b8e3a61) SHA1(1b5682b1328d6c97b604fb71512e8f72322a688f) )
	ROM_LOAD( "golf_grom03.grom3", 0x60000, 0x20000, CRC(b6e9fb15) SHA1(c1b28ea911696cb4ed56bfba212848693530b59f) )
	ROM_LOAD( "golf_grom04.grom4", 0x80000, 0x20000, CRC(faa16729) SHA1(5d46cddda66b6d23c9ebdf2fb4cebce15586b4ad) )
	ROM_LOAD( "golf_grom05.grom5", 0xa0000, 0x10000, CRC(62a523d2) SHA1(431f89fa044bf0ed3c197745d9c1a61f666da658) ) /* sldh */

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "golf_srom0.srom0", 0x00000, 0x20000, CRC(d041e0c9) SHA1(1d90f37071d92c714ff69ab2b0337c7c66147995) ) /* sldh */
ROM_END


ROM_START( gtgt10 )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "gtg_bim_1.0.u5", 0x00000, 0x10000, CRC(ec70b510) SHA1(318984d77eb1df6258b855781ae1c9a09aa74f15) ) /* Trackball version */

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "golf_snd_u27.u27", 0x08000, 0x8000, CRC(471da557) SHA1(32bfe450a42d9eb6c14edcfa2b4e33f65a11126e) ) /* Labeled GOLF SND (U27) - slds */

	ROM_REGION( 0xb0000, "grom", 0 )
	ROM_LOAD( "golf_grom00.grom0", 0x00000, 0x20000, CRC(a29c688a) SHA1(32dbb996a5e4c23cfd44b79312ac4a767658f20a) )
	ROM_LOAD( "golf_grom01.grom1", 0x20000, 0x20000, CRC(b52a23f6) SHA1(092961acf47875179b44342e2dd8955670e67ea2) )
	ROM_LOAD( "golf_grom02.grom2", 0x40000, 0x20000, CRC(9b8e3a61) SHA1(1b5682b1328d6c97b604fb71512e8f72322a688f) )
	ROM_LOAD( "golf_grom03.grom3", 0x60000, 0x20000, CRC(b6e9fb15) SHA1(c1b28ea911696cb4ed56bfba212848693530b59f) )
	ROM_LOAD( "golf_grom04.grom4", 0x80000, 0x20000, CRC(faa16729) SHA1(5d46cddda66b6d23c9ebdf2fb4cebce15586b4ad) )
	ROM_LOAD( "golf_grom05.grom5", 0xa0000, 0x10000, CRC(44b47015) SHA1(5dde4c932a697b51fe02eab8d948889b3fe7baff) ) /* sldh */

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "golf_srom0.srom0", 0x00000, 0x20000, CRC(d041e0c9) SHA1(1d90f37071d92c714ff69ab2b0337c7c66147995) ) /* sldh */
ROM_END


ROM_START( gtg2t )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "gtgii_tb_v1.1.u5", 0x00000, 0x10000, CRC(c7b3a9f3) SHA1(5edaca6fd6ee58bd1676dc9b2c86da4dd2f51687) ) /* Trackball version - labeled GTGII TB V1.1 (U5) */

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "gtgii_snd_v1_u27.u27", 0x08000, 0x8000, CRC(dd2a5905) SHA1(dc93f13de3953852a6757361eb9683a57d3ed326) ) /* labeled GTGII SND V1 (U27) */

	ROM_REGION( 0xc0000, "grom", 0 )
	ROM_LOAD( "gtgii-grom0.grom0", 0x00000, 0x20000, CRC(a29c688a) SHA1(32dbb996a5e4c23cfd44b79312ac4a767658f20a) )
	ROM_LOAD( "gtgii-grom1.grom1", 0x20000, 0x20000, CRC(a4182776) SHA1(9f4704d1a61a4fffce454c82cb3eb4629d9a2006) )
	ROM_LOAD( "gtgii-grom2.grom2", 0x40000, 0x20000, CRC(0580bb99) SHA1(012d473bc63632a5ed1a250daa54a00a4e30e8b2) )
	ROM_LOAD( "gtgii-grom3.grom3", 0x60000, 0x20000, CRC(89edb624) SHA1(e77d5ecd04f44f184e76eb2506660a04b6bc7fe8) )
	ROM_LOAD( "gtgii-grom4.grom4", 0x80000, 0x20000, CRC(f6557950) SHA1(fd2bbd6b852a9f2775a23d9fdaa20d31d6766d7f) )
	ROM_LOAD( "gtgii-grom5.grom5", 0xa0000, 0x20000, CRC(a680ce6a) SHA1(fe51546933c093cea81858b213afb0e926e9eb4e) )

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "gtgii_vr_srom0.srom0", 0x00000, 0x20000, CRC(4dd4db42) SHA1(0dffb51e8de36d8747f443fd65fe9927815eaff0) ) /* labeled GTGII VR-SROM0 */

	ROM_REGION( 0x0200, "plds", 0 )
	ROM_LOAD( "tibpal16l8.u11", 0x0000, 0x0104, CRC(9bf5a75f) SHA1(79786f7ce656f30a33a92887a290b767a7cbbf31) )
ROM_END


ROM_START( gtg2j )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "gtg2.bim_1.0.u5", 0x00000, 0x10000, CRC(9c95ceaa) SHA1(d9fd2b2419c026822a07d2ba51d6ab40b7cd0d49) ) /* Joystick version */

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "gtgii_snd_v1_u27.u27", 0x08000, 0x8000, CRC(dd2a5905) SHA1(dc93f13de3953852a6757361eb9683a57d3ed326) ) /* labeled GTGII SND V1 (U27) */

	ROM_REGION( 0xc0000, "grom", 0 )
	ROM_LOAD( "gtgii-grom0.grom0", 0x00000, 0x20000, CRC(a29c688a) SHA1(32dbb996a5e4c23cfd44b79312ac4a767658f20a) )
	ROM_LOAD( "gtgii-grom1.grom1", 0x20000, 0x20000, CRC(a4182776) SHA1(9f4704d1a61a4fffce454c82cb3eb4629d9a2006) )
	ROM_LOAD( "gtgii-grom2.grom2", 0x40000, 0x20000, CRC(0580bb99) SHA1(012d473bc63632a5ed1a250daa54a00a4e30e8b2) )
	ROM_LOAD( "gtgii-grom3.grom3", 0x60000, 0x20000, CRC(89edb624) SHA1(e77d5ecd04f44f184e76eb2506660a04b6bc7fe8) )
	ROM_LOAD( "gtgii-grom4.grom4", 0x80000, 0x20000, CRC(f6557950) SHA1(fd2bbd6b852a9f2775a23d9fdaa20d31d6766d7f) )
	ROM_LOAD( "gtgii-grom5.grom5", 0xa0000, 0x20000, CRC(a680ce6a) SHA1(fe51546933c093cea81858b213afb0e926e9eb4e) )

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "srom0.bin", 0x00000, 0x20000, CRC(1cccbfdf) SHA1(546059fea2e7cd5627a666d80b1fc3ed8fcc0762) )

	ROM_REGION( 0x0200, "plds", 0 )
	ROM_LOAD( "tibpal16l8.u11", 0x0000, 0x0104, CRC(9bf5a75f) SHA1(79786f7ce656f30a33a92887a290b767a7cbbf31) )
ROM_END


ROM_START( slikshot )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "poolpgm-20.u5",  0x00000, 0x10000, CRC(370a00eb) SHA1(b2878f161f4931d9fc3979a84b29660941e2608f) )

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "pool_snd-u27.u27", 0x08000, 0x8000, CRC(a96ce0f7) SHA1(c1fec3aeef97c846fd1a20b91af54f6bf9723a71) )

	ROM_REGION( 0x10000, "sub", 0 )
	ROM_LOAD( "z80_pgm_u53.u53", 0x00000, 0x0800, CRC(04b85918) SHA1(409aef2e71937c7654334999df9313909d757966) ) /* labeled Z80 PGM (U53) */
	ROM_CONTINUE(                0x00000, 0x0800 )
	ROM_CONTINUE(                0x00000, 0x0800 )
	ROM_CONTINUE(                0x00000, 0x0800 )

	ROM_REGION( 0xc0000, "grom", 0 )
	ROM_LOAD( "pool-grom0.grom0", 0x00000, 0x20000, CRC(e60c2804) SHA1(e62d11b6c4439a70a2f32df72c8c64e2f110351e) )
	ROM_LOAD( "pool-grom1.grom1", 0x20000, 0x20000, CRC(d764d542) SHA1(43fc0c9b627484a670d87da91e212741b137e995) )

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "pool_vr-srom0.srom0", 0x00000, 0x10000, CRC(4b075f5e) SHA1(d1ac2c06352a5b96486a7e8cf8baae0c0e5b1883) )
ROM_END


ROM_START( slikshot17 )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "poolpgm-17.u5", 0x00000, 0x10000, CRC(09d70554) SHA1(a009cd3b22261c60f1028694baef51f61713154f) )

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "pool_snd-u27.u27", 0x08000, 0x8000, CRC(a96ce0f7) SHA1(c1fec3aeef97c846fd1a20b91af54f6bf9723a71) )

	ROM_REGION( 0x10000, "sub", 0 )
	ROM_LOAD( "z80_pgm_u53.u53", 0x00000, 0x0800, CRC(04b85918) SHA1(409aef2e71937c7654334999df9313909d757966) ) /* labeled Z80 PGM (U53) */
	ROM_CONTINUE(                0x00000, 0x0800 )
	ROM_CONTINUE(                0x00000, 0x0800 )
	ROM_CONTINUE(                0x00000, 0x0800 )

	ROM_REGION( 0xc0000, "grom", 0 )
	ROM_LOAD( "pool-grom0.grom0", 0x00000, 0x20000, CRC(e60c2804) SHA1(e62d11b6c4439a70a2f32df72c8c64e2f110351e) )
	ROM_LOAD( "pool-grom1.grom1", 0x20000, 0x20000, CRC(d764d542) SHA1(43fc0c9b627484a670d87da91e212741b137e995) )

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "pool_vr-srom0.srom0", 0x00000, 0x10000, CRC(4b075f5e) SHA1(d1ac2c06352a5b96486a7e8cf8baae0c0e5b1883) )
ROM_END


ROM_START( slikshot16 )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "poolpgm-16.u5", 0x00000, 0x10000, CRC(c0f17012) SHA1(5d466e058daf91b4f52e634498df9d2a03627aaa) )

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "pool_snd-u27.u27", 0x08000, 0x8000, CRC(a96ce0f7) SHA1(c1fec3aeef97c846fd1a20b91af54f6bf9723a71) )

	ROM_REGION( 0x10000, "sub", 0 )
	ROM_LOAD( "z80_pgm_u53.u53", 0x00000, 0x0800, CRC(04b85918) SHA1(409aef2e71937c7654334999df9313909d757966) ) /* labeled Z80 PGM (U53) */
	ROM_CONTINUE(                0x00000, 0x0800 )
	ROM_CONTINUE(                0x00000, 0x0800 )
	ROM_CONTINUE(                0x00000, 0x0800 )

	ROM_REGION( 0xc0000, "grom", 0 )
	ROM_LOAD( "grom0-pool.grom0", 0x00000, 0x10000, CRC(e6d0edc6) SHA1(5287a31bbdde1e4291d8e9e6b99d3aa12bfb6e18) )
	ROM_LOAD( "grom1-pool.grom1", 0x10000, 0x10000, CRC(5a071aa2) SHA1(9c5506e37625d213429b1231d457d7ce8a7a81ff) )
	ROM_LOAD( "grom2-pool.grom2", 0x20000, 0x10000, CRC(c0bdf4e0) SHA1(3b7c635375c5e5fddcbc1bd1b186c960081ec37e) )
	ROM_LOAD( "grom3-pool.grom3", 0x30000, 0x10000, CRC(cb0bd9a3) SHA1(107ff127f9adad84a5f92077851423249fce8e30) )

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "pool_vr-srom0.srom0", 0x00000, 0x10000, CRC(4b075f5e) SHA1(d1ac2c06352a5b96486a7e8cf8baae0c0e5b1883) )
ROM_END


ROM_START( dynobop ) /* known to be labeled as DYNO PGM 1.0 U5, but same data - a "true" v1.0 most likely labeled as DYNO BIM U5 */
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "dyno_pgm_1.1_u5.u5", 0x00000, 0x10000, CRC(98452c40) SHA1(9b9316fc258792e0d825f16e0fadf8e0c35a864e) )

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "dyno_snd-u27.u27", 0x08000, 0x8000, CRC(a37d862b) SHA1(922eeae184df2c5c28040da27699dd55744f8dca) )

	ROM_REGION( 0x10000, "sub", 0 )
	ROM_LOAD( "z80_pgm_u53.u53", 0x00000, 0x0800, CRC(04b85918) SHA1(409aef2e71937c7654334999df9313909d757966) ) /* labeled Z80 PGM (U53) */
	ROM_CONTINUE(                0x00000, 0x0800 )
	ROM_CONTINUE(                0x00000, 0x0800 )
	ROM_CONTINUE(                0x00000, 0x0800 )

	ROM_REGION( 0xc0000, "grom", 0 )
	ROM_LOAD( "grom00-dyno.grom0", 0x00000, 0x20000, CRC(3525a7a3) SHA1(fe0b08203c135d55507506936dc34e1503e4906b) )
	ROM_LOAD( "grom01-dyno.grom1", 0x20000, 0x20000, CRC(1544a232) SHA1(60bba76537c82887db8e38c6a87c528afdd385d0) )

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "dyno_vr-srom0.srom0", 0x00000, 0x10000, CRC(b355bf1d) SHA1(c9745638feb33ec181b8d9a894421f9854286cd9) )
ROM_END


ROM_START( sstrike )
	ROM_REGION( 0x8000, "maincpu", 0 )
	ROM_LOAD( "sstrik_prg-v1_u5.u5", 0x00000, 0x8000, CRC(af00cddf) SHA1(b866e8dfce1449f7462a79efa385ea6b55cdc6e7) ) /* labeled SSTRIKE PRG-V1(U5) */

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "sstrik_snd1.4_u27.u27", 0x08000, 0x8000, CRC(efab7252) SHA1(eb3b2002531e551e3d67958ea3cc56a69fa660e2) ) /* labeled SSTRIKE SND1.4(U27) */

	ROM_REGION( 0x10000, "sub", 0 )
	ROM_LOAD( "spstku53.u53", 0x00000, 0x0800, CRC(04b85918) SHA1(409aef2e71937c7654334999df9313909d757966) )
	ROM_CONTINUE(        0x00000, 0x0800 )
	ROM_CONTINUE(        0x00000, 0x0800 )
	ROM_CONTINUE(        0x00000, 0x0800 )

	ROM_REGION( 0xc0000, "grom", 0 )
	ROM_LOAD( "super_strike_grom0.grom0", 0x00000, 0x20000, CRC(9cfb9849) SHA1(5aa860c0c6e3916ebdb8898ee44f633bf3347ca8) )
	ROM_LOAD( "super_strike_grom1.grom1", 0x20000, 0x20000, CRC(d9ea14e1) SHA1(4cddf3237c203b0a3f7ae770f85f1be35e9e1b78) )
	ROM_LOAD( "super_strike_grom2.grom2", 0x40000, 0x20000, CRC(dcd97bf7) SHA1(95361222ac58bf74539f2a7e80574bcd848c615e) )

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "super_strike_srom0.srom0", 0x00000, 0x20000, CRC(6ff390b9) SHA1(f31dae9e31f3fc83b9253e49fd4204820db3587e) )
ROM_END


ROM_START( stratabs )
	ROM_REGION( 0x8000, "maincpu", 0 )
	ROM_LOAD( "sb_prog-v4t.u5",  0x00000, 0x8000, CRC(38ddae75) SHA1(71a9cbd36cf7b180a88bab3ab92a4dff93ce365f) )

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "sb_snds_1.4.u27", 0x08000, 0x8000, CRC(526ef093) SHA1(884f9149b3d5eb33e47258e466ad9cd9ce5ffddb) )

	ROM_REGION( 0x10000, "sub", 0 )
	ROM_LOAD( "z80_prg_v1.0t.u53", 0x00000, 0x0800, CRC(08203f9f) SHA1(be5044938c73bf6fc1a6130dc473997e8c219b24) )
	ROM_CONTINUE(        0x00000, 0x0800 )
	ROM_CONTINUE(        0x00000, 0x0800 )
	ROM_CONTINUE(        0x00000, 0x0800 )

	ROM_REGION( 0xc0000, "grom", 0 )
	ROM_LOAD( "sb_grom00", 0x00000, 0x20000, CRC(22f6ce56) SHA1(8e423122384257d9c5d8f48192ff6fa1f544fd97) )
	ROM_LOAD( "sb_grom01", 0x20000, 0x20000, CRC(6cc7ad6f) SHA1(d601b9bb81f26ad86e5cc053cab055831331ccde) )
	ROM_LOAD( "sb_grom02", 0x40000, 0x20000, CRC(475134ef) SHA1(5920e7a211f1b2234e8a3f51e570303c3787d8fd) )

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "sb_srom0", 0x00000, 0x20000, CRC(6ff390b9) SHA1(f31dae9e31f3fc83b9253e49fd4204820db3587e) )
ROM_END


ROM_START( pokrdice )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "pd-v1.7_u5.u5", 0x00000, 0x10000, CRC(5e24be82) SHA1(97e50cc023ff651fb09cc5e85a1bef1bc234ccb9) )

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "pd-snd.u27", 0x08000, 0x8000, CRC(4925401c) SHA1(e35983bec4a0dd4cb1d942fd909790b1adeb415d) )

	ROM_REGION( 0xc0000, "grom", 0 )
	ROM_LOAD( "pd-grom0.grom0", 0x00000, 0x20000, CRC(7c2573e7) SHA1(d6a2a16277ad854c66927d88c5617d05eefe1057) )
	ROM_LOAD( "pd-grom1.grom1", 0x20000, 0x20000, CRC(e7c06aeb) SHA1(4be54b078d886359bf6ed376019cc1f6f04f52d6) )

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "pd-srom0.srom0", 0x00000, 0x10000, CRC(f85dbd6f) SHA1(fce53019432e4a84e52f0ae1996b2e0d94b32bc0) )
ROM_END


ROM_START( hstennis ) /* PCB  p/n 1030 */
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "ten.bim_v1.1_u5.u5", 0x00000, 0x10000, CRC(faffab5c) SHA1(4de525f6adb16205c47788b78aecdebd57008295) ) /* labeled as TEN.BIM V1.1 (U5) */

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "tensnd.bim_v1_u27.u27", 0x08000, 0x8000, CRC(f034a694) SHA1(3540e2edff2ce47504260ec856bab9b638d9260d) ) /* labeled as TENSND.BIM V1 U27 */

	ROM_REGION( 0xc0000, "grom", 0 )
	ROM_LOAD( "t_grom0.bim.grom0", 0x00000, 0x20000, CRC(1e69ebae) SHA1(a32e2c2f4e4a527ae6b57adeccd2c4d2045ab5fe) ) /* labeled as T GROM0.BIM, ect... */
	ROM_LOAD( "t_grom1.bim.grom1", 0x20000, 0x20000, CRC(4e6a22d5) SHA1(3c2d51dd874f61c0a557ea2c1968afa02d9bfc42) )
	ROM_LOAD( "t_grom2.bim.grom2", 0x40000, 0x20000, CRC(c0b643a9) SHA1(d240f703a55c39ce4a969612fbb9cd76e4b849ac) )
	ROM_LOAD( "t_grom3.bim.grom3", 0x60000, 0x20000, CRC(54afb456) SHA1(0a7f10b1a490825a85489ac3f678eb1e11a0ac65) )
	ROM_LOAD( "t_grom4.bim.grom4", 0x80000, 0x20000, CRC(ee09d645) SHA1(8233cfbd17d9de548ae3ca5e30928061712e0781) )

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "tennis.vr_srom0.srom0", 0x00000, 0x20000, CRC(d9ce58c3) SHA1(92574e60497d86b8608fba6278ccfc9036cb7f22) ) /* labeled as TENNIS.VR SROM0 */

	ROM_REGION( 0x0200, "plds", 0 )
	ROM_LOAD( "pal16l8-itvs.u11", 0x0000, 0x0104, CRC(fee03727) SHA1(e784ff18505cdccc1020dbe5cb0e7cc9efc068a4) )
ROM_END


ROM_START( hstennis10 ) /* PCB  p/n 1030 */
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "ten.bim_v1.0_u5.u5", 0x00000, 0x10000, CRC(d108a6e0) SHA1(1041e1d95b10245fc50f6484e710803db2706f9a) ) /* labeled as TEN.BIM V1.0 (U5) */

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "tensnd.bim_v1_u27.u27", 0x08000, 0x8000, CRC(f034a694) SHA1(3540e2edff2ce47504260ec856bab9b638d9260d) ) /* labeled as TENSND.BIM V1 U27 */

	ROM_REGION( 0xc0000, "grom", 0 )
	ROM_LOAD( "t_grom0.bim.grom0", 0x00000, 0x20000, CRC(1e69ebae) SHA1(a32e2c2f4e4a527ae6b57adeccd2c4d2045ab5fe) ) /* labeled as T GROM0.BIM, ect... */
	ROM_LOAD( "t_grom1.bim.grom1", 0x20000, 0x20000, CRC(4e6a22d5) SHA1(3c2d51dd874f61c0a557ea2c1968afa02d9bfc42) )
	ROM_LOAD( "t_grom2.bim.grom2", 0x40000, 0x20000, CRC(c0b643a9) SHA1(d240f703a55c39ce4a969612fbb9cd76e4b849ac) )
	ROM_LOAD( "t_grom3.bim.grom3", 0x60000, 0x20000, CRC(54afb456) SHA1(0a7f10b1a490825a85489ac3f678eb1e11a0ac65) )
	ROM_LOAD( "t_grom4.bim.grom4", 0x80000, 0x20000, CRC(ee09d645) SHA1(8233cfbd17d9de548ae3ca5e30928061712e0781) )

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "tennis.vr_srom0.srom0", 0x00000, 0x20000, CRC(d9ce58c3) SHA1(92574e60497d86b8608fba6278ccfc9036cb7f22) ) /* labeled as TENNIS.VR SROM0 */

	ROM_REGION( 0x0200, "plds", 0 )
	ROM_LOAD( "pal16l8-itvs.u11", 0x0000, 0x0104, CRC(fee03727) SHA1(e784ff18505cdccc1020dbe5cb0e7cc9efc068a4) )
ROM_END


ROM_START( arlingtn ) /* PCB  p/n 1030 rev. 1A */
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "ahr-d_v_1.40_u5.u5", 0x00000, 0x10000, CRC(02338ddd) SHA1(8e8e0c319c0b7533511089aa7a671a112169a4a1) ) /* labeled AHR-D V 1.40 (U5) - service menu reports version as 1.40-D */

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "ahr_snd_v1.1_u27.u27", 0x08000, 0x8000, CRC(dec57dca) SHA1(21a8ead10b0434629f41f6b067c49b6622569a6c) ) /* labeled AHR SND V1.1 (U27) */

	ROM_REGION( 0xc0000, "grom", 0 )
	ROM_LOAD( "ahr_grom0.grom0", 0x00000, 0x20000, CRC(5ef57fe5) SHA1(e877979e034a61968b432037501e25a302a17a9a) )
	ROM_LOAD( "ahr_grom1.grom1", 0x20000, 0x20000, CRC(6aca95c0) SHA1(da7a899bf0812a7af178e48b5a626ce56a836579) )
	ROM_LOAD( "ahr_grom2.grom2", 0x40000, 0x10000, CRC(6d6fde1b) SHA1(aaabc45d4b566be42e8d28d767e4771a96d9caae) )
	/* GROM3, GROM4 & GROM5 are unpopulated */

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "ahr_srom0.srom0", 0x00000, 0x40000, CRC(56087f81) SHA1(1d4a1f396ee9d8ed51d0417ea94b0b379312d72f) )
ROM_END


ROM_START( arlingtna ) /* PCB  p/n 1030 rev. 1A */
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "ahr-d_v_1.21_u5.u5", 0x00000, 0x10000, CRC(00aae02e) SHA1(3bcfbd256c34ae222dde24ba9544f19da70b698e) ) /* labeled AHR-D V 1.21 (U5) - service menu reports version as 1.21-D */

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "ahr_snd_v1.1_u27.u27", 0x08000, 0x8000, CRC(dec57dca) SHA1(21a8ead10b0434629f41f6b067c49b6622569a6c) ) /* labeled AHR SND V 1.1 (U27) */

	ROM_REGION( 0xc0000, "grom", 0 )
	ROM_LOAD( "ahr_grom0.grom0", 0x00000, 0x20000, CRC(5ef57fe5) SHA1(e877979e034a61968b432037501e25a302a17a9a) )
	ROM_LOAD( "ahr_grom1.grom1", 0x20000, 0x20000, CRC(6aca95c0) SHA1(da7a899bf0812a7af178e48b5a626ce56a836579) )
	ROM_LOAD( "ahr_grom2.grom2", 0x40000, 0x10000, CRC(6d6fde1b) SHA1(aaabc45d4b566be42e8d28d767e4771a96d9caae) )
	/* GROM3, GROM4 & GROM5 are unpopulated */

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "ahr_srom0.srom0", 0x00000, 0x40000, CRC(56087f81) SHA1(1d4a1f396ee9d8ed51d0417ea94b0b379312d72f) )
ROM_END


ROM_START( peggle )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "j-stick.u5", 0x00000, 0x10000, CRC(140d5a9c) SHA1(841e5f45c6f306d9bd286e7d3e3c75b169c932e1) )

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "sound.u27", 0x08000, 0x8000, CRC(b99beb70) SHA1(8d82c3b081a1afb236afa658abb3aa605c6c2264) )

	ROM_REGION( 0xc0000, "grom", 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, CRC(5c02348d) SHA1(c85352728d94b3a5ca78c2493e98bdb4b3206bed) )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, CRC(85a7a3a2) SHA1(7fb7be67c75867fcccf4272f1ce42f2af94c16e6) )
	ROM_LOAD( "grom2.bin", 0x40000, 0x20000, CRC(bfe11f18) SHA1(5e8d36c6a86a8eb883f50620fbfcea3d8398faeb) )

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "srom0", 0x00000, 0x20000, CRC(001846ea) SHA1(6a6b77f6d51c4a68105c935804b896aad9981719) )
ROM_END


ROM_START( pegglet )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "trakball.u5", 0x00000, 0x10000, CRC(d2694868) SHA1(9945a308550c9d89a647d80257e3ab14f793ac6f) )

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "sound.u27", 0x08000, 0x8000, CRC(b99beb70) SHA1(8d82c3b081a1afb236afa658abb3aa605c6c2264) )

	ROM_REGION( 0xc0000, "grom", 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, CRC(5c02348d) SHA1(c85352728d94b3a5ca78c2493e98bdb4b3206bed) )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, CRC(85a7a3a2) SHA1(7fb7be67c75867fcccf4272f1ce42f2af94c16e6) )
	ROM_LOAD( "grom2.bin", 0x40000, 0x20000, CRC(bfe11f18) SHA1(5e8d36c6a86a8eb883f50620fbfcea3d8398faeb) )

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "srom0", 0x00000, 0x20000, CRC(001846ea) SHA1(6a6b77f6d51c4a68105c935804b896aad9981719) )
ROM_END


ROM_START( neckneck )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "nn_prg12.u5", 0x00000, 0x10000, CRC(8e51734a) SHA1(c184af73670235a9245bfdeec2b58acfe93170e3) )

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "nn_snd10.u27", 0x08000, 0x8000, CRC(74771b2f) SHA1(0a963d2962699bb1b4d08bd486979151d0a228da) )

	ROM_REGION( 0xc0000, "grom", 0 )
	ROM_LOAD( "nn_grom0.bin", 0x00000, 0x20000, CRC(064d1464) SHA1(4ce5bcadad93586c9af2b0c499d7a7140b080cdb) )
	ROM_LOAD( "nn_grom1.bin", 0x20000, 0x20000, CRC(622d9a0b) SHA1(f084dcf194cfc658f97f36972f2b2ff30e834fc5) )
	ROM_LOAD( "nn_grom2.bin", 0x40000, 0x20000, CRC(e7eb4020) SHA1(29ceb535d1dfddf8c12cf78919ea10f6b956b8b3) )
	ROM_LOAD( "nn_grom3.bin", 0x60000, 0x20000, CRC(765c8593) SHA1(a39d757a01d61b637b264bdb4449e090a1e90219) )

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "nn_srom0.bin", 0x00000, 0x40000, CRC(33687201) SHA1(7c84d453fb7e59b0a27cd68a049d019ff65f6f19) )
ROM_END


ROM_START( rimrockn )
	ROM_REGION( 0x20000, "maincpu", 0 )
	ROM_LOAD( "rrb.bim_2.2_u5.u5", 0x00000, 0x20000, CRC(97777683) SHA1(0998dde26daaa2d2b78e83647e03ba01b0ef31f2) ) /* Labeled as RRB.BIM 2.2 U5 */

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "rrbsnd_v1.1_u27.u27", 0x08000, 0x8000, CRC(59f87f0e) SHA1(46f38aca35a7c2faee227b4c950d20a6076c6fa7) ) /* Labeled as RRBSND V1.1 U27 */

	ROM_REGION( 0x100000, "grom", 0 )
	ROM_LOAD( "rbb-grom00",   0x00000, 0x40000, CRC(3eacbad9) SHA1(bff1ec6a24ccf983434e4e9453c30f36fa397534) )
	ROM_LOAD( "rbb-grom01",   0x40000, 0x40000, CRC(864cc269) SHA1(06f92889cd20881faeb59ec06ca1578ead2294f4) )
	ROM_LOAD( "rbb-grom02-2", 0x80000, 0x40000, CRC(47904233) SHA1(6a4d10e8f7b75582f706a74b37d59788613ffc61) )
	ROM_LOAD( "rbb-grom03-2", 0xc0000, 0x40000, CRC(f005f118) SHA1(aa39f12d07827e21eceb286557e37973e519b433) )

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "rbb-srom0", 0x00000, 0x40000, CRC(7ad42be0) SHA1(c9b519bad3c5c9a3315d1bf3292cc30ee0771db7) )

	ROM_REGION( 0x0600, "plds", 0 )
	ROM_LOAD( "pal16l8.u14", 0x0000, 0x0104, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pal16r4.u45", 0x0200, 0x0104, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pal16l8.u29", 0x0400, 0x0104, NO_DUMP ) /* PAL is read protected */
ROM_END


ROM_START( rimrockn20 )
	ROM_REGION( 0x20000, "maincpu", 0 )
	ROM_LOAD( "rrb.bim_2.0_u5.u5", 0x00000, 0x20000, CRC(7e9d5545) SHA1(2aa028b3f5d05bec4ee289e7d39eaad30b3d4d5f) ) /* Labeled as RRB.BIM 2.0 U5 */

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "rrbsnd_v1.1_u27.u27", 0x08000, 0x8000, CRC(59f87f0e) SHA1(46f38aca35a7c2faee227b4c950d20a6076c6fa7) ) /* Labeled as RRBSND V1.1 U27 */

	ROM_REGION( 0x100000, "grom", 0 )
	ROM_LOAD( "rbb-grom00",   0x00000, 0x40000, CRC(3eacbad9) SHA1(bff1ec6a24ccf983434e4e9453c30f36fa397534) )
	ROM_LOAD( "rbb-grom01",   0x40000, 0x40000, CRC(864cc269) SHA1(06f92889cd20881faeb59ec06ca1578ead2294f4) )
	ROM_LOAD( "rbb-grom02-2", 0x80000, 0x40000, CRC(47904233) SHA1(6a4d10e8f7b75582f706a74b37d59788613ffc61) )
	ROM_LOAD( "rbb-grom03-2", 0xc0000, 0x40000, CRC(f005f118) SHA1(aa39f12d07827e21eceb286557e37973e519b433) )

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "rbb-srom0", 0x00000, 0x40000, CRC(7ad42be0) SHA1(c9b519bad3c5c9a3315d1bf3292cc30ee0771db7) )
ROM_END


ROM_START( rimrockn16 )
	ROM_REGION( 0x20000, "maincpu", 0 )
	ROM_LOAD( "rrb.bim_1.6_u5.u5",0x00000, 0x20000, CRC(999cd502) SHA1(8ad0d641a9f853eff27be1d4de04ab86b9275d57) ) /* Labeled as RRB.BIM 1.6 U5 */

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "rrbsnd_v1.1_u27.u27", 0x08000, 0x8000, CRC(59f87f0e) SHA1(46f38aca35a7c2faee227b4c950d20a6076c6fa7) ) /* Labeled as RRBSND V1.1 U27 */

	ROM_REGION( 0x100000, "grom", 0 )
	ROM_LOAD( "rbb-grom00", 0x00000, 0x40000, CRC(3eacbad9) SHA1(bff1ec6a24ccf983434e4e9453c30f36fa397534) )
	ROM_LOAD( "rbb-grom01", 0x40000, 0x40000, CRC(864cc269) SHA1(06f92889cd20881faeb59ec06ca1578ead2294f4) )
	ROM_LOAD( "rbb-grom02", 0x80000, 0x40000, CRC(34e567d5) SHA1(d0eb6fd0da8b9c3bfe7d4ecfb4bd903e4926b63a) )
	ROM_LOAD( "rbb-grom03", 0xc0000, 0x40000, CRC(fd18045d) SHA1(a1b98e4a2aa6f3cd33a3e2f5744160e05cc9f8d1) )

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "rbb-srom0", 0x00000, 0x40000, CRC(7ad42be0) SHA1(c9b519bad3c5c9a3315d1bf3292cc30ee0771db7) )
ROM_END


ROM_START( rimrockn15 )
	ROM_REGION( 0x20000, "maincpu", 0 )
	ROM_LOAD( "rrb.bim_1.5_u5.u5",0x00000, 0x20000, CRC(d6c25bdf) SHA1(8255313e93a4afbe537ae61f5219e51fcf60d6b7) ) /* Labeled as RRB.BIM 1.5 U5 */

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "rrbsnd_v1.1_u27.u27", 0x08000, 0x8000, CRC(59f87f0e) SHA1(46f38aca35a7c2faee227b4c950d20a6076c6fa7) ) /* Labeled as RRBSND V1.1 U27 */

	ROM_REGION( 0x100000, "grom", 0 )
	ROM_LOAD( "rbb-grom00", 0x00000, 0x40000, CRC(3eacbad9) SHA1(bff1ec6a24ccf983434e4e9453c30f36fa397534) )
	ROM_LOAD( "rbb-grom01", 0x40000, 0x40000, CRC(864cc269) SHA1(06f92889cd20881faeb59ec06ca1578ead2294f4) )
	ROM_LOAD( "rbb-grom02", 0x80000, 0x40000, CRC(34e567d5) SHA1(d0eb6fd0da8b9c3bfe7d4ecfb4bd903e4926b63a) )
	ROM_LOAD( "rbb-grom03", 0xc0000, 0x40000, CRC(fd18045d) SHA1(a1b98e4a2aa6f3cd33a3e2f5744160e05cc9f8d1) )

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "rbb-srom0", 0x00000, 0x40000, CRC(7ad42be0) SHA1(c9b519bad3c5c9a3315d1bf3292cc30ee0771db7) )
ROM_END


ROM_START( rimrockn12 )
	ROM_REGION( 0x20000, "maincpu", 0 )
	ROM_LOAD( "rrb.bim_1.2_u5.u5",0x00000, 0x20000, CRC(661761a6) SHA1(7224b1eac2fd0969d70657448ab241a433143df4) ) /* Labeled as RRB.BIM 1.2 U5 */

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "rrbsnd_v1_u27.u27", 0x08000, 0x8000, CRC(8eda5f53) SHA1(f256544a8c87125587719460ed0fef14efef9015) ) /* Labeled as RRBSND V1 U27 */

	ROM_REGION( 0x100000, "grom", 0 )
	ROM_LOAD( "rbb-grom00", 0x00000, 0x40000, CRC(3eacbad9) SHA1(bff1ec6a24ccf983434e4e9453c30f36fa397534) )
	ROM_LOAD( "rbb-grom01", 0x40000, 0x40000, CRC(864cc269) SHA1(06f92889cd20881faeb59ec06ca1578ead2294f4) )
	ROM_LOAD( "rbb-grom02", 0x80000, 0x40000, CRC(34e567d5) SHA1(d0eb6fd0da8b9c3bfe7d4ecfb4bd903e4926b63a) )
	ROM_LOAD( "rbb-grom03", 0xc0000, 0x40000, CRC(fd18045d) SHA1(a1b98e4a2aa6f3cd33a3e2f5744160e05cc9f8d1) )

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "rbb-srom0", 0x00000, 0x40000, CRC(7ad42be0) SHA1(c9b519bad3c5c9a3315d1bf3292cc30ee0771db7) )
ROM_END


ROM_START( rimrockn12b )
	ROM_REGION( 0x20000, "maincpu", 0 )
	ROM_LOAD( "rbba-1.u5", 0x00000, 0x20000, CRC(f99561a8) SHA1(ee59b934839ef5599bb393e0860d54c95f6df1b0) ) // ZX2880JL

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "rrbsndv1.u27", 0x08000, 0x8000, CRC(8eda5f53) SHA1(f256544a8c87125587719460ed0fef14efef9015) ) // Found on a double-sized ROM (27C512) with its first half empty

	ROM_REGION( 0x100000, "grom", 0 )
	ROM_LOAD( "rbb-grom00", 0x00000, 0x40000, CRC(3eacbad9) SHA1(bff1ec6a24ccf983434e4e9453c30f36fa397534) )
	ROM_LOAD( "rbb-grom01", 0x40000, 0x40000, CRC(864cc269) SHA1(06f92889cd20881faeb59ec06ca1578ead2294f4) )
	ROM_LOAD( "rbb-grom02", 0x80000, 0x40000, CRC(34e567d5) SHA1(d0eb6fd0da8b9c3bfe7d4ecfb4bd903e4926b63a) )
	ROM_LOAD( "rbb-grom03", 0xc0000, 0x40000, CRC(fd18045d) SHA1(a1b98e4a2aa6f3cd33a3e2f5744160e05cc9f8d1) )

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "rbb-srom0", 0x00000, 0x40000, CRC(7ad42be0) SHA1(c9b519bad3c5c9a3315d1bf3292cc30ee0771db7) )

	/* Unused */
	ROM_REGION( 0x0096b, "plds", 0 )
	ROM_LOAD( "a-palce16v8h.u53", 0x00000, 0x00117, NO_DUMP ) // Undumped (solderded)
	ROM_LOAD( "a-palce16v8q.u45", 0x00117, 0x00117, NO_DUMP ) // Undumped (solderded)
	ROM_LOAD( "a-palce16v8h.u14", 0x0022e, 0x00117, NO_DUMP ) // Undumped (solderded)
	ROM_LOAD( "a-gal22v10.u55",   0x00345, 0x002e1, NO_DUMP ) // Undumped (solderded)
	ROM_LOAD( "a-palce16v8h.u65", 0x00626, 0x00117, NO_DUMP ) // Undumped (solderded)
	ROM_LOAD( "a-palce16v8h.u50", 0x0073d, 0x00117, NO_DUMP ) // Undumped (solderded)
	ROM_LOAD( "b-palce16v8h.u29", 0x00854, 0x00117, NO_DUMP ) // Undumped (solderded)
ROM_END


ROM_START( ninclown )
	ROM_REGION( 0x40000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "prog1", 0x00000, 0x20000, CRC(fabfdcd2) SHA1(7a9852838cf7772d8f8f956b03823c4222520a5a) )
	ROM_LOAD16_BYTE( "prog0", 0x00001, 0x20000, CRC(eca63db5) SHA1(b86d75ee81e155a21de18c2e0fa898f15d61560d) )

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "nc-snd", 0x08000, 0x8000, CRC(f9d5b4e1) SHA1(e5c3774db349b60baf11baecf55ac432871e612c) )

	ROM_REGION( 0x180000, "grom", 0 )
	ROM_LOAD( "nc-grom0", 0x000000, 0x40000, CRC(532f7bff) SHA1(cc7a64ad1581f37ff4bcad78c11aad355a6e9aa8) )
	ROM_LOAD( "nc-grom1", 0x040000, 0x40000, CRC(45640d4a) SHA1(1b59ae2a74fef535c646586422f17b39c9ae247b) )
	ROM_LOAD( "nc-grom2", 0x080000, 0x40000, CRC(c8281d06) SHA1(f75d4e858f9bc19e5234f184b75818e448638c15) )
	ROM_LOAD( "nc-grom3", 0x0c0000, 0x40000, CRC(2a6d33ac) SHA1(0773a53ece0b09e7c33f0e5beb4518059f76b7af) )
	ROM_LOAD( "nc-grom4", 0x100000, 0x40000, CRC(910876ba) SHA1(d65d0a5043afe2239596ba76174a27fea60ffb35) )
	ROM_LOAD( "nc-grom5", 0x140000, 0x40000, CRC(2533279b) SHA1(c21b218b11d20f2cdac2321ffcb1334ec9ec9ffc) )

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "srom0.bin", 0x00000, 0x40000, CRC(f6b501e1) SHA1(0f1eb6f40f891d11822f1255c0a84975526b2b1b) )
ROM_END


ROM_START( gpgolf ) /* P/N 1047 REV. 1 main board + P/N 1038 REV2 sound board */
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "gpg_v1.1.u5", 0x00000, 0x10000, CRC(631e77e0) SHA1(847ba1e00d31441620a2a1f45a9aa58df84bde8b) ) /* Joystick version 1.1 */

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "sndv1.u27", 0x08000, 0x8000, CRC(55734876) SHA1(eb5ef816acbc6e35642749e38a2908b7ba359b9d) )

	ROM_REGION( 0xc0000, "grom", 0 )
	ROM_LOAD( "grom00.grom0", 0x00000, 0x40000, CRC(c3a7b54b) SHA1(414d693bc5337d578d2630817dd647cf7e5cbcf7) )
	ROM_LOAD( "grom01.grom1", 0x40000, 0x40000, CRC(b7fe172d) SHA1(1ad0f3ce0f240ac1a23c0c5bdd9f99ec81bc14f1) )
	ROM_LOAD( "grom02.grom2", 0x80000, 0x40000, CRC(aebe6c45) SHA1(15e64fcb36cb1064988ee5cd45699d501a6e7f01) )
	/* GROM3 not socketted or populated */

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "srom00.srom0", 0x00000, 0x20000, CRC(4dd4db42) SHA1(0dffb51e8de36d8747f443fd65fe9927815eaff0) )
ROM_END


ROM_START( gpgolfa ) /* P/N 1047 REV. 1 main board + P/N 1038 REV2 sound board */
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "gpar.bin.u5", 0x00000, 0x10000, CRC(bcb030b0) SHA1(6fbe0ccd50c3769050d86e2376950fd06ce2abdc) ) /* Joystick version 1.0 - handwritten label */

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "golf_sound_12-19-91_v.96.u27", 0x00000, 0x10000, CRC(f46b4300) SHA1(7be1878b72c55fb83b2cae3b79b1f65fe8825b4a) ) /* 27C512 with the first 0x8000 as 0xFF fill - handwritten label  GOLF SOUND  12/19/91  V.96 */
//  ROM_LOAD( "golf_sound.u27", 0x08000, 0x8000, CRC(3183d7f3) SHA1(482411947aa3074cec7d4491f6ee64785894d27c) ) /* Different than sndv1.u27 */

	ROM_REGION( 0xc0000, "grom", 0 )
	ROM_LOAD( "grom00.grom0", 0x00000, 0x40000, CRC(c3a7b54b) SHA1(414d693bc5337d578d2630817dd647cf7e5cbcf7) )
	ROM_LOAD( "grom01.grom1", 0x40000, 0x40000, CRC(b7fe172d) SHA1(1ad0f3ce0f240ac1a23c0c5bdd9f99ec81bc14f1) )
	ROM_LOAD( "grom02.grom2", 0x80000, 0x40000, CRC(aebe6c45) SHA1(15e64fcb36cb1064988ee5cd45699d501a6e7f01) )
	/* GROM3 not socketted or populated */

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "golf_speech_12-19-91_v.96.srom0", 0x00000, 0x20000, CRC(4dd4db42) SHA1(0dffb51e8de36d8747f443fd65fe9927815eaff0) ) /* == srom00.srom0 - handwritten label  GOLF SPEECh  12/19/91  V.96 */
ROM_END


ROM_START( gtg2 )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD( "gtg2_v2_2.u5", 0x00000, 0x10000, CRC(4a61580f) SHA1(7c64648d47418fbcc0f9b5bd91f88856209bc0f5) ) /* Trackball version */

	ROM_REGION( 0x10000, "soundcpu", 0 )
	ROM_LOAD( "sndv1.u27", 0x08000, 0x8000, CRC(55734876) SHA1(eb5ef816acbc6e35642749e38a2908b7ba359b9d) )

	ROM_REGION( 0xc0000, "grom", 0 )
	ROM_LOAD( "grom0.bin", 0x00000, 0x20000, CRC(a29c688a) SHA1(32dbb996a5e4c23cfd44b79312ac4a767658f20a) )
	ROM_LOAD( "grom1.bin", 0x20000, 0x20000, CRC(a4182776) SHA1(9f4704d1a61a4fffce454c82cb3eb4629d9a2006) )
	ROM_LOAD( "grom2.bin", 0x40000, 0x20000, CRC(0580bb99) SHA1(012d473bc63632a5ed1a250daa54a00a4e30e8b2) )
	ROM_LOAD( "grom3.bin", 0x60000, 0x20000, CRC(89edb624) SHA1(e77d5ecd04f44f184e76eb2506660a04b6bc7fe8) )
	ROM_LOAD( "grom4.bin", 0x80000, 0x20000, CRC(f6557950) SHA1(fd2bbd6b852a9f2775a23d9fdaa20d31d6766d7f) )
	ROM_LOAD( "grom5.bin", 0xa0000, 0x20000, CRC(a680ce6a) SHA1(fe51546933c093cea81858b213afb0e926e9eb4e) )

	ROM_REGION( 0x40000, "oki", 0 )
	ROM_LOAD( "srom00.bin", 0x00000, 0x20000, CRC(4dd4db42) SHA1(0dffb51e8de36d8747f443fd65fe9927815eaff0) )

	ROM_REGION( 0x0200, "plds", 0 )
	ROM_LOAD( "tibpal16l8.u29", 0x0000, 0x0104, NO_DUMP ) /* PAL is read protected */
ROM_END



/*************************************
 *
 *  Driver-specific init
 *
 *************************************/

void itech8_state::init_invbank()
{
	m_bankxor = 1;
}


void itech8_state::init_slikshot()
{
	save_item(NAME(m_z80_ctrl));
	save_item(NAME(m_z80_port_val));
	save_item(NAME(m_z80_clear_to_send));
	save_item(NAME(m_sensor0));
	save_item(NAME(m_sensor1));
	save_item(NAME(m_sensor2));
	save_item(NAME(m_sensor3));
	save_item(NAME(m_curvx));
	save_item(NAME(m_curvy));
	save_item(NAME(m_curx));
	save_item(NAME(m_xbuffer));
	save_item(NAME(m_ybuffer));
	save_item(NAME(m_ybuffer_next));
	save_item(NAME(m_curxpos));
	save_item(NAME(m_last_ytotal));
	save_item(NAME(m_crosshair_vis));
}


void itech8_state::init_hstennis()
{
	m_visarea.set(0, 375, 0, 239);
}


void itech8_state::init_arligntn()
{
	init_invbank();
	m_visarea.set(16, 389, 0, 239);
}


void itech8_state::init_peggle()
{
	m_visarea.set(18, 367, 0, 239);
}


void itech8_state::init_neckneck()
{
	m_visarea.set(8, 375, 0, 239);
}

/*************************************
 *
 *  Game drivers
 *
 *************************************/

/* Wheel of Fortune-style PCB */
GAME( 1989, wfortune,   0,         wfortune,          wfortune, itech8_state, empty_init,    ROT0,   "GameTek", "Wheel Of Fortune (set 1)", 0 )
GAME( 1989, wfortunea,  wfortune,  wfortune,          wfortune, itech8_state, empty_init,    ROT0,   "GameTek", "Wheel Of Fortune (set 2)", 0 ) /* program ROM label states "R1" */

/* Grudge Match-style PCB */
GAME( 1989, grmatch,    0,         grmatch,           grmatch,  grmatch_state, empty_init,   ROT0,   "Yankee Game Technology", "Grudge Match (Yankee Game Technology)", 0 )

/* Strata Bowling-style PCB */
GAME( 1990, stratab,    0,         stratab_hi,        stratab,  itech8_state, empty_init,    ROT270, "Strata/Incredible Technologies", "Strata Bowling (V3)", 0 ) // still says V1 in service mode?
GAME( 1990, stratab1,   stratab,   stratab_hi,        stratab,  itech8_state, empty_init,    ROT270, "Strata/Incredible Technologies", "Strata Bowling (V1)", 0 )
GAME( 1990, gtg,        0,         stratab_hi,        gtg,      itech8_state, empty_init,    ROT0,   "Strata/Incredible Technologies", "Golden Tee Golf (Joystick, v3.3)", 0 )
GAME( 1990, gtgj31,     gtg,       stratab_hi,        gtg,      itech8_state, empty_init,    ROT0,   "Strata/Incredible Technologies", "Golden Tee Golf (Joystick, v3.1)", 0 )
GAME( 1989, gtgt21,     gtg,       stratab_hi,        gtgt,     itech8_state, empty_init,    ROT0,   "Strata/Incredible Technologies", "Golden Tee Golf (Trackball, v2.1)", 0 )
GAME( 1989, gtgt20,     gtg,       stratab_hi,        gtgt,     itech8_state, empty_init,    ROT0,   "Strata/Incredible Technologies", "Golden Tee Golf (Trackball, v2.0)", 0 )
GAME( 1989, gtgt10,     gtg,       stratab_hi,        gtgt,     itech8_state, empty_init,    ROT0,   "Strata/Incredible Technologies", "Golden Tee Golf (Trackball, v1.0)", 0 )
GAME( 1989, gtg2t,      gtg2,      stratab_hi,        gtg2t,    itech8_state, init_invbank,  ROT0,   "Strata/Incredible Technologies", "Golden Tee Golf II (Trackball, V1.1)", 0 )
GAME( 1991, gtg2j,      gtg2,      stratab_lo,        gtg,      itech8_state, empty_init,    ROT0,   "Strata/Incredible Technologies", "Golden Tee Golf II (Joystick, V1.0)", 0 )

/* Slick Shot-style PCB */
GAME( 1990, slikshot,   0,         slikshot_hi,       slikshot, itech8_state, init_slikshot, ROT90,  "Grand Products/Incredible Technologies", "Slick Shot (V2.2)", MACHINE_MECHANICAL )
GAME( 1990, slikshot17, slikshot,  slikshot_hi,       slikshot, itech8_state, init_slikshot, ROT90,  "Grand Products/Incredible Technologies", "Slick Shot (V1.7)", MACHINE_MECHANICAL )
GAME( 1990, slikshot16, slikshot,  slikshot_hi,       slikshot, itech8_state, init_slikshot, ROT90,  "Grand Products/Incredible Technologies", "Slick Shot (V1.6)", MACHINE_MECHANICAL )
GAME( 1990, dynobop,    0,         slikshot_hi,       dynobop,  itech8_state, init_slikshot, ROT90,  "Grand Products/Incredible Technologies", "Dyno Bop (V1.1)", MACHINE_MECHANICAL )
GAME( 1990, sstrike,    0,         sstrike,           sstrike,  itech8_state, empty_init,    ROT270, "Strata/Incredible Technologies", "Super Strike Bowling (V1)", MACHINE_MECHANICAL )
GAME( 1990, stratabs,   stratab,   sstrike,           stratabs, itech8_state, empty_init,    ROT270, "Strata/Incredible Technologies", "Strata Bowling (V1 4T, Super Strike Bowling type PCB)", MACHINE_NOT_WORKING ) // need to figure out the control hookup for this set, service mode indicates it's still a trackball like stratab
GAME( 1991, pokrdice,   0,         slikshot_lo_noz80, pokrdice, itech8_state, empty_init,    ROT90,  "Strata/Incredible Technologies", "Poker Dice (V1.7)", 0 )

/* Hot Shots Tennis-style PCB */
GAME( 1990, hstennis,   0,         hstennis_hi,       hstennis, itech8_state, init_hstennis, ROT90,  "Strata/Incredible Technologies", "Hot Shots Tennis (V1.1)", 0 )
GAME( 1990, hstennis10, hstennis,  hstennis_hi,       hstennis, itech8_state, init_hstennis, ROT90,  "Strata/Incredible Technologies", "Hot Shots Tennis (V1.0)", 0 )
GAME( 1991, arlingtn,   0,         hstennis_hi,       arlingtn, itech8_state, init_arligntn, ROT0,   "Strata/Incredible Technologies", "Arlington Horse Racing (v1.40-D)", 0 )
GAME( 1991, arlingtna,  arlingtn,  hstennis_hi,       arlingtn, itech8_state, init_arligntn, ROT0,   "Strata/Incredible Technologies", "Arlington Horse Racing (v1.21-D)", 0 )
GAME( 1991, peggle,     0,         hstennis_lo,       peggle,   itech8_state, init_peggle,   ROT90,  "Strata/Incredible Technologies", "Peggle (Joystick, v1.0)", 0 )
GAME( 1991, pegglet,    peggle,    hstennis_lo,       pegglet,  itech8_state, init_peggle,   ROT90,  "Strata/Incredible Technologies", "Peggle (Trackball, v1.0)", 0 )
GAME( 1992, neckneck,   0,         hstennis_lo,       neckneck, itech8_state, init_neckneck, ROT0,   "Bundra Games/Incredible Technologies", "Neck-n-Neck (v1.2)", 0 )

/* Rim Rockin' Basketball-style PCB */
GAME( 1991, rimrockn,    0,        rimrockn,          rimrockn, itech8_state, empty_init,    ROT0,   "Strata/Incredible Technologies", "Rim Rockin' Basketball (V2.2)", 0 )
GAME( 1991, rimrockn20,  rimrockn, rimrockn,          rimrockn, itech8_state, empty_init,    ROT0,   "Strata/Incredible Technologies", "Rim Rockin' Basketball (V2.0)", 0 )
GAME( 1991, rimrockn16,  rimrockn, rimrockn,          rimrockn, itech8_state, empty_init,    ROT0,   "Strata/Incredible Technologies", "Rim Rockin' Basketball (V1.6)", 0 )
GAME( 1991, rimrockn15,  rimrockn, rimrockn,          rimrockn, itech8_state, empty_init,    ROT0,   "Strata/Incredible Technologies", "Rim Rockin' Basketball (V1.5)", 0 )
GAME( 1991, rimrockn12,  rimrockn, rimrockn,          rimrockn, itech8_state, empty_init,    ROT0,   "Strata/Incredible Technologies", "Rim Rockin' Basketball (V1.2)", 0 )
GAME( 1991, rimrockn12b, rimrockn, rimrockn,          rimrockn, itech8_state, empty_init,    ROT0,   "bootleg",                        "Rim Rockin' Basketball (V1.2, bootleg)", 0 )

/* Ninja Clowns-style PCB */
GAME( 1991, ninclown,   0,         ninclown,          ninclown, itech8_state, empty_init,    ROT0,   "Strata/Incredible Technologies", "Ninja Clowns (27 oct 91)", 0 )

/* Golden Tee Golf II-style PCB */
GAME( 1992, gpgolf,     0,         gtg2,              gpgolf,   itech8_state, empty_init,    ROT0,   "Strata/Incredible Technologies", "Golden Par Golf (Joystick, V1.1)", 0 ) /* Seems to stall during Demo Mode?? */
GAME( 1991, gpgolfa,    gpgolf,    gtg2,              gpgolf,   itech8_state, empty_init,    ROT0,   "Strata/Incredible Technologies", "Golden Par Golf (Joystick, V1.0)", 0 ) /* Seems to stall during Demo Mode?? */
GAME( 1992, gtg2,       0,         gtg2,              gtg2,     itech8_state, init_invbank,  ROT0,   "Strata/Incredible Technologies", "Golden Tee Golf II (Trackball, V2.2)", 0 )
