/*
    $Id: wchar.c 2966 2023-01-08 21:28:13Z soci $

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/
#include "wchar.h"

#if defined __DJGPP__ || (defined __WATCOMC__ && !defined _WIN32)
#include <errno.h>
#include <ctype.h>
#include <string.h>
#include "attributes.h"
#ifdef __DJGPP__
#include <dpmi.h>
#include <go32.h>
#else
#include <i86.h>
#endif

static const wchar_t *cp;
static uint32_t revcp[128];

static const wchar_t cp437[128] = {
    0x00c7, 0x00fc, 0x00e9, 0x00e2, 0x00e4, 0x00e0, 0x00e5, 0x00e7, 0x00ea,
    0x00eb, 0x00e8, 0x00ef, 0x00ee, 0x00ec, 0x00c4, 0x00c5, 0x00c9, 0x00e6,
    0x00c6, 0x00f4, 0x00f6, 0x00f2, 0x00fb, 0x00f9, 0x00ff, 0x00d6, 0x00dc,
    0x00a2, 0x00a3, 0x00a5, 0x20a7, 0x0192, 0x00e1, 0x00ed, 0x00f3, 0x00fa,
    0x00f1, 0x00d1, 0x00aa, 0x00ba, 0x00bf, 0x2310, 0x00ac, 0x00bd, 0x00bc,
    0x00a1, 0x00ab, 0x00bb, 0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561,
    0x2562, 0x2556, 0x2555, 0x2563, 0x2551, 0x2557, 0x255d, 0x255c, 0x255b,
    0x2510, 0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x255e, 0x255f,
    0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x2567, 0x2568,
    0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256b, 0x256a, 0x2518,
    0x250c, 0x2588, 0x2584, 0x258c, 0x2590, 0x2580, 0x03b1, 0x00df, 0x0393,
    0x03c0, 0x03a3, 0x03c3, 0x00b5, 0x03c4, 0x03a6, 0x0398, 0x03a9, 0x03b4,
    0x221e, 0x03c6, 0x03b5, 0x2229, 0x2261, 0x00b1, 0x2265, 0x2264, 0x2320,
    0x2321, 0x00f7, 0x2248, 0x00b0, 0x2219, 0x00b7, 0x221a, 0x207f, 0x00b2,
    0x25a0, 0x00a0
};

static const wchar_t cp737[128] = {
    0x0391, 0x0392, 0x0393, 0x0394, 0x0395, 0x0396, 0x0397, 0x0398, 0x0399,
    0x039a, 0x039b, 0x039c, 0x039d, 0x039e, 0x039f, 0x03a0, 0x03a1, 0x03a3,
    0x03a4, 0x03a5, 0x03a6, 0x03a7, 0x03a8, 0x03a9, 0x03b1, 0x03b2, 0x03b3,
    0x03b4, 0x03b5, 0x03b6, 0x03b7, 0x03b8, 0x03b9, 0x03ba, 0x03bb, 0x03bc,
    0x03bd, 0x03be, 0x03bf, 0x03c0, 0x03c1, 0x03c3, 0x03c2, 0x03c4, 0x03c5,
    0x03c6, 0x03c7, 0x03c8, 0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561,
    0x2562, 0x2556, 0x2555, 0x2563, 0x2551, 0x2557, 0x255d, 0x255c, 0x255b,
    0x2510, 0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x255e, 0x255f,
    0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x2567, 0x2568,
    0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256b, 0x256a, 0x2518,
    0x250c, 0x2588, 0x2584, 0x258c, 0x2590, 0x2580, 0x03c9, 0x03ac, 0x03ad,
    0x03ae, 0x03ca, 0x03af, 0x03cc, 0x03cd, 0x03cb, 0x03ce, 0x0386, 0x0388,
    0x0389, 0x038a, 0x038c, 0x038e, 0x038f, 0x00b1, 0x2265, 0x2264, 0x03aa,
    0x03ab, 0x00f7, 0x2248, 0x00b0, 0x2219, 0x00b7, 0x221a, 0x207f, 0x00b2,
    0x25a0, 0x00a0
};

static const wchar_t cp775[128] = {
    0x0106, 0x00fc, 0x00e9, 0x0101, 0x00e4, 0x0123, 0x00e5, 0x0107, 0x0142,
    0x0113, 0x0156, 0x0157, 0x012b, 0x0179, 0x00c4, 0x00c5, 0x00c9, 0x00e6,
    0x00c6, 0x014d, 0x00f6, 0x0122, 0x00a2, 0x015a, 0x015b, 0x00d6, 0x00dc,
    0x00f8, 0x00a3, 0x00d8, 0x00d7, 0x00a4, 0x0100, 0x012a, 0x00f3, 0x017b,
    0x017c, 0x017a, 0x201d, 0x00a6, 0x00a9, 0x00ae, 0x00ac, 0x00bd, 0x00bc,
    0x0141, 0x00ab, 0x00bb, 0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x0104,
    0x010c, 0x0118, 0x0116, 0x2563, 0x2551, 0x2557, 0x255d, 0x012e, 0x0160,
    0x2510, 0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x0172, 0x016a,
    0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x017d, 0x0105,
    0x010d, 0x0119, 0x0117, 0x012f, 0x0161, 0x0173, 0x016b, 0x017e, 0x2518,
    0x250c, 0x2588, 0x2584, 0x258c, 0x2590, 0x2580, 0x00d3, 0x00df, 0x014c,
    0x0143, 0x00f5, 0x00d5, 0x00b5, 0x0144, 0x0136, 0x0137, 0x013b, 0x013c,
    0x0146, 0x0112, 0x0145, 0x2019, 0x00ad, 0x00b1, 0x201c, 0x00be, 0x00b6,
    0x00a7, 0x00f7, 0x201e, 0x00b0, 0x2219, 0x00b7, 0x00b9, 0x00b3, 0x00b2,
    0x25a0, 0x00a0
};

static const wchar_t cp850[128] = {
    0x00c7, 0x00fc, 0x00e9, 0x00e2, 0x00e4, 0x00e0, 0x00e5, 0x00e7, 0x00ea,
    0x00eb, 0x00e8, 0x00ef, 0x00ee, 0x00ec, 0x00c4, 0x00c5, 0x00c9, 0x00e6,
    0x00c6, 0x00f4, 0x00f6, 0x00f2, 0x00fb, 0x00f9, 0x00ff, 0x00d6, 0x00dc,
    0x00f8, 0x00a3, 0x00d8, 0x00d7, 0x0192, 0x00e1, 0x00ed, 0x00f3, 0x00fa,
    0x00f1, 0x00d1, 0x00aa, 0x00ba, 0x00bf, 0x00ae, 0x00ac, 0x00bd, 0x00bc,
    0x00a1, 0x00ab, 0x00bb, 0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x00c1,
    0x00c2, 0x00c0, 0x00a9, 0x2563, 0x2551, 0x2557, 0x255d, 0x00a2, 0x00a5,
    0x2510, 0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x00e3, 0x00c3,
    0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x00a4, 0x00f0,
    0x00d0, 0x00ca, 0x00cb, 0x00c8, 0x0131, 0x00cd, 0x00ce, 0x00cf, 0x2518,
    0x250c, 0x2588, 0x2584, 0x00a6, 0x00cc, 0x2580, 0x00d3, 0x00df, 0x00d4,
    0x00d2, 0x00f5, 0x00d5, 0x00b5, 0x00fe, 0x00de, 0x00da, 0x00db, 0x00d9,
    0x00fd, 0x00dd, 0x00af, 0x00b4, 0x00ad, 0x00b1, 0x2017, 0x00be, 0x00b6,
    0x00a7, 0x00f7, 0x00b8, 0x00b0, 0x00a8, 0x00b7, 0x00b9, 0x00b3, 0x00b2,
    0x25a0, 0x00a0
};

static const wchar_t cp852[128] = {
    0x00c7, 0x00fc, 0x00e9, 0x00e2, 0x00e4, 0x016f, 0x0107, 0x00e7, 0x0142,
    0x00eb, 0x0150, 0x0151, 0x00ee, 0x0179, 0x00c4, 0x0106, 0x00c9, 0x0139,
    0x013a, 0x00f4, 0x00f6, 0x013d, 0x013e, 0x015a, 0x015b, 0x00d6, 0x00dc,
    0x0164, 0x0165, 0x0141, 0x00d7, 0x010d, 0x00e1, 0x00ed, 0x00f3, 0x00fa,
    0x0104, 0x0105, 0x017d, 0x017e, 0x0118, 0x0119, 0x00ac, 0x017a, 0x010c,
    0x015f, 0x00ab, 0x00bb, 0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x00c1,
    0x00c2, 0x011a, 0x015e, 0x2563, 0x2551, 0x2557, 0x255d, 0x017b, 0x017c,
    0x2510, 0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x0102, 0x0103,
    0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x00a4, 0x0111,
    0x0110, 0x010e, 0x00cb, 0x010f, 0x0147, 0x00cd, 0x00ce, 0x011b, 0x2518,
    0x250c, 0x2588, 0x2584, 0x0162, 0x016e, 0x2580, 0x00d3, 0x00df, 0x00d4,
    0x0143, 0x0144, 0x0148, 0x0160, 0x0161, 0x0154, 0x00da, 0x0155, 0x0170,
    0x00fd, 0x00dd, 0x0163, 0x00b4, 0x00ad, 0x02dd, 0x02db, 0x02c7, 0x02d8,
    0x00a7, 0x00f7, 0x00b8, 0x00b0, 0x00a8, 0x02d9, 0x0171, 0x0158, 0x0159,
    0x25a0, 0x00a0
};

static const wchar_t cp855[128] = {
    0x0452, 0x0402, 0x0453, 0x0403, 0x0451, 0x0401, 0x0454, 0x0404, 0x0455,
    0x0405, 0x0456, 0x0406, 0x0457, 0x0407, 0x0458, 0x0408, 0x0459, 0x0409,
    0x045a, 0x040a, 0x045b, 0x040b, 0x045c, 0x040c, 0x045e, 0x040e, 0x045f,
    0x040f, 0x044e, 0x042e, 0x044a, 0x042a, 0x0430, 0x0410, 0x0431, 0x0411,
    0x0446, 0x0426, 0x0434, 0x0414, 0x0435, 0x0415, 0x0444, 0x0424, 0x0433,
    0x0413, 0x00ab, 0x00bb, 0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x0445,
    0x0425, 0x0438, 0x0418, 0x2563, 0x2551, 0x2557, 0x255d, 0x0439, 0x0419,
    0x2510, 0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x043a, 0x041a,
    0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x00a4, 0x043b,
    0x041b, 0x043c, 0x041c, 0x043d, 0x041d, 0x043e, 0x041e, 0x043f, 0x2518,
    0x250c, 0x2588, 0x2584, 0x041f, 0x044f, 0x2580, 0x042f, 0x0440, 0x0420,
    0x0441, 0x0421, 0x0442, 0x0422, 0x0443, 0x0423, 0x0436, 0x0416, 0x0432,
    0x0412, 0x044c, 0x042c, 0x2116, 0x00ad, 0x044b, 0x042b, 0x0437, 0x0417,
    0x0448, 0x0428, 0x044d, 0x042d, 0x0449, 0x0429, 0x0447, 0x0427, 0x00a7,
    0x25a0, 0x00a0
};

static const wchar_t cp857[128] = {
    0x00c7, 0x00fc, 0x00e9, 0x00e2, 0x00e4, 0x00e0, 0x00e5, 0x00e7, 0x00ea,
    0x00eb, 0x00e8, 0x00ef, 0x00ee, 0x0131, 0x00c4, 0x00c5, 0x00c9, 0x00e6,
    0x00c6, 0x00f4, 0x00f6, 0x00f2, 0x00fb, 0x00f9, 0x0130, 0x00d6, 0x00dc,
    0x00f8, 0x00a3, 0x00d8, 0x015e, 0x015f, 0x00e1, 0x00ed, 0x00f3, 0x00fa,
    0x00f1, 0x00d1, 0x011e, 0x011f, 0x00bf, 0x00ae, 0x00ac, 0x00bd, 0x00bc,
    0x00a1, 0x00ab, 0x00bb, 0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x00c1,
    0x00c2, 0x00c0, 0x00a9, 0x2563, 0x2551, 0x2557, 0x255d, 0x00a2, 0x00a5,
    0x2510, 0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x00e3, 0x00c3,
    0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x00a4, 0x00ba,
    0x00aa, 0x00ca, 0x00cb, 0x00c8, 0x20ac, 0x00cd, 0x00ce, 0x00cf, 0x2518,
    0x250c, 0x2588, 0x2584, 0x00a6, 0x00cc, 0x2580, 0x00d3, 0x00df, 0x00d4,
    0x00d2, 0x00f5, 0x00d5, 0x00b5,      0, 0x00d7, 0x00da, 0x00db, 0x00d9,
    0x00ec, 0x00ff, 0x00af, 0x00b4, 0x00ad, 0x00b1,      0, 0x00be, 0x00b6,
    0x00a7, 0x00f7, 0x00b8, 0x00b0, 0x00a8, 0x00b7, 0x00b9, 0x00b3, 0x00b2,
    0x25a0, 0x00a0
};

static const wchar_t cp860[128] = {
    0x00c7, 0x00fc, 0x00e9, 0x00e2, 0x00e3, 0x00e0, 0x00c1, 0x00e7, 0x00ea,
    0x00ca, 0x00e8, 0x00cd, 0x00d4, 0x00ec, 0x00c3, 0x00c2, 0x00c9, 0x00c0,
    0x00c8, 0x00f4, 0x00f5, 0x00f2, 0x00da, 0x00f9, 0x00cc, 0x00d5, 0x00dc,
    0x00a2, 0x00a3, 0x00d9, 0x20a7, 0x00d3, 0x00e1, 0x00ed, 0x00f3, 0x00fa,
    0x00f1, 0x00d1, 0x00aa, 0x00ba, 0x00bf, 0x00d2, 0x00ac, 0x00bd, 0x00bc,
    0x00a1, 0x00ab, 0x00bb, 0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561,
    0x2562, 0x2556, 0x2555, 0x2563, 0x2551, 0x2557, 0x255d, 0x255c, 0x255b,
    0x2510, 0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x255e, 0x255f,
    0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x2567, 0x2568,
    0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256b, 0x256a, 0x2518,
    0x250c, 0x2588, 0x2584, 0x258c, 0x2590, 0x2580, 0x03b1, 0x00df, 0x0393,
    0x03c0, 0x03a3, 0x03c3, 0x00b5, 0x03c4, 0x03a6, 0x0398, 0x03a9, 0x03b4,
    0x221e, 0x03c6, 0x03b5, 0x2229, 0x2261, 0x00b1, 0x2265, 0x2264, 0x2320,
    0x2321, 0x00f7, 0x2248, 0x00b0, 0x2219, 0x00b7, 0x221a, 0x207f, 0x00b2,
    0x25a0, 0x00a0
};

static const wchar_t cp861[128] = {
    0x00c7, 0x00fc, 0x00e9, 0x00e2, 0x00e4, 0x00e0, 0x00e5, 0x00e7, 0x00ea,
    0x00eb, 0x00e8, 0x00d0, 0x00f0, 0x00de, 0x00c4, 0x00c5, 0x00c9, 0x00e6,
    0x00c6, 0x00f4, 0x00f6, 0x00fe, 0x00fb, 0x00dd, 0x00fd, 0x00d6, 0x00dc,
    0x00f8, 0x00a3, 0x00d8, 0x20a7, 0x0192, 0x00e1, 0x00ed, 0x00f3, 0x00fa,
    0x00c1, 0x00cd, 0x00d3, 0x00da, 0x00bf, 0x2310, 0x00ac, 0x00bd, 0x00bc,
    0x00a1, 0x00ab, 0x00bb, 0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561,
    0x2562, 0x2556, 0x2555, 0x2563, 0x2551, 0x2557, 0x255d, 0x255c, 0x255b,
    0x2510, 0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x255e, 0x255f,
    0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x2567, 0x2568,
    0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256b, 0x256a, 0x2518,
    0x250c, 0x2588, 0x2584, 0x258c, 0x2590, 0x2580, 0x03b1, 0x00df, 0x0393,
    0x03c0, 0x03a3, 0x03c3, 0x00b5, 0x03c4, 0x03a6, 0x0398, 0x03a9, 0x03b4,
    0x221e, 0x03c6, 0x03b5, 0x2229, 0x2261, 0x00b1, 0x2265, 0x2264, 0x2320,
    0x2321, 0x00f7, 0x2248, 0x00b0, 0x2219, 0x00b7, 0x221a, 0x207f, 0x00b2,
    0x25a0, 0x00a0
};

static const wchar_t cp862[128] = {
    0x05d0, 0x05d1, 0x05d2, 0x05d3, 0x05d4, 0x05d5, 0x05d6, 0x05d7, 0x05d8,
    0x05d9, 0x05da, 0x05db, 0x05dc, 0x05dd, 0x05de, 0x05df, 0x05e0, 0x05e1,
    0x05e2, 0x05e3, 0x05e4, 0x05e5, 0x05e6, 0x05e7, 0x05e8, 0x05e9, 0x05ea,
    0x00a2, 0x00a3, 0x00a5, 0x20a7, 0x0192, 0x00e1, 0x00ed, 0x00f3, 0x00fa,
    0x00f1, 0x00d1, 0x00aa, 0x00ba, 0x00bf, 0x2310, 0x00ac, 0x00bd, 0x00bc,
    0x00a1, 0x00ab, 0x00bb, 0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561,
    0x2562, 0x2556, 0x2555, 0x2563, 0x2551, 0x2557, 0x255d, 0x255c, 0x255b,
    0x2510, 0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x255e, 0x255f,
    0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x2567, 0x2568,
    0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256b, 0x256a, 0x2518,
    0x250c, 0x2588, 0x2584, 0x258c, 0x2590, 0x2580, 0x03b1, 0x00df, 0x0393,
    0x03c0, 0x03a3, 0x03c3, 0x00b5, 0x03c4, 0x03a6, 0x0398, 0x03a9, 0x03b4,
    0x221e, 0x03c6, 0x03b5, 0x2229, 0x2261, 0x00b1, 0x2265, 0x2264, 0x2320,
    0x2321, 0x00f7, 0x2248, 0x00b0, 0x2219, 0x00b7, 0x221a, 0x207f, 0x00b2,
    0x25a0, 0x00a0
};

static const wchar_t cp863[128] = {
    0x00c7, 0x00fc, 0x00e9, 0x00e2, 0x00c2, 0x00e0, 0x00b6, 0x00e7, 0x00ea,
    0x00eb, 0x00e8, 0x00ef, 0x00ee, 0x2017, 0x00c0, 0x00a7, 0x00c9, 0x00c8,
    0x00ca, 0x00f4, 0x00cb, 0x00cf, 0x00fb, 0x00f9, 0x00a4, 0x00d4, 0x00dc,
    0x00a2, 0x00a3, 0x00d9, 0x00db, 0x0192, 0x00a6, 0x00b4, 0x00f3, 0x00fa,
    0x00a8, 0x00b8, 0x00b3, 0x00af, 0x00ce, 0x2310, 0x00ac, 0x00bd, 0x00bc,
    0x00be, 0x00ab, 0x00bb, 0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561,
    0x2562, 0x2556, 0x2555, 0x2563, 0x2551, 0x2557, 0x255d, 0x255c, 0x255b,
    0x2510, 0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x255e, 0x255f,
    0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x2567, 0x2568,
    0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256b, 0x256a, 0x2518,
    0x250c, 0x2588, 0x2584, 0x258c, 0x2590, 0x2580, 0x03b1, 0x00df, 0x0393,
    0x03c0, 0x03a3, 0x03c3, 0x00b5, 0x03c4, 0x03a6, 0x0398, 0x03a9, 0x03b4,
    0x221e, 0x03c6, 0x03b5, 0x2229, 0x2261, 0x00b1, 0x2265, 0x2264, 0x2320,
    0x2321, 0x00f7, 0x2248, 0x00b0, 0x2219, 0x00b7, 0x221a, 0x207f, 0x00b2,
    0x25a0, 0x00a0
};

static const wchar_t cp864[128] = {
    0x00b0, 0x00b7, 0x2219, 0x221a, 0x2592, 0x2500, 0x2502, 0x253c, 0x2524,
    0x252c, 0x251c, 0x2534, 0x2510, 0x250c, 0x2514, 0x2518, 0x03b2, 0x221e,
    0x03c6, 0x00b1, 0x00bd, 0x00bc, 0x2248, 0x00ab, 0x00bb, 0xfef7, 0xfef8,
         0,      0, 0xfefb, 0xfefc,      0, 0x00a0, 0x00ad, 0xfe82, 0x00a3,
    0x00a4, 0xfe84,      0,      0, 0xfe8e, 0xfe8f, 0xfe95, 0xfe99, 0x060c,
    0xfe9d, 0xfea1, 0xfea5, 0x0660, 0x0661, 0x0662, 0x0663, 0x0664, 0x0665,
    0x0666, 0x0667, 0x0668, 0x0669, 0xfed1, 0x061b, 0xfeb1, 0xfeb5, 0xfeb9,
    0x061f, 0x00a2, 0xfe80, 0xfe81, 0xfe83, 0xfe85, 0xfeca, 0xfe8b, 0xfe8d,
    0xfe91, 0xfe93, 0xfe97, 0xfe9b, 0xfe9f, 0xfea3, 0xfea7, 0xfea9, 0xfeab,
    0xfead, 0xfeaf, 0xfeb3, 0xfeb7, 0xfebb, 0xfebf, 0xfec1, 0xfec5, 0xfecb,
    0xfecf, 0x00a6, 0x00ac, 0x00f7, 0x00d7, 0xfec9, 0x0640, 0xfed3, 0xfed7,
    0xfedb, 0xfedf, 0xfee3, 0xfee7, 0xfeeb, 0xfeed, 0xfeef, 0xfef3, 0xfebd,
    0xfecc, 0xfece, 0xfecd, 0xfee1, 0xfe7d, 0x0651, 0xfee5, 0xfee9, 0xfeec,
    0xfef0, 0xfef2, 0xfed0, 0xfed5, 0xfef5, 0xfef6, 0xfedd, 0xfed9, 0xfef1,
    0x25a0, 0x20
};

static const wchar_t cp865[128] = {
    0x00c7, 0x00fc, 0x00e9, 0x00e2, 0x00e4, 0x00e0, 0x00e5, 0x00e7, 0x00ea,
    0x00eb, 0x00e8, 0x00ef, 0x00ee, 0x00ec, 0x00c4, 0x00c5, 0x00c9, 0x00e6,
    0x00c6, 0x00f4, 0x00f6, 0x00f2, 0x00fb, 0x00f9, 0x00ff, 0x00d6, 0x00dc,
    0x00f8, 0x00a3, 0x00d8, 0x20a7, 0x0192, 0x00e1, 0x00ed, 0x00f3, 0x00fa,
    0x00f1, 0x00d1, 0x00aa, 0x00ba, 0x00bf, 0x2310, 0x00ac, 0x00bd, 0x00bc,
    0x00a1, 0x00ab, 0x00a4, 0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561,
    0x2562, 0x2556, 0x2555, 0x2563, 0x2551, 0x2557, 0x255d, 0x255c, 0x255b,
    0x2510, 0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x255e, 0x255f,
    0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x2567, 0x2568,
    0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256b, 0x256a, 0x2518,
    0x250c, 0x2588, 0x2584, 0x258c, 0x2590, 0x2580, 0x03b1, 0x00df, 0x0393,
    0x03c0, 0x03a3, 0x03c3, 0x00b5, 0x03c4, 0x03a6, 0x0398, 0x03a9, 0x03b4,
    0x221e, 0x03c6, 0x03b5, 0x2229, 0x2261, 0x00b1, 0x2265, 0x2264, 0x2320,
    0x2321, 0x00f7, 0x2248, 0x00b0, 0x2219, 0x00b7, 0x221a, 0x207f, 0x00b2,
    0x25a0, 0x00a0
};

static const wchar_t cp866[128] = {
    0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0416, 0x0417, 0x0418,
    0x0419, 0x041a, 0x041b, 0x041c, 0x041d, 0x041e, 0x041f, 0x0420, 0x0421,
    0x0422, 0x0423, 0x0424, 0x0425, 0x0426, 0x0427, 0x0428, 0x0429, 0x042a,
    0x042b, 0x042c, 0x042d, 0x042e, 0x042f, 0x0430, 0x0431, 0x0432, 0x0433,
    0x0434, 0x0435, 0x0436, 0x0437, 0x0438, 0x0439, 0x043a, 0x043b, 0x043c,
    0x043d, 0x043e, 0x043f, 0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561,
    0x2562, 0x2556, 0x2555, 0x2563, 0x2551, 0x2557, 0x255d, 0x255c, 0x255b,
    0x2510, 0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x255e, 0x255f,
    0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x2567, 0x2568,
    0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256b, 0x256a, 0x2518,
    0x250c, 0x2588, 0x2584, 0x258c, 0x2590, 0x2580, 0x0440, 0x0441, 0x0442,
    0x0443, 0x0444, 0x0445, 0x0446, 0x0447, 0x0448, 0x0449, 0x044a, 0x044b,
    0x044c, 0x044d, 0x044e, 0x044f, 0x0401, 0x0451, 0x0404, 0x0454, 0x0407,
    0x0457, 0x040e, 0x045e, 0x00b0, 0x2219, 0x00b7, 0x221a, 0x2116, 0x00a4,
    0x25a0, 0x00a0
};

static const wchar_t cp869[128] = {
         0,      0,      0,      0,      0,      0, 0x0386,      0, 0x00b7,
    0x00ac, 0x00a6, 0x2018, 0x2019, 0x0388, 0x2015, 0x0389, 0x038a, 0x03aa,
    0x038c,      0,      0, 0x038e, 0x03ab, 0x00a9, 0x038f, 0x00b2, 0x00b3,
    0x03ac, 0x00a3, 0x03ad, 0x03ae, 0x03af, 0x03ca, 0x0390, 0x03cc, 0x03cd,
    0x0391, 0x0392, 0x0393, 0x0394, 0x0395, 0x0396, 0x0397, 0x00bd, 0x0398,
    0x0399, 0x00ab, 0x00bb, 0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x039a,
    0x039b, 0x039c, 0x039d, 0x2563, 0x2551, 0x2557, 0x255d, 0x039e, 0x039f,
    0x2510, 0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x03a0, 0x03a1,
    0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x03a3, 0x03a4,
    0x03a5, 0x03a6, 0x03a7, 0x03a8, 0x03a9, 0x03b1, 0x03b2, 0x03b3, 0x2518,
    0x250c, 0x2588, 0x2584, 0x03b4, 0x03b5, 0x2580, 0x03b6, 0x03b7, 0x03b8,
    0x03b9, 0x03ba, 0x03bb, 0x03bc, 0x03bd, 0x03be, 0x03bf, 0x03c0, 0x03c1,
    0x03c3, 0x03c2, 0x03c4, 0x0384, 0x00ad, 0x00b1, 0x03c5, 0x03c6, 0x03c7,
    0x00a7, 0x03c8, 0x0385, 0x00b0, 0x00a8, 0x03c9, 0x03cb, 0x03b0, 0x03ce,
    0x25a0, 0x00a0
};

static const wchar_t cp874[128] = {
    0x20ac,      0,      0,      0,      0, 0x2026,      0,      0,      0,
         0,      0,      0,      0,      0,      0,      0,      0, 0x2018,
    0x2019, 0x201c, 0x201d, 0x2022, 0x2013, 0x2014,      0,      0,      0,
         0,      0,      0,      0,      0, 0x00a0, 0x0e01, 0x0e02, 0x0e03,
    0x0e04, 0x0e05, 0x0e06, 0x0e07, 0x0e08, 0x0e09, 0x0e0a, 0x0e0b, 0x0e0c,
    0x0e0d, 0x0e0e, 0x0e0f, 0x0e10, 0x0e11, 0x0e12, 0x0e13, 0x0e14, 0x0e15,
    0x0e16, 0x0e17, 0x0e18, 0x0e19, 0x0e1a, 0x0e1b, 0x0e1c, 0x0e1d, 0x0e1e,
    0x0e1f, 0x0e20, 0x0e21, 0x0e22, 0x0e23, 0x0e24, 0x0e25, 0x0e26, 0x0e27,
    0x0e28, 0x0e29, 0x0e2a, 0x0e2b, 0x0e2c, 0x0e2d, 0x0e2e, 0x0e2f, 0x0e30,
    0x0e31, 0x0e32, 0x0e33, 0x0e34, 0x0e35, 0x0e36, 0x0e37, 0x0e38, 0x0e39,
    0x0e3a,      0,      0,      0,      0, 0x0e3f, 0x0e40, 0x0e41, 0x0e42,
    0x0e43, 0x0e44, 0x0e45, 0x0e46, 0x0e47, 0x0e48, 0x0e49, 0x0e4a, 0x0e4b,
    0x0e4c, 0x0e4d, 0x0e4e, 0x0e4f, 0x0e50, 0x0e51, 0x0e52, 0x0e53, 0x0e54,
    0x0e55, 0x0e56, 0x0e57, 0x0e58, 0x0e59, 0x0e5a, 0x0e5b,      0,      0,
         0, 0x20
};

static int compcp(const void *a, const void *b) {
    return (int)*(uint32_t *)a - (int)*(uint32_t *)b;
}

static int compcp2(const void *a, const void *b) {
    return (int)(*(uint32_t *)a & ~0xffu) - (int)(*(uint32_t *)b & ~0xffu);
}

enum {
    GET_GLOBAL_CODE_PAGE_TABLE = 0x6601,
    IOCTL_GENERIC_CHARACTER_DEVICE_REQUEST = 0x440c,
    STDOUT_HANDLE = 1,
    DEVICE_CATEGORY_CON = 3,
    QUERY_SELECTED_CODE_PAGE = 0x6a,
    DISPLAY_SYS_GET_ACTIVE_CODE_PAGE = 0xad02,
    DPMI_ALLOC = 0x100,
    DPMI_FREE = 0x101,
    DPMI_INT = 0x300
};

#ifndef __DJGPP__
struct rmi_s {
    uint32_t edi, esi, ebp, res, ebx, edx, ecx, eax;
    uint16_t flags, es, ds, fs, gs, ip, cs, sp, ss;
};

static int dpmi_int(int nr, struct rmi_s *rmi) {
    union REGS regs;
    struct SREGS sregs;
    memset(&sregs, 0, sizeof sregs);
    memset(&regs, 0, sizeof regs);
    regs.w.ax = DPMI_INT;
    regs.h.bl = (uint8_t)nr;
    sregs.es = FP_SEG(rmi);
    regs.x.edi = FP_OFF(rmi);
    regs.x.cflag = 1;
    int386x(0x31, &regs, &regs, &sregs);
    return regs.x.cflag & 1;
}
#endif

static int get_codepage(void) {
#ifdef __DJGPP__
    __dpmi_regs regs;
    uint16_t w[2];
    memset(&regs, 0, sizeof regs);
    regs.x.ax = IOCTL_GENERIC_CHARACTER_DEVICE_REQUEST;
    regs.x.bx = STDOUT_HANDLE;
    regs.h.ch = DEVICE_CATEGORY_CON;
    regs.h.cl = QUERY_SELECTED_CODE_PAGE;
    regs.x.ds = (__tb >> 4) & 0xffff;
    regs.x.dx = __tb & 0xf;
    regs.x.flags = 1;
    memset(w, 0, sizeof w);
    dosmemput(w, sizeof w, __tb);
    if (__dpmi_int(0x21, &regs) == 0 && (regs.x.flags & 1) == 0) {
        dosmemget(__tb, sizeof w, w);
        if (w[1] != 0) {
            return w[1];
        }
    }
    regs.x.ax = DISPLAY_SYS_GET_ACTIVE_CODE_PAGE;
    regs.x.bx = 0;
    regs.x.flags = 1;
    if (__dpmi_int(0x2f, &regs) == 0 && (regs.x.flags & 1) == 0) {
        if (regs.x.bx != 0) {
            return regs.x.bx;
        }
    }
    regs.x.ax = GET_GLOBAL_CODE_PAGE_TABLE; 
    regs.x.bx = 0;
    regs.x.flags = 1;
    if (__dpmi_int(0x21, &regs) == 0 && (regs.x.flags & 1) == 0) {
        return regs.x.bx;
    }
#else
    struct rmi_s rmi;
    union REGS regs;
    int dcp = 0;
    memset(&rmi, 0, sizeof rmi);
    regs.w.ax = DPMI_ALLOC;
    regs.w.bx = 1;
    regs.w.cflag = 1;
    int386(0x31, &regs, &regs);
    if ((regs.w.cflag & 1) == 0) {
        uint16_t seg = regs.w.ax;
        uint16_t sel = regs.w.dx;
        uint16_t *w = (uint16_t *)(seg << 4);

        memset(w, 0, sizeof *w * 2);
        rmi.eax = IOCTL_GENERIC_CHARACTER_DEVICE_REQUEST;
        rmi.ebx = STDOUT_HANDLE;
        rmi.ecx = DEVICE_CATEGORY_CON * 0x100u + QUERY_SELECTED_CODE_PAGE;
        rmi.ds = seg;
        rmi.edx = 0;
        rmi.flags = 1;
        if (dpmi_int(0x21, &rmi)) {
            rmi.flags = 1;
        }
        if ((rmi.flags & 1) == 0) {
            dcp = w[1];
        }

        regs.w.ax = DPMI_FREE;
        regs.w.dx = sel;
        int386(0x31, &regs, &regs);
    }
    if (dcp != 0) {
        return dcp;
    }

    rmi.eax = DISPLAY_SYS_GET_ACTIVE_CODE_PAGE;
    rmi.ebx = 0;
    rmi.flags = 1;
    if (dpmi_int(0x2f, &rmi)) {
        rmi.flags = 1;
    }
    if ((rmi.flags & 1) == 0) {
        if ((uint16_t)rmi.ebx != 0) {
            return (uint16_t)rmi.ebx;
        }
    }

    rmi.eax = GET_GLOBAL_CODE_PAGE_TABLE;
    rmi.ebx = 0;
    rmi.flags = 1;
    if (dpmi_int(0x21, &rmi)) {
        rmi.flags = 1;
    }
    if ((rmi.flags & 1) == 0) {
        return (uint16_t)rmi.ebx;
    }
#endif
    return 0;
}

static void set_cp(void) {
    unsigned int i;
    int dcp = get_codepage();
    switch (dcp) {
    case 737: cp = cp737; break;
    case 775: cp = cp775; break;
    case 850: cp = cp850; break;
    case 852: cp = cp852; break;
    case 855: cp = cp855; break;
    case 857: cp = cp857; break;
    case 860: cp = cp860; break;
    case 861: cp = cp861; break;
    case 862: cp = cp862; break;
    case 863: cp = cp863; break;
    case 864: cp = cp864; break;
    case 865: cp = cp865; break;
    case 866: cp = cp866; break;
    case 869: cp = cp869; break;
    case 874: cp = cp874; break;
    default:
    case 437: cp = cp437; break;
    }
    for (i = 0; i < 128; i++) {
        revcp[i] = (cp[i] * 0x100u) | i | 0x80;
    }
    qsort(revcp, lenof(revcp), sizeof *revcp, compcp);
}

size_t mbrtowc(wchar_t *wc, const char *s, size_t n, mbstate_t *UNUSED(ps)) {
    uint8_t ch;
    wchar_t w;
    if (n == 0) return (size_t)-2;
    ch = (uint8_t)*s;
    if (ch < 0x80) {
        if (!ch) return 0;
        *wc = ch;
        return 1;
    }
    if (cp == NULL) set_cp();
    *wc = w = cp[ch - 0x80];
    if (w != 0) return 1;
    errno = EILSEQ;
    return (size_t)-1;
}

size_t wcrtomb(char *s, wchar_t wc, mbstate_t *UNUSED(ps)) {
    uint32_t *ch, c;
    if (wc < 0x80) {
        *s = (char)wc;
        return 1;
    }
    if (cp == NULL) set_cp();
    c = wc * 0x100u;
    ch = (uint32_t *)bsearch(&c, revcp, lenof(revcp), sizeof *revcp, compcp2);
    if (ch != NULL) {
        *s = (char)*ch;
        return 1;
    }
    errno = EILSEQ;
    return (size_t)-1;
}
#elif defined __GNUC__ || defined _MSC_VER || defined __WATCOMC__
#elif __STDC_VERSION__ >= 199901L && !defined __VBCC__
#else
#include <errno.h>
#include "attributes.h"

size_t mbrtowc(wchar_t *wc, const char *s, size_t n, mbstate_t *UNUSED(ps)) {
    uint8_t ch;
    if (n == 0) return (size_t)-2;
    ch = (uint8_t)*s;
    if (ch < 0x80) {
        if (!ch) return 0;
        *wc = ch;
        return 1;
    }
    if (ch < 0xa0) {
#ifdef EILSEQ
        errno = EILSEQ;
#endif
        return (size_t)-1;
    }
    *wc = ch;
    return 1;
}

size_t wcrtomb(char *s, wchar_t wc, mbstate_t *UNUSED(ps)) {
    if (wc < 0x80 || (wc >= 0xa0 && wc <= 0xff)) {
        *s = (char)wc;
        return 1;
    }
#ifdef EILSEQ
    errno = EILSEQ;
#endif
    return (size_t)-1;
}
#endif

struct w16_s {
    uint16_t start;
    uint16_t end;
};

static const struct w16_s zw16a[] = {
    {0x0300, 0x036F}, {0x0483, 0x0489}, {0x0591, 0x05BD}, {0x05BF, 0x05BF},
    {0x05C1, 0x05C2}, {0x05C4, 0x05C5}, {0x05C7, 0x05C7}, {0x0600, 0x0605},
    {0x0610, 0x061A}, {0x061C, 0x061C}, {0x064B, 0x065F}, {0x0670, 0x0670},
    {0x06D6, 0x06DD}, {0x06DF, 0x06E4}, {0x06E7, 0x06E8}, {0x06EA, 0x06ED},
    {0x070F, 0x070F}, {0x0711, 0x0711}, {0x0730, 0x074A}, {0x07A6, 0x07B0},
    {0x07EB, 0x07F3}, {0x07FD, 0x07FD}, {0x0816, 0x0819}, {0x081B, 0x0823},
    {0x0825, 0x0827}, {0x0829, 0x082D}, {0x0859, 0x085B}, {0x08D3, 0x0902},
    {0x093A, 0x093A}, {0x093C, 0x093C}, {0x0941, 0x0948}, {0x094D, 0x094D},
    {0x0951, 0x0957}, {0x0962, 0x0963}, {0x0981, 0x0981}, {0x09BC, 0x09BC},
    {0x09C1, 0x09C4}, {0x09CD, 0x09CD}, {0x09E2, 0x09E3}, {0x09FE, 0x09FE},
    {0x0A01, 0x0A02}, {0x0A3C, 0x0A3C}, {0x0A41, 0x0A42}, {0x0A47, 0x0A48},
    {0x0A4B, 0x0A4D}, {0x0A51, 0x0A51}, {0x0A70, 0x0A71}, {0x0A75, 0x0A75},
    {0x0A81, 0x0A82}, {0x0ABC, 0x0ABC}, {0x0AC1, 0x0AC5}, {0x0AC7, 0x0AC8},
    {0x0ACD, 0x0ACD}, {0x0AE2, 0x0AE3}, {0x0AFA, 0x0AFF}, {0x0B01, 0x0B01},
    {0x0B3C, 0x0B3C}, {0x0B3F, 0x0B3F}, {0x0B41, 0x0B44}, {0x0B4D, 0x0B4D},
    {0x0B55, 0x0B56}, {0x0B62, 0x0B63}, {0x0B82, 0x0B82}, {0x0BC0, 0x0BC0},
    {0x0BCD, 0x0BCD}, {0x0C00, 0x0C00}, {0x0C04, 0x0C04}, {0x0C3E, 0x0C40},
    {0x0C46, 0x0C48}, {0x0C4A, 0x0C4D}, {0x0C55, 0x0C56}, {0x0C62, 0x0C63},
    {0x0C81, 0x0C81}, {0x0CBC, 0x0CBC}, {0x0CBF, 0x0CBF}, {0x0CC6, 0x0CC6},
    {0x0CCC, 0x0CCD}, {0x0CE2, 0x0CE3}, {0x0D00, 0x0D01}, {0x0D3B, 0x0D3C},
    {0x0D41, 0x0D44}, {0x0D4D, 0x0D4D}, {0x0D62, 0x0D63}, {0x0D81, 0x0D81},
    {0x0DCA, 0x0DCA}, {0x0DD2, 0x0DD4}, {0x0DD6, 0x0DD6}, {0x0E31, 0x0E31},
    {0x0E34, 0x0E3A}, {0x0E47, 0x0E4E}, {0x0EB1, 0x0EB1}, {0x0EB4, 0x0EBC},
    {0x0EC8, 0x0ECD}, {0x0F18, 0x0F19}, {0x0F35, 0x0F35}, {0x0F37, 0x0F37},
    {0x0F39, 0x0F39}, {0x0F71, 0x0F7E}, {0x0F80, 0x0F84}, {0x0F86, 0x0F87},
    {0x0F8D, 0x0F97}, {0x0F99, 0x0FBC}, {0x0FC6, 0x0FC6}, {0x102D, 0x1030},
    {0x1032, 0x1037}, {0x1039, 0x103A}, {0x103D, 0x103E}, {0x1058, 0x1059},
    {0x105E, 0x1060}, {0x1071, 0x1074}, {0x1082, 0x1082}, {0x1085, 0x1086},
    {0x108D, 0x108D}, {0x109D, 0x109D}, {0x1160, 0x11FF}, {0x135D, 0x135F},
    {0x1712, 0x1714}, {0x1732, 0x1734}, {0x1752, 0x1753}, {0x1772, 0x1773},
    {0x17B4, 0x17B5}, {0x17B7, 0x17BD}, {0x17C6, 0x17C6}, {0x17C9, 0x17D3},
    {0x17DD, 0x17DD}, {0x180B, 0x180E}, {0x1885, 0x1886}, {0x18A9, 0x18A9},
    {0x1920, 0x1922}, {0x1927, 0x1928}, {0x1932, 0x1932}, {0x1939, 0x193B},
    {0x1A17, 0x1A18}, {0x1A1B, 0x1A1B}, {0x1A56, 0x1A56}, {0x1A58, 0x1A5E},
    {0x1A60, 0x1A60}, {0x1A62, 0x1A62}, {0x1A65, 0x1A6C}, {0x1A73, 0x1A7C},
    {0x1A7F, 0x1A7F}, {0x1AB0, 0x1AC0}, {0x1B00, 0x1B03}, {0x1B34, 0x1B34},
    {0x1B36, 0x1B3A}, {0x1B3C, 0x1B3C}, {0x1B42, 0x1B42}, {0x1B6B, 0x1B73},
    {0x1B80, 0x1B81}, {0x1BA2, 0x1BA5}, {0x1BA8, 0x1BA9}, {0x1BAB, 0x1BAD},
    {0x1BE6, 0x1BE6}, {0x1BE8, 0x1BE9}, {0x1BED, 0x1BED}, {0x1BEF, 0x1BF1},
    {0x1C2C, 0x1C33}, {0x1C36, 0x1C37}, {0x1CD0, 0x1CD2}, {0x1CD4, 0x1CE0},
    {0x1CE2, 0x1CE8}, {0x1CED, 0x1CED}, {0x1CF4, 0x1CF4}, {0x1CF8, 0x1CF9},
    {0x1DC0, 0x1DF9}, {0x1DFB, 0x1DFF}, {0x200B, 0x200F}, {0x202A, 0x202E},
    {0x2060, 0x2064}, {0x2066, 0x206F}, {0x20D0, 0x20F0}, {0x2CEF, 0x2CF1},
    {0x2D7F, 0x2D7F}, {0x2DE0, 0x2DFF}, {0x302A, 0x302D}, {0x3099, 0x309A},
    {0xA66F, 0xA672}, {0xA674, 0xA67D}, {0xA69E, 0xA69F}, {0xA6F0, 0xA6F1},
    {0xA802, 0xA802}, {0xA806, 0xA806}, {0xA80B, 0xA80B}, {0xA825, 0xA826},
    {0xA82C, 0xA82C}, {0xA8C4, 0xA8C5}, {0xA8E0, 0xA8F1}, {0xA8FF, 0xA8FF},
    {0xA926, 0xA92D}, {0xA947, 0xA951}, {0xA980, 0xA982}, {0xA9B3, 0xA9B3},
    {0xA9B6, 0xA9B9}, {0xA9BC, 0xA9BD}, {0xA9E5, 0xA9E5}, {0xAA29, 0xAA2E},
    {0xAA31, 0xAA32}, {0xAA35, 0xAA36}, {0xAA43, 0xAA43}, {0xAA4C, 0xAA4C},
    {0xAA7C, 0xAA7C}, {0xAAB0, 0xAAB0}, {0xAAB2, 0xAAB4}, {0xAAB7, 0xAAB8},
    {0xAABE, 0xAABF}, {0xAAC1, 0xAAC1}, {0xAAEC, 0xAAED}, {0xAAF6, 0xAAF6},
    {0xABE5, 0xABE5}, {0xABE8, 0xABE8}, {0xABED, 0xABED}, {0xFB1E, 0xFB1E},
    {0xFE00, 0xFE0F}, {0xFE20, 0xFE2F}, {0xFEFF, 0xFEFF}, {0xFFF9, 0xFFFB}
};

static const struct w16_s dw16a[] = {
    {0x1100, 0x115F}, {0x231A, 0x231B}, {0x2329, 0x232A}, {0x23E9, 0x23EC},
    {0x23F0, 0x23F0}, {0x23F3, 0x23F3}, {0x25FD, 0x25FE}, {0x2614, 0x2615},
    {0x2648, 0x2653}, {0x267F, 0x267F}, {0x2693, 0x2693}, {0x26A1, 0x26A1},
    {0x26AA, 0x26AB}, {0x26BD, 0x26BE}, {0x26C4, 0x26C5}, {0x26CE, 0x26CE},
    {0x26D4, 0x26D4}, {0x26EA, 0x26EA}, {0x26F2, 0x26F3}, {0x26F5, 0x26F5},
    {0x26FA, 0x26FA}, {0x26FD, 0x26FD}, {0x2705, 0x2705}, {0x270A, 0x270B},
    {0x2728, 0x2728}, {0x274C, 0x274C}, {0x274E, 0x274E}, {0x2753, 0x2755},
    {0x2757, 0x2757}, {0x2795, 0x2797}, {0x27B0, 0x27B0}, {0x27BF, 0x27BF},
    {0x2B1B, 0x2B1C}, {0x2B50, 0x2B50}, {0x2B55, 0x2B55}, {0x2E80, 0x2E99},
    {0x2E9B, 0x2EF3}, {0x2F00, 0x2FD5}, {0x2FF0, 0x2FFB}, {0x3000, 0x303E},
    {0x3041, 0x3096}, {0x3099, 0x30FF}, {0x3105, 0x312F}, {0x3131, 0x318E},
    {0x3190, 0x31E3}, {0x31F0, 0x321E}, {0x3220, 0x3247}, {0x3250, 0x4DBF},
    {0x4E00, 0xA48C}, {0xA490, 0xA4C6}, {0xA960, 0xA97C}, {0xAC00, 0xD7A3},
    {0xF900, 0xFAFF}, {0xFE10, 0xFE19}, {0xFE30, 0xFE52}, {0xFE54, 0xFE66},
    {0xFE68, 0xFE6B}, {0xFF01, 0xFF60}, {0xFFE0, 0xFFE6}
};

static const struct w16_s zw16b[] = {
    {0x01FD, 0x01FD}, {0x02E0, 0x02E0}, {0x0376, 0x037A}, {0x0A01, 0x0A03},
    {0x0A05, 0x0A06}, {0x0A0C, 0x0A0F}, {0x0A38, 0x0A3A}, {0x0A3F, 0x0A3F},
    {0x0AE5, 0x0AE6}, {0x0D24, 0x0D27}, {0x0EAB, 0x0EAC}, {0x0F46, 0x0F50},
    {0x1001, 0x1001}, {0x1038, 0x1046}, {0x107F, 0x1081}, {0x10B3, 0x10B6},
    {0x10B9, 0x10BA}, {0x10BD, 0x10BD}, {0x10CD, 0x10CD}, {0x1100, 0x1102},
    {0x1127, 0x112B}, {0x112D, 0x1134}, {0x1173, 0x1173}, {0x1180, 0x1181},
    {0x11B6, 0x11BE}, {0x11C9, 0x11CC}, {0x11CF, 0x11CF}, {0x122F, 0x1231},
    {0x1234, 0x1234}, {0x1236, 0x1237}, {0x123E, 0x123E}, {0x12DF, 0x12DF},
    {0x12E3, 0x12EA}, {0x1300, 0x1301}, {0x133B, 0x133C}, {0x1340, 0x1340},
    {0x1366, 0x136C}, {0x1370, 0x1374}, {0x1438, 0x143F}, {0x1442, 0x1444},
    {0x1446, 0x1446}, {0x145E, 0x145E}, {0x14B3, 0x14B8}, {0x14BA, 0x14BA},
    {0x14BF, 0x14C0}, {0x14C2, 0x14C3}, {0x15B2, 0x15B5}, {0x15BC, 0x15BD},
    {0x15BF, 0x15C0}, {0x15DC, 0x15DD}, {0x1633, 0x163A}, {0x163D, 0x163D},
    {0x163F, 0x1640}, {0x16AB, 0x16AB}, {0x16AD, 0x16AD}, {0x16B0, 0x16B5},
    {0x16B7, 0x16B7}, {0x171D, 0x171F}, {0x1722, 0x1725}, {0x1727, 0x172B},
    {0x182F, 0x1837}, {0x1839, 0x183A}, {0x193B, 0x193C}, {0x193E, 0x193E},
    {0x1943, 0x1943}, {0x19D4, 0x19D7}, {0x19DA, 0x19DB}, {0x19E0, 0x19E0},
    {0x1A01, 0x1A0A}, {0x1A33, 0x1A38}, {0x1A3B, 0x1A3E}, {0x1A47, 0x1A47},
    {0x1A51, 0x1A56}, {0x1A59, 0x1A5B}, {0x1A8A, 0x1A96}, {0x1A98, 0x1A99},
    {0x1C30, 0x1C36}, {0x1C38, 0x1C3D}, {0x1C3F, 0x1C3F}, {0x1C92, 0x1CA7},
    {0x1CAA, 0x1CB0}, {0x1CB2, 0x1CB3}, {0x1CB5, 0x1CB6}, {0x1D31, 0x1D36},
    {0x1D3A, 0x1D3A}, {0x1D3C, 0x1D3D}, {0x1D3F, 0x1D45}, {0x1D47, 0x1D47},
    {0x1D90, 0x1D91}, {0x1D95, 0x1D95}, {0x1D97, 0x1D97}, {0x1EF3, 0x1EF4},
    {0x3430, 0x3438}, {0x6AF0, 0x6AF4}, {0x6B30, 0x6B36}, {0x6F4F, 0x6F4F},
    {0x6F8F, 0x6F92}, {0x6FE4, 0x6FE4}, {0xBC9D, 0xBC9E}, {0xBCA0, 0xBCA3},
    {0xD167, 0xD169}, {0xD173, 0xD182}, {0xD185, 0xD18B}, {0xD1AA, 0xD1AD},
    {0xD242, 0xD244}, {0xDA00, 0xDA36}, {0xDA3B, 0xDA6C}, {0xDA75, 0xDA75},
    {0xDA84, 0xDA84}, {0xDA9B, 0xDA9F}, {0xDAA1, 0xDAAF}, {0xE000, 0xE006},
    {0xE008, 0xE018}, {0xE01B, 0xE021}, {0xE023, 0xE024}, {0xE026, 0xE02A},
    {0xE130, 0xE136}, {0xE2EC, 0xE2EF}, {0xE8D0, 0xE8D6}, {0xE944, 0xE94A}
};

static const struct w16_s dw16b[] = {
    {0x6FE0, 0x6FE4}, {0x6FF0, 0x6FF1}, {0x7000, 0x87F7}, {0x8800, 0x8CD5},
    {0x8D00, 0x8D08}, {0xB000, 0xB11E}, {0xB150, 0xB152}, {0xB164, 0xB167},
    {0xB170, 0xB2FB}, {0xF004, 0xF004}, {0xF0CF, 0xF0CF}, {0xF18E, 0xF18E},
    {0xF191, 0xF19A}, {0xF200, 0xF202}, {0xF210, 0xF23B}, {0xF240, 0xF248},
    {0xF250, 0xF251}, {0xF260, 0xF265}, {0xF300, 0xF320}, {0xF32D, 0xF335},
    {0xF337, 0xF37C}, {0xF37E, 0xF393}, {0xF3A0, 0xF3CA}, {0xF3CF, 0xF3D3},
    {0xF3E0, 0xF3F0}, {0xF3F4, 0xF3F4}, {0xF3F8, 0xF43E}, {0xF440, 0xF440},
    {0xF442, 0xF4FC}, {0xF4FF, 0xF53D}, {0xF54B, 0xF54E}, {0xF550, 0xF567},
    {0xF57A, 0xF57A}, {0xF595, 0xF596}, {0xF5A4, 0xF5A4}, {0xF5FB, 0xF64F},
    {0xF680, 0xF6C5}, {0xF6CC, 0xF6CC}, {0xF6D0, 0xF6D2}, {0xF6D5, 0xF6D7},
    {0xF6EB, 0xF6EC}, {0xF6F4, 0xF6FC}, {0xF7E0, 0xF7EB}, {0xF90C, 0xF93A},
    {0xF93C, 0xF945}, {0xF947, 0xF978}, {0xF97A, 0xF9CB}, {0xF9CD, 0xF9FF},
    {0xFA70, 0xFA74}, {0xFA78, 0xFA7A}, {0xFA80, 0xFA86}, {0xFA90, 0xFAA8},
    {0xFAB0, 0xFAB6}, {0xFAC0, 0xFAC2}, {0xFAD0, 0xFAD6}
};

static int compw16(const void *aa, const void *bb) {
    unichar_t key = *(const unichar_t *)aa;
    const struct w16_s *b = (const struct w16_s *)bb;

    if (key > b->end) return 1;
    if (key < b->start) return -1;
    return 0;
}

int wcwidth_v13(unichar_t ch) {
    if (ch < 0x300) return 1;

    if (ch < 0x10000) {
        if (bsearch(&ch, zw16a, lenof(zw16a), sizeof *zw16a, compw16) != NULL) return 0;
        if (ch < 0x1100) return 1;
        if (bsearch(&ch, dw16a, lenof(dw16a), sizeof *dw16a, compw16) != NULL) return 2;
        return 1;
    }
    if (ch < 0x20000) {
        ch &= 0xffff;
        if (bsearch(&ch, zw16b, lenof(zw16b), sizeof *zw16b, compw16) != NULL) return 0;
        if (bsearch(&ch, dw16b, lenof(dw16b), sizeof *dw16b, compw16) != NULL) return 2;
        return 1;
    }
    if (ch == 0xE0001 || (ch >= 0xE0020 && 0xE007F >= ch) || (ch >= 0xE0100 && 0xE01EF >= ch)) {
        return 0;
    }
    if ((ch >= 0x20000 && 0x2FFFD >= ch) || (0x30000 >= ch && 0x3FFFD >= ch)) {
        return 2;
    }
    return 1;
}

static const struct w16_s pr16a[349] = {
    {0x0020, 0x007E}, {0x00A1, 0x00AC}, {0x00AE, 0x0377}, {0x037A, 0x037F},
    {0x0384, 0x038A}, {0x038C, 0x038C}, {0x038E, 0x03A1}, {0x03A3, 0x052F},
    {0x0531, 0x0556}, {0x0559, 0x058A}, {0x058D, 0x058F}, {0x0591, 0x05C7},
    {0x05D0, 0x05EA}, {0x05EF, 0x05F4}, {0x0606, 0x061B}, {0x061E, 0x06DC},
    {0x06DE, 0x070D}, {0x0710, 0x074A}, {0x074D, 0x07B1}, {0x07C0, 0x07FA},
    {0x07FD, 0x082D}, {0x0830, 0x083E}, {0x0840, 0x085B}, {0x085E, 0x085E},
    {0x0860, 0x086A}, {0x08A0, 0x08B4}, {0x08B6, 0x08C7}, {0x08D3, 0x08E1},
    {0x08E3, 0x0983}, {0x0985, 0x098C}, {0x098F, 0x0990}, {0x0993, 0x09A8},
    {0x09AA, 0x09B0}, {0x09B2, 0x09B2}, {0x09B6, 0x09B9}, {0x09BC, 0x09C4},
    {0x09C7, 0x09C8}, {0x09CB, 0x09CE}, {0x09D7, 0x09D7}, {0x09DC, 0x09DD},
    {0x09DF, 0x09E3}, {0x09E6, 0x09FE}, {0x0A01, 0x0A03}, {0x0A05, 0x0A0A},
    {0x0A0F, 0x0A10}, {0x0A13, 0x0A28}, {0x0A2A, 0x0A30}, {0x0A32, 0x0A33},
    {0x0A35, 0x0A36}, {0x0A38, 0x0A39}, {0x0A3C, 0x0A3C}, {0x0A3E, 0x0A42},
    {0x0A47, 0x0A48}, {0x0A4B, 0x0A4D}, {0x0A51, 0x0A51}, {0x0A59, 0x0A5C},
    {0x0A5E, 0x0A5E}, {0x0A66, 0x0A76}, {0x0A81, 0x0A83}, {0x0A85, 0x0A8D},
    {0x0A8F, 0x0A91}, {0x0A93, 0x0AA8}, {0x0AAA, 0x0AB0}, {0x0AB2, 0x0AB3},
    {0x0AB5, 0x0AB9}, {0x0ABC, 0x0AC5}, {0x0AC7, 0x0AC9}, {0x0ACB, 0x0ACD},
    {0x0AD0, 0x0AD0}, {0x0AE0, 0x0AE3}, {0x0AE6, 0x0AF1}, {0x0AF9, 0x0AFF},
    {0x0B01, 0x0B03}, {0x0B05, 0x0B0C}, {0x0B0F, 0x0B10}, {0x0B13, 0x0B28},
    {0x0B2A, 0x0B30}, {0x0B32, 0x0B33}, {0x0B35, 0x0B39}, {0x0B3C, 0x0B44},
    {0x0B47, 0x0B48}, {0x0B4B, 0x0B4D}, {0x0B55, 0x0B57}, {0x0B5C, 0x0B5D},
    {0x0B5F, 0x0B63}, {0x0B66, 0x0B77}, {0x0B82, 0x0B83}, {0x0B85, 0x0B8A},
    {0x0B8E, 0x0B90}, {0x0B92, 0x0B95}, {0x0B99, 0x0B9A}, {0x0B9C, 0x0B9C},
    {0x0B9E, 0x0B9F}, {0x0BA3, 0x0BA4}, {0x0BA8, 0x0BAA}, {0x0BAE, 0x0BB9},
    {0x0BBE, 0x0BC2}, {0x0BC6, 0x0BC8}, {0x0BCA, 0x0BCD}, {0x0BD0, 0x0BD0},
    {0x0BD7, 0x0BD7}, {0x0BE6, 0x0BFA}, {0x0C00, 0x0C0C}, {0x0C0E, 0x0C10},
    {0x0C12, 0x0C28}, {0x0C2A, 0x0C39}, {0x0C3D, 0x0C44}, {0x0C46, 0x0C48},
    {0x0C4A, 0x0C4D}, {0x0C55, 0x0C56}, {0x0C58, 0x0C5A}, {0x0C60, 0x0C63},
    {0x0C66, 0x0C6F}, {0x0C77, 0x0C8C}, {0x0C8E, 0x0C90}, {0x0C92, 0x0CA8},
    {0x0CAA, 0x0CB3}, {0x0CB5, 0x0CB9}, {0x0CBC, 0x0CC4}, {0x0CC6, 0x0CC8},
    {0x0CCA, 0x0CCD}, {0x0CD5, 0x0CD6}, {0x0CDE, 0x0CDE}, {0x0CE0, 0x0CE3},
    {0x0CE6, 0x0CEF}, {0x0CF1, 0x0CF2}, {0x0D00, 0x0D0C}, {0x0D0E, 0x0D10},
    {0x0D12, 0x0D44}, {0x0D46, 0x0D48}, {0x0D4A, 0x0D4F}, {0x0D54, 0x0D63},
    {0x0D66, 0x0D7F}, {0x0D81, 0x0D83}, {0x0D85, 0x0D96}, {0x0D9A, 0x0DB1},
    {0x0DB3, 0x0DBB}, {0x0DBD, 0x0DBD}, {0x0DC0, 0x0DC6}, {0x0DCA, 0x0DCA},
    {0x0DCF, 0x0DD4}, {0x0DD6, 0x0DD6}, {0x0DD8, 0x0DDF}, {0x0DE6, 0x0DEF},
    {0x0DF2, 0x0DF4}, {0x0E01, 0x0E3A}, {0x0E3F, 0x0E5B}, {0x0E81, 0x0E82},
    {0x0E84, 0x0E84}, {0x0E86, 0x0E8A}, {0x0E8C, 0x0EA3}, {0x0EA5, 0x0EA5},
    {0x0EA7, 0x0EBD}, {0x0EC0, 0x0EC4}, {0x0EC6, 0x0EC6}, {0x0EC8, 0x0ECD},
    {0x0ED0, 0x0ED9}, {0x0EDC, 0x0EDF}, {0x0F00, 0x0F47}, {0x0F49, 0x0F6C},
    {0x0F71, 0x0F97}, {0x0F99, 0x0FBC}, {0x0FBE, 0x0FCC}, {0x0FCE, 0x0FDA},
    {0x1000, 0x10C5}, {0x10C7, 0x10C7}, {0x10CD, 0x10CD}, {0x10D0, 0x1248},
    {0x124A, 0x124D}, {0x1250, 0x1256}, {0x1258, 0x1258}, {0x125A, 0x125D},
    {0x1260, 0x1288}, {0x128A, 0x128D}, {0x1290, 0x12B0}, {0x12B2, 0x12B5},
    {0x12B8, 0x12BE}, {0x12C0, 0x12C0}, {0x12C2, 0x12C5}, {0x12C8, 0x12D6},
    {0x12D8, 0x1310}, {0x1312, 0x1315}, {0x1318, 0x135A}, {0x135D, 0x137C},
    {0x1380, 0x1399}, {0x13A0, 0x13F5}, {0x13F8, 0x13FD}, {0x1400, 0x167F},
    {0x1681, 0x169C}, {0x16A0, 0x16F8}, {0x1700, 0x170C}, {0x170E, 0x1714},
    {0x1720, 0x1736}, {0x1740, 0x1753}, {0x1760, 0x176C}, {0x176E, 0x1770},
    {0x1772, 0x1773}, {0x1780, 0x17DD}, {0x17E0, 0x17E9}, {0x17F0, 0x17F9},
    {0x1800, 0x180D}, {0x1810, 0x1819}, {0x1820, 0x1878}, {0x1880, 0x18AA},
    {0x18B0, 0x18F5}, {0x1900, 0x191E}, {0x1920, 0x192B}, {0x1930, 0x193B},
    {0x1940, 0x1940}, {0x1944, 0x196D}, {0x1970, 0x1974}, {0x1980, 0x19AB},
    {0x19B0, 0x19C9}, {0x19D0, 0x19DA}, {0x19DE, 0x1A1B}, {0x1A1E, 0x1A5E},
    {0x1A60, 0x1A7C}, {0x1A7F, 0x1A89}, {0x1A90, 0x1A99}, {0x1AA0, 0x1AAD},
    {0x1AB0, 0x1AC0}, {0x1B00, 0x1B4B}, {0x1B50, 0x1B7C}, {0x1B80, 0x1BF3},
    {0x1BFC, 0x1C37}, {0x1C3B, 0x1C49}, {0x1C4D, 0x1C88}, {0x1C90, 0x1CBA},
    {0x1CBD, 0x1CC7}, {0x1CD0, 0x1CFA}, {0x1D00, 0x1DF9}, {0x1DFB, 0x1F15},
    {0x1F18, 0x1F1D}, {0x1F20, 0x1F45}, {0x1F48, 0x1F4D}, {0x1F50, 0x1F57},
    {0x1F59, 0x1F59}, {0x1F5B, 0x1F5B}, {0x1F5D, 0x1F5D}, {0x1F5F, 0x1F7D},
    {0x1F80, 0x1FB4}, {0x1FB6, 0x1FC4}, {0x1FC6, 0x1FD3}, {0x1FD6, 0x1FDB},
    {0x1FDD, 0x1FEF}, {0x1FF2, 0x1FF4}, {0x1FF6, 0x1FFE}, {0x2010, 0x2027},
    {0x2030, 0x205E}, {0x2070, 0x2071}, {0x2074, 0x208E}, {0x2090, 0x209C},
    {0x20A0, 0x20BF}, {0x20D0, 0x20F0}, {0x2100, 0x218B}, {0x2190, 0x2426},
    {0x2440, 0x244A}, {0x2460, 0x2B73}, {0x2B76, 0x2B95}, {0x2B97, 0x2C2E},
    {0x2C30, 0x2C5E}, {0x2C60, 0x2CF3}, {0x2CF9, 0x2D25}, {0x2D27, 0x2D27},
    {0x2D2D, 0x2D2D}, {0x2D30, 0x2D67}, {0x2D6F, 0x2D70}, {0x2D7F, 0x2D96},
    {0x2DA0, 0x2DA6}, {0x2DA8, 0x2DAE}, {0x2DB0, 0x2DB6}, {0x2DB8, 0x2DBE},
    {0x2DC0, 0x2DC6}, {0x2DC8, 0x2DCE}, {0x2DD0, 0x2DD6}, {0x2DD8, 0x2DDE},
    {0x2DE0, 0x2E52}, {0x2E80, 0x2E99}, {0x2E9B, 0x2EF3}, {0x2F00, 0x2FD5},
    {0x2FF0, 0x2FFB}, {0x3001, 0x303F}, {0x3041, 0x3096}, {0x3099, 0x30FF},
    {0x3105, 0x312F}, {0x3131, 0x318E}, {0x3190, 0x31E3}, {0x31F0, 0x321E},
    {0x3220, 0x9FFC}, {0xA000, 0xA48C}, {0xA490, 0xA4C6}, {0xA4D0, 0xA62B},
    {0xA640, 0xA6F7}, {0xA700, 0xA7BF}, {0xA7C2, 0xA7CA}, {0xA7F5, 0xA82C},
    {0xA830, 0xA839}, {0xA840, 0xA877}, {0xA880, 0xA8C5}, {0xA8CE, 0xA8D9},
    {0xA8E0, 0xA953}, {0xA95F, 0xA97C}, {0xA980, 0xA9CD}, {0xA9CF, 0xA9D9},
    {0xA9DE, 0xA9FE}, {0xAA00, 0xAA36}, {0xAA40, 0xAA4D}, {0xAA50, 0xAA59},
    {0xAA5C, 0xAAC2}, {0xAADB, 0xAAF6}, {0xAB01, 0xAB06}, {0xAB09, 0xAB0E},
    {0xAB11, 0xAB16}, {0xAB20, 0xAB26}, {0xAB28, 0xAB2E}, {0xAB30, 0xAB6B},
    {0xAB70, 0xABED}, {0xABF0, 0xABF9}, {0xAC00, 0xD7A3}, {0xD7B0, 0xD7C6},
    {0xD7CB, 0xD7FB}, {0xF900, 0xFA6D}, {0xFA70, 0xFAD9}, {0xFB00, 0xFB06},
    {0xFB13, 0xFB17}, {0xFB1D, 0xFB36}, {0xFB38, 0xFB3C}, {0xFB3E, 0xFB3E},
    {0xFB40, 0xFB41}, {0xFB43, 0xFB44}, {0xFB46, 0xFBC1}, {0xFBD3, 0xFD3F},
    {0xFD50, 0xFD8F}, {0xFD92, 0xFDC7}, {0xFDF0, 0xFDFD}, {0xFE00, 0xFE19},
    {0xFE20, 0xFE52}, {0xFE54, 0xFE66}, {0xFE68, 0xFE6B}, {0xFE70, 0xFE74},
    {0xFE76, 0xFEFC}, {0xFF01, 0xFFBE}, {0xFFC2, 0xFFC7}, {0xFFCA, 0xFFCF},
    {0xFFD2, 0xFFD7}, {0xFFDA, 0xFFDC}, {0xFFE0, 0xFFE6}, {0xFFE8, 0xFFEE},
    {0xFFFC, 0xFFFD},
};

static const struct w16_s pr16b[321] = {
    {0x0000, 0x000B}, {0x000D, 0x0026}, {0x0028, 0x003A}, {0x003C, 0x003D},
    {0x003F, 0x004D}, {0x0050, 0x005D}, {0x0080, 0x00FA}, {0x0100, 0x0102},
    {0x0107, 0x0133}, {0x0137, 0x018E}, {0x0190, 0x019C}, {0x01A0, 0x01A0},
    {0x01D0, 0x01FD}, {0x0280, 0x029C}, {0x02A0, 0x02D0}, {0x02E0, 0x02FB},
    {0x0300, 0x0323}, {0x032D, 0x034A}, {0x0350, 0x037A}, {0x0380, 0x039D},
    {0x039F, 0x03C3}, {0x03C8, 0x03D5}, {0x0400, 0x049D}, {0x04A0, 0x04A9},
    {0x04B0, 0x04D3}, {0x04D8, 0x04FB}, {0x0500, 0x0527}, {0x0530, 0x0563},
    {0x056F, 0x056F}, {0x0600, 0x0736}, {0x0740, 0x0755}, {0x0760, 0x0767},
    {0x0800, 0x0805}, {0x0808, 0x0808}, {0x080A, 0x0835}, {0x0837, 0x0838},
    {0x083C, 0x083C}, {0x083F, 0x0855}, {0x0857, 0x089E}, {0x08A7, 0x08AF},
    {0x08E0, 0x08F2}, {0x08F4, 0x08F5}, {0x08FB, 0x091B}, {0x091F, 0x0939},
    {0x093F, 0x093F}, {0x0980, 0x09B7}, {0x09BC, 0x09CF}, {0x09D2, 0x0A03},
    {0x0A05, 0x0A06}, {0x0A0C, 0x0A13}, {0x0A15, 0x0A17}, {0x0A19, 0x0A35},
    {0x0A38, 0x0A3A}, {0x0A3F, 0x0A48}, {0x0A50, 0x0A58}, {0x0A60, 0x0A9F},
    {0x0AC0, 0x0AE6}, {0x0AEB, 0x0AF6}, {0x0B00, 0x0B35}, {0x0B39, 0x0B55},
    {0x0B58, 0x0B72}, {0x0B78, 0x0B91}, {0x0B99, 0x0B9C}, {0x0BA9, 0x0BAF},
    {0x0C00, 0x0C48}, {0x0C80, 0x0CB2}, {0x0CC0, 0x0CF2}, {0x0CFA, 0x0D27},
    {0x0D30, 0x0D39}, {0x0E60, 0x0E7E}, {0x0E80, 0x0EA9}, {0x0EAB, 0x0EAD},
    {0x0EB0, 0x0EB1}, {0x0F00, 0x0F27}, {0x0F30, 0x0F59}, {0x0FB0, 0x0FCB},
    {0x0FE0, 0x0FF6}, {0x1000, 0x104D}, {0x1052, 0x106F}, {0x107F, 0x10BC},
    {0x10BE, 0x10C1}, {0x10D0, 0x10E8}, {0x10F0, 0x10F9}, {0x1100, 0x1134},
    {0x1136, 0x1147}, {0x1150, 0x1176}, {0x1180, 0x11DF}, {0x11E1, 0x11F4},
    {0x1200, 0x1211}, {0x1213, 0x123E}, {0x1280, 0x1286}, {0x1288, 0x1288},
    {0x128A, 0x128D}, {0x128F, 0x129D}, {0x129F, 0x12A9}, {0x12B0, 0x12EA},
    {0x12F0, 0x12F9}, {0x1300, 0x1303}, {0x1305, 0x130C}, {0x130F, 0x1310},
    {0x1313, 0x1328}, {0x132A, 0x1330}, {0x1332, 0x1333}, {0x1335, 0x1339},
    {0x133B, 0x1344}, {0x1347, 0x1348}, {0x134B, 0x134D}, {0x1350, 0x1350},
    {0x1357, 0x1357}, {0x135D, 0x1363}, {0x1366, 0x136C}, {0x1370, 0x1374},
    {0x1400, 0x145B}, {0x145D, 0x1461}, {0x1480, 0x14C7}, {0x14D0, 0x14D9},
    {0x1580, 0x15B5}, {0x15B8, 0x15DD}, {0x1600, 0x1644}, {0x1650, 0x1659},
    {0x1660, 0x166C}, {0x1680, 0x16B8}, {0x16C0, 0x16C9}, {0x1700, 0x171A},
    {0x171D, 0x172B}, {0x1730, 0x173F}, {0x1800, 0x183B}, {0x18A0, 0x18F2},
    {0x18FF, 0x1906}, {0x1909, 0x1909}, {0x190C, 0x1913}, {0x1915, 0x1916},
    {0x1918, 0x1935}, {0x1937, 0x1938}, {0x193B, 0x1946}, {0x1950, 0x1959},
    {0x19A0, 0x19A7}, {0x19AA, 0x19D7}, {0x19DA, 0x19E4}, {0x1A00, 0x1A47},
    {0x1A50, 0x1AA2}, {0x1AC0, 0x1AF8}, {0x1C00, 0x1C08}, {0x1C0A, 0x1C36},
    {0x1C38, 0x1C45}, {0x1C50, 0x1C6C}, {0x1C70, 0x1C8F}, {0x1C92, 0x1CA7},
    {0x1CA9, 0x1CB6}, {0x1D00, 0x1D06}, {0x1D08, 0x1D09}, {0x1D0B, 0x1D36},
    {0x1D3A, 0x1D3A}, {0x1D3C, 0x1D3D}, {0x1D3F, 0x1D47}, {0x1D50, 0x1D59},
    {0x1D60, 0x1D65}, {0x1D67, 0x1D68}, {0x1D6A, 0x1D8E}, {0x1D90, 0x1D91},
    {0x1D93, 0x1D98}, {0x1DA0, 0x1DA9}, {0x1EE0, 0x1EF8}, {0x1FB0, 0x1FB0},
    {0x1FC0, 0x1FF1}, {0x1FFF, 0x2399}, {0x2400, 0x246E}, {0x2470, 0x2474},
    {0x2480, 0x2543}, {0x3000, 0x342E}, {0x4400, 0x4646}, {0x6800, 0x6A38},
    {0x6A40, 0x6A5E}, {0x6A60, 0x6A69}, {0x6A6E, 0x6A6F}, {0x6AD0, 0x6AED},
    {0x6AF0, 0x6AF5}, {0x6B00, 0x6B45}, {0x6B50, 0x6B59}, {0x6B5B, 0x6B61},
    {0x6B63, 0x6B77}, {0x6B7D, 0x6B8F}, {0x6E40, 0x6E9A}, {0x6F00, 0x6F4A},
    {0x6F4F, 0x6F87}, {0x6F8F, 0x6F9F}, {0x6FE0, 0x6FE4}, {0x6FF0, 0x6FF1},
    {0x7000, 0x87F7}, {0x8800, 0x8CD5}, {0x8D00, 0x8D08}, {0xB000, 0xB11E},
    {0xB150, 0xB152}, {0xB164, 0xB167}, {0xB170, 0xB2FB}, {0xBC00, 0xBC6A},
    {0xBC70, 0xBC7C}, {0xBC80, 0xBC88}, {0xBC90, 0xBC99}, {0xBC9C, 0xBC9F},
    {0xD000, 0xD0F5}, {0xD100, 0xD126}, {0xD129, 0xD172}, {0xD17B, 0xD1E8},
    {0xD200, 0xD245}, {0xD2E0, 0xD2F3}, {0xD300, 0xD356}, {0xD360, 0xD378},
    {0xD400, 0xD454}, {0xD456, 0xD49C}, {0xD49E, 0xD49F}, {0xD4A2, 0xD4A2},
    {0xD4A5, 0xD4A6}, {0xD4A9, 0xD4AC}, {0xD4AE, 0xD4B9}, {0xD4BB, 0xD4BB},
    {0xD4BD, 0xD4C3}, {0xD4C5, 0xD505}, {0xD507, 0xD50A}, {0xD50D, 0xD514},
    {0xD516, 0xD51C}, {0xD51E, 0xD539}, {0xD53B, 0xD53E}, {0xD540, 0xD544},
    {0xD546, 0xD546}, {0xD54A, 0xD550}, {0xD552, 0xD6A5}, {0xD6A8, 0xD7CB},
    {0xD7CE, 0xDA8B}, {0xDA9B, 0xDA9F}, {0xDAA1, 0xDAAF}, {0xE000, 0xE006},
    {0xE008, 0xE018}, {0xE01B, 0xE021}, {0xE023, 0xE024}, {0xE026, 0xE02A},
    {0xE100, 0xE12C}, {0xE130, 0xE13D}, {0xE140, 0xE149}, {0xE14E, 0xE14F},
    {0xE2C0, 0xE2F9}, {0xE2FF, 0xE2FF}, {0xE800, 0xE8C4}, {0xE8C7, 0xE8D6},
    {0xE900, 0xE94B}, {0xE950, 0xE959}, {0xE95E, 0xE95F}, {0xEC71, 0xECB4},
    {0xED01, 0xED3D}, {0xEE00, 0xEE03}, {0xEE05, 0xEE1F}, {0xEE21, 0xEE22},
    {0xEE24, 0xEE24}, {0xEE27, 0xEE27}, {0xEE29, 0xEE32}, {0xEE34, 0xEE37},
    {0xEE39, 0xEE39}, {0xEE3B, 0xEE3B}, {0xEE42, 0xEE42}, {0xEE47, 0xEE47},
    {0xEE49, 0xEE49}, {0xEE4B, 0xEE4B}, {0xEE4D, 0xEE4F}, {0xEE51, 0xEE52},
    {0xEE54, 0xEE54}, {0xEE57, 0xEE57}, {0xEE59, 0xEE59}, {0xEE5B, 0xEE5B},
    {0xEE5D, 0xEE5D}, {0xEE5F, 0xEE5F}, {0xEE61, 0xEE62}, {0xEE64, 0xEE64},
    {0xEE67, 0xEE6A}, {0xEE6C, 0xEE72}, {0xEE74, 0xEE77}, {0xEE79, 0xEE7C},
    {0xEE7E, 0xEE7E}, {0xEE80, 0xEE89}, {0xEE8B, 0xEE9B}, {0xEEA1, 0xEEA3},
    {0xEEA5, 0xEEA9}, {0xEEAB, 0xEEBB}, {0xEEF0, 0xEEF1}, {0xF000, 0xF02B},
    {0xF030, 0xF093}, {0xF0A0, 0xF0AE}, {0xF0B1, 0xF0BF}, {0xF0C1, 0xF0CF},
    {0xF0D1, 0xF0F5}, {0xF100, 0xF1AD}, {0xF1E6, 0xF202}, {0xF210, 0xF23B},
    {0xF240, 0xF248}, {0xF250, 0xF251}, {0xF260, 0xF265}, {0xF300, 0xF6D7},
    {0xF6E0, 0xF6EC}, {0xF6F0, 0xF6FC}, {0xF700, 0xF773}, {0xF780, 0xF7D8},
    {0xF7E0, 0xF7EB}, {0xF800, 0xF80B}, {0xF810, 0xF847}, {0xF850, 0xF859},
    {0xF860, 0xF887}, {0xF890, 0xF8AD}, {0xF8B0, 0xF8B1}, {0xF900, 0xF978},
    {0xF97A, 0xF9CB}, {0xF9CD, 0xFA53}, {0xFA60, 0xFA6D}, {0xFA70, 0xFA74},
    {0xFA78, 0xFA7A}, {0xFA80, 0xFA86}, {0xFA90, 0xFAA8}, {0xFAB0, 0xFAB6},
    {0xFAC0, 0xFAC2}, {0xFAD0, 0xFAD6}, {0xFB00, 0xFB92}, {0xFB94, 0xFBCA},
    {0xFBF0, 0xFBF9},
};

struct w32_s {
    uint32_t start;
    uint32_t end;
};

static const struct w32_s pr32[8] = {
    {0x20000, 0x2A6DD}, {0x2A700, 0x2B734}, {0x2B740, 0x2B81D},
    {0x2B820, 0x2CEA1}, {0x2CEB0, 0x2EBE0}, {0x2F800, 0x2FA1D},
    {0x30000, 0x3134A}, {0xE0100, 0xE01EF},
};

static int compw32(const void *aa, const void *bb) {
    unichar_t key = *(const unichar_t *)aa;
    const struct w32_s *b = (const struct w32_s *)bb;

    if (key > b->end) return 1;
    if (key < b->start) return -1;
    return 0;
}

int isprint_v13(unichar_t ch) {
    if (ch < 0x10000) {
        if (bsearch(&ch, pr16a, lenof(pr16a), sizeof *pr16a, compw16) != NULL) return 1;
        return 0;
    } 
    if (ch < 0x20000) {
        ch &= 0xffff;
        if (bsearch(&ch, pr16b, lenof(pr16b), sizeof *pr16b, compw16) != NULL) return 1;
        return 0;
    } 
    if (bsearch(&ch, pr32, lenof(pr32), sizeof *pr32, compw32) != NULL) return 1;
    return 0;
}
