#include <assert.h>
#include <float.h>
#include <math.h>
#include "asn1crt.h"

#ifndef FREXP_UNSUPPORTED

#define ExpoBitMask  0x7FF0000000000000ULL
#define MantBitMask  0x000FFFFFFFFFFFFFULL
#define MantBitMask2 0xFFE0000000000000ULL

void CalculateMantissaAndExponent(double d, int* exponent, asn1SccUint64* mantissa)
{
    union {
       double in;
       asn1SccUint64 out;
    } double2uint;

    asn1SccUint64 ll = 0;

    double2uint.in = d;
    ll = double2uint.out;

    *exponent = 0;
    *mantissa = 0;

    *exponent = (int)(((ll & ExpoBitMask)>>52) - 1023 - 52);

    *mantissa = ll & MantBitMask;
    (*mantissa) |= 0x0010000000000000ULL;
}

double GetDoubleByMantissaAndExp(asn1SccUint mantissa, int exponent)
{
    union {
        double ret;
        asn1SccUint64 u64;
    } u;

    asn1SccUint64 ll = 0;
    asn1SccUint64 exponent2 = 0;

    if (mantissa == 0)
        return 0.0;


    while ( (mantissa & MantBitMask2)>0) {
        mantissa>>=1;
        exponent += 1;
    }
    while ( (mantissa & 0x0010000000000000ULL) == 0) {
        mantissa<<=1;
        exponent += -1;
    }


    exponent2 = exponent + 1023 + 52;


    ll |= mantissa & MantBitMask;
    ll |= (exponent2<<52);

    u.u64 = ll;

    return u.ret;
}



#else

#define MAX_MANTISSA 4503599627370496

double pospow2[] = {
1, 2, 4, 8, 16, 32, 64, 128, 
256, 512, 1024, 2048, 4096, 8192, 16384, 32768, 
65536, 131072, 262144, 524288, 1048576, 2097152, 4194304, 8388608, 
16777216, 33554432, 67108864, 134217728, 268435456, 536870912, 1073741824, 2147483648, 
4294967296, 8589934592, 17179869184, 34359738368, 68719476736, 137438953472, 274877906944, 549755813888, 
1099511627776, 2199023255552, 4398046511104, 8796093022208, 17592186044416, 35184372088832, 70368744177664, 140737488355328, 
281474976710656, 562949953421312, 1.12589990684262E+15, 2.25179981368525E+15, 4.5035996273705E+15, 9.00719925474099E+15, 1.8014398509482E+16, 3.6028797018964E+16, 
7.20575940379279E+16, 1.44115188075856E+17, 2.88230376151712E+17, 5.76460752303423E+17, 1.15292150460685E+18, 2.30584300921369E+18, 4.61168601842739E+18, 9.22337203685478E+18, 
1.84467440737096E+19, 3.68934881474191E+19, 7.37869762948382E+19, 1.47573952589676E+20, 2.95147905179353E+20, 5.90295810358706E+20, 1.18059162071741E+21, 2.36118324143482E+21, 
4.72236648286965E+21, 9.44473296573929E+21, 1.88894659314786E+22, 3.77789318629572E+22, 7.55578637259143E+22, 1.51115727451829E+23, 3.02231454903657E+23, 6.04462909807315E+23, 
1.20892581961463E+24, 2.41785163922926E+24, 4.83570327845852E+24, 9.67140655691703E+24, 1.93428131138341E+25, 3.86856262276681E+25, 7.73712524553363E+25, 1.54742504910673E+26, 
3.09485009821345E+26, 6.1897001964269E+26, 1.23794003928538E+27, 2.47588007857076E+27, 4.95176015714152E+27, 9.90352031428304E+27, 1.98070406285661E+28, 3.96140812571322E+28, 
7.92281625142643E+28, 1.58456325028529E+29, 3.16912650057057E+29, 6.33825300114115E+29, 1.26765060022823E+30, 2.53530120045646E+30, 5.07060240091292E+30, 1.01412048018258E+31, 
2.02824096036517E+31, 4.05648192073033E+31, 8.11296384146067E+31, 1.62259276829213E+32, 3.24518553658427E+32, 6.49037107316853E+32, 1.29807421463371E+33, 2.59614842926741E+33, 
5.19229685853483E+33, 1.03845937170697E+34, 2.07691874341393E+34, 4.15383748682786E+34, 8.30767497365572E+34, 1.66153499473114E+35, 3.32306998946229E+35, 6.64613997892458E+35, 
1.32922799578492E+36, 2.65845599156983E+36, 5.31691198313966E+36, 1.06338239662793E+37, 2.12676479325587E+37, 4.25352958651173E+37, 8.50705917302346E+37, 1.70141183460469E+38, 
3.40282366920938E+38, 6.80564733841877E+38, 1.36112946768375E+39, 2.72225893536751E+39, 5.44451787073502E+39, 1.088903574147E+40, 2.17780714829401E+40, 4.35561429658801E+40, 
8.71122859317602E+40, 1.7422457186352E+41, 3.48449143727041E+41, 6.96898287454082E+41, 1.39379657490816E+42, 2.78759314981633E+42, 5.57518629963266E+42, 1.11503725992653E+43, 
2.23007451985306E+43, 4.46014903970612E+43, 8.92029807941225E+43, 1.78405961588245E+44, 3.5681192317649E+44, 7.1362384635298E+44, 1.42724769270596E+45, 2.85449538541192E+45, 
5.70899077082384E+45, 1.14179815416477E+46, 2.28359630832954E+46, 4.56719261665907E+46, 9.13438523331814E+46, 1.82687704666363E+47, 3.65375409332726E+47, 7.30750818665451E+47, 
1.4615016373309E+48, 2.92300327466181E+48, 5.84600654932361E+48, 1.16920130986472E+49, 2.33840261972944E+49, 4.67680523945889E+49, 9.35361047891778E+49, 1.87072209578356E+50, 
3.74144419156711E+50, 7.48288838313422E+50, 1.49657767662684E+51, 2.99315535325369E+51, 5.98631070650738E+51, 1.19726214130148E+52, 2.39452428260295E+52, 4.7890485652059E+52, 
9.57809713041181E+52, 1.91561942608236E+53, 3.83123885216472E+53, 7.66247770432944E+53, 1.53249554086589E+54, 3.06499108173178E+54, 6.12998216346356E+54, 1.22599643269271E+55, 
2.45199286538542E+55, 4.90398573077084E+55, 9.80797146154169E+55, 1.96159429230834E+56, 3.92318858461668E+56, 7.84637716923335E+56, 1.56927543384667E+57, 3.13855086769334E+57, 
6.27710173538668E+57, 1.25542034707734E+58, 2.51084069415467E+58, 5.02168138830934E+58, 1.00433627766187E+59, 2.00867255532374E+59, 4.01734511064748E+59, 8.03469022129495E+59, 
1.60693804425899E+60, 3.21387608851798E+60, 6.42775217703596E+60, 1.28555043540719E+61, 2.57110087081438E+61, 5.14220174162877E+61, 1.02844034832575E+62, 2.05688069665151E+62, 
4.11376139330302E+62, 8.22752278660603E+62, 1.64550455732121E+63, 3.29100911464241E+63, 6.58201822928482E+63, 1.31640364585696E+64, 2.63280729171393E+64, 5.26561458342786E+64, 
1.05312291668557E+65, 2.10624583337114E+65, 4.21249166674229E+65, 8.42498333348457E+65, 1.68499666669692E+66, 3.36999333339383E+66, 6.73998666678766E+66, 1.34799733335753E+67, 
2.69599466671506E+67, 5.39198933343013E+67, 1.07839786668603E+68, 2.15679573337205E+68, 4.3135914667441E+68, 8.6271829334882E+68, 1.72543658669764E+69, 3.45087317339528E+69, 
6.90174634679056E+69, 1.38034926935811E+70, 2.76069853871623E+70, 5.52139707743245E+70, 1.10427941548649E+71, 2.20855883097298E+71, 4.41711766194596E+71, 8.83423532389192E+71, 
1.76684706477838E+72, 3.53369412955677E+72, 7.06738825911354E+72, 1.41347765182271E+73, 2.82695530364541E+73, 5.65391060729083E+73, 1.13078212145817E+74, 2.26156424291633E+74, 
4.52312848583266E+74, 9.04625697166533E+74, 1.80925139433307E+75, 3.61850278866613E+75, 7.23700557733226E+75, 1.44740111546645E+76, 2.8948022309329E+76, 5.78960446186581E+76, 
1.15792089237316E+77, 2.31584178474632E+77, 4.63168356949265E+77, 9.2633671389853E+77, 1.85267342779706E+78, 3.70534685559412E+78, 7.41069371118824E+78, 1.48213874223765E+79, 
2.96427748447529E+79, 5.92855496895059E+79, 1.18571099379012E+80, 2.37142198758024E+80, 4.74284397516047E+80, 9.48568795032094E+80, 1.89713759006419E+81, 3.79427518012838E+81, 
7.58855036025675E+81, 1.51771007205135E+82, 3.0354201441027E+82, 6.0708402882054E+82, 1.21416805764108E+83, 2.42833611528216E+83, 4.85667223056432E+83, 9.71334446112865E+83, 
1.94266889222573E+84, 3.88533778445146E+84, 7.77067556890292E+84, 1.55413511378058E+85, 3.10827022756117E+85, 6.21654045512233E+85, 1.24330809102447E+86, 2.48661618204893E+86, 
4.97323236409787E+86, 9.94646472819573E+86, 1.98929294563915E+87, 3.97858589127829E+87, 7.95717178255659E+87, 1.59143435651132E+88, 3.18286871302263E+88, 6.36573742604527E+88, 
1.27314748520905E+89, 2.54629497041811E+89, 5.09258994083622E+89, 1.01851798816724E+90, 2.03703597633449E+90, 4.07407195266897E+90, 8.14814390533794E+90, 1.62962878106759E+91, 
3.25925756213518E+91, 6.51851512427036E+91, 1.30370302485407E+92, 2.60740604970814E+92, 5.21481209941628E+92, 1.04296241988326E+93, 2.08592483976651E+93, 4.17184967953303E+93, 
8.34369935906606E+93, 1.66873987181321E+94, 3.33747974362642E+94, 6.67495948725284E+94, 1.33499189745057E+95, 2.66998379490114E+95, 5.33996758980228E+95, 1.06799351796046E+96, 
2.13598703592091E+96, 4.27197407184182E+96, 8.54394814368364E+96, 1.70878962873673E+97, 3.41757925747346E+97, 6.83515851494691E+97, 1.36703170298938E+98, 2.73406340597876E+98, 
5.46812681195753E+98, 1.09362536239151E+99, 2.18725072478301E+99, 4.37450144956602E+99, 8.74900289913205E+99, 1.74980057982641E+100, 3.49960115965282E+100, 6.99920231930564E+100, 
1.39984046386113E+101, 2.79968092772226E+101, 5.59936185544451E+101, 1.1198723710889E+102, 2.2397447421778E+102, 4.47948948435561E+102, 8.95897896871122E+102, 1.79179579374224E+103, 
3.58359158748449E+103, 7.16718317496897E+103, 1.43343663499379E+104, 2.86687326998759E+104, 5.73374653997518E+104, 1.14674930799504E+105, 2.29349861599007E+105, 4.58699723198014E+105, 
9.17399446396029E+105, 1.83479889279206E+106, 3.66959778558411E+106, 7.33919557116823E+106, 1.46783911423365E+107, 2.93567822846729E+107, 5.87135645693458E+107, 1.17427129138692E+108, 
2.34854258277383E+108, 4.69708516554767E+108, 9.39417033109533E+108, 1.87883406621907E+109, 3.75766813243813E+109, 7.51533626487627E+109, 1.50306725297525E+110, 3.00613450595051E+110, 
6.01226901190101E+110, 1.2024538023802E+111, 2.40490760476041E+111, 4.80981520952081E+111, 9.61963041904162E+111, 1.92392608380832E+112, 3.84785216761665E+112, 7.6957043352333E+112, 
1.53914086704666E+113, 3.07828173409332E+113, 6.15656346818664E+113, 1.23131269363733E+114, 2.46262538727466E+114, 4.92525077454931E+114, 9.85050154909862E+114, 1.97010030981972E+115, 
3.94020061963945E+115, 7.8804012392789E+115, 1.57608024785578E+116, 3.15216049571156E+116, 6.30432099142312E+116, 1.26086419828462E+117, 2.52172839656925E+117, 5.04345679313849E+117, 
1.0086913586277E+118, 2.0173827172554E+118, 4.03476543451079E+118, 8.06953086902159E+118, 1.61390617380432E+119, 3.22781234760864E+119, 6.45562469521727E+119, 1.29112493904345E+120, 
2.58224987808691E+120, 5.16449975617382E+120, 1.03289995123476E+121, 2.06579990246953E+121, 4.13159980493905E+121, 8.26319960987811E+121, 1.65263992197562E+122, 3.30527984395124E+122, 
6.61055968790249E+122, 1.3221119375805E+123, 2.64422387516099E+123, 5.28844775032199E+123, 1.0576895500644E+124, 2.1153791001288E+124, 4.23075820025759E+124, 8.46151640051518E+124, 
1.69230328010304E+125, 3.38460656020607E+125, 6.76921312041215E+125, 1.35384262408243E+126, 2.70768524816486E+126, 5.41537049632972E+126, 1.08307409926594E+127, 2.16614819853189E+127, 
4.33229639706377E+127, 8.66459279412755E+127, 1.73291855882551E+128, 3.46583711765102E+128, 6.93167423530204E+128, 1.38633484706041E+129, 2.77266969412081E+129, 5.54533938824163E+129, 
1.10906787764833E+130, 2.21813575529665E+130, 4.4362715105933E+130, 8.87254302118661E+130, 1.77450860423732E+131, 3.54901720847464E+131, 7.09803441694929E+131, 1.41960688338986E+132, 
2.83921376677971E+132, 5.67842753355943E+132, 1.13568550671189E+133, 2.27137101342377E+133, 4.54274202684754E+133, 9.08548405369509E+133, 1.81709681073902E+134, 3.63419362147803E+134, 
7.26838724295607E+134, 1.45367744859121E+135, 2.90735489718243E+135, 5.81470979436486E+135, 1.16294195887297E+136, 2.32588391774594E+136, 4.65176783549188E+136, 9.30353567098377E+136, 
1.86070713419675E+137, 3.72141426839351E+137, 7.44282853678701E+137, 1.4885657073574E+138, 2.97713141471481E+138, 5.95426282942961E+138, 1.19085256588592E+139, 2.38170513177184E+139, 
4.76341026354369E+139, 9.52682052708738E+139, 1.90536410541748E+140, 3.81072821083495E+140, 7.6214564216699E+140, 1.52429128433398E+141, 3.04858256866796E+141, 6.09716513733592E+141, 
1.21943302746718E+142, 2.43886605493437E+142, 4.87773210986874E+142, 9.75546421973748E+142, 1.9510928439475E+143, 3.90218568789499E+143, 7.80437137578998E+143, 1.560874275158E+144, 
3.12174855031599E+144, 6.24349710063198E+144, 1.2486994201264E+145, 2.49739884025279E+145, 4.99479768050559E+145, 9.98959536101118E+145, 1.99791907220224E+146, 3.99583814440447E+146, 
7.99167628880894E+146, 1.59833525776179E+147, 3.19667051552358E+147, 6.39334103104715E+147, 1.27866820620943E+148, 2.55733641241886E+148, 5.11467282483772E+148, 1.02293456496754E+149, 
2.04586912993509E+149, 4.09173825987018E+149, 8.18347651974035E+149, 1.63669530394807E+150, 3.27339060789614E+150, 6.54678121579228E+150, 1.30935624315846E+151, 2.61871248631691E+151, 
5.23742497263383E+151, 1.04748499452677E+152, 2.09496998905353E+152, 4.18993997810706E+152, 8.37987995621412E+152, 1.67597599124282E+153, 3.35195198248565E+153, 6.7039039649713E+153, 
1.34078079299426E+154, 2.68156158598852E+154, 5.36312317197704E+154, 1.07262463439541E+155, 2.14524926879082E+155, 4.29049853758163E+155, 8.58099707516326E+155, 1.71619941503265E+156, 
3.4323988300653E+156, 6.86479766013061E+156, 1.37295953202612E+157, 2.74591906405224E+157, 5.49183812810449E+157, 1.0983676256209E+158, 2.1967352512418E+158, 4.39347050248359E+158, 
8.78694100496718E+158, 1.75738820099344E+159, 3.51477640198687E+159, 7.02955280397374E+159, 1.40591056079475E+160, 2.8118211215895E+160, 5.623642243179E+160, 1.1247284486358E+161, 
2.2494568972716E+161, 4.4989137945432E+161, 8.99782758908639E+161, 1.79956551781728E+162, 3.59913103563456E+162, 7.19826207126911E+162, 1.43965241425382E+163, 2.87930482850765E+163, 
5.75860965701529E+163, 1.15172193140306E+164, 2.30344386280612E+164, 4.60688772561223E+164, 9.21377545122447E+164, 1.84275509024489E+165, 3.68551018048979E+165, 7.37102036097957E+165, 
1.47420407219591E+166, 2.94840814439183E+166, 5.89681628878366E+166, 1.17936325775673E+167, 2.35872651551346E+167, 4.71745303102693E+167, 9.43490606205385E+167, 1.88698121241077E+168, 
3.77396242482154E+168, 7.54792484964308E+168, 1.50958496992862E+169, 3.01916993985723E+169, 6.03833987971447E+169, 1.20766797594289E+170, 2.41533595188579E+170, 4.83067190377157E+170, 
9.66134380754315E+170, 1.93226876150863E+171, 3.86453752301726E+171, 7.72907504603452E+171, 1.5458150092069E+172, 3.09163001841381E+172, 6.18326003682761E+172, 1.23665200736552E+173, 
2.47330401473105E+173, 4.94660802946209E+173, 9.89321605892418E+173, 1.97864321178484E+174, 3.95728642356967E+174, 7.91457284713935E+174, 1.58291456942787E+175, 3.16582913885574E+175, 
6.33165827771148E+175, 1.2663316555423E+176, 2.53266331108459E+176, 5.06532662216918E+176, 1.01306532443384E+177, 2.02613064886767E+177, 4.05226129773534E+177, 8.10452259547069E+177, 
1.62090451909414E+178, 3.24180903818828E+178, 6.48361807637655E+178, 1.29672361527531E+179, 2.59344723055062E+179, 5.18689446110124E+179, 1.03737889222025E+180, 2.0747577844405E+180, 
4.14951556888099E+180, 8.29903113776199E+180, 1.6598062275524E+181, 3.31961245510479E+181, 6.63922491020959E+181, 1.32784498204192E+182, 2.65568996408384E+182, 5.31137992816767E+182, 
1.06227598563353E+183, 2.12455197126707E+183, 4.24910394253414E+183, 8.49820788506827E+183, 1.69964157701365E+184, 3.39928315402731E+184, 6.79856630805462E+184, 1.35971326161092E+185, 
2.71942652322185E+185, 5.4388530464437E+185, 1.08777060928874E+186, 2.17554121857748E+186, 4.35108243715496E+186, 8.70216487430991E+186, 1.74043297486198E+187, 3.48086594972396E+187, 
6.96173189944793E+187, 1.39234637988959E+188, 2.78469275977917E+188, 5.56938551955834E+188, 1.11387710391167E+189, 2.22775420782334E+189, 4.45550841564668E+189, 8.91101683129335E+189, 
1.78220336625867E+190, 3.56440673251734E+190, 7.12881346503468E+190, 1.42576269300694E+191, 2.85152538601387E+191, 5.70305077202774E+191, 1.14061015440555E+192, 2.2812203088111E+192, 
4.5624406176222E+192, 9.12488123524439E+192, 1.82497624704888E+193, 3.64995249409776E+193, 7.29990498819551E+193, 1.4599809976391E+194, 2.9199619952782E+194, 5.83992399055641E+194, 
1.16798479811128E+195, 2.33596959622256E+195, 4.67193919244513E+195, 9.34387838489026E+195, 1.86877567697805E+196, 3.7375513539561E+196, 7.4751027079122E+196, 1.49502054158244E+197, 
2.99004108316488E+197, 5.98008216632976E+197, 1.19601643326595E+198, 2.39203286653191E+198, 4.78406573306381E+198, 9.56813146612762E+198, 1.91362629322552E+199, 3.82725258645105E+199, 
7.6545051729021E+199, 1.53090103458042E+200, 3.06180206916084E+200, 6.12360413832168E+200, 1.22472082766434E+201, 2.44944165532867E+201, 4.89888331065734E+201, 9.79776662131468E+201, 
1.95955332426294E+202, 3.91910664852587E+202, 7.83821329705175E+202, 1.56764265941035E+203, 3.1352853188207E+203, 6.2705706376414E+203, 1.25411412752828E+204, 2.50822825505656E+204, 
5.01645651011312E+204, 1.00329130202262E+205, 2.00658260404525E+205, 4.01316520809049E+205, 8.02633041618099E+205, 1.6052660832362E+206, 3.2105321664724E+206, 6.42106433294479E+206, 
1.28421286658896E+207, 2.56842573317792E+207, 5.13685146635583E+207, 1.02737029327117E+208, 2.05474058654233E+208, 4.10948117308467E+208, 8.21896234616933E+208, 1.64379246923387E+209, 
3.28758493846773E+209, 6.57516987693547E+209, 1.31503397538709E+210, 2.63006795077419E+210, 5.26013590154837E+210, 1.05202718030967E+211, 2.10405436061935E+211, 4.2081087212387E+211, 
8.4162174424774E+211, 1.68324348849548E+212, 3.36648697699096E+212, 6.73297395398192E+212, 1.34659479079638E+213, 2.69318958159277E+213, 5.38637916318553E+213, 1.07727583263711E+214, 
2.15455166527421E+214, 4.30910333054843E+214, 8.61820666109686E+214, 1.72364133221937E+215, 3.44728266443874E+215, 6.89456532887748E+215, 1.3789130657755E+216, 2.75782613155099E+216, 
5.51565226310199E+216, 1.1031304526204E+217, 2.20626090524079E+217, 4.41252181048159E+217, 8.82504362096318E+217, 1.76500872419264E+218, 3.53001744838527E+218, 7.06003489677054E+218, 
1.41200697935411E+219, 2.82401395870822E+219, 5.64802791741644E+219, 1.12960558348329E+220, 2.25921116696657E+220, 4.51842233393315E+220, 9.0368446678663E+220, 1.80736893357326E+221, 
3.61473786714652E+221, 7.22947573429304E+221, 1.44589514685861E+222, 2.89179029371721E+222, 5.78358058743443E+222, 1.15671611748689E+223, 2.31343223497377E+223, 4.62686446994754E+223, 
9.25372893989509E+223, 1.85074578797902E+224, 3.70149157595803E+224, 7.40298315191607E+224, 1.48059663038321E+225, 2.96119326076643E+225, 5.92238652153286E+225, 1.18447730430657E+226, 
2.36895460861314E+226, 4.73790921722628E+226, 9.47581843445257E+226, 1.89516368689051E+227, 3.79032737378103E+227, 7.58065474756206E+227, 1.51613094951241E+228, 3.03226189902482E+228, 
6.06452379804964E+228, 1.21290475960993E+229, 2.42580951921986E+229, 4.85161903843972E+229, 9.70323807687943E+229, 1.94064761537589E+230, 3.88129523075177E+230, 7.76259046150354E+230, 
1.55251809230071E+231, 3.10503618460142E+231, 6.21007236920284E+231, 1.24201447384057E+232, 2.48402894768113E+232, 4.96805789536227E+232, 9.93611579072454E+232, 1.98722315814491E+233, 
3.97444631628981E+233, 7.94889263257963E+233, 1.58977852651593E+234, 3.17955705303185E+234, 6.3591141060637E+234, 1.27182282121274E+235, 2.54364564242548E+235, 5.08729128485096E+235, 
1.01745825697019E+236, 2.03491651394039E+236, 4.06983302788077E+236, 8.13966605576154E+236, 1.62793321115231E+237, 3.25586642230462E+237, 6.51173284460923E+237, 1.30234656892185E+238, 
2.60469313784369E+238, 5.20938627568739E+238, 1.04187725513748E+239, 2.08375451027495E+239, 4.16750902054991E+239, 8.33501804109982E+239, 1.66700360821996E+240, 3.33400721643993E+240, 
6.66801443287985E+240, 1.33360288657597E+241, 2.66720577315194E+241, 5.33441154630388E+241, 1.06688230926078E+242, 2.13376461852155E+242, 4.26752923704311E+242, 8.53505847408621E+242, 
1.70701169481724E+243, 3.41402338963449E+243, 6.82804677926897E+243, 1.36560935585379E+244, 2.73121871170759E+244, 5.46243742341518E+244, 1.09248748468304E+245, 2.18497496936607E+245, 
4.36994993873214E+245, 8.73989987746428E+245, 1.74797997549286E+246, 3.49595995098571E+246, 6.99191990197143E+246, 1.39838398039429E+247, 2.79676796078857E+247, 5.59353592157714E+247, 
1.11870718431543E+248, 2.23741436863086E+248, 4.47482873726171E+248, 8.94965747452343E+248, 1.78993149490469E+249, 3.57986298980937E+249, 7.15972597961874E+249, 1.43194519592375E+250, 
2.8638903918475E+250, 5.72778078369499E+250, 1.145556156739E+251, 2.291112313478E+251, 4.58222462695599E+251, 9.16444925391199E+251, 1.8328898507824E+252, 3.6657797015648E+252, 
7.33155940312959E+252, 1.46631188062592E+253, 2.93262376125184E+253, 5.86524752250367E+253, 1.17304950450073E+254, 2.34609900900147E+254, 4.69219801800294E+254, 9.38439603600588E+254, 
1.87687920720118E+255, 3.75375841440235E+255, 7.5075168288047E+255, 1.50150336576094E+256, 3.00300673152188E+256, 6.00601346304376E+256, 1.20120269260875E+257, 2.4024053852175E+257, 
4.80481077043501E+257, 9.60962154087002E+257, 1.921924308174E+258, 3.84384861634801E+258, 7.68769723269601E+258, 1.5375394465392E+259, 3.07507889307841E+259, 6.15015778615681E+259, 
1.23003155723136E+260, 2.46006311446272E+260, 4.92012622892545E+260, 9.8402524578509E+260, 1.96805049157018E+261, 3.93610098314036E+261, 7.87220196628072E+261, 1.57444039325614E+262, 
3.14888078651229E+262, 6.29776157302457E+262, 1.25955231460491E+263, 2.51910462920983E+263, 5.03820925841966E+263, 1.00764185168393E+264, 2.01528370336786E+264, 4.03056740673573E+264, 
8.06113481347145E+264, 1.61222696269429E+265, 3.22445392538858E+265, 6.44890785077716E+265, 1.28978157015543E+266, 2.57956314031087E+266, 5.15912628062173E+266, 1.03182525612435E+267, 
2.06365051224869E+267, 4.12730102449738E+267, 8.25460204899477E+267, 1.65092040979895E+268, 3.30184081959791E+268, 6.60368163919582E+268, 1.32073632783916E+269, 2.64147265567833E+269, 
5.28294531135665E+269, 1.05658906227133E+270, 2.11317812454266E+270, 4.22635624908532E+270, 8.45271249817064E+270, 1.69054249963413E+271, 3.38108499926826E+271, 6.76216999853652E+271, 
1.3524339997073E+272, 2.70486799941461E+272, 5.40973599882921E+272, 1.08194719976584E+273, 2.16389439953168E+273, 4.32778879906337E+273, 8.65557759812674E+273, 1.73111551962535E+274, 
3.4622310392507E+274, 6.92446207850139E+274, 1.38489241570028E+275, 2.76978483140056E+275, 5.53956966280111E+275, 1.10791393256022E+276, 2.21582786512045E+276, 4.43165573024089E+276, 
8.86331146048178E+276, 1.77266229209636E+277, 3.54532458419271E+277, 7.09064916838542E+277, 1.41812983367708E+278, 2.83625966735417E+278, 5.67251933470834E+278, 1.13450386694167E+279, 
2.26900773388334E+279, 4.53801546776667E+279, 9.07603093553334E+279, 1.81520618710667E+280, 3.63041237421334E+280, 7.26082474842668E+280, 1.45216494968534E+281, 2.90432989937067E+281, 
5.80865979874134E+281, 1.16173195974827E+282, 2.32346391949654E+282, 4.64692783899307E+282, 9.29385567798614E+282, 1.85877113559723E+283, 3.71754227119446E+283, 7.43508454238892E+283, 
1.48701690847778E+284, 2.97403381695557E+284, 5.94806763391113E+284, 1.18961352678223E+285, 2.37922705356445E+285, 4.75845410712891E+285, 9.51690821425781E+285, 1.90338164285156E+286, 
3.80676328570312E+286, 7.61352657140625E+286, 1.52270531428125E+287, 3.0454106285625E+287, 6.090821257125E+287, 1.218164251425E+288, 2.43632850285E+288, 4.8726570057E+288, 
9.7453140114E+288, 1.94906280228E+289, 3.89812560456E+289, 7.79625120912E+289, 1.559250241824E+290, 3.118500483648E+290, 6.237000967296E+290, 1.2474001934592E+291, 
2.4948003869184E+291, 4.9896007738368E+291, 9.9792015476736E+291, 1.99584030953472E+292, 3.99168061906944E+292, 7.98336123813888E+292, 1.59667224762778E+293, 3.19334449525555E+293, 
6.3866889905111E+293, 1.27733779810222E+294, 2.55467559620444E+294, 5.10935119240888E+294, 1.02187023848178E+295, 2.04374047696355E+295, 4.08748095392711E+295, 8.17496190785421E+295, 
1.63499238157084E+296, 3.26998476314168E+296, 6.53996952628337E+296, 1.30799390525667E+297, 2.61598781051335E+297, 5.2319756210267E+297, 1.04639512420534E+298, 2.09279024841068E+298, 
4.18558049682136E+298, 8.37116099364271E+298, 1.67423219872854E+299, 3.34846439745709E+299, 6.69692879491417E+299, 1.33938575898283E+300, 2.67877151796567E+300, 5.35754303593134E+300, 
1.07150860718627E+301, 2.14301721437253E+301, 4.28603442874507E+301, 8.57206885749014E+301, 1.71441377149803E+302, 3.42882754299606E+302, 6.85765508599211E+302, 1.37153101719842E+303, 
2.74306203439684E+303, 5.48612406879369E+303, 1.09722481375874E+304, 2.19444962751748E+304, 4.38889925503495E+304, 8.7777985100699E+304, 1.75555970201398E+305, 3.51111940402796E+305, 
7.02223880805592E+305, 1.40444776161118E+306, 2.80889552322237E+306, 5.61779104644474E+306, 1.12355820928895E+307, 2.24711641857789E+307, 4.49423283715579E+307
};

double negpow2[] = {
1, 0.5, 0.25, 0.125, 0.0625, 0.03125, 0.015625, 0.0078125, 
0.00390625, 0.001953125, 0.0009765625, 0.00048828125, 0.000244140625, 0.0001220703125, 6.103515625E-05, 3.0517578125E-05, 
1.52587890625E-05, 7.62939453125E-06, 3.814697265625E-06, 1.9073486328125E-06, 9.5367431640625E-07, 4.76837158203125E-07, 2.38418579101563E-07, 1.19209289550781E-07, 
5.96046447753906E-08, 2.98023223876953E-08, 1.49011611938477E-08, 7.45058059692383E-09, 3.72529029846191E-09, 1.86264514923096E-09, 9.31322574615479E-10, 4.65661287307739E-10, 
2.3283064365387E-10, 1.16415321826935E-10, 5.82076609134674E-11, 2.91038304567337E-11, 1.45519152283669E-11, 7.27595761418343E-12, 3.63797880709171E-12, 1.81898940354586E-12, 
9.09494701772928E-13, 4.54747350886464E-13, 2.27373675443232E-13, 1.13686837721616E-13, 5.6843418860808E-14, 2.8421709430404E-14, 1.4210854715202E-14, 7.105427357601E-15, 
3.5527136788005E-15, 1.77635683940025E-15, 8.88178419700125E-16, 4.44089209850063E-16, 2.22044604925031E-16, 1.11022302462516E-16, 5.55111512312578E-17, 2.77555756156289E-17, 
1.38777878078145E-17, 6.93889390390723E-18, 3.46944695195361E-18, 1.73472347597681E-18, 8.67361737988404E-19, 4.33680868994202E-19, 2.16840434497101E-19, 1.0842021724855E-19, 
5.42101086242752E-20, 2.71050543121376E-20, 1.35525271560688E-20, 6.7762635780344E-21, 3.3881317890172E-21, 1.6940658945086E-21, 8.470329472543E-22, 4.2351647362715E-22, 
2.11758236813575E-22, 1.05879118406788E-22, 5.29395592033938E-23, 2.64697796016969E-23, 1.32348898008484E-23, 6.61744490042422E-24, 3.30872245021211E-24, 1.65436122510606E-24, 
8.27180612553028E-25, 4.13590306276514E-25, 2.06795153138257E-25, 1.03397576569128E-25, 5.16987882845642E-26, 2.58493941422821E-26, 1.29246970711411E-26, 6.46234853557053E-27, 
3.23117426778526E-27, 1.61558713389263E-27, 8.07793566946316E-28, 4.03896783473158E-28, 2.01948391736579E-28, 1.0097419586829E-28, 5.04870979341448E-29, 2.52435489670724E-29, 
1.26217744835362E-29, 6.31088724176809E-30, 3.15544362088405E-30, 1.57772181044202E-30, 7.88860905221012E-31, 3.94430452610506E-31, 1.97215226305253E-31, 9.86076131526265E-32, 
4.93038065763132E-32, 2.46519032881566E-32, 1.23259516440783E-32, 6.16297582203915E-33, 3.08148791101958E-33, 1.54074395550979E-33, 7.70371977754894E-34, 3.85185988877447E-34, 
1.92592994438724E-34, 9.62964972193618E-35, 4.81482486096809E-35, 2.40741243048404E-35, 1.20370621524202E-35, 6.01853107621011E-36, 3.00926553810506E-36, 1.50463276905253E-36, 
7.52316384526264E-37, 3.76158192263132E-37, 1.88079096131566E-37, 9.4039548065783E-38, 4.70197740328915E-38, 2.35098870164458E-38, 1.17549435082229E-38, 5.87747175411144E-39, 
2.93873587705572E-39, 1.46936793852786E-39, 7.3468396926393E-40, 3.67341984631965E-40, 1.83670992315982E-40, 9.18354961579912E-41, 4.59177480789956E-41, 2.29588740394978E-41, 
1.14794370197489E-41, 5.73971850987445E-42, 2.86985925493723E-42, 1.43492962746861E-42, 7.17464813734306E-43, 3.58732406867153E-43, 1.79366203433577E-43, 8.96831017167883E-44, 
4.48415508583941E-44, 2.24207754291971E-44, 1.12103877145985E-44, 5.60519385729927E-45, 2.80259692864963E-45, 1.40129846432482E-45, 7.00649232162409E-46, 3.50324616081204E-46, 
1.75162308040602E-46, 8.75811540203011E-47, 4.37905770101505E-47, 2.18952885050753E-47, 1.09476442525376E-47, 5.47382212626882E-48, 2.73691106313441E-48, 1.3684555315672E-48, 
6.84227765783602E-49, 3.42113882891801E-49, 1.71056941445901E-49, 8.55284707229503E-50, 4.27642353614751E-50, 2.13821176807376E-50, 1.06910588403688E-50, 5.34552942018439E-51, 
2.6727647100922E-51, 1.3363823550461E-51, 6.68191177523049E-52, 3.34095588761524E-52, 1.67047794380762E-52, 8.35238971903811E-53, 4.17619485951906E-53, 2.08809742975953E-53, 
1.04404871487976E-53, 5.22024357439882E-54, 2.61012178719941E-54, 1.3050608935997E-54, 6.52530446799852E-55, 3.26265223399926E-55, 1.63132611699963E-55, 8.15663058499816E-56, 
4.07831529249908E-56, 2.03915764624954E-56, 1.01957882312477E-56, 5.09789411562385E-57, 2.54894705781192E-57, 1.27447352890596E-57, 6.37236764452981E-58, 3.1861838222649E-58, 
1.59309191113245E-58, 7.96545955566226E-59, 3.98272977783113E-59, 1.99136488891557E-59, 9.95682444457783E-60, 4.97841222228891E-60, 2.48920611114446E-60, 1.24460305557223E-60, 
6.22301527786114E-61, 3.11150763893057E-61, 1.55575381946529E-61, 7.77876909732643E-62, 3.88938454866321E-62, 1.94469227433161E-62, 9.72346137165803E-63, 4.86173068582902E-63, 
2.43086534291451E-63, 1.21543267145725E-63, 6.07716335728627E-64, 3.03858167864314E-64, 1.51929083932157E-64, 7.59645419660784E-65, 3.79822709830392E-65, 1.89911354915196E-65, 
9.4955677457598E-66, 4.7477838728799E-66, 2.37389193643995E-66, 1.18694596821997E-66, 5.93472984109987E-67, 2.96736492054994E-67, 1.48368246027497E-67, 7.41841230137484E-68, 
3.70920615068742E-68, 1.85460307534371E-68, 9.27301537671855E-69, 4.63650768835928E-69, 2.31825384417964E-69, 1.15912692208982E-69, 5.7956346104491E-70, 2.89781730522455E-70, 
1.44890865261227E-70, 7.24454326306137E-71, 3.62227163153068E-71, 1.81113581576534E-71, 9.05567907882671E-72, 4.52783953941336E-72, 2.26391976970668E-72, 1.13195988485334E-72, 
5.6597994242667E-73, 2.82989971213335E-73, 1.41494985606667E-73, 7.07474928033337E-74, 3.53737464016668E-74, 1.76868732008334E-74, 8.84343660041671E-75, 4.42171830020836E-75, 
2.21085915010418E-75, 1.10542957505209E-75, 5.52714787526044E-76, 2.76357393763022E-76, 1.38178696881511E-76, 6.90893484407556E-77, 3.45446742203778E-77, 1.72723371101889E-77, 
8.63616855509444E-78, 4.31808427754722E-78, 2.15904213877361E-78, 1.07952106938681E-78, 5.39760534693403E-79, 2.69880267346701E-79, 1.34940133673351E-79, 6.74700668366753E-80, 
3.37350334183377E-80, 1.68675167091688E-80, 8.43375835458442E-81, 4.21687917729221E-81, 2.1084395886461E-81, 1.05421979432305E-81, 5.27109897161526E-82, 2.63554948580763E-82, 
1.31777474290382E-82, 6.58887371451908E-83, 3.29443685725954E-83, 1.64721842862977E-83, 8.23609214314885E-84, 4.11804607157442E-84, 2.05902303578721E-84, 1.02951151789361E-84, 
5.14755758946803E-85, 2.57377879473401E-85, 1.28688939736701E-85, 6.43444698683504E-86, 3.21722349341752E-86, 1.60861174670876E-86, 8.0430587335438E-87, 4.0215293667719E-87, 
2.01076468338595E-87, 1.00538234169297E-87, 5.02691170846487E-88, 2.51345585423244E-88, 1.25672792711622E-88, 6.28363963558109E-89, 3.14181981779055E-89, 1.57090990889527E-89, 
7.85454954447636E-90, 3.92727477223818E-90, 1.96363738611909E-90, 9.81818693059545E-91, 4.90909346529773E-91, 2.45454673264886E-91, 1.22727336632443E-91, 6.13636683162216E-92, 
3.06818341581108E-92, 1.53409170790554E-92, 7.6704585395277E-93, 3.83522926976385E-93, 1.91761463488192E-93, 9.58807317440962E-94, 4.79403658720481E-94, 2.39701829360241E-94, 
1.1985091468012E-94, 5.99254573400601E-95, 2.99627286700301E-95, 1.4981364335015E-95, 7.49068216750752E-96, 3.74534108375376E-96, 1.87267054187688E-96, 9.3633527093844E-97, 
4.6816763546922E-97, 2.3408381773461E-97, 1.17041908867305E-97, 5.85209544336525E-98, 2.92604772168262E-98, 1.46302386084131E-98, 7.31511930420656E-99, 3.65755965210328E-99, 
1.82877982605164E-99, 9.1438991302582E-100, 4.5719495651291E-100, 2.28597478256455E-100, 1.14298739128227E-100, 5.71493695641137E-101, 2.85746847820569E-101, 1.42873423910284E-101, 
7.14367119551422E-102, 3.57183559775711E-102, 1.78591779887855E-102, 8.92958899439277E-103, 4.46479449719639E-103, 2.23239724859819E-103, 1.1161986242991E-103, 5.58099312149548E-104, 
2.79049656074774E-104, 1.39524828037387E-104, 6.97624140186935E-105, 3.48812070093468E-105, 1.74406035046734E-105, 8.72030175233669E-106, 4.36015087616835E-106, 2.18007543808417E-106, 
1.09003771904209E-106, 5.45018859521043E-107, 2.72509429760522E-107, 1.36254714880261E-107, 6.81273574401304E-108, 3.40636787200652E-108, 1.70318393600326E-108, 8.5159196800163E-109, 
4.25795984000815E-109, 2.12897992000408E-109, 1.06448996000204E-109, 5.32244980001019E-110, 2.66122490000509E-110, 1.33061245000255E-110, 6.65306225001274E-111, 3.32653112500637E-111, 
1.66326556250318E-111, 8.31632781251592E-112, 4.15816390625796E-112, 2.07908195312898E-112, 1.03954097656449E-112, 5.19770488282245E-113, 2.59885244141122E-113, 1.29942622070561E-113, 
6.49713110352806E-114, 3.24856555176403E-114, 1.62428277588202E-114, 8.12141387941008E-115, 4.06070693970504E-115, 2.03035346985252E-115, 1.01517673492626E-115, 5.0758836746313E-116, 
2.53794183731565E-116, 1.26897091865782E-116, 6.34485459328912E-117, 3.17242729664456E-117, 1.58621364832228E-117, 7.9310682416114E-118, 3.9655341208057E-118, 1.98276706040285E-118, 
9.91383530201425E-119, 4.95691765100713E-119, 2.47845882550356E-119, 1.23922941275178E-119, 6.19614706375891E-120, 3.09807353187945E-120, 1.54903676593973E-120, 7.74518382969864E-121, 
3.87259191484932E-121, 1.93629595742466E-121, 9.6814797871233E-122, 4.84073989356165E-122, 2.42036994678082E-122, 1.21018497339041E-122, 6.05092486695206E-123, 3.02546243347603E-123, 
1.51273121673802E-123, 7.56365608369007E-124, 3.78182804184504E-124, 1.89091402092252E-124, 9.45457010461259E-125, 4.7272850523063E-125, 2.36364252615315E-125, 1.18182126307657E-125, 
5.90910631538287E-126, 2.95455315769144E-126, 1.47727657884572E-126, 7.38638289422859E-127, 3.69319144711429E-127, 1.84659572355715E-127, 9.23297861778574E-128, 4.61648930889287E-128, 
2.30824465444643E-128, 1.15412232722322E-128, 5.77061163611608E-129, 2.88530581805804E-129, 1.44265290902902E-129, 7.21326454514511E-130, 3.60663227257255E-130, 1.80331613628628E-130, 
9.01658068143138E-131, 4.50829034071569E-131, 2.25414517035785E-131, 1.12707258517892E-131, 5.63536292589461E-132, 2.81768146294731E-132, 1.40884073147365E-132, 7.04420365736827E-133, 
3.52210182868413E-133, 1.76105091434207E-133, 8.80525457171033E-134, 4.40262728585517E-134, 2.20131364292758E-134, 1.10065682146379E-134, 5.50328410731896E-135, 2.75164205365948E-135, 
1.37582102682974E-135, 6.8791051341487E-136, 3.43955256707435E-136, 1.71977628353717E-136, 8.59888141768587E-137, 4.29944070884294E-137, 2.14972035442147E-137, 1.07486017721073E-137, 
5.37430088605367E-138, 2.68715044302684E-138, 1.34357522151342E-138, 6.71787610756709E-139, 3.35893805378354E-139, 1.67946902689177E-139, 8.39734513445886E-140, 4.19867256722943E-140, 
2.09933628361472E-140, 1.04966814180736E-140, 5.24834070903679E-141, 2.62417035451839E-141, 1.3120851772592E-141, 6.56042588629599E-142, 3.28021294314799E-142, 1.640106471574E-142, 
8.20053235786998E-143, 4.10026617893499E-143, 2.0501330894675E-143, 1.02506654473375E-143, 5.12533272366874E-144, 2.56266636183437E-144, 1.28133318091718E-144, 6.40666590458592E-145, 
3.20333295229296E-145, 1.60166647614648E-145, 8.0083323807324E-146, 4.0041661903662E-146, 2.0020830951831E-146, 1.00104154759155E-146, 5.00520773795775E-147, 2.50260386897888E-147, 
1.25130193448944E-147, 6.25650967244719E-148, 3.1282548362236E-148, 1.5641274181118E-148, 7.82063709055899E-149, 3.91031854527949E-149, 1.95515927263975E-149, 9.77579636319874E-150, 
4.88789818159937E-150, 2.44394909079968E-150, 1.22197454539984E-150, 6.10987272699921E-151, 3.0549363634996E-151, 1.5274681817498E-151, 7.63734090874901E-152, 3.81867045437451E-152, 
1.90933522718725E-152, 9.54667613593626E-153, 4.77333806796813E-153, 2.38666903398407E-153, 1.19333451699203E-153, 5.96667258496017E-154, 2.98333629248008E-154, 1.49166814624004E-154, 
7.45834073120021E-155, 3.7291703656001E-155, 1.86458518280005E-155, 9.32292591400026E-156, 4.66146295700013E-156, 2.33073147850006E-156, 1.16536573925003E-156, 5.82682869625016E-157, 
2.91341434812508E-157, 1.45670717406254E-157, 7.2835358703127E-158, 3.64176793515635E-158, 1.82088396757818E-158, 9.10441983789088E-159, 4.55220991894544E-159, 2.27610495947272E-159, 
1.13805247973636E-159, 5.6902623986818E-160, 2.8451311993409E-160, 1.42256559967045E-160, 7.11282799835225E-161, 3.55641399917612E-161, 1.77820699958806E-161, 8.89103499794031E-162, 
4.44551749897016E-162, 2.22275874948508E-162, 1.11137937474254E-162, 5.55689687371269E-163, 2.77844843685635E-163, 1.38922421842817E-163, 6.94612109214087E-164, 3.47306054607043E-164, 
1.73653027303522E-164, 8.68265136517608E-165, 4.34132568258804E-165, 2.17066284129402E-165, 1.08533142064701E-165, 5.42665710323505E-166, 2.71332855161753E-166, 1.35666427580876E-166, 
6.78332137904382E-167, 3.39166068952191E-167, 1.69583034476095E-167, 8.47915172380477E-168, 4.23957586190238E-168, 2.11978793095119E-168, 1.0598939654756E-168, 5.29946982737798E-169, 
2.64973491368899E-169, 1.3248674568445E-169, 6.62433728422248E-170, 3.31216864211124E-170, 1.65608432105562E-170, 8.2804216052781E-171, 4.14021080263905E-171, 2.07010540131952E-171, 
1.03505270065976E-171, 5.17526350329881E-172, 2.5876317516494E-172, 1.2938158758247E-172, 6.46907937912351E-173, 3.23453968956176E-173, 1.61726984478088E-173, 8.08634922390439E-174, 
4.04317461195219E-174, 2.0215873059761E-174, 1.01079365298805E-174, 5.05396826494024E-175, 2.52698413247012E-175, 1.26349206623506E-175, 6.3174603311753E-176, 3.15873016558765E-176, 
1.57936508279383E-176, 7.89682541396913E-177, 3.94841270698457E-177, 1.97420635349228E-177, 9.87103176746141E-178, 4.93551588373071E-178, 2.46775794186535E-178, 1.23387897093268E-178, 
6.16939485466338E-179, 3.08469742733169E-179, 1.54234871366585E-179, 7.71174356832923E-180, 3.85587178416461E-180, 1.92793589208231E-180, 9.63967946041154E-181, 4.81983973020577E-181, 
2.40991986510288E-181, 1.20495993255144E-181, 6.02479966275721E-182, 3.01239983137861E-182, 1.5061999156893E-182, 7.53099957844651E-183, 3.76549978922326E-183, 1.88274989461163E-183, 
9.41374947305814E-184, 4.70687473652907E-184, 2.35343736826454E-184, 1.17671868413227E-184, 5.88359342066134E-185, 2.94179671033067E-185, 1.47089835516533E-185, 7.35449177582667E-186, 
3.67724588791334E-186, 1.83862294395667E-186, 9.19311471978334E-187, 4.59655735989167E-187, 2.29827867994584E-187, 1.14913933997292E-187, 5.74569669986459E-188, 2.87284834993229E-188, 
1.43642417496615E-188, 7.18212087483074E-189, 3.59106043741537E-189, 1.79553021870768E-189, 8.97765109353842E-190, 4.48882554676921E-190, 2.2444127733846E-190, 1.1222063866923E-190, 
5.61103193346151E-191, 2.80551596673076E-191, 1.40275798336538E-191, 7.01378991682689E-192, 3.50689495841344E-192, 1.75344747920672E-192, 8.76723739603361E-193, 4.38361869801681E-193, 
2.1918093490084E-193, 1.0959046745042E-193, 5.47952337252101E-194, 2.7397616862605E-194, 1.36988084313025E-194, 6.84940421565126E-195, 3.42470210782563E-195, 1.71235105391281E-195, 
8.56175526956407E-196, 4.28087763478204E-196, 2.14043881739102E-196, 1.07021940869551E-196, 5.35109704347755E-197, 2.67554852173877E-197, 1.33777426086939E-197, 6.68887130434693E-198, 
3.34443565217347E-198, 1.67221782608673E-198, 8.36108913043367E-199, 4.18054456521683E-199, 2.09027228260842E-199, 1.04513614130421E-199, 5.22568070652104E-200, 2.61284035326052E-200, 
1.30642017663026E-200, 6.5321008831513E-201, 3.26605044157565E-201, 1.63302522078783E-201, 8.16512610393913E-202, 4.08256305196956E-202, 2.04128152598478E-202, 1.02064076299239E-202, 
5.10320381496195E-203, 2.55160190748098E-203, 1.27580095374049E-203, 6.37900476870244E-204, 3.18950238435122E-204, 1.59475119217561E-204, 7.97375596087805E-205, 3.98687798043903E-205, 
1.99343899021951E-205, 9.96719495109757E-206, 4.98359747554878E-206, 2.49179873777439E-206, 1.2458993688872E-206, 6.22949684443598E-207, 3.11474842221799E-207, 1.55737421110899E-207, 
7.78687105554497E-208, 3.89343552777249E-208, 1.94671776388624E-208, 9.73358881943122E-209, 4.86679440971561E-209, 2.4333972048578E-209, 1.2166986024289E-209, 6.08349301214451E-210, 
3.04174650607226E-210, 1.52087325303613E-210, 7.60436626518064E-211, 3.80218313259032E-211, 1.90109156629516E-211, 9.5054578314758E-212, 4.7527289157379E-212, 2.37636445786895E-212, 
1.18818222893447E-212, 5.94091114467237E-213, 2.97045557233619E-213, 1.48522778616809E-213, 7.42613893084047E-214, 3.71306946542023E-214, 1.85653473271012E-214, 9.28267366355059E-215, 
4.64133683177529E-215, 2.32066841588765E-215, 1.16033420794382E-215, 5.80167103971912E-216, 2.90083551985956E-216, 1.45041775992978E-216, 7.25208879964889E-217, 3.62604439982445E-217, 
1.81302219991222E-217, 9.06511099956112E-218, 4.53255549978056E-218, 2.26627774989028E-218, 1.13313887494514E-218, 5.6656943747257E-219, 2.83284718736285E-219, 1.41642359368142E-219, 
7.08211796840712E-220, 3.54105898420356E-220, 1.77052949210178E-220, 8.8526474605089E-221, 4.42632373025445E-221, 2.21316186512723E-221, 1.10658093256361E-221, 5.53290466281807E-222, 
2.76645233140903E-222, 1.38322616570452E-222, 6.91613082852258E-223, 3.45806541426129E-223, 1.72903270713065E-223, 8.64516353565323E-224, 4.32258176782661E-224, 2.16129088391331E-224, 
1.08064544195665E-224, 5.40322720978327E-225, 2.70161360489163E-225, 1.35080680244582E-225, 6.75403401222908E-226, 3.37701700611454E-226, 1.68850850305727E-226, 8.44254251528635E-227, 
4.22127125764318E-227, 2.11063562882159E-227, 1.05531781441079E-227, 5.27658907205397E-228, 2.63829453602699E-228, 1.31914726801349E-228, 6.59573634006746E-229, 3.29786817003373E-229, 
1.64893408501687E-229, 8.24467042508433E-230, 4.12233521254217E-230, 2.06116760627108E-230, 1.03058380313554E-230, 5.15291901567771E-231, 2.57645950783885E-231, 1.28822975391943E-231, 
6.44114876959713E-232, 3.22057438479857E-232, 1.61028719239928E-232, 8.05143596199642E-233, 4.02571798099821E-233, 2.0128589904991E-233, 1.00642949524955E-233, 5.03214747624776E-234, 
2.51607373812388E-234, 1.25803686906194E-234, 6.2901843453097E-235, 3.14509217265485E-235, 1.57254608632743E-235, 7.86273043163713E-236, 3.93136521581856E-236, 1.96568260790928E-236, 
9.82841303954641E-237, 4.9142065197732E-237, 2.4571032598866E-237, 1.2285516299433E-237, 6.1427581497165E-238, 3.07137907485825E-238, 1.53568953742913E-238, 7.67844768714563E-239, 
3.83922384357282E-239, 1.91961192178641E-239, 9.59805960893204E-240, 4.79902980446602E-240, 2.39951490223301E-240, 1.1997574511165E-240, 5.99878725558252E-241, 2.99939362779126E-241, 
1.49969681389563E-241, 7.49848406947815E-242, 3.74924203473908E-242, 1.87462101736954E-242, 9.37310508684769E-243, 4.68655254342385E-243, 2.34327627171192E-243, 1.17163813585596E-243, 
5.85819067927981E-244, 2.9290953396399E-244, 1.46454766981995E-244, 7.32273834909976E-245, 3.66136917454988E-245, 1.83068458727494E-245, 9.1534229363747E-246, 4.57671146818735E-246, 
2.28835573409368E-246, 1.14417786704684E-246, 5.72088933523419E-247, 2.86044466761709E-247, 1.43022233380855E-247, 7.15111166904273E-248, 3.57555583452137E-248, 1.78777791726068E-248, 
8.93888958630342E-249, 4.46944479315171E-249, 2.23472239657585E-249, 1.11736119828793E-249, 5.58680599143964E-250, 2.79340299571982E-250, 1.39670149785991E-250, 6.98350748929955E-251, 
3.49175374464977E-251, 1.74587687232489E-251, 8.72938436162443E-252, 4.36469218081222E-252, 2.18234609040611E-252, 1.09117304520305E-252, 5.45586522601527E-253, 2.72793261300764E-253, 
1.36396630650382E-253, 6.81983153251909E-254, 3.40991576625954E-254, 1.70495788312977E-254, 8.52478941564886E-255, 4.26239470782443E-255, 2.13119735391221E-255, 1.06559867695611E-255, 
5.32799338478054E-256, 2.66399669239027E-256, 1.33199834619513E-256, 6.65999173097567E-257, 3.32999586548784E-257, 1.66499793274392E-257, 8.32498966371959E-258, 4.16249483185979E-258, 
2.0812474159299E-258, 1.04062370796495E-258, 5.20311853982474E-259, 2.60155926991237E-259, 1.30077963495619E-259, 6.50389817478093E-260, 3.25194908739046E-260, 1.62597454369523E-260, 
8.12987271847616E-261, 4.06493635923808E-261, 2.03246817961904E-261, 1.01623408980952E-261, 5.0811704490476E-262, 2.5405852245238E-262, 1.2702926122619E-262, 6.3514630613095E-263, 
3.17573153065475E-263, 1.58786576532738E-263, 7.93932882663688E-264, 3.96966441331844E-264, 1.98483220665922E-264, 9.9241610332961E-265, 4.96208051664805E-265, 2.48104025832402E-265, 
1.24052012916201E-265, 6.20260064581006E-266, 3.10130032290503E-266, 1.55065016145251E-266, 7.75325080726257E-267, 3.87662540363129E-267, 1.93831270181564E-267, 9.69156350907822E-268, 
4.84578175453911E-268, 2.42289087726955E-268, 1.21144543863478E-268, 6.05722719317389E-269, 3.02861359658694E-269, 1.51430679829347E-269, 7.57153399146736E-270, 3.78576699573368E-270, 
1.89288349786684E-270, 9.4644174893342E-271, 4.7322087446671E-271, 2.36610437233355E-271, 1.18305218616677E-271, 5.91526093083387E-272, 2.95763046541694E-272, 1.47881523270847E-272, 
7.39407616354234E-273, 3.69703808177117E-273, 1.84851904088559E-273, 9.24259520442793E-274, 4.62129760221396E-274, 2.31064880110698E-274, 1.15532440055349E-274, 5.77662200276745E-275, 
2.88831100138373E-275, 1.44415550069186E-275, 7.22077750345932E-276, 3.61038875172966E-276, 1.80519437586483E-276, 9.02597187932415E-277, 4.51298593966207E-277, 2.25649296983104E-277, 
1.12824648491552E-277, 5.64123242457759E-278, 2.8206162122888E-278, 1.4103081061444E-278, 7.05154053072199E-279, 3.525770265361E-279, 1.7628851326805E-279, 8.81442566340249E-280, 
4.40721283170124E-280, 2.20360641585062E-280, 1.10180320792531E-280, 5.50901603962656E-281, 2.75450801981328E-281, 1.37725400990664E-281, 6.88627004953319E-282, 3.4431350247666E-282, 
1.7215675123833E-282, 8.60783756191649E-283, 4.30391878095825E-283, 2.15195939047912E-283, 1.07597969523956E-283, 5.37989847619781E-284, 2.6899492380989E-284, 1.34497461904945E-284, 
6.72487309524726E-285, 3.36243654762363E-285, 1.68121827381181E-285, 8.40609136905907E-286, 4.20304568452954E-286, 2.10152284226477E-286, 1.05076142113238E-286, 5.25380710566192E-287, 
2.62690355283096E-287, 1.31345177641548E-287, 6.5672588820774E-288, 3.2836294410387E-288, 1.64181472051935E-288, 8.20907360259675E-289, 4.10453680129838E-289, 2.05226840064919E-289, 
1.02613420032459E-289, 5.13067100162297E-290, 2.56533550081149E-290, 1.28266775040574E-290, 6.41333875202871E-291, 3.20666937601436E-291, 1.60333468800718E-291, 8.01667344003589E-292, 
4.00833672001795E-292, 2.00416836000897E-292, 1.00208418000449E-292, 5.01042090002243E-293, 2.50521045001122E-293, 1.25260522500561E-293, 6.26302612502804E-294, 3.13151306251402E-294, 
1.56575653125701E-294, 7.82878265628505E-295, 3.91439132814253E-295, 1.95719566407126E-295, 9.78597832035631E-296, 4.89298916017816E-296, 2.44649458008908E-296, 1.22324729004454E-296, 
6.1162364502227E-297, 3.05811822511135E-297, 1.52905911255567E-297, 7.64529556277837E-298, 3.82264778138918E-298, 1.91132389069459E-298, 9.55661945347296E-299, 4.77830972673648E-299, 
2.38915486336824E-299, 1.19457743168412E-299, 5.9728871584206E-300, 2.9864435792103E-300, 1.49322178960515E-300, 7.46610894802575E-301, 3.73305447401288E-301, 1.86652723700644E-301, 
9.33263618503219E-302, 4.66631809251609E-302, 2.33315904625805E-302, 1.16657952312902E-302, 5.83289761564512E-303, 2.91644880782256E-303, 1.45822440391128E-303, 7.2911220195564E-304, 
3.6455610097782E-304, 1.8227805048891E-304, 9.1139025244455E-305, 4.55695126222275E-305, 2.27847563111137E-305, 1.13923781555569E-305, 5.69618907777844E-306, 2.84809453888922E-306, 
1.42404726944461E-306, 7.12023634722304E-307, 3.56011817361152E-307, 1.78005908680576E-307, 8.90029543402881E-308, 4.4501477170144E-308, 2.2250738585072E-308
};

//int GetNumberOfBitsForNonNegativeInteger(asn1SccUint v) 
int mylog2(double v)
{
    if (v<2) {
        v=1.0/v;
        return -GetNumberOfBitsForNonNegativeInteger((asn1SccUint)v);
    }

    return GetNumberOfBitsForNonNegativeInteger((asn1SccUint)v);
}


double log2(double v) 
{
    return log(v)/log(2.0);
}

double mypow2(int exp)
{
    asn1SccUint ret=1;
    if (exp>=0) {
/*      while(exp) 
        {
            ret<<=1;
            exp--;
        }*/
        return pospow2[exp];
    } else {
        exp = -exp;
/*      while(exp) 
        {
            ret<<=1;
            exp--;
        }
        return 1.0/(double)ret;*/
        return negpow2[exp];
    }
}

double pow2(double v)
{
    return pow(2.0, v);
}

double myReal(asn1SccUint* mantissa, int exp) 
{
    return (*mantissa) * mypow2(exp);
//  return (*mantissa) * pow2(exp);
}


void CalculateMantissaAndExponent(double d, int* exp, asn1SccUint64* mantissa)
{
    double error;
    double dmantissa;
    int nCount=100;

    assert(d>0.0);

    /*
    Let mantissa be 1
    then exponent is the logarithm of the input value.
    However, since we need the exponent to be stored in an INT we get the Floor 
    Floor return the largest integer less than or equal to the specified double-precision floating-point number
    */
    *exp = mylog2(d);

    /*
        Since exponent was 'Floored' mantissa is not 1 anymore but the following value
        now mantissa has a value in the range [1..base)
    */
    dmantissa = d/mypow2(*exp);
    *mantissa = (asn1SccUint64)dmantissa;

    error = fabs((double)(d-myReal(mantissa,*exp)))/d;
    while ( (*mantissa <= MAX_MANTISSA) && (error > DBL_EPSILON) && nCount--)
    {
        dmantissa *=2;
        *mantissa = (asn1SccUint64)dmantissa;
        (*exp)--;
        error = fabs((double)(d-myReal(mantissa,*exp)))/d;
    }
}

#endif
