"use strict";
/**
 * Copyright (c) "Neo4j"
 * Neo4j Sweden AB [http://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        if (typeof b !== "function" && b !== null)
            throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.Unpacker = exports.Packer = void 0;
var v1 = __importStar(require("./packstream-v1"));
var neo4j_driver_core_1 = require("neo4j-driver-core");
var temporal_factory_1 = require("./temporal-factory");
var _a = neo4j_driver_core_1.internal.temporalUtil, dateToEpochDay = _a.dateToEpochDay, localDateTimeToEpochSecond = _a.localDateTimeToEpochSecond, localTimeToNanoOfDay = _a.localTimeToNanoOfDay;
var POINT_2D = 0x58;
var POINT_2D_STRUCT_SIZE = 3;
var POINT_3D = 0x59;
var POINT_3D_STRUCT_SIZE = 4;
var DURATION = 0x45;
var DURATION_STRUCT_SIZE = 4;
var LOCAL_TIME = 0x74;
var LOCAL_TIME_STRUCT_SIZE = 1;
var TIME = 0x54;
var TIME_STRUCT_SIZE = 2;
var DATE = 0x44;
var DATE_STRUCT_SIZE = 1;
var LOCAL_DATE_TIME = 0x64;
var LOCAL_DATE_TIME_STRUCT_SIZE = 2;
var DATE_TIME_WITH_ZONE_OFFSET = 0x46;
var DATE_TIME_WITH_ZONE_OFFSET_STRUCT_SIZE = 3;
var DATE_TIME_WITH_ZONE_ID = 0x66;
var DATE_TIME_WITH_ZONE_ID_STRUCT_SIZE = 3;
var Packer = /** @class */ (function (_super) {
    __extends(Packer, _super);
    function Packer() {
        return _super !== null && _super.apply(this, arguments) || this;
    }
    Packer.prototype.disableByteArrays = function () {
        throw new Error('Bolt V2 should always support byte arrays');
    };
    Packer.prototype.packable = function (obj) {
        var _this = this;
        if ((0, neo4j_driver_core_1.isPoint)(obj)) {
            return function () { return packPoint(obj, _this); };
        }
        else if ((0, neo4j_driver_core_1.isDuration)(obj)) {
            return function () { return packDuration(obj, _this); };
        }
        else if ((0, neo4j_driver_core_1.isLocalTime)(obj)) {
            return function () { return packLocalTime(obj, _this); };
        }
        else if ((0, neo4j_driver_core_1.isTime)(obj)) {
            return function () { return packTime(obj, _this); };
        }
        else if ((0, neo4j_driver_core_1.isDate)(obj)) {
            return function () { return packDate(obj, _this); };
        }
        else if ((0, neo4j_driver_core_1.isLocalDateTime)(obj)) {
            return function () { return packLocalDateTime(obj, _this); };
        }
        else if ((0, neo4j_driver_core_1.isDateTime)(obj)) {
            return function () { return packDateTime(obj, _this); };
        }
        else {
            return _super.prototype.packable.call(this, obj);
        }
    };
    return Packer;
}(v1.Packer));
exports.Packer = Packer;
var Unpacker = /** @class */ (function (_super) {
    __extends(Unpacker, _super);
    /**
     * @constructor
     * @param {boolean} disableLosslessIntegers if this unpacker should convert all received integers to native JS numbers.
     * @param {boolean} useBigInt if this unpacker should convert all received integers to Bigint
     */
    function Unpacker(disableLosslessIntegers, useBigInt) {
        if (disableLosslessIntegers === void 0) { disableLosslessIntegers = false; }
        if (useBigInt === void 0) { useBigInt = false; }
        return _super.call(this, disableLosslessIntegers, useBigInt) || this;
    }
    Unpacker.prototype._unpackUnknownStruct = function (signature, structSize, buffer) {
        if (signature === POINT_2D) {
            return unpackPoint2D(this, structSize, buffer);
        }
        else if (signature === POINT_3D) {
            return unpackPoint3D(this, structSize, buffer);
        }
        else if (signature === DURATION) {
            return unpackDuration(this, structSize, buffer);
        }
        else if (signature === LOCAL_TIME) {
            return unpackLocalTime(this, structSize, buffer, this._disableLosslessIntegers, this._useBigInt);
        }
        else if (signature === TIME) {
            return unpackTime(this, structSize, buffer, this._disableLosslessIntegers, this._useBigInt);
        }
        else if (signature === DATE) {
            return unpackDate(this, structSize, buffer, this._disableLosslessIntegers, this._useBigInt);
        }
        else if (signature === LOCAL_DATE_TIME) {
            return unpackLocalDateTime(this, structSize, buffer, this._disableLosslessIntegers, this._useBigInt);
        }
        else if (signature === DATE_TIME_WITH_ZONE_OFFSET) {
            return unpackDateTimeWithZoneOffset(this, structSize, buffer, this._disableLosslessIntegers, this._useBigInt);
        }
        else if (signature === DATE_TIME_WITH_ZONE_ID) {
            return unpackDateTimeWithZoneId(this, structSize, buffer, this._disableLosslessIntegers, this._useBigInt);
        }
        else {
            return _super.prototype._unpackUnknownStruct.call(this, signature, structSize, buffer, this._disableLosslessIntegers, this._useBigInt);
        }
    };
    return Unpacker;
}(v1.Unpacker));
exports.Unpacker = Unpacker;
/**
 * Pack given 2D or 3D point.
 * @param {Point} point the point value to pack.
 * @param {Packer} packer the packer to use.
 */
function packPoint(point, packer) {
    var is2DPoint = point.z === null || point.z === undefined;
    if (is2DPoint) {
        packPoint2D(point, packer);
    }
    else {
        packPoint3D(point, packer);
    }
}
/**
 * Pack given 2D point.
 * @param {Point} point the point value to pack.
 * @param {Packer} packer the packer to use.
 */
function packPoint2D(point, packer) {
    var packableStructFields = [
        packer.packable((0, neo4j_driver_core_1.int)(point.srid)),
        packer.packable(point.x),
        packer.packable(point.y)
    ];
    packer.packStruct(POINT_2D, packableStructFields);
}
/**
 * Pack given 3D point.
 * @param {Point} point the point value to pack.
 * @param {Packer} packer the packer to use.
 */
function packPoint3D(point, packer) {
    var packableStructFields = [
        packer.packable((0, neo4j_driver_core_1.int)(point.srid)),
        packer.packable(point.x),
        packer.packable(point.y),
        packer.packable(point.z)
    ];
    packer.packStruct(POINT_3D, packableStructFields);
}
/**
 * Unpack 2D point value using the given unpacker.
 * @param {Unpacker} unpacker the unpacker to use.
 * @param {number} structSize the retrieved struct size.
 * @param {BaseBuffer} buffer the buffer to unpack from.
 * @return {Point} the unpacked 2D point value.
 */
function unpackPoint2D(unpacker, structSize, buffer) {
    unpacker._verifyStructSize('Point2D', POINT_2D_STRUCT_SIZE, structSize);
    return new neo4j_driver_core_1.Point(unpacker.unpack(buffer), // srid
    unpacker.unpack(buffer), // x
    unpacker.unpack(buffer), // y
    undefined // z
    );
}
/**
 * Unpack 3D point value using the given unpacker.
 * @param {Unpacker} unpacker the unpacker to use.
 * @param {number} structSize the retrieved struct size.
 * @param {BaseBuffer} buffer the buffer to unpack from.
 * @return {Point} the unpacked 3D point value.
 */
function unpackPoint3D(unpacker, structSize, buffer) {
    unpacker._verifyStructSize('Point3D', POINT_3D_STRUCT_SIZE, structSize);
    return new neo4j_driver_core_1.Point(unpacker.unpack(buffer), // srid
    unpacker.unpack(buffer), // x
    unpacker.unpack(buffer), // y
    unpacker.unpack(buffer) // z
    );
}
/**
 * Pack given duration.
 * @param {Duration} value the duration value to pack.
 * @param {Packer} packer the packer to use.
 */
function packDuration(value, packer) {
    var months = (0, neo4j_driver_core_1.int)(value.months);
    var days = (0, neo4j_driver_core_1.int)(value.days);
    var seconds = (0, neo4j_driver_core_1.int)(value.seconds);
    var nanoseconds = (0, neo4j_driver_core_1.int)(value.nanoseconds);
    var packableStructFields = [
        packer.packable(months),
        packer.packable(days),
        packer.packable(seconds),
        packer.packable(nanoseconds)
    ];
    packer.packStruct(DURATION, packableStructFields);
}
/**
 * Unpack duration value using the given unpacker.
 * @param {Unpacker} unpacker the unpacker to use.
 * @param {number} structSize the retrieved struct size.
 * @param {BaseBuffer} buffer the buffer to unpack from.
 * @return {Duration} the unpacked duration value.
 */
function unpackDuration(unpacker, structSize, buffer) {
    unpacker._verifyStructSize('Duration', DURATION_STRUCT_SIZE, structSize);
    var months = unpacker.unpack(buffer);
    var days = unpacker.unpack(buffer);
    var seconds = unpacker.unpack(buffer);
    var nanoseconds = unpacker.unpack(buffer);
    return new neo4j_driver_core_1.Duration(months, days, seconds, nanoseconds);
}
/**
 * Pack given local time.
 * @param {LocalTime} value the local time value to pack.
 * @param {Packer} packer the packer to use.
 */
function packLocalTime(value, packer) {
    var nanoOfDay = localTimeToNanoOfDay(value.hour, value.minute, value.second, value.nanosecond);
    var packableStructFields = [packer.packable(nanoOfDay)];
    packer.packStruct(LOCAL_TIME, packableStructFields);
}
/**
 * Unpack local time value using the given unpacker.
 * @param {Unpacker} unpacker the unpacker to use.
 * @param {number} structSize the retrieved struct size.
 * @param {BaseBuffer} buffer the buffer to unpack from.
 * @param {boolean} disableLosslessIntegers if integer properties in the result local time should be native JS numbers.
 * @return {LocalTime} the unpacked local time value.
 */
function unpackLocalTime(unpacker, structSize, buffer, disableLosslessIntegers) {
    unpacker._verifyStructSize('LocalTime', LOCAL_TIME_STRUCT_SIZE, structSize);
    var nanoOfDay = unpacker.unpackInteger(buffer);
    var result = (0, temporal_factory_1.nanoOfDayToLocalTime)(nanoOfDay);
    return convertIntegerPropsIfNeeded(result, disableLosslessIntegers);
}
/**
 * Pack given time.
 * @param {Time} value the time value to pack.
 * @param {Packer} packer the packer to use.
 */
function packTime(value, packer) {
    var nanoOfDay = localTimeToNanoOfDay(value.hour, value.minute, value.second, value.nanosecond);
    var offsetSeconds = (0, neo4j_driver_core_1.int)(value.timeZoneOffsetSeconds);
    var packableStructFields = [
        packer.packable(nanoOfDay),
        packer.packable(offsetSeconds)
    ];
    packer.packStruct(TIME, packableStructFields);
}
/**
 * Unpack time value using the given unpacker.
 * @param {Unpacker} unpacker the unpacker to use.
 * @param {number} structSize the retrieved struct size.
 * @param {BaseBuffer} buffer the buffer to unpack from.
 * @param {boolean} disableLosslessIntegers if integer properties in the result time should be native JS numbers.
 * @return {Time} the unpacked time value.
 */
function unpackTime(unpacker, structSize, buffer, disableLosslessIntegers, useBigInt) {
    unpacker._verifyStructSize('Time', TIME_STRUCT_SIZE, structSize);
    var nanoOfDay = unpacker.unpackInteger(buffer);
    var offsetSeconds = unpacker.unpackInteger(buffer);
    var localTime = (0, temporal_factory_1.nanoOfDayToLocalTime)(nanoOfDay);
    var result = new neo4j_driver_core_1.Time(localTime.hour, localTime.minute, localTime.second, localTime.nanosecond, offsetSeconds);
    return convertIntegerPropsIfNeeded(result, disableLosslessIntegers, useBigInt);
}
/**
 * Pack given neo4j date.
 * @param {Date} value the date value to pack.
 * @param {Packer} packer the packer to use.
 */
function packDate(value, packer) {
    var epochDay = dateToEpochDay(value.year, value.month, value.day);
    var packableStructFields = [packer.packable(epochDay)];
    packer.packStruct(DATE, packableStructFields);
}
/**
 * Unpack neo4j date value using the given unpacker.
 * @param {Unpacker} unpacker the unpacker to use.
 * @param {number} structSize the retrieved struct size.
 * @param {BaseBuffer} buffer the buffer to unpack from.
 * @param {boolean} disableLosslessIntegers if integer properties in the result date should be native JS numbers.
 * @return {Date} the unpacked neo4j date value.
 */
function unpackDate(unpacker, structSize, buffer, disableLosslessIntegers, useBigInt) {
    unpacker._verifyStructSize('Date', DATE_STRUCT_SIZE, structSize);
    var epochDay = unpacker.unpackInteger(buffer);
    var result = (0, temporal_factory_1.epochDayToDate)(epochDay);
    return convertIntegerPropsIfNeeded(result, disableLosslessIntegers, useBigInt);
}
/**
 * Pack given local date time.
 * @param {LocalDateTime} value the local date time value to pack.
 * @param {Packer} packer the packer to use.
 */
function packLocalDateTime(value, packer) {
    var epochSecond = localDateTimeToEpochSecond(value.year, value.month, value.day, value.hour, value.minute, value.second, value.nanosecond);
    var nano = (0, neo4j_driver_core_1.int)(value.nanosecond);
    var packableStructFields = [
        packer.packable(epochSecond),
        packer.packable(nano)
    ];
    packer.packStruct(LOCAL_DATE_TIME, packableStructFields);
}
/**
 * Unpack local date time value using the given unpacker.
 * @param {Unpacker} unpacker the unpacker to use.
 * @param {number} structSize the retrieved struct size.
 * @param {BaseBuffer} buffer the buffer to unpack from.
 * @param {boolean} disableLosslessIntegers if integer properties in the result local date-time should be native JS numbers.
 * @return {LocalDateTime} the unpacked local date time value.
 */
function unpackLocalDateTime(unpacker, structSize, buffer, disableLosslessIntegers, useBigInt) {
    unpacker._verifyStructSize('LocalDateTime', LOCAL_DATE_TIME_STRUCT_SIZE, structSize);
    var epochSecond = unpacker.unpackInteger(buffer);
    var nano = unpacker.unpackInteger(buffer);
    var result = (0, temporal_factory_1.epochSecondAndNanoToLocalDateTime)(epochSecond, nano);
    return convertIntegerPropsIfNeeded(result, disableLosslessIntegers, useBigInt);
}
/**
 * Pack given date time.
 * @param {DateTime} value the date time value to pack.
 * @param {Packer} packer the packer to use.
 */
function packDateTime(value, packer) {
    if (value.timeZoneId) {
        packDateTimeWithZoneId(value, packer);
    }
    else {
        packDateTimeWithZoneOffset(value, packer);
    }
}
/**
 * Pack given date time with zone offset.
 * @param {DateTime} value the date time value to pack.
 * @param {Packer} packer the packer to use.
 */
function packDateTimeWithZoneOffset(value, packer) {
    var epochSecond = localDateTimeToEpochSecond(value.year, value.month, value.day, value.hour, value.minute, value.second, value.nanosecond);
    var nano = (0, neo4j_driver_core_1.int)(value.nanosecond);
    var timeZoneOffsetSeconds = (0, neo4j_driver_core_1.int)(value.timeZoneOffsetSeconds);
    var packableStructFields = [
        packer.packable(epochSecond),
        packer.packable(nano),
        packer.packable(timeZoneOffsetSeconds)
    ];
    packer.packStruct(DATE_TIME_WITH_ZONE_OFFSET, packableStructFields);
}
/**
 * Unpack date time with zone offset value using the given unpacker.
 * @param {Unpacker} unpacker the unpacker to use.
 * @param {number} structSize the retrieved struct size.
 * @param {BaseBuffer} buffer the buffer to unpack from.
 * @param {boolean} disableLosslessIntegers if integer properties in the result date-time should be native JS numbers.
 * @return {DateTime} the unpacked date time with zone offset value.
 */
function unpackDateTimeWithZoneOffset(unpacker, structSize, buffer, disableLosslessIntegers, useBigInt) {
    unpacker._verifyStructSize('DateTimeWithZoneOffset', DATE_TIME_WITH_ZONE_OFFSET_STRUCT_SIZE, structSize);
    var epochSecond = unpacker.unpackInteger(buffer);
    var nano = unpacker.unpackInteger(buffer);
    var timeZoneOffsetSeconds = unpacker.unpackInteger(buffer);
    var localDateTime = (0, temporal_factory_1.epochSecondAndNanoToLocalDateTime)(epochSecond, nano);
    var result = new neo4j_driver_core_1.DateTime(localDateTime.year, localDateTime.month, localDateTime.day, localDateTime.hour, localDateTime.minute, localDateTime.second, localDateTime.nanosecond, timeZoneOffsetSeconds, null);
    return convertIntegerPropsIfNeeded(result, disableLosslessIntegers, useBigInt);
}
/**
 * Pack given date time with zone id.
 * @param {DateTime} value the date time value to pack.
 * @param {Packer} packer the packer to use.
 */
function packDateTimeWithZoneId(value, packer) {
    var epochSecond = localDateTimeToEpochSecond(value.year, value.month, value.day, value.hour, value.minute, value.second, value.nanosecond);
    var nano = (0, neo4j_driver_core_1.int)(value.nanosecond);
    var timeZoneId = value.timeZoneId;
    var packableStructFields = [
        packer.packable(epochSecond),
        packer.packable(nano),
        packer.packable(timeZoneId)
    ];
    packer.packStruct(DATE_TIME_WITH_ZONE_ID, packableStructFields);
}
/**
 * Unpack date time with zone id value using the given unpacker.
 * @param {Unpacker} unpacker the unpacker to use.
 * @param {number} structSize the retrieved struct size.
 * @param {BaseBuffer} buffer the buffer to unpack from.
 * @param {boolean} disableLosslessIntegers if integer properties in the result date-time should be native JS numbers.
 * @return {DateTime} the unpacked date time with zone id value.
 */
function unpackDateTimeWithZoneId(unpacker, structSize, buffer, disableLosslessIntegers, useBigInt) {
    unpacker._verifyStructSize('DateTimeWithZoneId', DATE_TIME_WITH_ZONE_ID_STRUCT_SIZE, structSize);
    var epochSecond = unpacker.unpackInteger(buffer);
    var nano = unpacker.unpackInteger(buffer);
    var timeZoneId = unpacker.unpack(buffer);
    var localDateTime = (0, temporal_factory_1.epochSecondAndNanoToLocalDateTime)(epochSecond, nano);
    var result = new neo4j_driver_core_1.DateTime(localDateTime.year, localDateTime.month, localDateTime.day, localDateTime.hour, localDateTime.minute, localDateTime.second, localDateTime.nanosecond, null, timeZoneId);
    return convertIntegerPropsIfNeeded(result, disableLosslessIntegers, useBigInt);
}
function convertIntegerPropsIfNeeded(obj, disableLosslessIntegers, useBigInt) {
    if (!disableLosslessIntegers && !useBigInt) {
        return obj;
    }
    var convert = function (value) {
        return useBigInt ? value.toBigInt() : value.toNumberOrInfinity();
    };
    var clone = Object.create(Object.getPrototypeOf(obj));
    for (var prop in obj) {
        if (obj.hasOwnProperty(prop)) {
            var value = obj[prop];
            clone[prop] = (0, neo4j_driver_core_1.isInt)(value) ? convert(value) : value;
        }
    }
    Object.freeze(clone);
    return clone;
}
