import Connection from './connection';
import { ConnectionHolder } from './internal/connection-holder';
import { Bookmark } from './internal/bookmark';
import { TxConfig } from './internal/tx-config';
import Result from './result';
import { Query } from './types';
/**
 * Represents a transaction in the Neo4j database.
 *
 * @access public
 */
declare class Transaction {
    private _connectionHolder;
    private _reactive;
    private _state;
    private _onClose;
    private _onBookmark;
    private _onConnection;
    private _onError;
    private _onComplete;
    private _fetchSize;
    private _results;
    private _impersonatedUser?;
    /**
     * @constructor
     * @param {ConnectionHolder} connectionHolder - the connection holder to get connection from.
     * @param {function()} onClose - Function to be called when transaction is committed or rolled back.
     * @param {function(bookmark: Bookmark)} onBookmark callback invoked when new bookmark is produced.
     * * @param {function()} onConnection - Function to be called when a connection is obtained to ensure the conneciton
     * is not yet released.
     * @param {boolean} reactive whether this transaction generates reactive streams
     * @param {number} fetchSize - the record fetch size in each pulling batch.
     * @param {string} impersonatedUser - The name of the user which should be impersonated for the duration of the session.
     */
    constructor({ connectionHolder, onClose, onBookmark, onConnection, reactive, fetchSize, impersonatedUser }: {
        connectionHolder: ConnectionHolder;
        onClose: () => void;
        onBookmark: (bookmark: Bookmark) => void;
        onConnection: () => void;
        reactive: boolean;
        fetchSize: number;
        impersonatedUser?: string;
    });
    /**
     * @private
     * @param {Bookmark | string |  string []} bookmark
     * @param {TxConfig} txConfig
     * @returns {void}
     */
    _begin(bookmark: Bookmark | string | string[], txConfig: TxConfig): void;
    /**
     * Run Cypher query
     * Could be called with a query object i.e.: `{text: "MATCH ...", parameters: {param: 1}}`
     * or with the query and parameters as separate arguments.
     * @param {mixed} query - Cypher query to execute
     * @param {Object} parameters - Map with parameters to use in query
     * @return {Result} New Result
     */
    run(query: Query, parameters?: any): Result;
    /**
     * Commits the transaction and returns the result.
     *
     * After committing the transaction can no longer be used.
     *
     * @returns {Promise<void>} An empty promise if committed successfully or error if any error happened during commit.
     */
    commit(): Promise<void>;
    /**
     * Rollbacks the transaction.
     *
     * After rolling back, the transaction can no longer be used.
     *
     * @returns {Promise<void>} An empty promise if rolled back successfully or error if any error happened during
     * rollback.
     */
    rollback(): Promise<void>;
    /**
     * Check if this transaction is active, which means commit and rollback did not happen.
     * @return {boolean} `true` when not committed and not rolled back, `false` otherwise.
     */
    isOpen(): boolean;
    /**
     * Closes the transaction
     *
     * This method will roll back the transaction if it is not already committed or rolled back.
     *
     * @returns {Promise<void>} An empty promise if closed successfully or error if any error happened during
     */
    close(): Promise<void>;
    _onErrorCallback(err: any): Promise<Connection | void>;
    /**
     * @private
     * @param {object} meta The meta with bookmark
     * @returns {void}
     */
    _onCompleteCallback(meta: {
        bookmark?: string | string[];
    }): void;
}
export default Transaction;
