"use strict";
/**
 * Copyright (c) "Neo4j"
 * Neo4j Sweden AB [http://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __generator = (this && this.__generator) || function (thisArg, body) {
    var _ = { label: 0, sent: function() { if (t[0] & 1) throw t[1]; return t[1]; }, trys: [], ops: [] }, f, y, t, g;
    return g = { next: verb(0), "throw": verb(1), "return": verb(2) }, typeof Symbol === "function" && (g[Symbol.iterator] = function() { return this; }), g;
    function verb(n) { return function (v) { return step([n, v]); }; }
    function step(op) {
        if (f) throw new TypeError("Generator is already executing.");
        while (_) try {
            if (f = 1, y && (t = op[0] & 2 ? y["return"] : op[0] ? y["throw"] || ((t = y["return"]) && t.call(y), 0) : y.next) && !(t = t.call(y, op[1])).done) return t;
            if (y = 0, t) op = [op[0] & 2, t.value];
            switch (op[0]) {
                case 0: case 1: t = op; break;
                case 4: _.label++; return { value: op[1], done: false };
                case 5: _.label++; y = op[1]; op = [0]; continue;
                case 7: op = _.ops.pop(); _.trys.pop(); continue;
                default:
                    if (!(t = _.trys, t = t.length > 0 && t[t.length - 1]) && (op[0] === 6 || op[0] === 2)) { _ = 0; continue; }
                    if (op[0] === 3 && (!t || (op[1] > t[0] && op[1] < t[3]))) { _.label = op[1]; break; }
                    if (op[0] === 6 && _.label < t[1]) { _.label = t[1]; t = op; break; }
                    if (t && _.label < t[2]) { _.label = t[2]; _.ops.push(op); break; }
                    if (t[2]) _.ops.pop();
                    _.trys.pop(); continue;
            }
            op = body.call(thisArg, _);
        } catch (e) { op = [6, e]; y = 0; } finally { f = t = 0; }
        if (op[0] & 5) throw op[1]; return { value: op[0] ? op[1] : void 0, done: true };
    }
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
var pool_config_1 = __importDefault(require("./pool-config"));
var neo4j_driver_core_1 = require("neo4j-driver-core");
var Logger = neo4j_driver_core_1.internal.logger.Logger;
var Pool = /** @class */ (function () {
    /**
     * @param {function(address: ServerAddress, function(address: ServerAddress, resource: object): Promise<object>): Promise<object>} create
     *                an allocation function that creates a promise with a new resource. It's given an address for which to
     *                allocate the connection and a function that will return the resource to the pool if invoked, which is
     *                meant to be called on .dispose or .close or whatever mechanism the resource uses to finalize.
     * @param {function(resource: object): Promise<void>} destroy
     *                called with the resource when it is evicted from this pool
     * @param {function(resource: object): boolean} validate
     *                called at various times (like when an instance is acquired and when it is returned.
     *                If this returns false, the resource will be evicted
     * @param {function(resource: object, observer: { onError }): void} installIdleObserver
     *                called when the resource is released back to pool
     * @param {function(resource: object): void} removeIdleObserver
     *                called when the resource is acquired from the pool
     * @param {PoolConfig} config configuration for the new driver.
     * @param {Logger} log the driver logger.
     */
    function Pool(_a) {
        var _b = _a === void 0 ? {} : _a, _c = _b.create, create = _c === void 0 ? function (address, release) { return Promise.resolve(); } : _c, _d = _b.destroy, destroy = _d === void 0 ? function (conn) { return Promise.resolve(); } : _d, _e = _b.validate, validate = _e === void 0 ? function (conn) { return true; } : _e, _f = _b.installIdleObserver, installIdleObserver = _f === void 0 ? function (conn, observer) { } : _f, _g = _b.removeIdleObserver, removeIdleObserver = _g === void 0 ? function (conn) { } : _g, _h = _b.config, config = _h === void 0 ? pool_config_1.default.defaultConfig() : _h, _j = _b.log, log = _j === void 0 ? Logger.noOp() : _j;
        this._create = create;
        this._destroy = destroy;
        this._validate = validate;
        this._installIdleObserver = installIdleObserver;
        this._removeIdleObserver = removeIdleObserver;
        this._maxSize = config.maxSize;
        this._acquisitionTimeout = config.acquisitionTimeout;
        this._pools = {};
        this._pendingCreates = {};
        this._acquireRequests = {};
        this._activeResourceCounts = {};
        this._poolState = {};
        this._release = this._release.bind(this);
        this._log = log;
        this._closed = false;
    }
    /**
     * Acquire and idle resource fom the pool or create a new one.
     * @param {ServerAddress} address the address for which we're acquiring.
     * @return {Object} resource that is ready to use.
     */
    Pool.prototype.acquire = function (address) {
        var _this = this;
        return this._acquire(address).then(function (resource) {
            var key = address.asKey();
            if (resource) {
                // New or existing resource acquired
                return resource;
            }
            // We're out of resources and will try to acquire later on when an existing resource is released.
            var allRequests = _this._acquireRequests;
            var requests = allRequests[key];
            if (!requests) {
                allRequests[key] = [];
            }
            return new Promise(function (resolve, reject) {
                var request;
                var timeoutId = setTimeout(function () {
                    // acquisition timeout fired
                    // remove request from the queue of pending requests, if it's still there
                    // request might've been taken out by the release operation
                    var pendingRequests = allRequests[key];
                    if (pendingRequests) {
                        allRequests[key] = pendingRequests.filter(function (item) { return item !== request; });
                    }
                    if (request.isCompleted()) {
                        // request already resolved/rejected by the release operation; nothing to do
                    }
                    else {
                        // request is still pending and needs to be failed
                        var activeCount = _this.activeResourceCount(address);
                        var idleCount = _this.has(address) ? _this._pools[key].length : 0;
                        request.reject((0, neo4j_driver_core_1.newError)("Connection acquisition timed out in " + _this._acquisitionTimeout + " ms. Pool status: Active conn count = " + activeCount + ", Idle conn count = " + idleCount + "."));
                    }
                }, _this._acquisitionTimeout);
                request = new PendingRequest(key, resolve, reject, timeoutId, _this._log);
                allRequests[key].push(request);
            });
        });
    };
    /**
     * Destroy all idle resources for the given address.
     * @param {ServerAddress} address the address of the server to purge its pool.
     * @returns {Promise<void>} A promise that is resolved when the resources are purged
     */
    Pool.prototype.purge = function (address) {
        return this._purgeKey(address.asKey());
    };
    /**
     * Destroy all idle resources in this pool.
     * @returns {Promise<void>} A promise that is resolved when the resources are purged
     */
    Pool.prototype.close = function () {
        return __awaiter(this, void 0, void 0, function () {
            var _this = this;
            return __generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        this._closed = true;
                        return [4 /*yield*/, Promise.all(Object.keys(this._pools).map(function (key) { return _this._purgeKey(key); }))];
                    case 1: 
                    /**
                     * The lack of Promise consuming was making the driver do not close properly in the scenario
                     * captured at result.test.js:it('should handle missing onCompleted'). The test was timing out
                     * because while wainting for the driver close.
                     *
                     * Consuming the Promise.all or by calling then or by awaiting in the result inside this method solved
                     * the issue somehow.
                     *
                     * PS: the return of this method was already awaited at PooledConnectionProvider.close, but the await bellow
                     * seems to be need also.
                     */
                    return [2 /*return*/, _a.sent()];
                }
            });
        });
    };
    /**
     * Keep the idle resources for the provided addresses and purge the rest.
     * @returns {Promise<void>} A promise that is resolved when the other resources are purged
     */
    Pool.prototype.keepAll = function (addresses) {
        var _this = this;
        var keysToKeep = addresses.map(function (a) { return a.asKey(); });
        var keysPresent = Object.keys(this._pools);
        var keysToPurge = keysPresent.filter(function (k) { return keysToKeep.indexOf(k) === -1; });
        return Promise.all(keysToPurge.map(function (key) { return _this._purgeKey(key); }));
    };
    /**
     * Check if this pool contains resources for the given address.
     * @param {ServerAddress} address the address of the server to check.
     * @return {boolean} `true` when pool contains entries for the given key, <code>false</code> otherwise.
     */
    Pool.prototype.has = function (address) {
        return address.asKey() in this._pools;
    };
    /**
     * Get count of active (checked out of the pool) resources for the given key.
     * @param {ServerAddress} address the address of the server to check.
     * @return {number} count of resources acquired by clients.
     */
    Pool.prototype.activeResourceCount = function (address) {
        return this._activeResourceCounts[address.asKey()] || 0;
    };
    Pool.prototype._acquire = function (address) {
        return __awaiter(this, void 0, void 0, function () {
            var key, pool, poolState, resource_1, numConnections, resource;
            var _this = this;
            return __generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        if (this._closed) {
                            throw (0, neo4j_driver_core_1.newError)('Pool is closed, it is no more able to serve requests.');
                        }
                        key = address.asKey();
                        pool = this._pools[key];
                        poolState = this._poolState[key];
                        if (!pool) {
                            pool = [];
                            poolState = new PoolState();
                            this._pools[key] = pool;
                            this._pendingCreates[key] = 0;
                            this._poolState[key] = poolState;
                        }
                        _a.label = 1;
                    case 1:
                        if (!pool.length) return [3 /*break*/, 5];
                        resource_1 = pool.pop();
                        if (!this._validate(resource_1)) return [3 /*break*/, 2];
                        if (this._removeIdleObserver) {
                            this._removeIdleObserver(resource_1);
                        }
                        // idle resource is valid and can be acquired
                        resourceAcquired(key, this._activeResourceCounts);
                        if (this._log.isDebugEnabled()) {
                            this._log.debug(resource_1 + " acquired from the pool " + key);
                        }
                        return [2 /*return*/, resource_1];
                    case 2: return [4 /*yield*/, this._destroy(resource_1)];
                    case 3:
                        _a.sent();
                        _a.label = 4;
                    case 4: return [3 /*break*/, 1];
                    case 5:
                        // Ensure requested max pool size
                        if (this._maxSize > 0) {
                            numConnections = this.activeResourceCount(address) + this._pendingCreates[key];
                            if (numConnections >= this._maxSize) {
                                // Will put this request in queue instead since the pool is full
                                return [2 /*return*/, null];
                            }
                        }
                        // there exist no idle valid resources, create a new one for acquisition
                        // Keep track of how many pending creates there are to avoid making too many connections.
                        this._pendingCreates[key] = this._pendingCreates[key] + 1;
                        _a.label = 6;
                    case 6:
                        _a.trys.push([6, , 8, 9]);
                        return [4 /*yield*/, this._create(address, function (address, resource) { return _this._release(poolState, address, resource); })];
                    case 7:
                        // Invoke callback that creates actual connection
                        resource = _a.sent();
                        resourceAcquired(key, this._activeResourceCounts);
                        if (this._log.isDebugEnabled()) {
                            this._log.debug(resource + " created for the pool " + key);
                        }
                        return [3 /*break*/, 9];
                    case 8:
                        this._pendingCreates[key] = this._pendingCreates[key] - 1;
                        return [7 /*endfinally*/];
                    case 9: return [2 /*return*/, resource];
                }
            });
        });
    };
    Pool.prototype._release = function (poolState, address, resource) {
        return __awaiter(this, void 0, void 0, function () {
            var key, pool;
            var _this = this;
            return __generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        key = address.asKey();
                        pool = this._pools[key];
                        if (!(pool && poolState && poolState.isActive())) return [3 /*break*/, 4];
                        if (!!this._validate(resource)) return [3 /*break*/, 2];
                        if (this._log.isDebugEnabled()) {
                            this._log.debug(resource + " destroyed and can't be released to the pool " + key + " because it is not functional");
                        }
                        return [4 /*yield*/, this._destroy(resource)];
                    case 1:
                        _a.sent();
                        return [3 /*break*/, 3];
                    case 2:
                        if (this._installIdleObserver) {
                            this._installIdleObserver(resource, {
                                onError: function (error) {
                                    _this._log.debug("Idle connection " + resource + " destroyed because of error: " + error);
                                    var pool = _this._pools[key];
                                    if (pool) {
                                        _this._pools[key] = pool.filter(function (r) { return r !== resource; });
                                    }
                                    // let's not care about background clean-ups due to errors but just trigger the destroy
                                    // process for the resource, we especially catch any errors and ignore them to avoid
                                    // unhandled promise rejection warnings
                                    _this._destroy(resource).catch(function () { });
                                }
                            });
                        }
                        pool.push(resource);
                        if (this._log.isDebugEnabled()) {
                            this._log.debug(resource + " released to the pool " + key);
                        }
                        _a.label = 3;
                    case 3: return [3 /*break*/, 6];
                    case 4:
                        // key has been purged, don't put it back, just destroy the resource
                        if (this._log.isDebugEnabled()) {
                            this._log.debug(resource + " destroyed and can't be released to the pool " + key + " because pool has been purged");
                        }
                        return [4 /*yield*/, this._destroy(resource)];
                    case 5:
                        _a.sent();
                        _a.label = 6;
                    case 6:
                        resourceReleased(key, this._activeResourceCounts);
                        this._processPendingAcquireRequests(address);
                        return [2 /*return*/];
                }
            });
        });
    };
    Pool.prototype._purgeKey = function (key) {
        return __awaiter(this, void 0, void 0, function () {
            var pool, poolState, resource;
            return __generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        pool = this._pools[key] || [];
                        poolState = this._poolState[key] || new PoolState();
                        _a.label = 1;
                    case 1:
                        if (!pool.length) return [3 /*break*/, 3];
                        resource = pool.pop();
                        if (this._removeIdleObserver) {
                            this._removeIdleObserver(resource);
                        }
                        return [4 /*yield*/, this._destroy(resource)];
                    case 2:
                        _a.sent();
                        return [3 /*break*/, 1];
                    case 3:
                        poolState.close();
                        delete this._pools[key];
                        delete this._poolState[key];
                        return [2 /*return*/];
                }
            });
        });
    };
    Pool.prototype._processPendingAcquireRequests = function (address) {
        var _this = this;
        var key = address.asKey();
        var requests = this._acquireRequests[key];
        var poolState = this._poolState[key];
        if (requests) {
            var pendingRequest_1 = requests.shift(); // pop a pending acquire request
            if (pendingRequest_1) {
                this._acquire(address)
                    .catch(function (error) {
                    // failed to acquire/create a new connection to resolve the pending acquire request
                    // propagate the error by failing the pending request
                    pendingRequest_1.reject(error);
                    return null;
                })
                    .then(function (resource) {
                    if (resource) {
                        // managed to acquire a valid resource from the pool
                        if (pendingRequest_1.isCompleted()) {
                            // request has been completed, most likely failed by a timeout
                            // return the acquired resource back to the pool
                            _this._release(poolState, address, resource);
                        }
                        else {
                            // request is still pending and can be resolved with the newly acquired resource
                            pendingRequest_1.resolve(resource); // resolve the pending request with the acquired resource
                        }
                    }
                });
            }
            else {
                delete this._acquireRequests[key];
            }
        }
    };
    return Pool;
}());
/**
 * Increment active (checked out of the pool) resource counter.
 * @param {string} key the resource group identifier (server address for connections).
 * @param {Object.<string, number>} activeResourceCounts the object holding active counts per key.
 */
function resourceAcquired(key, activeResourceCounts) {
    var currentCount = activeResourceCounts[key] || 0;
    activeResourceCounts[key] = currentCount + 1;
}
/**
 * Decrement active (checked out of the pool) resource counter.
 * @param {string} key the resource group identifier (server address for connections).
 * @param {Object.<string, number>} activeResourceCounts the object holding active counts per key.
 */
function resourceReleased(key, activeResourceCounts) {
    var currentCount = activeResourceCounts[key] || 0;
    var nextCount = currentCount - 1;
    if (nextCount > 0) {
        activeResourceCounts[key] = nextCount;
    }
    else {
        delete activeResourceCounts[key];
    }
}
var PendingRequest = /** @class */ (function () {
    function PendingRequest(key, resolve, reject, timeoutId, log) {
        this._key = key;
        this._resolve = resolve;
        this._reject = reject;
        this._timeoutId = timeoutId;
        this._log = log;
        this._completed = false;
    }
    PendingRequest.prototype.isCompleted = function () {
        return this._completed;
    };
    PendingRequest.prototype.resolve = function (resource) {
        if (this._completed) {
            return;
        }
        this._completed = true;
        clearTimeout(this._timeoutId);
        if (this._log.isDebugEnabled()) {
            this._log.debug(resource + " acquired from the pool " + this._key);
        }
        this._resolve(resource);
    };
    PendingRequest.prototype.reject = function (error) {
        if (this._completed) {
            return;
        }
        this._completed = true;
        clearTimeout(this._timeoutId);
        this._reject(error);
    };
    return PendingRequest;
}());
var PoolState = /** @class */ (function () {
    function PoolState() {
        this._active = true;
    }
    PoolState.prototype.isActive = function () {
        return this._active;
    };
    PoolState.prototype.close = function () {
        this._active = false;
    };
    return PoolState;
}());
exports.default = Pool;
