# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""UI helpers for authentication models."""

from functools import cached_property

from django.core.exceptions import ImproperlyConfigured

from debusine.db.context import context
from debusine.db.models import Group, User
from debusine.web.views.places import ResourcePlace
from debusine.web.views.ui.base import UI


class UserUI(UI[User]):
    """UI helpers for User instances."""

    @cached_property
    def place(self) -> ResourcePlace:
        """Return a place to show this user."""
        if full_name := self.instance.get_full_name():
            description = f"User {full_name} <{self.instance.username}>"
        else:
            description = f"User <{self.instance.username}>"
        place = ResourcePlace(
            title=self.instance.get_full_name() or self.instance.username,
            breadcrumb=self.instance.username,
            url=self.instance.get_absolute_url(),
            description=description,
        )
        if context.scope is not None:
            place.parent = context.scope.ui(self.request).place
        return place


class GroupUI(UI[Group]):
    """UI helpers for Group instances."""

    @cached_property
    def place(self) -> ResourcePlace:
        """Return a place to show this group."""
        user = context.user
        if user is None or not user.is_authenticated:
            raise ImproperlyConfigured(
                "GroupUI.place invoked without a current user"
            )

        scope_ui = context.require_scope().ui(self.request)
        return ResourcePlace(
            title=f"Group {self.instance}",
            breadcrumb=self.instance.name,
            url=self.instance.get_absolute_url(),
            parent=scope_ui.place_group_list,
        )
