/* $Id$ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __COMP_INST_STAT_HPP_INCLUDED
#define __COMP_INST_STAT_HPP_INCLUDED

#include "frontend/ast/Name.hpp"
#include "frontend/ast/ConcurrentStat.hpp"
#include "frontend/ast/AssociationElement.hpp"
#include "frontend/ast/Entity.hpp"

namespace ast {

/** a VHDL component instantiation */
class CompInstStat : public ConcurrentStat {
public:
	//! c'tor
	/** @param entityN name of the entity.
          * @param genericM generic map
          * @param portM port map
	  * @param loc location of the concurrent statement
          */
	CompInstStat(
		Name *entityN,
		std::list<AssociationElement*> *genericM,
		std::list<AssociationElement*> *portM,
		Location loc
		) : 	ConcurrentStat(loc),
			entityName(entityN),
			genericMap(genericM),
			portMap(portM) {

		assert(entityN != NULL);
		if (entityN->candidates.size() == 1) {
			Symbol *sym = entityN->candidates.front();
			this->entity = 
				dynamic_cast<Entity*>(&sym->declaration);
		} else {
			this->entity = NULL;
		}
	}

	//! Accept a Visitor.
 	/** All leaf AST nodes need to implement this method.
         *
         *  @param visitor the Visitor that can visit this node.
         */
	virtual void accept(Visitor& visitor) {
		visitor.visit(*this);
	}

	/** Put a textual representation of the AstNode on the stream.
	 *  @param stream stream to put the textual representation to.
	 */
	virtual void put(std::ostream &stream) const {
		if (this->name != NULL) {
			stream << *this->name;
		} else {
			stream << "(null)";
		}

		stream << " : " << this->entityName 
			<< " generic map (";
		util::MiscUtil::listPut(this->genericMap, stream, ", ");
		stream << ") port map (";
		util::MiscUtil::listPut(this->portMap, stream, ", ");
		stream << ");";
	}

	/** name referring to the entity */
	Name *entityName;
	/** generic map */
	std::list<AssociationElement*> *genericMap;
	/** port map */
	std::list<AssociationElement*> *portMap;
	/** referred to entity */
	const Entity *entity;

protected:
	/** Destructor */
	virtual ~CompInstStat() {
		util::MiscUtil::terminate(entityName);
		util::MiscUtil::lterminate(genericMap);
		util::MiscUtil::lterminate(portMap);
	}

};

}; /* namespace ast */

#endif /* __COMP_INST_STAT_HPP_INCLUDED */
