IPP Data File API
=================

API to read and write IPP attributes and other commands or data using a common base format that supports tools such as `ipptool` and `ippeveprinter`.


Base Types
----------

    typedef struct _ipp_file_s ipp_file_t;

An open IPP tools file with associated variables and callbacks.  The following standard variables are typically defined:

- "date-current": Current date in ISO-8601 format
- "date-start": Start date (when file opened) in ISO-8601 format
- "filename": Associated data/document filename, if any
- "filetype": MIME media type of associated data/document filename, if any
- "hostname": Hostname or IP address from the "uri" value, if any
- "port": Port number from the "uri" value, if any
- "resource": Resource path from the "uri" value, if any
- "scheme": URI scheme from the "uri" value, if any
- "uri": URI, if any
- "uriuser": Username from the "uri" value, if any
- "uripassword": Password from the "uri" value, if any
- "user": Current login user

    typedef bool (*ipp_fattr_cb_t)(ipp_file_t *file, void *cb_data, const char *name);

An IPP attribute filter callback.  The "name" parameter is the name of an IPP attribute.  The callback returns `true` to use the attribute and `false` otherwise.  Typically this is used to filter out attributes that are generated by the application or that cannot be configured through the file (status, supported operations, etc.)  The default `NULL` callback does no filtering.

    typedef bool (*ipp_ferror_cb_t)(ipp_file_t *file, void *cb_data, const char *error);

An IPP error message callback.  The "error" parameter contains an error message.  The callback returns `true` to continue or `false` to stop.  The default `NULL` callback writes the error message to `stderr` and returns `true`.

    typedef bool (*ipp_ftoken_cb_t)(ipp_file_t *file, void *cb_data, const char *token);

An IPP token callback.  The "token" parameter contains the token to be processed.  The callback can use the `ippFileReadToken` function to read additional tokens from the file and the `ippFileExpandToken` function to expand any variables in the token string.  Return `false` to stop reading the file and `true` to continue.  The default `NULL` callback reports an unknown token error through the error callback end returns `false`.


Public API
----------

    ipp_file_t *
    ippFileNew(ipp_file_t *parent, ipp_fattr_cb_t attr_cb,
               ipp_ferror_cb_t error_cb, void *cb_data);

Create a new IPP data file object.  The "parent" pointer causes the new file to contain the same variables as the parent and is typically used when processing the `INCLUDE` directive.

    bool
    ippFileOpen(ipp_file_t *file, const char *filename, const char *mode);

Open an IPP tools file for reading (mode = "r") or writing (mode = "w").

    bool
    ippFileClose(ipp_file_t *file);

Closes an IPP tools file and returns `true` on success or `false` on failure.

    bool
    ippFileDelete(ipp_file_t *file);

Closes an IPP tools file (if needed), frees all memory used, and returns `true` on success or `false` on failure.

    size_t
    ippFileExpandVars(ipp_file_t *file, char *dst, const char *src,
                      size_t dstsize);

Expands variables in the source string, copying to the destination string.

    ipp_attribute_t *
    ippFileGetAttribute(ipp_file_t *file, const char *name,
                        ipp_tag_t value_tag);

Get the named IPP attribute if it has been read.

    ipp_t *
    ippFileGetAttributes(ipp_file_t *file);

Get the IPP attributes that have been read so far.

    const char *
    ippFileGetFilename(ipp_file_t *file);

Gets the current filename.

    int
    ippFileGetLineNumber(ipp_file_t *file);

Gets the current line number in the file.

    const char *
    ippFileGetVar(ipp_file_t *file, const char *name);

    void
    ippFileSetVar(ipp_file_t *file, const char *name, const char *value);

Get and set file variables.  Setting the "uri" variable also sets the "hostname", "port", "resource", "scheme", and "uriuser" variables.

    bool
    ippFileRead(ipp_file_t *file, ipp_ftoken_cb_t token_cb, bool with_groups);

Reads the IPP tools file using the specified token callback.

    bool
    ippFileReadToken(ipp_file_t *file, char *buffer, size_t bufsize);

Reads a single token from the file, returning `false` on end-of-file.

    bool
    ippFileRestorePosition(ipp_file_t *file);

    bool
    ippFileSavePosition(ipp_file_t *file);

Saves and restores the current read position in a file.

    bool
    ippFileSetAttributes(ipp_file_t *file, ipp_t *attrs);

Set the IPP message to add attributes to when reading.

    bool
    ippFileSetGroupTag(ipp_file_t *file, ipp_tag_t group_tag);

Set the current IPP group tag for new attributes.

    bool
    ippFileSetVar(ipp_file_t *file, const char *name, const char *value);

    bool
    ippFileSetVarf(ipp_file_t *file, const char *name, const char *value, ...);

Set the named variable.

    bool
    ippFileWriteAttributes(ipp_file_t *file, ipp_t *ipp, bool with_groups);

Writes the attributes in "ipp" to the file, including "GROUP" directives when "with_groups" is `true`.  Returns `false` on error or `true` on success.

    bool
    ippFileWriteComment(ipp_file_t *file, const char *comment, ...);

Writes a comment to the file.

    bool
    ippFileWriteToken(ipp_file_t *file, const char *token);

    bool
    ippFileWriteTokenf(ipp_file_t *file, const char *token, ...);

Write a single token to the file.  The second form allows for printf-style format strings.
