"use strict";
/*
 * Copyright (c) 2014-2023 Bjoern Kimminich & the OWASP Juice Shop contributors.
 * SPDX-License-Identifier: MIT
 */
Object.defineProperty(exports, "__esModule", { value: true });
const basketitem_1 = require("../models/basketitem");
const quantity_1 = require("../models/quantity");
const challengeUtils = require("../lib/challengeUtils");
const utils = require('../lib/utils');
const challenges = require('../data/datacache').challenges;
const security = require('../lib/insecurity');
module.exports.addBasketItem = function addBasketItem() {
    return (req, res, next) => {
        const result = utils.parseJsonCustom(req.rawBody);
        const productIds = [];
        const basketIds = [];
        const quantities = [];
        for (let i = 0; i < result.length; i++) {
            if (result[i].key === 'ProductId') {
                productIds.push(result[i].value);
            }
            else if (result[i].key === 'BasketId') {
                basketIds.push(result[i].value);
            }
            else if (result[i].key === 'quantity') {
                quantities.push(result[i].value);
            }
        }
        const user = security.authenticatedUsers.from(req);
        if (user && basketIds[0] && basketIds[0] !== 'undefined' && Number(user.bid) != Number(basketIds[0])) { // eslint-disable-line eqeqeq
            res.status(401).send('{\'error\' : \'Invalid BasketId\'}');
        }
        else {
            const basketItem = {
                ProductId: productIds[productIds.length - 1],
                BasketId: basketIds[basketIds.length - 1],
                quantity: quantities[quantities.length - 1]
            };
            challengeUtils.solveIf(challenges.basketManipulateChallenge, () => { return user && basketItem.BasketId && basketItem.BasketId !== 'undefined' && user.bid != basketItem.BasketId; }); // eslint-disable-line eqeqeq
            const basketItemInstance = basketitem_1.BasketItemModel.build(basketItem);
            basketItemInstance.save().then((addedBasketItem) => {
                res.json({ status: 'success', data: addedBasketItem });
            }).catch((error) => {
                next(error);
            });
        }
    };
};
module.exports.quantityCheckBeforeBasketItemAddition = function quantityCheckBeforeBasketItemAddition() {
    return (req, res, next) => {
        void quantityCheck(req, res, next, req.body.ProductId, req.body.quantity).catch((error) => {
            next(error);
        });
    };
};
module.exports.quantityCheckBeforeBasketItemUpdate = function quantityCheckBeforeBasketItemUpdate() {
    return (req, res, next) => {
        basketitem_1.BasketItemModel.findOne({ where: { id: req.params.id } }).then((item) => {
            const user = security.authenticatedUsers.from(req);
            challengeUtils.solveIf(challenges.basketManipulateChallenge, () => { return user && req.body.BasketId && user.bid != req.body.BasketId; }); // eslint-disable-line eqeqeq
            if (req.body.quantity) {
                if (!item) {
                    throw new Error('No such item found!');
                }
                void quantityCheck(req, res, next, item.ProductId, req.body.quantity);
            }
            else {
                next();
            }
        }).catch((error) => {
            next(error);
        });
    };
};
async function quantityCheck(req, res, next, id, quantity) {
    const product = await quantity_1.QuantityModel.findOne({ where: { ProductId: id } });
    if (!product) {
        throw new Error('No such product found!');
    }
    // is product limited per user and order, except if user is deluxe?
    if (!product.limitPerUser || (product.limitPerUser && product.limitPerUser >= quantity) || security.isDeluxe(req)) {
        if (product.quantity >= quantity) { // enough in stock?
            next();
        }
        else {
            res.status(400).json({ error: res.__('We are out of stock! Sorry for the inconvenience.') });
        }
    }
    else {
        res.status(400).json({ error: res.__('You can order only up to {{quantity}} items of this product.', { quantity: product.limitPerUser.toString() }) });
    }
}
//# sourceMappingURL=basketItems.js.map