/*
 * Copyright (c) 2000, 2024, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package nsk.jdi.ReferenceType.visibleFields;

import nsk.share.*;
import nsk.share.jpda.*;
import nsk.share.jdi.*;

import com.sun.jdi.*;
import java.util.*;
import java.io.*;

/**
 * This test checks the method <code>visibleFields()</code>
 * of the JDI interface <code>ReferenceType</code> of com.sun.jdi package
 */

public class visibfield004 extends Log {
    static java.io.PrintStream out_stream;
    static boolean verbose_mode = false;  // test argument -vbs or -verbose switches to static
                                          // - for more easy failure evaluation

    /** The main class names of the debugger & debugee applications. */
    private final static String
        package_prefix = "nsk.jdi.ReferenceType.visibleFields.",
//        package_prefix = "",    //  for DEBUG without package
        thisClassName = package_prefix + "visibfield004",
        debugeeName   = thisClassName + "a";

    /** Debugee's classes for check **/
    private final static String class_for_check = package_prefix + "visibfield004aClassForCheck";


    public static void main (String argv[]) {
        int result = run(argv,System.out);
        if (result != 0) {
            throw new RuntimeException("TEST FAILED with result " + result);
        }

    }

    /**
     * JCK-like entry point to the test: perform testing, and
     * return exit code 0 (PASSED) or either 2 (FAILED).
     */
    public static int run (String argv[], PrintStream out) {
        out_stream = out;

        int v_test_result = new visibfield004().runThis(argv,out_stream);
        if ( v_test_result == 2/*STATUS_FAILED*/ ) {
            out_stream.println("\n==> nsk/jdi/ReferenceType/visibleFields/visibfield004 test FAILED");
        }
        else {
            out_stream.println("\n==> nsk/jdi/ReferenceType/visibleFields/visibfield004 test PASSED");
        }
        return v_test_result;
    }

    private void print_log_on_verbose(String message) {
        display(message);
    }

    /**
     * Non-static variant of the method <code>run(args,out)</code>
     */
    private int runThis (String argv[], PrintStream out) {
        ArgumentHandler argHandler = new ArgumentHandler(argv);
        verbose_mode = argHandler.verbose();

        if ( out_stream == null ) {
            out_stream = out;
        }

        out_stream.println("==> nsk/jdi/ReferenceType/visibleFields/visibfield004 test LOG:");
        out_stream.println("==> test checks visibleFields() method of ReferenceType interface ");
        out_stream.println("    of the com.sun.jdi package for class without visible fields\n");

        String debugee_launch_command = debugeeName;
        if (verbose_mode) {
            logTo(out_stream);
        }

        Binder binder = new Binder(argHandler,this);
        Debugee debugee = binder.bindToDebugee(debugee_launch_command);
        IOPipe pipe = new IOPipe(debugee);

        debugee.redirectStderr(out);
        print_log_on_verbose("--> visibfield004: visibfield004a debugee launched");
        debugee.resume();

        String line = pipe.readln();
        if (line == null) {
            out_stream.println
                ("##> visibfield004: UNEXPECTED debugee's signal (not \"ready\") - " + line);
            return 2/*STATUS_FAILED*/;
        }
        if (!line.equals("ready")) {
            out_stream.println
                ("##> visibfield004: UNEXPECTED debugee's signal (not \"ready\") - " + line);
            return 2/*STATUS_FAILED*/;
        }
        else {
            print_log_on_verbose("--> visibfield004: debugee's \"ready\" signal recieved!");
        }

        out_stream.println
            ("--> visibfield004: check ReferenceType.visibleFields() method for debugee's "
            + class_for_check + " class...");
        boolean class_not_found_error = false;
        boolean visibleFields_method_error = false;
        int visible_fields_number = 0;

        while ( true ) {
            ReferenceType refType = debugee.classByName(class_for_check);
            if (refType == null) {
                out_stream.println("##> visibfield004: Could NOT FIND class: " + class_for_check);
                class_not_found_error = true;
                break;
            }
            List<Field> visible_fields_list = null;
            try {
                visible_fields_list = refType.visibleFields();
            }
            catch (Throwable thrown) {
                out_stream.println("##> visibfield004: FAILED: ReferenceType.visibleFields() throws unexpected "
                    + thrown);
                visibleFields_method_error = true;
                break;
            }
            visible_fields_number = visible_fields_list.size();
            if ( visible_fields_number == 0 ) {
                break;
            }
            Field visible_fields[] = new Field[visible_fields_number];
            visible_fields_list.toArray(visible_fields);
            for (int i=0; i<visible_fields_number; i++) {
                Field checked_field = visible_fields[i];
                String checked_field_name = checked_field.name();
                String checked_field_typename = checked_field.typeName();
                String checked_field_static;
                if ( checked_field.isStatic() ) {
                    checked_field_static = " static";
                }
                else {
                    checked_field_static = "";
                }
                String full_checked_field = checked_field_static
                    + " " + checked_field_typename + " " + checked_field_name;
                out_stream.println
                    ("##> visibfield004: FAILED: unexpected visible field: " + full_checked_field);
            }
            break;
        }

        int v_test_result = 0/*STATUS_PASSED*/;
        if ( class_not_found_error || visibleFields_method_error ) {
            v_test_result = 2/*STATUS_FAILED*/;
        }

        if ( visible_fields_number > 0 ) {
            out_stream.println
                ("##> visibfield004: UNEXPECTED visible fields number = " + visible_fields_number);
            v_test_result = 2/*STATUS_FAILED*/;
        }
        else {
            out_stream.println
                ("--> visibfield004: PASSED: returned list of fields is empty!");
        }

        print_log_on_verbose("--> visibfield004: waiting for debugee finish...");
        pipe.println("quit");
        debugee.waitFor();

        int status = debugee.getStatus();
        if (status != 0/*STATUS_PASSED*/ + 95/*STATUS_TEMP*/) {
            out_stream.println
                ("##> visibfield004: UNEXPECTED Debugee's exit status (not 95) - " + status);
            v_test_result = 2/*STATUS_FAILED*/;
        }
        else {
            print_log_on_verbose
                ("--> visibfield004: expected Debugee's exit status - " + status);
        }

        return v_test_result;
    }
}
