/*
 * Access to ALPS QUERY methods
 *
 * Copyright (c) 2009-2011 Centro Svizzero di Calcolo Scientifico (CSCS)
 * Licensed under the GPLv2.
 */
#include "parser_internal.h"

/**
 * get_alps_engine  -  run QUERY of type ENGINE
 * This uses the convention of returning the Engine.version attribute via 'msg'.
 * Returns pointer to @buf, NULL on error.
 */
const char *get_alps_engine(char *buf, size_t buflen)
{
	struct basil_parse_data bp = {0};

	/* For this query use Basil 1.0 as lowest common denominator */
	bp.version = BV_1_0;
	bp.method  = BM_engine;

	if (basil_request(&bp) < 0)
		return NULL;
	strncpy(buf, bp.msg, buflen);
	return buf;
}

/**
 * get_basil_version  -  Detect highest BASIL version supported by ALPS.
 *
 * This uses the following correspondence table to find the highest supported
 * BASIL version. Failing that, it falls back to Basil 1.0 as last resort.
 *
 * +------------+---------------+------+----------------+----------------------+
 * | CLE release| Engine.version| ALPS | Basil Protocol |       Remarks        |
 * +------------+---------------+------+----------------+----------------------+
 * |  <= 2.2.48B|         1.1.0 |  1.1 |   1.0, 1.1     | see below            |
 * |  >= 2.2.67 |         1.2.0 |  1.2 |   1.0, 1.1     | last CLE 2.2 update  |
 * |     3.0    |         1.3.0 |  3.0 |   1.0, 1.1     | Cray ticket #762417  |
 * |     3.1    |         3.1.0 |  3.1 |   1.0, 1.1     | Cray ticket #762035  |
 * |     4.0    |         4.0.0 |  4.0 |   1.0,1.1,1.2  | starts GPU support   |
 * +------------+---------------+------+----------------+----------------------+
 *
 * The 'ALPS' column shows the name of the ALPS engine; the 'Basil Protocol'
 * column shows the supported versions for the BasilRequest.protocol attribute.
 *
 * No CLE 2 versions were released between 2.2.48B and 2.2.67; the Basil 1.2
 * variant that came with the latter release behaved identically to Basil 1.1.
 *
 * Starting from Basil 3.1, there is also a 'basil_support' attribute to query
 * the supported 'Basil Protocol' list.
 *
 * However, we can not presuppose this version and thus use Basil 1.0 as lowest
 * common denominator supported by all CLE releases.
 */
enum basil_version get_basil_version(void)
{
	char engine_version[BASIL_STRING_LONG];
	unsigned major, minor, micro;

	if (!get_alps_engine(engine_version, sizeof(engine_version))) {
		error("can not determine ALPS Engine version, using BASIL 1.0");
		return BV_1_0;
	}
	if (parse_version_string(engine_version, &major, &minor, &micro) < 0) {
		error("bad ALPS Engine version '%s', using BASIL 1.0",
		      engine_version);
		return BV_1_0;
	}

	switch (major) {
	default:
		if (major > 4) {
			error("got new ALPS Engine version %s, using BASIL 4.1",
			      engine_version);
			return BV_4_1;
		}
		break;
	case 4:
		switch (minor) {
		default:
			error("unknown BASIL version 4.%u, using 4.1", minor);
			/* fall through */
		case 1:
			return BV_4_1;
		case 0:
			return BV_4_0;
		}
	case 3:
		if (minor != 1)
			error("unknown BASIL version 3.%u, using 3.1", minor);
		return BV_3_1;
	case 1:
		switch (minor) {
		case 3:
			/*
			 * Cray Bug#762417 - strictly speaking, we should be
			 * returning BV_3_0 here. Alps Engine Version 1.3.0
			 * is reserved for the Cozla release (CLE 3.0), which
			 * however was only a short time on the market.
			 */
			return BV_3_1;
		default:
			error("unknown BASIL version 1.%u, using 1.2", minor);
			/* fall through */
		case 2:
			return BV_1_2;
		case 1:
			return BV_1_1;
		case 0:
			return BV_1_0;
		}
	}
	error("unsupported ALPS Engine version '%s', falling back to BASIL 1.0",
		engine_version);
	return BV_1_0;
}

/**
 * get_inventory  -  generic INVENTORY request
 * Caller must free result structure by calling free_inv().
 */
static struct basil_inventory *get_inventory(enum basil_version version,
					     bool do_full_inventory)
{
	struct basil_parse_data bp = {0};

	bp.version   = version;
	bp.method    = BM_inventory;
	bp.mdata.inv = alloc_inv(do_full_inventory);

	if (bp.mdata.inv == NULL)
		return NULL;

	if (basil_request(&bp) < 0) {
		free_inv(bp.mdata.inv);
		return NULL;
	}

	return bp.mdata.inv;
}

/** Basic inventory (avail/total counts only) */
struct basil_inventory *get_sched_inventory(enum basil_version version)
{
	return get_inventory(version, false);
}

/** Perform a detailed inventory */
struct basil_inventory *get_full_inventory(enum basil_version version)
{
	return get_inventory(version, true);
}
