#! /usr/bin/env python

# Genetic
# Copyright (C) 2001 Jean-Baptiste LAMY
#
# This program is free software. See README or LICENSE for the license terms.

from __future__ import nested_scopes

import random, string, math, operator
from genetic import organism, lifecycle

# This will solve the traveller saleman problem (=TSP).
# SEE demo_saleman_2.py FOR A MUCH EFFICIENT IMPLEMENTATION WITH TK INTERFACE !
#
# The saleman must travel from city to city, and pass in each city; but he
# wants to walk as less as possible... Are genetic algos his best chance ?


# Define a City class. Cities are defined by 2 coordinates x and y on a map.
class City:
  next_id = 0
  
  def __init__(self, x, y):
    self.x, self.y = x, y
    
    self.id = City.next_id
    self.name = "city" + str(self.id)
    City.next_id = City.next_id + 1


# Define a list of cities. You can add some if you want.
# With this value, the minimum distance is 4 -- but this demo doesn't give
# really good results.

cities = [
City(0.0, 0.0),
City(0.8, 0.0),
City(0.1, 0.0),
City(1.0, 0.0),

City(0.0, 1.0),
City(0.8, 1.0),
City(0.1, 1.0),
City(1.0, 1.0),
]


# Define the function to minimize.
#
# We consider having a gene for each city, called "city<x>", except for the
# first city. The saleman always start in the first city !
#
# Each gene indicates the priority of the corresponding city : the higher the
# corresponding gene value is, the first the city will be visited.
#
# journey_distance takes the list of city-priorities (= the genes values) and
# return the length of the journey.

def journey_distance(*city_priorities):
  # city_priorities is : [city1_priority, city2_priority, ...]
  
  # 1 - Associate each priority with its city index, in a tuple : [(city1_priority, city1), (city2_priority, city2), ...]
  city_list = map(None, city_priorities, cities[1:])
  
  # 2 - Sort the city_list by priority (= the first value in the tuple)
  city_list.sort()
  
  # 3 - Remove the priorities -- there are useless now
  city_list = map(lambda tuple: tuple[1], city_list)
  
  # 4 - Add the first city at the beginning
  city_list.insert(0, cities[0])
  
  # 5 - Compute the journey distance
  return reduce(operator.add, map(lambda cityA, cityB: math.sqrt((cityA.x - cityB.x) ** 2 + (cityA.y - cityB.y) ** 2), city_list, city_list[1:] + [city_list[0]]))

 
class TSPSolver(organism.Organism):
  characteristics = [
    organism.Characteristic("journey_distance",
                            journey_distance,
                            organism.PERCHROMOSOM_DOMINANCY_PHENOTYPE,
                            [ city.name for city in cities[1:]] # = ["city0", "city1", ...] the args names must be explicitely specified, because they cannot be guessed from the journey_distance function.
                            )
    ]
  
  def __cmp__(self, other):
    return cmp(self.journey_distance, other.journey_distance)

  # Override the Organism.__repr__ method, so it will give the city order
  def __repr__(self):
    # self.journey_distance_args is the args that has been used to compute
    # journey_distance -- the city_priorities !
    city_list = map(None, self.journey_distance_args, cities[1:])
    
    # 2 - Sort the city_list by priority (= the first value in the tuple)
    city_list.sort()
    
    # 3 - Remove the priorities -- there are useless now
    city_list = map(lambda tuple: tuple[1], city_list)
    
    # 4 - Add the first city at the beginning
    city_list.insert(0, cities[0])
    
    repr = "Cities order : %s\n" % map(lambda city: city.name, city_list)
    return repr + organism.Organism.__repr__(self)


# Create an initial set of organisms.
#
# This complex dict manipulation allows to automatically give the needed genes,
# but you can also use organism.Chromosom(city1 = 0.0, city2 = 0.0, ...)

args_names = TSPSolver.characteristics[0].funcargs

genesA = { "__mutation__" : 10.0, "__mutampl__" : 5.0 }
map(genesA.__setitem__, args_names, [0.0] * len(args_names))

organismA = TSPSolver([organism.Chromosom(**genesA)])


genesB = { "__mutation__" : 10.0, "__mutampl__" : 5.0 }
map(genesB.__setitem__, args_names, [10.0] * len(args_names))

organismB = TSPSolver([organism.Chromosom(**genesB)])


organisms = [organismA, organismB]


# Define some parameters

# True for elitism (= always keep the best organism)
ELITISM = 1

# Number of generation to run
NB_GENERATION = 6

# Number of children each generation made
NB_CHILDREN  = 100

# Number of organisms kept per generation (other children doesn't survive)
NB_ORGANISMS = 4


# Ready !

lifecycle.run(organisms, ELITISM, NB_GENERATION, NB_CHILDREN, NB_ORGANISMS, 0)
