/*
 * storagebackendsheepdogtest.c: storage backend for Sheepdog handling
 *
 * Copyright (C) 2014 Red Hat, Inc.
 * Copyright (C) 2012 Sebastian Wiedenroth
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library.  If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include <config.h>

#include <unistd.h>

#include <fcntl.h>

#include "internal.h"
#include "testutils.h"
#define LIBVIRT_STORAGE_BACKEND_SHEEPDOG_PRIV_H_ALLOW
#include "storage/storage_backend_sheepdog_priv.h"
#include "virstring.h"

#define VIR_FROM_THIS VIR_FROM_NONE

typedef struct {
    const char *output;
    int expected_return;
    uint64_t expected_capacity;
    uint64_t expected_allocation;
} collie_test;

struct testNodeInfoParserData {
    collie_test data;
    const char *poolxml;
};

struct testVDIListParserData {
    collie_test data;
    const char *poolxml;
    const char *volxml;
};


static int
test_node_info_parser(const void *opaque)
{
    const struct testNodeInfoParserData *data = opaque;
    collie_test test = data->data;
    g_autofree char *output = NULL;
    g_autoptr(virStoragePoolDef) pool = NULL;

    if (!(pool = virStoragePoolDefParseFile(data->poolxml)))
        return -1;

    output = g_strdup(test.output);

    if (virStorageBackendSheepdogParseNodeInfo(pool, output) !=
        test.expected_return)
        return -1;

    if (test.expected_return)
        return 0;

    if (pool->capacity == test.expected_capacity &&
        pool->allocation == test.expected_allocation)
        return 0;

    return -1;
}

static int
test_vdi_list_parser(const void *opaque)
{
    const struct testVDIListParserData *data = opaque;
    collie_test test = data->data;
    g_autofree char *output = NULL;
    g_autoptr(virStoragePoolDef) pool = NULL;
    g_autoptr(virStorageVolDef) vol = NULL;

    if (!(pool = virStoragePoolDefParseFile(data->poolxml)))
        return -1;

    if (!(vol = virStorageVolDefParseFile(pool, data->volxml, 0)))
        return -1;

    output = g_strdup(test.output);

    if (virStorageBackendSheepdogParseVdiList(vol, output) !=
        test.expected_return)
        return -1;

    if (test.expected_return)
        return 0;

    if (vol->target.capacity == test.expected_capacity &&
        vol->target.allocation == test.expected_allocation)
        return 0;

    return -1;
}


static int
mymain(void)
{
    int ret = 0;
    g_autofree char *poolxml = NULL;
    g_autofree char *volxml = NULL;

    collie_test node_info_tests[] = {
        {"", -1, 0, 0},
        {"Total 15245667872 117571104 0% 20972341\n", 0, 15245667872, 117571104},
        {"To", -1, 0, 0},
        {"asdf\nasdf", -1, 0, 0},
        {"Total ", -1, 0, 0},
        {"Total 1", -1, 0, 0},
        {"Total 1\n", -1, 0, 0},
        {"Total 1 ", -1, 0, 0},
        {"Total 1 2", -1, 0, 0},
        {"Total 1 2 ", -1, 0, 0},
        {"Total 1 2\n", 0, 1, 2},
        {"Total 1 2 \n", 0, 1, 2},
        {"Total a 2 \n", -1, 0, 0},
        {"Total 1 b \n", -1, 0, 0},
        {"Total a b \n", -1, 0, 0},
        {"stuff\nTotal 1 2 \n", 0, 1, 2},
        {"0 1 2 3\nTotal 1 2 \n", 0, 1, 2},
        {NULL, 0, 0, 0}
    };

    collie_test vdi_list_tests[] = {
        {"", -1, 0, 0},
        {"= test 3 10 20 0 1336557216 7c2b27\n", 0, 10, 20},
        {"= test\\ with\\ spaces 3 10 20 0 1336557216 7c2b27\n", 0, 10, 20},
        {"= backslashattheend\\\\ 3 10 20 0 1336557216 7c2b27\n", 0, 10, 20},
        {"s test 1 10 20 0 1336556634 7c2b25\n= test 3 50 60 0 1336557216 7c2b27\n", 0, 50, 60},
        {"=", -1, 0, 0},
        {"= test", -1, 0, 0},
        {"= test ", -1, 0, 0},
        {"= test 1", -1, 0, 0},
        {"= test 1 ", -1, 0, 0},
        {"= test 1 2", -1, 0, 0},
        {"= test 1 2 ", -1, 0, 0},
        {"= test 1 2 3", -1, 0, 0},
        {NULL, 0, 0, 0}
    };

    collie_test *test = node_info_tests;

    poolxml = g_strdup_printf("%s/storagepoolxml2xmlin/pool-sheepdog.xml",
                              abs_srcdir);

    volxml = g_strdup_printf("%s/storagevolxml2xmlin/vol-sheepdog.xml",
                             abs_srcdir);

#define DO_TEST_NODE(collie) \
    do { \
        struct testNodeInfoParserData data = { \
            .data = collie, \
            .poolxml = poolxml, \
        }; \
        if (virTestRun("node_info_parser", test_node_info_parser, \
                       &data) < 0) \
            ret = -1; \
    } while (0)

    while (test->output != NULL) {
        DO_TEST_NODE(*test);
        ++test;
    }


#define DO_TEST_VDI(collie) \
    do { \
        struct testVDIListParserData data = { \
            .data = collie, \
            .poolxml = poolxml, \
            .volxml = volxml, \
        }; \
        if (virTestRun("vdi_list_parser", test_vdi_list_parser, \
                       &data) < 0) \
            ret = -1; \
    } while (0)

    test = vdi_list_tests;

    while (test->output != NULL) {
        DO_TEST_VDI(*test);
        ++test;
    }

    return ret == 0 ? EXIT_SUCCESS : EXIT_FAILURE;
}

VIR_TEST_MAIN(mymain)
