/*
 *  stream.c
 *
 *  Copyright 2009 Arnaud Soyez <weboide@codealpha.net>
 *
 *  This file is part of AudioPreview.
 *
 *  AudioPreview is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  AudioPreview is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with AudioPreview.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "stream.h"

/**
 * Return a new stream struct using the given GFile
 * Returns a ptr to Stream if success, NULL otherwise.
 */
Stream* stream_new (GFile *file)
{
  Stream *stream;
  
  if (file == NULL)
    return NULL;
  
  stream = g_new (Stream, 1);
  
  if (stream == NULL)
    return NULL;
  
  stream->file      = file;
  stream->duration  = -1;
  stream->position  = -1;
  stream->artist    = NULL;
  stream->title     = NULL;
  stream->uri       = stream_get_uri (stream);
  stream->filepath  = stream_get_filepath (stream);
  stream->buffering_perc = -1;
  stream->startposition  = -1;
  stream->update_function_source_id          = 0;
  stream->paused    = FALSE;
  
  return stream;
}

/**
 * Define the title of the given *stream to *title
 * Returns TRUE if success, FALSE otherwise.
 */
gboolean stream_set_title (Stream *stream, const gchar *title)
{
  static GStaticMutex mutex = G_STATIC_MUTEX_INIT;
  
  if (!stream)
    return FALSE;
  g_static_mutex_lock (&mutex);
  if (stream->title)
    g_free (stream->title);
  stream->title = g_strdup (title);
  g_static_mutex_unlock (&mutex);
  return TRUE;
}

/**
 * Define the artist of the given *stream to *artist
 * Returns TRUE if success, FALSE otherwise.
 */
gboolean stream_set_artist (Stream *stream, const gchar *artist)
{
  static GStaticMutex mutex = G_STATIC_MUTEX_INIT;
  
  if (!stream)
    return FALSE;
  
  g_static_mutex_lock (&mutex);
  if (stream->artist)
    g_free (stream->artist);
  stream->artist = g_strdup (artist);
  g_static_mutex_unlock (&mutex);
  return TRUE;
}

/**
 * Define the duration of the given *stream to the given duration
 * Returns TRUE if success, FALSE otherwise.
 */
gboolean stream_set_duration (Stream *stream, const gint64 duration)
{
  if (!stream)
    return FALSE;
  stream->duration = duration;
  return TRUE;
}

/**
 * Define the playing position of the given *stream to the given position
 * This is just an assignment to stream->position.
 * DO NOT USE TO SEEK. To seek, see player_seek().
 * Returns TRUE if success, FALSE otherwise.
 */
gboolean stream_set_position (Stream *stream, const gint64 position)
{
  if (!stream)
    return FALSE;
  stream->position = position;
  return TRUE;
}

/**
 * Define the buffering perc of the given *stream to the given percentage
 * Returns TRUE if success, FALSE otherwise.
 */
gboolean stream_set_buffering_perc (Stream *stream, const gint perc)
{
  if (!stream)
    return FALSE;
  stream->buffering_perc = perc;
  return TRUE;
}

/**
 * Returns the uri of the stream
 * Returns a gchar* if success, NULL otherwise.
 */
gchar* stream_get_uri (Stream *stream)
{
  if (!stream)
    return NULL;
  return g_file_get_uri (stream->file);
}

/**
 * Returns the filepath of the stream
 * Returns a gchar* if success, NULL otherwise.
 */
gchar* stream_get_filepath (Stream *stream)
{
  if (!stream)
    return NULL;
  return g_file_get_path (stream->file);
}

/**
 * Returns the name (track/stream info) of the stream
 * Returns a gchar* if success, NULL otherwise.
 */
gchar* stream_get_name_string (Stream *stream)
{
  if (!stream)
    return NULL;
  
  gchar *info;
  
  if (stream->artist)
  {
    if (stream->title)
      info = g_strdup_printf ("%s - %s", stream->artist, stream->title);
    else
      info = g_strdup_printf ("%s", stream->artist);
  }
  else
  {
    if (stream->title)
      info = g_strdup_printf ("%s", stream->title);
    else
    {
      GFileInfo* fileinfo = g_file_query_info (stream->file,
                                  G_FILE_ATTRIBUTE_STANDARD_DISPLAY_NAME,
                                  G_FILE_QUERY_INFO_NONE, NULL, NULL);
      if (fileinfo)
      {
        info = g_strdup (g_file_info_get_display_name (fileinfo));
        g_object_unref (fileinfo);
      }
      else
      {
        info = g_strdup_printf (_("(unknown)"));
      }
    }
  }
  
  return info;
}

/**
 * Returns the duration (as a user-readable string) of the stream
 * Minimum length is 7+7+3 characters long (maximum is estimated at 13+13+3)
 * Returns a gchar* if success, NULL otherwise.
 */
gchar* stream_get_duration_string (Stream *stream)
{
  if (!stream)
    return NULL;
  
  gchar *duration, *position, *ret;
  
  if (stream->duration >= 0)
  {
    gint hour, min, sec;
    split_nanoseconds (stream->duration, &hour, &min, &sec);
    duration = g_strdup_printf ("%u:%02u:%02u", hour, min, sec);
  }
  else
  {
    duration = g_strdup_printf ("-:--:--");
  }
  
  if (stream->position >= 0)
  {
    gint hour, min, sec;
    split_nanoseconds (stream->position, &hour, &min, &sec);
    position = g_strdup_printf ("%u:%02u:%02u", hour, min, sec);
  }
  else
  {
    position = g_strdup_printf ("-:--:--");
  }
  
  ret = g_strdup_printf ("(%s/%s)", position, duration);
  
  g_free (duration);
  g_free (position);
  return ret;
}

/**
 * Splits a gint64 nanoseconds into gint's hour/min/secs
 */
void split_nanoseconds (gint64 nanosecs, gint *hour, gint *min, gint *secs)
{
  *secs = (gint) (nanosecs / GST_SECOND);
  *hour = *secs / 3600;
  *secs -= (*hour * 3600);
  *min = *secs / 60;
  *secs -= (*min * 60);
}

/**
 * Free a stream and its content
 */
void stream_free (Stream* stream)
{
  if (stream->title)
    g_free (stream->title);
  if (stream->artist)
    g_free (stream->artist);
  if (stream->filepath)
    g_free (stream->filepath);
  if (stream->uri)
    g_free (stream->uri);
}

/**
 * Add an updater func to be called every STREAM_INFO_UPDATER_RATE millesconds
 * with the given stream as a parameter.
 * There should only be one call per stream.
 */
void stream_add_updater (Stream* stream, GSourceFunc func)
{
  g_debug ("%s: adding update function", __FUNCTION__);
  g_assert (stream->update_function_source_id == 0);
  
  stream->update_function_source_id = 
      g_timeout_add (STREAM_INFO_UPDATER_RATE, func, stream);
}

/**
 * Remove the added updater function from the stream
 * (previously added with stream_add_updater)
 */
void stream_remove_updater (Stream* stream)
{
  if (stream->update_function_source_id != 0)
  {
    g_debug ("%s: removing update function", __FUNCTION__);
    g_source_remove (stream->update_function_source_id);
    stream->update_function_source_id = 0;
  }
}

/**
 * Clear the line and print the given stream info to stdout
 * (artist,title,duration,position)
 */
void stream_print_info (Stream *stream)
{
  gchar *namestr = stream_get_name_string (stream);
  gchar *durstr  = stream_get_duration_string (stream);
  gboolean isbuffering = (stream->buffering_perc<100 && stream->buffering_perc>=0);
  gint max_term_size = ap_config.terminal_cols;
  
  // if tty, clear whole line first
  if (isatty (STDOUT_FILENO))
    g_print ("\r");
  
  COLOR_START_PRINT (color_magenta);
  if (isbuffering)
  {
    g_printf ("[buffering %i%%]", stream->buffering_perc);  // max 19 character long (with the %i%%)
    max_term_size -= 16;
  }
  else if (stream->paused)
  {
    g_printf ("[paused]");
    max_term_size -= 8;
  }
  else
  {
    g_printf ("[playing]");
    max_term_size -= 9;
  }
  COLOR_END_PRINT (color_magenta);
  
  
  glong max_name_size = (glong)max_term_size - 1 - ((glong)strlen (durstr) + 2);
  
  // If we don't have space at all, we don't print
  if (max_name_size > 0)
  {
    
    /* Check if namestr is too long */
    if ( g_utf8_strlen (namestr, -1) > max_name_size) 
    {
      // FIXME: this needs to use a wide-character-size-function
      gchar *tmp = malloc (strlen (namestr) + 1);
      g_utf8_strncpy (tmp, namestr, max_name_size);
      g_free (namestr);
      namestr = tmp;
    }
    
    // Print playing stream info
    //~ COLOR_START_PRINT (color_bold_magenta);
    g_printf (" %s", namestr); // 10 character long (without the %s)
    //~ COLOR_END_PRINT (color_bold_magenta);
    
    max_term_size -= max_name_size;
  }
  g_free (namestr);
  
  if (max_term_size > (gint)strlen (durstr) + 2)
  {
    // Print duration info
    COLOR_START_PRINT (color_magenta);
    g_printf (" %s", durstr); // 1 character long (without the %s)
    COLOR_END_PRINT (color_magenta);
  }
  g_free (durstr);
  
  /* clear the end of line (if tty) */
  if (isatty (STDOUT_FILENO))
    g_print ("%s", CLEAREOL);
}

/**
 * Get the current elapsed time of the given stream
 */
void stream_get_elapsed_time (Stream *stream, gint64 *returnvalue)
{
  if (stream->position < stream->startposition)
  {
    g_warning ("Debug: position < startposition (%" G_GINT64_FORMAT " < %" G_GINT64_FORMAT ")",
             stream->position, stream->startposition);
    *returnvalue = 0;
    return;
  }
  *returnvalue = stream->position - stream->startposition;
}

void stream_set_paused (Stream *stream, gboolean b)
{
  if (!stream)
    return;
  
  stream->paused = b;
}
