from __future__ import absolute_import, division, print_function

import libtbx.load_env
from libtbx.test_utils import show_diff, assert_lines_in_text
from mmtbx.regression.pdb_interpretation.tst_using_ncs_1 import get_geometry_stats

pdb_str = """\
ATOM      1  N   GLY A   1      -9.009   4.612   6.102  1.00 16.77           N
ATOM      2  CA  GLY A   1      -9.052   4.207   4.651  1.00 16.57           C
ATOM      3  C   GLY A   1      -8.015   3.140   4.419  1.00 16.16           C
ATOM      4  O   GLY A   1      -7.523   2.521   5.381  1.00 16.78           O
ATOM      5  N   ASN A   2      -7.656   2.923   3.155  1.00 15.02           N
ATOM      6  CA  ASN A   2      -6.522   2.038   2.831  1.00 14.10           C
ATOM      7  C   ASN A   2      -5.241   2.537   3.427  1.00 13.13           C
ATOM      8  O   ASN A   2      -4.978   3.742   3.426  1.00 11.91           O
ATOM      9  CB  ASN A   2      -6.346   1.881   1.341  1.00 15.38           C
ATOM     10  CG  ASN A   2      -7.584   1.342   0.692  1.00 14.08           C
ATOM     11  OD1 ASN A   2      -8.025   0.227   1.016  1.00 17.46           O
ATOM     12  ND2 ASN A   2      -8.204   2.155  -0.169  1.00 11.72           N
ATOM     13  N   ASN A   3      -4.438   1.590   3.905  1.00 12.26           N
ATOM     14  CA  ASN A   3      -3.193   1.904   4.589  1.00 11.74           C
ATOM     15  C   ASN A   3      -1.955   1.332   3.895  1.00 11.10           C
ATOM     16  O   ASN A   3      -1.872   0.119   3.648  1.00 10.42           O
ATOM     17  CB  ASN A   3      -3.259   1.378   6.042  1.00 12.15           C
ATOM     18  CG  ASN A   3      -2.006   1.739   6.861  1.00 12.82           C
ATOM     19  OD1 ASN A   3      -1.702   2.925   7.072  1.00 15.05           O
ATOM     20  ND2 ASN A   3      -1.271   0.715   7.306  1.00 13.48           N
ATOM     21  N   GLN A   4      -1.005   2.228   3.598  1.00 10.29           N
ATOM     22  CA  GLN A   4       0.384   1.888   3.199  1.00 10.53           C
ATOM     23  C   GLN A   4       1.435   2.606   4.088  1.00 10.24           C
ATOM     24  O   GLN A   4       1.547   3.843   4.115  1.00  8.86           O
ATOM     25  CB  GLN A   4       0.656   2.148   1.711  1.00  9.80           C
ATOM     26  CG  GLN A   4       1.944   1.458   1.213  1.00 10.25           C
ATOM     27  CD  GLN A   4       2.504   2.044  -0.089  1.00 12.43           C
ATOM     28  OE1 GLN A   4       2.744   3.268  -0.190  1.00 14.62           O
ATOM     29  NE2 GLN A   4       2.750   1.161  -1.091  1.00  9.05           N
ATOM     30  N   GLN A   5       2.154   1.821   4.871  1.00 10.38           N
ATOM     31  CA  GLN A   5       3.270   2.361   5.640  1.00 11.39           C
ATOM     32  C   GLN A   5       4.594   1.768   5.172  1.00 11.52           C
ATOM     33  O   GLN A   5       4.768   0.546   5.054  1.00 12.05           O
ATOM     34  CB  GLN A   5       3.056   2.183   7.147  1.00 11.96           C
ATOM     35  CG  GLN A   5       1.829   2.950   7.647  1.00 10.81           C
ATOM     36  CD  GLN A   5       1.344   2.414   8.954  1.00 13.10           C
ATOM     37  OE1 GLN A   5       0.774   1.325   9.002  1.00 10.65           O
ATOM     38  NE2 GLN A   5       1.549   3.187  10.039  1.00 12.30           N
ATOM     39  N   ASN A   6       5.514   2.664   4.856  1.00 11.99           N
ATOM     40  CA  ASN A   6       6.831   2.310   4.318  1.00 12.30           C
ATOM     41  C   ASN A   6       7.854   2.761   5.324  1.00 13.40           C
ATOM     42  O   ASN A   6       8.219   3.943   5.374  1.00 13.92           O
ATOM     43  CB  ASN A   6       7.065   3.016   2.993  1.00 12.13           C
ATOM     44  CG  ASN A   6       5.961   2.735   2.003  1.00 12.77           C
ATOM     45  OD1 ASN A   6       5.798   1.604   1.551  1.00 14.27           O
ATOM     46  ND2 ASN A   6       5.195   3.747   1.679  1.00 10.07           N
ATOM     47  N   TYR A   7       8.292   1.817   6.147  1.00 14.70           N
ATOM     48  CA  TYR A   7       9.159   2.144   7.299  1.00 15.18           C
ATOM     49  C   TYR A   7      10.603   2.331   6.885  1.00 15.91           C
ATOM     50  O   TYR A   7      11.041   1.811   5.855  1.00 15.76           O
ATOM     51  CB  TYR A   7       9.061   1.065   8.369  1.00 15.35           C
ATOM     52  CG  TYR A   7       7.665   0.929   8.902  1.00 14.45           C
ATOM     53  CD1 TYR A   7       7.210   1.756   9.920  1.00 14.80           C
ATOM     54  CD2 TYR A   7       6.771   0.021   8.327  1.00 15.68           C
ATOM     55  CE1 TYR A   7       5.904   1.649  10.416  1.00 14.33           C
ATOM     56  CE2 TYR A   7       5.480  -0.094   8.796  1.00 13.46           C
ATOM     57  CZ  TYR A   7       5.047   0.729   9.831  1.00 15.09           C
ATOM     58  OH  TYR A   7       3.766   0.589  10.291  1.00 14.39           O
ATOM     59  OXT TYR A   7      11.358   2.999   7.612  1.00 17.49           O
TER
HETATM   60  O   HOH A   8      -6.471   5.227   7.124  1.00 22.62           O
HETATM   61  O   HOH A   9      10.431   1.858   3.216  1.00 19.71           O
HETATM   62  O   HOH A  10     -11.286   1.756  -1.468  1.00 17.08           O
HETATM   63  O   HOH A  11      11.808   4.179   9.970  1.00 23.99           O
HETATM   64  O   HOH A  12      13.605   1.327   9.198  1.00 26.17           O
HETATM   65  O   HOH A  13      -2.749   3.429  10.024  1.00 39.15           O
HETATM   66  O   HOH A  14      -1.500   0.682  10.967  1.00 43.49           O
ATOM     67  N   GLY B   1      -9.009  -0.254   6.102  1.00 16.77           N
ATOM     68  CA  GLY B   1      -9.052  -0.659   4.651  1.00 16.57           C
ATOM     69  C   GLY B   1      -8.015  -1.726   4.419  1.00 16.16           C
ATOM     70  O   GLY B   1      -7.523  -2.345   5.381  1.00 16.78           O
ATOM     71  N   ASN B   2      -7.656  -1.943   3.155  1.00 15.02           N
ATOM     72  CA  ASN B   2      -6.522  -2.828   2.831  1.00 14.10           C
ATOM     73  C   ASN B   2      -5.241  -2.329   3.427  1.00 13.13           C
ATOM     74  O   ASN B   2      -4.978  -1.124   3.426  1.00 11.91           O
ATOM     75  CB  ASN B   2      -6.346  -2.985   1.341  1.00 15.38           C
ATOM     76  CG  ASN B   2      -7.584  -3.524   0.692  1.00 14.08           C
ATOM     77  OD1 ASN B   2      -8.025  -4.639   1.016  1.00 17.46           O
ATOM     78  ND2 ASN B   2      -8.204  -2.711  -0.169  1.00 11.72           N
ATOM     79  N   ASN B   3      -4.438  -3.276   3.905  1.00 12.26           N
ATOM     80  CA  ASN B   3      -3.193  -2.962   4.589  1.00 11.74           C
ATOM     81  C   ASN B   3      -1.955  -3.534   3.895  1.00 11.10           C
ATOM     82  O   ASN B   3      -1.872  -4.747   3.648  1.00 10.42           O
ATOM     83  CB  ASN B   3      -3.259  -3.488   6.042  1.00 12.15           C
ATOM     84  CG  ASN B   3      -2.006  -3.127   6.861  1.00 12.82           C
ATOM     85  OD1 ASN B   3      -1.702  -1.941   7.072  1.00 15.05           O
ATOM     86  ND2 ASN B   3      -1.271  -4.151   7.306  1.00 13.48           N
ATOM     87  N   GLN B   4      -1.005  -2.638   3.598  1.00 10.29           N
ATOM     88  CA  GLN B   4       0.384  -2.978   3.199  1.00 10.53           C
ATOM     89  C   GLN B   4       1.435  -2.260   4.088  1.00 10.24           C
ATOM     90  O   GLN B   4       1.547  -1.023   4.115  1.00  8.86           O
ATOM     91  CB  GLN B   4       0.656  -2.718   1.711  1.00  9.80           C
ATOM     92  CG  GLN B   4       1.944  -3.408   1.213  1.00 10.25           C
ATOM     93  CD  GLN B   4       2.504  -2.822  -0.089  1.00 12.43           C
ATOM     94  OE1 GLN B   4       2.744  -1.598  -0.190  1.00 14.62           O
ATOM     95  NE2 GLN B   4       2.750  -3.705  -1.091  1.00  9.05           N
ATOM     96  N   GLN B   5       2.154  -3.045   4.871  1.00 10.38           N
ATOM     97  CA  GLN B   5       3.270  -2.505   5.640  1.00 11.39           C
ATOM     98  C   GLN B   5       4.594  -3.098   5.172  1.00 11.52           C
ATOM     99  O   GLN B   5       4.768  -4.320   5.054  1.00 12.05           O
ATOM    100  CB  GLN B   5       3.056  -2.683   7.147  1.00 11.96           C
ATOM    101  CG  GLN B   5       1.829  -1.916   7.647  1.00 10.81           C
ATOM    102  CD  GLN B   5       1.344  -2.452   8.954  1.00 13.10           C
ATOM    103  OE1 GLN B   5       0.774  -3.541   9.002  1.00 10.65           O
ATOM    104  NE2 GLN B   5       1.549  -1.679  10.039  1.00 12.30           N
ATOM    105  N   ASN B   6       5.514  -2.202   4.856  1.00 11.99           N
ATOM    106  CA  ASN B   6       6.831  -2.556   4.318  1.00 12.30           C
ATOM    107  C   ASN B   6       7.854  -2.105   5.324  1.00 13.40           C
ATOM    108  O   ASN B   6       8.219  -0.923   5.374  1.00 13.92           O
ATOM    109  CB  ASN B   6       7.065  -1.850   2.993  1.00 12.13           C
ATOM    110  CG  ASN B   6       5.961  -2.131   2.003  1.00 12.77           C
ATOM    111  OD1 ASN B   6       5.798  -3.262   1.551  1.00 14.27           O
ATOM    112  ND2 ASN B   6       5.195  -1.119   1.679  1.00 10.07           N
ATOM    113  N   TYR B   7       8.292  -3.049   6.147  1.00 14.70           N
ATOM    114  CA  TYR B   7       9.159  -2.722   7.299  1.00 15.18           C
ATOM    115  C   TYR B   7      10.603  -2.535   6.885  1.00 15.91           C
ATOM    116  O   TYR B   7      11.041  -3.055   5.855  1.00 15.76           O
ATOM    117  CB  TYR B   7       9.061  -3.801   8.369  1.00 15.35           C
ATOM    118  CG  TYR B   7       7.665  -3.937   8.902  1.00 14.45           C
ATOM    119  CD1 TYR B   7       7.210  -3.110   9.920  1.00 14.80           C
ATOM    120  CD2 TYR B   7       6.771  -4.845   8.327  1.00 15.68           C
ATOM    121  CE1 TYR B   7       5.904  -3.217  10.416  1.00 14.33           C
ATOM    122  CE2 TYR B   7       5.480  -4.960   8.796  1.00 13.46           C
ATOM    123  CZ  TYR B   7       5.047  -4.137   9.831  1.00 15.09           C
ATOM    124  OH  TYR B   7       3.766  -4.277  10.291  1.00 14.39           O
ATOM    125  OXT TYR B   7      11.358  -1.867   7.612  1.00 17.49           O
TER
HETATM  126  O   HOH C   8      -6.471   0.361   7.124  1.00 22.62           O
HETATM  127  O   HOH C   9      10.431  -3.008   3.216  1.00 19.71           O
HETATM  128  O   HOH C  10     -11.286  -3.110  -1.468  1.00 17.08           O
HETATM  129  O   HOH C  11      11.808  -0.687   9.970  1.00 23.99           O
HETATM  130  O   HOH C  12      13.605  -3.539   9.198  1.00 26.17           O
HETATM  131  O   HOH C  13      -2.749  -1.437  10.024  1.00 39.15           O
HETATM  132  O   HOH C  14      -1.500  -4.184  10.967  1.00 43.49           O
HETATM  133  O   HOH E  13      -4.146   0.996  12.418  1.00 39.15           O
HETATM  134  O   HOH E  14      -5.395  -1.751  11.475  1.00 43.49           O
HETATM  135  O   HOH G  13      -4.146   5.862  12.418  1.00 39.15           O
HETATM  136  O   HOH G  14      -5.395   3.115  11.475  1.00 43.49           O
ATOM    137  N   GLY H   1      -9.009   9.478   6.102  1.00 16.77           N
ATOM    138  CA  GLY H   1      -9.052   9.073   4.651  1.00 16.57           C
ATOM    139  C   GLY H   1      -8.015   8.006   4.419  1.00 16.16           C
ATOM    140  O   GLY H   1      -7.523   7.387   5.381  1.00 16.78           O
ATOM    141  N   ASN H   2      -7.656   7.789   3.155  1.00 15.02           N
ATOM    142  CA  ASN H   2      -6.522   6.904   2.831  1.00 14.10           C
ATOM    143  C   ASN H   2      -5.241   7.403   3.427  1.00 13.13           C
ATOM    144  O   ASN H   2      -4.978   8.608   3.426  1.00 11.91           O
ATOM    145  CB  ASN H   2      -6.346   6.747   1.341  1.00 15.38           C
ATOM    146  CG  ASN H   2      -7.584   6.208   0.692  1.00 14.08           C
ATOM    147  OD1 ASN H   2      -8.025   5.093   1.016  1.00 17.46           O
ATOM    148  ND2 ASN H   2      -8.204   7.021  -0.169  1.00 11.72           N
ATOM    149  N   ASN H   3      -4.438   6.456   3.905  1.00 12.26           N
ATOM    150  CA  ASN H   3      -3.193   6.770   4.589  1.00 11.74           C
ATOM    151  C   ASN H   3      -1.955   6.198   3.895  1.00 11.10           C
ATOM    152  O   ASN H   3      -1.872   4.985   3.648  1.00 10.42           O
ATOM    153  CB  ASN H   3      -3.259   6.244   6.042  1.00 12.15           C
ATOM    154  CG  ASN H   3      -2.006   6.605   6.861  1.00 12.82           C
ATOM    155  OD1 ASN H   3      -1.702   7.791   7.072  1.00 15.05           O
ATOM    156  ND2 ASN H   3      -1.271   5.581   7.306  1.00 13.48           N
ATOM    157  N   GLN H   4      -1.005   7.094   3.598  1.00 10.29           N
ATOM    158  CA  GLN H   4       0.384   6.754   3.199  1.00 10.53           C
ATOM    159  C   GLN H   4       1.435   7.472   4.088  1.00 10.24           C
ATOM    160  O   GLN H   4       1.547   8.709   4.115  1.00  8.86           O
ATOM    161  CB  GLN H   4       0.656   7.014   1.711  1.00  9.80           C
ATOM    162  CG  GLN H   4       1.944   6.324   1.213  1.00 10.25           C
ATOM    163  CD  GLN H   4       2.504   6.910  -0.089  1.00 12.43           C
ATOM    164  OE1 GLN H   4       2.744   8.134  -0.190  1.00 14.62           O
ATOM    165  NE2 GLN H   4       2.750   6.027  -1.091  1.00  9.05           N
ATOM    166  N   GLN H   5       2.154   6.687   4.871  1.00 10.38           N
ATOM    167  CA  GLN H   5       3.270   7.227   5.640  1.00 11.39           C
ATOM    168  C   GLN H   5       4.594   6.634   5.172  1.00 11.52           C
ATOM    169  O   GLN H   5       4.768   5.412   5.054  1.00 12.05           O
ATOM    170  CB  GLN H   5       3.056   7.049   7.147  1.00 11.96           C
ATOM    171  CG  GLN H   5       1.829   7.816   7.647  1.00 10.81           C
ATOM    172  CD  GLN H   5       1.344   7.280   8.954  1.00 13.10           C
ATOM    173  OE1 GLN H   5       0.774   6.191   9.002  1.00 10.65           O
ATOM    174  NE2 GLN H   5       1.549   8.053  10.039  1.00 12.30           N
ATOM    175  N   ASN H   6       5.514   7.530   4.856  1.00 11.99           N
ATOM    176  CA  ASN H   6       6.831   7.176   4.318  1.00 12.30           C
ATOM    177  C   ASN H   6       7.854   7.627   5.324  1.00 13.40           C
ATOM    178  O   ASN H   6       8.219   8.809   5.374  1.00 13.92           O
ATOM    179  CB  ASN H   6       7.065   7.882   2.993  1.00 12.13           C
ATOM    180  CG  ASN H   6       5.961   7.601   2.003  1.00 12.77           C
ATOM    181  OD1 ASN H   6       5.798   6.470   1.551  1.00 14.27           O
ATOM    182  ND2 ASN H   6       5.195   8.613   1.679  1.00 10.07           N
ATOM    183  N   TYR H   7       8.292   6.683   6.147  1.00 14.70           N
ATOM    184  CA  TYR H   7       9.159   7.010   7.299  1.00 15.18           C
ATOM    185  C   TYR H   7      10.603   7.197   6.885  1.00 15.91           C
ATOM    186  O   TYR H   7      11.041   6.677   5.855  1.00 15.76           O
ATOM    187  CB  TYR H   7       9.061   5.931   8.369  1.00 15.35           C
ATOM    188  CG  TYR H   7       7.665   5.795   8.902  1.00 14.45           C
ATOM    189  CD1 TYR H   7       7.210   6.622   9.920  1.00 14.80           C
ATOM    190  CD2 TYR H   7       6.771   4.887   8.327  1.00 15.68           C
ATOM    191  CE1 TYR H   7       5.904   6.515  10.416  1.00 14.33           C
ATOM    192  CE2 TYR H   7       5.480   4.772   8.796  1.00 13.46           C
ATOM    193  CZ  TYR H   7       5.047   5.595   9.831  1.00 15.09           C
ATOM    194  OH  TYR H   7       3.766   5.455  10.291  1.00 14.39           O
ATOM    195  OXT TYR H   7      11.358   7.865   7.612  1.00 17.49           O
TER
HETATM  196  O   HOH I   9      10.431   6.724   3.216  1.00 19.71           O
HETATM  197  O   HOH I  10     -11.286   6.622  -1.468  1.00 17.08           O
HETATM  198  O   HOH I  12      13.605   6.193   9.198  1.00 26.17           O
HETATM  199  O   HOH I  14      -1.500   5.548  10.967  1.00 43.49           O
ATOM    200  N   GLN L   5      12.888   4.254  17.571  1.00 10.38           N
ATOM    201  CA  GLN L   5      11.772   4.794  16.802  1.00 11.39           C
ATOM    202  C   GLN L   5      10.448   4.201  17.270  1.00 11.52           C
ATOM    203  O   GLN L   5      10.274   2.979  17.388  1.00 12.05           O
ATOM    204  CB  GLN L   5      11.986   4.616  15.295  1.00 11.96           C
ATOM    205  CG  GLN L   5      13.213   5.383  14.795  1.00 10.81           C
ATOM    206  CD  GLN L   5      13.698   4.847  13.488  1.00 13.10           C
ATOM    207  OE1 GLN L   5      14.268   3.758  13.440  1.00 10.65           O
ATOM    208  NE2 GLN L   5      13.493   5.620  12.403  1.00 12.30           N
ATOM    209  N   TYR L   7       6.750   4.250  16.295  1.00 14.70           N
ATOM    210  CA  TYR L   7       5.883   4.577  15.143  1.00 15.18           C
ATOM    211  C   TYR L   7       4.439   4.764  15.557  1.00 15.91           C
ATOM    212  O   TYR L   7       4.001   4.244  16.587  1.00 15.76           O
ATOM    213  CB  TYR L   7       5.981   3.498  14.073  1.00 15.35           C
ATOM    214  CG  TYR L   7       7.377   3.362  13.540  1.00 14.45           C
ATOM    215  CD1 TYR L   7       7.832   4.189  12.522  1.00 14.80           C
ATOM    216  CD2 TYR L   7       8.271   2.454  14.115  1.00 15.68           C
ATOM    217  CE1 TYR L   7       9.138   4.082  12.026  1.00 14.33           C
ATOM    218  CE2 TYR L   7       9.562   2.339  13.646  1.00 13.46           C
ATOM    219  CZ  TYR L   7       9.995   3.162  12.611  1.00 15.09           C
ATOM    220  OH  TYR L   7      11.276   3.022  12.151  1.00 14.39           O
ATOM    221  OXT TYR L   7       3.684   5.432  14.830  1.00 17.49           O
TER
HETATM  222  O   HOH M  11       3.234   6.612  12.472  1.00 23.99           O
HETATM  223  O   HOH M  12       1.437   3.760  13.244  1.00 26.17           O
HETATM  224  O   HOH M  14      16.542   3.115  11.475  1.00 43.49           O
ATOM    225  N   ASN N   2       7.656   0.490  -3.155  1.00 15.02           N
ATOM    226  CA  ASN N   2       6.522  -0.395  -2.831  1.00 14.10           C
ATOM    227  C   ASN N   2       5.241   0.104  -3.427  1.00 13.13           C
ATOM    228  O   ASN N   2       4.978   1.309  -3.426  1.00 11.91           O
ATOM    229  CB  ASN N   2       6.346  -0.552  -1.341  1.00 15.38           C
ATOM    230  CG  ASN N   2       7.584  -1.091  -0.692  1.00 14.08           C
ATOM    231  OD1 ASN N   2       8.025  -2.206  -1.016  1.00 17.46           O
ATOM    232  ND2 ASN N   2       8.204  -0.278   0.169  1.00 11.72           N
ATOM    233  N   ASN N   3       4.438  -0.843  -3.905  1.00 12.26           N
ATOM    234  CA  ASN N   3       3.193  -0.529  -4.589  1.00 11.74           C
ATOM    235  C   ASN N   3       1.955  -1.101  -3.895  1.00 11.10           C
ATOM    236  O   ASN N   3       1.872  -2.314  -3.648  1.00 10.42           O
ATOM    237  CB  ASN N   3       3.259  -1.055  -6.042  1.00 12.15           C
ATOM    238  CG  ASN N   3       2.006  -0.694  -6.861  1.00 12.82           C
ATOM    239  OD1 ASN N   3       1.702   0.492  -7.072  1.00 15.05           O
ATOM    240  ND2 ASN N   3       1.271  -1.718  -7.306  1.00 13.48           N
ATOM    241  N   GLN N   4       1.005  -0.205  -3.598  1.00 10.29           N
ATOM    242  CA  GLN N   4      -0.384  -0.545  -3.199  1.00 10.53           C
ATOM    243  C   GLN N   4      -1.435   0.173  -4.088  1.00 10.24           C
ATOM    244  O   GLN N   4      -1.547   1.410  -4.115  1.00  8.86           O
ATOM    245  CB  GLN N   4      -0.656  -0.285  -1.711  1.00  9.80           C
ATOM    246  CG  GLN N   4      -1.944  -0.975  -1.213  1.00 10.25           C
ATOM    247  CD  GLN N   4      -2.504  -0.389   0.089  1.00 12.43           C
ATOM    248  OE1 GLN N   4      -2.744   0.835   0.190  1.00 14.62           O
ATOM    249  NE2 GLN N   4      -2.750  -1.272   1.091  1.00  9.05           N
ATOM    250  N   ASN N   6      -5.514   0.231  -4.856  1.00 11.99           N
ATOM    251  CA  ASN N   6      -6.831  -0.123  -4.318  1.00 12.30           C
ATOM    252  C   ASN N   6      -7.854   0.328  -5.324  1.00 13.40           C
ATOM    253  O   ASN N   6      -8.219   1.510  -5.374  1.00 13.92           O
ATOM    254  CB  ASN N   6      -7.065   0.583  -2.993  1.00 12.13           C
ATOM    255  CG  ASN N   6      -5.961   0.302  -2.003  1.00 12.77           C
ATOM    256  OD1 ASN N   6      -5.798  -0.829  -1.551  1.00 14.27           O
ATOM    257  ND2 ASN N   6      -5.195   1.314  -1.679  1.00 10.07           N
TER
HETATM  258  O   HOH O   9     -10.431  -0.575  -3.216  1.00 19.71           O
HETATM  259  O   HOH O  10      11.286  -0.677   1.468  1.00 17.08           O
ATOM    260  N   ASN P   2       7.656   5.356  -3.155  1.00 15.02           N
ATOM    261  CA  ASN P   2       6.522   4.471  -2.831  1.00 14.10           C
ATOM    262  C   ASN P   2       5.241   4.970  -3.427  1.00 13.13           C
ATOM    263  O   ASN P   2       4.978   6.175  -3.426  1.00 11.91           O
ATOM    264  CB  ASN P   2       6.346   4.314  -1.341  1.00 15.38           C
ATOM    265  CG  ASN P   2       7.584   3.775  -0.692  1.00 14.08           C
ATOM    266  OD1 ASN P   2       8.025   2.660  -1.016  1.00 17.46           O
ATOM    267  ND2 ASN P   2       8.204   4.588   0.169  1.00 11.72           N
ATOM    268  N   ASN P   3       4.438   4.023  -3.905  1.00 12.26           N
ATOM    269  CA  ASN P   3       3.193   4.337  -4.589  1.00 11.74           C
ATOM    270  C   ASN P   3       1.955   3.765  -3.895  1.00 11.10           C
ATOM    271  O   ASN P   3       1.872   2.552  -3.648  1.00 10.42           O
ATOM    272  CB  ASN P   3       3.259   3.811  -6.042  1.00 12.15           C
ATOM    273  CG  ASN P   3       2.006   4.172  -6.861  1.00 12.82           C
ATOM    274  OD1 ASN P   3       1.702   5.358  -7.072  1.00 15.05           O
ATOM    275  ND2 ASN P   3       1.271   3.148  -7.306  1.00 13.48           N
ATOM    276  N   GLN P   4       1.005   4.661  -3.598  1.00 10.29           N
ATOM    277  CA  GLN P   4      -0.384   4.321  -3.199  1.00 10.53           C
ATOM    278  C   GLN P   4      -1.435   5.039  -4.088  1.00 10.24           C
ATOM    279  O   GLN P   4      -1.547   6.276  -4.115  1.00  8.86           O
ATOM    280  CB  GLN P   4      -0.656   4.581  -1.711  1.00  9.80           C
ATOM    281  CG  GLN P   4      -1.944   3.891  -1.213  1.00 10.25           C
ATOM    282  CD  GLN P   4      -2.504   4.477   0.089  1.00 12.43           C
ATOM    283  OE1 GLN P   4      -2.744   5.701   0.190  1.00 14.62           O
ATOM    284  NE2 GLN P   4      -2.750   3.594   1.091  1.00  9.05           N
ATOM    285  N   ASN P   6      -5.514   5.097  -4.856  1.00 11.99           N
ATOM    286  CA  ASN P   6      -6.831   4.743  -4.318  1.00 12.30           C
ATOM    287  C   ASN P   6      -7.854   5.194  -5.324  1.00 13.40           C
ATOM    288  O   ASN P   6      -8.219   6.376  -5.374  1.00 13.92           O
ATOM    289  CB  ASN P   6      -7.065   5.449  -2.993  1.00 12.13           C
ATOM    290  CG  ASN P   6      -5.961   5.168  -2.003  1.00 12.77           C
ATOM    291  OD1 ASN P   6      -5.798   4.037  -1.551  1.00 14.27           O
ATOM    292  ND2 ASN P   6      -5.195   6.180  -1.679  1.00 10.07           N
ATOM    293  N   TYR P   7      -8.292   4.250  -6.147  1.00 14.70           N
ATOM    294  CA  TYR P   7      -9.159   4.577  -7.299  1.00 15.18           C
ATOM    295  C   TYR P   7     -10.603   4.764  -6.885  1.00 15.91           C
ATOM    296  O   TYR P   7     -11.041   4.244  -5.855  1.00 15.76           O
ATOM    297  CB  TYR P   7      -9.061   3.498  -8.369  1.00 15.35           C
ATOM    298  CG  TYR P   7      -7.665   3.362  -8.902  1.00 14.45           C
ATOM    299  CD1 TYR P   7      -7.210   4.189  -9.920  1.00 14.80           C
ATOM    300  CD2 TYR P   7      -6.771   2.454  -8.327  1.00 15.68           C
ATOM    301  CE1 TYR P   7      -5.904   4.082 -10.416  1.00 14.33           C
ATOM    302  CE2 TYR P   7      -5.480   2.339  -8.796  1.00 13.46           C
ATOM    303  CZ  TYR P   7      -5.047   3.162  -9.831  1.00 15.09           C
ATOM    304  OH  TYR P   7      -3.766   3.022 -10.291  1.00 14.39           O
ATOM    305  OXT TYR P   7     -11.358   5.432  -7.612  1.00 17.49           O
TER
HETATM  306  O   HOH Q   9     -10.431   4.291  -3.216  1.00 19.71           O
HETATM  307  O   HOH Q  10      11.286   4.189   1.468  1.00 17.08           O
ATOM    308  N   GLN R   5      12.888  -0.612  17.571  1.00 10.38           N
ATOM    309  CA  GLN R   5      11.772  -0.072  16.802  1.00 11.39           C
ATOM    310  C   GLN R   5      10.448  -0.665  17.270  1.00 11.52           C
ATOM    311  O   GLN R   5      10.274  -1.887  17.388  1.00 12.05           O
ATOM    312  CB  GLN R   5      11.986  -0.250  15.295  1.00 11.96           C
ATOM    313  CG  GLN R   5      13.213   0.517  14.795  1.00 10.81           C
ATOM    314  CD  GLN R   5      13.698  -0.019  13.488  1.00 13.10           C
ATOM    315  OE1 GLN R   5      14.268  -1.108  13.440  1.00 10.65           O
ATOM    316  NE2 GLN R   5      13.493   0.754  12.403  1.00 12.30           N
ATOM    317  N   TYR R   7       6.750  -0.616  16.295  1.00 14.70           N
ATOM    318  CA  TYR R   7       5.883  -0.289  15.143  1.00 15.18           C
ATOM    319  C   TYR R   7       4.439  -0.102  15.557  1.00 15.91           C
ATOM    320  O   TYR R   7       4.001  -0.622  16.587  1.00 15.76           O
ATOM    321  CB  TYR R   7       5.981  -1.368  14.073  1.00 15.35           C
ATOM    322  CG  TYR R   7       7.377  -1.504  13.540  1.00 14.45           C
ATOM    323  CD1 TYR R   7       7.832  -0.677  12.522  1.00 14.80           C
ATOM    324  CD2 TYR R   7       8.271  -2.412  14.115  1.00 15.68           C
ATOM    325  CE1 TYR R   7       9.138  -0.784  12.026  1.00 14.33           C
ATOM    326  CE2 TYR R   7       9.562  -2.527  13.646  1.00 13.46           C
ATOM    327  CZ  TYR R   7       9.995  -1.704  12.611  1.00 15.09           C
ATOM    328  OH  TYR R   7      11.276  -1.844  12.151  1.00 14.39           O
ATOM    329  OXT TYR R   7       3.684   0.566  14.830  1.00 17.49           O
TER
HETATM  330  O   HOH S  11       3.234   1.746  12.472  1.00 23.99           O
HETATM  331  O   HOH S  12       1.437  -1.106  13.244  1.00 26.17           O
ATOM    332  N   ASN V   6     -16.423   2.664   4.856  1.00 11.99           N
ATOM    333  CA  ASN V   6     -15.106   2.310   4.318  1.00 12.30           C
ATOM    334  C   ASN V   6     -14.083   2.761   5.324  1.00 13.40           C
ATOM    335  O   ASN V   6     -13.718   3.943   5.374  1.00 13.92           O
ATOM    336  CB  ASN V   6     -14.872   3.016   2.993  1.00 12.13           C
ATOM    337  CG  ASN V   6     -15.976   2.735   2.003  1.00 12.77           C
ATOM    338  OD1 ASN V   6     -16.139   1.604   1.551  1.00 14.27           O
ATOM    339  ND2 ASN V   6     -16.742   3.747   1.679  1.00 10.07           N
ATOM    340  N   TYR V   7     -13.645   1.817   6.147  1.00 14.70           N
ATOM    341  CA  TYR V   7     -12.778   2.144   7.299  1.00 15.18           C
ATOM    342  C   TYR V   7     -11.334   2.331   6.885  1.00 15.91           C
ATOM    343  O   TYR V   7     -10.896   1.811   5.855  1.00 15.76           O
ATOM    344  CB  TYR V   7     -12.876   1.065   8.369  1.00 15.35           C
ATOM    345  CG  TYR V   7     -14.272   0.929   8.902  1.00 14.45           C
ATOM    346  CD1 TYR V   7     -14.727   1.756   9.920  1.00 14.80           C
ATOM    347  CD2 TYR V   7     -15.166   0.021   8.327  1.00 15.68           C
ATOM    348  CE1 TYR V   7     -16.033   1.649  10.416  1.00 14.33           C
ATOM    349  CE2 TYR V   7     -16.457  -0.094   8.796  1.00 13.46           C
ATOM    350  CZ  TYR V   7     -16.890   0.729   9.831  1.00 15.09           C
ATOM    351  OH  TYR V   7     -18.171   0.589  10.291  1.00 14.39           O
ATOM    352  OXT TYR V   7     -10.579   2.999   7.612  1.00 17.49           O
TER
HETATM  353  O   HOH W   9     -11.506   1.858   3.216  1.00 19.71           O
HETATM  354  O   HOH W  11     -10.129   4.179   9.970  1.00 23.99           O
HETATM  355  O   HOH W  12      -8.332   1.327   9.198  1.00 26.17           O
ATOM    356  N   GLY X   1     -12.928   2.179  -6.102  1.00 16.77           N
ATOM    357  CA  GLY X   1     -12.885   1.774  -4.651  1.00 16.57           C
ATOM    358  C   GLY X   1     -13.922   0.707  -4.419  1.00 16.16           C
ATOM    359  O   GLY X   1     -14.414   0.088  -5.381  1.00 16.78           O
ATOM    360  N   ASN X   2     -14.281   0.490  -3.155  1.00 15.02           N
ATOM    361  CA  ASN X   2     -15.415  -0.395  -2.831  1.00 14.10           C
ATOM    362  C   ASN X   2     -16.696   0.104  -3.427  1.00 13.13           C
ATOM    363  O   ASN X   2     -16.959   1.309  -3.426  1.00 11.91           O
ATOM    364  CB  ASN X   2     -15.591  -0.552  -1.341  1.00 15.38           C
ATOM    365  CG  ASN X   2     -14.353  -1.091  -0.692  1.00 14.08           C
ATOM    366  OD1 ASN X   2     -13.912  -2.206  -1.016  1.00 17.46           O
ATOM    367  ND2 ASN X   2     -13.733  -0.278   0.169  1.00 11.72           N
TER
HETATM  368  O   HOH Y  10     -10.651  -0.677   1.468  1.00 17.08           O
ATOM    369  N   ASN Z   2     -14.281   5.356  -3.155  1.00 15.02           N
ATOM    370  CA  ASN Z   2     -15.415   4.471  -2.831  1.00 14.10           C
ATOM    371  C   ASN Z   2     -16.696   4.970  -3.427  1.00 13.13           C
ATOM    372  O   ASN Z   2     -16.959   6.175  -3.426  1.00 11.91           O
ATOM    373  CB  ASN Z   2     -15.591   4.314  -1.341  1.00 15.38           C
ATOM    374  CG  ASN Z   2     -14.353   3.775  -0.692  1.00 14.08           C
ATOM    375  OD1 ASN Z   2     -13.912   2.660  -1.016  1.00 17.46           O
ATOM    376  ND2 ASN Z   2     -13.733   4.588   0.169  1.00 11.72           N
TER
HETATM  377  O   HOH 0  10     -10.651   4.189   1.468  1.00 17.08           O
ATOM    378  N   GLY 1   1      12.928  -0.254   6.102  1.00 16.77           N
ATOM    379  CA  GLY 1   1      12.885  -0.659   4.651  1.00 16.57           C
ATOM    380  C   GLY 1   1      13.922  -1.726   4.419  1.00 16.16           C
ATOM    381  O   GLY 1   1      14.414  -2.345   5.381  1.00 16.78           O
TER
HETATM  382  O   HOH 2   8      15.466   0.361   7.124  1.00 22.62           O
ATOM    383  N   GLY 3   1      12.928   4.612   6.102  1.00 16.77           N
ATOM    384  CA  GLY 3   1      12.885   4.207   4.651  1.00 16.57           C
ATOM    385  C   GLY 3   1      13.922   3.140   4.419  1.00 16.16           C
ATOM    386  O   GLY 3   1      14.414   2.521   5.381  1.00 16.78           O
ATOM    387  N   ASN 3   2      14.281   2.923   3.155  1.00 15.02           N
ATOM    388  CA  ASN 3   2      15.415   2.038   2.831  1.00 14.10           C
ATOM    389  C   ASN 3   2      16.696   2.537   3.427  1.00 13.13           C
ATOM    390  O   ASN 3   2      16.959   3.742   3.426  1.00 11.91           O
ATOM    391  CB  ASN 3   2      15.591   1.881   1.341  1.00 15.38           C
ATOM    392  CG  ASN 3   2      14.353   1.342   0.692  1.00 14.08           C
ATOM    393  OD1 ASN 3   2      13.912   0.227   1.016  1.00 17.46           O
ATOM    394  ND2 ASN 3   2      13.733   2.155  -0.169  1.00 11.72           N
TER
HETATM  395  O   HOH 4   8      15.466   5.227   7.124  1.00 22.62           O
HETATM  396  O   HOH 4  10      10.651   1.756  -1.468  1.00 17.08           O
HETATM  397  O   HOH 8  11       3.234  -3.120  12.472  1.00 23.99           O
ATOM    398  N   TYR 9   7     -13.645   6.683   6.147  1.00 14.70           N
ATOM    399  CA  TYR 9   7     -12.778   7.010   7.299  1.00 15.18           C
ATOM    400  C   TYR 9   7     -11.334   7.197   6.885  1.00 15.91           C
ATOM    401  O   TYR 9   7     -10.896   6.677   5.855  1.00 15.76           O
ATOM    402  CB  TYR 9   7     -12.876   5.931   8.369  1.00 15.35           C
ATOM    403  CG  TYR 9   7     -14.272   5.795   8.902  1.00 14.45           C
ATOM    404  CD1 TYR 9   7     -14.727   6.622   9.920  1.00 14.80           C
ATOM    405  CD2 TYR 9   7     -15.166   4.887   8.327  1.00 15.68           C
ATOM    406  CE1 TYR 9   7     -16.033   6.515  10.416  1.00 14.33           C
ATOM    407  CE2 TYR 9   7     -16.457   4.772   8.796  1.00 13.46           C
ATOM    408  CZ  TYR 9   7     -16.890   5.595   9.831  1.00 15.09           C
ATOM    409  OH  TYR 9   7     -18.171   5.455  10.291  1.00 14.39           O
ATOM    410  OXT TYR 9   7     -10.579   7.865   7.612  1.00 17.49           O
TER
HETATM  411  O   HOH a   9     -11.506   6.724   3.216  1.00 19.71           O
HETATM  412  O   HOH a  12      -8.332   6.193   9.198  1.00 26.17           O
ATOM    413  N   ASN f   2      14.281   7.789   3.155  1.00 15.02           N
ATOM    414  CA  ASN f   2      15.415   6.904   2.831  1.00 14.10           C
ATOM    415  C   ASN f   2      16.696   7.403   3.427  1.00 13.13           C
ATOM    416  O   ASN f   2      16.959   8.608   3.426  1.00 11.91           O
ATOM    417  CB  ASN f   2      15.591   6.747   1.341  1.00 15.38           C
ATOM    418  CG  ASN f   2      14.353   6.208   0.692  1.00 14.08           C
ATOM    419  OD1 ASN f   2      13.912   5.093   1.016  1.00 17.46           O
ATOM    420  ND2 ASN f   2      13.733   7.021  -0.169  1.00 11.72           N
TER
ATOM    421  N   TYR h   7       6.750   9.116  16.295  1.00 14.70           N
ATOM    422  CA  TYR h   7       5.883   9.443  15.143  1.00 15.18           C
ATOM    423  C   TYR h   7       4.439   9.630  15.557  1.00 15.91           C
ATOM    424  O   TYR h   7       4.001   9.110  16.587  1.00 15.76           O
ATOM    425  CB  TYR h   7       5.981   8.364  14.073  1.00 15.35           C
ATOM    426  CG  TYR h   7       7.377   8.228  13.540  1.00 14.45           C
ATOM    427  CD1 TYR h   7       7.832   9.055  12.522  1.00 14.80           C
ATOM    428  CD2 TYR h   7       8.271   7.320  14.115  1.00 15.68           C
ATOM    429  CE1 TYR h   7       9.138   8.948  12.026  1.00 14.33           C
ATOM    430  CE2 TYR h   7       9.562   7.205  13.646  1.00 13.46           C
ATOM    431  CZ  TYR h   7       9.995   8.028  12.611  1.00 15.09           C
ATOM    432  OH  TYR h   7      11.276   7.888  12.151  1.00 14.39           O
ATOM    433  OXT TYR h   7       3.684  10.298  14.830  1.00 17.49           O
TER
"""

def exercise_01():
  geom_ncs, log_ncs=get_geometry_stats(pdb_str, True)
  geom_no_ncs, log_no_ncs=get_geometry_stats(pdb_str, False)
  assert not show_diff(geom_ncs, geom_no_ncs)
  # assert not show_diff(log_ncs, log_no_ncs)
  assert_lines_in_text(log_ncs, """   Restraints were copied for chains:
    X, h, B, H, R, f
""")
  # print(log_ncs)

if(__name__ == "__main__"):
  if libtbx.env.find_in_repositories(relative_path="chem_data") is None:
    print("Skipping exercise_01(): chem_data directory not available")
  else:
    exercise_01()
    print('OK')

