/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcmaps_arguments.h
    \ingroup APIforLcmapsPlugins
    \author Martijn Steenbakkers and Oscar Koeroo for the EU DataGrid.
    \brief  Public header file to be used by plugins

    Routines to access the plugin arguments.

    The interface is composed of:
    -# lcmaps_setArgValue(): Set the value of argument with name argName of argType to value
    -# lcmaps_getArgValue(): Get the value of argument with name argName of argType
    -# lcmaps_findArgName(): Get index of argument with name argName
    -# lcmaps_cntArgs(): Count the number of arguments
*/

#ifndef LCMAPS_ARGUMENTS_H
#define LCMAPS_ARGUMENTS_H

/******************************************************************************
                               Type definitions
******************************************************************************/

/*!
    \struct lcmaps_argument_s
    \brief structure representing an LCMAPS plugin run argument
*/
/*!
    \typedef lcmaps_argument_t
    \brief Type of LCMAPS plugin run argument (to be passed to the plugin by plugin_run())
*/
typedef struct lcmaps_argument_s
{
    char *argName; /*!< name of argument */
    char *argType; /*!< type of the argument */
    int argInOut;  /*!< input or output argument (0 = false = Input / 1 = true = Out) */
    void *value;   /*!< value of argument */
} lcmaps_argument_t;


/******************************************************************************
 *                 Module definition
 *****************************************************************************/ 

/******************************************************************************
Function:   lcmaps_setArgValue
Description:
    Set the value of argType on the reserved place in values. 
    The place within values is determined by the place where argName is found
    in the arguments list

Parameters:
    argName: name of argument
    argType: type of argument
    argcx:   number of arguments
    argvx:   array of arguments structures
Returns:
    0:  success
    -1: failure (could not find structure with name argName)
******************************************************************************/
extern int         lcmaps_setArgValue(char *argName, char *argType, void *value,  int argcx, lcmaps_argument_t **argvx);

/******************************************************************************
Function:   lcmaps_getArgValue
Description:
    Gets the value of argType from values.
    The place within the lcmaps_argument_t values is determined by the argName listed in 
    lcmaps_argument_t *argvx.
    Returns a void pointer to the value.

Parameters:
    argName: name of argument
    argType: type of argument
    argcx:   number of arguments
    argvx:   array of arguments structures
Returns:
    void pointer to the value or NULL
******************************************************************************/
extern void *      lcmaps_getArgValue(char *argName, char *argType, int argcx, lcmaps_argument_t *argvx);

/******************************************************************************
Function:   lcmaps_findArgName
Description:
    Search for argName in the arguments list.
    Returns the index to lcmaps_argument_t element.

Parameters:
    argName: name of argument
    argcx:   number of arguments
    argvx:   array of arguments structures
Returns:
    index to lcmaps_argument_t element
******************************************************************************/
extern int         lcmaps_findArgName(char *argName, int argcx, lcmaps_argument_t *argvx);

/******************************************************************************
Function:   lcmaps_findArgNameAndType
Description:
    Search for argName and Type in the arguments list.
    Returns the index to lcmaps_argument_t element.

Parameters:
    argName: name of argument
    argType: type of argument
    argcx:   number of arguments
    argvx:   array of arguments structures
Returns:
    index to lcmaps_argument_t element
******************************************************************************/
extern int         lcmaps_findArgNameAndType(char *argName, char *argType, int argcx, lcmaps_argument_t *argvx);

/******************************************************************************
Function:   lcmaps_cntArgs
Description:
    Count the number of arguments that are defined in a plug-in
    Returns this number.

Parameters:
    argvx:   array of arguments structures
Returns:
    the number of arguments
******************************************************************************/
extern int         lcmaps_cntArgs(lcmaps_argument_t *argvx);

#endif /* LCMAPS_ARGUMENTS_H */

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps/interface/lcmaps_arguments.h,v $
    $Date: 2011-03-04 21:55:13 +0100 (Fri, 04 Mar 2011) $
    $Revision: 14868 $
    $Author: dennisvd $
******************************************************************************/
