/*  Lziprecover - Data recovery tool for the lzip format
    Copyright (C) 2009-2015 Antonio Diaz Diaz.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
/*
    Exit status: 0 for a normal exit, 1 for environmental problems
    (file not found, invalid flags, I/O errors, etc), 2 to indicate a
    corrupt or invalid input file, 3 for an internal consistency error
    (eg, bug) which caused lziprecover to panic.
*/

#define _FILE_OFFSET_BITS 64

#include <algorithm>
#include <cerrno>
#include <climits>
#include <csignal>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <string>
#include <vector>
#include <fcntl.h>
#include <stdint.h>
#include <unistd.h>
#include <utime.h>
#include <sys/stat.h>
#if defined(__MSVCRT__)
#include <io.h>
#define fchmod(x,y) 0
#define fchown(x,y,z) 0
#define SIGHUP SIGTERM
#define S_ISSOCK(x) 0
#define S_IRGRP 0
#define S_IWGRP 0
#define S_IROTH 0
#define S_IWOTH 0
#endif
#if defined(__OS2__)
#include <io.h>
#endif

#include "arg_parser.h"
#include "lzip.h"
#include "decoder.h"
#include "block.h"

#ifndef O_BINARY
#define O_BINARY 0
#endif

#if CHAR_BIT != 8
#error "Environments where CHAR_BIT != 8 are not supported."
#endif


namespace {

const char * const Program_name = "Lziprecover";
const char * const program_name = "lziprecover";
const char * const program_year = "2015";
const char * invocation_name = 0;

struct { const char * from; const char * to; } const known_extensions[] = {
  { ".lz",  ""     },
  { ".tlz", ".tar" },
  { 0,      0      } };

enum Mode { m_none, m_debug_delay, m_debug_repair, m_decompress, m_list,
            m_merge, m_range_dec, m_repair, m_split, m_test };

std::string output_filename;
int outfd = -1;
int verbosity = 0;
const mode_t usr_rw = S_IRUSR | S_IWUSR;
const mode_t all_rw = usr_rw | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH;
mode_t outfd_mode = usr_rw;
bool delete_output_on_interrupt = false;


void show_help()
  {
  std::printf( "%s - Data recovery tool and decompressor for the lzip format.\n", Program_name );
  std::printf( "\nLziprecover can repair perfectly most files with small errors (up to one\n"
               "single-byte error per member), without the need of any extra redundance\n"
               "at all. Losing an entire archive just because of a corrupt byte near the\n"
               "beginning is a thing of the past.\n"
               "Lziprecover can also produce a correct file by merging the good parts of\n"
               "two or more damaged copies, extract data from damaged files, decompress\n"
               "files and test integrity of files.\n"
               "\nUsage: %s [options] [files]\n", invocation_name );
  std::printf( "\nOptions:\n"
               "  -h, --help                      display this help and exit\n"
               "  -V, --version                   output version information and exit\n"
               "  -c, --stdout                    send decompressed output to standard output\n"
               "  -d, --decompress                decompress\n"
               "  -D, --range-decompress=<range>  decompress only a range of bytes (N-M)\n"
               "  -f, --force                     overwrite existing output files\n"
               "  -i, --ignore-errors             make '--range-decompress' ignore data errors\n"
               "  -k, --keep                      keep (don't delete) input files\n"
               "  -l, --list                      print total file sizes and ratios\n"
               "  -m, --merge                     correct errors in file using several copies\n"
               "  -o, --output=<file>             place the output into <file>\n"
               "  -q, --quiet                     suppress all messages\n"
               "  -R, --repair                    try to repair a small error in file\n"
               "  -s, --split                     split multi-member file in single-member files\n"
               "  -t, --test                      test compressed file integrity\n"
               "  -v, --verbose                   be verbose (a 2nd -v gives more)\n" );
  if( verbosity >= 1 )
    {
    std::printf( "  -y, --debug-delay=<range>       find max error detection delay in <range>\n"
                 "  -z, --debug-repair=<pos>,<val>  test repair one-byte error at <pos>\n" );
    }
  std::printf( "Numbers may be followed by a multiplier: k = kB = 10^3 = 1000,\n"
               "Ki = KiB = 2^10 = 1024, M = 10^6, Mi = 2^20, G = 10^9, Gi = 2^30, etc...\n"
               "\nExit status: 0 for a normal exit, 1 for environmental problems (file\n"
               "not found, invalid flags, I/O errors, etc), 2 to indicate a corrupt or\n"
               "invalid input file, 3 for an internal consistency error (eg, bug) which\n"
               "caused lziprecover to panic.\n"
               "\nReport bugs to lzip-bug@nongnu.org\n"
               "Lziprecover home page: http://www.nongnu.org/lzip/lziprecover.html\n" );
  }


void show_version()
  {
  std::printf( "%s %s\n", program_name, PROGVERSION );
  std::printf( "Copyright (C) %s Antonio Diaz Diaz.\n", program_year );
  std::printf( "License GPLv2+: GNU GPL version 2 or later <http://gnu.org/licenses/gpl.html>\n"
               "This is free software: you are free to change and redistribute it.\n"
               "There is NO WARRANTY, to the extent permitted by law.\n" );
  }

} // end namespace

void show_header( const unsigned dictionary_size )
  {
  if( verbosity >= 3 )
    {
    const char * const prefix[8] =
      { "Ki", "Mi", "Gi", "Ti", "Pi", "Ei", "Zi", "Yi" };
    enum { factor = 1024 };
    const char * p = "";
    const char * np = "  ";
    unsigned num = dictionary_size;
    bool exact = ( num % factor == 0 );

    for( int i = 0; i < 8 && ( num > 9999 || ( exact && num >= factor ) ); ++i )
      { num /= factor; if( num % factor != 0 ) exact = false;
        p = prefix[i]; np = ""; }
    std::fprintf( stderr, "dictionary size %s%4u %sB.  ", np, num, p );
    }
  }

namespace {

// Returns the number of chars read, or 0 if error.
//
int parse_long_long( const char * const ptr, long long & value )
  {
  char * tail;
  errno = 0;
  value = strtoll( ptr, &tail, 0 );
  if( tail == ptr || errno || value < 0 ) return 0;
  int c = tail - ptr;

  if( ptr[c] )
    {
    const int factor = ( ptr[c+1] == 'i' ) ? 1024 : 1000;
    int exponent = 0;
    switch( ptr[c] )
      {
      case 'Y': exponent = 8; break;
      case 'Z': exponent = 7; break;
      case 'E': exponent = 6; break;
      case 'P': exponent = 5; break;
      case 'T': exponent = 4; break;
      case 'G': exponent = 3; break;
      case 'M': exponent = 2; break;
      case 'K': if( factor == 1024 ) exponent = 1; else return 0; break;
      case 'k': if( factor == 1000 ) exponent = 1; else return 0; break;
      }
    if( exponent > 0 )
      {
      ++c;
      if( ptr[c] == 'i' ) { ++c; if( value ) format_num( 0, 0, -1 ); }
      if( ptr[c] == 'B' ) ++c;
      for( int i = 0; i < exponent; ++i )
        {
        if( INT64_MAX / factor >= value ) value *= factor;
        else return 0;
        }
      }
    }
  return c;
  }


// Recognized formats: <begin> <begin>-<end> <begin>,<size>
//
void parse_range( const char * const ptr, Block & range )
  {
  long long value = 0;
  int c = parse_long_long( ptr, value );		// pos
  if( c && value >= 0 && value < INT64_MAX &&
      ( ptr[c] == 0 || ptr[c] == ',' || ptr[c] == '-' ) )
    {
    range.pos( value );
    if( ptr[c] == 0 ) { range.size( INT64_MAX - value ); return; }
    const bool issize = ( ptr[c] == ',' );
    c = parse_long_long( ptr + c + 1, value );		// size
    if( c && value > 0 && ( issize || value > range.pos() ) )
      {
      if( !issize ) value -= range.pos();
      if( INT64_MAX - range.pos() >= value ) { range.size( value ); return; }
      }
    }
  show_error( "Bad decompression range.", 0, true );
  std::exit( 1 );
  }


// Recognized format: <pos>,<value>
//
void parse_pos_value( const char * const ptr, long long & pos, uint8_t & value )
  {
  long long val = 0;
  int c = parse_long_long( ptr, val );			// pos
  if( c && val >= 0 && val < INT64_MAX && ptr[c] == ',' )
    {
    pos = val;
    c = parse_long_long( ptr + c + 1, val );		// value
    if( c && val >= 0 && val < 256 )
      { value = val; return; }
    }
  show_error( "Bad file position or byte value.", 0, true );
  std::exit( 1 );
  }


void one_file( const int files )
  {
  if( files != 1 )
    {
    show_error( "You must specify exactly 1 file.", 0, true );
    std::exit( 1 );
    }
  }


void set_mode( Mode & program_mode, const Mode new_mode )
  {
  if( program_mode != m_none && program_mode != new_mode )
    {
    show_error( "Only one operation can be specified.", 0, true );
    std::exit( 1 );
    }
  program_mode = new_mode;
  }


int extension_index( const std::string & name )
  {
  for( int i = 0; known_extensions[i].from; ++i )
    {
    const std::string ext( known_extensions[i].from );
    if( name.size() > ext.size() &&
        name.compare( name.size() - ext.size(), ext.size(), ext ) == 0 )
      return i;
    }
  return -1;
  }

} // end namespace

int open_instream( const char * const name, struct stat * const in_statsp,
                   const bool no_ofile, const bool reg_only )
  {
  int infd = open( name, O_RDONLY | O_BINARY );
  if( infd < 0 )
    {
    if( verbosity >= 0 )
      std::fprintf( stderr, "%s: Can't open input file '%s': %s.\n",
                    program_name, name, std::strerror( errno ) );
    }
  else
    {
    const int i = fstat( infd, in_statsp );
    const mode_t mode = in_statsp->st_mode;
    const bool can_read = ( i == 0 && !reg_only &&
                            ( S_ISBLK( mode ) || S_ISCHR( mode ) ||
                              S_ISFIFO( mode ) || S_ISSOCK( mode ) ) );
    if( i != 0 || ( !S_ISREG( mode ) && ( !can_read || !no_ofile ) ) )
      {
      if( verbosity >= 0 )
        std::fprintf( stderr, "%s: Input file '%s' is not a regular file%s.\n",
                      program_name, name,
                      ( can_read && !no_ofile ) ?
                      " and '--stdout' was not specified" : "" );
      close( infd );
      infd = -1;
      }
    }
  return infd;
  }

namespace {

void set_d_outname( const std::string & name, const int i )
  {
  if( i >= 0 )
    {
    const std::string from( known_extensions[i].from );
    if( name.size() > from.size() )
      {
      output_filename.assign( name, 0, name.size() - from.size() );
      output_filename += known_extensions[i].to;
      return;
      }
    }
  output_filename = name; output_filename += ".out";
  if( verbosity >= 1 )
    std::fprintf( stderr, "%s: Can't guess original name for '%s' -- using '%s'.\n",
                  program_name, name.c_str(), output_filename.c_str() );
  }


bool open_outstream( const bool force )
  {
  int flags = O_CREAT | O_WRONLY | O_BINARY;
  if( force ) flags |= O_TRUNC; else flags |= O_EXCL;

  outfd = open( output_filename.c_str(), flags, outfd_mode );
  if( outfd < 0 && verbosity >= 0 )
    {
    if( errno == EEXIST )
      std::fprintf( stderr, "%s: Output file '%s' already exists, skipping.\n",
                    program_name, output_filename.c_str() );
    else
      std::fprintf( stderr, "%s: Can't create output file '%s': %s.\n",
                    program_name, output_filename.c_str(), std::strerror( errno ) );
    }
  return ( outfd >= 0 );
  }


void cleanup_and_fail( const int retval )
  {
  if( delete_output_on_interrupt )
    {
    delete_output_on_interrupt = false;
    if( verbosity >= 0 )
      std::fprintf( stderr, "%s: Deleting output file '%s', if it exists.\n",
                    program_name, output_filename.c_str() );
    if( outfd >= 0 ) { close( outfd ); outfd = -1; }
    if( std::remove( output_filename.c_str() ) != 0 && errno != ENOENT )
      show_error( "WARNING: deletion of output file (apparently) failed." );
    }
  std::exit( retval );
  }


     /* Set permissions, owner and times. */
void close_and_set_permissions( const struct stat * const in_statsp )
  {
  bool warning = false;
  if( in_statsp )
    {
    const mode_t mode = in_statsp->st_mode;
    /* fchown will in many cases return with EPERM, which can be safely ignored. */
    if( fchown( outfd, in_statsp->st_uid, in_statsp->st_gid ) == 0 )
      { if( fchmod( outfd, mode ) != 0 ) warning = true; }
    else
      if( errno != EPERM ||
          fchmod( outfd, mode & ~( S_ISUID | S_ISGID | S_ISVTX ) ) != 0 )
        warning = true;
    }
  if( close( outfd ) != 0 ) cleanup_and_fail( 1 );
  outfd = -1;
  delete_output_on_interrupt = false;
  if( in_statsp )
    {
    struct utimbuf t;
    t.actime = in_statsp->st_atime;
    t.modtime = in_statsp->st_mtime;
    if( utime( output_filename.c_str(), &t ) != 0 ) warning = true;
    }
  if( warning && verbosity >= 1 )
    show_error( "Can't change output file attributes." );
  }


std::string insert_fixed( std::string name )
  {
  if( name.size() > 7 && name.compare( name.size() - 7, 7, ".tar.lz" ) == 0 )
    name.insert( name.size() - 7, "_fixed" );
  else if( name.size() > 3 && name.compare( name.size() - 3, 3, ".lz" ) == 0 )
    name.insert( name.size() - 3, "_fixed" );
  else if( name.size() > 4 && name.compare( name.size() - 4, 4, ".tlz" ) == 0 )
    name.insert( name.size() - 4, "_fixed" );
  else name += "_fixed.lz";
  return name;
  }


unsigned char xdigit( const int value )
  {
  if( value >= 0 && value <= 9 ) return '0' + value;
  if( value >= 10 && value <= 15 ) return 'A' + value - 10;
  return 0;
  }


void show_trailing_garbage( const uint8_t * const data, const int size,
                            const Pretty_print & pp, const bool all )
  {
  std::string garbage_msg;
  if( !all ) garbage_msg = "first bytes of ";
  garbage_msg += "trailing garbage found = ";
  bool text = true;
  for( int i = 0; i < size; ++i )
    if( !std::isprint( data[i] ) ) { text = false; break; }
  if( text )
    {
    garbage_msg += '\'';
    garbage_msg.append( (const char *)data, size );
    garbage_msg += '\'';
    }
  else
    {
    for( int i = 0; i < size; ++i )
      {
      if( i > 0 ) garbage_msg += ' ';
      garbage_msg += xdigit( data[i] >> 4 );
      garbage_msg += xdigit( data[i] & 0x0F );
      }
    }
  garbage_msg += '.';
  pp( garbage_msg.c_str() );
  }


int decompress( const int infd, const Pretty_print & pp, const bool testing )
  {
  int retval = 0;

  try {
    unsigned long long partial_file_pos = 0;
    Range_decoder rdec( infd );
    for( bool first_member = true; ; first_member = false )
      {
      File_header header;
      rdec.reset_member_position();
      const int size = rdec.read_data( header.data, File_header::size );
      if( rdec.finished() )			// End Of File
        {
        if( first_member )
          { pp( "File ends unexpectedly at member header." ); retval = 2; }
        else if( verbosity >= 4 && size > 0 )
          show_trailing_garbage( header.data, size, pp, true );
        break;
        }
      if( !header.verify_magic() )
        {
        if( first_member )
          { pp( "Bad magic number (file not in lzip format)." ); retval = 2; }
        else if( verbosity >= 4 )
          show_trailing_garbage( header.data, size, pp, false );
        break;
        }
      if( !header.verify_version() )
        {
        if( verbosity >= 0 )
          { pp();
            std::fprintf( stderr, "Version %d member format not supported.\n",
                          header.version() ); }
        retval = 2; break;
        }
      const unsigned dictionary_size = header.dictionary_size();
      if( dictionary_size < min_dictionary_size ||
          dictionary_size > max_dictionary_size )
        { pp( "Invalid dictionary size in member header." ); retval = 2; break; }

      if( verbosity >= 2 || ( verbosity == 1 && first_member ) )
        { pp(); show_header( dictionary_size ); }

      LZ_decoder decoder( header, rdec, outfd );
      const int result = decoder.decode_member( pp );
      partial_file_pos += rdec.member_position();
      if( result != 0 )
        {
        if( verbosity >= 0 && result <= 2 )
          {
          pp();
          if( result == 2 )
            std::fprintf( stderr, "File ends unexpectedly at pos %llu.\n",
                          partial_file_pos );
          else
            std::fprintf( stderr, "Decoder error at pos %llu.\n",
                          partial_file_pos );
          }
        retval = 2; break;
        }
      if( verbosity >= 2 )
        { std::fprintf( stderr, testing ? "ok\n" : "done\n" ); pp.reset(); }
      }
    }
  catch( std::bad_alloc ) { pp( "Not enough memory." ); retval = 1; }
  catch( Error e ) { pp(); show_error( e.msg, errno ); retval = 1; }
  if( verbosity == 1 && retval == 0 )
    std::fprintf( stderr, testing ? "ok\n" : "done\n" );
  return retval;
  }


extern "C" void signal_handler( int )
  {
  show_error( "Control-C or similar caught, quitting." );
  cleanup_and_fail( 1 );
  }


void set_signals()
  {
  std::signal( SIGHUP, signal_handler );
  std::signal( SIGINT, signal_handler );
  std::signal( SIGTERM, signal_handler );
  }

} // end namespace


int open_outstream_rw( const std::string & output_filename, const bool force )
  {
  int flags = O_CREAT | O_RDWR | O_BINARY;
  if( force ) flags |= O_TRUNC; else flags |= O_EXCL;

  int outfd = open( output_filename.c_str(), flags, all_rw );
  if( outfd < 0 && verbosity >= 0 )
    {
    if( errno == EEXIST )
      std::fprintf( stderr, "%s: Output file '%s' already exists."
                            " Use '--force' to overwrite it.\n",
                    program_name, output_filename.c_str() );
    else
      std::fprintf( stderr, "%s: Can't create output file '%s': %s.\n",
                    program_name, output_filename.c_str(), std::strerror( errno ) );
    }
  return outfd;
  }


void show_error( const char * const msg, const int errcode, const bool help )
  {
  if( verbosity >= 0 )
    {
    if( msg && msg[0] )
      {
      std::fprintf( stderr, "%s: %s", program_name, msg );
      if( errcode > 0 )
        std::fprintf( stderr, ": %s.", std::strerror( errcode ) );
      std::fprintf( stderr, "\n" );
      }
    if( help )
      std::fprintf( stderr, "Try '%s --help' for more information.\n",
                    invocation_name );
    }
  }


void internal_error( const char * const msg )
  {
  if( verbosity >= 0 )
    std::fprintf( stderr, "%s: internal error: %s\n", program_name, msg );
  std::exit( 3 );
  }


void show_error2( const char * const msg1, const char * const name,
                  const char * const msg2 )
  {
  if( verbosity >= 0 )
    std::fprintf( stderr, "%s: %s '%s' %s\n", program_name, msg1, name, msg2 );
  }


int main( const int argc, const char * const argv[] )
  {
  Block range( 0, 0 );
  long long bad_pos = 0;
  std::string input_filename;
  std::string default_output_filename;
  std::vector< std::string > filenames;
  int infd = -1;
  Mode program_mode = m_none;
  uint8_t bad_value = 0;
  bool force = false;
  bool ignore = false;
  bool keep_input_files = false;
  bool to_stdout = false;
  invocation_name = argv[0];

  const Arg_parser::Option options[] =
    {
    { 'c', "stdout",           Arg_parser::no  },
    { 'd', "decompress",       Arg_parser::no  },
    { 'D', "range-decompress", Arg_parser::yes },
    { 'f', "force",            Arg_parser::no  },
    { 'h', "help",             Arg_parser::no  },
    { 'i', "ignore-errors",    Arg_parser::no  },
    { 'k', "keep",             Arg_parser::no  },
    { 'l', "list",             Arg_parser::no  },
    { 'm', "merge",            Arg_parser::no  },
    { 'n', "threads",          Arg_parser::yes },
    { 'o', "output",           Arg_parser::yes },
    { 'q', "quiet",            Arg_parser::no  },
    { 'R', "repair",           Arg_parser::no  },
    { 's', "split",            Arg_parser::no  },
    { 't', "test",             Arg_parser::no  },
    { 'v', "verbose",          Arg_parser::no  },
    { 'V', "version",          Arg_parser::no  },
    { 'y', "debug-delay",      Arg_parser::yes },
    { 'z', "debug-repair",     Arg_parser::yes },
    {  0 ,  0,                 Arg_parser::no  } };

  const Arg_parser parser( argc, argv, options );
  if( parser.error().size() )				// bad option
    { show_error( parser.error().c_str(), 0, true ); return 1; }

  int argind = 0;
  for( ; argind < parser.arguments(); ++argind )
    {
    const int code = parser.code( argind );
    if( !code ) break;					/* no more options */
    const std::string & arg = parser.argument( argind );
    switch( code )
      {
      case 'c': to_stdout = true; break;
      case 'd': set_mode( program_mode, m_decompress ); break;
      case 'D': set_mode( program_mode, m_range_dec );
                parse_range( arg.c_str(), range ); break;
      case 'f': force = true; break;
      case 'h': show_help(); return 0;
      case 'i': ignore = true; break;
      case 'k': keep_input_files = true; break;
      case 'l': set_mode( program_mode, m_list ); break;
      case 'm': set_mode( program_mode, m_merge ); break;
      case 'n': break;
      case 'o': default_output_filename = arg; break;
      case 'q': verbosity = -1; break;
      case 'R': set_mode( program_mode, m_repair ); break;
      case 's': set_mode( program_mode, m_split ); break;
      case 't': set_mode( program_mode, m_test ); break;
      case 'v': if( verbosity < 4 ) ++verbosity; break;
      case 'V': show_version(); return 0;
      case 'y': set_mode( program_mode, m_debug_delay );
                parse_range( arg.c_str(), range ); break;
      case 'z': set_mode( program_mode, m_debug_repair );
                parse_pos_value( arg.c_str(), bad_pos, bad_value ); break;
      default : internal_error( "uncaught option." );
      }
    } /* end process options */

#if defined(__MSVCRT__) || defined(__OS2__)
  setmode( STDIN_FILENO, O_BINARY );
  setmode( STDOUT_FILENO, O_BINARY );
#endif

  if( program_mode == m_none )
    {
    show_error( "You must specify the operation to be performed.", 0, true );
    return 1;
    }

  bool filenames_given = false;
  for( ; argind < parser.arguments(); ++argind )
    {
    filenames.push_back( parser.argument( argind ) );
    if( filenames.back() != "-" ) filenames_given = true;
    }

  try {
  switch( program_mode )
    {
    case m_none: internal_error( "invalid operation." ); break;
    case m_debug_delay:
      one_file( filenames.size() );
      return debug_delay( filenames[0], range, verbosity );
    case m_debug_repair:
      one_file( filenames.size() );
      return debug_repair( filenames[0], bad_pos, verbosity, bad_value );
    case m_decompress: break;
    case m_list:
      if( filenames.size() < 1 )
        { show_error( "You must specify at least 1 file.", 0, true ); return 1; }
      return list_files( filenames, verbosity );
    case m_merge:
      if( filenames.size() < 2 )
        { show_error( "You must specify at least 2 files.", 0, true ); return 1; }
      if( default_output_filename.empty() )
        default_output_filename = insert_fixed( filenames[0] );
      return merge_files( filenames, default_output_filename, verbosity, force );
    case m_range_dec:
      one_file( filenames.size() );
      return range_decompress( filenames[0], default_output_filename,
                               range, verbosity, force, ignore, to_stdout );
    case m_repair:
      one_file( filenames.size() );
      if( default_output_filename.empty() )
        default_output_filename = insert_fixed( filenames[0] );
      return repair_file( filenames[0], default_output_filename,
                          verbosity, force );
    case m_split:
      one_file( filenames.size() );
      return split_file( filenames[0], default_output_filename, verbosity, force );
    case m_test: break;
    }
    }
  catch( std::bad_alloc ) { show_error( "Not enough memory." ); return 1; }
  catch( Error e ) { show_error( e.msg, errno ); return 1; }

  if( program_mode == m_test )
    outfd = -1;
  else if( program_mode != m_decompress )
    internal_error( "invalid decompressor operation." );

  if( filenames.empty() ) filenames.push_back("-");
  if( !to_stdout && program_mode != m_test &&
      ( filenames_given || default_output_filename.size() ) )
    set_signals();

  Pretty_print pp( filenames, verbosity );

  int retval = 0;
  for( unsigned i = 0; i < filenames.size(); ++i )
    {
    struct stat in_stats;
    output_filename.clear();

    if( filenames[i].empty() || filenames[i] == "-" )
      {
      input_filename.clear();
      infd = STDIN_FILENO;
      if( program_mode != m_test )
        {
        if( to_stdout || default_output_filename.empty() )
          outfd = STDOUT_FILENO;
        else
          {
          output_filename = default_output_filename;
          outfd_mode = all_rw;
          if( !open_outstream( force ) )
            {
            if( retval < 1 ) retval = 1;
            close( infd ); infd = -1;
            continue;
            }
          }
        }
      }
    else
      {
      input_filename = filenames[i];
      infd = open_instream( input_filename.c_str(), &in_stats,
                            to_stdout || program_mode == m_test );
      if( infd < 0 ) { if( retval < 1 ) retval = 1; continue; }
      if( program_mode != m_test )
        {
        if( to_stdout ) outfd = STDOUT_FILENO;
        else
          {
          set_d_outname( input_filename, extension_index( input_filename ) );
          outfd_mode = usr_rw;
          if( !open_outstream( force ) )
            {
            if( retval < 1 ) retval = 1;
            close( infd ); infd = -1;
            continue;
            }
          }
        }
      }

    if( isatty( infd ) )
      {
      show_error( "I won't read compressed data from a terminal.", 0, true );
      return 1;
      }

    if( output_filename.size() && !to_stdout && program_mode != m_test )
      delete_output_on_interrupt = true;
    const struct stat * const in_statsp = input_filename.size() ? &in_stats : 0;
    pp.set_name( input_filename );
    const int tmp = decompress( infd, pp, program_mode == m_test );
    if( tmp > retval ) retval = tmp;
    if( tmp && program_mode != m_test ) cleanup_and_fail( retval );

    if( delete_output_on_interrupt )
      close_and_set_permissions( in_statsp );
    if( input_filename.size() )
      {
      close( infd ); infd = -1;
      if( !keep_input_files && !to_stdout && program_mode != m_test )
        std::remove( input_filename.c_str() );
      }
    }
  if( outfd >= 0 && close( outfd ) != 0 )
    {
    show_error( "Can't close stdout", errno );
    if( retval < 1 ) retval = 1;
    }
  return retval;
  }
