/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2010 Oracle and/or its affiliates. All rights reserved.
 *
 * Oracle and Java are registered trademarks of Oracle and/or its affiliates.
 * Other names may be trademarks of their respective owners.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 *
 * Contributor(s):
 *
 * Portions Copyrighted 2009 Sun Microsystems, Inc.
 */

package org.netbeans.modules.cnd.toolchain.compilerset;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;
import org.netbeans.modules.cnd.api.toolchain.AbstractCompiler;
import org.netbeans.modules.cnd.api.toolchain.Tool;
import org.netbeans.modules.cnd.api.toolchain.CompilerFlavor;
import org.netbeans.modules.cnd.spi.toolchain.CompilerProvider;
import org.netbeans.modules.cnd.api.toolchain.CompilerSet;
import org.netbeans.modules.cnd.api.toolchain.PlatformTypes;
import org.netbeans.modules.cnd.api.toolchain.PredefinedToolKind;
import org.netbeans.modules.nativeexecution.api.ExecutionEnvironment;
import org.netbeans.modules.nativeexecution.api.ExecutionEnvironmentFactory;
import org.openide.util.Exceptions;
import org.openide.util.NbPreferences;

/**
 *
 * @author Alexander Simon
 */
public final class CompilerSetPreferences {
    /* Persistance information */
    private static final String csm_version = "1.2"; // NOI18N
    private static final String CSM = "csm."; // NOI18N
    private static final String VERSION = "version"; // NOI18N
    private static final String NO_SETS = ".noOfSets"; // NOI18N
    private static final String SET_NAME = ".setName."; // NOI18N
    private static final String SET_FLAVOR = ".setFlavor."; // NOI18N
    private static final String SET_DIRECTORY = ".setDirectory."; // NOI18N
    private static final String SET_ENCODING = ".setEncoding."; // NOI18N
    private static final String SET_AUTO = ".autoGenerated."; // NOI18N
    private static final String SET_DEFAULT = ".defaultSet"; // NOI18N
    private static final String SET_PLATFORM = ".setPlatform."; // NOI18N
    private static final String NO_TOOLS = ".noOfTools."; // NOI18N
    private static final String TOOL_NAME = ".toolName."; // NOI18N
    private static final String TOOL_DISPLAYNAME = ".toolDisplayName."; // NOI18N
    private static final String TOOL_KIND = ".toolKind."; // NOI18N
    private static final String TOOL_PATH = ".toolPath."; // NOI18N
    private static final String TOOL_FLAVOR = ".toolFlavor."; // NOI18N
    private static final String TOOL_SETTINGS = ".toolSettings."; // NOI18N
    private static final Logger log = Logger.getLogger("cnd.remote.logger"); // NOI18N
    public static final String VERSION_KEY = CSM + VERSION;

    private static CompilerProvider compilerProvider = null;

    private CompilerSetPreferences() {
    }

    /*
     * Persistence ...
     */
    public static Preferences getPreferences() {
        return NbPreferences.forModule(CompilerSetPreferences.class);
    }

    public static void putEnv(ExecutionEnvironment executionEnvironment, int platform){
        getPreferences().putInt(CSM + ExecutionEnvironmentFactory.toUniqueID(executionEnvironment) + SET_PLATFORM, platform);
    }

    public static void storeExecutionEnvironmentList(List<ExecutionEnvironment> liveServers){
        if (liveServers == null){
            return;
        }
        try {
            List<String> keys = new ArrayList<String>();
            for(ExecutionEnvironment env : liveServers){
                String key = CSM + ExecutionEnvironmentFactory.toUniqueID(env) + "."; // NOI18N
                keys.add(key);
                log.log(Level.FINE, "CSM.storeExecutionEnvironmentList: host = {0}", env); // NOI18N
            }
            getPreferences().remove(CSM + VERSION);
            loop:for(String key : getPreferences().keys()){
                if (key.startsWith(CSM)) {
                    for(String start : keys) {
                        if (key.startsWith(start)) {
                            continue loop;
                        }
                    }
                    log.log(Level.FINE, "CSM.storeExecutionEnvironmentList: remove {0}={1}", new Object[]{key, getPreferences().get(key, "")}); // NOI18N
                    getPreferences().remove(key);
                }
            }
            getPreferences().put(CSM + VERSION, csm_version);
        } catch (BackingStoreException ex) {
            Exceptions.printStackTrace(ex);
        }
    }

    private static void clearPersistence(ExecutionEnvironment env) {
        try {
            String aKey = CSM + ExecutionEnvironmentFactory.toUniqueID(env) + "."; // NOI18N
            for(String key : getPreferences().keys()){
                if (key.startsWith(aKey)) {
                    getPreferences().remove(key);
                }
            }
        } catch (BackingStoreException ex) {
            Exceptions.printStackTrace(ex);
        }
    }

    public static void saveToDisk(CompilerSetManagerImpl manager) {
        log.log(Level.FINE, "CSM.Save to disk {0} size={1}", new Object[]{manager.getExecutionEnvironment(), manager.getCompilerSets().size()}); // NOI18N
        if (!manager.getCompilerSets().isEmpty()) {
            clearPersistence(manager.getExecutionEnvironment());
            getPreferences().put(CSM + VERSION, csm_version);
            String executionEnvironmentKey = ExecutionEnvironmentFactory.toUniqueID(manager.getExecutionEnvironment());
            getPreferences().putInt(CSM + executionEnvironmentKey + NO_SETS, manager.getCompilerSets().size());
            getPreferences().putInt(CSM + executionEnvironmentKey + SET_PLATFORM, manager.getPlatform());
            int setCount = 0;
            for (CompilerSet cs : manager.getCompilerSets()) {
                getPreferences().put(CSM + executionEnvironmentKey + SET_NAME + setCount, cs.getName());
                getPreferences().put(CSM + executionEnvironmentKey + SET_FLAVOR + setCount, cs.getCompilerFlavor().toString());
                getPreferences().put(CSM + executionEnvironmentKey + SET_DIRECTORY + setCount, cs.getDirectory());
                getPreferences().put(CSM + executionEnvironmentKey + SET_ENCODING + setCount, cs.getEncoding().name());
                getPreferences().putBoolean(CSM + executionEnvironmentKey + SET_AUTO + setCount, cs.isAutoGenerated());
                getPreferences().putBoolean(CSM + executionEnvironmentKey + SET_DEFAULT + setCount, manager.isDefaultCompilerSet(cs));
                List<Tool> tools = cs.getTools();
                getPreferences().putInt(CSM + executionEnvironmentKey + NO_TOOLS + setCount, tools.size());
                int toolCount = 0;
                for (Tool tool : tools) {
                    getPreferences().put(CSM + executionEnvironmentKey + TOOL_NAME + setCount + '.' + toolCount, tool.getName());
                    getPreferences().put(CSM + executionEnvironmentKey + TOOL_DISPLAYNAME + '-' + setCount + '.' + toolCount, tool.getDisplayName());
                    getPreferences().putInt(CSM + executionEnvironmentKey + TOOL_KIND + setCount + '.' + toolCount, tool.getKind().ordinal());
                    getPreferences().put(CSM + executionEnvironmentKey + TOOL_PATH + setCount + '.' + toolCount, tool.getPath());
                    getPreferences().put(CSM + executionEnvironmentKey + TOOL_FLAVOR + setCount + '.' + toolCount, tool.getFlavor().toString());
                    if (tool instanceof AbstractCompiler) {
                        ((AbstractCompiler) tool).saveSettings(getPreferences(), CSM + executionEnvironmentKey + TOOL_SETTINGS + setCount + '.' + toolCount);
                    }
                    toolCount++;
                }
                setCount++;
            }
        }
    }

    public static CompilerSetManagerImpl restoreFromDisk(ExecutionEnvironment env) {
        String version = getPreferences().get(CSM + VERSION, "1.0"); // NOI18N
        if ("1.0".equals(version) && env.isLocal()) { // NOI18N
            return restoreFromDisk10();
        }
        String executionEnvironmentKey = ExecutionEnvironmentFactory.toUniqueID(env);
        int noSets = getPreferences().getInt(CSM + executionEnvironmentKey + NO_SETS, -1);
        log.log(Level.FINE, "CSM.Restore from disk {0} size={1}", new Object[]{env, noSets}); // NOI18N
        if (noSets < 0) {
            return null;
        }
        int pform = getPreferences().getInt(CSM + executionEnvironmentKey + SET_PLATFORM, -1);
        if (pform < 0) {
            if (env.isLocal()) {
                pform = ToolUtils.computeLocalPlatform();
            }
        } else {
            if (env.isLocal() && pform != ToolUtils.computeLocalPlatform()) {
                // it seems user dir is shared among different platforms
                // return null and reinit tool collection
                return null;
            }
        }

        ArrayList<CompilerSet> css = new ArrayList<CompilerSet>();
        for (int setCount = 0; setCount < noSets; setCount++) {
            String setName = getPreferences().get(CSM + executionEnvironmentKey + SET_NAME + setCount, null);
            String setFlavorName = getPreferences().get(CSM + executionEnvironmentKey + SET_FLAVOR + setCount, null);
            CompilerFlavor flavor = null;
            if (setFlavorName != null) {
                flavor = CompilerFlavorImpl.toFlavor(setFlavorName, pform);
            }
            String setDirectory = getPreferences().get(CSM + executionEnvironmentKey + SET_DIRECTORY + setCount, null);
            if (setName == null || setFlavorName == null || flavor == null) {
                // FIXUP: error
                continue;
            }
            String setEncoding = getPreferences().get(CSM + executionEnvironmentKey + SET_ENCODING + setCount, null);
            Boolean auto = getPreferences().getBoolean(CSM + executionEnvironmentKey + SET_AUTO + setCount, false);
            Boolean isDefault = getPreferences().getBoolean(CSM + executionEnvironmentKey + SET_DEFAULT + setCount, false);
            CompilerSetImpl cs = new CompilerSetImpl(flavor, setDirectory, setName);
            cs.setAutoGenerated(auto);
            cs.setAsDefault(isDefault);
            if (setEncoding != null) {
                Charset charset = Charset.forName(setEncoding);
                if (charset != null) {
                    cs.setEncoding(charset);
                }
            }
            int noTools = getPreferences().getInt(CSM + executionEnvironmentKey + NO_TOOLS + setCount, -1);
            for (int toolCount = 0; toolCount < noTools; toolCount++) {
                String toolName = getPreferences().get(CSM + executionEnvironmentKey + TOOL_NAME + setCount + '.' + toolCount, null);
                String toolDisplayName = getPreferences().get(CSM + executionEnvironmentKey + TOOL_DISPLAYNAME + '-' + setCount + '.' + toolCount, null);
                int toolKind = getPreferences().getInt(CSM + executionEnvironmentKey + TOOL_KIND + setCount + '.' + toolCount, -1);
                String toolPath = getPreferences().get(CSM + executionEnvironmentKey + TOOL_PATH + setCount + '.' + toolCount, null);
                String toolFlavorName = getPreferences().get(CSM + executionEnvironmentKey + TOOL_FLAVOR + setCount + '.' + toolCount, null);
                CompilerFlavor toolFlavor = null;
                if (toolFlavorName != null) {
                    toolFlavor = CompilerFlavorImpl.toFlavor(toolFlavorName, pform);
                }
                Tool tool = null;
                if (toolFlavor != null) {
                    tool = CompilerSetPreferences.getCompilerProvider().createCompiler(env, toolFlavor, PredefinedToolKind.getTool(toolKind), toolName, toolDisplayName, toolPath);
                }
                if (tool instanceof AbstractCompiler) {
                    ((AbstractCompiler) tool).loadSettings(getPreferences(), CSM + executionEnvironmentKey + TOOL_SETTINGS + setCount + '.' + toolCount);
                }
                if (tool != null) {
                    cs.addTool(tool);
                }
            }
            css.add(cs);
        }

        CompilerSetManagerImpl csm = new CompilerSetManagerImpl(env, css, pform);
        return csm;
    }

    private static CompilerSetManagerImpl restoreFromDisk10() {
        int noSets = getPreferences().getInt(CSM + NO_SETS, -1);
        if (noSets < 0) {
            return null;
        }

        ArrayList<CompilerSet> css = new ArrayList<CompilerSet>();
        getPreferences().remove(CSM + NO_SETS);
        for (int setCount = 0; setCount < noSets; setCount++) {
            String setName = getPreferences().get(CSM + SET_NAME + setCount, null);
            getPreferences().remove(CSM + SET_NAME + setCount);
            String setFlavorName = getPreferences().get(CSM + SET_FLAVOR + setCount, null);
            getPreferences().remove(CSM + SET_FLAVOR + setCount);
            CompilerFlavor flavor = null;
            if (setFlavorName != null) {
                flavor = CompilerFlavorImpl.toFlavor(setFlavorName, PlatformTypes.getDefaultPlatform());
            }
            String setDirectory = getPreferences().get(CSM + SET_DIRECTORY + setCount, null);
            getPreferences().remove(CSM + SET_DIRECTORY + setCount);
            if (setName == null || setFlavorName == null || flavor == null) {
                // FIXUP: error
                continue;
            }
            Boolean auto = getPreferences().getBoolean(CSM + SET_AUTO + setCount, false);
            getPreferences().remove(CSM + SET_AUTO + setCount);
            CompilerSetImpl cs = new CompilerSetImpl(flavor, setDirectory, setName);
            cs.setAutoGenerated(auto);
            int noTools = getPreferences().getInt(CSM + NO_TOOLS + setCount, -1);
            getPreferences().remove(CSM + NO_TOOLS + setCount);
            for (int toolCount = 0; toolCount < noTools; toolCount++) {
                String toolName = getPreferences().get(CSM + TOOL_NAME + setCount + '.' + toolCount, null);
                String toolDisplayName = getPreferences().get(CSM + TOOL_DISPLAYNAME + '-' + setCount + '.' + toolCount, null);
                int toolKind = getPreferences().getInt(CSM + TOOL_KIND + setCount + '.' + toolCount, -1);
                String toolPath = getPreferences().get(CSM + TOOL_PATH + setCount + '.' + toolCount, null);
                String toolFlavorName = getPreferences().get(CSM + TOOL_FLAVOR + setCount + '.' + toolCount, null);
                getPreferences().remove(CSM + TOOL_NAME + setCount + '.' + toolCount);
                getPreferences().remove(CSM + TOOL_DISPLAYNAME + '-' + setCount + '.' + toolCount);
                getPreferences().remove(CSM + TOOL_KIND + setCount + '.' + toolCount);
                getPreferences().remove(CSM + TOOL_PATH + setCount + '.' + toolCount);
                getPreferences().remove(CSM + TOOL_FLAVOR + setCount + '.' + toolCount);
                CompilerFlavor toolFlavor = null;
                if (toolFlavorName != null) {
                    toolFlavor = CompilerFlavorImpl.toFlavor(toolFlavorName, PlatformTypes.getDefaultPlatform());
                }
                Tool tool = null;
                if (toolFlavor != null) {
                    tool = CompilerSetPreferences.getCompilerProvider().createCompiler(ExecutionEnvironmentFactory.getLocal(),
                        toolFlavor, PredefinedToolKind.getTool(toolKind), toolName, toolDisplayName, toolPath); //NOI18N
                }
                if (tool != null) {
                    cs.addTool(tool);
                }
            }
            CompilerSetManagerImpl.completeCompilerSet(ExecutionEnvironmentFactory.getLocal(), cs, css);
            css.add(cs);
        }
        CompilerSetManagerImpl csm = new CompilerSetManagerImpl(ExecutionEnvironmentFactory.getLocal(),  css, ToolUtils.computeLocalPlatform());
        return csm;
    }

    public static CompilerProvider getCompilerProvider() {
        if (compilerProvider == null) {
            compilerProvider = CompilerProvider.getInstance();
        }
        return compilerProvider;
    }
}
