#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2 of the License, or (at
# your option) any later version.  A copy of this license is included
# with this module (LICENSE).
# 
# Copyright (c) 2003, Andrew Molloy.  All Rights Reserved.
# Andrew Molloy <amolloy@kaizolabs.com>
# 
# NOTE: Module has been heavily optimized for Sorune's use!
#       Darren Smith (09/04/04)
#

package Ogg::Vorbis::Header::PurePerl;
use 5.005;
use strict;

use Fcntl qw/SEEK_SET SEEK_CUR SEEK_END/;

our $VERSION = '0.05';

sub new 
{
    my $class = shift;
    my $file = shift;

    return load($class, $file);
}

sub load 
{
    my $class    = shift;
    my $file     = shift;
    my $from_new = shift;
    my %data;
    my $self;

    # there must be a better way...
    if ($class eq 'Ogg::Vorbis::Header::PurePerl') {
        $self = bless \%data, $class;
    }
    else {
        $self = $class;
    }

    if ($self->{'FILE_LOADED'}) {
        return $self;
    }

    $self->{'FILE_LOADED'} = 1;

    # check that the file is readable
    if (!-r $file) {
        return undef;
    }

    # open up the file
    if (open(FILE,$file)) {
        binmode FILE;

        $data{'filename'} = $file;
        $data{'fileHandle'} = \*FILE;

        _init(\%data);
        _loadInfo(\%data);
        _loadComments(\%data);
        _calculateTrackLength(\%data);

        close FILE;
    }

    return $self;
}

sub info 
{
    my $self = shift;
    my $key = shift;

    # if the user did not supply a key, return the entire hash
    unless ($key) {
        return $self->{'INFO'};
    }

    # otherwise, return the value for the given key
    return $self->{'INFO'}{lc $key};
}

sub comment_tags 
{
    my $self = shift;

    return @{$self->{'COMMENT_KEYS'}};
}

sub comment 
{
    my $self = shift;
    my $key = shift;

    # if the user supplied key does not exist, return undef
    unless($self->{'COMMENTS'}{lc $key}) {
        return undef;
    }

    return @{$self->{'COMMENTS'}{lc $key}};
}

# "private" methods

sub _init
{
    my $data = shift;
    my $byteCount = 0;

    # check the header to make sure this is actually an Ogg-Vorbis file
    $byteCount = _checkHeader($data);

    unless($byteCount) {
        return undef;
    }

    $data->{'startInfoHeader'} = $byteCount;
}

sub _checkHeader
{
    my $data = shift;
    my $fh = $data->{'fileHandle'};
    my $buffer;
    my $pageSegCount;
    my $byteCount = 0; # stores how far into the file we've read,
                       # so later reads into the file can skip right
                       # past all of the header stuff

    # check that the first four bytes are 'OggS'
    read($fh, $buffer, 4);
    if ($buffer ne 'OggS') {
        return undef;
    }
    $byteCount += 4;

    # check the stream structure version (1 byte, should be 0x00)
    read($fh, $buffer, 1);
    if (ord($buffer) != 0x00) {
        return undef;
    }
    $byteCount += 1;

    # check the header type flag 
    # This is a bitfield, so technically we should check all of the bits
    # that could potentially be set. However, the only value this should
    # possibly have at the beginning of a proper Ogg-Vorbis file is 0x02,
    # so we just check for that. If it's not that, we go on anyway, but
    # give a warning (this behavior may (should?) be modified in the future.
    read($fh, $buffer, 1);
    $byteCount += 1;

    # skip to the page_segments count
    seek $fh, 20, SEEK_CUR;
    $byteCount += 20;

    # read the number of page segments
    read($fh, $buffer, 1);
    $pageSegCount = ord($buffer);
    $byteCount += 1;

    # skip $pageSegCount bytes
    seek $fh, $pageSegCount, SEEK_CUR;
    $byteCount += $pageSegCount;

    # check packet type. Should be 0x01 (for indentification header)
    read($fh, $buffer, 1);
    if (ord($buffer) != 0x01) {
        return undef;
    }
    $byteCount += 1;

    # check that the packet identifies itself as 'vorbis'
    read($fh, $buffer, 6);
    if ($buffer ne 'vorbis') {
        return undef;
    }
    $byteCount += 6;

    # at this point, we assume the bitstream is valid
    return $byteCount;
}

sub _loadInfo
{
    my $data = shift;
    my $start = 0;
    my $fh = $data->{'fileHandle'};
    my $buffer;
    my %info;

    if (!defined $data->{'startInfoHeader'}) {
        return;
    } else {
        $start = $data->{'startInfoHeader'} + 5;
    }
    
    seek $fh, $start, SEEK_SET;

    # read the sample rate
    read($fh, $buffer, 4);
    $info{'rate'} = _decodeInt($buffer);
    
    $data->{'startCommentHeader'} = $start + 18; # header is 23 bytes
    $data->{'INFO'} = \%info;
}

sub _loadComments
{
    my $data = shift;
    my $fh = $data->{'fileHandle'};
    my $start = 0;
    my $buffer;
    my $page_segments;
    my $vendor_length;
    my $user_comment_count;
    my $byteCount = $start;
    my %comments;

    if (!defined $data->{'startCommentHeader'}) {
        return;
    } else {
        $start = $data->{'startCommentHeader'};
    }

    seek $fh, $start + 26, SEEK_SET;
    $byteCount += 26;

    # get the number of entries in the segment_table...
    read($fh, $buffer, 1);
    $page_segments = _decodeInt($buffer);
    $byteCount += 1;

    # skip on past it
    seek $fh, $page_segments, SEEK_CUR;
    $byteCount += $page_segments;

    # check the header type (should be 0x03)
    read($fh, $buffer, 1);
    $byteCount += 1;

    # now we should see 'vorbis'
    read($fh, $buffer, 6);
    if ($buffer ne 'vorbis') {
        return;
    }
    $byteCount += 6;

    # get the vendor length
    read($fh, $buffer, 4);
    $vendor_length = _decodeInt($buffer);
    $byteCount += 4;

    # read in the vendor
    read($fh, $buffer, $vendor_length);
    $comments{'vendor'} = $buffer;
    $byteCount += $vendor_length;

    # read in the number of user comments
    read($fh, $buffer, 4);
    $user_comment_count = _decodeInt($buffer);
    $byteCount += 4;

    $data->{'COMMENT_KEYS'} = [];

    # finally, read the comments
    for (my $i = 0; $i < $user_comment_count; $i++) {
        # first read the length
        read($fh, $buffer, 4);
        my $comment_length = _decodeInt($buffer);
        $byteCount += 4;

        # then the comment itself
        read($fh, $buffer, $comment_length);
        $byteCount += $comment_length;

        my ($key, $value) = split(/=/, $buffer);

        push @{$comments{lc $key}}, $value;
        push @{$data->{'COMMENT_KEYS'}}, lc $key;
    }
    
    # skip the framing_bit
    seek $fh, 1, SEEK_CUR;
    $byteCount += 1;

    $data->{'INFO'}{'offset'} = $byteCount;
    $data->{'COMMENTS'} = \%comments;
}

sub _calculateTrackLength
{
    my $data = shift;
    my $fh = $data->{'fileHandle'};
    my $buffer;
    my $granule_position;
    my $index;

    seek $fh, -8500, SEEK_END;
    # that magic number is from vorbisfile.c in the constant CHUNKSIZE,
    # which comes with the comment /* a shade over 8k; anyone using pages
    # well over 8k gets what they deserve */

    read $fh, $buffer, 8500;
    $index = rindex $buffer, "OggS";

    if ($index >= 0) {
        # seek to the position past OggS
        seek $fh, (-8500 + $index + 4), SEEK_END;

        # stream structure version - must be 0x00
        read($fh, $buffer, 1);
        if (ord($buffer) != 0x00) {
            return;
        }

        # skip header type flag
        seek $fh, 1, SEEK_CUR;

        # absolute granule position - this is what we need!
        read($fh, $buffer, 8);
        $granule_position = _decodeInt($buffer);

        # cool, we're done now
        $data->{'INFO'}{'length'} = $granule_position / $data->{'INFO'}{'rate'};
    } else {
        $data->{'INFO'}{'length'} = 0;
    }
}

sub _decodeInt
{
    my @bytes = split //, shift;
    my $count = scalar @bytes;
    my ($num,$mult) = (0,1);

    for (my $i = 0; $i < $count; $i++) {
        $num += ord($bytes[$i]) * $mult;
        $mult *= 256;
    }
    return $num;
}

1;
