use core:lang;
use core:asm;
use lang:bs:macro;

// Do type checking and -casting for a conditional.
Expr castCond(Block parent, Expr cond) {
	Value type = cond.result.type;

	// TODO: References!

	if (isCppPtr(type)) {
		return CheckPtr(cond);
	}

	if (type.isAsmType) {
		return CheckIntegral(cond);
	}

	// TODO: We don't need to be this restrictive.
	if (type.type !is named{Bool})
		throw SyntaxError(cond.pos, "A condition must evaluate to a boolean, not ${type}.");

	cond;
}


/**
 * Check if a pointer is null.
 */
class CheckPtr extends Expr {
	init(Expr e) {
		init(e.pos) { e = e; }
	}

	Expr e;

	ExprResult result() : override {
		Value(named{Bool});
	}

	void code(CodeGen gen, CodeResult res) : override {
		CodeResult inner(e.result.type.asRef(false), gen.block);
		e.code(gen, inner);

		Var ptr = inner.location(gen);
		gen.l << cmp(ptrRel(ptr), ptrConst(0));
		gen.l << setCond(res.location(gen), CondFlag:ifNotEqual);
	}

	void codePtr(CodeGen gen, Type type) : override {
		CodeResult inner(e.result.type.asRef(false), gen.block);
		e.code(gen, inner);

		CppVar var = allocTemporary(gen, named{Bool});
		var.adjust(gen, ptrA);
		// We only need to check the pointer-part of the pointer.
		gen.l << cmp(ptrRel(inner.location(gen)), ptrConst(0));
		gen.l << setCond(byteRel(ptrA), CondFlag:ifNotEqual);
		var.created(gen);
		gen.l << location(SrcPos());
		var.ptr(gen);
	}
}


/**
 * Check if an integer value is nonzero.
 */
class CheckIntegral extends Expr {
	init(Expr e) {
		init(e.pos) { e = e; }
	}

	Expr e;

	ExprResult result() : override {
		Value(named{Bool});
	}

	void code(CodeGen gen, CodeResult res) : override {
		CodeResult inner(e.result.type.asRef(false), gen.block);
		e.code(gen, inner);

		Var val = inner.location(gen);
		gen.l << cmp(val, xConst(val.size, 0));
		gen.l << setCond(res.location(gen), CondFlag:ifNotEqual);
	}

	void codePtr(CodeGen gen, Type type) : override {
		CodeResult inner(e.result.type.asRef(false), gen.block);
		e.code(gen, inner);
		Var val = inner.location(gen);

		CppVar var = allocTemporary(gen, named{Bool});
		var.adjust(gen, ptrA);
		gen.l << cmp(val, xConst(val.size, 0));
		gen.l << setCond(byteRel(ptrA), CondFlag:ifNotEqual);
		var.created(gen);
		gen.l << location(SrcPos());
		var.ptr(gen);
	}
}


/**
 * An if-statement.
 */
class IfStmt extends Block {
	init(SrcPos pos, Block parent) {
		init(pos, parent) {}
	}

	// Condition.
	Expr? cond;

	// True branch.
	Stmt? ifBranch;

	// False branch.
	Stmt? elseBranch;

	// Setters for the grammar.
	void cond(Expr e) { cond = castCond(this, e); }
	void ifTrue(Stmt s) { ifBranch = s; }
	void ifFalse(Stmt s) { elseBranch = s; }

	Bool blockCode(CodeGen gen) : override {
		unless (cond)
			throw SyntaxError(pos, "This if-statement has no condition set.");

		var onFalse = gen.l.label();
		var exit = gen.l.label();

		CodeResult condResult(Value(named{Bool}), gen.block);
		CodeGen sub = gen.child();
		if (step)
			gen.l << location(cond.pos);
		gen.l << begin(sub.block);
		cond.code(sub, condResult);
		gen.l << end(sub.block);

		gen.l << cmp(condResult.location(gen), byteConst(0));
		gen.l << jmp(onFalse, CondFlag:ifEqual);

		Bool returns = false;
		if (ifBranch)
			returns |= executeStmt(gen, ifBranch);
		else
			returns |= true;

		gen.l << jmp(exit);
		gen.l << onFalse;

		if (elseBranch)
			returns |= executeStmt(gen, elseBranch);
		else
			returns |= true;


		gen.l << exit;

		returns;
	}
}

/**
 * Logical and/or (short circuiting).
 */
class LogicOp extends ExprRef {
	// If "and" is true, we're an and operation, otherwise we're an or operation.
	init(SrcPos pos, Block parent, Expr lhs, Expr rhs, Bool and) {
		init(pos) {
			lhs = castCond(parent, lhs);
			rhs = castCond(parent, rhs);
			and = and;
			step = parent.step;
		}
	}

	private Expr lhs;
	private Expr rhs;
	private Bool and;
	private Bool step;

	ExprResult result() : override {
		Value(named{Bool}).asRef(true);
	}

	void code(CodeGen gen, CodeResult res) : override {
		if (res.type.ref) {
			CppVar var = allocType(gen, named{Bool});
			codeRef(gen, var);

			var.adjust(gen, res.safeLocation(gen, Value(named{Bool}).asRef()));
		} else {
			code(gen, res.safeLocation(gen, Value(named{Bool})));
		}
		res.created(gen);
	}

	void codeRef(CodeGen gen, CppVar var) : override {
		code(gen, al);
		var.adjust(gen, ptrB);
		gen.l << mov(byteRel(ptrB), al);
		var.created(gen);
	}

	private void code(CodeGen gen, Operand result) {
		Value bool(named{Bool});
		CodeResult lResult(bool, gen.block);
		if (step)
			gen.l << location(lhs.pos);
		lhs.code(gen, lResult);

		Label skip = gen.l.label();

		gen.l << mov(result, lResult.location(gen));
		gen.l << cmp(result, byteConst(0));

		if (and)
			gen.l << jmp(skip, CondFlag:ifEqual);
		else
			gen.l << jmp(skip, CondFlag:ifNotEqual);

		CodeResult rResult(bool, gen.block);
		if (step)
			gen.l << location(rhs.pos);
		rhs.code(gen, rResult);

		gen.l << mov(result, rResult.location(gen));

		gen.l << skip;
	}
}

/**
 * Negation operator (!)
 */
class NegateOp extends ExprRef {
	init(SrcPos pos, Block parent, Expr expr) {
		init(pos) {
			expr = castCond(parent, expr);
		}
	}

	private Expr expr;

	ExprResult result() : override {
		Value(named{Bool}).asRef(true);
	}

	void code(CodeGen gen, CodeResult res) : override {
		if (res.type.ref) {
			CppVar var = allocType(gen, named{Bool});
			codeRef(gen, var);

			var.adjust(gen, res.safeLocation(gen, Value(named{Bool}).asRef()));
		} else {
			code(gen, res.safeLocation(gen, Value(named{Bool})));
		}
		res.created(gen);
	}

	void codeRef(CodeGen gen, CppVar var) : override {
		code(gen, al);
		var.adjust(gen, ptrB);
		gen.l << mov(byteRel(ptrB), al);
		var.created(gen);
	}

	private void code(CodeGen gen, Operand result) {
		Value bool(named{Bool});
		CodeResult res(bool, gen.block);
		expr.code(gen, res);

		gen.l << mov(result, res.location(gen));
		gen.l << cmp(result, byteConst(0));
		gen.l << setCond(result, CondFlag:ifEqual);
	}
}
