#!/usr/bin/env python

"""SWIG Static Binding Copier

This script copies over the Python bindings generated by SWIG from the build
directory to the source directory. Before using the script, make sure you have
LLDB_USE_STATIC_BINDINGS set to OFF by looking at CMakeCache.txt in the LLDB
build directory.

The scripts knows the location of the static bindings in the source directory
based on its own location. The build directory must be specified as a position
argument.

$ copy-static-bindings.py <path to LLDB build directory>

Run this script whenever you're changing any of the .i interface files in the
bindings directory.
"""

import argparse
import os
import sys
import shutil


def copy_bindings(build_dir, source_dir, language, extensions=['.cpp']):
    binding_build_dir = os.path.join(build_dir, 'bindings', language)
    binding_source_dir = os.path.join(source_dir, 'bindings', language,
                                      'static-binding')

    for root, _, files in os.walk(binding_build_dir):
        for file in files:
            _, extension = os.path.splitext(file)
            filepath = os.path.join(root, file)
            if extension in extensions:
                shutil.copy(filepath, os.path.join(binding_source_dir, file))


def main():
    parser = argparse.ArgumentParser(description='Copy the static bindings')
    parser.add_argument('build_dir',
                        type=str,
                        help='Path to the root of the LLDB build directory')

    args = parser.parse_args()

    build_dir = args.build_dir
    if not os.path.exists(build_dir):
        print("error: the build directory does not exist: {}".format(
            args.build_dir))
        sys.exit(1)

    source_dir = os.path.dirname(os.path.dirname(os.path.realpath(__file__)))
    if not os.path.exists(source_dir):
        print("error: the source directory does not exist: {}".format(
            source_dir))
        sys.exit(1)

    copy_bindings(build_dir, source_dir, 'python', ['.py', '.cpp'])
    copy_bindings(build_dir, source_dir, 'lua', ['.cpp'])


if __name__ == "__main__":
    main()
