#!/usr/bin/env python
copyright = '''
/*
 * Copyright 2009 VMware, Inc.
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * on the rights to use, copy, modify, merge, publish, distribute, sub
 * license, and/or sell copies of the Software, and to permit persons to whom
 * the Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT.  IN NO EVENT SHALL
 * VMWARE AND/OR THEIR SUPPLIERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
 * USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
'''

GENERATE, UBYTE, USHORT, UINT = 'generate', 'ubyte', 'ushort', 'uint'
FIRST, LAST = 'first', 'last'

INTYPES = (GENERATE, UBYTE, USHORT, UINT)
OUTTYPES = (USHORT, UINT)
PVS=(FIRST, LAST)
PRIMS=('points', 
       'lines', 
       'linestrip', 
       'lineloop', 
       'tris', 
       'trifan', 
       'tristrip', 
       'quads', 
       'quadstrip', 
       'polygon')

LONGPRIMS=('PIPE_PRIM_POINTS', 
           'PIPE_PRIM_LINES', 
           'PIPE_PRIM_LINE_STRIP', 
           'PIPE_PRIM_LINE_LOOP', 
           'PIPE_PRIM_TRIANGLES', 
           'PIPE_PRIM_TRIANGLE_FAN', 
           'PIPE_PRIM_TRIANGLE_STRIP', 
           'PIPE_PRIM_QUADS', 
           'PIPE_PRIM_QUAD_STRIP', 
           'PIPE_PRIM_POLYGON')

longprim = dict(zip(PRIMS, LONGPRIMS))
intype_idx = dict(ubyte='IN_UBYTE', ushort='IN_USHORT', uint='IN_UINT')
outtype_idx = dict(ushort='OUT_USHORT', uint='OUT_UINT')
pv_idx = dict(first='PV_FIRST', last='PV_LAST')


def prolog():
    print '''/* File automatically generated by indices.py */'''
    print copyright
    print r'''

/**
 * @file
 * Functions to translate and generate index lists
 */

#include "indices/u_indices.h"
#include "indices/u_indices_priv.h"
#include "pipe/p_compiler.h"
#include "util/u_debug.h"
#include "pipe/p_defines.h"
#include "util/u_memory.h"


static unsigned out_size_idx( unsigned index_size )
{
   switch (index_size) {
   case 4: return OUT_UINT;
   case 2: return OUT_USHORT;
   default: assert(0); return OUT_USHORT;
   }
}

static unsigned in_size_idx( unsigned index_size )
{
   switch (index_size) {
   case 4: return IN_UINT;
   case 2: return IN_USHORT;
   case 1: return IN_UBYTE;
   default: assert(0); return IN_UBYTE;
   }
}


static u_translate_func translate[IN_COUNT][OUT_COUNT][PV_COUNT][PV_COUNT][PRIM_COUNT];
static u_generate_func  generate[OUT_COUNT][PV_COUNT][PV_COUNT][PRIM_COUNT];


'''

def vert( intype, outtype, v0 ):
    if intype == GENERATE:
        return '(' + outtype + ')(' + v0 + ')'
    else:
        return '(' + outtype + ')in[' + v0 + ']'

def point( intype, outtype, ptr, v0 ):
    print '      (' + ptr + ')[0] = ' + vert( intype, outtype, v0 ) + ';'

def line( intype, outtype, ptr, v0, v1 ):
    print '      (' + ptr + ')[0] = ' + vert( intype, outtype, v0 ) + ';'
    print '      (' + ptr + ')[1] = ' + vert( intype, outtype, v1 ) + ';'

def tri( intype, outtype, ptr, v0, v1, v2 ):
    print '      (' + ptr + ')[0] = ' + vert( intype, outtype, v0 ) + ';'
    print '      (' + ptr + ')[1] = ' + vert( intype, outtype, v1 ) + ';'
    print '      (' + ptr + ')[2] = ' + vert( intype, outtype, v2 ) + ';'

def do_point( intype, outtype, ptr, v0 ):
    point( intype, outtype, ptr, v0 )

def do_line( intype, outtype, ptr, v0, v1, inpv, outpv ):
    if inpv == outpv:
        line( intype, outtype, ptr, v0, v1 )
    else:
        line( intype, outtype, ptr, v1, v0 )

def do_tri( intype, outtype, ptr, v0, v1, v2, inpv, outpv ):
    if inpv == outpv:
        tri( intype, outtype, ptr, v0, v1, v2 )
    else: 
        if inpv == FIRST:
            tri( intype, outtype, ptr, v1, v2, v0 )
        else:
            tri( intype, outtype, ptr, v2, v0, v1 )

def do_quad( intype, outtype, ptr, v0, v1, v2, v3, inpv, outpv ):
    do_tri( intype, outtype, ptr+'+0',  v0, v1, v3, inpv, outpv );
    do_tri( intype, outtype, ptr+'+3',  v1, v2, v3, inpv, outpv );

def name(intype, outtype, inpv, outpv, prim):
    if intype == GENERATE:
        return 'generate_' + prim + '_' + outtype + '_' + inpv + '2' + outpv
    else:
        return 'translate_' + prim + '_' + intype + '2' + outtype + '_' + inpv + '2' + outpv

def preamble(intype, outtype, inpv, outpv, prim):
    print 'static void ' + name( intype, outtype, inpv, outpv, prim ) + '('
    if intype != GENERATE:
        print '    const void * _in,'
    print '    unsigned start,'
    print '    unsigned nr,'
    print '    void *_out )'
    print '{'
    if intype != GENERATE:
        print '  const ' + intype + '*in = (const ' + intype + '*)_in;'
    print '  ' + outtype + ' *out = (' + outtype + '*)_out;'
    print '  unsigned i, j;'
    print '  (void)j;'

def postamble():
    print '}'


def points(intype, outtype, inpv, outpv):
    preamble(intype, outtype, inpv, outpv, prim='points')
    print '  for (i = start; i < (nr+start); i++) { '
    do_point( intype, outtype, 'out+i',  'i' );
    print '   }'
    postamble()

def lines(intype, outtype, inpv, outpv):
    preamble(intype, outtype, inpv, outpv, prim='lines')
    print '  for (i = start; i < (nr+start); i+=2) { '
    do_line( intype, outtype, 'out+i',  'i', 'i+1', inpv, outpv );
    print '   }'
    postamble()

def linestrip(intype, outtype, inpv, outpv):
    preamble(intype, outtype, inpv, outpv, prim='linestrip')
    print '  for (i = start, j = 0; j < nr; j+=2, i++) { '
    do_line( intype, outtype, 'out+j',  'i', 'i+1', inpv, outpv );
    print '   }'
    postamble()

def lineloop(intype, outtype, inpv, outpv):
    preamble(intype, outtype, inpv, outpv, prim='lineloop')
    print '  for (i = start, j = 0; j < nr - 2; j+=2, i++) { '
    do_line( intype, outtype, 'out+j',  'i', 'i+1', inpv, outpv );
    print '   }'
    do_line( intype, outtype, 'out+j',  'i', '0', inpv, outpv );
    postamble()

def tris(intype, outtype, inpv, outpv):
    preamble(intype, outtype, inpv, outpv, prim='tris')
    print '  for (i = start; i < (nr+start); i+=3) { '
    do_tri( intype, outtype, 'out+i',  'i', 'i+1', 'i+2', inpv, outpv );
    print '   }'
    postamble()


def tristrip(intype, outtype, inpv, outpv):
    preamble(intype, outtype, inpv, outpv, prim='tristrip')
    print '  for (i = start, j = 0; j < nr; j+=3, i++) { '
    if inpv == FIRST:
        do_tri( intype, outtype, 'out+j',  'i', 'i+1+(i&1)', 'i+2-(i&1)', inpv, outpv );
    else:
        do_tri( intype, outtype, 'out+j',  'i+(i&1)', 'i+1-(i&1)', 'i+2', inpv, outpv );
    print '   }'
    postamble()


def trifan(intype, outtype, inpv, outpv):
    preamble(intype, outtype, inpv, outpv, prim='trifan')
    print '  for (i = start, j = 0; j < nr; j+=3, i++) { '
    do_tri( intype, outtype, 'out+j',  '0', 'i+1', 'i+2', inpv, outpv );
    print '   }'
    postamble()



def polygon(intype, outtype, inpv, outpv):
    preamble(intype, outtype, inpv, outpv, prim='polygon')
    print '  for (i = start, j = 0; j < nr; j+=3, i++) { '
    if inpv == FIRST:
        do_tri( intype, outtype, 'out+j',  '0', 'i+1', 'i+2', inpv, outpv );
    else:
        do_tri( intype, outtype, 'out+j',  'i+1', 'i+2', '0', inpv, outpv );
    print '   }'
    postamble()


def quads(intype, outtype, inpv, outpv):
    preamble(intype, outtype, inpv, outpv, prim='quads')
    print '  for (i = start, j = 0; j < nr; j+=6, i+=4) { '
    do_quad( intype, outtype, 'out+j', 'i+0', 'i+1', 'i+2', 'i+3', inpv, outpv );
    print '   }'
    postamble()


def quadstrip(intype, outtype, inpv, outpv):
    preamble(intype, outtype, inpv, outpv, prim='quadstrip')
    print '  for (i = start, j = 0; j < nr; j+=6, i+=2) { '
    do_quad( intype, outtype, 'out+j', 'i+2', 'i+0', 'i+1', 'i+3', inpv, outpv );
    print '   }'
    postamble()


def emit_funcs():
    for intype in INTYPES:
        for outtype in OUTTYPES:
            for inpv in (FIRST, LAST):
                for outpv in (FIRST, LAST):
                    points(intype, outtype, inpv, outpv)
                    lines(intype, outtype, inpv, outpv)
                    linestrip(intype, outtype, inpv, outpv)
                    lineloop(intype, outtype, inpv, outpv)
                    tris(intype, outtype, inpv, outpv)
                    tristrip(intype, outtype, inpv, outpv)
                    trifan(intype, outtype, inpv, outpv)
                    quads(intype, outtype, inpv, outpv)
                    quadstrip(intype, outtype, inpv, outpv)
                    polygon(intype, outtype, inpv, outpv)

def init(intype, outtype, inpv, outpv, prim):
    if intype == GENERATE:
        print ('generate[' + 
               outtype_idx[outtype] + 
               '][' + pv_idx[inpv] + 
               '][' + pv_idx[outpv] + 
               '][' + longprim[prim] + 
               '] = ' + name( intype, outtype, inpv, outpv, prim ) + ';')
    else:
        print ('translate[' + 
               intype_idx[intype] + 
               '][' + outtype_idx[outtype] + 
               '][' + pv_idx[inpv] + 
               '][' + pv_idx[outpv] + 
               '][' + longprim[prim] + 
               '] = ' + name( intype, outtype, inpv, outpv, prim ) + ';')


def emit_all_inits():
    for intype in INTYPES:
        for outtype in OUTTYPES:
            for inpv in PVS:
                for outpv in PVS:
                    for prim in PRIMS:
                        init(intype, outtype, inpv, outpv, prim)

def emit_init():
    print 'void u_index_init( void )'
    print '{'
    print '  static int firsttime = 1;'
    print '  if (!firsttime) return;'
    print '  firsttime = 0;'
    emit_all_inits()
    print '}'


    

def epilog():
    print '#include "indices/u_indices.c"'


def main():
    prolog()
    emit_funcs()
    emit_init()
    epilog()


if __name__ == '__main__':
    main()
