/*
 * Copyright © 2007-2019 Advanced Micro Devices, Inc.
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sub license, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NON-INFRINGEMENT. IN NO EVENT SHALL THE COPYRIGHT HOLDERS, AUTHORS
 * AND/OR ITS SUPPLIERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
 * USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 */

/**
************************************************************************************************************************
* @file  gfx10SwizzlePattern.h
* @brief swizzle pattern for gfx10.
************************************************************************************************************************
*/

#ifndef __GFX10_SWIZZLE_PATTERN_H__
#define __GFX10_SWIZZLE_PATTERN_H__

namespace Addr
{
namespace V2
{
/**
************************************************************************************************************************
* @brief Bit setting for swizzle pattern
************************************************************************************************************************
*/
union ADDR_BIT_SETTING
{
    struct
    {
        UINT_16 x;
        UINT_16 y;
        UINT_16 z;
        UINT_16 s;
    };
    UINT_64 value;
};

/**
************************************************************************************************************************
*   InitBit
*
*   @brief
*       Initialize bit setting value via a return value
************************************************************************************************************************
*/
#define InitBit(c, index) (1ull << ((c << 4) + index))

const UINT_64 X0  = InitBit(0,  0);
const UINT_64 X1  = InitBit(0,  1);
const UINT_64 X2  = InitBit(0,  2);
const UINT_64 X3  = InitBit(0,  3);
const UINT_64 X4  = InitBit(0,  4);
const UINT_64 X5  = InitBit(0,  5);
const UINT_64 X6  = InitBit(0,  6);
const UINT_64 X7  = InitBit(0,  7);
const UINT_64 X8  = InitBit(0,  8);
const UINT_64 X9  = InitBit(0,  9);
const UINT_64 X10 = InitBit(0, 10);
const UINT_64 X11 = InitBit(0, 11);

const UINT_64 Y0  = InitBit(1,  0);
const UINT_64 Y1  = InitBit(1,  1);
const UINT_64 Y2  = InitBit(1,  2);
const UINT_64 Y3  = InitBit(1,  3);
const UINT_64 Y4  = InitBit(1,  4);
const UINT_64 Y5  = InitBit(1,  5);
const UINT_64 Y6  = InitBit(1,  6);
const UINT_64 Y7  = InitBit(1,  7);
const UINT_64 Y8  = InitBit(1,  8);
const UINT_64 Y9  = InitBit(1,  9);
const UINT_64 Y10 = InitBit(1, 10);
const UINT_64 Y11 = InitBit(1, 11);

const UINT_64 Z0  = InitBit(2,  0);
const UINT_64 Z1  = InitBit(2,  1);
const UINT_64 Z2  = InitBit(2,  2);
const UINT_64 Z3  = InitBit(2,  3);
const UINT_64 Z4  = InitBit(2,  4);
const UINT_64 Z5  = InitBit(2,  5);
const UINT_64 Z6  = InitBit(2,  6);
const UINT_64 Z7  = InitBit(2,  7);
const UINT_64 Z8  = InitBit(2,  8);

const UINT_64 S0  = InitBit(3,  0);
const UINT_64 S1  = InitBit(3,  1);
const UINT_64 S2  = InitBit(3,  2);

// Color data swizzle pattern
const UINT_64 SW_256_S[][8]=
{
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1              },
};

const UINT_64 SW_256_D[][8]=
{
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1              },
};

const UINT_64 SW_4K_S[][12]=
{
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3              },
};

const UINT_64 SW_4K_D[][12]=
{
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3              },
};

const UINT_64 SW_4K_S_X[][12]=
{
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z0 ^ Y3 ^ X4,   X4,             Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Z0 ^ Y2 ^ X3,   X3,             Y3,             X4              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Z0 ^ X2 ^ Y2,   X2,             Y3,             X3              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4,   Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4,   Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Z1 ^ Y2 ^ X4,   Z0 ^ X3 ^ Y3,   Y3,             X4              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Z1 ^ Y2 ^ X3,   Z0 ^ X2 ^ Y3,   Y3,             X3              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z2 ^ Y3 ^ X6,   Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z2 ^ Y3 ^ X5,   Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ Z2 ^ X5,   Z1 ^ X3 ^ Y4,   Z0 ^ Y3 ^ X4,   X4              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ Z2 ^ X4,   Z1 ^ X2 ^ Y4,   Z0 ^ X3 ^ Y3,   X3              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ Z3 ^ X5,   X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ Z3 ^ X5,   X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ Z3 ^ X5,   X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z0 ^ X4 ^ Y4,   Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z0 ^ X3 ^ Y4,   Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             Z0 ^ X3 ^ Y3,   Y3,             X4              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             Z0 ^ X2 ^ Y3,   Y3,             X3              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             Z1 ^ X3 ^ Y4,   Z0 ^ Y3 ^ X4,   X4              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             Z1 ^ X2 ^ Y4,   Z0 ^ X3 ^ Y3,   X3              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Z0 ^ X5 ^ Y5,   X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z0 ^ Y4 ^ X5,   X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z0 ^ X4 ^ Y4,   X4              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Z0 ^ Y3 ^ X4,   X4              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Z0 ^ X3 ^ Y3,   X3              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
};

const UINT_64 SW_4K_D_X[][12]=
{
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z0 ^ Y3 ^ X4,   X4,             Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z0 ^ Y2 ^ X3,   X3,             Y3,             X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z0 ^ X2 ^ Y2,   X2,             Y3,             X3              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4,   Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4,   Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z1 ^ Y2 ^ X4,   Z0 ^ X3 ^ Y3,   Y3,             X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z1 ^ Y2 ^ X3,   Z0 ^ X2 ^ Y3,   Y3,             X3              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z2 ^ Y3 ^ X6,   Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z2 ^ Y3 ^ X5,   Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ Z2 ^ X5,   Z1 ^ X3 ^ Y4,   Z0 ^ Y3 ^ X4,   X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ Z2 ^ X4,   Z1 ^ X2 ^ Y4,   Z0 ^ X3 ^ Y3,   X3              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ Z3 ^ X5,   X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ Z3 ^ X5,   X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ Z3 ^ X5,   X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z0 ^ X4 ^ Y4,   Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z0 ^ X3 ^ Y4,   Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z0 ^ X3 ^ Y3,   Y3,             X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             Z0 ^ X2 ^ Y3,   Y3,             X3              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z1 ^ X3 ^ Y4,   Z0 ^ Y3 ^ X4,   X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             Z1 ^ X2 ^ Y4,   Z0 ^ X3 ^ Y3,   X3              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Z0 ^ X5 ^ Y5,   X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z0 ^ Y4 ^ X5,   X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z0 ^ X4 ^ Y4,   X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z0 ^ Y3 ^ X4,   X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Z0 ^ X3 ^ Y3,   X3              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
};

const UINT_64 SW_64K_S[][16]=
{
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
};

const UINT_64 SW_64K_D[][16]=
{
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
};

const UINT_64 SW_64K_S_T[][16]=
{
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X4 ^ Y4,        X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X4,        X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3 ^ Y3,        X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X3,        X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             X2 ^ Y2,        X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X4,        X3 ^ Y3,        Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X3,        X2 ^ Y3,        Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X5,        X3 ^ Y4,        Y3 ^ X4,        X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X4,        X2 ^ Y4,        X3 ^ Y3,        X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X7,        X4 ^ Y7,        Y5 ^ X6,        X5 ^ Y6,        Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X7,        X4 ^ Y6,        Y4 ^ X6,        X5 ^ Y5,        Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X6,        X3 ^ Y6,        Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X6,        X3 ^ Y5,        Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X5,        X2 ^ Y5,        Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X7,        X5 ^ Y7,        X6 ^ Y6,        X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6,        X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6 ^ X7,        X6 ^ Y7,        Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X7,        X6 ^ Y6,        Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Y5,        Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y4,        Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y4,        Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Y3,        Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Y3,        Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Y4,        Y3 ^ X4,        X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Y4,        X3 ^ Y3,        X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Y7,        Y5 ^ X6,        X5 ^ Y6,        Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Y4 ^ X6,        X5 ^ Y5,        Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y6,        Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Y5,        Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Y5,        Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X7,        X5 ^ Y7,        X6 ^ Y6,        X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6,        X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6 ^ X7,        X6 ^ Y7,        Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X7,        X6 ^ Y6,        Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             X7 ^ Y7,        X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6 ^ X7,        X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             X6 ^ Y6,        X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X6,        X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             X5 ^ Y5,        X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             X5 ^ Y5,        X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X5,        X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             X4 ^ Y4,        X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X4,        X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             X3 ^ Y3,        X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X7,        X5 ^ Y7,        X6 ^ Y6,        X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6,        X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6 ^ X7,        X6 ^ Y7,        Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X7,        X6 ^ Y6,        Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             X7 ^ Y7,        X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6 ^ X7,        X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             X6 ^ Y6,        X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X6,        X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             X5 ^ Y5,        X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
};

const UINT_64 SW_64K_D_T[][16]=
{
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             X4 ^ Y4,        X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X4,        X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3 ^ Y3,        X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X3,        X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2 ^ Y2,        X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X4,        X3 ^ Y3,        Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X3,        X2 ^ Y3,        Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X5,        X3 ^ Y4,        Y3 ^ X4,        X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X4,        X2 ^ Y4,        X3 ^ Y3,        X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X7,        X4 ^ Y7,        Y5 ^ X6,        X5 ^ Y6,        Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X7,        X4 ^ Y6,        Y4 ^ X6,        X5 ^ Y5,        Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X6,        X3 ^ Y6,        Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X6,        X3 ^ Y5,        Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X5,        X2 ^ Y5,        Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X7,        X5 ^ Y7,        X6 ^ Y6,        X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6,        X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6 ^ X7,        X6 ^ Y7,        Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X7,        X6 ^ Y6,        Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Y5,        Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y4,        Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y4,        Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y3,        Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Y3,        Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y4,        Y3 ^ X4,        X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Y4,        X3 ^ Y3,        X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Y7,        Y5 ^ X6,        X5 ^ Y6,        Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Y4 ^ X6,        X5 ^ Y5,        Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y6,        Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y5,        Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Y5,        Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X7,        X5 ^ Y7,        X6 ^ Y6,        X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6,        X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6 ^ X7,        X6 ^ Y7,        Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X7,        X6 ^ Y6,        Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             X7 ^ Y7,        X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6 ^ X7,        X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             X6 ^ Y6,        X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X6,        X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             X5 ^ Y5,        X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             X5 ^ Y5,        X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X5,        X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             X4 ^ Y4,        X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X4,        X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             X3 ^ Y3,        X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X7,        X5 ^ Y7,        X6 ^ Y6,        X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6,        X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6 ^ X7,        X6 ^ Y7,        Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X7,        X6 ^ Y6,        Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             X7 ^ Y7,        X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6 ^ X7,        X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             X6 ^ Y6,        X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X6,        X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             X5 ^ Y5,        X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
};

const UINT_64 SW_64K_S_X[][16]=
{
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z0 ^ Y3 ^ X4,   X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Z0 ^ Y2 ^ X3,   X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Z0 ^ X2 ^ Y2,   X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Z1 ^ Y2 ^ X4,   Z0 ^ X3 ^ Y3,   Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Z1 ^ Y2 ^ X3,   Z0 ^ X2 ^ Y3,   Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z2 ^ Y3 ^ X6,   Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z2 ^ Y3 ^ X5,   Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ Z2 ^ X5,   Z1 ^ X3 ^ Y4,   Z0 ^ Y3 ^ X4,   X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ Z2 ^ X4,   Z1 ^ X2 ^ Y4,   Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ Z3 ^ X5,   X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ Z4 ^ X8,   Z3 ^ X4 ^ Y8,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ Z4 ^ X8,   Z3 ^ X4 ^ Y7,   Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ Z4 ^ X7,   X3 ^ Z3 ^ Y7,   Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ Z4 ^ X7,   X3 ^ Z3 ^ Y6,   Z2 ^ Y3 ^ X6,   Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ Z4 ^ X6,   X2 ^ Z3 ^ Y6,   Z2 ^ Y3 ^ X5,   Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ Z5 ^ X9,   X4 ^ Z4 ^ Y9,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ Z5 ^ X9,   X4 ^ Z4 ^ Y8,   Z3 ^ Y4 ^ X8,   Z2 ^ X5 ^ Y7,   Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6,   Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ Z5 ^ X8,   X3 ^ Z4 ^ Y8,   Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ Z5 ^ X8,   X3 ^ Z4 ^ Y7,   Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ Z5 ^ X7,   X2 ^ Z4 ^ Y7,   Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             Z0 ^ X3 ^ Y3,   Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             Z0 ^ X2 ^ Y3,   Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             Z1 ^ X3 ^ Y4,   Z0 ^ Y3 ^ X4,   X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             Z1 ^ X2 ^ Y4,   Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             Z3 ^ X4 ^ Y8,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z3 ^ X4 ^ Y7,   Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Z3 ^ Y7,   Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Z3 ^ Y6,   Z2 ^ Y3 ^ X6,   Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Z3 ^ Y6,   Z2 ^ Y3 ^ X5,   Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Z4 ^ Y9,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Z4 ^ Y8,   Z3 ^ Y4 ^ X8,   Z2 ^ X5 ^ Y7,   Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6,   Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Z4 ^ Y8,   Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Z4 ^ Y7,   Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Z4 ^ Y7,   Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Z5 ^Y10,   Z4 ^ Y5 ^ X9,   Z3 ^ X5 ^ Y9,   Z2 ^ Y6 ^ X8,   Z1 ^ X6 ^ Y8,   Z0 ^ X7 ^ Y7,   X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Z5 ^ Y9,   Y4 ^ Z4 ^ X9,   Z3 ^ X5 ^ Y8,   Z2 ^ Y5 ^ X8,   Z1 ^ X6 ^ Y7,   Z0 ^ Y6 ^ X7,   X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Z5 ^ Y9,   Y4 ^ Z4 ^ X8,   Z3 ^ X4 ^ Y8,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Z5 ^ Y8,   Y3 ^ Z4 ^ X8,   Z3 ^ X4 ^ Y7,   Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Z5 ^ Y8,   Y3 ^ Z4 ^ X7,   X3 ^ Z3 ^ Y7,   Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Z0 ^ X5 ^ Y5,   X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z0 ^ Y4 ^ X5,   X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Z0 ^ Y3 ^ X4,   X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Z2 ^ Y3 ^ X6,   Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Z2 ^ Y3 ^ X5,   Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z3 ^ Y4 ^ X8,   Z2 ^ X5 ^ Y7,   Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6,   Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Z4 ^ Y5 ^ X9,   Z3 ^ X5 ^ Y9,   Z2 ^ Y6 ^ X8,   Z1 ^ X6 ^ Y8,   Z0 ^ X7 ^ Y7,   X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ Z4 ^ X9,   Z3 ^ X5 ^ Y8,   Z2 ^ Y5 ^ X8,   Z1 ^ X6 ^ Y7,   Z0 ^ Y6 ^ X7,   X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ Z4 ^ X8,   Z3 ^ X4 ^ Y8,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ Z4 ^ X8,   Z3 ^ X4 ^ Y7,   Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ Z4 ^ X7,   X3 ^ Z3 ^ Y7,   Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ Z5 ^X10,   Z4 ^ X5 ^Y10,   Z3 ^ Y6 ^ X9,   Z2 ^ X6 ^ Y9,   Z1 ^ Y7 ^ X8,   Z0 ^ X7 ^ Y8    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ Z5 ^X10,   Z4 ^ X5 ^ Y9,   Z3 ^ Y5 ^ X9,   Z2 ^ X6 ^ Y8,   Z1 ^ Y6 ^ X8,   Z0 ^ X7 ^ Y7    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ Z5 ^ X9,   X4 ^ Z4 ^ Y9,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ Z5 ^ X9,   X4 ^ Z4 ^ Y8,   Z3 ^ Y4 ^ X8,   Z2 ^ X5 ^ Y7,   Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ Z5 ^ X8,   X3 ^ Z4 ^ Y8,   Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
};

const UINT_64 SW_64K_D_X[][16]=
{
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z0 ^ Y3 ^ X4,   X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z0 ^ Y2 ^ X3,   X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z0 ^ X2 ^ Y2,   X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z1 ^ Y2 ^ X4,   Z0 ^ X3 ^ Y3,   Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z1 ^ Y2 ^ X3,   Z0 ^ X2 ^ Y3,   Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z2 ^ Y3 ^ X6,   Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z2 ^ Y3 ^ X5,   Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ Z2 ^ X5,   Z1 ^ X3 ^ Y4,   Z0 ^ Y3 ^ X4,   X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ Z2 ^ X4,   Z1 ^ X2 ^ Y4,   Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ Z3 ^ X5,   X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ Z4 ^ X8,   Z3 ^ X4 ^ Y8,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ Z4 ^ X8,   Z3 ^ X4 ^ Y7,   Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ Z4 ^ X7,   X3 ^ Z3 ^ Y7,   Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ Z4 ^ X7,   X3 ^ Z3 ^ Y6,   Z2 ^ Y3 ^ X6,   Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ Z4 ^ X6,   X2 ^ Z3 ^ Y6,   Z2 ^ Y3 ^ X5,   Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ Z5 ^ X9,   X4 ^ Z4 ^ Y9,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ Z5 ^ X9,   X4 ^ Z4 ^ Y8,   Z3 ^ Y4 ^ X8,   Z2 ^ X5 ^ Y7,   Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6,   Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ Z5 ^ X8,   X3 ^ Z4 ^ Y8,   Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ Z5 ^ X8,   X3 ^ Z4 ^ Y7,   Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ Z5 ^ X7,   X2 ^ Z4 ^ Y7,   Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z0 ^ X3 ^ Y3,   Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             Z0 ^ X2 ^ Y3,   Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z1 ^ X3 ^ Y4,   Z0 ^ Y3 ^ X4,   X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             Z1 ^ X2 ^ Y4,   Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             Z3 ^ X4 ^ Y8,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z3 ^ X4 ^ Y7,   Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Z3 ^ Y7,   Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Z3 ^ Y6,   Z2 ^ Y3 ^ X6,   Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Z3 ^ Y6,   Z2 ^ Y3 ^ X5,   Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Z4 ^ Y9,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Z4 ^ Y8,   Z3 ^ Y4 ^ X8,   Z2 ^ X5 ^ Y7,   Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6,   Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Z4 ^ Y8,   Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Z4 ^ Y7,   Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Z4 ^ Y7,   Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Z5 ^Y10,   Z4 ^ Y5 ^ X9,   Z3 ^ X5 ^ Y9,   Z2 ^ Y6 ^ X8,   Z1 ^ X6 ^ Y8,   Z0 ^ X7 ^ Y7,   X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Z5 ^ Y9,   Y4 ^ Z4 ^ X9,   Z3 ^ X5 ^ Y8,   Z2 ^ Y5 ^ X8,   Z1 ^ X6 ^ Y7,   Z0 ^ Y6 ^ X7,   X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Z5 ^ Y9,   Y4 ^ Z4 ^ X8,   Z3 ^ X4 ^ Y8,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Z5 ^ Y8,   Y3 ^ Z4 ^ X8,   Z3 ^ X4 ^ Y7,   Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Z5 ^ Y8,   Y3 ^ Z4 ^ X7,   X3 ^ Z3 ^ Y7,   Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Z0 ^ X5 ^ Y5,   X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z0 ^ Y4 ^ X5,   X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z0 ^ Y3 ^ X4,   X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z2 ^ Y3 ^ X6,   Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Z2 ^ Y3 ^ X5,   Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z3 ^ Y4 ^ X8,   Z2 ^ X5 ^ Y7,   Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6,   Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Z4 ^ Y5 ^ X9,   Z3 ^ X5 ^ Y9,   Z2 ^ Y6 ^ X8,   Z1 ^ X6 ^ Y8,   Z0 ^ X7 ^ Y7,   X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ Z4 ^ X9,   Z3 ^ X5 ^ Y8,   Z2 ^ Y5 ^ X8,   Z1 ^ X6 ^ Y7,   Z0 ^ Y6 ^ X7,   X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ Z4 ^ X8,   Z3 ^ X4 ^ Y8,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ Z4 ^ X8,   Z3 ^ X4 ^ Y7,   Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ Z4 ^ X7,   X3 ^ Z3 ^ Y7,   Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ Z5 ^X10,   Z4 ^ X5 ^Y10,   Z3 ^ Y6 ^ X9,   Z2 ^ X6 ^ Y9,   Z1 ^ Y7 ^ X8,   Z0 ^ X7 ^ Y8    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ Z5 ^X10,   Z4 ^ X5 ^ Y9,   Z3 ^ Y5 ^ X9,   Z2 ^ X6 ^ Y8,   Z1 ^ Y6 ^ X8,   Z0 ^ X7 ^ Y7    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ Z5 ^ X9,   X4 ^ Z4 ^ Y9,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ Z5 ^ X9,   X4 ^ Z4 ^ Y8,   Z3 ^ Y4 ^ X8,   Z2 ^ X5 ^ Y7,   Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ Z5 ^ X8,   X3 ^ Z4 ^ Y8,   Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
};

const UINT_64 SW_64K_R_X_1xaa[][16]=
{
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Z0 ^ X3 ^ Y3,   X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z0 ^ X3 ^ Y3,   X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z0 ^ X3 ^ Y3,   X3,             Y2,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z0 ^ X3 ^ Y3,   X2,             Y2,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y3,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2,             X3,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2,             X2,             Y3,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X4,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X3,             Y4,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X3,             Y2,             X4,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X2,             Y2,             X3,             Y4,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y4,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y3,             X4,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y2,             X3,             Y4,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y2,             X2,             Y3,             X4              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X4,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X3,             Y4,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y2,             X3,             Y4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   X2,             Y3,             X4              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y4,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y3,             X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y3,             X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   Y3,             X4              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X4,             Z0 ^ X3 ^ Y3,   Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X4,             Z0 ^ X3 ^ Y3,   Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3,             Z0 ^ X3 ^ Y3,   Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3,             Z0 ^ X3 ^ Y3,   Y2,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Z0 ^ X3 ^ Y3,   Y2,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Y5,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X3,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X2,             Y3,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X5,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y4,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y2,             X4,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y2,             X3,             Y4,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Y6,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X4,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X3,             Y4,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X2,             Y3,             X4              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X6,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X4,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y4,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X3,             Y4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   Y3,             X4              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Y7,             X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y3,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   X4              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X4,             Y5,             Z0 ^ X3 ^ Y3,   X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X4,             Y4,             Z0 ^ X3 ^ Y3,   X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3,             Y4,             Z0 ^ X3 ^ Y3,   X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3,             Y2,             Z0 ^ X3 ^ Y3,   X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Y5,             X5,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             X5,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y3,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X5,             Y6,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X4,             Y5,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3,             Y2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X4,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X3,             Y4,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Y6,             X6,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             X6,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y4,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y3,             X4              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X6,             Y7,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X4,             Y6,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3,             Y4,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   X4              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Y7,             X7,             X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             X7,             X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X4,             X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y3,             X4,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y3,             X4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7    },
};

const UINT_64 SW_64K_R_X_2xaa[][16]=
{
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             S0 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             S0 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             S0 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             S0 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             S0 ^ Y6         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Z0 ^ X3 ^ Y3,   X4,             Y5,             X5,             Y6,             X6,             Y7,             S0 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z0 ^ X3 ^ Y3,   X4,             Y4,             X5,             Y5,             X6,             Y6,             S0 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5,             Y6,             S0 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z0 ^ X3 ^ Y3,   X3,             Y2,             X4,             Y4,             X5,             Y5,             S0 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z0 ^ X3 ^ Y3,   X2,             Y2,             X3,             Y4,             X4,             Y5,             S0 ^ Y6         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y5,             X5,             Y6,             X6,             Y7,             S0 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6,             Y6,             S0 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y3,             X4,             Y5,             X5,             Y6,             S0 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2,             X3,             Y4,             X5,             Y5,             S0 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2,             X2,             Y3,             X4,             Y5,             S0 ^ Y6         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6,             Y7,             S0 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X4,             Y5,             X6,             Y6,             S0 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X3,             Y4,             X5,             Y6,             S0 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X3,             Y2,             X4,             Y5,             S0 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X2,             Y2,             X3,             Y4,             S0 ^ Y6         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6,             Y7,             S0 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y4,             X6,             Y6,             S0 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y3,             X4,             Y6,             S0 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y2,             X3,             Y4,             S0 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   S0 ^ X5 ^ Y6,   Y2,             X2,             Y3,             X4              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             Y7,             S0 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X4,             Y6,             S0 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X3,             Y4,             S0 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   S0 ^ X6 ^ Y6,   Y2,             X3,             Y4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   S0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   X2,             Y3,             X4              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Z0 ^ Y6 ^ X7,   Z5 ^ X6 ^ Y7,   Y7,             S0 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Z0 ^ Y6 ^ X7,   Z5 ^ X6 ^ Y7,   Y4,             S0 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Z0 ^ Y6 ^ X7,   S0 ^ X6 ^ Y7,   Y3,             X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   S0 ^ Y6 ^ X7,   Y2 ^ X6 ^ Y7,   X3,             Y4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X8,   S0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   Y3,             X4              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             S0 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             S0 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             S0 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             S0 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             S0 ^ Y6         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X4,             Z0 ^ X3 ^ Y3,   Y5,             X5,             Y6,             X6,             Y7,             S0 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X4,             Z0 ^ X3 ^ Y3,   Y4,             X5,             Y5,             X6,             Y6,             S0 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3,             Z0 ^ X3 ^ Y3,   Y4,             X4,             Y5,             X5,             Y6,             S0 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3,             Z0 ^ X3 ^ Y3,   Y2,             X4,             Y4,             X5,             Y5,             S0 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Z0 ^ X3 ^ Y3,   Y2,             X3,             Y4,             X4,             Y5,             S0 ^ Y6         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Y5,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X5,             Y6,             X6,             Y7,             S0 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X5,             Y5,             X6,             Y6,             S0 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             S0 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X3,             Y4,             X5,             Y5,             S0 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X2,             Y3,             X4,             Y5,             S0 ^ Y6         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X5,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y6,             X6,             Y7,             S0 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y5,             X6,             Y6,             S0 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y4,             X5,             Y6,             S0 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y2,             X4,             Y5,             S0 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y2,             X3,             Y4,             S0 ^ Y6         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Y6,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6,             Y7,             S0 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6,             Y6,             S0 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X4,             Y6,             S0 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X3,             Y4,             S0 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   S0 ^ X5 ^ Y6,   X2,             Y3,             X4              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X6,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y7,             S0 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X4,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y6,             S0 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y4,             S0 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   S0 ^ X6 ^ Y6,   X3,             Y4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   S0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   Y3,             X4              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Y7,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Z0 ^ Y6 ^ X7,   Z5 ^ X6 ^ Y7,   S0 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Z0 ^ Y6 ^ X7,   Z5 ^ X6 ^ Y7,   S0 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Z0 ^ Y6 ^ X7,   S0 ^ X6 ^ Y7,   X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   S0 ^ Y6 ^ X7,   Y2 ^ X6 ^ Y7,   Y4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X8,   S0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   X4              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             S0 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             S0 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             S0 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             S0 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             S0 ^ Y6         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X4,             Y5,             Z0 ^ X3 ^ Y3,   X5,             Y6,             X6,             Y7,             S0 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X4,             Y4,             Z0 ^ X3 ^ Y3,   X5,             Y5,             X6,             Y6,             S0 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3,             Y4,             Z0 ^ X3 ^ Y3,   X4,             Y5,             X5,             Y6,             S0 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3,             Y2,             Z0 ^ X3 ^ Y3,   X4,             Y4,             X5,             Y5,             S0 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             S0 ^ Y6         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Y5,             X5,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y6,             X6,             Y7,             S0 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             X5,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y5,             X6,             Y6,             S0 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y5,             X5,             Y6,             S0 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             S0 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y3,             X4,             Y5,             S0 ^ Y6         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X5,             Y6,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X6,             Y7,             S0 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X4,             Y5,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X6,             Y6,             S0 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5,             Y6,             S0 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3,             Y2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X4,             Y5,             S0 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X3,             Y4,             S0 ^ Y6         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Y6,             X6,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y7,             S0 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             X6,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             S0 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             S0 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y4,             S0 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   S0 ^ X5 ^ Y6,   Y3,             X4              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X6,             Y7,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   S0 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X4,             Y6,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   S0 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3,             Y4,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   S0 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   S0 ^ X6 ^ Y6,   Y4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   S0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   X4              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Y7,             S0 ^ Y8,        X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Z0 ^ Y6 ^ X7,   Z5 ^ X6 ^ Y7    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             S0 ^ Y7,        X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Z0 ^ Y6 ^ X7,   Z5 ^ X6 ^ Y7    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X4,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Z0 ^ Y6 ^ X7,   S0 ^ X6 ^ Y7    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3,             Y4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   S0 ^ Y6 ^ X7,   Y2 ^ X6 ^ Y7    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y3,             X4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X8,   S0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7    },
};

const UINT_64 SW_64K_R_X_4xaa[][16]=
{
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             S0 ^ Y7,        S1 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             S0 ^ Y5,        S1 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             S0 ^ Y5,        S1 ^ Y6         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Z0 ^ X3 ^ Y3,   X4,             Y5,             X5,             Y6,             X6,             S0 ^ Y7,        S1 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z0 ^ X3 ^ Y3,   X4,             Y4,             X5,             Y5,             X6,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z0 ^ X3 ^ Y3,   X3,             Y2,             X4,             Y4,             X5,             S0 ^ Y5,        S1 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z0 ^ X3 ^ Y3,   X2,             Y2,             X3,             Y4,             X4,             S0 ^ Y5,        S1 ^ Y6         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y5,             X5,             Y6,             X6,             S0 ^ Y7,        S1 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y3,             X4,             Y5,             X5,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2,             X3,             Y4,             X5,             S0 ^ Y5,        S1 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2,             X2,             Y3,             X4,             S0 ^ Y5,        S1 ^ Y6         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6,             S0 ^ Y7,        S1 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X4,             Y5,             X6,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X3,             Y4,             X5,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y2,             X3,             Y4,             S0 ^ Y5,        S1 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S1 ^ X5 ^ Y5,   X2,             Y2,             X3,             Y4,             S0 ^ Y6         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6,             S0 ^ Y7,        S1 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y4,             X6,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y3,             X4,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   S1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y2,             X3,             Y4,             S0 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S1 ^ Y5 ^ X6,   S0 ^ X5 ^ Y6,   Y2,             X2,             Y3,             X4              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Z4 ^ X6 ^ Y6,   X6,             S0 ^ Y7,        S1 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Z4 ^ X6 ^ Y6,   Y4,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   S1 ^ X6 ^ Y6,   X3,             Y4,             S0 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   S1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   S0 ^ X6 ^ Y6,   Y2,             X3,             Y4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S1 ^ Y5 ^ X7,   S0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   X2,             Y3,             X4              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Z5 ^ Y6 ^ X7,   Z4 ^ X6 ^ Y7,   S0 ^ Y7,        S1 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   S1 ^ Y6 ^ X7,   Z4 ^ X6 ^ Y7,   Y4,             S0 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   S1 ^ Y6 ^ X7,   S0 ^ X6 ^ Y7,   Y3,             X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   S1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   S0 ^ Y6 ^ X7,   Y2 ^ X6 ^ Y7,   X3,             Y4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S1 ^ Y5 ^ X8,   S0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   Y3,             X4              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             S0 ^ Y7,        S1 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             S0 ^ Y5,        S1 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             S0 ^ Y5,        S1 ^ Y6         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X4,             Z0 ^ X3 ^ Y3,   Y5,             X5,             Y6,             X6,             S0 ^ Y7,        S1 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X4,             Z0 ^ X3 ^ Y3,   Y4,             X5,             Y5,             X6,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3,             Z0 ^ X3 ^ Y3,   Y4,             X4,             Y5,             X5,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3,             Z0 ^ X3 ^ Y3,   Y2,             X4,             Y4,             X5,             S0 ^ Y5,        S1 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Z0 ^ X3 ^ Y3,   Y2,             X3,             Y4,             X4,             S0 ^ Y5,        S1 ^ Y6         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Y5,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X5,             Y6,             X6,             S0 ^ Y7,        S1 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X5,             Y5,             X6,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X3,             Y4,             X5,             S0 ^ Y5,        S1 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X2,             Y3,             X4,             S0 ^ Y5,        S1 ^ Y6         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X5,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y6,             X6,             S0 ^ Y7,        S1 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y5,             X6,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y4,             X5,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X3,             Y4,             S0 ^ Y5,        S1 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S1 ^ X5 ^ Y5,   Y2,             X3,             Y4,             S0 ^ Y6         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Y6,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6,             S0 ^ Y7,        S1 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X4,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   S1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X3,             Y4,             S0 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S1 ^ Y5 ^ X6,   S0 ^ X5 ^ Y6,   X2,             Y3,             X4              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X6,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Z4 ^ X6 ^ Y6,   S0 ^ Y7,        S1 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Z4 ^ X6 ^ Y6,   S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   S1 ^ X6 ^ Y6,   Y4,             S0 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   S1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   S0 ^ X6 ^ Y6,   X3,             Y4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S1 ^ Y5 ^ X7,   S0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   Y3,             X4              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             S0 ^ Y7,        X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Z5 ^ Y6 ^ X7,   Z4 ^ X6 ^ Y7,   S1 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   S1 ^ Y6 ^ X7,   Z4 ^ X6 ^ Y7,   S0 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   S1 ^ Y6 ^ X7,   S0 ^ X6 ^ Y7,   X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   S1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   S0 ^ Y6 ^ X7,   Y2 ^ X6 ^ Y7,   Y4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S1 ^ Y5 ^ X8,   S0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   X4              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             S0 ^ Y7,        S1 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             S0 ^ Y5,        S1 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             S0 ^ Y5,        S1 ^ Y6         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X4,             Y5,             Z0 ^ X3 ^ Y3,   X5,             Y6,             X6,             S0 ^ Y7,        S1 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X4,             Y4,             Z0 ^ X3 ^ Y3,   X5,             Y5,             X6,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3,             Y4,             Z0 ^ X3 ^ Y3,   X4,             Y5,             X5,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3,             Y2,             Z0 ^ X3 ^ Y3,   X4,             Y4,             X5,             S0 ^ Y5,        S1 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             S0 ^ Y5,        S1 ^ Y6         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Y5,             X5,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y6,             X6,             S0 ^ Y7,        S1 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             X5,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y5,             X6,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y5,             X5,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y4,             X5,             S0 ^ Y5,        S1 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y3,             X4,             S0 ^ Y5,        S1 ^ Y6         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X5,             Y6,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X6,             S0 ^ Y7,        S1 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X4,             Y5,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X6,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y4,             S0 ^ Y5,        S1 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S1 ^ X5 ^ Y5,   X3,             Y4,             S0 ^ Y6         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Y6,             X6,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   S0 ^ Y7,        S1 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             X6,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   S1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y4,             S0 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S1 ^ Y5 ^ X6,   S0 ^ X5 ^ Y6,   Y3,             X4              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X6,             S0 ^ Y7,        X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Z4 ^ X6 ^ Y6,   S1 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             S0 ^ Y6,        X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Z4 ^ X6 ^ Y6,   S1 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3,             Y4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   S1 ^ X6 ^ Y6,   S0 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   S1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   S0 ^ X6 ^ Y6,   Y4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S1 ^ Y5 ^ X7,   S0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   X4              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             S0 ^ Y7,        S1 ^ Y8,        X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Z5 ^ Y6 ^ X7,   Z4 ^ X6 ^ Y7    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             S0 ^ Y7,        X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   S1 ^ Y6 ^ X7,   Z4 ^ X6 ^ Y7    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   S1 ^ Y6 ^ X7,   S0 ^ X6 ^ Y7    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   S1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   S0 ^ Y6 ^ X7,   Y2 ^ X6 ^ Y7    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y3,             X4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S1 ^ Y5 ^ X8,   S0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7    },
};

const UINT_64 SW_64K_R_X_8xaa[][16]=
{
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             S0 ^ Y7,        S1 ^ Y8,        S2 ^ Y9         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             S0 ^ Y6,        S1 ^ Y7,        S2 ^ Y8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             S0 ^ Y6,        S1 ^ Y7,        S2 ^ Y8         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             S0 ^ Y5,        S1 ^ Y6,        S2 ^ Y7         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             S0 ^ Y5,        S1 ^ Y6,        S2 ^ Y7         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Z0 ^ X3 ^ Y3,   X4,             Y5,             X5,             Y6,             S0 ^ Y7,        S1 ^ Y8,        S2 ^ Y9         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z0 ^ X3 ^ Y3,   X4,             Y4,             X5,             Y5,             S0 ^ Y6,        S1 ^ Y7,        S2 ^ Y8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             S0 ^ Y6,        S1 ^ Y7,        S2 ^ Y8         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z0 ^ X3 ^ Y3,   X3,             Y2,             X4,             Y4,             S0 ^ Y5,        S1 ^ Y6,        S2 ^ Y7         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z0 ^ X3 ^ Y3,   X2,             Y2,             X3,             Y4,             S0 ^ Y5,        S1 ^ Y6,        S2 ^ Y7         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y5,             X5,             Y6,             S0 ^ Y7,        S1 ^ Y8,        S2 ^ Y9         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             S0 ^ Y6,        S1 ^ Y7,        S2 ^ Y8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y3,             X4,             Y5,             S0 ^ Y6,        S1 ^ Y7,        S2 ^ Y8         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2,             X3,             Y4,             S0 ^ Y5,        S1 ^ Y6,        S2 ^ Y7         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X2,             Y2,             X3,             S0 ^ Y5,        S1 ^ Y6,        S2 ^ Y7         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5,             Y6,             S0 ^ Y7,        S1 ^ Y8,        S2 ^ Y9         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X4,             Y5,             S0 ^ Y6,        S1 ^ Y7,        S2 ^ Y8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X3,             Y4,             S0 ^ Y6,        S1 ^ Y7,        S2 ^ Y8         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S2 ^ X5 ^ Y5,   Y2,             X3,             Y4,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S2 ^ X5 ^ Y5,   X2,             Y2,             X3,             S0 ^ Y6,        S1 ^ Y7         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   Z3 ^ X5 ^ Y6,   Y6,             S0 ^ Y7,        S1 ^ Y8,        S2 ^ Y9         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   Z3 ^ X5 ^ Y6,   Y4,             S0 ^ Y6,        S1 ^ Y7,        S2 ^ Y8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   S2 ^ X5 ^ Y6,   Y3,             X4,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S2 ^ Y5 ^ X6,   S1 ^ X5 ^ Y6,   Y2,             X3,             Y4,             S0 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S2 ^ Y5 ^ X6,   S1 ^ X5 ^ Y6,   X2,             Y2,             X3,             S0 ^ Y6         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   Z4 ^ X5 ^ Y7,   Z3 ^ X6 ^ Y6,   S0 ^ Y7,        S1 ^ Y8,        S2 ^ Y9         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   Z3 ^ X5 ^ Y7,   S2 ^ X6 ^ Y6,   Y4,             S0 ^ Y7,        S1 ^ Y8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   S2 ^ X5 ^ Y7,   S1 ^ X6 ^ Y6,   X3,             Y4,             S0 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S2 ^ Y5 ^ X7,   S1 ^ X5 ^ Y7,   S0 ^ X6 ^ Y6,   Y2,             X3,             Y4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S2 ^ Y5 ^ X7,   S1 ^ X5 ^ Y7,   S0 ^ X6 ^ Y6,   X2,             Y2,             X3              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X8,   Z4 ^ X5 ^ Y8,   Z3 ^ Y6 ^ X7,   S2 ^ X6 ^ Y7,   S0 ^ Y7,        S1 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X8,   Z3 ^ X5 ^ Y8,   S2 ^ Y6 ^ X7,   S1 ^ X6 ^ Y7,   Y4,             S0 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X8,   S2 ^ X5 ^ Y8,   S1 ^ Y6 ^ X7,   S0 ^ X6 ^ Y7,   Y3,             X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S2 ^ Y5 ^ X8,   S1 ^ X5 ^ Y8,   S0 ^ Y6 ^ X7,   Y2 ^ X6 ^ Y7,   X3,             Y4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S2 ^ Y5 ^ X8,   S1 ^ X5 ^ Y8,   S0 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   Y2,             X3              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             S0 ^ Y7,        S1 ^ Y8,        S2 ^ Y9         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             S0 ^ Y6,        S1 ^ Y7,        S2 ^ Y8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             S0 ^ Y6,        S1 ^ Y7,        S2 ^ Y8         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             S0 ^ Y5,        S1 ^ Y6,        S2 ^ Y7         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             S0 ^ Y5,        S1 ^ Y6,        S2 ^ Y7         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X4,             Z0 ^ X3 ^ Y3,   Y5,             X5,             Y6,             S0 ^ Y7,        S1 ^ Y8,        S2 ^ Y9         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X4,             Z0 ^ X3 ^ Y3,   Y4,             X5,             Y5,             S0 ^ Y6,        S1 ^ Y7,        S2 ^ Y8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3,             Z0 ^ X3 ^ Y3,   Y4,             X4,             Y5,             S0 ^ Y6,        S1 ^ Y7,        S2 ^ Y8         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3,             Z0 ^ X3 ^ Y3,   Y2,             X4,             Y4,             S0 ^ Y5,        S1 ^ Y6,        S2 ^ Y7         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Z0 ^ X3 ^ Y3,   Y2,             X3,             Y4,             S0 ^ Y5,        S1 ^ Y6,        S2 ^ Y7         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Y5,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X5,             Y6,             S0 ^ Y7,        S1 ^ Y8,        S2 ^ Y9         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X5,             Y5,             S0 ^ Y6,        S1 ^ Y7,        S2 ^ Y8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X4,             Y5,             S0 ^ Y6,        S1 ^ Y7,        S2 ^ Y8         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X3,             Y4,             S0 ^ Y5,        S1 ^ Y6,        S2 ^ Y7         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2,             X3,             S0 ^ Y5,        S1 ^ Y6,        S2 ^ Y7         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X5,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y6,             S0 ^ Y7,        S1 ^ Y8,        S2 ^ Y9         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y5,             S0 ^ Y6,        S1 ^ Y7,        S2 ^ Y8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y4,             S0 ^ Y6,        S1 ^ Y7,        S2 ^ Y8         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S2 ^ X5 ^ Y5,   X3,             Y4,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S2 ^ X5 ^ Y5,   Y2,             X3,             S0 ^ Y6,        S1 ^ Y7         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Y6,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   Z3 ^ X5 ^ Y6,   S0 ^ Y7,        S1 ^ Y8,        S2 ^ Y9         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   Z3 ^ X5 ^ Y6,   S0 ^ Y6,        S1 ^ Y7,        S2 ^ Y8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   S2 ^ X5 ^ Y6,   X4,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S2 ^ Y5 ^ X6,   S1 ^ X5 ^ Y6,   X3,             Y4,             S0 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S2 ^ Y5 ^ X6,   S1 ^ X5 ^ Y6,   Y2,             X3,             S0 ^ Y6         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             S0 ^ Y7,        Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   Z4 ^ X5 ^ Y7,   Z3 ^ X6 ^ Y6,   S1 ^ Y8,        S2 ^ Y9         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   Z3 ^ X5 ^ Y7,   S2 ^ X6 ^ Y6,   S0 ^ Y7,        S1 ^ Y8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   S2 ^ X5 ^ Y7,   S1 ^ X6 ^ Y6,   Y4,             S0 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S2 ^ Y5 ^ X7,   S1 ^ X5 ^ Y7,   S0 ^ X6 ^ Y6,   X3,             Y4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S2 ^ Y5 ^ X7,   S1 ^ X5 ^ Y7,   S0 ^ X6 ^ Y6,   Y2,             X3              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             S0 ^ Y7,        Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X8,   Z4 ^ X5 ^ Y8,   Z3 ^ Y6 ^ X7,   S2 ^ X6 ^ Y7,   S1 ^ Y8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X8,   Z3 ^ X5 ^ Y8,   S2 ^ Y6 ^ X7,   S1 ^ X6 ^ Y7,   S0 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X8,   S2 ^ X5 ^ Y8,   S1 ^ Y6 ^ X7,   S0 ^ X6 ^ Y7,   X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S2 ^ Y5 ^ X8,   S1 ^ X5 ^ Y8,   S0 ^ Y6 ^ X7,   Y2 ^ X6 ^ Y7,   Y4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S2 ^ Y5 ^ X8,   S1 ^ X5 ^ Y8,   S0 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   X3              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             S0 ^ Y7,        S1 ^ Y8,        S2 ^ Y9         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             S0 ^ Y6,        S1 ^ Y7,        S2 ^ Y8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             S0 ^ Y6,        S1 ^ Y7,        S2 ^ Y8         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             S0 ^ Y5,        S1 ^ Y6,        S2 ^ Y7         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             S0 ^ Y5,        S1 ^ Y6,        S2 ^ Y7         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X4,             Y5,             Z0 ^ X3 ^ Y3,   X5,             Y6,             S0 ^ Y7,        S1 ^ Y8,        S2 ^ Y9         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X4,             Y4,             Z0 ^ X3 ^ Y3,   X5,             Y5,             S0 ^ Y6,        S1 ^ Y7,        S2 ^ Y8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3,             Y4,             Z0 ^ X3 ^ Y3,   X4,             Y5,             S0 ^ Y6,        S1 ^ Y7,        S2 ^ Y8         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3,             Y2,             Z0 ^ X3 ^ Y3,   X4,             Y4,             S0 ^ Y5,        S1 ^ Y6,        S2 ^ Y7         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             S0 ^ Y5,        S1 ^ Y6,        S2 ^ Y7         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Y5,             X5,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y6,             S0 ^ Y7,        S1 ^ Y8,        S2 ^ Y9         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             X5,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y5,             S0 ^ Y6,        S1 ^ Y7,        S2 ^ Y8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y5,             S0 ^ Y6,        S1 ^ Y7,        S2 ^ Y8         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y4,             S0 ^ Y5,        S1 ^ Y6,        S2 ^ Y7         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X3,             S0 ^ Y5,        S1 ^ Y6,        S2 ^ Y7         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             X5,             Y6,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   S0 ^ Y7,        S1 ^ Y8,        S2 ^ Y9         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             X4,             Y5,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   S0 ^ Y6,        S1 ^ Y7,        S2 ^ Y8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   S0 ^ Y6,        S1 ^ Y7,        S2 ^ Y8         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S2 ^ X5 ^ Y5,   Y4,             S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S2 ^ X5 ^ Y5,   X3,             S0 ^ Y6,        S1 ^ Y7         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             Y6,             S0 ^ Y7,        Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   Z3 ^ X5 ^ Y6,   S1 ^ Y8,        S2 ^ Y9         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             S0 ^ Y6,        Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   Z3 ^ X5 ^ Y6,   S1 ^ Y7,        S2 ^ Y8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   S2 ^ X5 ^ Y6,   S0 ^ Y6,        S1 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S2 ^ Y5 ^ X6,   S1 ^ X5 ^ Y6,   Y4,             S0 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S2 ^ Y5 ^ X6,   S1 ^ X5 ^ Y6,   X3,             S0 ^ Y6         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             S0 ^ Y7,        S1 ^ Y8,        Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   Z4 ^ X5 ^ Y7,   Z3 ^ X6 ^ Y6,   S2 ^ Y9         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             S0 ^ Y7,        Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   Z3 ^ X5 ^ Y7,   S2 ^ X6 ^ Y6,   S1 ^ Y8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   S2 ^ X5 ^ Y7,   S1 ^ X6 ^ Y6,   S0 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S2 ^ Y5 ^ X7,   S1 ^ X5 ^ Y7,   S0 ^ X6 ^ Y6,   Y4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S2 ^ Y5 ^ X7,   S1 ^ X5 ^ Y7,   S0 ^ X6 ^ Y6,   X3              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y4,             S0 ^ Y7,        S1 ^ Y8,        Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X8,   Z4 ^ X5 ^ Y8,   Z3 ^ Y6 ^ X7,   S2 ^ X6 ^ Y7    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4,             S0 ^ Y7,        Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X8,   Z3 ^ X5 ^ Y8,   S2 ^ Y6 ^ X7,   S1 ^ X6 ^ Y7    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X8,   S2 ^ X5 ^ Y8,   S1 ^ Y6 ^ X7,   S0 ^ X6 ^ Y7    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3,             Y4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S2 ^ Y5 ^ X8,   S1 ^ X5 ^ Y8,   S0 ^ Y6 ^ X7,   Y2 ^ X6 ^ Y7    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   S2 ^ Y5 ^ X8,   S1 ^ X5 ^ Y8,   S0 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7    },
};

const UINT_64 SW_64K_Z_X_1xaa[][16]=
{
    {X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             Z0 ^ X3 ^ Y3,   X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Z0 ^ X3 ^ Y3,   X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Z0 ^ X3 ^ Y3,   X3,             Y2,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z0 ^ X3 ^ Y3,   X2,             Y2,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y3,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2,             X3,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2,             X2,             Y3,             X4,             Y5,             X5              },
    {X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X4,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X3,             Y4,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X3,             Y2,             X4,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X2,             Y2,             X3,             Y4,             X5              },
    {X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y4,             X6,             Y6,             X7              },
    {0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y3,             X4,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y2,             X3,             Y4,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y2,             X2,             Y3,             X4              },
    {X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X4,             Y6,             X7              },
    {0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X3,             Y4,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y2,             X3,             Y4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   X2,             Y3,             X4              },
    {X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y7,             X7              },
    {0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y4,             X7              },
    {0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y3,             X4              },
    {0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y3,             X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   Y3,             X4              },
    {X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             X4,             Z0 ^ X3 ^ Y3,   Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             X4,             Z0 ^ X3 ^ Y3,   Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Z0 ^ X3 ^ Y3,   Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             X3,             Z0 ^ X3 ^ Y3,   Y2,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Z0 ^ X3 ^ Y3,   Y2,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             Y5,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X3,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X2,             Y3,             X4,             Y5,             X5              },
    {X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             X5,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             X4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y4,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y2,             X4,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y2,             X3,             Y4,             X5              },
    {X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             Y6,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6,             Y6,             X7              },
    {0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X4,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X3,             Y4,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X2,             Y3,             X4              },
    {X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             X6,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y7,             X7              },
    {0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             X4,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y6,             X7              },
    {0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y4,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X3,             Y4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   Y3,             X4              },
    {X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             Y7,             X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   X7              },
    {0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   X7              },
    {0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3,             X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   X4              },
    {0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y3,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   X4              },
    {X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             X4,             Y5,             Z0 ^ X3 ^ Y3,   X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             X4,             Y4,             Z0 ^ X3 ^ Y3,   X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             Z0 ^ X3 ^ Y3,   X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             X3,             Y2,             Z0 ^ X3 ^ Y3,   X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5              },
    {X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             Y5,             X5,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             X5,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3,             X4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y3,             X4,             Y5,             X5              },
    {X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             X5,             Y6,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X6,             Y7,             X7              },
    {0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             X4,             Y5,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X6,             Y6,             X7              },
    {0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             X3,             Y2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X4,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X3,             Y4,             X5              },
    {X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             Y6,             X6,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y7,             X7              },
    {0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             X6,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X7              },
    {0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3,             X4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y4,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y3,             X4              },
    {X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             X6,             Y7,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X7              },
    {0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             X4,             Y6,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X7              },
    {0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6              },
    {0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   X4              },
    {X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             Y7,             X7,             X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             X7,             X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3,             X4,             X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y3,             X4,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y3,             X4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7    },
};

const UINT_64 SW_64K_Z_X_2xaa[][16]=
{
    {S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7              },
    {0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             Y2 ^ Y7         },
    {0,              0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y2 ^ Y6         },
    {0,              0,              0,              0,              S0,             X0,             Y0,             X1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             Y1 ^ Y6         },
    {S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Z0 ^ X3 ^ Y3,   Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7              },
    {0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5,             Y6,             Y2 ^ Y7         },
    {0,              0,              0,              S0,             X0,             Y0,             X1,             Y1,             Z0 ^ X3 ^ Y3,   X2,             X3,             Y4,             X4,             Y5,             X5,             Y2 ^ Y6         },
    {0,              0,              0,              0,              S0,             X0,             Y0,             X1,             Z0 ^ X3 ^ Y3,   X2,             Y2,             X3,             Y4,             X4,             Y5,             Y1 ^ Y6         },
    {S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y4,             Y5,             X5,             Y6,             X6,             Y7              },
    {0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y3,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y3,             X4,             Y5,             X5,             Y6,             Y2 ^ Y7         },
    {0,              0,              0,              S0,             X0,             Y0,             X1,             Y1,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X2,             Y3,             X4,             Y5,             X5,             Y2 ^ Y6         },
    {0,              0,              0,              0,              S0,             X0,             Y0,             X1,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2,             X2,             Y3,             X4,             Y5,             Y1 ^ Y6         },
    {S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y4,             X5,             Y6,             X6,             Y7              },
    {0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X3,             Y4,             X5,             Y6,             X6              },
    {0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X3,             Y4,             X5,             Y6,             Y2 ^ Y7         },
    {0,              0,              0,              S0,             X0,             Y0,             X1,             Y1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X2,             X3,             Y4,             X5,             Y2 ^ Y6         },
    {0,              0,              0,              0,              S0,             X0,             Y0,             X1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X2,             Y2,             X3,             Y4,             Y1 ^ Y6         },
    {S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y4,             Y6,             X6,             Y7              },
    {0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y3,             X4,             Y6,             X6              },
    {0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y3,             X4,             Y6,             Y2 ^ Y7         },
    {0,              0,              0,              S0,             X0,             Y0,             X1,             Y1,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X2,             Y3,             X4,             Y2 ^ Y6         },
    {0,              0,              0,              0,              S0,             X0,             Y0,             X1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   Y2 ^ X5 ^ Y6,   Y1,             X3,             Y4,             X2 ^ Y6         },
    {S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y4,             X6,             Y7              },
    {0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X3,             Y4,             X6              },
    {0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X3,             Y4,             Y2 ^ Y7         },
    {0,              0,              0,              S0,             X0,             Y0,             X1,             Y1,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   Y3,             X4,             X2 ^ Y7         },
    {0,              0,              0,              0,              S0,             X0,             Y0,             X1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   Y2 ^ X5 ^ Y7,   X2 ^ X6 ^ Y6,   X3,             Y4,             Y1 ^ Y7         },
    {S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Z0 ^ Y6 ^ X7,   Z5 ^ X6 ^ Y7,   Y4,             Y7              },
    {0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Z0 ^ Y6 ^ X7,   Z5 ^ X6 ^ Y7,   Y3,             X4              },
    {0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Z0 ^ Y6 ^ X7,   Y2 ^ X6 ^ Y7,   X3,             Y4              },
    {0,              0,              0,              S0,             X0,             Y0,             X1,             Y1,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   Y3,             X4              },
    {0,              0,              0,              0,              S0,             X0,             Y0,             X1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X8,   Y2 ^ X5 ^ Y8,   X2 ^ Y6 ^ X7,   Y1 ^ X6 ^ Y7,   X3,             Y4              },
    {S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7              },
    {0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             Y2 ^ Y7         },
    {0,              0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y2 ^ Y6         },
    {0,              0,              0,              0,              S0,             X0,             Y0,             X1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             Y1 ^ Y6         },
    {S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             Z0 ^ X3 ^ Y3,   X4,             Y5,             X5,             Y6,             X6,             Y7              },
    {0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Z0 ^ X3 ^ Y3,   Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             X3,             Z0 ^ X3 ^ Y3,   Y4,             X4,             Y5,             X5,             Y6,             Y2 ^ Y7         },
    {0,              0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5,             Y2 ^ Y6         },
    {0,              0,              0,              0,              S0,             X0,             Y0,             X1,             X2,             Z0 ^ X3 ^ Y3,   Y2,             X3,             Y4,             X4,             Y5,             Y1 ^ Y6         },
    {S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y5,             X5,             Y6,             X6,             Y7              },
    {0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             Y2 ^ Y7         },
    {0,              0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y3,             X4,             Y5,             X5,             Y2 ^ Y6         },
    {0,              0,              0,              0,              S0,             X0,             Y0,             X1,             Y2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X2,             Y3,             X4,             Y5,             Y1 ^ Y6         },
    {S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6,             Y7              },
    {0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y4,             X5,             Y6,             X6              },
    {0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y4,             X5,             Y6,             Y2 ^ Y7         },
    {0,              0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X3,             Y4,             X5,             Y2 ^ Y6         },
    {0,              0,              0,              0,              S0,             X0,             Y0,             X1,             X2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y2,             X3,             Y4,             Y1 ^ Y6         },
    {S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6,             Y7              },
    {0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X4,             Y6,             X6              },
    {0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X4,             Y6,             Y2 ^ Y7         },
    {0,              0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y3,             X4,             Y2 ^ Y6         },
    {0,              0,              0,              0,              S0,             X0,             Y0,             X1,             Y1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   Y2 ^ X5 ^ Y6,   X3,             Y4,             X2 ^ Y6         },
    {S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             Y7              },
    {0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y4,             X6              },
    {0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             X3,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y4,             Y2 ^ Y7         },
    {0,              0,              0,              S0,             X0,             Y0,             X1,             Y1,             Y3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   X4,             X2 ^ Y7         },
    {0,              0,              0,              0,              S0,             X0,             Y0,             X1,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   Y2 ^ X5 ^ Y7,   X2 ^ X6 ^ Y6,   Y4,             Y1 ^ Y7         },
    {S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Z0 ^ Y6 ^ X7,   Z5 ^ X6 ^ Y7,   Y7              },
    {0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Z0 ^ Y6 ^ X7,   Z5 ^ X6 ^ Y7,   X4              },
    {0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             X3,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Z0 ^ Y6 ^ X7,   Y2 ^ X6 ^ Y7,   Y4              },
    {0,              0,              0,              S0,             X0,             Y0,             X1,             Y1,             Y3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   X4              },
    {0,              0,              0,              0,              S0,             X0,             Y0,             X1,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X8,   Y2 ^ X5 ^ Y8,   X2 ^ Y6 ^ X7,   Y1 ^ X6 ^ Y7,   Y4              },
    {S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7              },
    {0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             Y2 ^ Y7         },
    {0,              0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y2 ^ Y6         },
    {0,              0,              0,              0,              S0,             X0,             Y0,             X1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             Y1 ^ Y6         },
    {S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             X4,             Z0 ^ X3 ^ Y3,   Y5,             X5,             Y6,             X6,             Y7              },
    {0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             Z0 ^ X3 ^ Y3,   X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             X3,             Y4,             Z0 ^ X3 ^ Y3,   X4,             Y5,             X5,             Y6,             Y2 ^ Y7         },
    {0,              0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             X3,             Z0 ^ X3 ^ Y3,   Y4,             X4,             Y5,             X5,             Y2 ^ Y6         },
    {0,              0,              0,              0,              S0,             X0,             Y0,             X1,             X2,             Y2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             Y1 ^ Y6         },
    {S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             Y5,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X5,             Y6,             X6,             Y7              },
    {0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3,             X4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y5,             X5,             Y6,             X6              },
    {0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y3,             X4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y5,             X5,             Y6,             Y2 ^ Y7         },
    {0,              0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y2 ^ Y6         },
    {0,              0,              0,              0,              S0,             X0,             Y0,             X1,             Y2,             X2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y3,             X4,             Y5,             Y1 ^ Y6         },
    {S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             X5,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y6,             X6,             Y7              },
    {0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5,             Y6,             Y2 ^ Y7         },
    {0,              0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y4,             X5,             Y2 ^ Y6         },
    {0,              0,              0,              0,              S0,             X0,             Y0,             X1,             X2,             Y2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X3,             Y4,             Y1 ^ Y6         },
    {S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             Y6,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6,             Y7              },
    {0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3,             X4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6              },
    {0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y3,             X4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             Y2 ^ Y7         },
    {0,              0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X4,             Y2 ^ Y6         },
    {0,              0,              0,              0,              S0,             X0,             Y0,             X1,             Y1,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   Y2 ^ X5 ^ Y6,   Y4,             X2 ^ Y6         },
    {S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             X6,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y7              },
    {0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6              },
    {0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             X3,             Y4,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y2 ^ Y7         },
    {0,              0,              0,              S0,             X0,             Y0,             X1,             Y1,             Y3,             X4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   X2 ^ Y7         },
    {0,              0,              0,              0,              S0,             X0,             Y0,             X1,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   Y2 ^ X5 ^ Y7,   X2 ^ X6 ^ Y6,   Y1 ^ Y7         },
    {S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             Y7,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Z0 ^ Y6 ^ X7,   Z5 ^ X6 ^ Y7    },
    {0,              S0,             X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3,             X4,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Z0 ^ Y6 ^ X7,   Z5 ^ X6 ^ Y7    },
    {0,              0,              S0,             X0,             Y0,             X1,             Y1,             X2,             X3,             Y4,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Z0 ^ Y6 ^ X7,   Y2 ^ X6 ^ Y7    },
    {0,              0,              0,              S0,             X0,             Y0,             X1,             Y1,             Y3,             X4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7    },
    {0,              0,              0,              0,              S0,             X0,             Y0,             X1,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X8,   Y2 ^ X5 ^ Y8,   X2 ^ Y6 ^ X7,   Y1 ^ X6 ^ Y7    },
};

const UINT_64 SW_64K_Z_X_4xaa[][16]=
{
    {S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y2 ^ Y6         },
    {0,              0,              S0,             S1,             X0,             Y0,             X1,             Y1,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y2 ^ Y6,        X2 ^ Y7         },
    {0,              0,              0,              S0,             S1,             X0,             Y0,             X1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y1 ^ Y5,        X2 ^ Y6         },
    {0,              0,              0,              0,              S0,             S1,             X0,             Y0,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y1 ^ Y5,        X1 ^ Y6         },
    {S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             Y2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             Z0 ^ X3 ^ Y3,   X3,             X4,             Y4,             X5,             Y5,             X6,             Y2 ^ Y6         },
    {0,              0,              S0,             S1,             X0,             Y0,             X1,             Y1,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5,             X2 ^ Y6,        Y2 ^ Y7         },
    {0,              0,              0,              S0,             S1,             X0,             Y0,             X1,             Z0 ^ X3 ^ Y3,   X3,             Y2,             X4,             Y4,             X5,             X2 ^ Y5,        Y1 ^ Y6         },
    {0,              0,              0,              0,              S0,             S1,             X0,             Y0,             Z0 ^ X3 ^ Y3,   X2,             Y2,             X3,             Y4,             X4,             X1 ^ Y5,        Y1 ^ Y6         },
    {S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             Y2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y3,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X3,             Y4,             X5,             Y5,             X6,             Y2 ^ Y6         },
    {0,              0,              S0,             S1,             X0,             Y0,             X1,             Y1,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y3,             X4,             Y5,             X5,             Y2 ^ Y6,        X2 ^ Y7         },
    {0,              0,              0,              S0,             S1,             X0,             Y0,             X1,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2,             X3,             Y4,             X5,             Y1 ^ Y5,        X2 ^ Y6         },
    {0,              0,              0,              0,              S0,             S1,             X0,             Y0,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2,             X2,             Y3,             X4,             Y1 ^ Y5,        X1 ^ Y6         },
    {S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             Y2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X3,             Y4,             X5,             Y6,             X6              },
    {0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X3,             X4,             Y5,             X6,             Y2 ^ Y6         },
    {0,              0,              S0,             S1,             X0,             Y0,             X1,             Y1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X3,             Y4,             X5,             X2 ^ Y6,        Y2 ^ Y7         },
    {0,              0,              0,              S0,             S1,             X0,             Y0,             X1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y2,             X3,             Y4,             Y1 ^ Y5,        X2 ^ Y6         },
    {0,              0,              0,              0,              S0,             S1,             X0,             Y0,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2 ^ X5 ^ Y5,   X2,             Y3,             X4,             X1 ^ Y6,        Y1 ^ Y7         },
    {S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y3,             X4,             Y6,             X6              },
    {0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X3,             Y4,             X6,             Y2 ^ Y6         },
    {0,              0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y3,             X4,             Y2 ^ Y6,        X2 ^ Y7         },
    {0,              0,              0,              S0,             S1,             X0,             Y0,             X1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Y2 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y3,             X4,             Y1 ^ Y6,        X2 ^ Y7         },
    {0,              0,              0,              0,              S0,             S1,             X0,             Y0,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2 ^ Y5 ^ X6,   X1 ^ X5 ^ Y6,   Y3,             X4,             Y1 ^ Y6,        X2 ^ Y7         },
    {S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Z4 ^ X6 ^ Y6,   X3,             Y4,             X6              },
    {0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Z4 ^ X6 ^ Y6,   X3,             Y4,             Y2 ^ Y6         },
    {0,              0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   Y3,             X4,             X2 ^ Y7         },
    {0,              0,              0,              S0,             S1,             X0,             Y0,             X1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Y2 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Y1 ^ X6 ^ Y6,   Y3,             X4,             X2 ^ Y7         },
    {0,              0,              0,              0,              S0,             S1,             X0,             Y0,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2 ^ Y5 ^ X7,   X1 ^ X5 ^ Y7,   Y1 ^ X6 ^ Y6,   Y3,             X4,             X2 ^ Y7         },
    {S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Z5 ^ Y6 ^ X7,   Z4 ^ X6 ^ Y7,   Y3,             X4              },
    {0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   Z4 ^ X6 ^ Y7,   Y3,             X4              },
    {0,              0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   Y3,             X4              },
    {0,              0,              0,              S0,             S1,             X0,             Y0,             X1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Y2 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y1 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   Y3,             X4              },
    {0,              0,              0,              0,              S0,             S1,             X0,             Y0,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2 ^ Y5 ^ X8,   X1 ^ X5 ^ Y8,   Y1 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   Y3,             X4              },
    {S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y2 ^ Y6         },
    {0,              0,              S0,             S1,             X0,             Y0,             X1,             Y1,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y2 ^ Y6,        X2 ^ Y7         },
    {0,              0,              0,              S0,             S1,             X0,             Y0,             X1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y1 ^ Y5,        X2 ^ Y6         },
    {0,              0,              0,              0,              S0,             S1,             X0,             Y0,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y1 ^ Y5,        X1 ^ Y6         },
    {S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Z0 ^ X3 ^ Y3,   Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             X3,             Z0 ^ X3 ^ Y3,   X4,             Y4,             X5,             Y5,             X6,             Y2 ^ Y6         },
    {0,              0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X3,             Z0 ^ X3 ^ Y3,   Y4,             X4,             Y5,             X5,             X2 ^ Y6,        Y2 ^ Y7         },
    {0,              0,              0,              S0,             S1,             X0,             Y0,             X1,             X3,             Z0 ^ X3 ^ Y3,   Y2,             X4,             Y4,             X5,             X2 ^ Y5,        Y1 ^ Y6         },
    {0,              0,              0,              0,              S0,             S1,             X0,             Y0,             X2,             Z0 ^ X3 ^ Y3,   Y2,             X3,             Y4,             X4,             X1 ^ Y5,        Y1 ^ Y6         },
    {S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             X6              },
    {0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             X3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6,             Y2 ^ Y6         },
    {0,              0,              S0,             S1,             X0,             Y0,             X1,             Y1,             Y3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y2 ^ Y6,        X2 ^ Y7         },
    {0,              0,              0,              S0,             S1,             X0,             Y0,             X1,             Y2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X3,             Y4,             X5,             Y1 ^ Y5,        X2 ^ Y6         },
    {0,              0,              0,              0,              S0,             S1,             X0,             Y0,             Y2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X2,             Y3,             X4,             Y1 ^ Y5,        X1 ^ Y6         },
    {S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y4,             X5,             Y6,             X6              },
    {0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X4,             Y5,             X6,             Y2 ^ Y6         },
    {0,              0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y4,             X5,             X2 ^ Y6,        Y2 ^ Y7         },
    {0,              0,              0,              S0,             S1,             X0,             Y0,             X1,             Y2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X3,             Y4,             Y1 ^ Y5,        X2 ^ Y6         },
    {0,              0,              0,              0,              S0,             S1,             X0,             Y0,             X2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2 ^ X5 ^ Y5,   Y3,             X4,             X1 ^ Y6,        Y1 ^ Y7         },
    {S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X4,             Y6,             X6              },
    {0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             X3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y4,             X6,             Y2 ^ Y6         },
    {0,              0,              S0,             S1,             X0,             Y0,             X1,             Y1,             Y3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X4,             Y2 ^ Y6,        X2 ^ Y7         },
    {0,              0,              0,              S0,             S1,             X0,             Y0,             X1,             Y3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Y2 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X4,             Y1 ^ Y6,        X2 ^ Y7         },
    {0,              0,              0,              0,              S0,             S1,             X0,             Y0,             Y3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2 ^ Y5 ^ X6,   X1 ^ X5 ^ Y6,   X4,             Y1 ^ Y6,        X2 ^ Y7         },
    {S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Z4 ^ X6 ^ Y6,   Y4,             X6              },
    {0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             X3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Z4 ^ X6 ^ Y6,   Y4,             Y2 ^ Y6         },
    {0,              0,              S0,             S1,             X0,             Y0,             X1,             Y1,             Y3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   X4,             X2 ^ Y7         },
    {0,              0,              0,              S0,             S1,             X0,             Y0,             X1,             Y3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Y2 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Y1 ^ X6 ^ Y6,   X4,             X2 ^ Y7         },
    {0,              0,              0,              0,              S0,             S1,             X0,             Y0,             Y3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2 ^ Y5 ^ X7,   X1 ^ X5 ^ Y7,   Y1 ^ X6 ^ Y6,   X4,             X2 ^ Y7         },
    {S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Z5 ^ Y6 ^ X7,   Z4 ^ X6 ^ Y7,   X4              },
    {0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             Y3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   Z4 ^ X6 ^ Y7,   X4              },
    {0,              0,              S0,             S1,             X0,             Y0,             X1,             Y1,             Y3,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   X4              },
    {0,              0,              0,              S0,             S1,             X0,             Y0,             X1,             Y3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Y2 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y1 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   X4              },
    {0,              0,              0,              0,              S0,             S1,             X0,             Y0,             Y3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2 ^ Y5 ^ X8,   X1 ^ X5 ^ Y8,   Y1 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   X4              },
    {S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y2 ^ Y6         },
    {0,              0,              S0,             S1,             X0,             Y0,             X1,             Y1,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y2 ^ Y6,        X2 ^ Y7         },
    {0,              0,              0,              S0,             S1,             X0,             Y0,             X1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y1 ^ Y5,        X2 ^ Y6         },
    {0,              0,              0,              0,              S0,             S1,             X0,             Y0,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y1 ^ Y5,        X1 ^ Y6         },
    {S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             Z0 ^ X3 ^ Y3,   X4,             Y5,             X5,             Y6,             X6              },
    {0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             X3,             X4,             Z0 ^ X3 ^ Y3,   Y4,             X5,             Y5,             X6,             Y2 ^ Y6         },
    {0,              0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X3,             Y4,             Z0 ^ X3 ^ Y3,   X4,             Y5,             X5,             X2 ^ Y6,        Y2 ^ Y7         },
    {0,              0,              0,              S0,             S1,             X0,             Y0,             X1,             X3,             Y2,             Z0 ^ X3 ^ Y3,   X4,             Y4,             X5,             X2 ^ Y5,        Y1 ^ Y6         },
    {0,              0,              0,              0,              S0,             S1,             X0,             Y0,             X2,             Y2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             X1 ^ Y5,        Y1 ^ Y6         },
    {S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3,             X4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y5,             X5,             Y6,             X6              },
    {0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             X3,             Y4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X5,             Y5,             X6,             Y2 ^ Y6         },
    {0,              0,              S0,             S1,             X0,             Y0,             X1,             Y1,             Y3,             X4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y5,             X5,             Y2 ^ Y6,        X2 ^ Y7         },
    {0,              0,              0,              S0,             S1,             X0,             Y0,             X1,             Y2,             X3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y1 ^ Y5,        X2 ^ Y6         },
    {0,              0,              0,              0,              S0,             S1,             X0,             Y0,             Y2,             X2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y3,             X4,             Y1 ^ Y5,        X1 ^ Y6         },
    {S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             X3,             X4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y5,             X6,             Y2 ^ Y6         },
    {0,              0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5,             X2 ^ Y6,        Y2 ^ Y7         },
    {0,              0,              0,              S0,             S1,             X0,             Y0,             X1,             Y2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y4,             Y1 ^ Y5,        X2 ^ Y6         },
    {0,              0,              0,              0,              S0,             S1,             X0,             Y0,             X2,             Y3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2 ^ X5 ^ Y5,   X4,             X1 ^ Y6,        Y1 ^ Y7         },
    {S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3,             X4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6              },
    {0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             X3,             Y4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6,             Y2 ^ Y6         },
    {0,              0,              S0,             S1,             X0,             Y0,             X1,             Y1,             Y3,             X4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y2 ^ Y6,        X2 ^ Y7         },
    {0,              0,              0,              S0,             S1,             X0,             Y0,             X1,             Y3,             X4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Y2 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y1 ^ Y6,        X2 ^ Y7         },
    {0,              0,              0,              0,              S0,             S1,             X0,             Y0,             Y3,             X4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2 ^ Y5 ^ X6,   X1 ^ X5 ^ Y6,   Y1 ^ Y6,        X2 ^ Y7         },
    {S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             Y2,             X3,             Y4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Z4 ^ X6 ^ Y6,   X6              },
    {0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             X3,             Y4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Z4 ^ X6 ^ Y6,   Y2 ^ Y6         },
    {0,              0,              S0,             S1,             X0,             Y0,             X1,             Y1,             Y3,             X4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   X2 ^ Y7         },
    {0,              0,              0,              S0,             S1,             X0,             Y0,             X1,             Y3,             X4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Y2 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Y1 ^ X6 ^ Y6,   X2 ^ Y7         },
    {0,              0,              0,              0,              S0,             S1,             X0,             Y0,             Y3,             X4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2 ^ Y5 ^ X7,   X1 ^ X5 ^ Y7,   Y1 ^ X6 ^ Y6,   X2 ^ Y7         },
    {S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3,             X4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Z5 ^ Y6 ^ X7,   Z4 ^ X6 ^ Y7    },
    {0,              S0,             S1,             X0,             Y0,             X1,             Y1,             X2,             Y3,             X4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   Z4 ^ X6 ^ Y7    },
    {0,              0,              S0,             S1,             X0,             Y0,             X1,             Y1,             Y3,             X4,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7    },
    {0,              0,              0,              S0,             S1,             X0,             Y0,             X1,             Y3,             X4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Y2 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y1 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7    },
    {0,              0,              0,              0,              S0,             S1,             X0,             Y0,             Y3,             X4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2 ^ Y5 ^ X8,   X1 ^ X5 ^ Y8,   Y1 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7    },
};

const UINT_64 SW_64K_Z_X_8xaa[][16]=
{
    {S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             Y2 ^ Y7         },
    {0,              S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y2 ^ Y6,        X2 ^ Y7         },
    {0,              0,              S0,             S1,             S2,             X0,             Y0,             X1,             Y3,             X3,             Y4,             X4,             Y5,             Y1 ^ Y6,        Y2 ^ Y7,        X2 ^ Y8         },
    {0,              0,              0,              S0,             S1,             S2,             X0,             Y0,             X2,             Y3,             X3,             Y4,             X4,             Y1 ^ Y5,        X1 ^ Y6,        Y2 ^ Y7         },
    {0,              0,              0,              0,              S0,             S1,             S2,             X0,             Y2,             X2,             Y3,             X3,             Y4,             Y0 ^ Y5,        Y1 ^ Y6,        X1 ^ Y7         },
    {S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             X2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5,             Y6,             Y2 ^ Y7         },
    {0,              S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5,             X2 ^ Y6,        Y2 ^ Y7         },
    {0,              0,              S0,             S1,             S2,             X0,             Y0,             X1,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             Y1 ^ Y6,        X2 ^ Y7,        Y2 ^ Y8         },
    {0,              0,              0,              S0,             S1,             S2,             X0,             Y0,             Z0 ^ X3 ^ Y3,   Y2,             X3,             Y4,             X4,             X1 ^ Y5,        Y1 ^ Y6,        X2 ^ Y7         },
    {0,              0,              0,              0,              S0,             S1,             S2,             X0,             Z0 ^ X3 ^ Y3,   X2,             Y2,             X3,             Y4,             Y0 ^ Y5,        X1 ^ Y6,        Y1 ^ Y7         },
    {S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             X2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y3,             X4,             Y5,             X5,             Y6,             Y2 ^ Y7         },
    {0,              S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y3,             X4,             Y5,             X5,             Y2 ^ Y6,        X2 ^ Y7         },
    {0,              0,              S0,             S1,             S2,             X0,             Y0,             X1,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y3,             X4,             Y5,             Y1 ^ Y6,        Y2 ^ Y7,        X2 ^ Y8         },
    {0,              0,              0,              S0,             S1,             S2,             X0,             Y0,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X2,             Y3,             X4,             Y1 ^ Y5,        X1 ^ Y6,        Y2 ^ Y7         },
    {0,              0,              0,              0,              S0,             S1,             S2,             X0,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X2,             Y2,             X3,             Y0 ^ Y5,        X1 ^ Y6,        Y1 ^ Y7         },
    {S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             X2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X3,             Y4,             X5,             Y6,             Y2 ^ Y7         },
    {0,              S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X3,             Y4,             X5,             X2 ^ Y6,        Y2 ^ Y7         },
    {0,              0,              S0,             S1,             S2,             X0,             Y0,             X1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X3,             Y4,             Y1 ^ Y6,        X2 ^ Y7,        Y2 ^ Y8         },
    {0,              0,              0,              S0,             S1,             S2,             X0,             Y0,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2 ^ X5 ^ Y5,   Y3,             X4,             X1 ^ Y6,        Y1 ^ Y7,        X2 ^ Y8         },
    {0,              0,              0,              0,              S0,             S1,             S2,             X0,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y0 ^ X5 ^ Y5,   Y2,             X3,             X1 ^ Y6,        Y1 ^ Y7,        X2 ^ Y8         },
    {S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             X2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   Z3 ^ X5 ^ Y6,   Y3,             X4,             Y6,             Y2 ^ Y7         },
    {0,              S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   Z3 ^ X5 ^ Y6,   Y3,             X4,             Y2 ^ Y6,        X2 ^ Y7         },
    {0,              0,              S0,             S1,             S2,             X0,             Y0,             X1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   Y2 ^ X5 ^ Y6,   X3,             Y4,             X2 ^ Y6,        Y1 ^ Y7         },
    {0,              0,              0,              S0,             S1,             S2,             X0,             Y0,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2 ^ Y5 ^ X6,   X1 ^ X5 ^ Y6,   Y3,             X4,             Y1 ^ Y6,        X2 ^ Y7         },
    {0,              0,              0,              0,              S0,             S1,             S2,             X0,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y0 ^ Y5 ^ X6,   X1 ^ X5 ^ Y6,   X3,             Y1 ^ Y6,        X2 ^ Y7,        Y2 ^ Y8         },
    {S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             X2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   Z4 ^ X5 ^ Y7,   Z3 ^ X6 ^ Y6,   X3,             Y4,             Y2 ^ Y7         },
    {0,              S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   Z3 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   Y3,             X4,             X2 ^ Y7         },
    {0,              0,              S0,             S1,             S2,             X0,             Y0,             X1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   Y2 ^ X5 ^ Y7,   X2 ^ X6 ^ Y6,   X3,             Y4,             Y1 ^ Y7         },
    {0,              0,              0,              S0,             S1,             S2,             X0,             Y0,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2 ^ Y5 ^ X7,   X1 ^ X5 ^ Y7,   Y1 ^ X6 ^ Y6,   Y3,             X4,             X2 ^ Y7         },
    {0,              0,              0,              0,              S0,             S1,             S2,             X0,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y0 ^ Y5 ^ X7,   X1 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   X3,             X2 ^ Y7,        Y1 ^ Y8         },
    {S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             X2,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X8,   Z4 ^ X5 ^ Y8,   Z3 ^ Y6 ^ X7,   Y2 ^ X6 ^ Y7,   X3,             Y4              },
    {0,              S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X8,   Z3 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   Y3,             X4              },
    {0,              0,              S0,             S1,             S2,             X0,             Y0,             X1,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X8,   Y2 ^ X5 ^ Y8,   X2 ^ Y6 ^ X7,   Y1 ^ X6 ^ Y7,   X3,             Y4              },
    {0,              0,              0,              S0,             S1,             S2,             X0,             Y0,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2 ^ Y5 ^ X8,   X1 ^ X5 ^ Y8,   Y1 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   Y3,             X4              },
    {0,              0,              0,              0,              S0,             S1,             S2,             X0,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y0 ^ Y5 ^ X8,   X1 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   X3,             Y1 ^ Y7         },
    {S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             Y2 ^ Y7         },
    {0,              S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y2 ^ Y6,        X2 ^ Y7         },
    {0,              0,              S0,             S1,             S2,             X0,             Y0,             X1,             Y3,             X3,             Y4,             X4,             Y5,             Y1 ^ Y6,        Y2 ^ Y7,        X2 ^ Y8         },
    {0,              0,              0,              S0,             S1,             S2,             X0,             Y0,             X2,             Y3,             X3,             Y4,             X4,             Y1 ^ Y5,        X1 ^ Y6,        Y2 ^ Y7         },
    {0,              0,              0,              0,              S0,             S1,             S2,             X0,             Y2,             X2,             Y3,             X3,             Y4,             Y0 ^ Y5,        Y1 ^ Y6,        X1 ^ Y7         },
    {S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             X2,             X3,             Z0 ^ X3 ^ Y3,   Y4,             X4,             Y5,             X5,             Y6,             Y2 ^ Y7         },
    {0,              S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             X3,             Z0 ^ X3 ^ Y3,   Y4,             X4,             Y5,             X5,             X2 ^ Y6,        Y2 ^ Y7         },
    {0,              0,              S0,             S1,             S2,             X0,             Y0,             X1,             X3,             Z0 ^ X3 ^ Y3,   Y4,             X4,             Y5,             Y1 ^ Y6,        X2 ^ Y7,        Y2 ^ Y8         },
    {0,              0,              0,              S0,             S1,             S2,             X0,             Y0,             Y2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             X1 ^ Y5,        Y1 ^ Y6,        X2 ^ Y7         },
    {0,              0,              0,              0,              S0,             S1,             S2,             X0,             X2,             Z0 ^ X3 ^ Y3,   Y2,             X3,             Y4,             Y0 ^ Y5,        X1 ^ Y6,        Y1 ^ Y7         },
    {S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             X2,             Y3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             Y2 ^ Y7         },
    {0,              S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             Y3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y2 ^ Y6,        X2 ^ Y7         },
    {0,              0,              S0,             S1,             S2,             X0,             Y0,             X1,             Y3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X4,             Y5,             Y1 ^ Y6,        Y2 ^ Y7,        X2 ^ Y8         },
    {0,              0,              0,              S0,             S1,             S2,             X0,             Y0,             X2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y3,             X4,             Y1 ^ Y5,        X1 ^ Y6,        Y2 ^ Y7         },
    {0,              0,              0,              0,              S0,             S1,             S2,             X0,             X2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2,             X3,             Y0 ^ Y5,        X1 ^ Y6,        Y1 ^ Y7         },
    {S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             X2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y4,             X5,             Y6,             Y2 ^ Y7         },
    {0,              S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y4,             X5,             X2 ^ Y6,        Y2 ^ Y7         },
    {0,              0,              S0,             S1,             S2,             X0,             Y0,             X1,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y4,             Y1 ^ Y6,        X2 ^ Y7,        Y2 ^ Y8         },
    {0,              0,              0,              S0,             S1,             S2,             X0,             Y0,             Y3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2 ^ X5 ^ Y5,   X4,             X1 ^ Y6,        Y1 ^ Y7,        X2 ^ Y8         },
    {0,              0,              0,              0,              S0,             S1,             S2,             X0,             Y2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y0 ^ X5 ^ Y5,   X3,             X1 ^ Y6,        Y1 ^ Y7,        X2 ^ Y8         },
    {S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             X2,             Y3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   Z3 ^ X5 ^ Y6,   X4,             Y6,             Y2 ^ Y7         },
    {0,              S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             Y3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   Z3 ^ X5 ^ Y6,   X4,             Y2 ^ Y6,        X2 ^ Y7         },
    {0,              0,              S0,             S1,             S2,             X0,             Y0,             X1,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   Y2 ^ X5 ^ Y6,   Y4,             X2 ^ Y6,        Y1 ^ Y7         },
    {0,              0,              0,              S0,             S1,             S2,             X0,             Y0,             Y3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2 ^ Y5 ^ X6,   X1 ^ X5 ^ Y6,   X4,             Y1 ^ Y6,        X2 ^ Y7         },
    {0,              0,              0,              0,              S0,             S1,             S2,             X0,             X3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y0 ^ Y5 ^ X6,   X1 ^ X5 ^ Y6,   Y1 ^ Y6,        X2 ^ Y7,        Y2 ^ Y8         },
    {S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             X2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   Z4 ^ X5 ^ Y7,   Z3 ^ X6 ^ Y6,   Y4,             Y2 ^ Y7         },
    {0,              S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             Y3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   Z3 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   X4,             X2 ^ Y7         },
    {0,              0,              S0,             S1,             S2,             X0,             Y0,             X1,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   Y2 ^ X5 ^ Y7,   X2 ^ X6 ^ Y6,   Y4,             Y1 ^ Y7         },
    {0,              0,              0,              S0,             S1,             S2,             X0,             Y0,             Y3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2 ^ Y5 ^ X7,   X1 ^ X5 ^ Y7,   Y1 ^ X6 ^ Y6,   X4,             X2 ^ Y7         },
    {0,              0,              0,              0,              S0,             S1,             S2,             X0,             X3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y0 ^ Y5 ^ X7,   X1 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   X2 ^ Y7,        Y1 ^ Y8         },
    {S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             X2,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X8,   Z4 ^ X5 ^ Y8,   Z3 ^ Y6 ^ X7,   Y2 ^ X6 ^ Y7,   Y4              },
    {0,              S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             Y3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X8,   Z3 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   X4              },
    {0,              0,              S0,             S1,             S2,             X0,             Y0,             X1,             X3,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X8,   Y2 ^ X5 ^ Y8,   X2 ^ Y6 ^ X7,   Y1 ^ X6 ^ Y7,   Y4              },
    {0,              0,              0,              S0,             S1,             S2,             X0,             Y0,             Y3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2 ^ Y5 ^ X8,   X1 ^ X5 ^ Y8,   Y1 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   X4              },
    {0,              0,              0,              0,              S0,             S1,             S2,             X0,             X3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y0 ^ Y5 ^ X8,   X1 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   Y1 ^ Y7         },
    {S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             Y2 ^ Y7         },
    {0,              S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y2 ^ Y6,        X2 ^ Y7         },
    {0,              0,              S0,             S1,             S2,             X0,             Y0,             X1,             Y3,             X3,             Y4,             X4,             Y5,             Y1 ^ Y6,        Y2 ^ Y7,        X2 ^ Y8         },
    {0,              0,              0,              S0,             S1,             S2,             X0,             Y0,             X2,             Y3,             X3,             Y4,             X4,             Y1 ^ Y5,        X1 ^ Y6,        Y2 ^ Y7         },
    {0,              0,              0,              0,              S0,             S1,             S2,             X0,             Y2,             X2,             Y3,             X3,             Y4,             Y0 ^ Y5,        Y1 ^ Y6,        X1 ^ Y7         },
    {S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             X2,             X3,             Y4,             Z0 ^ X3 ^ Y3,   X4,             Y5,             X5,             Y6,             Y2 ^ Y7         },
    {0,              S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             X3,             Y4,             Z0 ^ X3 ^ Y3,   X4,             Y5,             X5,             X2 ^ Y6,        Y2 ^ Y7         },
    {0,              0,              S0,             S1,             S2,             X0,             Y0,             X1,             X3,             Y4,             Z0 ^ X3 ^ Y3,   X4,             Y5,             Y1 ^ Y6,        X2 ^ Y7,        Y2 ^ Y8         },
    {0,              0,              0,              S0,             S1,             S2,             X0,             Y0,             Y2,             X3,             Z0 ^ X3 ^ Y3,   Y4,             X4,             X1 ^ Y5,        Y1 ^ Y6,        X2 ^ Y7         },
    {0,              0,              0,              0,              S0,             S1,             S2,             X0,             X2,             Y2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             Y0 ^ Y5,        X1 ^ Y6,        Y1 ^ Y7         },
    {S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             X2,             Y3,             X4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y5,             X5,             Y6,             Y2 ^ Y7         },
    {0,              S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             Y3,             X4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y5,             X5,             Y2 ^ Y6,        X2 ^ Y7         },
    {0,              0,              S0,             S1,             S2,             X0,             Y0,             X1,             Y3,             X4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y5,             Y1 ^ Y6,        Y2 ^ Y7,        X2 ^ Y8         },
    {0,              0,              0,              S0,             S1,             S2,             X0,             Y0,             X2,             Y3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X4,             Y1 ^ Y5,        X1 ^ Y6,        Y2 ^ Y7         },
    {0,              0,              0,              0,              S0,             S1,             S2,             X0,             X2,             Y2,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X3,             Y0 ^ Y5,        X1 ^ Y6,        Y1 ^ Y7         },
    {S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             X2,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5,             Y6,             Y2 ^ Y7         },
    {0,              S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5,             X2 ^ Y6,        Y2 ^ Y7         },
    {0,              0,              S0,             S1,             S2,             X0,             Y0,             X1,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y1 ^ Y6,        X2 ^ Y7,        Y2 ^ Y8         },
    {0,              0,              0,              S0,             S1,             S2,             X0,             Y0,             Y3,             X4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2 ^ X5 ^ Y5,   X1 ^ Y6,        Y1 ^ Y7,        X2 ^ Y8         },
    {0,              0,              0,              0,              S0,             S1,             S2,             X0,             Y2,             X3,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y0 ^ X5 ^ Y5,   X1 ^ Y6,        Y1 ^ Y7,        X2 ^ Y8         },
    {S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             X2,             Y3,             X4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   Z3 ^ X5 ^ Y6,   Y6,             Y2 ^ Y7         },
    {0,              S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             Y3,             X4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   Z3 ^ X5 ^ Y6,   Y2 ^ Y6,        X2 ^ Y7         },
    {0,              0,              S0,             S1,             S2,             X0,             Y0,             X1,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   Y2 ^ X5 ^ Y6,   X2 ^ Y6,        Y1 ^ Y7         },
    {0,              0,              0,              S0,             S1,             S2,             X0,             Y0,             Y3,             X4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2 ^ Y5 ^ X6,   X1 ^ X5 ^ Y6,   Y1 ^ Y6,        X2 ^ Y7         },
    {0,              0,              0,              0,              S0,             S1,             S2,             X0,             X3,             Y1 ^ Y6,        Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y0 ^ Y5 ^ X6,   X1 ^ X5 ^ Y6,   X2 ^ Y7,        Y2 ^ Y8         },
    {S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             X2,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   Z4 ^ X5 ^ Y7,   Z3 ^ X6 ^ Y6,   Y2 ^ Y7         },
    {0,              S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             Y3,             X4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   Z3 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   X2 ^ Y7         },
    {0,              0,              S0,             S1,             S2,             X0,             Y0,             X1,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   Y2 ^ X5 ^ Y7,   X2 ^ X6 ^ Y6,   Y1 ^ Y7         },
    {0,              0,              0,              S0,             S1,             S2,             X0,             Y0,             Y3,             X4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2 ^ Y5 ^ X7,   X1 ^ X5 ^ Y7,   Y1 ^ X6 ^ Y6,   X2 ^ Y7         },
    {0,              0,              0,              0,              S0,             S1,             S2,             X0,             X3,             X2 ^ Y7,        Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y0 ^ Y5 ^ X7,   X1 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   Y1 ^ Y8         },
    {S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             X2,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X8,   Z4 ^ X5 ^ Y8,   Z3 ^ Y6 ^ X7,   Y2 ^ X6 ^ Y7    },
    {0,              S0,             S1,             S2,             X0,             Y0,             X1,             Y1,             Y3,             X4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X8,   Z3 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7    },
    {0,              0,              S0,             S1,             S2,             X0,             Y0,             X1,             X3,             Y4,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X8,   Y2 ^ X5 ^ Y8,   X2 ^ Y6 ^ X7,   Y1 ^ X6 ^ Y7    },
    {0,              0,              0,              S0,             S1,             S2,             X0,             Y0,             Y3,             X4,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y2 ^ Y5 ^ X8,   X1 ^ X5 ^ Y8,   Y1 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7    },
    {0,              0,              0,              0,              S0,             S1,             S2,             X0,             X3,             Y1 ^ Y7,        Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y0 ^ Y5 ^ X8,   X1 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7    },
};

const UINT_64 SW_4K_S3[][12]=
{
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2,             Y3              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2,             Y2              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2,             Y2              },
};

const UINT_64 SW_4K_S3_X[][12]=
{
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2,             Y3              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2,             Y2              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2,             Y2              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2 ^ X3 ^ Z3,   X3,             Z3,             Y3              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             X2 ^ Y2 ^ Z3,   X2,             Z3,             Y3              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             X2 ^ Y2 ^ Z2,   X2,             Z2,             Y3              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1 ^ X2 ^ Z2,   X2,             Z2,             Y2              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             X1 ^ Y1 ^ Z2,   X1,             Z2,             Y2              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2 ^ X4 ^ Z4,   X3 ^ Y3 ^ Z3,   Z3,             Y3              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2 ^ X3 ^ Z4,   X2 ^ Y3 ^ Z3,   Z3,             Y3              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2 ^ X3 ^ Z3,   X2 ^ Z2 ^ Y3,   Z2,             Y3              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1 ^ X3 ^ Z3,   X2 ^ Y2 ^ Z2,   Z2,             Y2              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1 ^ X2 ^ Z3,   X1 ^ Y2 ^ Z2,   Z2,             Y2              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2 ^ X5 ^ Z5,   X3 ^ Y4 ^ Z4,   Y3 ^ Z3 ^ X4,   Y3              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2 ^ X4 ^ Z5,   X2 ^ Y4 ^ Z4,   X3 ^ Y3 ^ Z3,   Y3              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2 ^ X4 ^ Z4,   X2 ^ Z3 ^ Y4,   Z2 ^ X3 ^ Y3,   Y3              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1 ^ X4 ^ Z4,   X2 ^ Y3 ^ Z3,   Y2 ^ Z2 ^ X3,   Y2              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1 ^ X3 ^ Z4,   X1 ^ Y3 ^ Z3,   X2 ^ Y2 ^ Z2,   Y2              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2 ^ X6 ^ Z6,   X3 ^ Y5 ^ Z5,   Z3 ^ Y4 ^ X5,   Y3 ^ X4 ^ Z4    },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2 ^ X5 ^ Z6,   X2 ^ Y5 ^ Z5,   Z3 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z4    },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2 ^ X5 ^ Z5,   X2 ^ Z4 ^ Y5,   Z2 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z3    },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1 ^ X5 ^ Z5,   X2 ^ Y4 ^ Z4,   Z2 ^ Y3 ^ X4,   Y2 ^ X3 ^ Z3    },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1 ^ X4 ^ Z5,   X1 ^ Y4 ^ Z4,   Z2 ^ X3 ^ Y3,   X2 ^ Y2 ^ Z3    },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2 ^ X6 ^ Z6,   X3 ^ Y5 ^ Z5,   Z3 ^ Y4 ^ X5,   Y3 ^ X4 ^ Z4    },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2 ^ X5 ^ Z6,   X2 ^ Y5 ^ Z5,   Z3 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z4    },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2 ^ X5 ^ Z5,   X2 ^ Z4 ^ Y5,   Z2 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z3    },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1 ^ X5 ^ Z5,   X2 ^ Y4 ^ Z4,   Z2 ^ Y3 ^ X4,   Y2 ^ X3 ^ Z3    },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1 ^ X4 ^ Z5,   X1 ^ Y4 ^ Z4,   Z2 ^ X3 ^ Y3,   X2 ^ Y2 ^ Z3    },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2 ^ X6 ^ Z6,   X3 ^ Y5 ^ Z5,   Z3 ^ Y4 ^ X5,   Y3 ^ X4 ^ Z4    },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2 ^ X5 ^ Z6,   X2 ^ Y5 ^ Z5,   Z3 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z4    },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2 ^ X5 ^ Z5,   X2 ^ Z4 ^ Y5,   Z2 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z3    },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1 ^ X5 ^ Z5,   X2 ^ Y4 ^ Z4,   Z2 ^ Y3 ^ X4,   Y2 ^ X3 ^ Z3    },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1 ^ X4 ^ Z5,   X1 ^ Y4 ^ Z4,   Z2 ^ X3 ^ Y3,   X2 ^ Y2 ^ Z3    },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2,             Y3              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2,             Y2              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2,             Y2              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3 ^ Y3 ^ Z3,   Z3,             Y3              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2 ^ Y3 ^ Z3,   Z3,             Y3              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2 ^ Z2 ^ Y3,   Z2,             Y3              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2 ^ Y2 ^ Z2,   Z2,             Y2              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1 ^ Y2 ^ Z2,   Z2,             Y2              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3 ^ Y4 ^ Z4,   Y3 ^ Z3 ^ X4,   Y3              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2 ^ Y4 ^ Z4,   X3 ^ Y3 ^ Z3,   Y3              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2 ^ Z3 ^ Y4,   Z2 ^ X3 ^ Y3,   Y3              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2 ^ Y3 ^ Z3,   Y2 ^ Z2 ^ X3,   Y2              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1 ^ Y3 ^ Z3,   X2 ^ Y2 ^ Z2,   Y2              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3 ^ Y5 ^ Z5,   Z3 ^ Y4 ^ X5,   Y3 ^ X4 ^ Z4    },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2 ^ Y5 ^ Z5,   Z3 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z4    },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2 ^ Z4 ^ Y5,   Z2 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z3    },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2 ^ Y4 ^ Z4,   Z2 ^ Y3 ^ X4,   Y2 ^ X3 ^ Z3    },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1 ^ Y4 ^ Z4,   Z2 ^ X3 ^ Y3,   X2 ^ Y2 ^ Z3    },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3 ^ Y5 ^ Z5,   Z3 ^ Y4 ^ X5,   Y3 ^ X4 ^ Z4    },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2 ^ Y5 ^ Z5,   Z3 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z4    },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2 ^ Z4 ^ Y5,   Z2 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z3    },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2 ^ Y4 ^ Z4,   Z2 ^ Y3 ^ X4,   Y2 ^ X3 ^ Z3    },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1 ^ Y4 ^ Z4,   Z2 ^ X3 ^ Y3,   X2 ^ Y2 ^ Z3    },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3 ^ Y5 ^ Z5,   Z3 ^ Y4 ^ X5,   Y3 ^ X4 ^ Z4    },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2 ^ Y5 ^ Z5,   Z3 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z4    },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2 ^ Z4 ^ Y5,   Z2 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z3    },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2 ^ Y4 ^ Z4,   Z2 ^ Y3 ^ X4,   Y2 ^ X3 ^ Z3    },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1 ^ Y4 ^ Z4,   Z2 ^ X3 ^ Y3,   X2 ^ Y2 ^ Z3    },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3 ^ Y5 ^ Z5,   Z3 ^ Y4 ^ X5,   Y3 ^ X4 ^ Z4    },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2 ^ Y5 ^ Z5,   Z3 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z4    },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2 ^ Z4 ^ Y5,   Z2 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z3    },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2 ^ Y4 ^ Z4,   Z2 ^ Y3 ^ X4,   Y2 ^ X3 ^ Z3    },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1 ^ Y4 ^ Z4,   Z2 ^ X3 ^ Y3,   X2 ^ Y2 ^ Z3    },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2,             Y3              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2,             Y2              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2,             Y2              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Y3 ^ Z3 ^ X4,   Y3              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             X3 ^ Y3 ^ Z3,   Y3              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2 ^ X3 ^ Y3,   Y3              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Y2 ^ Z2 ^ X3,   Y2              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             X2 ^ Y2 ^ Z2,   Y2              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3 ^ Y4 ^ X5,   Y3 ^ X4 ^ Z4    },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z4    },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z3    },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2 ^ Y3 ^ X4,   Y2 ^ X3 ^ Z3    },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2 ^ X3 ^ Y3,   X2 ^ Y2 ^ Z3    },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3 ^ Y4 ^ X5,   Y3 ^ X4 ^ Z4    },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z4    },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z3    },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2 ^ Y3 ^ X4,   Y2 ^ X3 ^ Z3    },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2 ^ X3 ^ Y3,   X2 ^ Y2 ^ Z3    },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3 ^ Y4 ^ X5,   Y3 ^ X4 ^ Z4    },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z4    },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z3    },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2 ^ Y3 ^ X4,   Y2 ^ X3 ^ Z3    },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2 ^ X3 ^ Y3,   X2 ^ Y2 ^ Z3    },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3 ^ Y4 ^ X5,   Y3 ^ X4 ^ Z4    },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z4    },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z3    },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2 ^ Y3 ^ X4,   Y2 ^ X3 ^ Z3    },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2 ^ X3 ^ Y3,   X2 ^ Y2 ^ Z3    },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3 ^ Y4 ^ X5,   Y3 ^ X4 ^ Z4    },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z4    },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z3    },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2 ^ Y3 ^ X4,   Y2 ^ X3 ^ Z3    },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2 ^ X3 ^ Y3,   X2 ^ Y2 ^ Z3    },
};

const UINT_64 SW_64K_S3[][16]=
{
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3,             X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2,             Y3,             X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3,             X3              },
};

const UINT_64 SW_64K_S3_X[][16]=
{
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3,             X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2,             Y3,             X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2 ^ X3 ^ Z3,   X3,             Z3,             Y3,             X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             X2 ^ Y2 ^ Z3,   X2,             Z3,             Y3,             X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             X2 ^ Y2 ^ Z2,   X2,             Z2,             Y3,             X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1 ^ X2 ^ Z2,   X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             X1 ^ Y1 ^ Z2,   X1,             Z2,             Y2,             X2,             Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2 ^ X4 ^ Z4,   X3 ^ Y3 ^ Z3,   Z3,             Y3,             X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2 ^ X3 ^ Z4,   X2 ^ Y3 ^ Z3,   Z3,             Y3,             X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2 ^ X3 ^ Z3,   X2 ^ Z2 ^ Y3,   Z2,             Y3,             X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1 ^ X3 ^ Z3,   X2 ^ Y2 ^ Z2,   Z2,             Y2,             X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1 ^ X2 ^ Z3,   X1 ^ Y2 ^ Z2,   Z2,             Y2,             X2,             Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2 ^ X5 ^ Z5,   X3 ^ Y4 ^ Z4,   Y3 ^ Z3 ^ X4,   Y3,             X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2 ^ X4 ^ Z5,   X2 ^ Y4 ^ Z4,   X3 ^ Y3 ^ Z3,   Y3,             X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2 ^ X4 ^ Z4,   X2 ^ Z3 ^ Y4,   Z2 ^ X3 ^ Y3,   Y3,             X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1 ^ X4 ^ Z4,   X2 ^ Y3 ^ Z3,   Y2 ^ Z2 ^ X3,   Y2,             X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1 ^ X3 ^ Z4,   X1 ^ Y3 ^ Z3,   X2 ^ Y2 ^ Z2,   Y2,             X2,             Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2 ^ X6 ^ Z6,   X3 ^ Y5 ^ Z5,   Z3 ^ Y4 ^ X5,   Y3 ^ X4 ^ Z4,   X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2 ^ X5 ^ Z6,   X2 ^ Y5 ^ Z5,   Z3 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z4,   X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2 ^ X5 ^ Z5,   X2 ^ Z4 ^ Y5,   Z2 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z3,   X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1 ^ X5 ^ Z5,   X2 ^ Y4 ^ Z4,   Z2 ^ Y3 ^ X4,   Y2 ^ X3 ^ Z3,   X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1 ^ X4 ^ Z5,   X1 ^ Y4 ^ Z4,   Z2 ^ X3 ^ Y3,   X2 ^ Y2 ^ Z3,   X2,             Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2 ^ X7 ^ Z7,   X3 ^ Y6 ^ Z6,   Z3 ^ Y5 ^ X6,   Y3 ^ X5 ^ Z5,   X4 ^ Y4 ^ Z4,   Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2 ^ X6 ^ Z7,   X2 ^ Y6 ^ Z6,   Z3 ^ X5 ^ Y5,   Y3 ^ X4 ^ Z5,   X3 ^ Y4 ^ Z4,   Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2 ^ X6 ^ Z6,   X2 ^ Z5 ^ Y6,   Z2 ^ X5 ^ Y5,   Y3 ^ X4 ^ Z4,   X3 ^ Z3 ^ Y4,   Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1 ^ X6 ^ Z6,   X2 ^ Y5 ^ Z5,   Z2 ^ Y4 ^ X5,   Y2 ^ X4 ^ Z4,   X3 ^ Y3 ^ Z3,   Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1 ^ X5 ^ Z6,   X1 ^ Y5 ^ Z5,   Z2 ^ X4 ^ Y4,   Y2 ^ X3 ^ Z4,   X2 ^ Y3 ^ Z3,   Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2 ^ X8 ^ Z8,   X3 ^ Y7 ^ Z7,   Z3 ^ Y6 ^ X7,   Y3 ^ X6 ^ Z6,   X4 ^ Y5 ^ Z5,   Y4 ^ Z4 ^ X5,   Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2 ^ X7 ^ Z8,   X2 ^ Y7 ^ Z7,   Z3 ^ X6 ^ Y6,   Y3 ^ X5 ^ Z6,   X3 ^ Y5 ^ Z5,   X4 ^ Y4 ^ Z4,   Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2 ^ X7 ^ Z7,   X2 ^ Z6 ^ Y7,   Z2 ^ X6 ^ Y6,   Y3 ^ X5 ^ Z5,   X3 ^ Z4 ^ Y5,   Z3 ^ X4 ^ Y4,   Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1 ^ X7 ^ Z7,   X2 ^ Y6 ^ Z6,   Z2 ^ Y5 ^ X6,   Y2 ^ X5 ^ Z5,   X3 ^ Y4 ^ Z4,   Y3 ^ Z3 ^ X4,   Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1 ^ X6 ^ Z7,   X1 ^ Y6 ^ Z6,   Z2 ^ X5 ^ Y5,   Y2 ^ X4 ^ Z5,   X2 ^ Y4 ^ Z4,   X3 ^ Y3 ^ Z3,   Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3,             X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2,             Y3,             X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3 ^ Y3 ^ Z3,   Z3,             Y3,             X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2 ^ Y3 ^ Z3,   Z3,             Y3,             X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2 ^ Z2 ^ Y3,   Z2,             Y3,             X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2 ^ Y2 ^ Z2,   Z2,             Y2,             X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1 ^ Y2 ^ Z2,   Z2,             Y2,             X2,             Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3 ^ Y4 ^ Z4,   Y3 ^ Z3 ^ X4,   Y3,             X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2 ^ Y4 ^ Z4,   X3 ^ Y3 ^ Z3,   Y3,             X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2 ^ Z3 ^ Y4,   Z2 ^ X3 ^ Y3,   Y3,             X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2 ^ Y3 ^ Z3,   Y2 ^ Z2 ^ X3,   Y2,             X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1 ^ Y3 ^ Z3,   X2 ^ Y2 ^ Z2,   Y2,             X2,             Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3 ^ Y5 ^ Z5,   Z3 ^ Y4 ^ X5,   Y3 ^ X4 ^ Z4,   X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2 ^ Y5 ^ Z5,   Z3 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z4,   X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2 ^ Z4 ^ Y5,   Z2 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z3,   X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2 ^ Y4 ^ Z4,   Z2 ^ Y3 ^ X4,   Y2 ^ X3 ^ Z3,   X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1 ^ Y4 ^ Z4,   Z2 ^ X3 ^ Y3,   X2 ^ Y2 ^ Z3,   X2,             Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3 ^ Y6 ^ Z6,   Z3 ^ Y5 ^ X6,   Y3 ^ X5 ^ Z5,   X4 ^ Y4 ^ Z4,   Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2 ^ Y6 ^ Z6,   Z3 ^ X5 ^ Y5,   Y3 ^ X4 ^ Z5,   X3 ^ Y4 ^ Z4,   Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2 ^ Z5 ^ Y6,   Z2 ^ X5 ^ Y5,   Y3 ^ X4 ^ Z4,   X3 ^ Z3 ^ Y4,   Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2 ^ Y5 ^ Z5,   Z2 ^ Y4 ^ X5,   Y2 ^ X4 ^ Z4,   X3 ^ Y3 ^ Z3,   Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1 ^ Y5 ^ Z5,   Z2 ^ X4 ^ Y4,   Y2 ^ X3 ^ Z4,   X2 ^ Y3 ^ Z3,   Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3 ^ Y7 ^ Z7,   Z3 ^ Y6 ^ X7,   Y3 ^ X6 ^ Z6,   X4 ^ Y5 ^ Z5,   Y4 ^ Z4 ^ X5,   Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2 ^ Y7 ^ Z7,   Z3 ^ X6 ^ Y6,   Y3 ^ X5 ^ Z6,   X3 ^ Y5 ^ Z5,   X4 ^ Y4 ^ Z4,   Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2 ^ Z6 ^ Y7,   Z2 ^ X6 ^ Y6,   Y3 ^ X5 ^ Z5,   X3 ^ Z4 ^ Y5,   Z3 ^ X4 ^ Y4,   Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2 ^ Y6 ^ Z6,   Z2 ^ Y5 ^ X6,   Y2 ^ X5 ^ Z5,   X3 ^ Y4 ^ Z4,   Y3 ^ Z3 ^ X4,   Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1 ^ Y6 ^ Z6,   Z2 ^ X5 ^ Y5,   Y2 ^ X4 ^ Z5,   X2 ^ Y4 ^ Z4,   X3 ^ Y3 ^ Z3,   Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3 ^ Y8 ^ Z8,   Z3 ^ Y7 ^ X8,   Y3 ^ X7 ^ Z7,   X4 ^ Y6 ^ Z6,   Z4 ^ Y5 ^ X6,   Y4 ^ X5 ^ Z5,   X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2 ^ Y8 ^ Z8,   Z3 ^ X7 ^ Y7,   Y3 ^ X6 ^ Z7,   X3 ^ Y6 ^ Z6,   Z4 ^ X5 ^ Y5,   X4 ^ Y4 ^ Z5,   X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2 ^ Z7 ^ Y8,   Z2 ^ X7 ^ Y7,   Y3 ^ X6 ^ Z6,   X3 ^ Z5 ^ Y6,   Z3 ^ X5 ^ Y5,   X4 ^ Y4 ^ Z4,   X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2 ^ Y7 ^ Z7,   Z2 ^ Y6 ^ X7,   Y2 ^ X6 ^ Z6,   X3 ^ Y5 ^ Z5,   Z3 ^ Y4 ^ X5,   Y3 ^ X4 ^ Z4,   X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1 ^ Y7 ^ Z7,   Z2 ^ X6 ^ Y6,   Y2 ^ X5 ^ Z6,   X2 ^ Y5 ^ Z5,   Z3 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z4,   X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3,             X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2,             Y3,             X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Y3 ^ Z3 ^ X4,   Y3,             X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             X3 ^ Y3 ^ Z3,   Y3,             X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2 ^ X3 ^ Y3,   Y3,             X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Y2 ^ Z2 ^ X3,   Y2,             X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             X2 ^ Y2 ^ Z2,   Y2,             X2,             Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3 ^ Y4 ^ X5,   Y3 ^ X4 ^ Z4,   X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z4,   X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z3,   X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2 ^ Y3 ^ X4,   Y2 ^ X3 ^ Z3,   X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2 ^ X3 ^ Y3,   X2 ^ Y2 ^ Z3,   X2,             Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3 ^ Y5 ^ X6,   Y3 ^ X5 ^ Z5,   X4 ^ Y4 ^ Z4,   Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3 ^ X5 ^ Y5,   Y3 ^ X4 ^ Z5,   X3 ^ Y4 ^ Z4,   Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2 ^ X5 ^ Y5,   Y3 ^ X4 ^ Z4,   X3 ^ Z3 ^ Y4,   Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2 ^ Y4 ^ X5,   Y2 ^ X4 ^ Z4,   X3 ^ Y3 ^ Z3,   Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2 ^ X4 ^ Y4,   Y2 ^ X3 ^ Z4,   X2 ^ Y3 ^ Z3,   Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3 ^ Y6 ^ X7,   Y3 ^ X6 ^ Z6,   X4 ^ Y5 ^ Z5,   Y4 ^ Z4 ^ X5,   Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3 ^ X6 ^ Y6,   Y3 ^ X5 ^ Z6,   X3 ^ Y5 ^ Z5,   X4 ^ Y4 ^ Z4,   Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2 ^ X6 ^ Y6,   Y3 ^ X5 ^ Z5,   X3 ^ Z4 ^ Y5,   Z3 ^ X4 ^ Y4,   Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2 ^ Y5 ^ X6,   Y2 ^ X5 ^ Z5,   X3 ^ Y4 ^ Z4,   Y3 ^ Z3 ^ X4,   Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2 ^ X5 ^ Y5,   Y2 ^ X4 ^ Z5,   X2 ^ Y4 ^ Z4,   X3 ^ Y3 ^ Z3,   Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3 ^ Y7 ^ X8,   Y3 ^ X7 ^ Z7,   X4 ^ Y6 ^ Z6,   Z4 ^ Y5 ^ X6,   Y4 ^ X5 ^ Z5,   X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3 ^ X7 ^ Y7,   Y3 ^ X6 ^ Z7,   X3 ^ Y6 ^ Z6,   Z4 ^ X5 ^ Y5,   X4 ^ Y4 ^ Z5,   X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2 ^ X7 ^ Y7,   Y3 ^ X6 ^ Z6,   X3 ^ Z5 ^ Y6,   Z3 ^ X5 ^ Y5,   X4 ^ Y4 ^ Z4,   X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2 ^ Y6 ^ X7,   Y2 ^ X6 ^ Z6,   X3 ^ Y5 ^ Z5,   Z3 ^ Y4 ^ X5,   Y3 ^ X4 ^ Z4,   X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2 ^ X6 ^ Y6,   Y2 ^ X5 ^ Z6,   X2 ^ Y5 ^ Z5,   Z3 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z4,   X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3 ^ Y8 ^ X9,   Y3 ^ X8 ^ Z8,   X4 ^ Y7 ^ Z7,   Z4 ^ Y6 ^ X7,   Y4 ^ X6 ^ Z6,   X5 ^ Y5 ^ Z5    },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3 ^ X8 ^ Y8,   Y3 ^ X7 ^ Z8,   X3 ^ Y7 ^ Z7,   Z4 ^ X6 ^ Y6,   Y4 ^ X5 ^ Z6,   X4 ^ Y5 ^ Z5    },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2 ^ X8 ^ Y8,   Y3 ^ X7 ^ Z7,   X3 ^ Z6 ^ Y7,   Z3 ^ X6 ^ Y6,   Y4 ^ X5 ^ Z5,   X4 ^ Z4 ^ Y5    },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2 ^ Y7 ^ X8,   Y2 ^ X7 ^ Z7,   X3 ^ Y6 ^ Z6,   Z3 ^ Y5 ^ X6,   Y3 ^ X5 ^ Z5,   X4 ^ Y4 ^ Z4    },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2 ^ X7 ^ Y7,   Y2 ^ X6 ^ Z7,   X2 ^ Y6 ^ Z6,   Z3 ^ X5 ^ Y5,   Y3 ^ X4 ^ Z5,   X3 ^ Y4 ^ Z4    },
};

const UINT_64 SW_64K_S3_T[][16]=
{
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3,             X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2,             Y3,             X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2 ^ X3 ^ Z3,   X3,             Z3,             Y3,             X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             X2 ^ Y2 ^ Z3,   X2,             Z3,             Y3,             X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             X2 ^ Y2 ^ Z2,   X2,             Z2,             Y3,             X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1 ^ X2 ^ Z2,   X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             X1 ^ Y1 ^ Z2,   X1,             Z2,             Y2,             X2,             Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2 ^ X4 ^ Z4,   X3 ^ Y3 ^ Z3,   Z3,             Y3,             X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2 ^ X3 ^ Z4,   X2 ^ Y3 ^ Z3,   Z3,             Y3,             X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2 ^ X3 ^ Z3,   X2 ^ Z2 ^ Y3,   Z2,             Y3,             X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1 ^ X3 ^ Z3,   X2 ^ Y2 ^ Z2,   Z2,             Y2,             X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1 ^ X2 ^ Z3,   X1 ^ Y2 ^ Z2,   Z2,             Y2,             X2,             Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2 ^ X5,        X3 ^ Y4 ^ Z4,   Y3 ^ Z3 ^ X4,   Y3,             X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2 ^ X4,        X2 ^ Y4 ^ Z4,   X3 ^ Y3 ^ Z3,   Y3,             X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2 ^ X4,        X2 ^ Z3 ^ Y4,   Z2 ^ X3 ^ Y3,   Y3,             X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1 ^ X4,        X2 ^ Y3 ^ Z3,   Y2 ^ Z2 ^ X3,   Y2,             X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1 ^ X3,        X1 ^ Y3 ^ Z3,   X2 ^ Y2 ^ Z2,   Y2,             X2,             Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3 ^ Y4 ^ X5,   Y3 ^ X4 ^ Z4,   X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z4,   X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z3,   X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2 ^ Y3 ^ X4,   Y2 ^ X3 ^ Z3,   X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2 ^ X3 ^ Y3,   X2 ^ Y2 ^ Z3,   X2,             Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3 ^ X5,        X4 ^ Y4 ^ Z4,   Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3 ^ X4,        X3 ^ Y4 ^ Z4,   Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2,             Y3 ^ X4,        X3 ^ Z3 ^ Y4,   Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2,             Y2 ^ X4,        X3 ^ Y3 ^ Z3,   Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2,             Y2 ^ X3,        X2 ^ Y3 ^ Z3,   Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4,             Y4 ^ Z4 ^ X5,   Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3,             X3,             X4 ^ Y4 ^ Z4,   Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2,             Y3,             X3,             Z3 ^ X4 ^ Y4,   Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Y3 ^ Z3 ^ X4,   Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             X3 ^ Y3 ^ Z3,   Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3,             X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2,             Y3,             X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3 ^ Y3 ^ Z3,   Z3,             Y3,             X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2 ^ Y3 ^ Z3,   Z3,             Y3,             X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2 ^ Z2 ^ Y3,   Z2,             Y3,             X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2 ^ Y2 ^ Z2,   Z2,             Y2,             X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1 ^ Y2 ^ Z2,   Z2,             Y2,             X2,             Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3 ^ Y4 ^ Z4,   Y3 ^ Z3 ^ X4,   Y3,             X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2 ^ Y4 ^ Z4,   X3 ^ Y3 ^ Z3,   Y3,             X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2 ^ Z3 ^ Y4,   Z2 ^ X3 ^ Y3,   Y3,             X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2 ^ Y3 ^ Z3,   Y2 ^ Z2 ^ X3,   Y2,             X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1 ^ Y3 ^ Z3,   X2 ^ Y2 ^ Z2,   Y2,             X2,             Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3 ^ Y4 ^ X5,   Y3 ^ X4 ^ Z4,   X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z4,   X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z3,   X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2 ^ Y3 ^ X4,   Y2 ^ X3 ^ Z3,   X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2 ^ X3 ^ Y3,   X2 ^ Y2 ^ Z3,   X2,             Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3 ^ X5,        X4 ^ Y4 ^ Z4,   Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3 ^ X4,        X3 ^ Y4 ^ Z4,   Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2,             Y3 ^ X4,        X3 ^ Z3 ^ Y4,   Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2,             Y2 ^ X4,        X3 ^ Y3 ^ Z3,   Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2,             Y2 ^ X3,        X2 ^ Y3 ^ Z3,   Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4,             Y4 ^ Z4 ^ X5,   Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3,             X3,             X4 ^ Y4 ^ Z4,   Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2,             Y3,             X3,             Z3 ^ X4 ^ Y4,   Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Y3 ^ Z3 ^ X4,   Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             X3 ^ Y3 ^ Z3,   Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4,             Z4,             Y4 ^ X5,        X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3,             X3,             Z4,             X4 ^ Y4,        X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2,             Y3,             X3,             Z3,             X4 ^ Y4,        X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3 ^ X4,        X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             X3 ^ Y3,        X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3,             X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2,             Y3,             X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Y3 ^ Z3 ^ X4,   Y3,             X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             X3 ^ Y3 ^ Z3,   Y3,             X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2 ^ X3 ^ Y3,   Y3,             X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Y2 ^ Z2 ^ X3,   Y2,             X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             X2 ^ Y2 ^ Z2,   Y2,             X2,             Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3 ^ Y4 ^ X5,   Y3 ^ X4 ^ Z4,   X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z4,   X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2 ^ X4 ^ Y4,   X3 ^ Y3 ^ Z3,   X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2 ^ Y3 ^ X4,   Y2 ^ X3 ^ Z3,   X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2 ^ X3 ^ Y3,   X2 ^ Y2 ^ Z3,   X2,             Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3 ^ X5,        X4 ^ Y4 ^ Z4,   Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3 ^ X4,        X3 ^ Y4 ^ Z4,   Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2,             Y3 ^ X4,        X3 ^ Z3 ^ Y4,   Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2,             Y2 ^ X4,        X3 ^ Y3 ^ Z3,   Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2,             Y2 ^ X3,        X2 ^ Y3 ^ Z3,   Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4,             Y4 ^ Z4 ^ X5,   Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3,             X3,             X4 ^ Y4 ^ Z4,   Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2,             Y3,             X3,             Z3 ^ X4 ^ Y4,   Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Y3 ^ Z3 ^ X4,   Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             X3 ^ Y3 ^ Z3,   Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4,             Z4,             Y4 ^ X5,        X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3,             X3,             Z4,             X4 ^ Y4,        X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2,             Y3,             X3,             Z3,             X4 ^ Y4,        X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3 ^ X4,        X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             X3 ^ Y3,        X3              },
    {X0,             X1,             Z0,             Y0,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3,             X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             Z0,             Y1,             X1,             Z1,             Y2,             X2,             Z2,             Y3,             X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              Z0,             Y0,             X0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3,             X3              },
};

const UINT_64 SW_64K_D3_X[][16]=
{
    {X0,             X1,             Z0,             Y0,             Y1,             Z1,             X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             Z2,             Y2,             X2,             Z3,             Y3,             X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             X1,             Z0,             Y1,             Z1,             Y2,             X2,             Z2,             Y3,             X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             Z0,             X1,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Y1,             Z1,             X2,             Z2,             X3 ^ Y3,        X3,             Z3,             Y2,             X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             Z2,             X3 ^ Y3,        X2,             Z3,             Y2,             X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             X1,             Z0,             Y1,             Z1,             X3 ^ Y3,        X2,             Z2,             Y2,             X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             Z0,             X1,             Z1,             X3 ^ Y3,        X2,             Z2,             Y1,             X3,             Z3,             Y2,             X4              },
    {0,              0,              0,              0,              X0,             Z0,             Y0,             Z1,             X3 ^ Y3,        X1,             Z2,             Y1,             X2,             Z3,             Y2,             X3              },
    {X0,             X1,             Z0,             Y0,             Y1,             Z1,             X2,             Z2,             X3 ^ Y3,        X4 ^ Y4,        Z3,             Y2,             X3,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             Z2,             X3 ^ Y3,        X4 ^ Y4,        Z3,             Y2,             X2,             Z4,             Y3,             X4              },
    {0,              0,              X0,             Y0,             X1,             Z0,             Y1,             Z1,             X3 ^ Y3,        X4 ^ Y4,        Z2,             Y2,             X2,             Z3,             Y3,             X4              },
    {0,              0,              0,              X0,             Y0,             Z0,             X1,             Z1,             X3 ^ Y3,        X4 ^ Y4,        Z2,             Y1,             X2,             Z3,             Y2,             X3              },
    {0,              0,              0,              0,              X0,             Z0,             Y0,             Z1,             X3 ^ Y3,        X1 ^ X4 ^ Y4,   Z2,             Y1,             X2,             Z3,             Y2,             X3              },
    {X0,             X1,             Z0,             Y0,             Y1,             Z1,             X2,             Z2,             X3 ^ Y3,        X4 ^ Y4,        X5 ^ Y5,        Z3,             Y2,             X3,             Z4,             Y4              },
    {0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             Z2,             X3 ^ Y3,        X4 ^ Y4,        Z3 ^ X5 ^ Y5,   Y2,             X2,             Z4,             Y3,             X4              },
    {0,              0,              X0,             Y0,             X1,             Z0,             Y1,             Z1,             X3 ^ Y3,        X4 ^ Y4,        Z2 ^ X5 ^ Y5,   Y2,             X2,             Z3,             Y3,             X4              },
    {0,              0,              0,              X0,             Y0,             Z0,             X1,             Z1,             X3 ^ Y3,        X4 ^ Y4,        Z2 ^ X5 ^ Y5,   Y1,             X2,             Z3,             Y2,             X3              },
    {0,              0,              0,              0,              X0,             Z0,             Y0,             Z1,             X3 ^ Y3,        X1 ^ X4 ^ Y4,   Z2 ^ X5 ^ Y5,   Y1,             X2,             Z3,             Y2,             X3              },
    {X0,             X1,             Z0,             Y0,             Y1,             Z1,             X2,             Z2,             X3 ^ Y3,        X4 ^ Y4,        Y2 ^ Y5 ^ X6,   X5 ^ Y6,        Z3,             Y3,             X4,             Z4              },
    {0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             Z2,             X3 ^ Y3,        X4 ^ Y4,        Z3 ^ Y5 ^ X6,   Y2 ^ X5 ^ Y6,   X2,             Z4,             Y3,             X4              },
    {0,              0,              X0,             Y0,             X1,             Z0,             Y1,             Z1,             X3 ^ Y3,        X4 ^ Y4,        Z2 ^ Y5 ^ X6,   Y2 ^ X5 ^ Y6,   X2,             Z3,             Y3,             X4              },
    {0,              0,              0,              X0,             Y0,             Z0,             X1,             Z1,             X3 ^ Y3,        X4 ^ Y4,        Z2 ^ Y5 ^ X6,   Y1 ^ X5 ^ Y6,   X2,             Z3,             Y2,             X3              },
    {0,              0,              0,              0,              X0,             Z0,             Y0,             Z1,             X3 ^ Y3,        X1 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X6,   Y1 ^ X5 ^ Y6,   X2,             Z3,             Y2,             X3              },
    {X0,             X1,             Z0,             Y0,             Y1,             Z1,             X2,             Z2,             X3 ^ Y3,        X4 ^ Y4,        Y2 ^ Y5 ^ X7,   X5 ^ Y7,        Z3 ^ X6 ^ Y6,   Y3,             X4,             Z4              },
    {0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             Z2,             X3 ^ Y3,        X4 ^ Y4,        Z3 ^ Y5 ^ X7,   Y2 ^ X5 ^ Y7,   X2 ^ X6 ^ Y6,   Z4,             Y3,             X4              },
    {0,              0,              X0,             Y0,             X1,             Z0,             Y1,             Z1,             X3 ^ Y3,        X4 ^ Y4,        Z2 ^ Y5 ^ X7,   Y2 ^ X5 ^ Y7,   X2 ^ X6 ^ Y6,   Z3,             Y3,             X4              },
    {0,              0,              0,              X0,             Y0,             Z0,             X1,             Z1,             X3 ^ Y3,        X4 ^ Y4,        Z2 ^ Y5 ^ X7,   Y1 ^ X5 ^ Y7,   X2 ^ X6 ^ Y6,   Z3,             Y2,             X3              },
    {0,              0,              0,              0,              X0,             Z0,             Y0,             Z1,             X3 ^ Y3,        X1 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Y1 ^ X5 ^ Y7,   X2 ^ X6 ^ Y6,   Z3,             Y2,             X3              },
    {X0,             X1,             Z0,             Y0,             Y1,             Z1,             X2,             Z2,             X3 ^ Y3,        X4 ^ Y4,        Y2 ^ Y5 ^ X8,   X5 ^ Y8,        Z3 ^ Y6 ^ X7,   Z4 ^ X6 ^ Y7,   Y3,             X4              },
    {0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             Z2,             X3 ^ Y3,        X4 ^ Y4,        Z3 ^ Y5 ^ X8,   Y2 ^ X5 ^ Y8,   X2 ^ Y6 ^ X7,   Z4 ^ X6 ^ Y7,   Y3,             X4              },
    {0,              0,              X0,             Y0,             X1,             Z0,             Y1,             Z1,             X3 ^ Y3,        X4 ^ Y4,        Z2 ^ Y5 ^ X8,   Y2 ^ X5 ^ Y8,   X2 ^ Y6 ^ X7,   Z3 ^ X6 ^ Y7,   Y3,             X4              },
    {0,              0,              0,              X0,             Y0,             Z0,             X1,             Z1,             X3 ^ Y3,        X4 ^ Y4,        Z2 ^ Y5 ^ X8,   Y1 ^ X5 ^ Y8,   X2 ^ Y6 ^ X7,   Z3 ^ X6 ^ Y7,   Y2,             X3              },
    {0,              0,              0,              0,              X0,             Z0,             Y0,             Z1,             X3 ^ Y3,        X1 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Y1 ^ X5 ^ Y8,   X2 ^ Y6 ^ X7,   Z3 ^ X6 ^ Y7,   Y2,             X3              },
    {X0,             X1,             Z0,             Y0,             Y1,             Z1,             X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             Z2,             Y2,             X2,             Z3,             Y3,             X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             X1,             Z0,             Y1,             Z1,             Y2,             X2,             Z2,             Y3,             X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             Z0,             X1,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Y1,             Z1,             X2,             Z2,             X3,             X3 ^ Y3,        Z3,             Y2,             X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             Z2,             X2,             X3 ^ Y3,        Z3,             Y2,             X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             X1,             Z0,             Y1,             Z1,             X2,             X3 ^ Y3,        Z2,             Y2,             X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             Z0,             X1,             Z1,             X2,             X3 ^ Y3,        Z2,             Y1,             X3,             Z3,             Y2,             X4              },
    {0,              0,              0,              0,              X0,             Z0,             Y0,             Z1,             X1,             X3 ^ Y3,        Z2,             Y1,             X2,             Z3,             Y2,             X3              },
    {X0,             X1,             Z0,             Y0,             Y1,             Z1,             X2,             Z2,             Z3,             X3 ^ Y3,        X4 ^ Y4,        Y2,             X3,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             Z2,             Z3,             X3 ^ Y3,        X4 ^ Y4,        Y2,             X2,             Z4,             Y3,             X4              },
    {0,              0,              X0,             Y0,             X1,             Z0,             Y1,             Z1,             Z2,             X3 ^ Y3,        X4 ^ Y4,        Y2,             X2,             Z3,             Y3,             X4              },
    {0,              0,              0,              X0,             Y0,             Z0,             X1,             Z1,             Z2,             X3 ^ Y3,        X4 ^ Y4,        Y1,             X2,             Z3,             Y2,             X3              },
    {0,              0,              0,              0,              X0,             Z0,             Y0,             Z1,             Z2,             X3 ^ Y3,        X1 ^ X4 ^ Y4,   Y1,             X2,             Z3,             Y2,             X3              },
    {X0,             X1,             Z0,             Y0,             Y1,             Z1,             X2,             Z2,             Z3,             X3 ^ Y3,        X4 ^ Y4,        X5 ^ Y5,        Y2,             X3,             Z4,             Y4              },
    {0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             Z2,             Y2,             X3 ^ Y3,        X4 ^ Y4,        Z3 ^ X5 ^ Y5,   X2,             Z4,             Y3,             X4              },
    {0,              0,              X0,             Y0,             X1,             Z0,             Y1,             Z1,             Y2,             X3 ^ Y3,        X4 ^ Y4,        Z2 ^ X5 ^ Y5,   X2,             Z3,             Y3,             X4              },
    {0,              0,              0,              X0,             Y0,             Z0,             X1,             Z1,             Y1,             X3 ^ Y3,        X4 ^ Y4,        Z2 ^ X5 ^ Y5,   X2,             Z3,             Y2,             X3              },
    {0,              0,              0,              0,              X0,             Z0,             Y0,             Z1,             Y1,             X3 ^ Y3,        X1 ^ X4 ^ Y4,   Z2 ^ X5 ^ Y5,   X2,             Z3,             Y2,             X3              },
    {X0,             X1,             Z0,             Y0,             Y1,             Z1,             X2,             Z2,             Z3,             X3 ^ Y3,        X4 ^ Y4,        Y2 ^ Y5 ^ X6,   X5 ^ Y6,        Y3,             X4,             Z4              },
    {0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             Z2,             X2,             X3 ^ Y3,        X4 ^ Y4,        Z3 ^ Y5 ^ X6,   Y2 ^ X5 ^ Y6,   Z4,             Y3,             X4              },
    {0,              0,              X0,             Y0,             X1,             Z0,             Y1,             Z1,             X2,             X3 ^ Y3,        X4 ^ Y4,        Z2 ^ Y5 ^ X6,   Y2 ^ X5 ^ Y6,   Z3,             Y3,             X4              },
    {0,              0,              0,              X0,             Y0,             Z0,             X1,             Z1,             X2,             X3 ^ Y3,        X4 ^ Y4,        Z2 ^ Y5 ^ X6,   Y1 ^ X5 ^ Y6,   Z3,             Y2,             X3              },
    {0,              0,              0,              0,              X0,             Z0,             Y0,             Z1,             X2,             X3 ^ Y3,        X1 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X6,   Y1 ^ X5 ^ Y6,   Z3,             Y2,             X3              },
    {X0,             X1,             Z0,             Y0,             Y1,             Z1,             X2,             Z2,             Y3,             X3 ^ Y3,        X4 ^ Y4,        Y2 ^ Y5 ^ X7,   X5 ^ Y7,        Z3 ^ X6 ^ Y6,   X4,             Z4              },
    {0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             Z2,             Z4,             X3 ^ Y3,        X4 ^ Y4,        Z3 ^ Y5 ^ X7,   Y2 ^ X5 ^ Y7,   X2 ^ X6 ^ Y6,   Y3,             X4              },
    {0,              0,              X0,             Y0,             X1,             Z0,             Y1,             Z1,             Z3,             X3 ^ Y3,        X4 ^ Y4,        Z2 ^ Y5 ^ X7,   Y2 ^ X5 ^ Y7,   X2 ^ X6 ^ Y6,   Y3,             X4              },
    {0,              0,              0,              X0,             Y0,             Z0,             X1,             Z1,             Z3,             X3 ^ Y3,        X4 ^ Y4,        Z2 ^ Y5 ^ X7,   Y1 ^ X5 ^ Y7,   X2 ^ X6 ^ Y6,   Y2,             X3              },
    {0,              0,              0,              0,              X0,             Z0,             Y0,             Z1,             Z3,             X3 ^ Y3,        X1 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Y1 ^ X5 ^ Y7,   X2 ^ X6 ^ Y6,   Y2,             X3              },
    {X0,             X1,             Z0,             Y0,             Y1,             Z1,             X2,             Z2,             Y3,             X3 ^ Y3,        X4 ^ Y4,        Y2 ^ Y5 ^ X8,   X5 ^ Y8,        Z3 ^ Y6 ^ X7,   Z4 ^ X6 ^ Y7,   X4              },
    {0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             Z2,             Y3,             X3 ^ Y3,        X4 ^ Y4,        Z3 ^ Y5 ^ X8,   Y2 ^ X5 ^ Y8,   X2 ^ Y6 ^ X7,   Z4 ^ X6 ^ Y7,   X4              },
    {0,              0,              X0,             Y0,             X1,             Z0,             Y1,             Z1,             Y3,             X3 ^ Y3,        X4 ^ Y4,        Z2 ^ Y5 ^ X8,   Y2 ^ X5 ^ Y8,   X2 ^ Y6 ^ X7,   Z3 ^ X6 ^ Y7,   X4              },
    {0,              0,              0,              X0,             Y0,             Z0,             X1,             Z1,             Y2,             X3 ^ Y3,        X4 ^ Y4,        Z2 ^ Y5 ^ X8,   Y1 ^ X5 ^ Y8,   X2 ^ Y6 ^ X7,   Z3 ^ X6 ^ Y7,   X3              },
    {0,              0,              0,              0,              X0,             Z0,             Y0,             Z1,             Y2,             X3 ^ Y3,        X1 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Y1 ^ X5 ^ Y8,   X2 ^ Y6 ^ X7,   Z3 ^ X6 ^ Y7,   X3              },
    {X0,             X1,             Z0,             Y0,             Y1,             Z1,             X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             Z2,             Y2,             X2,             Z3,             Y3,             X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             X1,             Z0,             Y1,             Z1,             Y2,             X2,             Z2,             Y3,             X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             Z0,             X1,             Z1,             Y1,             X2,             Z2,             Y2,             X3,             Z3,             Y3,             X4              },
    {0,              0,              0,              0,              X0,             Z0,             Y0,             Z1,             Y1,             X1,             Z2,             Y2,             X2,             Z3,             Y3,             X3              },
    {X0,             X1,             Z0,             Y0,             Y1,             Z1,             X2,             Z2,             X3,             Z3,             X3 ^ Y3,        Y2,             X4,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             Z2,             X2,             Z3,             X3 ^ Y3,        Y2,             X3,             Z4,             Y4,             X4              },
    {0,              0,              X0,             Y0,             X1,             Z0,             Y1,             Z1,             X2,             Z2,             X3 ^ Y3,        Y2,             X3,             Z3,             Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             Z0,             X1,             Z1,             X2,             Z2,             X3 ^ Y3,        Y1,             X3,             Z3,             Y2,             X4              },
    {0,              0,              0,              0,              X0,             Z0,             Y0,             Z1,             X1,             Z2,             X3 ^ Y3,        Y1,             X2,             Z3,             Y2,             X3              },
    {X0,             X1,             Z0,             Y0,             Y1,             Z1,             X2,             Z2,             Z3,             Y2,             X3 ^ Y3,        X4 ^ Y4,        X3,             Z4,             Y4,             X5              },
    {0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             Z2,             Z3,             Y2,             X3 ^ Y3,        X4 ^ Y4,        X2,             Z4,             Y3,             X4              },
    {0,              0,              X0,             Y0,             X1,             Z0,             Y1,             Z1,             Z2,             Y2,             X3 ^ Y3,        X4 ^ Y4,        X2,             Z3,             Y3,             X4              },
    {0,              0,              0,              X0,             Y0,             Z0,             X1,             Z1,             Z2,             Y1,             X3 ^ Y3,        X4 ^ Y4,        X2,             Z3,             Y2,             X3              },
    {0,              0,              0,              0,              X0,             Z0,             Y0,             Z1,             Z2,             Y1,             X3 ^ Y3,        X1 ^ X4 ^ Y4,   X2,             Z3,             Y2,             X3              },
    {X0,             X1,             Z0,             Y0,             Y1,             Z1,             X2,             Z2,             Z3,             Y2,             X3 ^ Y3,        X4 ^ Y4,        X5 ^ Y5,        X3,             Z4,             Y4              },
    {0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             Z2,             Y2,             X2,             X3 ^ Y3,        X4 ^ Y4,        Z3 ^ X5 ^ Y5,   Z4,             Y3,             X4              },
    {0,              0,              X0,             Y0,             X1,             Z0,             Y1,             Z1,             Y2,             X2,             X3 ^ Y3,        X4 ^ Y4,        Z2 ^ X5 ^ Y5,   Z3,             Y3,             X4              },
    {0,              0,              0,              X0,             Y0,             Z0,             X1,             Z1,             Y1,             X2,             X3 ^ Y3,        X4 ^ Y4,        Z2 ^ X5 ^ Y5,   Z3,             Y2,             X3              },
    {0,              0,              0,              0,              X0,             Z0,             Y0,             Z1,             Y1,             X2,             X3 ^ Y3,        X1 ^ X4 ^ Y4,   Z2 ^ X5 ^ Y5,   Z3,             Y2,             X3              },
    {X0,             X1,             Z0,             Y0,             Y1,             Z1,             X2,             Z2,             Z3,             Y3,             X3 ^ Y3,        X4 ^ Y4,        Y2 ^ Y5 ^ X6,   X5 ^ Y6,        X4,             Z4              },
    {0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             Z2,             X2,             Z4,             X3 ^ Y3,        X4 ^ Y4,        Z3 ^ Y5 ^ X6,   Y2 ^ X5 ^ Y6,   Y3,             X4              },
    {0,              0,              X0,             Y0,             X1,             Z0,             Y1,             Z1,             X2,             Z3,             X3 ^ Y3,        X4 ^ Y4,        Z2 ^ Y5 ^ X6,   Y2 ^ X5 ^ Y6,   Y3,             X4              },
    {0,              0,              0,              X0,             Y0,             Z0,             X1,             Z1,             X2,             Z3,             X3 ^ Y3,        X4 ^ Y4,        Z2 ^ Y5 ^ X6,   Y1 ^ X5 ^ Y6,   Y2,             X3              },
    {0,              0,              0,              0,              X0,             Z0,             Y0,             Z1,             X2,             Z3,             X3 ^ Y3,        X1 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X6,   Y1 ^ X5 ^ Y6,   Y2,             X3              },
    {X0,             X1,             Z0,             Y0,             Y1,             Z1,             X2,             Z2,             Y3,             X4,             X3 ^ Y3,        X4 ^ Y4,        Y2 ^ Y5 ^ X7,   X5 ^ Y7,        Z3 ^ X6 ^ Y6,   Z4              },
    {0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             Z2,             Z4,             Y3,             X3 ^ Y3,        X4 ^ Y4,        Z3 ^ Y5 ^ X7,   Y2 ^ X5 ^ Y7,   X2 ^ X6 ^ Y6,   X4              },
    {0,              0,              X0,             Y0,             X1,             Z0,             Y1,             Z1,             Z3,             Y3,             X3 ^ Y3,        X4 ^ Y4,        Z2 ^ Y5 ^ X7,   Y2 ^ X5 ^ Y7,   X2 ^ X6 ^ Y6,   X4              },
    {0,              0,              0,              X0,             Y0,             Z0,             X1,             Z1,             Z3,             Y2,             X3 ^ Y3,        X4 ^ Y4,        Z2 ^ Y5 ^ X7,   Y1 ^ X5 ^ Y7,   X2 ^ X6 ^ Y6,   X3              },
    {0,              0,              0,              0,              X0,             Z0,             Y0,             Z1,             Z3,             Y2,             X3 ^ Y3,        X1 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Y1 ^ X5 ^ Y7,   X2 ^ X6 ^ Y6,   X3              },
    {X0,             X1,             Z0,             Y0,             Y1,             Z1,             X2,             Z2,             Y3,             X4,             X3 ^ Y3,        X4 ^ Y4,        Y2 ^ Y5 ^ X8,   X5 ^ Y8,        Z3 ^ Y6 ^ X7,   Z4 ^ X6 ^ Y7    },
    {0,              X0,             Z0,             Y0,             X1,             Z1,             Y1,             Z2,             Y3,             X4,             X3 ^ Y3,        X4 ^ Y4,        Z3 ^ Y5 ^ X8,   Y2 ^ X5 ^ Y8,   X2 ^ Y6 ^ X7,   Z4 ^ X6 ^ Y7    },
    {0,              0,              X0,             Y0,             X1,             Z0,             Y1,             Z1,             Y3,             X4,             X3 ^ Y3,        X4 ^ Y4,        Z2 ^ Y5 ^ X8,   Y2 ^ X5 ^ Y8,   X2 ^ Y6 ^ X7,   Z3 ^ X6 ^ Y7    },
    {0,              0,              0,              X0,             Y0,             Z0,             X1,             Z1,             Y2,             X3,             X3 ^ Y3,        X4 ^ Y4,        Z2 ^ Y5 ^ X8,   Y1 ^ X5 ^ Y8,   X2 ^ Y6 ^ X7,   Z3 ^ X6 ^ Y7    },
    {0,              0,              0,              0,              X0,             Z0,             Y0,             Z1,             Y2,             X3,             X3 ^ Y3,        X1 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Y1 ^ X5 ^ Y8,   X2 ^ Y6 ^ X7,   Z3 ^ X6 ^ Y7    },
};

// Meta data swizzle pattern
const UINT_64 HTILE_64K[][18]=
{
    {0,              0,              0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             X4,             X5,             Y5,             X6,             Z0 ^ X3 ^ Y3,   Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             X4,             X5,             Y5,             X6,             Z0 ^ X3 ^ Y3,   Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             X4,             X5,             Y5,             X6,             Z0 ^ X3 ^ Y3,   Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             X4,             X5,             Y5,             X6,             Z0 ^ X3 ^ Y3,   Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             X5,             Y5,             X6,             Y6,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X7,             Y7,             X8,             0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             X5,             Y5,             X6,             Y6,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X7,             Y7,             X8,             0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             X5,             Y5,             X6,             Y6,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X7,             Y7,             X8,             0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             X5,             Y5,             X6,             Y6,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X7,             Y7,             X8,             0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             Y5,             X6,             Y6,             X7,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y7,             X8,             Y8,             0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             Y5,             X6,             Y6,             X7,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y7,             X8,             Y8,             0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             Y5,             X6,             Y6,             X7,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y7,             X8,             Y8,             0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             Y5,             X6,             Y6,             X7,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y7,             X8,             Y8,             0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             X6,             Y6,             X7,             Y7,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X8,             Y8,             X9,             0,              0               },
    {0,              0,              0,              X3,             Y4,             X6,             Y6,             X7,             Y7,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X8,             Y8,             X9,             0,              0               },
    {0,              0,              0,              X3,             Y4,             X6,             Y6,             X7,             Y7,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X8,             Y8,             X9,             0,              0               },
    {0,              0,              0,              X3,             Y4,             X6,             Y6,             X7,             Y7,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   X5 ^ Y6,        X8,             Y8,             X9,             0,              0               },
    {0,              0,              0,              X3,             Y4,             Y6,             X7,             Y7,             X8,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y8,             X9,             Y9,             0               },
    {0,              0,              0,              X3,             Y4,             Y6,             X7,             Y7,             X8,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y8,             X9,             Y9,             0               },
    {0,              0,              0,              X3,             Y4,             Y6,             X7,             Y7,             X8,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   X6 ^ Y6,        Y8,             X9,             Y9,             0               },
    {0,              0,              0,              X3,             Y4,             Y6,             X7,             Y7,             X8,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   X5 ^ Y7,        X6 ^ Y6,        Y8,             X9,             Y9,             0               },
    {0,              0,              0,              X3,             Y4,             X7,             Y7,             X8,             Y8,             X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   X9,             Y9,             X10             },
    {0,              0,              0,              X3,             Y4,             X7,             Y7,             X8,             Y8,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Z0 ^ Y6 ^ X7,   X6 ^ Y7,        X9,             Y9,             X10             },
    {0,              0,              0,              X3,             Y4,             X7,             Y7,             X8,             Y8,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y6 ^ X7,        X6 ^ Y7,        X9,             Y9,             X10             },
    {0,              0,              0,              X3,             Y4,             X7,             Y7,             X8,             Y8,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X8,   X5 ^ Y8,        Y6 ^ X7,        X6 ^ Y7,        X9,             Y9,             X10             },
    {0,              0,              0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             Z0 ^ X3 ^ Y3,   X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             Z0 ^ X3 ^ Y3,   X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             Z0 ^ X3 ^ Y3,   X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             Z0 ^ X3 ^ Y3,   X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y7,             X8,             0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y7,             X8,             0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y7,             X8,             0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y7,             X8,             0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X8,             Y8,             0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X8,             Y8,             0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X8,             Y8,             0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X8,             Y8,             0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y8,             X9,             0,              0               },
    {0,              0,              0,              X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y8,             X9,             0,              0               },
    {0,              0,              0,              X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y8,             X9,             0,              0               },
    {0,              0,              0,              X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   X5 ^ Y6,        Y8,             X9,             0,              0               },
    {0,              0,              0,              X3,             Y4,             Y6,             X7,             Y7,             X8,             Y8,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X9,             Y9,             0               },
    {0,              0,              0,              X3,             Y4,             Y6,             X7,             Y7,             X8,             Y8,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X9,             Y9,             0               },
    {0,              0,              0,              X3,             Y4,             Y6,             X7,             Y7,             X8,             Y8,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   X6 ^ Y6,        X9,             Y9,             0               },
    {0,              0,              0,              X3,             Y4,             Y6,             X7,             Y7,             X8,             Y8,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   X5 ^ Y7,        X6 ^ Y6,        X9,             Y9,             0               },
    {0,              0,              0,              X3,             Y4,             X7,             Y7,             X8,             Y8,             X9,             X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y9,             X10             },
    {0,              0,              0,              X3,             Y4,             X7,             Y7,             X8,             Y8,             X9,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Z0 ^ Y6 ^ X7,   X6 ^ Y7,        Y9,             X10             },
    {0,              0,              0,              X3,             Y4,             X7,             Y7,             X8,             Y8,             X9,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y6 ^ X7,        X6 ^ Y7,        Y9,             X10             },
    {0,              0,              0,              X3,             Y4,             X7,             Y7,             X8,             Y8,             X9,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X8,   X5 ^ Y8,        Y6 ^ X7,        X6 ^ Y7,        Y9,             X10             },
    {0,              0,              0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             X7,             Z0 ^ X3 ^ Y3,   Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             X7,             Z0 ^ X3 ^ Y3,   Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             X7,             Z0 ^ X3 ^ Y3,   Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             X7,             Z0 ^ X3 ^ Y3,   Y7,             0,              0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X8,             0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X8,             0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X8,             0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X8,             0,              0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y8,             0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y8,             0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y8,             0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y8,             0,              0,              0               },
    {0,              0,              0,              X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X9,             0,              0               },
    {0,              0,              0,              X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X9,             0,              0               },
    {0,              0,              0,              X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X9,             0,              0               },
    {0,              0,              0,              X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             Y8,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   X5 ^ Y6,        X9,             0,              0               },
    {0,              0,              0,              X3,             Y4,             Y6,             X7,             Y7,             X8,             Y8,             X9,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y9,             0               },
    {0,              0,              0,              X3,             Y4,             Y6,             X7,             Y7,             X8,             Y8,             X9,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y9,             0               },
    {0,              0,              0,              X3,             Y4,             Y6,             X7,             Y7,             X8,             Y8,             X9,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   X6 ^ Y6,        Y9,             0               },
    {0,              0,              0,              X3,             Y4,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   X5 ^ Y7,        X6 ^ Y6,        Y9,             0               },
    {0,              0,              0,              X3,             Y4,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   X10             },
    {0,              0,              0,              X3,             Y4,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Z0 ^ Y6 ^ X7,   X6 ^ Y7,        X10             },
    {0,              0,              0,              X3,             Y4,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y6 ^ X7,        X6 ^ Y7,        X10             },
    {0,              0,              0,              X3,             Y4,             X7,             Y7,             X8,             Y8,             X9,             Y9,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X8,   X5 ^ Y8,        Y6 ^ X7,        X6 ^ Y7,        X10             },
};

const UINT_64 CMASK_64K[][17]=
{
    {X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             Z0 ^ X3 ^ Y3,   X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             Z0 ^ X3 ^ Y3,   X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             Z0 ^ X3 ^ Y3,   X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             Z0 ^ X3 ^ Y3,   X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X9,             0,              0,              0,              0               },
    {X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X9,             0,              0,              0,              0               },
    {X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X9,             0,              0,              0,              0               },
    {X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X9,             0,              0,              0,              0               },
    {X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   0,              0,              0,              0               },
    {X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   0,              0,              0,              0               },
    {X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   0,              0,              0,              0               },
    {X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   0,              0,              0,              0               },
    {X3,             Y4,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   0,              0,              0               },
    {X3,             Y4,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   0,              0,              0               },
    {X3,             Y4,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   0,              0,              0               },
    {X3,             Y4,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   0,              0,              0               },
    {X3,             Y4,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   0,              0               },
    {X3,             Y4,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   0,              0               },
    {X3,             Y4,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   0,              0               },
    {X3,             Y4,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Y6 ^ X7,        Z0 ^ X6 ^ Y7,   0,              0               },
    {X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Z0 ^ X3 ^ Y3,   Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Z0 ^ X3 ^ Y3,   Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Z0 ^ X3 ^ Y3,   Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Z0 ^ X3 ^ Y3,   Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X9,             0,              0,              0,              0               },
    {X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X9,             0,              0,              0,              0               },
    {X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X9,             0,              0,              0,              0               },
    {X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X9,             0,              0,              0,              0               },
    {X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   0,              0,              0,              0               },
    {X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   0,              0,              0,              0               },
    {X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   0,              0,              0,              0               },
    {X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   0,              0,              0,              0               },
    {X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   0,              0,              0               },
    {X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   0,              0,              0               },
    {X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   0,              0,              0               },
    {X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   0,              0,              0               },
    {X3,             Y4,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   0,              0               },
    {X3,             Y4,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   0,              0               },
    {X3,             Y4,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   0,              0               },
    {X3,             Y4,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   0,              0               },
    {X3,             Y4,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            Y10,            X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   0               },
    {X3,             Y4,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            Y10,            X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   0               },
    {X3,             Y4,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            Y10,            X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   0               },
    {X3,             Y4,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            Y10,            X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Y6 ^ X7,        Z0 ^ X6 ^ Y7,   0               },
    {X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             Z0 ^ X3 ^ Y3,   X9,             0,              0,              0,              0               },
    {X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             Z0 ^ X3 ^ Y3,   X9,             0,              0,              0,              0               },
    {X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             Z0 ^ X3 ^ Y3,   X9,             0,              0,              0,              0               },
    {X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             Z0 ^ X3 ^ Y3,   X9,             0,              0,              0,              0               },
    {X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   0,              0,              0,              0               },
    {X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   0,              0,              0,              0               },
    {X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   0,              0,              0,              0               },
    {X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   0,              0,              0,              0               },
    {X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   0,              0,              0               },
    {X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   0,              0,              0               },
    {X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   0,              0,              0               },
    {X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   0,              0,              0               },
    {X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   0,              0               },
    {X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   0,              0               },
    {X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   0,              0               },
    {X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   0,              0               },
    {X3,             Y4,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            Y10,            X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   0               },
    {X3,             Y4,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            Y10,            X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   0               },
    {X3,             Y4,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            Y10,            X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   0               },
    {X3,             Y4,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            Y10,            X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   0               },
    {X3,             Y4,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            Y10,            X11,            X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {X3,             Y4,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            Y10,            X11,            X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {X3,             Y4,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            Y10,            X11,            X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {X3,             Y4,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            Y10,            X11,            X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Y6 ^ X7,        Z0 ^ X6 ^ Y7    },
};

const UINT_64 DCC_64K_R_X[][17]=
{
    {0,              X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             0,              0,              0,              0               },
    {0,              Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             0,              0,              0,              0               },
    {0,              Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             0,              0,              0,              0               },
    {0,              X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0               },
    {0,              X3 ^ Y3,        X4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             0,              0,              0,              0               },
    {0,              X3 ^ Y3,        X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             0,              0,              0,              0               },
    {0,              Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             0,              0,              0,              0               },
    {0,              X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0               },
    {0,              X3 ^ Y3,        X4 ^ Y4,        X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             0,              0,              0,              0               },
    {0,              X3 ^ Y3,        X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             0,              0,              0,              0               },
    {0,              Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             0,              0,              0,              0               },
    {0,              X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0               },
    {0,              X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             0,              0,              0,              0               },
    {0,              Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             0,              0,              0,              0               },
    {0,              Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             0,              0,              0,              0               },
    {0,              X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0               },
    {0,              X4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Z0 ^ X3 ^ Y3,   Y8,             X9,             Y9,             0,              0,              0,              0               },
    {0,              Y4,             X4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             Z0 ^ X3 ^ Y3,   X8,             Y8,             X9,             0,              0,              0,              0               },
    {0,              X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             X7,             Z0 ^ X3 ^ Y3,   Y7,             X8,             Y8,             0,              0,              0,              0               },
    {0,              Y2,             X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             Z0 ^ X3 ^ Y3,   X7,             Y7,             X8,             0,              0,              0,              0               },
    {0,              X2,             Y2,             X3,             Y4,             X4,             X5,             Y5,             X6,             Z0 ^ X3 ^ Y3,   Y6,             X7,             Y7,             0,              0,              0,              0               },
    {0,              X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X9,             Y9,             0,              0,              0,              0               },
    {0,              Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y8,             X9,             0,              0,              0,              0               },
    {0,              X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X8,             Y8,             0,              0,              0,              0               },
    {0,              Y2,             X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y7,             X8,             0,              0,              0,              0               },
    {0,              X2,             Y2,             X3,             Y4,             X5,             Y5,             X6,             Y6,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X7,             Y7,             0,              0,              0,              0               },
    {0,              Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y9,             0,              0,              0,              0               },
    {0,              Y4,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X9,             0,              0,              0,              0               },
    {0,              X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y8,             0,              0,              0,              0               },
    {0,              Y2,             X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X8,             0,              0,              0,              0               },
    {0,              X2,             Y2,             X3,             Y4,             Y5,             X6,             Y6,             X7,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y7,             0,              0,              0,              0               },
    {0,              X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   0,              0,              0,              0               },
    {0,              Y4,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   0,              0,              0,              0               },
    {0,              X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   0,              0,              0,              0               },
    {0,              Y2,             X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   0,              0,              0,              0               },
    {0,              X2,             Y2,             X3,             Y4,             X6,             Y6,             X7,             Y7,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   0,              0,              0,              0               },
    {0,              Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   0,              0,              0               },
    {0,              Y4,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   0,              0,              0               },
    {0,              X3,             Y4,             Y6,             X7,             Y7,             X8,             Y8,             X9,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   0,              0,              0               },
    {0,              Y2,             X3,             Y4,             Y6,             X7,             Y7,             X8,             Y8,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   0,              0,              0               },
    {0,              X2,             X3,             Y4,             Y6,             X7,             Y7,             Y2,             X8,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   0,              0,              0               },
    {0,              X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            Y10,            X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   0,              0               },
    {0,              Y4,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   0,              0               },
    {0,              X3,             Y4,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   0,              0               },
    {0,              X3,             Y4,             X7,             Y7,             X8,             Y8,             Y2,             X9,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   0,              0               },
    {0,              X3,             Y4,             X7,             Y7,             X8,             Y8,             X2,             Y2,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   0,              0               },
    {0,              X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             0,              0,              0,              0               },
    {0,              Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             0,              0,              0,              0               },
    {0,              Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             0,              0,              0,              0               },
    {0,              X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0               },
    {0,              X3 ^ Y3,        X4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             0,              0,              0,              0               },
    {0,              X3 ^ Y3,        X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             0,              0,              0,              0               },
    {0,              Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             0,              0,              0,              0               },
    {0,              X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0               },
    {0,              X3 ^ Y3,        X4 ^ Y4,        X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             0,              0,              0,              0               },
    {0,              X3 ^ Y3,        X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             0,              0,              0,              0               },
    {0,              Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             0,              0,              0,              0               },
    {0,              X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0               },
    {0,              X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             0,              0,              0,              0               },
    {0,              Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             0,              0,              0,              0               },
    {0,              Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             0,              0,              0,              0               },
    {0,              X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0               },
    {0,              X4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             Z0 ^ X3 ^ Y3,   X9,             Y9,             0,              0,              0,              0               },
    {0,              Y4,             X4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Z0 ^ X3 ^ Y3,   Y8,             X9,             0,              0,              0,              0               },
    {0,              X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             Z0 ^ X3 ^ Y3,   X8,             Y8,             0,              0,              0,              0               },
    {0,              Y2,             X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             X7,             Z0 ^ X3 ^ Y3,   Y7,             X8,             0,              0,              0,              0               },
    {0,              X2,             Y2,             X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             Z0 ^ X3 ^ Y3,   X7,             Y7,             0,              0,              0,              0               },
    {0,              X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y9,             0,              0,              0,              0               },
    {0,              Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X9,             0,              0,              0,              0               },
    {0,              X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y8,             0,              0,              0,              0               },
    {0,              Y2,             X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   X8,             0,              0,              0,              0               },
    {0,              X2,             Y2,             X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   Y7,             0,              0,              0,              0               },
    {0,              Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   0,              0,              0,              0               },
    {0,              Y4,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   0,              0,              0,              0               },
    {0,              X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   0,              0,              0,              0               },
    {0,              Y2,             X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   0,              0,              0,              0               },
    {0,              X2,             Y2,             X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   0,              0,              0,              0               },
    {0,              X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   0,              0,              0               },
    {0,              Y4,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   0,              0,              0               },
    {0,              X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   0,              0,              0               },
    {0,              Y2,             X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   0,              0,              0               },
    {0,              X2,             Y2,             X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   0,              0,              0               },
    {0,              Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            Y10,            X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   0,              0               },
    {0,              Y4,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   0,              0               },
    {0,              X3,             Y4,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   0,              0               },
    {0,              Y2,             X3,             Y4,             Y6,             X7,             Y7,             X8,             Y8,             X9,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   0,              0               },
    {0,              X2,             X3,             Y4,             Y6,             X7,             Y7,             Y2,             X8,             Y8,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   0,              0               },
    {0,              X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            Y10,            X11,            X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   0               },
    {0,              Y4,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            Y10,            X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   0               },
    {0,              X3,             Y4,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   0               },
    {0,              X3,             Y4,             X7,             Y7,             X8,             Y8,             Y2,             X9,             Y9,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   0               },
    {0,              X3,             Y4,             X7,             Y7,             X8,             Y8,             X2,             Y2,             X9,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   0               },
    {0,              X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             0,              0,              0,              0               },
    {0,              Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             0,              0,              0,              0               },
    {0,              Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             0,              0,              0,              0               },
    {0,              X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0               },
    {0,              X3 ^ Y3,        X4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             0,              0,              0,              0               },
    {0,              X3 ^ Y3,        X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             0,              0,              0,              0               },
    {0,              Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             0,              0,              0,              0               },
    {0,              X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0               },
    {0,              X3 ^ Y3,        X4 ^ Y4,        X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             0,              0,              0,              0               },
    {0,              X3 ^ Y3,        X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             0,              0,              0,              0               },
    {0,              Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             0,              0,              0,              0               },
    {0,              X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0               },
    {0,              X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             0,              0,              0,              0               },
    {0,              Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             0,              0,              0,              0               },
    {0,              X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             0,              0,              0,              0               },
    {0,              Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             0,              0,              0,              0               },
    {0,              X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             0,              0,              0,              0               },
    {0,              X4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Z0 ^ X3 ^ Y3,   Y9,             0,              0,              0,              0               },
    {0,              Y4,             X4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             Z0 ^ X3 ^ Y3,   X9,             0,              0,              0,              0               },
    {0,              X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Z0 ^ X3 ^ Y3,   Y8,             0,              0,              0,              0               },
    {0,              Y2,             X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             Z0 ^ X3 ^ Y3,   X8,             0,              0,              0,              0               },
    {0,              X2,             Y2,             X3,             Y4,             X4,             X5,             Y5,             X6,             Y6,             X7,             Z0 ^ X3 ^ Y3,   Y7,             0,              0,              0,              0               },
    {0,              X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   0,              0,              0,              0               },
    {0,              Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   0,              0,              0,              0               },
    {0,              X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   0,              0,              0,              0               },
    {0,              Y2,             X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   0,              0,              0,              0               },
    {0,              X2,             Y2,             X3,             Y4,             X5,             Y5,             X6,             Y6,             X7,             Y7,             Z1 ^ X3 ^ Y3,   Z0 ^ X4 ^ Y4,   0,              0,              0,              0               },
    {0,              Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   0,              0,              0               },
    {0,              Y4,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   0,              0,              0               },
    {0,              X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   0,              0,              0               },
    {0,              Y2,             X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Y8,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   0,              0,              0               },
    {0,              X2,             Y2,             X3,             Y4,             Y5,             X6,             Y6,             X7,             Y7,             X8,             Z2 ^ X3 ^ Y3,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   0,              0,              0               },
    {0,              X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            Y10,            X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   0,              0               },
    {0,              Y4,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   0,              0               },
    {0,              X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   0,              0               },
    {0,              Y2,             X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X9,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   0,              0               },
    {0,              X2,             Y2,             X3,             Y4,             X6,             Y6,             X7,             Y7,             X8,             Y8,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   0,              0               },
    {0,              Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            Y10,            X11,            X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   0               },
    {0,              Y4,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            Y10,            X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   0               },
    {0,              X3,             Y4,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   0               },
    {0,              Y2,             X3,             Y4,             Y6,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   0               },
    {0,              X2,             X3,             Y4,             Y6,             X7,             Y7,             Y2,             X8,             Y8,             X9,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   0               },
    {0,              X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            Y10,            X11,            Y11,            X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {0,              Y4,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            Y10,            X11,            X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {0,              X3,             Y4,             X7,             Y7,             X8,             Y8,             X9,             Y9,             X10,            Y10,            X3 ^ Y3 ^ Z5,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {0,              X3,             Y4,             X7,             Y7,             X8,             Y8,             Y2,             X9,             Y9,             X10,            X3 ^ Y3 ^ Z4,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {0,              X3,             Y4,             X7,             Y7,             X8,             Y8,             X2,             Y2,             X9,             Y9,             X3 ^ Y3 ^ Z3,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7    },
};

// Rb plus color data swizzle pattern
const UINT_64 SW_256_S_RBPLUS[][8]=
{
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1              },
};

const UINT_64 SW_256_D_RBPLUS[][8]=
{
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1              },
};

const UINT_64 SW_4K_S_RBPLUS[][12]=
{
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3              },
};

const UINT_64 SW_4K_D_RBPLUS[][12]=
{
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3              },
};

const UINT_64 SW_4K_S_X_RBPLUS[][12]=
{
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z0 ^ Y3 ^ X4,   X4,             Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Z0 ^ Y2 ^ X3,   X3,             Y3,             X4              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Z0 ^ X2 ^ Y2,   X2,             Y3,             X3              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X5,        Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X5,        Z0 ^ X4 ^ Y4,   Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X4,        Z0 ^ X3 ^ Y4,   Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X4,        Z0 ^ X3 ^ Y3,   Y3,             X4              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X3,        Z0 ^ X2 ^ Y3,   Y3,             X3              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X6,        X4 ^ Y6,        Z0 ^ X5 ^ Y5,   X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Z0 ^ Y4 ^ X5,   X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        Z0 ^ X4 ^ Y4,   X4              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X5,        X3 ^ Y4,        Z0 ^ Y3 ^ X4,   X4              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X4,        X2 ^ Y4,        Z0 ^ X3 ^ Y3,   X3              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4,   Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4,   Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Z1 ^ Y2 ^ X4,   Z0 ^ X3 ^ Y3,   Y3,             X4              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Z1 ^ Y2 ^ X3,   Z0 ^ X2 ^ Y3,   Y3,             X3              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X6,        Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X6,        Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X5,        Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X5,        Z1 ^ X3 ^ Y4,   Z0 ^ Y3 ^ X4,   X4              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X4,        Z1 ^ X2 ^ Y4,   Z0 ^ X3 ^ Y3,   X3              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X7,        X4 ^ Y7,        Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X7,        X4 ^ Y6,        Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X6,        X3 ^ Y6,        Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X6,        X3 ^ Y5,        Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X5,        X2 ^ Y5,        Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z2 ^ Y3 ^ X6,   Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z2 ^ Y3 ^ X5,   Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ Z2 ^ X5,   Z1 ^ X3 ^ Y4,   Z0 ^ Y3 ^ X4,   X4              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ Z2 ^ X4,   Z1 ^ X2 ^ Y4,   Z0 ^ X3 ^ Y3,   X3              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X7,        Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X7,        Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X6,        Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X6,        Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X5,        X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X7,        X4 ^ Y7,        Z2 ^ Y5 ^ X6,   Z1 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X7,        X4 ^ Y6,        Z2 ^ Y4 ^ X6,   Z1 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X6,        X3 ^ Y6,        Z2 ^ Y4 ^ X5,   Z1 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X6,        X3 ^ Y5,        Z2 ^ Y3 ^ X5,   Z1 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X5,        X2 ^ Y5,        Z2 ^ Y3 ^ X4,   Z1 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ Z3 ^ X5,   X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X7,        Z3 ^ X4 ^ Y7,   Z2 ^ Y5 ^ X6,   Z1 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X7,        Z3 ^ X4 ^ Y6,   Z2 ^ Y4 ^ X6,   Z1 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X6,        X3 ^ Z3 ^ Y6,   Z2 ^ Y4 ^ X5,   Z1 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X6,        X3 ^ Z3 ^ Y5,   Z2 ^ Y3 ^ X5,   Z1 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X5,        X2 ^ Z3 ^ Y5,   Z2 ^ Y3 ^ X4,   Z1 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X7,        X4 ^ Y7,        Z3 ^ Y5 ^ X6,   Z2 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X7,        X4 ^ Y6,        Z3 ^ Y4 ^ X6,   Z2 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X6,        X3 ^ Y6,        Z3 ^ Y4 ^ X5,   Z2 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X6,        X3 ^ Y5,        Y3 ^ Z3 ^ X5,   Z2 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X5,        X2 ^ Y5,        Y3 ^ Z3 ^ X4,   Z2 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ Z3 ^ X5,   X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X7,        Z3 ^ X4 ^ Y7,   Z2 ^ Y5 ^ X6,   Z1 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X7,        Z3 ^ X4 ^ Y6,   Z2 ^ Y4 ^ X6,   Z1 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X6,        X3 ^ Z3 ^ Y6,   Z2 ^ Y4 ^ X5,   Z1 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X6,        X3 ^ Z3 ^ Y5,   Z2 ^ Y3 ^ X5,   Z1 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X5,        X2 ^ Z3 ^ Y5,   Z2 ^ Y3 ^ X4,   Z1 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z0 ^ X4 ^ Y4,   Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z0 ^ X3 ^ Y4,   Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             Z0 ^ X3 ^ Y3,   Y3,             X4              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             Z0 ^ X2 ^ Y3,   Y3,             X3              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Y6,        Z0 ^ X5 ^ Y5,   X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y5,        Z0 ^ Y4 ^ X5,   X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y5,        Z0 ^ X4 ^ Y4,   X4              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Y4,        Z0 ^ Y3 ^ X4,   X4              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Y4,        Z0 ^ X3 ^ Y3,   X3              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Y7,        Y5 ^ X6,        Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Y4 ^ X6,        Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y6,        Y4 ^ X5,        Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Y5,        Y3 ^ X5,        Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Y5,        Y3 ^ X4,        Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             Z1 ^ X3 ^ Y4,   Z0 ^ Y3 ^ X4,   X4              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             Z1 ^ X2 ^ Y4,   Z0 ^ X3 ^ Y3,   X3              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Y7,        Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y6,        Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Y5,        Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Y5,        Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Y7,        Y5 ^ X6,        Z1 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Y4 ^ X6,        Z1 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y6,        Y4 ^ X5,        Z1 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Y5,        Y3 ^ X5,        Z1 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Y5,        Y3 ^ X4,        Z1 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Y7,        Z2 ^ Y5 ^ X6,   Z1 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Z2 ^ Y4 ^ X6,   Z1 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y6,        Z2 ^ Y4 ^ X5,   Z1 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Y5,        Z2 ^ Y3 ^ X5,   Z1 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Y5,        Z2 ^ Y3 ^ X4,   Z1 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Y7,        Y5 ^ X6,        Z2 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Y4 ^ X6,        Z2 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y6,        Y4 ^ X5,        Z2 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Y5,        Y3 ^ X5,        Z2 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Y5,        Y3 ^ X4,        Z2 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Y7,        Z2 ^ Y5 ^ X6,   Z1 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Z2 ^ Y4 ^ X6,   Z1 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y6,        Z2 ^ Y4 ^ X5,   Z1 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Y5,        Z2 ^ Y3 ^ X5,   Z1 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Y5,        Z2 ^ Y3 ^ X4,   Z1 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Y7,        Y5 ^ X6,        Z2 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Y4 ^ X6,        Z2 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y6,        Y4 ^ X5,        Z2 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Y5,        Y3 ^ X5,        Z2 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Y5,        Y3 ^ X4,        Z2 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Y7,        Z2 ^ Y5 ^ X6,   Z1 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Z2 ^ Y4 ^ X6,   Z1 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y6,        Z2 ^ Y4 ^ X5,   Z1 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Y5,        Z2 ^ Y3 ^ X5,   Z1 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Y5,        Z2 ^ Y3 ^ X4,   Z1 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Z0 ^ X5 ^ Y5,   X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z0 ^ Y4 ^ X5,   X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z0 ^ X4 ^ Y4,   X4              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Z0 ^ Y3 ^ X4,   X4              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Z0 ^ X3 ^ Y3,   X3              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X6,        Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X5,        Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X4,        Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5         },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X5,        X4 ^ Y4         },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X4,        X3 ^ Y4         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X6,        Z1 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        Z1 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        Z1 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X5,        Z1 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X4,        Z1 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5         },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X5,        X4 ^ Y4         },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X4,        X3 ^ Y4         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X6,        Z1 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        Z1 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        Z1 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X5,        Z1 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X4,        Z1 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5         },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X5,        X4 ^ Y4         },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X4,        X3 ^ Y4         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X6,        Z1 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        Z1 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        Z1 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X5,        Z1 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X4,        Z1 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5         },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5         },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X5,        X4 ^ Y4         },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X4,        X3 ^ Y4         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X6,        Z1 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        Z1 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        Z1 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X5,        Z1 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X4,        Z1 ^ X3 ^ Y4    },
};

const UINT_64 SW_4K_D_X_RBPLUS[][12]=
{
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z0 ^ Y3 ^ X4,   X4,             Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z0 ^ Y2 ^ X3,   X3,             Y3,             X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z0 ^ X2 ^ Y2,   X2,             Y3,             X3              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X5,        Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X5,        Z0 ^ X4 ^ Y4,   Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ X4,        Z0 ^ X3 ^ Y4,   Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X4,        Z0 ^ X3 ^ Y3,   Y3,             X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X3,        Z0 ^ X2 ^ Y3,   Y3,             X3              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X6,        X4 ^ Y6,        Z0 ^ X5 ^ Y5,   X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Z0 ^ Y4 ^ X5,   X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ X5,        X3 ^ Y5,        Z0 ^ X4 ^ Y4,   X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X5,        X3 ^ Y4,        Z0 ^ Y3 ^ X4,   X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X4,        X2 ^ Y4,        Z0 ^ X3 ^ Y3,   X3              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4,   Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4,   Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z1 ^ Y2 ^ X4,   Z0 ^ X3 ^ Y3,   Y3,             X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z1 ^ Y2 ^ X3,   Z0 ^ X2 ^ Y3,   Y3,             X3              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X6,        Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X6,        Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ X5,        Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X5,        Z1 ^ X3 ^ Y4,   Z0 ^ Y3 ^ X4,   X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X4,        Z1 ^ X2 ^ Y4,   Z0 ^ X3 ^ Y3,   X3              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X7,        X4 ^ Y7,        Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X7,        X4 ^ Y6,        Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ X6,        X3 ^ Y6,        Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X6,        X3 ^ Y5,        Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X5,        X2 ^ Y5,        Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z2 ^ Y3 ^ X6,   Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Z2 ^ Y3 ^ X5,   Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ Z2 ^ X5,   Z1 ^ X3 ^ Y4,   Z0 ^ Y3 ^ X4,   X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ Z2 ^ X4,   Z1 ^ X2 ^ Y4,   Z0 ^ X3 ^ Y3,   X3              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X7,        Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X7,        Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ X6,        Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X6,        Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X5,        X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X7,        X4 ^ Y7,        Z2 ^ Y5 ^ X6,   Z1 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X7,        X4 ^ Y6,        Z2 ^ Y4 ^ X6,   Z1 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ X6,        X3 ^ Y6,        Z2 ^ Y4 ^ X5,   Z1 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X6,        X3 ^ Y5,        Z2 ^ Y3 ^ X5,   Z1 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X5,        X2 ^ Y5,        Z2 ^ Y3 ^ X4,   Z1 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ Z3 ^ X5,   X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X7,        Z3 ^ X4 ^ Y7,   Z2 ^ Y5 ^ X6,   Z1 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X7,        Z3 ^ X4 ^ Y6,   Z2 ^ Y4 ^ X6,   Z1 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ X6,        X3 ^ Z3 ^ Y6,   Z2 ^ Y4 ^ X5,   Z1 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X6,        X3 ^ Z3 ^ Y5,   Z2 ^ Y3 ^ X5,   Z1 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X5,        X2 ^ Z3 ^ Y5,   Z2 ^ Y3 ^ X4,   Z1 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X7,        X4 ^ Y7,        Z3 ^ Y5 ^ X6,   Z2 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X7,        X4 ^ Y6,        Z3 ^ Y4 ^ X6,   Z2 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ X6,        X3 ^ Y6,        Z3 ^ Y4 ^ X5,   Z2 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X6,        X3 ^ Y5,        Y3 ^ Z3 ^ X5,   Z2 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X5,        X2 ^ Y5,        Y3 ^ Z3 ^ X4,   Z2 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ Z3 ^ X5,   X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X7,        Z3 ^ X4 ^ Y7,   Z2 ^ Y5 ^ X6,   Z1 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X7,        Z3 ^ X4 ^ Y6,   Z2 ^ Y4 ^ X6,   Z1 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ X6,        X3 ^ Z3 ^ Y6,   Z2 ^ Y4 ^ X5,   Z1 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X6,        X3 ^ Z3 ^ Y5,   Z2 ^ Y3 ^ X5,   Z1 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X5,        X2 ^ Z3 ^ Y5,   Z2 ^ Y3 ^ X4,   Z1 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z0 ^ X4 ^ Y4,   Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             Z0 ^ X3 ^ Y4,   Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z0 ^ X3 ^ Y3,   Y3,             X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             Z0 ^ X2 ^ Y3,   Y3,             X3              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Y6,        Z0 ^ X5 ^ Y5,   X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y5,        Z0 ^ Y4 ^ X5,   X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3 ^ Y5,        Z0 ^ X4 ^ Y4,   X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y4,        Z0 ^ Y3 ^ X4,   X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Y4,        Z0 ^ X3 ^ Y3,   X3              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Y7,        Y5 ^ X6,        Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Y4 ^ X6,        Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3 ^ Y6,        Y4 ^ X5,        Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y5,        Y3 ^ X5,        Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Y5,        Y3 ^ X4,        Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z1 ^ X3 ^ Y4,   Z0 ^ Y3 ^ X4,   X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             Z1 ^ X2 ^ Y4,   Z0 ^ X3 ^ Y3,   X3              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Y7,        Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3 ^ Y6,        Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y5,        Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Y5,        Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Y7,        Y5 ^ X6,        Z1 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Y4 ^ X6,        Z1 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3 ^ Y6,        Y4 ^ X5,        Z1 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y5,        Y3 ^ X5,        Z1 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Y5,        Y3 ^ X4,        Z1 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Y7,        Z2 ^ Y5 ^ X6,   Z1 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Z2 ^ Y4 ^ X6,   Z1 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3 ^ Y6,        Z2 ^ Y4 ^ X5,   Z1 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y5,        Z2 ^ Y3 ^ X5,   Z1 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Y5,        Z2 ^ Y3 ^ X4,   Z1 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Y7,        Y5 ^ X6,        Z2 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Y4 ^ X6,        Z2 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3 ^ Y6,        Y4 ^ X5,        Z2 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y5,        Y3 ^ X5,        Z2 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Y5,        Y3 ^ X4,        Z2 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Y7,        Z2 ^ Y5 ^ X6,   Z1 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Z2 ^ Y4 ^ X6,   Z1 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3 ^ Y6,        Z2 ^ Y4 ^ X5,   Z1 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y5,        Z2 ^ Y3 ^ X5,   Z1 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Y5,        Z2 ^ Y3 ^ X4,   Z1 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Y7,        Y5 ^ X6,        Z2 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Y4 ^ X6,        Z2 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3 ^ Y6,        Y4 ^ X5,        Z2 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y5,        Y3 ^ X5,        Z2 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Y5,        Y3 ^ X4,        Z2 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Y7,        Z2 ^ Y5 ^ X6,   Z1 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Z2 ^ Y4 ^ X6,   Z1 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3 ^ Y6,        Z2 ^ Y4 ^ X5,   Z1 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y5,        Z2 ^ Y3 ^ X5,   Z1 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Y5,        Z2 ^ Y3 ^ X4,   Z1 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Z0 ^ X5 ^ Y5,   X5              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z0 ^ Y4 ^ X5,   X5              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Z0 ^ X4 ^ Y4,   X4              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z0 ^ Y3 ^ X4,   X4              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Z0 ^ X3 ^ Y3,   X3              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X6,        Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X5,        Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X5,        Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X4,        Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X5,        X4 ^ Y4         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X4,        X3 ^ Y4         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X6,        Z1 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        Z1 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X5,        Z1 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X5,        Z1 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X4,        Z1 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X5,        X4 ^ Y4         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X4,        X3 ^ Y4         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X6,        Z1 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        Z1 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X5,        Z1 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X5,        Z1 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X4,        Z1 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X5,        X4 ^ Y4         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X4,        X3 ^ Y4         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X6,        Z1 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        Z1 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X5,        Z1 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X5,        Z1 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X4,        Z1 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X5,        X4 ^ Y4         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X4,        X3 ^ Y4         },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X6,        Z1 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        Z1 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X5,        Z1 ^ X4 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X5,        Z1 ^ X4 ^ Y4    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X4,        Z1 ^ X3 ^ Y4    },
};

const UINT_64 SW_64K_S_RBPLUS[][16]=
{
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
};

const UINT_64 SW_64K_D_RBPLUS[][16]=
{
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
};

const UINT_64 SW_64K_S_T_RBPLUS[][16]=
{
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X4 ^ Y4,        X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X4,        X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             X3 ^ Y3,        X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X3,        X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             X2 ^ Y2,        X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X4,        X3 ^ Y3,        Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X3,        X2 ^ Y3,        Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X5,        X3 ^ Y4,        Y3 ^ X4,        X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X4,        X2 ^ Y4,        X3 ^ Y3,        X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X4,        X3 ^ Y3,        Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X3,        X2 ^ Y3,        Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X5,        X3 ^ Y4,        Y3 ^ X4,        X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X4,        X2 ^ Y4,        X3 ^ Y3,        X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X7,        X4 ^ Y7,        Y5 ^ X6,        X5 ^ Y6,        Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X7,        X4 ^ Y6,        Y4 ^ X6,        X5 ^ Y5,        Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X6,        X3 ^ Y6,        Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X6,        X3 ^ Y5,        Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X5,        X2 ^ Y5,        Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X5,        X3 ^ Y4,        Y3 ^ X4,        X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X4,        X2 ^ Y4,        X3 ^ Y3,        X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X7,        X4 ^ Y7,        Y5 ^ X6,        X5 ^ Y6,        Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X7,        X4 ^ Y6,        Y4 ^ X6,        X5 ^ Y5,        Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X6,        X3 ^ Y6,        Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X6,        X3 ^ Y5,        Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X5,        X2 ^ Y5,        Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X7,        X5 ^ Y7,        X6 ^ Y6,        X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6,        X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X7,        X4 ^ Y7,        Y5 ^ X6,        X5 ^ Y6,        Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X7,        X4 ^ Y6,        Y4 ^ X6,        X5 ^ Y5,        Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X6,        X3 ^ Y6,        Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X6,        X3 ^ Y5,        Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X5,        X2 ^ Y5,        Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X7,        X5 ^ Y7,        X6 ^ Y6,        X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6,        X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6 ^ X7,        X6 ^ Y7,        Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X7,        X6 ^ Y6,        Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X7,        X5 ^ Y7,        X6 ^ Y6,        X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6,        X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6 ^ X7,        X6 ^ Y7,        Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X7,        X6 ^ Y6,        Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Y5,        Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y4,        Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y4,        Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Y3,        Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Y3,        Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Y4,        Y3 ^ X4,        X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Y4,        X3 ^ Y3,        X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Y7,        Y5 ^ X6,        X5 ^ Y6,        Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Y4 ^ X6,        X5 ^ Y5,        Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y6,        Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Y5,        Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Y5,        Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Y4,        Y3 ^ X4,        X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Y4,        X3 ^ Y3,        X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Y7,        Y5 ^ X6,        X5 ^ Y6,        Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Y4 ^ X6,        X5 ^ Y5,        Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y6,        Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Y5,        Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Y5,        Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X7,        X5 ^ Y7,        X6 ^ Y6,        X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6,        X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Y7,        Y5 ^ X6,        X5 ^ Y6,        Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Y4 ^ X6,        X5 ^ Y5,        Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y6,        Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Y5,        Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Y5,        Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X7,        X5 ^ Y7,        X6 ^ Y6,        X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6,        X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6 ^ X7,        X6 ^ Y7,        Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X7,        X6 ^ Y6,        Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X7,        X5 ^ Y7,        X6 ^ Y6,        X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6,        X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6 ^ X7,        X6 ^ Y7,        Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X7,        X6 ^ Y6,        Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             X7 ^ Y7,        X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6 ^ X7,        X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             X6 ^ Y6,        X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X6,        X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             X5 ^ Y5,        X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6 ^ X7,        X6 ^ Y7,        Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X7,        X6 ^ Y6,        Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             X7 ^ Y7,        X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6 ^ X7,        X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             X6 ^ Y6,        X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X6,        X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             X5 ^ Y5,        X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             X5 ^ Y5,        X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X5,        X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             X4 ^ Y4,        X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X4,        X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             X3 ^ Y3,        X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X7,        X5 ^ Y7,        X6 ^ Y6,        X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6,        X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X7,        X5 ^ Y7,        X6 ^ Y6,        X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6,        X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6 ^ X7,        X6 ^ Y7,        Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X7,        X6 ^ Y6,        Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X7,        X5 ^ Y7,        X6 ^ Y6,        X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6,        X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6 ^ X7,        X6 ^ Y7,        Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X7,        X6 ^ Y6,        Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             X7 ^ Y7,        X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6 ^ X7,        X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             X6 ^ Y6,        X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X6,        X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             X5 ^ Y5,        X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6 ^ X7,        X6 ^ Y7,        Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X7,        X6 ^ Y6,        Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             X7 ^ Y7,        X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6 ^ X7,        X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             X6 ^ Y6,        X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X6,        X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             X5 ^ Y5,        X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             X7 ^ Y7,        X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6 ^ X7,        X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             X6 ^ Y6,        X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X6,        X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             X5 ^ Y5,        X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
};

const UINT_64 SW_64K_D_T_RBPLUS[][16]=
{
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             X4 ^ Y4,        X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X4,        X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3 ^ Y3,        X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X3,        X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2 ^ Y2,        X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X4,        X3 ^ Y3,        Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X3,        X2 ^ Y3,        Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X5,        X3 ^ Y4,        Y3 ^ X4,        X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X4,        X2 ^ Y4,        X3 ^ Y3,        X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X4,        X3 ^ Y3,        Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X3,        X2 ^ Y3,        Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X5,        X3 ^ Y4,        Y3 ^ X4,        X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X4,        X2 ^ Y4,        X3 ^ Y3,        X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X7,        X4 ^ Y7,        Y5 ^ X6,        X5 ^ Y6,        Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X7,        X4 ^ Y6,        Y4 ^ X6,        X5 ^ Y5,        Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ X6,        X3 ^ Y6,        Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X6,        X3 ^ Y5,        Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X5,        X2 ^ Y5,        Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X5,        X3 ^ Y4,        Y3 ^ X4,        X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X4,        X2 ^ Y4,        X3 ^ Y3,        X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X7,        X4 ^ Y7,        Y5 ^ X6,        X5 ^ Y6,        Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X7,        X4 ^ Y6,        Y4 ^ X6,        X5 ^ Y5,        Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ X6,        X3 ^ Y6,        Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X6,        X3 ^ Y5,        Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X5,        X2 ^ Y5,        Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X7,        X5 ^ Y7,        X6 ^ Y6,        X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6,        X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X7,        X4 ^ Y7,        Y5 ^ X6,        X5 ^ Y6,        Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X7,        X4 ^ Y6,        Y4 ^ X6,        X5 ^ Y5,        Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ X6,        X3 ^ Y6,        Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X6,        X3 ^ Y5,        Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X5,        X2 ^ Y5,        Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X7,        X5 ^ Y7,        X6 ^ Y6,        X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6,        X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6 ^ X7,        X6 ^ Y7,        Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X7,        X6 ^ Y6,        Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X7,        X5 ^ Y7,        X6 ^ Y6,        X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6,        X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6 ^ X7,        X6 ^ Y7,        Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X7,        X6 ^ Y6,        Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Y5,        Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y4,        Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3 ^ Y4,        Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y3,        Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Y3,        Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y4,        Y3 ^ X4,        X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Y4,        X3 ^ Y3,        X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Y7,        Y5 ^ X6,        X5 ^ Y6,        Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Y4 ^ X6,        X5 ^ Y5,        Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3 ^ Y6,        Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y5,        Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Y5,        Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y4,        Y3 ^ X4,        X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Y4,        X3 ^ Y3,        X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Y7,        Y5 ^ X6,        X5 ^ Y6,        Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Y4 ^ X6,        X5 ^ Y5,        Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3 ^ Y6,        Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y5,        Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Y5,        Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X7,        X5 ^ Y7,        X6 ^ Y6,        X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6,        X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Y7,        Y5 ^ X6,        X5 ^ Y6,        Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Y4 ^ X6,        X5 ^ Y5,        Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3 ^ Y6,        Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y5,        Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Y5,        Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X7,        X5 ^ Y7,        X6 ^ Y6,        X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6,        X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6 ^ X7,        X6 ^ Y7,        Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X7,        X6 ^ Y6,        Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X7,        X5 ^ Y7,        X6 ^ Y6,        X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6,        X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6 ^ X7,        X6 ^ Y7,        Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X7,        X6 ^ Y6,        Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             X7 ^ Y7,        X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6 ^ X7,        X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             X6 ^ Y6,        X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X6,        X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             X5 ^ Y5,        X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6 ^ X7,        X6 ^ Y7,        Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X7,        X6 ^ Y6,        Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             X7 ^ Y7,        X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6 ^ X7,        X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             X6 ^ Y6,        X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X6,        X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             X5 ^ Y5,        X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             X5 ^ Y5,        X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X5,        X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             X4 ^ Y4,        X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X4,        X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             X3 ^ Y3,        X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X7,        X5 ^ Y7,        X6 ^ Y6,        X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6,        X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X5,        X4 ^ Y4,        Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X4,        X3 ^ Y4,        Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X7,        X5 ^ Y7,        X6 ^ Y6,        X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6,        X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6 ^ X7,        X6 ^ Y7,        Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X7,        X6 ^ Y6,        Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X7,        X5 ^ Y7,        X6 ^ Y6,        X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6,        X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X6,        X4 ^ Y6,        X5 ^ Y5,        X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Y4 ^ X5,        X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        X4 ^ Y4,        X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6 ^ X7,        X6 ^ Y7,        Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X7,        X6 ^ Y6,        Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             X7 ^ Y7,        X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6 ^ X7,        X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             X6 ^ Y6,        X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X6,        X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             X5 ^ Y5,        X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6 ^ X7,        X6 ^ Y7,        Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X7,        X6 ^ Y6,        Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5 ^ X6,        X5 ^ Y6,        Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        X5 ^ Y5,        Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        X4 ^ Y5,        Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             X7 ^ Y7,        X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6 ^ X7,        X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             X6 ^ Y6,        X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X6,        X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             X5 ^ Y5,        X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             X7 ^ Y7,        X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6 ^ X7,        X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             X6 ^ Y6,        X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5 ^ X6,        X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             X5 ^ Y5,        X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
};

const UINT_64 SW_64K_S_X_RBPLUS[][16]=
{
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z0 ^ Y3 ^ X4,   X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Z0 ^ Y2 ^ X3,   X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Z0 ^ X2 ^ Y2,   X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X5,        Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X5,        Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X4,        Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X4,        Z0 ^ X3 ^ Y3,   Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X3,        Z0 ^ X2 ^ Y3,   Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X6,        X4 ^ Y6,        Z0 ^ X5 ^ Y5,   X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Z0 ^ Y4 ^ X5,   X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X5,        X3 ^ Y4,        Z0 ^ Y3 ^ X4,   X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X4,        X2 ^ Y4,        Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Z1 ^ Y2 ^ X4,   Z0 ^ X3 ^ Y3,   Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Z1 ^ Y2 ^ X3,   Z0 ^ X2 ^ Y3,   Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X6,        Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X6,        Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X5,        Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X5,        Z1 ^ X3 ^ Y4,   Z0 ^ Y3 ^ X4,   X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X4,        Z1 ^ X2 ^ Y4,   Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X7,        X4 ^ Y7,        Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X7,        X4 ^ Y6,        Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X6,        X3 ^ Y6,        Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X6,        X3 ^ Y5,        Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X5,        X2 ^ Y5,        Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z2 ^ Y3 ^ X6,   Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Z2 ^ Y3 ^ X5,   Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ Z2 ^ X5,   Z1 ^ X3 ^ Y4,   Z0 ^ Y3 ^ X4,   X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ Z2 ^ X4,   Z1 ^ X2 ^ Y4,   Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X7,        Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X7,        Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X6,        Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X6,        Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X5,        X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X8,        X4 ^ Y8,        Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X8,        X4 ^ Y7,        Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X7,        X3 ^ Y7,        Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X7,        X3 ^ Y6,        Z2 ^ Y3 ^ X6,   Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X6,        X2 ^ Y6,        Z2 ^ Y3 ^ X5,   Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ Z3 ^ X5,   X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X8,        Z3 ^ X4 ^ Y8,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X8,        Z3 ^ X4 ^ Y7,   Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X7,        X3 ^ Z3 ^ Y7,   Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X7,        X3 ^ Z3 ^ Y6,   Z2 ^ Y3 ^ X6,   Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X6,        X2 ^ Z3 ^ Y6,   Z2 ^ Y3 ^ X5,   Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X9,        X4 ^ Y9,        Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X9,        X4 ^ Y8,        Z3 ^ Y4 ^ X8,   Z2 ^ X5 ^ Y7,   Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6,   Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X8,        X3 ^ Y8,        Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X8,        X3 ^ Y7,        Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X7,        X2 ^ Y7,        Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ Z4 ^ X8,   Z3 ^ X4 ^ Y8,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ Z4 ^ X8,   Z3 ^ X4 ^ Y7,   Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ Z4 ^ X7,   X3 ^ Z3 ^ Y7,   Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ Z4 ^ X7,   X3 ^ Z3 ^ Y6,   Z2 ^ Y3 ^ X6,   Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ Z4 ^ X6,   X2 ^ Z3 ^ Y6,   Z2 ^ Y3 ^ X5,   Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X9,        X4 ^ Z4 ^ Y9,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X9,        X4 ^ Z4 ^ Y8,   Z3 ^ Y4 ^ X8,   Z2 ^ X5 ^ Y7,   Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6,   Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3 ^ X8,        X3 ^ Z4 ^ Y8,   Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2 ^ X8,        X3 ^ Z4 ^ Y7,   Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2 ^ X7,        X2 ^ Z4 ^ Y7,   Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             Z0 ^ X3 ^ Y3,   Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             Z0 ^ X2 ^ Y3,   Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Y6,        Z0 ^ X5 ^ Y5,   X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y5,        Z0 ^ Y4 ^ X5,   X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y5,        Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Y4,        Z0 ^ Y3 ^ X4,   X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Y4,        Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Y7,        Y5 ^ X6,        Z0 ^ X5 ^ Y6,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Y4 ^ X6,        Z0 ^ X5 ^ Y5,   Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y6,        Y4 ^ X5,        Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Y5,        Y3 ^ X5,        Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Y5,        Y3 ^ X4,        Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             Z1 ^ X3 ^ Y4,   Z0 ^ Y3 ^ X4,   X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             Z1 ^ X2 ^ Y4,   Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Y7,        Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y6,        Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Y5,        Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Y5,        Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Y8,        Y5 ^ X7,        Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y7,        Y4 ^ X7,        Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y7,        Y4 ^ X6,        Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Y6,        Y3 ^ X6,        Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Y6,        Y3 ^ X5,        Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Y8,        Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y7,        Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y7,        Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Y6,        Z2 ^ Y3 ^ X6,   Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Y6,        Z2 ^ Y3 ^ X5,   Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Y9,        Y5 ^ X8,        Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y8,        Y4 ^ X8,        Z2 ^ X5 ^ Y7,   Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6,   Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y8,        Y4 ^ X7,        Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Y7,        Y3 ^ X7,        Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Y7,        Y3 ^ X6,        Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             Z3 ^ X4 ^ Y8,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z3 ^ X4 ^ Y7,   Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Z3 ^ Y7,   Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Z3 ^ Y6,   Z2 ^ Y3 ^ X6,   Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Z3 ^ Y6,   Z2 ^ Y3 ^ X5,   Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Y9,        Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y8,        Z3 ^ Y4 ^ X8,   Z2 ^ X5 ^ Y7,   Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6,   Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y8,        Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Y7,        Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Y7,        Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^Y10,        Y5 ^ X9,        Z3 ^ X5 ^ Y9,   Z2 ^ Y6 ^ X8,   Z1 ^ X6 ^ Y8,   Z0 ^ X7 ^ Y7,   X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y9,        Y4 ^ X9,        Z3 ^ X5 ^ Y8,   Z2 ^ Y5 ^ X8,   Z1 ^ X6 ^ Y7,   Z0 ^ Y6 ^ X7,   X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y9,        Y4 ^ X8,        Z3 ^ X4 ^ Y8,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Y8,        Y3 ^ X8,        Z3 ^ X4 ^ Y7,   Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Y8,        Y3 ^ X7,        X3 ^ Z3 ^ Y7,   Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^ Z4 ^ Y9,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Z4 ^ Y8,   Z3 ^ Y4 ^ X8,   Z2 ^ X5 ^ Y7,   Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6,   Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Z4 ^ Y8,   Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Z4 ^ Y7,   Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Z4 ^ Y7,   Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4 ^Y10,        Z4 ^ Y5 ^ X9,   Z3 ^ X5 ^ Y9,   Z2 ^ Y6 ^ X8,   Z1 ^ X6 ^ Y8,   Z0 ^ X7 ^ Y7,   X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y9,        Y4 ^ Z4 ^ X9,   Z3 ^ X5 ^ Y8,   Z2 ^ Y5 ^ X8,   Z1 ^ X6 ^ Y7,   Z0 ^ Y6 ^ X7,   X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3 ^ Y9,        Y4 ^ Z4 ^ X8,   Z3 ^ X4 ^ Y8,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3 ^ Y8,        Y3 ^ Z4 ^ X8,   Z3 ^ X4 ^ Y7,   Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2 ^ Y8,        Y3 ^ Z4 ^ X7,   X3 ^ Z3 ^ Y7,   Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Z0 ^ X5 ^ Y5,   X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z0 ^ Y4 ^ X5,   X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Z0 ^ Y3 ^ X4,   X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X6,        Z0 ^ X5 ^ Y6,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        Z0 ^ X5 ^ Y5,   Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X5,        Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X5,        Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X4,        Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X7,        X5 ^ Y7,        Z0 ^ X6 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        X5 ^ Y6,        Z0 ^ Y5 ^ X6,   X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X6,        X4 ^ Y6,        Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Z0 ^ Y4 ^ X5,   X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X7,        Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X6,        Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X6,        Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X5,        Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X8,        X5 ^ Y8,        Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X8,        X5 ^ Y7,        Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6,   Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X7,        X4 ^ Y7,        Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X7,        X4 ^ Y6,        Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X6,        X3 ^ Y6,        Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Z2 ^ Y3 ^ X6,   Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Z2 ^ Y3 ^ X5,   Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X8,        Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X8,        Z2 ^ X5 ^ Y7,   Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6,   Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X7,        Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X7,        Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X6,        Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X9,        X5 ^ Y9,        Z2 ^ Y6 ^ X8,   Z1 ^ X6 ^ Y8,   Z0 ^ X7 ^ Y7,   X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X9,        X5 ^ Y8,        Z2 ^ Y5 ^ X8,   Z1 ^ X6 ^ Y7,   Z0 ^ Y6 ^ X7,   X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X8,        X4 ^ Y8,        Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X8,        X4 ^ Y7,        Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X7,        X3 ^ Y7,        Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z3 ^ Y4 ^ X8,   Z2 ^ X5 ^ Y7,   Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6,   Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^ X9,        Z3 ^ X5 ^ Y9,   Z2 ^ Y6 ^ X8,   Z1 ^ X6 ^ Y8,   Z0 ^ X7 ^ Y7,   X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X9,        Z3 ^ X5 ^ Y8,   Z2 ^ Y5 ^ X8,   Z1 ^ X6 ^ Y7,   Z0 ^ Y6 ^ X7,   X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X8,        Z3 ^ X4 ^ Y8,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X8,        Z3 ^ X4 ^ Y7,   Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X7,        X3 ^ Z3 ^ Y7,   Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^X10,        X5 ^Y10,        Z3 ^ Y6 ^ X9,   Z2 ^ X6 ^ Y9,   Z1 ^ Y7 ^ X8,   Z0 ^ X7 ^ Y8    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^X10,        X5 ^ Y9,        Z3 ^ Y5 ^ X9,   Z2 ^ X6 ^ Y8,   Z1 ^ Y6 ^ X8,   Z0 ^ X7 ^ Y7    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X9,        X4 ^ Y9,        Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X9,        X4 ^ Y8,        Z3 ^ Y4 ^ X8,   Z2 ^ X5 ^ Y7,   Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X8,        X3 ^ Y8,        Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Z4 ^ Y5 ^ X9,   Z3 ^ X5 ^ Y9,   Z2 ^ Y6 ^ X8,   Z1 ^ X6 ^ Y8,   Z0 ^ X7 ^ Y7,   X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ Z4 ^ X9,   Z3 ^ X5 ^ Y8,   Z2 ^ Y5 ^ X8,   Z1 ^ X6 ^ Y7,   Z0 ^ Y6 ^ X7,   X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ Z4 ^ X8,   Z3 ^ X4 ^ Y8,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6              },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ Z4 ^ X8,   Z3 ^ X4 ^ Y7,   Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6              },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ Z4 ^ X7,   X3 ^ Z3 ^ Y7,   Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X4,             Y5 ^X10,        Z4 ^ X5 ^Y10,   Z3 ^ Y6 ^ X9,   Z2 ^ X6 ^ Y9,   Z1 ^ Y7 ^ X8,   Z0 ^ X7 ^ Y8    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^X10,        Z4 ^ X5 ^ Y9,   Z3 ^ Y5 ^ X9,   Z2 ^ X6 ^ Y8,   Z1 ^ Y6 ^ X8,   Z0 ^ X7 ^ Y7    },
    {0,              0,              X0,             X1,             Y0,             Y1,             Y2,             X2,             Y3,             X3,             Y4 ^ X9,        X4 ^ Z4 ^ Y9,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {0,              0,              0,              X0,             Y0,             Y1,             X1,             X2,             Y2,             X3,             Y3 ^ X9,        X4 ^ Z4 ^ Y8,   Z3 ^ Y4 ^ X8,   Z2 ^ X5 ^ Y7,   Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6    },
    {0,              0,              0,              0,              Y0,             Y1,             X0,             X1,             Y2,             X2,             Y3 ^ X8,        X3 ^ Z4 ^ Y8,   Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
};

const UINT_64 SW_64K_D_X_RBPLUS[][16]=
{
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z0 ^ Y3 ^ X4,   X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z0 ^ Y2 ^ X3,   X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z0 ^ X2 ^ Y2,   X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X5,        Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X5,        Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ X4,        Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X4,        Z0 ^ X3 ^ Y3,   Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X3,        Z0 ^ X2 ^ Y3,   Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X6,        X4 ^ Y6,        Z0 ^ X5 ^ Y5,   X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Z0 ^ Y4 ^ X5,   X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ X5,        X3 ^ Y5,        Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X5,        X3 ^ Y4,        Z0 ^ Y3 ^ X4,   X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X4,        X2 ^ Y4,        Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z1 ^ Y2 ^ X4,   Z0 ^ X3 ^ Y3,   Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z1 ^ Y2 ^ X3,   Z0 ^ X2 ^ Y3,   Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X6,        Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X6,        Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ X5,        Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X5,        Z1 ^ X3 ^ Y4,   Z0 ^ Y3 ^ X4,   X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X4,        Z1 ^ X2 ^ Y4,   Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X7,        X4 ^ Y7,        Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X7,        X4 ^ Y6,        Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ X6,        X3 ^ Y6,        Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X6,        X3 ^ Y5,        Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X5,        X2 ^ Y5,        Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z2 ^ Y3 ^ X6,   Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Z2 ^ Y3 ^ X5,   Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ Z2 ^ X5,   Z1 ^ X3 ^ Y4,   Z0 ^ Y3 ^ X4,   X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ Z2 ^ X4,   Z1 ^ X2 ^ Y4,   Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X7,        Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X7,        Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ X6,        Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X6,        Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X5,        X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X8,        X4 ^ Y8,        Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X8,        X4 ^ Y7,        Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ X7,        X3 ^ Y7,        Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X7,        X3 ^ Y6,        Z2 ^ Y3 ^ X6,   Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X6,        X2 ^ Y6,        Z2 ^ Y3 ^ X5,   Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ Z3 ^ X5,   X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X8,        Z3 ^ X4 ^ Y8,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X8,        Z3 ^ X4 ^ Y7,   Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ X7,        X3 ^ Z3 ^ Y7,   Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X7,        X3 ^ Z3 ^ Y6,   Z2 ^ Y3 ^ X6,   Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X6,        X2 ^ Z3 ^ Y6,   Z2 ^ Y3 ^ X5,   Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X9,        X4 ^ Y9,        Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X9,        X4 ^ Y8,        Z3 ^ Y4 ^ X8,   Z2 ^ X5 ^ Y7,   Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6,   Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ X8,        X3 ^ Y8,        Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X8,        X3 ^ Y7,        Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X7,        X2 ^ Y7,        Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ Z4 ^ X8,   Z3 ^ X4 ^ Y8,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ Z4 ^ X8,   Z3 ^ X4 ^ Y7,   Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ Z4 ^ X7,   X3 ^ Z3 ^ Y7,   Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ Z4 ^ X7,   X3 ^ Z3 ^ Y6,   Z2 ^ Y3 ^ X6,   Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ Z4 ^ X6,   X2 ^ Z3 ^ Y6,   Z2 ^ Y3 ^ X5,   Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4 ^ X9,        X4 ^ Z4 ^ Y9,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3 ^ X9,        X4 ^ Z4 ^ Y8,   Z3 ^ Y4 ^ X8,   Z2 ^ X5 ^ Y7,   Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6,   Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3 ^ X8,        X3 ^ Z4 ^ Y8,   Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2 ^ X8,        X3 ^ Z4 ^ Y7,   Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2 ^ X7,        X2 ^ Z4 ^ Y7,   Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z0 ^ X3 ^ Y3,   Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             Z0 ^ X2 ^ Y3,   Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Y6,        Z0 ^ X5 ^ Y5,   X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y5,        Z0 ^ Y4 ^ X5,   X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3 ^ Y5,        Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y4,        Z0 ^ Y3 ^ X4,   X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Y4,        Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Y7,        Y5 ^ X6,        Z0 ^ X5 ^ Y6,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Y4 ^ X6,        Z0 ^ X5 ^ Y5,   Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3 ^ Y6,        Y4 ^ X5,        Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y5,        Y3 ^ X5,        Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Y5,        Y3 ^ X4,        Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z1 ^ X3 ^ Y4,   Z0 ^ Y3 ^ X4,   X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             Z1 ^ X2 ^ Y4,   Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Y7,        Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y6,        Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3 ^ Y6,        Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y5,        Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Y5,        Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Y8,        Y5 ^ X7,        Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y7,        Y4 ^ X7,        Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3 ^ Y7,        Y4 ^ X6,        Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y6,        Y3 ^ X6,        Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Y6,        Y3 ^ X5,        Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z2 ^ X3 ^ Y5,   Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Z2 ^ Y5,   Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Y8,        Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y7,        Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3 ^ Y7,        Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y6,        Z2 ^ Y3 ^ X6,   Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Y6,        Z2 ^ Y3 ^ X5,   Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Y9,        Y5 ^ X8,        Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y8,        Y4 ^ X8,        Z2 ^ X5 ^ Y7,   Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6,   Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3 ^ Y8,        Y4 ^ X7,        Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y7,        Y3 ^ X7,        Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Y7,        Y3 ^ X6,        Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             Z3 ^ X4 ^ Y8,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z3 ^ X4 ^ Y7,   Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3 ^ Z3 ^ Y7,   Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Z3 ^ Y6,   Z2 ^ Y3 ^ X6,   Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Z3 ^ Y6,   Z2 ^ Y3 ^ X5,   Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Y9,        Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y8,        Z3 ^ Y4 ^ X8,   Z2 ^ X5 ^ Y7,   Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6,   Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3 ^ Y8,        Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y7,        Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Y7,        Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^Y10,        Y5 ^ X9,        Z3 ^ X5 ^ Y9,   Z2 ^ Y6 ^ X8,   Z1 ^ X6 ^ Y8,   Z0 ^ X7 ^ Y7,   X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y9,        Y4 ^ X9,        Z3 ^ X5 ^ Y8,   Z2 ^ Y5 ^ X8,   Z1 ^ X6 ^ Y7,   Z0 ^ Y6 ^ X7,   X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3 ^ Y9,        Y4 ^ X8,        Z3 ^ X4 ^ Y8,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y8,        Y3 ^ X8,        Z3 ^ X4 ^ Y7,   Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Y8,        Y3 ^ X7,        X3 ^ Z3 ^ Y7,   Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^ Z4 ^ Y9,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Z4 ^ Y8,   Z3 ^ Y4 ^ X8,   Z2 ^ X5 ^ Y7,   Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6,   Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3 ^ Z4 ^ Y8,   Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Z4 ^ Y7,   Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Z4 ^ Y7,   Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4 ^Y10,        Z4 ^ Y5 ^ X9,   Z3 ^ X5 ^ Y9,   Z2 ^ Y6 ^ X8,   Z1 ^ X6 ^ Y8,   Z0 ^ X7 ^ Y7,   X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4 ^ Y9,        Y4 ^ Z4 ^ X9,   Z3 ^ X5 ^ Y8,   Z2 ^ Y5 ^ X8,   Z1 ^ X6 ^ Y7,   Z0 ^ Y6 ^ X7,   X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3 ^ Y9,        Y4 ^ Z4 ^ X8,   Z3 ^ X4 ^ Y8,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3 ^ Y8,        Y3 ^ Z4 ^ X8,   Z3 ^ X4 ^ Y7,   Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2 ^ Y8,        Y3 ^ Z4 ^ X7,   X3 ^ Z3 ^ Y7,   Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5,             X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4,             X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3,             X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3,             X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Z0 ^ X5 ^ Y5,   X5,             Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z0 ^ Y4 ^ X5,   X5,             Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Z0 ^ X4 ^ Y4,   X4,             Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z0 ^ Y3 ^ X4,   X4,             Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Z0 ^ X3 ^ Y3,   X3,             Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X6,        Z0 ^ X5 ^ Y6,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X6,        Z0 ^ X5 ^ Y5,   Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X5,        Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X5,        Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X4,        Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X7,        X5 ^ Y7,        Z0 ^ X6 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        X5 ^ Y6,        Z0 ^ Y5 ^ X6,   X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X6,        X4 ^ Y6,        Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X6,        X4 ^ Y5,        Z0 ^ Y4 ^ X5,   X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X5,        X3 ^ Y5,        Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z1 ^ Y3 ^ X5,   Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Z1 ^ Y3 ^ X4,   Z0 ^ X3 ^ Y4,   Y4,             X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X7,        Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X7,        Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X6,        Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X6,        Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X5,        Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X8,        X5 ^ Y8,        Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X8,        X5 ^ Y7,        Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6,   Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X7,        X4 ^ Y7,        Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X7,        X4 ^ Y6,        Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X6,        X3 ^ Y6,        Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6,             Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5,             Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z2 ^ Y3 ^ X6,   Z1 ^ X4 ^ Y5,   Z0 ^ Y4 ^ X5,   X5,             Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Z2 ^ Y3 ^ X5,   Z1 ^ X3 ^ Y5,   Z0 ^ X4 ^ Y4,   X4,             Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X8,        Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X8,        Z2 ^ X5 ^ Y7,   Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6,   Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X7,        Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X7,        Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X6,        Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X9,        X5 ^ Y9,        Z2 ^ Y6 ^ X8,   Z1 ^ X6 ^ Y8,   Z0 ^ X7 ^ Y7,   X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X9,        X5 ^ Y8,        Z2 ^ Y5 ^ X8,   Z1 ^ X6 ^ Y7,   Z0 ^ Y6 ^ X7,   X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X8,        X4 ^ Y8,        Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X8,        X4 ^ Y7,        Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X7,        X3 ^ Y7,        Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y7,             X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Z3 ^ Y4 ^ X8,   Z2 ^ X5 ^ Y7,   Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6,   Y6,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ Z3 ^ X7,   Z2 ^ X4 ^ Y6,   Z1 ^ Y4 ^ X6,   Z0 ^ X5 ^ Y5,   Y5,             X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ Z3 ^ X6,   Z2 ^ X3 ^ Y6,   Z1 ^ Y4 ^ X5,   Z0 ^ X4 ^ Y5,   Y5,             X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^ X9,        Z3 ^ X5 ^ Y9,   Z2 ^ Y6 ^ X8,   Z1 ^ X6 ^ Y8,   Z0 ^ X7 ^ Y7,   X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ X9,        Z3 ^ X5 ^ Y8,   Z2 ^ Y5 ^ X8,   Z1 ^ X6 ^ Y7,   Z0 ^ Y6 ^ X7,   X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X8,        Z3 ^ X4 ^ Y8,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X8,        Z3 ^ X4 ^ Y7,   Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X7,        X3 ^ Z3 ^ Y7,   Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^X10,        X5 ^Y10,        Z3 ^ Y6 ^ X9,   Z2 ^ X6 ^ Y9,   Z1 ^ Y7 ^ X8,   Z0 ^ X7 ^ Y8    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^X10,        X5 ^ Y9,        Z3 ^ Y5 ^ X9,   Z2 ^ X6 ^ Y8,   Z1 ^ Y6 ^ X8,   Z0 ^ X7 ^ Y7    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X9,        X4 ^ Y9,        Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X9,        X4 ^ Y8,        Z3 ^ Y4 ^ X8,   Z2 ^ X5 ^ Y7,   Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X8,        X3 ^ Y8,        Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Z4 ^ Y5 ^ X9,   Z3 ^ X5 ^ Y9,   Z2 ^ Y6 ^ X8,   Z1 ^ X6 ^ Y8,   Z0 ^ X7 ^ Y7,   X7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^ Z4 ^ X9,   Z3 ^ X5 ^ Y8,   Z2 ^ Y5 ^ X8,   Z1 ^ X6 ^ Y7,   Z0 ^ Y6 ^ X7,   X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ Z4 ^ X8,   Z3 ^ X4 ^ Y8,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ Z4 ^ X8,   Z3 ^ X4 ^ Y7,   Z2 ^ Y4 ^ X7,   Z1 ^ X5 ^ Y6,   Z0 ^ Y5 ^ X6,   X6              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ Z4 ^ X7,   X3 ^ Z3 ^ Y7,   Z2 ^ Y4 ^ X6,   Z1 ^ X4 ^ Y6,   Z0 ^ X5 ^ Y5,   X5              },
    {X0,             X1,             X2,             Y1,             Y0,             Y2,             X3,             Y3,             Y4,             X4,             Y5 ^X10,        Z4 ^ X5 ^Y10,   Z3 ^ Y6 ^ X9,   Z2 ^ X6 ^ Y9,   Z1 ^ Y7 ^ X8,   Z0 ^ X7 ^ Y8    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^X10,        Z4 ^ X5 ^ Y9,   Z3 ^ Y5 ^ X9,   Z2 ^ X6 ^ Y8,   Z1 ^ Y6 ^ X8,   Z0 ^ X7 ^ Y7    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y3,             X3,             Y4 ^ X9,        X4 ^ Z4 ^ Y9,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X9,        X4 ^ Z4 ^ Y8,   Z3 ^ Y4 ^ X8,   Z2 ^ X5 ^ Y7,   Z1 ^ Y5 ^ X7,   Z0 ^ X6 ^ Y6    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y2,             X2,             Y3 ^ X8,        X3 ^ Z4 ^ Y8,   Z3 ^ Y4 ^ X7,   Z2 ^ X4 ^ Y7,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6    },
};

const UINT_64 SW_64K_R_X_1xaa_RBPLUS[][16]=
{
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X4,             Y4,             X5 ^Y10,        Y5 ^X10,        X6 ^ Y9,        Y6 ^ X9,        X7 ^ Y8,        Y7 ^ X8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^X10,        X5 ^ Y9,        Y5 ^ X9,        X6 ^ Y8,        Y6 ^ X8,        X7 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y3,             X4 ^ Y9,        Y4 ^ X9,        X5 ^ Y8,        Y5 ^ X8,        X6 ^ Y7,        Y6 ^ X7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X9,        X4 ^ Y8,        Y4 ^ X8,        X5 ^ Y7,        Y5 ^ X7,        X6 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3 ^ Y8,        Y3 ^ X8,        X4 ^ Y7,        Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5 ^X10,        X6 ^ Y9,        Y6 ^ X9,        X7 ^ Y8,        Y7 ^ X8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z0 ^ X4 ^ Y4,   Y3,             Y4,             X5 ^ Y9,        X6 ^ Y8,        Y5 ^ X9,        X7 ^ Y7,        Y6 ^ X8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Z0 ^ X4 ^ Y4,   X3,             Y3,             Y4 ^ X9,        X5 ^ Y8,        Y5 ^ X8,        X6 ^ Y7,        Y6 ^ X7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z0 ^ X4 ^ Y4,   Y2,             X3,             Y3 ^ X9,        X5 ^ Y7,        Y4 ^ X8,        X6 ^ Y6,        Y5 ^ X7         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z0 ^ X4 ^ Y4,   X2,             Y2,             Y3 ^ X8,        X3 ^ Y7,        Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X5,             Y5,             X6 ^ Y9,        Y6 ^ X9,        X7 ^ Y8,        Y7 ^ X8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   Y3,             X5,             X6 ^ Y8,        Y5 ^ X9,        X7 ^ Y7,        Y6 ^ X8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X3,             Y3,             X5 ^ Y8,        Y5 ^ X8,        X6 ^ Y7,        Y6 ^ X7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   Y2,             X3,             Y3 ^ X8,        X5 ^ Y7,        X6 ^ Y6,        Y5 ^ X7         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X2,             Y2,             Y3 ^ X7,        X3 ^ Y7,        X5 ^ Y6,        Y5 ^ X6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X5 ^ X6 ^ Y6,   Y5,             X6,             Y6 ^ X9,        X7 ^ Y8,        Y7 ^ X8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X5 ^ X6 ^ Y6,   Y3,             Y5,             X6 ^ Y8,        X7 ^ Y7,        Y6 ^ X8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X5 ^ X6 ^ Y6,   X3,             Y3,             Y5 ^ X8,        X6 ^ Y7,        Y6 ^ X7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X5 ^ X6 ^ Y6,   Y2,             X3,             Y3 ^ X8,        X6 ^ Y6,        Y5 ^ X7         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X5 ^ X6 ^ Y6,   X2,             Y2,             Y3 ^ X7,        X3 ^ Y6,        Y5 ^ X6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   X5,             X6,             Y6 ^ X9,        X7 ^ Y8,        Y7 ^ X8,        Z0 ^ X5 ^ Y5    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Y3,             X5,             X6 ^ Y8,        Y6 ^ X8,        X7 ^ Y7,        Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   X3,             Y3,             X5 ^ Y8,        X6 ^ Y7,        Y6 ^ X7,        Z0 ^ X5 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Y2,             X3,             Y3 ^ X7,        X5 ^ Y7,        X6 ^ Y6,        Z0 ^ X5 ^ Y5    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   X2,             Y2,             X3 ^ Y7,        Y3 ^ X6,        X5 ^ Y6,        Z0 ^ X5 ^ Y5    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5,             X6,             Y6 ^ X9,        X7 ^ Y8,        Y7 ^ X8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y3,             X5,             X6 ^ Y8,        Y6 ^ X8,        X7 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X3,             Y3,             X5 ^ Y8,        X6 ^ Y7,        Y6 ^ X7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y2,             X3,             Y3 ^ X7,        X5 ^ Y7,        X6 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X2,             Y2,             X3 ^ Y7,        Y3 ^ X6,        X5 ^ Y6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5 ^ X7 ^ Y7,   X6,             Y6,             X7 ^ Y8,        Y7 ^ X8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5 ^ X7 ^ Y7,   Y3,             X6,             Y6 ^ X8,        X7 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5 ^ X7 ^ Y7,   X3,             Y3,             X6 ^ Y7,        Y6 ^ X7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5 ^ X7 ^ Y7,   Y2,             X3,             Y3 ^ X7,        X6 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5 ^ X7 ^ Y7,   X2,             Y2,             X3 ^ Y6,        Y3 ^ X6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   X6,             Y6,             X7 ^ Y8,        Y7 ^ X8,        Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Y3,             X6,             X7 ^ Y7,        Y6 ^ X8,        Z0 ^ X5 ^ Y6    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   X3,             Y3,             X6 ^ Y7,        Y6 ^ X7,        Z0 ^ X5 ^ Y6    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Y2,             X3,             Y3 ^ X7,        X6 ^ Y6,        Z0 ^ X5 ^ Y6    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   X2,             Y2,             Y3 ^ X6,        X3 ^ Y6,        Z0 ^ X5 ^ Y6    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6,             Y6,             X7 ^ Y8,        Y7 ^ X8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y3,             X6,             X7 ^ Y7,        Y6 ^ X8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X3,             Y3,             X6 ^ Y7,        Y6 ^ X7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y2,             X3,             Y3 ^ X7,        X6 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X2,             Y2,             Y3 ^ X6,        X3 ^ Y6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6 ^ X8 ^ Y8,   Y6,             X7,             Y7 ^ X8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6 ^ X8 ^ Y8,   Y3,             Y6,             X7 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6 ^ X8 ^ Y8,   X3,             Y3,             Y6 ^ X7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6 ^ X8 ^ Y8,   Y2,             X3,             Y3 ^ X7         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X3 ^ X8 ^ Y8,   X2,             Y2,             Y3 ^ X6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   X6,             X7,             Y7 ^ X8,        Z0 ^ X6 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Y3,             X6,             X7 ^ Y7,        Z0 ^ X6 ^ Y6    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   X3,             Y3,             X6 ^ Y7,        Z0 ^ X6 ^ Y6    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Y2,             X3,             Y3 ^ X7,        Z0 ^ X6 ^ Y6    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y4 ^ X8 ^ Y8,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   X2,             X3,             Y3 ^ X6,        Y2 ^ X6 ^ Y6    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             X7,             Y7 ^ X8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y3,             X6,             X7 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X3,             Y3,             X6 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y2,             X3,             Y3 ^ X7         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y4 ^ X8 ^ Y8,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   X2,             X3,             Y3 ^ X6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6 ^ X9 ^ Y9,   X7,             Y7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6 ^ X9 ^ Y9,   Y3,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6 ^ X9 ^ Y9,   X3,             Y3              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X3 ^ X9 ^ Y9,   Y2,             Y3              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y4 ^ X8 ^ Y8,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   X3 ^ X9 ^ Y9,   X2,             Y3              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X9 ^ Y9,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   X7,             Y7,             Z0 ^ X6 ^ Y7    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4 ^ X9 ^ Y9,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Y3,             X7,             Z0 ^ X6 ^ Y7    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y4 ^ X9 ^ Y9,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   X3,             Y3,             Z0 ^ X6 ^ Y7    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y4 ^ X9 ^ Y9,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X3,             Y3,             Z0 ^ X6 ^ Y7    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y4 ^ X9 ^ Y9,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X3,             Y3,             X2 ^ X6 ^ Y7    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X9 ^ Y9,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   X7,             Y7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4 ^ X9 ^ Y9,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y3,             X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y4 ^ X9 ^ Y9,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   X3,             Y3              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y4 ^ X9 ^ Y9,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   X3,             Y3              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y4 ^ X9 ^ Y9,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   X3,             Y3              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X4,             Y4,             X5 ^Y10,        Y5 ^X10,        X6 ^ Y9,        Y6 ^ X9,        X7 ^ Y8,        Y7 ^ X8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^X10,        X5 ^ Y9,        Y5 ^ X9,        X6 ^ Y8,        Y6 ^ X8,        X7 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y3,             X4 ^ Y9,        Y4 ^ X9,        X5 ^ Y8,        Y5 ^ X8,        X6 ^ Y7,        Y6 ^ X7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X9,        X4 ^ Y8,        Y4 ^ X8,        X5 ^ Y7,        Y5 ^ X7,        X6 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3 ^ Y8,        Y3 ^ X8,        X4 ^ Y7,        Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             Z0 ^ X4 ^ Y4,   X5,             Y5 ^X10,        X6 ^ Y9,        Y6 ^ X9,        X7 ^ Y8,        Y7 ^ X8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Z0 ^ X4 ^ Y4,   Y4,             X5 ^ Y9,        X6 ^ Y8,        Y5 ^ X9,        X7 ^ Y7,        Y6 ^ X8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Z0 ^ X4 ^ Y4,   Y3,             Y4 ^ X9,        X5 ^ Y8,        Y5 ^ X8,        X6 ^ Y7,        Y6 ^ X7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Z0 ^ X4 ^ Y4,   X3,             Y3 ^ X9,        X5 ^ Y7,        Y4 ^ X8,        X6 ^ Y6,        Y5 ^ X7         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Z0 ^ X4 ^ Y4,   Y2,             Y3 ^ X8,        X3 ^ Y7,        Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X5,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   Y5,             X6 ^ Y9,        Y6 ^ X9,        X7 ^ Y8,        Y7 ^ X8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X5,             X6 ^ Y8,        Y5 ^ X9,        X7 ^ Y7,        Y6 ^ X8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   Y3,             X5 ^ Y8,        Y5 ^ X8,        X6 ^ Y7,        Y6 ^ X7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X3,             Y3 ^ X8,        X5 ^ Y7,        X6 ^ Y6,        Y5 ^ X7         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   Y2,             Y3 ^ X7,        X3 ^ Y7,        X5 ^ Y6,        Y5 ^ X6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y5,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X5 ^ X6 ^ Y6,   X6,             Y6 ^ X9,        X7 ^ Y8,        Y7 ^ X8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X5 ^ X6 ^ Y6,   Y5,             X6 ^ Y8,        X7 ^ Y7,        Y6 ^ X8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X5 ^ X6 ^ Y6,   Y3,             Y5 ^ X8,        X6 ^ Y7,        Y6 ^ X7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X5 ^ X6 ^ Y6,   X3,             Y3 ^ X8,        X6 ^ Y6,        Y5 ^ X7         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X5 ^ X6 ^ Y6,   Y2,             Y3 ^ X7,        X3 ^ Y6,        Y5 ^ X6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X5,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   X6,             Y6 ^ X9,        X7 ^ Y8,        Y7 ^ X8,        Z0 ^ X5 ^ Y5    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   X5,             X6 ^ Y8,        Y6 ^ X8,        X7 ^ Y7,        Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Y3,             X5 ^ Y8,        X6 ^ Y7,        Y6 ^ X7,        Z0 ^ X5 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   X3,             Y3 ^ X7,        X5 ^ Y7,        X6 ^ Y6,        Z0 ^ X5 ^ Y5    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Y2,             X3 ^ Y7,        Y3 ^ X6,        X5 ^ Y6,        Z0 ^ X5 ^ Y5    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X5,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X6,             Y6 ^ X9,        X7 ^ Y8,        Y7 ^ X8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5,             X6 ^ Y8,        Y6 ^ X8,        X7 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y3,             X5 ^ Y8,        X6 ^ Y7,        Y6 ^ X7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X3,             Y3 ^ X7,        X5 ^ Y7,        X6 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y2,             X3 ^ Y7,        Y3 ^ X6,        X5 ^ Y6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X6,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5 ^ X7 ^ Y7,   Y6,             X7 ^ Y8,        Y7 ^ X8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5 ^ X7 ^ Y7,   X6,             Y6 ^ X8,        X7 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5 ^ X7 ^ Y7,   Y3,             X6 ^ Y7,        Y6 ^ X7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5 ^ X7 ^ Y7,   X3,             Y3 ^ X7,        X6 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5 ^ X7 ^ Y7,   Y2,             X3 ^ Y6,        Y3 ^ X6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X6,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Y6,             X7 ^ Y8,        Y7 ^ X8,        Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   X6,             X7 ^ Y7,        Y6 ^ X8,        Z0 ^ X5 ^ Y6    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Y3,             X6 ^ Y7,        Y6 ^ X7,        Z0 ^ X5 ^ Y6    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   X3,             Y3 ^ X7,        X6 ^ Y6,        Z0 ^ X5 ^ Y6    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Y2,             Y3 ^ X6,        X3 ^ Y6,        Z0 ^ X5 ^ Y6    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X6,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y6,             X7 ^ Y8,        Y7 ^ X8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6,             X7 ^ Y7,        Y6 ^ X8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y3,             X6 ^ Y7,        Y6 ^ X7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X3,             Y3 ^ X7,        X6 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y2,             Y3 ^ X6,        X3 ^ Y6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y6,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6 ^ X8 ^ Y8,   X7,             Y7 ^ X8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6 ^ X8 ^ Y8,   Y6,             X7 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6 ^ X8 ^ Y8,   Y3,             Y6 ^ X7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6 ^ X8 ^ Y8,   X3,             Y3 ^ X7         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X3 ^ X8 ^ Y8,   Y2,             Y3 ^ X6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X6,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   X7,             Y7 ^ X8,        Z0 ^ X6 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   X6,             X7 ^ Y7,        Z0 ^ X6 ^ Y6    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Y3,             X6 ^ Y7,        Z0 ^ X6 ^ Y6    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   X3,             Y3 ^ X7,        Z0 ^ X6 ^ Y6    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y4 ^ X8 ^ Y8,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   X3,             Y3 ^ X6,        Y2 ^ X6 ^ Y6    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X6,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X7,             Y7 ^ X8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             X7 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y3,             X6 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X3,             Y3 ^ X7         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y4 ^ X8 ^ Y8,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   X3,             Y3 ^ X6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X7,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6 ^ X9 ^ Y9,   Y7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6 ^ X9 ^ Y9,   X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6 ^ X9 ^ Y9,   Y3              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X3 ^ X9 ^ Y9,   Y3              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y4 ^ X8 ^ Y8,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   X3 ^ X9 ^ Y9,   Y3              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X7,             Y4 ^ X9 ^ Y9,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Y7,             Z0 ^ X6 ^ Y7    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Y4 ^ X9 ^ Y9,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   X7,             Z0 ^ X6 ^ Y7    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y4 ^ X9 ^ Y9,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Y3,             Z0 ^ X6 ^ Y7    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3,             Y4 ^ X9 ^ Y9,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   Y3,             Z0 ^ X6 ^ Y7    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X3,             Y4 ^ X9 ^ Y9,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   Y3,             X2 ^ X6 ^ Y7    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X7,             Y4 ^ X9 ^ Y9,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y7              },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Y4 ^ X9 ^ Y9,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   X7              },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y4 ^ X9 ^ Y9,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y3              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3,             Y4 ^ X9 ^ Y9,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7,   Y3              },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X3,             Y4 ^ X9 ^ Y9,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7,   Y3              },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X4,             Y4,             X5 ^Y10,        Y5 ^X10,        X6 ^ Y9,        Y6 ^ X9,        X7 ^ Y8,        Y7 ^ X8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^X10,        X5 ^ Y9,        Y5 ^ X9,        X6 ^ Y8,        Y6 ^ X8,        X7 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y3,             X4 ^ Y9,        Y4 ^ X9,        X5 ^ Y8,        Y5 ^ X8,        X6 ^ Y7,        Y6 ^ X7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X9,        X4 ^ Y8,        Y4 ^ X8,        X5 ^ Y7,        Y5 ^ X7,        X6 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             X3 ^ Y8,        Y3 ^ X8,        X4 ^ Y7,        Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4,             X5,             Z0 ^ X4 ^ Y4,   Y5 ^X10,        X6 ^ Y9,        Y6 ^ X9,        X7 ^ Y8,        Y7 ^ X8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Y4,             Z0 ^ X4 ^ Y4,   X5 ^ Y9,        X6 ^ Y8,        Y5 ^ X9,        X7 ^ Y7,        Y6 ^ X8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y3,             Z0 ^ X4 ^ Y4,   Y4 ^ X9,        X5 ^ Y8,        Y5 ^ X8,        X6 ^ Y7,        Y6 ^ X7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Z0 ^ X4 ^ Y4,   Y3 ^ X9,        X5 ^ Y7,        Y4 ^ X8,        X6 ^ Y6,        Y5 ^ X7         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Z0 ^ X4 ^ Y4,   Y3 ^ X8,        X3 ^ Y7,        Y4 ^ X7,        X5 ^ Y6,        Y5 ^ X6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X5,             Y5,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X6 ^ Y9,        Y6 ^ X9,        X7 ^ Y8,        Y7 ^ X8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X5,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X6 ^ Y8,        Y5 ^ X9,        X7 ^ Y7,        Y6 ^ X8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y3,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X5 ^ Y8,        Y5 ^ X8,        X6 ^ Y7,        Y6 ^ X7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   Y3 ^ X8,        X5 ^ Y7,        X6 ^ Y6,        Y5 ^ X7         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   Y3 ^ X7,        X3 ^ Y7,        X5 ^ Y6,        Y5 ^ X6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y5,             X6,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X5 ^ X6 ^ Y6,   Y6 ^ X9,        X7 ^ Y8,        Y7 ^ X8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Y5,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X5 ^ X6 ^ Y6,   X6 ^ Y8,        X7 ^ Y7,        Y6 ^ X8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y3,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X5 ^ X6 ^ Y6,   Y5 ^ X8,        X6 ^ Y7,        Y6 ^ X7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X5 ^ X6 ^ Y6,   Y3 ^ X8,        X6 ^ Y6,        Y5 ^ X7         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X5 ^ X6 ^ Y6,   Y3 ^ X7,        X3 ^ Y6,        Y5 ^ X6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X5,             X6,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Y6 ^ X9,        X7 ^ Y8,        Y7 ^ X8,        Z0 ^ X5 ^ Y5    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X5,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   X6 ^ Y8,        Y6 ^ X8,        X7 ^ Y7,        Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y3,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   X5 ^ Y8,        X6 ^ Y7,        Y6 ^ X7,        Z0 ^ X5 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Y3 ^ X7,        X5 ^ Y7,        X6 ^ Y6,        Z0 ^ X5 ^ Y5    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   X3 ^ Y7,        Y3 ^ X6,        X5 ^ Y6,        Z0 ^ X5 ^ Y5    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X5,             X6,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y6 ^ X9,        X7 ^ Y8,        Y7 ^ X8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X5,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X6 ^ Y8,        Y6 ^ X8,        X7 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y3,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5 ^ Y8,        X6 ^ Y7,        Y6 ^ X7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y3 ^ X7,        X5 ^ Y7,        X6 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X3 ^ Y7,        Y3 ^ X6,        X5 ^ Y6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X6,             Y6,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5 ^ X7 ^ Y7,   X7 ^ Y8,        Y7 ^ X8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X6,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5 ^ X7 ^ Y7,   Y6 ^ X8,        X7 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y3,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5 ^ X7 ^ Y7,   X6 ^ Y7,        Y6 ^ X7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5 ^ X7 ^ Y7,   Y3 ^ X7,        X6 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5 ^ X7 ^ Y7,   X3 ^ Y6,        Y3 ^ X6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X6,             Y6,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   X7 ^ Y8,        Y7 ^ X8,        Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X6,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   X7 ^ Y7,        Y6 ^ X8,        Z0 ^ X5 ^ Y6    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y3,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   X6 ^ Y7,        Y6 ^ X7,        Z0 ^ X5 ^ Y6    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Y3 ^ X7,        X6 ^ Y6,        Z0 ^ X5 ^ Y6    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Y3 ^ X6,        X3 ^ Y6,        Z0 ^ X5 ^ Y6    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X6,             Y6,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X7 ^ Y8,        Y7 ^ X8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X6,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X7 ^ Y7,        Y6 ^ X8         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y3,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6 ^ Y7,        Y6 ^ X7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y3 ^ X7,        X6 ^ Y6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y3 ^ X6,        X3 ^ Y6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y6,             X7,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6 ^ X8 ^ Y8,   Y7 ^ X8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             Y6,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6 ^ X8 ^ Y8,   X7 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y3,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6 ^ X8 ^ Y8,   Y6 ^ X7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6 ^ X8 ^ Y8,   Y3 ^ X7         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X3 ^ X8 ^ Y8,   Y3 ^ X6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X6,             X7,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Y7 ^ X8,        Z0 ^ X6 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X6,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   X7 ^ Y7,        Z0 ^ X6 ^ Y6    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y3,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   X6 ^ Y7,        Z0 ^ X6 ^ Y6    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Y3 ^ X7,        Z0 ^ X6 ^ Y6    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             X3,             Y4 ^ X8 ^ Y8,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Y3 ^ X6,        Y2 ^ X6 ^ Y6    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X6,             X7,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y7 ^ X8         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X6,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X7 ^ Y7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y3,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6 ^ Y7         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y3 ^ X7         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             X3,             Y4 ^ X8 ^ Y8,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   Y3 ^ X6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X7,             Y7,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6 ^ X9 ^ Y9    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X7,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6 ^ X9 ^ Y9    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y3,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6 ^ X9 ^ Y9    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             Y3,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X3 ^ X9 ^ Y9    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y3,             Y4 ^ X8 ^ Y8,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   X3 ^ X9 ^ Y9    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X7,             Y7,             Y4 ^ X9 ^ Y9,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X7,             Y4 ^ X9 ^ Y9,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y3,             Y4 ^ X9 ^ Y9,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3,             Y3,             Y4 ^ X9 ^ Y9,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X3,             Y3,             Y4 ^ X9 ^ Y9,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X7,             Y7,             Y4 ^ X9 ^ Y9,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X7,             Y4 ^ X9 ^ Y9,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y3,             Y4 ^ X9 ^ Y9,   X4 ^ Y4 ^ Z4,   Z3 ^ Y5 ^ X8,   Z2 ^ X5 ^ Y8,   Z1 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             X3,             Y3,             Y4 ^ X9 ^ Y9,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X8,   Z1 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   Z0 ^ X6 ^ Y7    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X3,             Y3,             Y4 ^ X9 ^ Y9,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X8,   Z0 ^ X5 ^ Y8,   Y2 ^ Y6 ^ X7,   X2 ^ X6 ^ Y7    },
};

const UINT_64 SW_64K_R_X_2xaa_RBPLUS[][16]=
{
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             X4,             Y4,             Y5 ^X10,        X5 ^Y10,        Y6 ^ X9,        X6 ^ Y9,        Y7 ^ X8,        S0 ^ X7 ^ Y8    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y3,             X4,             Y4 ^X10,        X5 ^ Y9,        Y5 ^ X9,        X6 ^ Y8,        Y6 ^ X8,        S0 ^ X7 ^ Y7    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             X3,             Y3,             Y4 ^ X9,        X4 ^ Y9,        Y5 ^ X8,        X5 ^ Y8,        Y6 ^ X7,        S0 ^ X6 ^ Y7    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y2,             X3,             Y3 ^ X9,        X4 ^ Y8,        Y4 ^ X8,        X5 ^ Y7,        Y5 ^ X7,        S0 ^ X6 ^ Y6    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             X2,             Y2,             Y3 ^ X8,        X3 ^ Y8,        Y4 ^ X7,        X4 ^ Y7,        Y5 ^ X6,        S0 ^ X5 ^ Y6    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Z0 ^ X4 ^ Y4,   Y4,             X5,             Y5 ^X10,        Y6 ^ X9,        X6 ^ Y9,        Y7 ^ X8,        S0 ^ X7 ^ Y8    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Z0 ^ X4 ^ Y4,   Y3,             Y4,             X5 ^ Y9,        Y5 ^ X9,        X6 ^ Y8,        Y6 ^ X8,        S0 ^ X7 ^ Y7    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Z0 ^ X4 ^ Y4,   X3,             Y3,             Y4 ^ X9,        Y5 ^ X8,        X5 ^ Y8,        Y6 ^ X7,        S0 ^ X6 ^ Y7    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Z0 ^ X4 ^ Y4,   Y2,             X3,             Y3 ^ X9,        Y4 ^ X8,        X5 ^ Y7,        Y5 ^ X7,        S0 ^ X6 ^ Y6    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Z0 ^ X4 ^ Y4,   X2,             Y2,             Y3 ^ X8,        Y4 ^ X7,        X3 ^ Y7,        Y5 ^ X6,        S0 ^ X5 ^ Y6    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X5,             Y5,             Y6 ^ X9,        X6 ^ Y9,        Y7 ^ X8,        S0 ^ X7 ^ Y8    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   Y3,             X5,             Y5 ^ X9,        X6 ^ Y8,        Y6 ^ X8,        S0 ^ X7 ^ Y7    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X3,             Y3,             Y5 ^ X8,        X5 ^ Y8,        Y6 ^ X7,        S0 ^ X6 ^ Y7    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   Y2,             X3,             Y3 ^ X8,        X5 ^ Y7,        Y5 ^ X7,        S0 ^ X6 ^ Y6    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X2,             Y2,             Y3 ^ X7,        X3 ^ Y7,        Y5 ^ X6,        S0 ^ X5 ^ Y6    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X5 ^ X6 ^ Y6,   Y5,             X6,             Y6 ^ X9,        Y7 ^ X8,        S0 ^ X7 ^ Y8    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X5 ^ X6 ^ Y6,   Y3,             Y5,             X6 ^ Y8,        Y6 ^ X8,        S0 ^ X7 ^ Y7    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X5 ^ X6 ^ Y6,   X3,             Y3,             Y5 ^ X8,        Y6 ^ X7,        S0 ^ X6 ^ Y7    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   X5 ^ X6 ^ Y6,   Y2,             X3,             Y3 ^ X8,        Y5 ^ X7,        S0 ^ X6 ^ Y6    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y4 ^ X5 ^ Y5,   Z0 ^ X4 ^ Y4,   S0 ^ X6 ^ Y6,   X2,             Y2,             Y3 ^ X6,        X3 ^ Y6,        X5 ^ Y5         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   X5,             X6,             Y6 ^ X9,        Y7 ^ X8,        S0 ^ X7 ^ Y8,   Z0 ^ X5 ^ Y5    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Y3,             X5,             X6 ^ Y8,        Y6 ^ X8,        S0 ^ X7 ^ Y7,   Z0 ^ X5 ^ Y5    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   X3,             Y3,             X5 ^ Y8,        Y6 ^ X7,        S0 ^ X6 ^ Y7,   Z0 ^ X5 ^ Y5    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Y2,             X3,             Y3 ^ X7,        X5 ^ Y7,        S0 ^ X6 ^ Y6,   Z0 ^ X5 ^ Y5    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   X2,             Y2,             Y3 ^ X6,        X3 ^ Y7,        S0 ^ X5 ^ Y6,   Z0 ^ X5 ^ Y5    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5,             X6,             Y6 ^ X9,        Y7 ^ X8,        S0 ^ X7 ^ Y8    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y3,             X5,             X6 ^ Y8,        Y6 ^ X8,        S0 ^ X7 ^ Y7    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X3,             Y3,             X5 ^ Y8,        Y6 ^ X7,        S0 ^ X6 ^ Y7    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   Y2,             X3,             Y3 ^ X7,        X5 ^ Y7,        S0 ^ X6 ^ Y6    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X2,             Y2,             Y3 ^ X6,        X3 ^ Y7,        S0 ^ X5 ^ Y6    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5 ^ X7 ^ Y7,   X6,             Y6,             Y7 ^ X8,        S0 ^ X7 ^ Y8    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5 ^ X7 ^ Y7,   Y3,             X6,             Y6 ^ X8,        S0 ^ X7 ^ Y7    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5 ^ X7 ^ Y7,   X3,             Y3,             Y6 ^ X7,        S0 ^ X6 ^ Y7    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   X5 ^ X7 ^ Y7,   Y2,             X3,             Y3 ^ X7,        S0 ^ X6 ^ Y6    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y4 ^ X6 ^ Y6,   Z1 ^ X4 ^ Y4,   Z0 ^ X5 ^ Y5,   S0 ^ X7 ^ Y7,   X2,             Y2,             Y3 ^ X5,        X3 ^ Y6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   X6,             Y6,             Y7 ^ X8,        S0 ^ X7 ^ Y8,   Z0 ^ X5 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Y3,             X6,             Y6 ^ X8,        S0 ^ X7 ^ Y7,   Z0 ^ X5 ^ Y6    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   X3,             Y3,             Y6 ^ X7,        S0 ^ X6 ^ Y7,   Z0 ^ X5 ^ Y6    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Y2,             X3,             Y3 ^ X7,        S0 ^ X6 ^ Y6,   Z0 ^ X5 ^ Y6    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y4 ^ X7 ^ Y7,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   X2,             Y2,             Y3 ^ X6,        X3 ^ Y6,        S0 ^ X5 ^ Y6    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6,             Y6,             Y7 ^ X8,        S0 ^ X7 ^ Y8    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y3,             X6,             Y6 ^ X8,        S0 ^ X7 ^ Y7    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X3,             Y3,             Y6 ^ X7,        S0 ^ X6 ^ Y7    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   Y2,             X3,             Y3 ^ X7,        S0 ^ X6 ^ Y6    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y4 ^ X7 ^ Y7,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   S0 ^ X5 ^ Y6,   X2,             Y2,             Y3 ^ X6,        X3 ^ Y6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6 ^ X8 ^ Y8,   Y6,             Y7,             S0 ^ X7 ^ Y8    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   X6 ^ X8 ^ Y8,   Y3,             Y6,             S0 ^ X7 ^ Y7    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   S0 ^ X8 ^ Y8,   X3,             Y3,             X6 ^ Y6         },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y4 ^ X7 ^ Y7,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X6,   Z0 ^ X5 ^ Y6,   S0 ^ X8 ^ Y8,   Y2,             X3,             Y3 ^ X6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y4 ^ X7 ^ Y7,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X6,   S0 ^ X5 ^ Y6,   X3 ^ X8 ^ Y8,   X2,             Y2,             Y3 ^ X6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   X6,             Y7,             S0 ^ X7 ^ Y8,   Z0 ^ X6 ^ Y6    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Y3,             X6,             S0 ^ X7 ^ Y7,   Z0 ^ X6 ^ Y6    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   X3,             Y3,             S0 ^ X6 ^ Y7,   Z0 ^ X6 ^ Y6    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y4 ^ X8 ^ Y8,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   Y2,             X3,             Y3 ^ X6,        S0 ^ X6 ^ Y6    },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y4 ^ X8 ^ Y8,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   S0 ^ X5 ^ Y7,   X2,             X3,             Y3 ^ X6,        Y2 ^ X6 ^ Y6    },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6,             Y7,             S0 ^ X7 ^ Y8    },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   Y3,             X6,             S0 ^ X7 ^ Y7    },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X3,             Y3,             S0 ^ X6 ^ Y7    },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y4 ^ X8 ^ Y8,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   S0 ^ X6 ^ Y6,   Y2,             X3,             Y3 ^ X6         },
    {0,              0,              0,              0,              X0,             Y0,             X1,             Y1,             Y4 ^ X8 ^ Y8,   Z1 ^ X4 ^ Y4,   Z0 ^ Y5 ^ X7,   S0 ^ X5 ^ Y7,   Y2 ^ X6 ^ Y6,   X2,             X3,             Y3 ^ X6         },
    {X0,             X1,             X2,             X3,             Y0,             Y1,             Y2,             Y3,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6 ^ X9 ^ Y9,   Y7,             S0 ^ X7         },
    {0,              X0,             X1,             X2,             Y0,             Y1,             Y2,             X3,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   X6 ^ X9 ^ Y9,   Y3,             S0 ^ X7         },
    {0,              0,              X0,             X1,             Y0,             Y1,             X2,             Y2,             Y4 ^ X8 ^ Y8,   Z3 ^ X4 ^ Y4,   Z2 ^ Y5 ^ X7,   Z1 ^ X5 ^ Y7,   Z0 ^ X6 ^ Y6,   S0 ^ X9 ^ Y9,   X3,             Y3              },
    {0,              0,              0,              X0,             Y0,             X1,             X2,             Y1,             Y4 ^ X8 ^ Y8,   Z2 ^ X4 ^ Y4,   Z1 ^ Y5 ^ X7,   Z0 ^ X5 ^ Y7,   S0 ^ X6 ^ Y6,   X3 ^ X9 ^ Y9,   Y2,             Y3              },
    {0,              0,              0,              0,              X0,          