#
#  Copyright (C) 2018, 2019 Codethink Limited
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU Lesser General Public
#  License as published by the Free Software Foundation; either
#  version 2 of the License, or (at your option) any later version.
#
#  This library is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
#  Lesser General Public License for more details.
#
#  You should have received a copy of the GNU Lesser General Public
#  License along with this library. If not, see <http://www.gnu.org/licenses/>.
#
#  Authors:
#         Valentin David <valentin.david@codethink.co.uk>


import collections.abc
import os
from ruamel import yaml
from buildstream import Element, ElementError, Scope

class SnapImageElement(Element):

    def configure(self, node):
        self.node_validate(node, [
            'directory', 'include', 'exclude', 'metadata',
            'include-orphans'
        ])
        self.directory = self.node_subst_member(node, 'directory')
        self.include = self.node_get_member(node, list, 'include')
        self.exclude = self.node_get_member(node, list, 'exclude')
        self.include_orphans = self.node_get_member(node, bool, 'include-orphans')
        self.metadata = self._clean_meta_data(node.get('metadata'))

    def _clean_meta_data(self, node):
        ret = {}
        for k, v in node.items():
            if not k.startswith('__bst'):
                if isinstance(v, collections.abc.Mapping):
                    ret[k] = self._clean_meta_data(v)
                elif isinstance(v, list):
                    ret[k] = self.node_subst_list(node, k)
                else:
                    ret[k] = self.node_subst_member(node, k)
        return ret

    def preflight(self):
        runtime_deps = list(self.dependencies(Scope.RUN, recurse=False))
        if runtime_deps:
            raise ElementError("{}: Only build type dependencies supported by flatpak_image elements"
                               .format(self))

        sources = list(self.sources())
        if sources:
            raise ElementError("{}: flatpak_image elements may not have sources".format(self))

    def get_unique_key(self):
        key = {}
        key['directory'] = self.directory
        key['include'] = sorted(self.include)
        key['exclude'] = sorted(self.exclude)
        key['include-orphans'] = self.include_orphans
        key['metadata'] = self.metadata
        key['version'] = 7
        return key

    def configure_sandbox(self, sandbox):
        pass

    def stage(self, sandbox):
        pass

    def assemble(self, sandbox):
        basedir = sandbox.get_directory()

        reldirectory = os.path.relpath(self.directory, os.sep)
        rootdir = os.path.join(basedir, reldirectory)

        metadir = os.path.join(rootdir, 'meta')
        metadata = os.path.join(metadir, 'snap.yaml')

        with self.timed_activity("Creating snap image", silent_nested=True):
            self.stage_dependency_artifacts(sandbox,
                                            Scope.BUILD,
                                            include=self.include,
                                            exclude=self.exclude,
                                            orphans=self.include_orphans)

            os.makedirs(metadir, exist_ok=True)

            with open(metadata, 'w') as file:
                yaml.dump(self.metadata, file)


        return os.path.join(os.sep, reldirectory)

def setup():
    return SnapImageElement
