/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/macros/check_annotation.t.cpp
 *
 * @brief [LEVEL: beta] testing @ref DIAGNOSTICS_BASE_CHECK
 *
 * $Id: check_annotation.t.cpp,v 1.11 2005/06/23 09:54:21 esdentem Exp $
 *
 * @author Christian Schallhart
 */

#include <diagnostics/unittest.hpp>

// component
#include <diagnostics/macros/check_annotation.hpp>

// used component
#include <diagnostics/frame/logging_config.hpp>

// backdoor
#include <diagnostics/unittest/test_system/current_test_logger.hpp>

// test support
#include <diagnostics/util/dummy_logger.ts.hpp>
#include <diagnostics/util/assert_record.ts.hpp>

#define TEST_COMPONENT_NAME check_annotation
#define TEST_COMPONENT_NAMESPACE diagnostics


DIAGNOSTICS_NAMESPACE_BEGIN;
TEST_NAMESPACE_BEGIN;
TEST_COMPONENT_TEST_NAMESPACE_BEGIN;
using namespace unittest;
using ::diagnostics::unittest::internal::Current_Test_Logger;


#define AC_THROW(NAME,STR_WHAT) throw NAME(STR_WHAT)
#define AC_NOOP ((void)0)

void use_check_throw(bool const succeed,
					 ::std::string const what)
{
    Dummy_Logger * logger=new Dummy_Logger; 
    Dummy_Logger::Records_t & records(logger->records()); 
    typedef Dummy_Logger::Records_t::iterator Iter_t;

    Logging_Config::register_logger(logger);
    if(succeed) {
		TEST_EXCEPTIONLESS_BLOCK_ENTER;
		TEST_NO_IMPLICIT_LOGGING_ENTER;
		DIAGNOSTICS_BASE_CHECK(LEVEL_PROD,0,what,succeed,AC_THROW(Test_Exception,what));
		DIAGNOSTICS_BASE_CHECK(LEVEL_DEBUG,1,what,succeed,AC_THROW(Test_Exception,what));
		DIAGNOSTICS_BASE_CHECK(LEVEL_AUDIT,2,what,succeed,AC_THROW(Test_Exception,what));
		DIAGNOSTICS_BASE_CHECK(LEVEL_TEST,3,what,succeed,AC_THROW(Test_Exception,what));
		TEST_NO_IMPLICIT_LOGGING_EXIT;
		TEST_EXCEPTIONLESS_BLOCK_EXIT;
    }
    else {
		TEST_THROWING_BLOCK_ENTER;
		TEST_NO_IMPLICIT_LOGGING_ENTER;
		DIAGNOSTICS_BASE_CHECK(LEVEL_PROD,0,what,succeed,AC_THROW(Test_Exception,what));
		TEST_NO_IMPLICIT_LOGGING_EXIT;
		TEST_THROWING_BLOCK_EXIT1(Test_Exception,what.c_str());

		TEST_THROWING_BLOCK_ENTER;
		TEST_NO_IMPLICIT_LOGGING_ENTER;
		DIAGNOSTICS_BASE_CHECK(LEVEL_DEBUG,1,what,succeed,AC_THROW(Test_Exception,what));
		TEST_NO_IMPLICIT_LOGGING_EXIT;
		TEST_THROWING_BLOCK_EXIT1(Test_Exception,what.c_str());

		TEST_THROWING_BLOCK_ENTER;
		TEST_NO_IMPLICIT_LOGGING_ENTER;
		DIAGNOSTICS_BASE_CHECK(LEVEL_AUDIT,2,what,succeed,AC_THROW(Test_Exception,what));
		TEST_NO_IMPLICIT_LOGGING_EXIT;
		TEST_THROWING_BLOCK_EXIT1(Test_Exception,what.c_str());

		TEST_THROWING_BLOCK_ENTER;
		TEST_NO_IMPLICIT_LOGGING_ENTER;
		DIAGNOSTICS_BASE_CHECK(LEVEL_TEST,3,what,succeed,AC_THROW(Test_Exception,what));
		TEST_NO_IMPLICIT_LOGGING_EXIT;
		TEST_THROWING_BLOCK_EXIT1(Test_Exception,what.c_str());
    }
    Logging_Config::unregister_logger(logger);
    
    // checking
    Iter_t current(records.begin()); 
    TEST_CHECK(current->type()==TYPE_LOG_OPEN);
    ++current;
    if(!succeed) {
		TEST_ASSERT_RECORD1(LEVEL_PROD,TYPE_FAILED_CHECK,0,what);
		++current;
		TEST_ASSERT_RECORD1(LEVEL_DEBUG,TYPE_FAILED_CHECK,1,what);
		++current;
		TEST_ASSERT_RECORD1(LEVEL_AUDIT,TYPE_FAILED_CHECK,2,what);
		++current;
		TEST_ASSERT_RECORD1(LEVEL_TEST,TYPE_FAILED_CHECK,3,what);
		++current;
    }
    TEST_CHECK(current->type()==TYPE_LOG_CLOSE);
    TEST_CHECK(current+1==records.end()); 
    
    delete logger;
}


void use_check_nothrow(bool const succeed,
					   ::std::string const what)
{
    Dummy_Logger * logger=new Dummy_Logger; 
    Dummy_Logger::Records_t & records(logger->records()); 
    typedef Dummy_Logger::Records_t::iterator Iter_t;

    Logging_Config::register_logger(logger);
    TEST_EXCEPTIONLESS_BLOCK_ENTER;
    TEST_NO_IMPLICIT_LOGGING_ENTER;
    DIAGNOSTICS_BASE_CHECK(LEVEL_PROD,0,what,succeed,AC_NOOP);
    DIAGNOSTICS_BASE_CHECK(LEVEL_DEBUG,1,what,succeed,AC_NOOP);
    DIAGNOSTICS_BASE_CHECK(LEVEL_AUDIT,2,what,succeed,AC_NOOP);
    DIAGNOSTICS_BASE_CHECK(LEVEL_TEST,3,what,succeed,AC_NOOP);
    TEST_NO_IMPLICIT_LOGGING_EXIT;
    TEST_EXCEPTIONLESS_BLOCK_EXIT;
    Logging_Config::unregister_logger(logger);
    
    // checking
    Iter_t current(records.begin()); 
    TEST_CHECK(current->type()==TYPE_LOG_OPEN);
    ++current;
    if(!succeed) {
		TEST_ASSERT_RECORD1(LEVEL_PROD,TYPE_FAILED_CHECK,0,what);
		++current;
		TEST_ASSERT_RECORD1(LEVEL_DEBUG,TYPE_FAILED_CHECK,1,what);
		++current;
		TEST_ASSERT_RECORD1(LEVEL_AUDIT,TYPE_FAILED_CHECK,2,what);
		++current;
		TEST_ASSERT_RECORD1(LEVEL_TEST,TYPE_FAILED_CHECK,3,what);
		++current;
    }
    TEST_CHECK(current->type()==TYPE_LOG_CLOSE);
    TEST_CHECK(current+1==records.end()); 
    
    delete logger;
}



void throw_succeed(Test_Data & data)
{
    use_check_throw(true,"bla");
    use_check_throw(true,"bla1");
}

void throw_fail(Test_Data & data)
{
    use_check_throw(false,"bla");
    use_check_throw(false,"bla1");
}

void nothrow_succeed(Test_Data & data)
{
    use_check_nothrow(true,"bla");
    use_check_nothrow(true,"bla1");
}

void nothrow_fail(Test_Data & data)
{
    use_check_nothrow(false,"bla");
    use_check_nothrow(false,"bla1");
}

TEST_COMPONENT_TEST_NAMESPACE_END;
TEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

TEST_SUITE_BEGIN;
TEST_NORMAL_CASE(&throw_succeed,LEVEL_PROD);
TEST_NORMAL_CASE(&throw_fail,LEVEL_PROD);
TEST_NORMAL_CASE(&nothrow_succeed,LEVEL_PROD);
TEST_NORMAL_CASE(&nothrow_fail,LEVEL_PROD);
TEST_SUITE_END;

STREAM_TEST_SYSTEM_MAIN;

// vim:ts=4:sw=4
