/* ============================================================
 *
 * This file is a part of digiKam project
 * https://www.digikam.org
 *
 * Date        : 2010-06-21
 * Description : unit test program for digiKam sqlite database init
 *
 * SPDX-FileCopyrightText: 2013 by Michael G. Hansen <mike at mghansen dot de>
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * ============================================================ */

#include "faceclassifier_utest.h"

// Qt includes

#include <QApplication>
#include <QSqlDatabase>
#include <QTimer>
#include <QCommandLineParser>
#include <QDir>
#include <QMultiMap>
#include <QUuid>

// KDE includes

#include <kaboutdata.h>

// Local includes

#include "digikam_debug.h"
#include "daboutdata.h"
#include "albummanager.h"
#include "coredbaccess.h"
#include "thumbsdbaccess.h"
#include "facedbaccess.h"
#include "dbengineparameters.h"
#include "scancontroller.h"
#include "digikam_version.h"
#include "dtestdatadir.h"
#include "wstoolutils.h"
#include "identityprovider.h"
#include "faceclassifier.h"

using namespace Digikam;

QTEST_MAIN(FaceClassifierTest)

void FaceClassifierTest::initTestCase()
{
    m_tempPath  = QString::fromLatin1(QTest::currentAppName());
    m_tempPath.replace(QLatin1String("./"), QString());
    m_tempDir   = WSToolUtils::makeTemporaryDir(m_tempPath.toLatin1().data());
    qCDebug(DIGIKAM_TESTS_LOG) << "Database Dir:" << m_tempDir.path();

    DbEngineParameters params;
    params.databaseType = DbEngineParameters::SQLiteDatabaseType();
    // params.setCoreDatabasePath(m_tempDir.path() + QLatin1String("/digikam-core-test.db"));
    // params.setThumbsDatabasePath(m_tempDir.path() + QLatin1String("/digikam-thumbs-test.db"));
    params.setFaceDatabasePath(m_tempDir.path() + QLatin1String("/digikam-faces-test.db"));
    // params.setFaceDatabasePath(QString::fromUtf8(":memory:"));
    // params.setSimilarityDatabasePath(m_tempDir.path() + QLatin1String("/digikam-similarity-test.db"));
    params.legacyAndDefaultChecks();

    CoreDbAccess::setParameters(params, CoreDbAccess::MainApplication);

    m_idProvider = IdentityProvider::instance();

    QTest::qWait(2000);
}

void FaceClassifierTest::cleanupTestCase()
{
    WSToolUtils::removeTemporaryDir(m_tempPath.toLatin1().data());
}

void FaceClassifierTest::testInit()
{
    qCDebug(DIGIKAM_TESTS_LOG) << "Initializing FaceClassifier...";
    QVERIFY2(FaceClassifier::instance() != nullptr,
             "Cannot initialize FaceClassifier");
}

void FaceClassifierTest::testRetrain()
{
    QVERIFY2(!FaceClassifier::instance()->retrain(),
             "retraining failed");
}

void FaceClassifierTest::testPredictPositiveSmall()
{
    FaceClassifier* classifier = FaceClassifier::instance();

    // FaceScanSettings parameters;
    // parameters.recognizeAccuracy = 7;
    // parameters.recognizeModel = FaceScanSettings::FaceRecognitionModel::SFace;
    // classifier->setParameters(parameters);
    classifier->setThreshold(0.5);

    for (int i = 0; i < 1; ++i)
    {
        QMultiMap<QString, QString> attributes;
        // attributes.insert(QLatin1String("uuid"), QUuid::createUuid().toString());
        attributes.insert(QLatin1String("name"), QLatin1String("test_id_") + QString::number(i));
        Identity id = m_idProvider->addIdentity(attributes);
    }

    QList<QString> featureList;
    featureList.append(QLatin1String("1,-0.047281403,-0.07474381,0.18678404,-0.05520716,0.010853133,-0.14348315,0.008160604,0.09736014,0.10964414,0.04066433,-0.026659422,0.100382514,-0.14132598,-0.07574784,-0.08431036,0.05169243,0.004651547,0.07322956,-0.25055045,0.034236744,-0.019558707,-0.11333167,0.000951085,-0.037991114,0.025830604,-0.009498599,0.015018532,0.006168048,0.082809515,-0.07373069,-0.19332403,0.21315591,-0.030255774,-0.023226283,-0.06836336,-0.1935418,-0.09257777,0.031265114,0.037895836,0.19380751,-0.043753523,0.16488996,-0.06359069,-0.081164,0.0806125,-0.0672546,-0.11743675,0.0202263,-0.048734594,0.088581495,-0.139474,0.08210075,-0.004360999,-0.000611505,0.027147964,0.03460771,-0.15236597,-0.051156193,0.028382896,0.061550345,-0.10292523,0.006337127,0.16700515,0.08835749,-0.011415936,-0.098105215,0.015307047,-0.041633777,0.023416625,-0.1344745,-0.008491004,0.05564281,-0.039523218,-0.08386901,0.11139244,0.003178574,0.003832626,0.03700139,-0.15400001,-0.012633625,0.05243851,-0.09997221,0.03865657,-0.00394618,0.13090804,0.22784027,-0.13002598,0.023789762,0.13102293,0.007242764,-0.088376895,-0.09132104,0.000236351,0.14750421,-0.042197328,-0.08691427,-0.10907041,0.11416536,0.0546626,0.11419356,-0.10883488,-0.11407062,0.071064055,-0.036179367,-0.037827253,0.006766912,-0.07194462,-0.02112297,0.117017455,-0.01176431,0.009570991,-0.059715625,0.026351372,-0.040147442,0.10661339,0.05411962,-0.005610226,-0.0595129,0.09074693,-0.001376847,-0.06547876,0.003314705,0.03970739,0.022410264,-0.05971824,0.08515292,-0.114376545,-0.105876766"));
    featureList.append(QLatin1String("1,-0.17861661,-0.040475924,0.090081684,0.08168912,0.042131767,-0.105440564,0.05213071,0.10890906,-0.005793085,-0.027744493,-0.06156276,0.09604177,-0.040716294,-0.10973724,-0.04968564,-0.006188312,-0.028958596,0.09010428,-0.09254466,0.085760765,0.06879532,-0.1420803,0.003392484,-0.001823023,0.008288479,0.012042236,-0.009894501,-0.041561715,0.124355294,0.005234093,-0.22946352,0.20422767,0.046762377,-0.099779636,-0.08162261,-0.08667685,-0.06419169,0.021214036,-0.02164374,0.09546424,-0.06975184,0.1641133,-0.037354812,-0.05630515,0.063229874,-0.1760392,-0.1212846,-0.0339956,-0.025867945,0.13633148,-0.12441383,0.07568663,-0.059415016,-0.015307453,0.082054466,0.003850304,-0.07530409,-0.12822503,0.058756106,-0.01664572,-0.07626505,0.02793208,0.14851916,0.0970666,-0.028266465,-0.0731352,0.001720438,-0.13753337,0.05200308,-0.14790036,-0.09750276,0.10381213,-0.027783908,-0.1125119,0.11904662,0.040635206,-0.002530122,0.07392372,-0.13502833,-0.03529188,-0.004829353,-0.04418042,-0.007376272,0.057115436,0.09940613,0.22096556,-0.054663345,0.049966995,0.115494154,-0.006208859,-0.05916577,0.083571315,-0.000248464,0.20099033,-0.07125319,-0.002540504,-0.1812666,0.057650816,0.09328636,0.1498713,-0.070790075,-0.05284402,0.07324293,-0.04722559,-0.086268336,0.019139389,-0.061146956,-0.07422941,0.11213745,0.015419872,0.037457317,-0.085921705,0.09004879,-0.044826407,0.12882109,0.073522426,0.061204065,-0.07676954,0.039913606,-0.112205125,-0.059481956,0.064906724,0.010922485,-0.022747723,0.02731935,0.11116442,-0.10254583,-0.13259776"));

    for (int i = 0; i < featureList.size(); ++i)
    {
        QStringList parts = featureList.at(i).split(QLatin1Char(','), Qt::SkipEmptyParts);

        if (parts.size() < 129)
        {
            continue;
        }

        cv::Mat features(1, 128, CV_32F);

        for (int j = 0; j < 128; ++j)
        {
            features.at<float>(0, j) = parts[j + 1].toFloat();
        }

        m_idProvider->addTraining(m_idProvider->identity(parts[0].toInt()), QString::number(i), features);
    }

    classifier->retrain();

    QTest::qWait(2000);

    QString target(QLatin1String("-0.033131853,-0.07562129,0.094871,0.14901902,0.015341093,-0.067466564,0.058864925,0.045275092,0.0323881,0.05091712,-0.10240302,0.19989395,-0.06241081,-0.11798663,0.016763724,-0.028603876,-0.12620921,0.1303773,-0.03530589,0.09929939,0.009072037,-0.13666216,-0.042646896,-0.044234294,0.09710265,-0.027400719,-0.029403822,0.10160872,0.018439105,-0.04638693,-0.18293802,0.19380355,0.03261224,-0.10567152,-0.014626429,-0.10011008,-0.03258871,0.068059996,0.0299799,0.07671056,0.016370911,0.065420635,0.0489549,-0.12299267,-0.04350005,-0.1663782,-0.06277769,-0.06890516,-0.012865229,0.23691796,-0.16800533,0.03299991,0.06303085,-0.08636596,0.04823346,0.006761336,0.023643654,-0.09836512,0.05002582,-0.03013158,-0.07199467,-0.0326686,0.058208134,-0.024371404,0.06284339,0.017114758,0.016073754,-0.000603331,0.04744236,-0.19955628,0.018515898,0.011568921,-0.026412513,-0.09919079,0.11714681,0.012179423,0.039082535,0.042947087,-0.06547415,-0.07371305,0.026597435,-0.005376318,-0.028710958,0.0413312,0.111582965,0.19080253,-0.12795049,-0.054398283,0.1316155,-0.017513053,-0.19880845,0.084753424,-0.015010008,0.16122478,-0.11810096,-0.008177375,-0.11818214,-0.005423737,0.06063142,0.09651744,-0.14992692,-0.1637755,0.017979505,-0.048526037,-0.09138023,0.059961386,-0.11750289,0.011654021,0.119147666,-0.01367995,-0.000895048,0.020841349,0.068358265,0.009349358,0.10202134,0.034986634,0.0401149,-0.08917925,-0.031993087,-0.03073732,-0.03426356,0.0816176,0.09034271,0.036180798,-0.013713819,0.22150229,-0.110783465,-0.10790335"));
    QStringList parts = target.split(QLatin1Char(','), Qt::SkipEmptyParts);
    cv::Mat targetFeatures(1, 128, CV_32F);

    for (int j = 0; j < 128; ++j)
    {
        targetFeatures.at<float>(0, j) = parts[j].toFloat();
    }

    QVERIFY2(FaceClassifier::instance()->predict(targetFeatures) == m_idProvider->identity(1).id(),
             "positive predict failed");

    m_idProvider->clearAllTraining();
    classifier->retrain();
    QTest::qWait(2000);
}

void FaceClassifierTest::testPredictPositiveLarge()
{
    FaceClassifier* classifier = FaceClassifier::instance();

    FaceScanSettings parameters;
    parameters.recognizeAccuracy = 7;
    parameters.recognizeModel = FaceScanSettings::FaceRecognitionModel::SFace;
    classifier->setParameters(parameters);
    classifier->setThreshold(0.5);

    for (int i = 0; i < 9; ++i)
    {
        QMultiMap<QString, QString> attributes;
        // attributes.insert(QLatin1String("uuid"), QUuid::createUuid().toString());
        attributes.insert(QLatin1String("name"), QLatin1String("test_id_") + QString::number(i));
        Identity id = m_idProvider->addIdentity(attributes);
    }

    QList<QString> featureList;
    featureList.append(QLatin1String("1,-0.047281403,-0.07474381,0.18678404,-0.05520716,0.010853133,-0.14348315,0.008160604,0.09736014,0.10964414,0.04066433,-0.026659422,0.100382514,-0.14132598,-0.07574784,-0.08431036,0.05169243,0.004651547,0.07322956,-0.25055045,0.034236744,-0.019558707,-0.11333167,0.000951085,-0.037991114,0.025830604,-0.009498599,0.015018532,0.006168048,0.082809515,-0.07373069,-0.19332403,0.21315591,-0.030255774,-0.023226283,-0.06836336,-0.1935418,-0.09257777,0.031265114,0.037895836,0.19380751,-0.043753523,0.16488996,-0.06359069,-0.081164,0.0806125,-0.0672546,-0.11743675,0.0202263,-0.048734594,0.088581495,-0.139474,0.08210075,-0.004360999,-0.000611505,0.027147964,0.03460771,-0.15236597,-0.051156193,0.028382896,0.061550345,-0.10292523,0.006337127,0.16700515,0.08835749,-0.011415936,-0.098105215,0.015307047,-0.041633777,0.023416625,-0.1344745,-0.008491004,0.05564281,-0.039523218,-0.08386901,0.11139244,0.003178574,0.003832626,0.03700139,-0.15400001,-0.012633625,0.05243851,-0.09997221,0.03865657,-0.00394618,0.13090804,0.22784027,-0.13002598,0.023789762,0.13102293,0.007242764,-0.088376895,-0.09132104,0.000236351,0.14750421,-0.042197328,-0.08691427,-0.10907041,0.11416536,0.0546626,0.11419356,-0.10883488,-0.11407062,0.071064055,-0.036179367,-0.037827253,0.006766912,-0.07194462,-0.02112297,0.117017455,-0.01176431,0.009570991,-0.059715625,0.026351372,-0.040147442,0.10661339,0.05411962,-0.005610226,-0.0595129,0.09074693,-0.001376847,-0.06547876,0.003314705,0.03970739,0.022410264,-0.05971824,0.08515292,-0.114376545,-0.105876766"));
    featureList.append(QLatin1String("1,-0.17861661,-0.040475924,0.090081684,0.08168912,0.042131767,-0.105440564,0.05213071,0.10890906,-0.005793085,-0.027744493,-0.06156276,0.09604177,-0.040716294,-0.10973724,-0.04968564,-0.006188312,-0.028958596,0.09010428,-0.09254466,0.085760765,0.06879532,-0.1420803,0.003392484,-0.001823023,0.008288479,0.012042236,-0.009894501,-0.041561715,0.124355294,0.005234093,-0.22946352,0.20422767,0.046762377,-0.099779636,-0.08162261,-0.08667685,-0.06419169,0.021214036,-0.02164374,0.09546424,-0.06975184,0.1641133,-0.037354812,-0.05630515,0.063229874,-0.1760392,-0.1212846,-0.0339956,-0.025867945,0.13633148,-0.12441383,0.07568663,-0.059415016,-0.015307453,0.082054466,0.003850304,-0.07530409,-0.12822503,0.058756106,-0.01664572,-0.07626505,0.02793208,0.14851916,0.0970666,-0.028266465,-0.0731352,0.001720438,-0.13753337,0.05200308,-0.14790036,-0.09750276,0.10381213,-0.027783908,-0.1125119,0.11904662,0.040635206,-0.002530122,0.07392372,-0.13502833,-0.03529188,-0.004829353,-0.04418042,-0.007376272,0.057115436,0.09940613,0.22096556,-0.054663345,0.049966995,0.115494154,-0.006208859,-0.05916577,0.083571315,-0.000248464,0.20099033,-0.07125319,-0.002540504,-0.1812666,0.057650816,0.09328636,0.1498713,-0.070790075,-0.05284402,0.07324293,-0.04722559,-0.086268336,0.019139389,-0.061146956,-0.07422941,0.11213745,0.015419872,0.037457317,-0.085921705,0.09004879,-0.044826407,0.12882109,0.073522426,0.061204065,-0.07676954,0.039913606,-0.112205125,-0.059481956,0.064906724,0.010922485,-0.022747723,0.02731935,0.11116442,-0.10254583,-0.13259776"));
    featureList.append(QLatin1String("2,-0.046440978,0.008136075,0.10827469,0.02431576,0.01281467,-0.000180961,0.057968494,0.07656529,0.15599342,-0.050390642,0.031135002,-0.06643697,0.023876004,0.004409098,-0.09247863,0.056261897,0.035400365,-0.123071946,0.001635031,0.075200856,0.011758638,0.007476777,-0.1610957,-0.04996513,-0.060669944,-0.09387314,-0.006189549,0.02893175,-0.13086846,0.09140149,-0.08109905,0.024505569,-0.04951021,-0.07872567,-0.026720088,0.11595194,0.142271,0.14733964,-0.0300847,0.048522025,0.07389007,0.03190392,0.004429078,-0.031010361,-0.21125083,0.10685117,0.10752529,-0.008691691,-0.020576097,0.04893602,-0.07423451,-0.08378916,-0.091961935,0.13856234,0.09465348,0.051068366,0.107319884,0.085669644,-0.040877294,-0.055282954,-0.12530363,-0.02398158,0.062930755,-0.05164011,0.12038398,-0.088912055,-0.096101,0.038551223,0.028211422,-0.1280518,0.04637744,0.049485363,-0.10096312,-0.16105862,0.17980611,0.056187216,-0.14294955,0.07070137,-0.16327511,-0.03276137,-0.002184951,0.061346527,0.23512411,0.15005058,0.018260906,0.18302555,0.14221488,-0.062292084,-0.0885868,0.09915376,-0.062642835,-0.08487809,0.1095285,0.067248106,-0.16962007,-0.09049176,-0.10273298,-0.003016656,-0.049281213,-0.123564266,-0.06357984,-0.07571997,-0.036526598,0.083957374,-0.12765086,-0.04606615,-0.009421278,0.013232984,0.006882576,-0.0538015,0.07357936,0.08157764,0.020169437,0.1000084,0.096220225,-0.030184163,0.06141162,0.009470192,0.15158291,-0.055109676,0.010198104,0.069823824,-0.10214417,-0.046035793,-0.07270918,0.006874017,-0.15956429,-0.100938156"));
    featureList.append(QLatin1String("2,-0.045438293,0.122451685,0.044172965,0.07235132,0.07980927,-0.10903624,0.122112714,-0.013662659,0.06151699,-0.056031477,0.014504878,-0.022653993,0.13990709,-0.019964136,-0.057700887,0.10626077,0.07645508,-0.08510915,0.04228058,0.022068797,0.035022296,-0.059729394,-0.13117473,-0.06941038,-0.04681897,-0.086726695,-0.0740306,-0.021451185,-0.08066909,0.021152863,-0.12037684,0.08828405,0.04258104,-0.02847625,-0.047149446,0.0780645,0.13632795,0.13653506,0.056806598,0.053699035,0.04794449,0.047351066,-0.036866017,0.099353544,-0.24572752,0.052447878,0.17370546,-0.10063953,-0.03935418,0.05727679,-0.08395674,-0.053348135,0.07336896,0.17974268,0.07470763,-0.029260173,0.14887168,0.08700397,0.005499004,-0.007407344,-0.17723791,0.13948585,-0.054067682,0.029848775,0.12185353,-0.13379025,-0.12297138,0.031690057,0.04037903,-0.106467605,0.13896662,-0.057002887,-0.11702129,-0.12600273,0.10542595,-0.017875344,-0.06205757,0.038740996,-0.08314455,-0.02729106,0.014421283,0.015305565,0.13196781,0.08493751,0.08476643,0.044674534,0.061368167,-0.01598244,-0.0960765,0.085997224,0.02568488,0.005857835,0.19375159,0.11358538,-0.119300716,-0.02658672,-0.003028054,0.011505259,0.021989668,0.020994948,0.08047175,-0.11751372,-0.06137797,0.05664697,-0.12075389,-0.16311318,0.06551584,0.13397953,0.05555349,0.09468686,0.02203257,-0.000523049,-0.01588385,0.066162564,0.12915641,0.012311908,0.000342087,0.000403134,0.18081354,-0.029097848,-0.078452416,0.18740802,-0.067765325,0.08222777,0.011766153,0.04478508,-0.127295,-0.08108815"));
    featureList.append(QLatin1String("2,-0.043536983,0.063943304,-0.06384248,0.05413271,0.041636553,0.10398487,0.12799843,-0.07303958,-0.006411377,0.027973168,0.06633813,-0.038538896,0.027089559,0.100090705,-0.13287438,0.028979788,0.005111534,-0.05003496,-0.02191197,0.023291426,-0.05506127,0.014641101,-0.062912785,-0.088273466,-0.109896615,-0.031202255,0.09556985,0.026126541,-0.02687753,0.03289996,-0.06401611,0.05264278,0.047669865,0.000288729,-0.09339796,0.100968815,0.255587,0.11349144,0.10349267,0.0638206,0.102010824,-0.14707167,-0.052243322,0.068379015,-0.23278816,-0.109161735,0.16105224,-0.106053546,-0.031862274,0.07705447,-0.11374664,-0.006582835,-0.016367445,0.23272717,0.10651476,-0.06257271,0.2014026,0.012354907,0.14008477,-0.14532925,-0.101440296,0.13172363,-0.001242178,0.013726965,0.08533494,0.018620938,-0.09216568,-0.017594606,0.113416955,-0.040933263,0.064321436,-0.040697157,-0.09481748,-0.03717326,0.0639688,0.07639607,-0.026709912,-0.014555211,-0.058500703,-0.022562599,0.041648205,0.1008404,0.102785274,0.039538175,0.13481598,0.023728643,0.014529581,0.039329156,-0.084330656,-0.0005139,-0.01604114,0.04694758,0.16994803,0.047708575,-0.18718663,-0.08458251,0.043403525,0.04239872,-0.001097934,-0.012443224,0.04225502,-0.14159004,-0.03752708,0.06250024,-0.09456901,-0.17982824,0.022026846,0.047662638,0.03428581,-0.07923993,0.007791665,0.14506322,0.025640436,0.06840789,0.22840148,-0.08841811,-0.032941185,0.052459754,0.065382235,0.067142844,-0.075021274,0.08317158,-0.007221692,0.08283944,0.005957593,0.09062707,-0.10765318,-0.02592048"));
    featureList.append(QLatin1String("3,-0.08831926,0.089735195,-0.077017345,0.08152622,0.038784735,0.006889239,-0.00674994,0.068560265,-0.028147776,0.09466478,0.07362538,-0.039608203,0.00147303,-0.099209644,0.08289112,-0.039649334,-0.089045115,-0.036715716,-0.11125611,0.09069161,0.040403575,0.14599212,0.12586883,0.006524743,-0.043252297,-0.06319995,-0.034274403,0.08433114,-0.027227486,0.053675976,0.3101404,0.17554776,0.06346292,0.08382879,0.1376148,-0.105081275,-0.10041212,0.04382165,-0.082087316,-0.022649769,0.05839938,0.001154675,0.090127654,-0.036717884,0.005601532,-0.059277225,0.034965158,0.079683796,-0.07228321,0.036683362,-0.028155228,-0.13534431,-0.111597955,0.003228124,0.021940451,0.006409085,0.12621067,0.055114254,-0.2588602,-0.11937279,-0.003039059,0.1398185,-0.14792217,0.039950162,-0.056150086,0.002566557,0.05737644,-0.12919675,0.047984023,0.13781385,0.035481703,0.059636924,-0.07110271,-0.0605143,0.029616691,-0.08805048,0.079367585,-0.05965519,0.06739711,0.057299,-0.07844374,-0.042108882,-0.137428,0.002542189,-0.03542341,-0.07819629,-0.012900068,0.04896712,-0.015006186,-0.122049056,-0.13597645,0.013540057,-0.110730566,-0.09647387,-0.019951874,0.018499218,0.01898401,-0.012947147,-0.040201813,0.07903037,0.099868536,-0.19613856,0.012283932,0.03167211,0.06878161,0.05611151,0.06444308,0.042942684,0.03430104,0.06419586,0.10314341,0.053225055,0.026788441,-0.04939538,0.1788629,0.1964139,-0.06811801,-0.019821508,0.086848065,0.023224516,0.19799684,0.0482959,0.17603263,0.11754142,-0.00309324,-0.04419163,-0.08449487,0.021783963"));
    featureList.append(QLatin1String("3,-0.07736632,0.1076011,-0.16817962,0.02174541,-0.00753109,0.018346652,0.02041989,-0.008790209,-0.025698459,0.07180746,0.09254847,0.025181964,-0.076284386,0.035015363,0.14231208,-0.089535594,-0.006729899,-0.045665413,-0.07119386,-0.01764481,-0.015809793,0.08101377,-0.039441288,0.005188806,0.042729598,0.06979966,0.026858585,-0.029595915,0.04560224,0.07660327,0.2179529,0.26054195,0.06871948,0.03762776,0.008035598,-0.03353555,-0.11717614,-0.01031411,-0.106292665,-0.026790189,-0.022792807,-0.000905816,0.038292263,-0.14476953,-0.0303619,-0.023439575,-0.02606918,0.022513995,-0.09580567,0.065082274,-0.05384805,-0.087970145,-0.09110262,0.015350869,0.026190855,-0.07597605,0.058816582,0.05140765,-0.18754028,-0.09759752,-0.016408455,0.122544244,-0.13591698,0.060487907,0.036077365,-0.18571621,0.08424925,-0.05417947,0.05102463,0.18604062,0.10678552,0.013468322,0.048976455,-0.10368417,-0.011387347,-0.06953553,0.05295705,0.044994127,0.04833041,0.072297074,-0.15788028,-0.026813569,-0.022594035,-0.043266837,-0.09654308,-0.028580882,-0.07597516,0.02404522,0.015489358,-0.17684259,-0.07679287,0.101809844,0.034428015,0.12978067,-0.04164183,-0.005472221,-0.033714533,-0.05973433,0.018324163,0.18893,0.104530744,-0.18392062,-0.012703613,-0.044636972,0.047731407,0.12834308,-0.07646612,-0.020378495,-0.07919957,-0.06414256,0.0926562,0.09112063,-0.07509701,-0.041616213,0.17182885,0.23117988,-0.035839543,0.06274595,0.01781051,0.07240746,0.16107656,0.038557008,0.1994643,0.1317114,-0.000554039,0.07707839,-0.010222112,0.001415688"));
    featureList.append(QLatin1String("3,-0.09112029,0.05216353,-0.085060224,0.09387434,-0.030540388,0.1346437,0.093205154,0.015873205,-0.047510453,0.07978757,-0.009933884,0.040273294,0.046198074,-0.07320592,0.15843447,-0.05967768,-0.09059312,-0.055454023,-0.16012268,0.10611639,0.043847036,0.11233881,-0.014382063,-0.043179497,-0.027166223,-0.027264915,0.02255857,0.112222224,0.103233,0.13145775,0.3012259,0.20895405,0.0876816,0.055416815,0.06666743,0.012039672,-0.15632711,0.10421986,-0.009843633,-0.04628256,0.023405634,-0.0185902,0.07077836,-0.015442901,-0.001305956,-0.050230324,-0.044859443,0.07218997,-0.04153806,-0.007063946,0.003094757,-0.050456893,-0.12767531,0.003087706,0.04599068,-0.053304154,0.08343648,0.016588306,-0.2026236,-0.15254107,0.046416905,0.17458767,-0.14289016,0.02441661,-0.044407357,-0.013578275,-0.06143877,-0.10585717,0.13900806,0.109437846,0.031349614,0.08906614,0.04315364,-0.1102259,0.071651086,-0.17552741,0.020419884,-0.038060736,0.021608079,0.08349426,-0.09731322,-0.12886596,0.024765698,-0.039709255,-0.09152878,-0.04741968,0.016128885,0.021004315,-0.014800006,-0.12674545,-0.06674381,0.026797002,-0.10383171,-0.04478739,-0.068067975,0.06272391,0.004288096,0.054430373,-0.03318977,0.012401555,0.028289173,-0.18812683,0.01607709,0.04991065,0.037704416,0.04983652,-0.023686675,0.029630192,-0.029661953,0.002585867,0.074469104,-0.001580714,-0.06949863,0.016718827,0.11430151,0.18958177,-0.027098294,0.056382947,0.033130147,0.079407714,0.25825298,-0.007227142,0.18047711,0.09733504,0.019348769,0.017169569,-0.047089897,0.07770864"));
    featureList.append(QLatin1String("4,-0.11547907,0.05084539,0.025948986,0.07645218,-0.05244145,0.10114689,-0.040877797,-0.041701257,-0.09668003,-0.13886772,-0.03204962,0.08647229,-0.06676862,0.024240252,0.011280532,-0.08357899,-0.019506536,-0.24937549,-0.07253256,0.1698672,-0.13980967,-0.08748476,0.044683874,-0.0754935,-0.011016874,-0.06292134,-0.0355244,-0.0453734,-0.005541085,-0.04123493,0.18331319,0.1309753,-0.10158171,0.112506926,-0.055848494,0.14843434,-0.022847459,0.078895144,-0.011287795,0.100646764,-0.106305785,-0.039973255,-0.018638743,0.08630954,0.028397217,-0.037381213,-0.07984496,-0.055046983,-0.038157355,0.029875763,-0.000804651,0.20390815,-0.007990341,0.04034689,0.059468303,-0.0332341,0.10360797,0.05017535,-0.060534243,0.1262729,0.004275549,-0.15439092,-0.13871959,0.08535297,0.012754493,-0.08283842,0.035178855,-0.012292413,-0.07219086,-0.03914396,0.12102442,0.11278702,0.063748285,0.067289054,0.06787706,-0.0716278,0.02153956,-0.02497203,0.002421161,0.100709505,-0.14884606,-0.14506036,0.054656953,0.12123453,-0.10729137,-0.006988721,0.12914422,0.008609582,0.09603466,-0.14082038,-0.18505357,0.052416064,0.05615737,-0.05299662,-0.029394519,-0.07609373,-0.020312931,-0.0685074,-0.035594657,-0.09080197,-0.04416345,0.14935257,-0.07947567,0.061884597,0.13181612,0.06310356,0.00256797,-0.002171143,-0.008803752,0.010133051,0.03516867,-0.079087235,0.08229559,0.06925083,0.003574943,0.018410217,0.00267468,0.115779,0.003206589,0.02392496,-0.073439136,0.023718704,0.08875774,-0.018548729,0.14024338,-0.14806692,0.07746627,0.28942814"));
    featureList.append(QLatin1String("4,0.00592304,-0.05952925,0.007357889,0.16008998,-0.0539848,0.08862096,0.045261215,0.04969675,-0.09070944,-0.18991993,-0.03315544,-0.031819988,-0.18487903,0.037242413,0.032942988,0.005658116,0.006527033,-0.103298865,-0.14874884,0.16342092,-0.14797331,0.058579396,-0.054402493,-0.11043291,-0.09883324,0.012739044,-0.0677564,-0.01850895,0.009700549,-0.06574693,0.14913587,-0.04059463,-0.15312892,0.062158514,0.009621932,0.122653894,0.009795403,0.024798915,0.08896562,0.081584014,-0.107672684,-0.018779451,-0.07173892,-0.1287168,-0.017836284,0.03774792,-0.10660621,0.018570045,-0.093991585,-0.005659851,0.007972562,0.07346815,-0.15354012,0.029910652,0.14573793,-0.001637889,0.09746144,0.013582564,-0.13569835,0.17296813,-0.004987102,-0.1045527,-0.26450148,-0.030037558,-0.005709802,-0.082491145,0.095338784,-0.05244306,-0.041856695,0.050438777,0.084890366,0.09391597,0.065954894,0.14305252,-0.018567039,-0.016420422,-0.061724138,0.020969229,0.16132842,-0.087863036,-0.0996516,-0.010715236,-0.007590779,0.066081315,-0.09495834,-0.095081896,0.010906762,-0.016611118,-0.00376584,-0.04057311,-0.13313375,-0.04224761,0.12512279,-0.09370394,-0.060741924,-0.080405444,0.080033064,0.00836008,0.1344387,-0.075907715,-0.020645685,0.121596016,-0.23820552,0.07443526,0.07964618,0.04173623,-0.06545487,-0.015112747,0.06099095,-0.00860845,0.011072363,-0.056731187,0.07841301,0.041577186,-0.058205694,-0.03449965,-0.020953186,0.06799382,0.10133786,0.02394463,0.024150988,-0.109615184,0.118526675,0.1335706,0.09270456,-0.12336008,0.06990618,0.06518085"));
    featureList.append(QLatin1String("4,-0.20404804,-0.0629781,0.025607757,0.14429837,-0.05732933,0.20664524,0.07398014,0.03563041,-0.13777326,-0.16011311,0.035559896,0.110687196,-0.1527489,0.058832638,-0.05636094,-0.1026529,-0.09783718,-0.11808156,-0.11554517,0.0514963,-0.061426405,0.07093389,0.07961582,-0.06754486,0.022169294,-0.18494211,-0.06491569,-0.054400664,-0.106846176,-0.1354662,0.21058214,0.119595654,0.04951047,0.06969538,0.000295875,0.06197647,-0.004274519,0.096055575,0.05439373,0.012220907,-0.08037573,-0.00743207,0.013457825,0.1170898,0.026290568,0.046331275,0.053161412,-0.048007965,-0.07520844,0.006082993,-0.039196387,0.12819257,0.004157716,0.14148377,0.002584574,-0.03160202,0.1284609,0.042204756,-0.13929194,0.15740027,-0.021502001,-0.02225755,-0.0877016,0.058389604,-0.041869,-0.20923032,0.11021321,-0.016231243,0.002716681,-0.0533938,0.10224036,0.15644401,-0.084693596,0.060298964,0.043383565,-0.06023888,0.048027653,0.032202676,-0.06837919,0.040441666,-0.000217465,-0.00971583,0.000830009,0.016590375,-0.1129052,-0.06791808,0.04606425,-0.034243442,-0.022476044,-0.10465934,-0.15715241,0.121407576,0.17122841,-0.063692525,-0.06359672,0.02110726,0.019725706,-0.09203939,-0.061763864,-0.07531931,0.023634065,0.13391593,-0.056822605,0.0886033,0.08176069,0.02940008,-0.024178503,0.020812713,0.06676077,-0.002311,0.049829308,-0.065650485,0.054161724,0.12918329,-0.066527456,0.059477977,-0.07105476,0.086294055,-0.029341051,0.034014344,0.00390901,0.017264822,0.1514802,-0.02683219,0.1487143,-0.082645826,0.030694297,0.16044463"));
    featureList.append(QLatin1String("5,-0.07795735,-0.040321037,-0.03916085,-0.049651835,0.25862932,0.00570056,0.05127941,-0.0755388,-0.012383703,-0.043222494,0.020283882,-0.010268405,-0.122333966,0.201655,0.0891718,-0.04146528,-0.027884874,0.028898513,0.022080196,-0.033944726,-0.04714022,0.061591472,-0.09628513,-0.037333056,0.12423254,0.13183571,-0.07205409,0.019665109,0.019863427,0.13146563,-0.056478765,-0.06345649,0.036724497,0.19683975,-0.13060553,0.062413584,0.07721429,-0.007569716,-0.02162854,0.021784218,0.005435362,-0.039712153,-0.062368065,0.034120213,0.14996785,-0.010624458,-0.05417054,0.06938774,-0.0553041,0.09356648,-0.17368414,-0.039853655,0.065354936,0.01572459,-0.15649079,0.112004414,-0.08866589,0.19302167,-0.14500903,0.026237154,-0.12438341,0.12346215,-0.12401543,0.07383546,0.018893218,0.07229844,0.12626925,-0.15177162,0.018144745,0.054433744,0.014517739,0.002199197,0.00349711,0.06523317,0.1479535,-0.13599594,-0.08385247,0.060409095,0.022529155,-0.003056157,-0.02129507,0.13602184,0.029611846,0.14025858,-0.04860785,0.029346822,0.14026462,-0.053849325,0.045707244,-0.03836237,-0.04652785,0.13733897,0.010326879,-0.06713988,0.12336759,-0.018479057,0.07758854,-0.015011335,0.13079147,-0.055187408,-0.043825965,-0.031442482,0.12833372,-0.024037005,0.09816802,-0.09392294,0.05982848,-0.014342564,0.072206125,-0.045102175,0.024598202,0.08619783,-0.11890161,0.002871333,0.13305134,0.09743104,0.004175024,-0.008439376,-0.11187496,0.04519197,0.11910631,-0.098905884,-0.001157953,0.023954302,0.0950444,-0.13742279,-0.20224476,0.07897276"));
    featureList.append(QLatin1String("5,-0.14090647,-0.007487848,0.17760116,0.03588947,0.2051916,0.040901437,-0.009705161,-0.06929519,-0.03412522,-0.018413914,0.101039186,0.17555821,-0.06507414,0.13536519,-0.011061361,-0.032402862,0.043598615,-0.013525368,-0.10696575,-0.000719473,0.17711394,-0.033104017,-0.028818302,0.16961037,-0.04133971,0.028862637,0.13446474,-0.1534778,-0.1386749,0.019102972,0.0297433,0.043567974,-0.034726016,0.09388143,0.09418191,0.12731919,0.05452857,0.13162272,-0.034843612,0.04999253,-0.002355777,-0.21543019,-0.16623414,-0.19176432,0.006502694,-0.009012869,0.088009216,0.1561966,-0.03560267,-0.007928073,-0.1640113,-0.15652427,-0.008519698,0.005797132,-0.068770915,0.035707373,0.030556025,0.18891856,-0.041264523,0.15160821,0.013061114,-0.046333477,-0.062248733,0.002184014,-0.06064399,0.10912303,0.10588913,-0.10414468,0.19477001,0.07657914,0.035303343,-0.07135801,0.047804262,0.08059509,0.10144526,-0.013714016,0.06713691,0.001314854,-0.01320576,0.059806015,-0.12555116,0.04146455,-0.0972235,0.05764292,-0.03501805,0.015413937,0.105225064,-0.06345417,0.004656451,0.015420157,0.01872252,0.11380082,0.032093886,-0.07016314,0.026526963,-0.043710366,-0.003306464,0.019343736,0.04949016,0.0696808,-0.00544752,-0.06493515,-0.07297637,0.0307902,0.08954465,-0.029824851,0.099450864,-0.07918933,0.06931748,0.004433528,0.13585842,0.01105643,0.013556433,0.042037163,0.017943542,0.000865693,0.042620312,-0.0957204,-0.09201957,0.023823598,0.19917904,-0.07610468,-0.08220547,0.11819687,-0.1255285,-0.07595889,-0.023234002,0.012074869"));
    featureList.append(QLatin1String("5,-0.117300466,-0.095548324,0.11231685,-0.048945025,0.03300034,0.062749885,0.014153725,-0.05049832,-0.090198174,0.002780051,-0.012694638,0.014194406,-0.070404924,0.14344172,0.052630108,-0.008896269,0.053804677,-0.05897849,-0.18888377,0.09259972,0.07680262,-0.070871435,-0.022193586,0.104911715,0.1424036,0.10211243,0.11037226,-0.080042824,-0.017989777,-0.005148625,-0.000701318,-0.029386105,-0.014027699,0.001760798,-0.05810076,0.08769802,0.16628337,0.08490879,-0.17376162,-0.00905629,0.030765435,-0.15985647,-0.1758778,-0.008381427,0.14349987,0.004194099,-0.03872747,0.113849886,-0.025667166,-0.046972953,-0.1595554,-0.10178945,-0.08615257,-0.102117255,-0.1584018,0.09625884,0.050068233,0.1196113,-0.1098127,0.11275398,-0.04353949,0.029321352,-0.113295004,-0.015261777,-0.1440824,0.035024207,0.093775414,-0.08299377,0.13994908,-0.06996799,0.03505275,0.00541218,0.04836991,0.062846996,0.13184676,-0.05557119,0.029589739,0.028560946,-0.03416784,-0.020739194,-0.067474656,0.045077085,-0.001812051,0.12601334,-0.13353737,-0.025351193,0.20865725,-0.024525598,0.00693321,-0.00453,-0.08577354,-0.02926582,0.06286822,-0.10086907,0.07571242,-0.07346548,0.07837185,-0.119455755,0.0925096,0.060433608,-0.04678934,0.00496404,0.015911182,0.037638698,-0.009063775,-0.047853265,0.09472808,-0.15746357,0.075160384,-0.020410452,0.20584291,-0.015696038,0.023379864,0.009097371,0.13011725,0.13472494,-0.04425348,-0.076692946,-0.06380387,-0.014934169,0.20467858,-0.14044599,0.072639056,0.10593097,-0.057185926,-0.110910095,-0.040056244,0.007704658"));
    featureList.append(QLatin1String("6,0.03148283,-0.025467105,0.01529519,-0.02299552,-0.041317463,-0.04356684,-0.15240823,0.050490174,-0.18664372,-0.059275094,-0.08521575,0.050487824,-0.016088095,-0.06592162,0.040437855,-0.20043938,-0.09725563,0.015939416,-0.25833288,0.000290489,-0.1364842,0.18179868,0.20423415,0.020442868,0.049349822,-0.18931605,0.007404295,-0.016193822,0.030233944,0.04711624,0.048535764,0.012833919,0.027449802,0.014605246,0.06915407,0.026279025,-0.03019648,-0.11352715,0.17365922,0.19174902,0.000667694,0.098667115,-0.0282992,0.009424043,-0.00191977,-0.06394483,0.14817917,-0.010148539,0.13268094,0.002885249,0.13692275,-0.034959868,0.07309669,-0.16978195,-0.06553694,0.056005396,0.105809905,-0.09947254,0.088599086,0.005655663,-0.020994136,0.14042072,-0.0725077,-0.12749387,-0.01729017,-0.0781478,0.17799556,-0.001280295,0.1293052,-0.027913025,-0.08576736,-0.05416071,0.05270558,-0.03329489,0.002485163,0.046151754,-0.000822237,-0.10183951,-0.19205551,-0.11962796,0.03176982,-0.08801038,0.045819204,-0.004317629,-0.04954206,-0.015274318,0.018422421,0.0061532,0.08191602,0.010100524,0.08276258,0.11423143,0.027313221,0.07050606,0.054770354,-0.08991048,0.15085623,0.083393335,0.060986537,0.06996307,0.04601802,-0.114367254,-0.09571294,-0.024080662,-0.044832483,-0.114983715,-0.016226605,-0.04705191,-0.007932434,0.10824738,0.003448061,0.06285623,-0.13048825,0.038346816,0.1011377,-0.018078905,0.0358174,-0.11169952,0.027295506,0.08073782,-0.103552096,-0.061775923,-0.11819783,-0.014043894,-0.12943186,-0.038997587,-0.007860908,-0.024714801"));
    featureList.append(QLatin1String("6,-0.089653164,-0.13200466,0.11362688,-0.16038792,-0.07029565,0.04296877,-0.1336664,0.07619257,-0.24125315,-0.037441462,-0.15214984,0.003234855,-0.09065372,-0.030131895,-0.027168091,-0.18006079,-0.07916607,0.08234124,-0.25910124,-0.1158616,-0.073570274,0.18315235,0.11720578,0.10884729,0.08295445,-0.1274256,-0.023799788,-0.015763408,0.001402826,0.03954177,0.074109405,0.026443,-0.013224111,-0.075360134,0.067211755,-0.02949543,0.05620331,-0.0867135,0.05228954,0.06203099,0.041729424,0.11383395,-0.07288656,-0.03957948,0.033027407,-0.062269222,0.079148196,-0.046074275,0.091052465,-0.14168777,0.08887421,0.003628204,0.088243365,-0.032303017,-0.07076445,0.012634001,0.060360465,-0.03659106,-0.001509036,-0.02418925,-0.024139782,0.09917243,-0.1103007,-0.16289659,-0.001445711,-0.122837014,0.18152355,-0.012669059,0.12540618,-0.021170048,-0.17396814,0.006911188,0.00629963,-0.07443539,0.13613538,-0.07413616,0.005446789,-0.04835364,-0.16379318,-0.059407968,-0.079536535,-0.080861025,-0.011771593,-0.025903504,-0.09174306,0.010762722,0.02711469,-0.026386864,-0.032027226,-0.01332065,-0.057527762,0.034575753,0.10464467,0.10553624,-0.0489859,0.003892506,0.11587158,-0.06882781,-0.03829639,0.12952833,-0.026554527,-0.072082326,-0.04085873,-0.026502386,0.059933886,0.022228735,0.033632528,-0.10344322,0.027431073,0.12518996,0.034180414,0.037784282,-0.089254394,-0.07078463,0.109974995,0.001903983,-0.041434888,-0.16672565,0.099455655,0.10126619,-0.08095043,0.003913239,0.014348845,0.08584615,-0.15570743,-0.13383134,0.0249221,-0.002002071"));
    featureList.append(QLatin1String("6,-0.035962958,-0.087412596,0.052523714,-0.124708146,-0.02830468,-0.05355835,-0.08010086,0.012490444,-0.18358205,-0.088247366,-0.11362766,0.039667998,-0.07676698,-0.045507636,-0.029143471,-0.12137066,-0.08239712,0.07573567,-0.27135795,0.008981113,-0.086223535,0.19188434,0.10731391,0.083648644,0.023653833,-0.1333074,-0.07632276,0.018027661,0.03576472,0.041328713,0.11747267,0.005058223,-0.060919497,-0.011663549,0.069683515,0.053423475,-0.05153772,-0.057290573,0.13573186,0.112094454,0.12985851,0.17494512,-0.1462541,-0.082049735,-0.022026226,0.021383116,0.14859457,-0.051519636,0.056267783,-0.040282767,0.15538314,-0.038976006,0.14431451,-0.03771265,-0.003750571,0.07763968,0.078465484,-0.13985576,0.00957847,-0.047427885,-0.028711945,0.102958724,-0.08567547,-0.117106885,-0.026434487,-0.09445041,0.11678262,0.003007245,0.17555916,0.0838696,-0.11646479,-0.09931239,-0.038179692,-0.10499811,0.012356589,-0.08546592,-0.04716715,-0.039984778,-0.07626846,-0.09186697,-0.009538022,-0.07180876,-0.055968564,0.007556612,-0.12510006,-0.037710037,0.02029136,0.018160865,-0.04648168,0.03047281,-0.09697718,0.16055769,0.18773675,0.043447938,-0.007492924,-0.024616623,0.11875247,0.017217075,-0.066891804,0.14883366,0.0966848,0.007647621,-0.037976917,0.013509846,-0.044161595,-0.063028514,0.050213937,-0.08097201,0.047391027,0.120595746,0.053379487,0.06928991,-0.095255144,-0.056954496,0.092873186,0.059038468,0.047025256,-0.13925488,0.12301967,0.09896473,-0.06949686,-0.096590936,-0.08651804,0.042427883,-0.098497346,-0.04453477,-0.033606764,0.007679901"));
    featureList.append(QLatin1String("7,-0.047624726,-0.09987522,-0.040628225,0.17137522,-0.024567248,-0.04921244,-0.02462002,-0.023471866,-0.08621894,-0.012911261,0.01690468,0.19787116,-0.005275248,-0.08848327,-0.06624566,-0.17711566,0.026520874,-0.04608696,-0.21901056,-0.1045268,0.09815729,-0.023479216,-0.011333387,0.10032144,-0.08764893,0.04430601,-0.009669333,-0.06442134,0.01393107,0.06943583,0.08235323,0.020156212,0.10666319,0.13515301,0.12406595,-0.04370503,-0.00653678,0.082116604,0.17921942,0.17242226,0.13158107,-0.103829876,0.07133222,0.0919064,0.004976614,-0.090283684,-0.06541935,-0.08294803,0.10508099,-0.045801505,-0.019407073,0.03618171,0.121812135,-0.015148655,-0.047372997,-0.015206956,0.022081165,-0.016107697,0.042619493,-0.007853973,0.08062039,0.15945378,-0.12136048,-0.044747338,0.05263072,-0.08235396,0.002322235,-0.006656334,0.02856368,-0.05292185,-0.114061035,0.05502498,0.028430538,-0.06044435,0.0957403,-0.014654109,0.005040105,0.08512557,-0.046506267,-0.02912499,0.10898997,-0.021833336,-0.15805104,0.08157098,-0.024339247,-0.002606699,-0.012198912,-0.058523014,0.019845653,0.04011727,-0.017475167,0.09134518,0.07778627,0.045796786,0.02665587,0.07205152,0.07265013,0.018034974,-0.09152265,-0.023953065,0.17913555,-0.17685111,-0.047514893,-0.07184792,0.05892041,0.057181615,-0.05150321,0.11518818,0.034884974,0.013409345,-0.016442318,-0.17950457,-0.062888026,-0.15134698,0.11695361,0.09368743,0.13127328,0.08347665,-0.03764737,0.069736354,-0.06022517,0.18630645,-0.15214215,0.036310557,0.007878901,-0.08667562,-0.27119324,0.040681597"));
    featureList.append(QLatin1String("7,0.037800908,0.010644773,0.04034214,0.1457713,-0.040860135,0.07418263,-0.05575516,0.002166969,-0.07451106,0.007065914,-0.043477453,0.18977804,-0.06820431,-0.13207762,0.0695795,-0.12633741,-0.10585798,-0.12704286,-0.13929039,0.033442263,0.06721744,-0.05840756,0.06516241,0.0647665,-0.12430605,0.005490481,-0.045648877,-0.09488999,0.10775407,-0.017189791,0.1460801,-0.025172822,0.028776541,0.08551697,0.07788033,-0.000543235,0.000733696,0.124023944,0.09617697,0.115045175,0.124115825,-0.12467893,-0.044813775,0.098327115,0.05221218,-0.036286615,-0.004878964,0.000481606,0.10834086,-0.008211323,0.0197937,0.062020946,0.07832136,-0.14856087,0.042999294,0.0816035,0.007522665,-0.07007972,-0.082406856,-0.16280186,0.015554993,0.020633265,-0.16382362,-0.14756878,-0.03791145,-0.007680663,-0.026664073,-0.0337654,0.024130944,-0.085891694,-0.002112167,0.000416481,-0.07200035,-0.029030852,0.03174874,-0.047245197,-0.0777445,-0.029702393,-0.14309965,0.03198205,0.122108586,-0.002736188,-0.09812959,0.06708401,0.009850698,-0.16841567,0.115543574,-0.080665305,-0.14113012,0.04120782,-0.06779945,0.07278675,0.12784617,0.20061313,-0.0936409,0.04440849,0.007180985,-0.12985353,-0.07496788,0.08235001,0.07756315,-0.06247286,0.004422798,-0.047444712,-0.016921584,0.033465683,-0.082334764,0.06714375,0.123279005,0.03929325,0.07776902,-0.14711882,-0.08936313,-0.20644033,0.090284534,0.11412873,0.14689393,-0.08143168,0.025584513,-0.013795801,0.018466644,0.2344618,0.023540663,0.00494808,-0.034668073,-0.07033819,-0.16519912,0.03350467"));
    featureList.append(QLatin1String("7,-0.030102307,0.032898195,-0.024700366,0.14695176,-0.06671806,0.010268066,-0.09987117,-0.02471053,0.035617977,0.018984094,-0.017654136,0.061743416,0.016738595,-0.14857045,0.05054462,-0.11824659,-0.07500825,-0.1252292,-0.21697892,0.021075172,0.03651798,0.026972638,0.000937765,0.04767143,-0.15379068,0.010826027,0.007389783,-0.049164064,-0.075363144,-0.08544139,0.1280728,-0.17120701,-0.032835294,0.10105097,0.2072884,-0.12703662,0.002997127,0.1334371,0.17581092,0.09929496,0.12979762,-0.1097747,0.000916982,0.015134899,0.06336133,-0.031284865,-0.1106672,0.027843565,-0.007889348,-0.033249862,-0.025669413,0.09607621,0.06277311,-0.091320805,0.07705313,-0.054817263,-0.061819263,-0.1749025,-0.031848747,-0.16936733,-0.030547727,0.02873596,-0.09690472,-0.16075861,-0.032520883,0.029091325,-0.004190866,-0.08589643,0.04248055,-0.02664049,0.009159517,-0.048977327,-0.03286233,-0.031100554,0.09739128,-0.09152542,-0.04833123,0.011545296,-0.069564596,0.000654226,0.14774422,0.04174108,0.022445207,0.10404252,0.035098195,-0.0814097,0.030702302,-0.13222533,-0.02036468,0.012374822,0.062990494,0.10683118,0.17740513,0.10135231,0.002696689,0.1342811,-0.02904325,0.018245844,-0.088094324,0.13969037,0.068036504,-0.033786684,-0.019195952,0.10112207,-0.018461548,-0.001423759,-0.17673321,0.06051832,0.08546715,0.16501589,0.048940755,-0.12389954,-0.02818944,-0.14693195,0.12051079,0.049868114,0.12023851,0.027780147,0.05722447,-0.11074447,0.01260594,0.16374056,0.017917074,-0.06194891,0.03742735,0.023348553,-0.18253769,0.032990437"));
    featureList.append(QLatin1String("8,0.008958684,-0.11661742,0.14442825,0.071045384,0.17005529,0.008875784,-0.050029904,-0.03365135,-0.10947284,0.08243518,-0.06425265,-0.037482534,0.09475055,0.03194454,-0.006824969,-0.12110239,0.04930836,0.04384268,0.02746999,-0.10861392,-0.051072255,-0.020153193,-0.016473023,-0.09978471,0.021310728,-0.06529319,0.025337256,0.005331838,0.044497725,-0.0422814,-0.020150265,0.014348511,-0.012543638,-0.107472226,0.15347137,-0.09487832,-0.04170827,0.03172302,0.037315104,-0.13126159,-0.106643125,0.013818642,0.15121731,-0.009556481,-0.023336021,-0.15143089,-0.08248163,0.1756631,-0.032645985,-0.00643199,0.014563743,0.105180204,0.04585234,0.019365259,0.05400441,0.08829015,0.08065893,0.0028214,-0.085432254,0.09514258,0.16292472,0.027022326,-0.019839669,-0.17040056,-0.15383163,0.06667214,-0.088311814,0.004805075,-0.027603742,-0.16891108,-0.10009337,-0.060388427,0.068982154,-0.100250445,0.05819608,-0.07228805,-0.10056851,-0.14693272,0.004189019,0.023129296,0.18890065,0.018237997,-0.00400145,-0.012071714,-0.101927064,0.016651368,0.046089336,0.057045266,-0.027185256,-0.1582772,0.09155087,0.16046408,-0.09099564,-0.081276484,-0.058077622,-0.049383074,-0.09030782,0.11680724,0.029308233,-0.028060066,0.06669161,-0.15603569,0.07962713,0.100232445,-0.13660423,-0.03974044,0.1996709,-0.047425948,0.16772698,0.088104315,0.059858713,-0.08705801,0.012699877,-0.07973061,-0.09262377,0.019658929,-0.012279351,-0.0781675,0.07997285,-0.080362335,-0.070226304,0.09965314,0.005815777,0.08405282,0.022761062,-0.21508828,0.15244557,-0.019512953"));
    featureList.append(QLatin1String("8,-0.09108315,-0.051381297,0.15333685,0.03401844,0.112858385,-0.009750281,-0.05186768,-0.12765597,-0.17715245,0.011257608,-0.09234586,-0.07392116,0.058365036,0.08861454,0.016132882,-0.087471396,0.007371954,0.06509519,0.028325362,0.029150425,-0.1093593,-0.06845466,-0.006551254,-0.074685305,0.017924942,-0.00876343,0.08425733,0.035073604,0.051401056,0.005914256,0.053740315,-0.040458906,0.028871888,-0.124741085,0.15256149,-0.08520348,-0.06290025,0.06958716,0.11488479,-0.18340586,-0.03460724,0.023521284,0.14761777,0.030324114,-0.09245862,-0.09144933,-0.09007785,0.24267402,-0.13293467,-0.03749615,0.011767913,0.045045886,-0.011453197,-0.059120316,0.07790625,0.11406629,0.06646973,0.033711087,-0.054786626,0.033763904,0.012580044,-0.09022586,-0.03855314,-0.04351324,-0.15856451,0.07078864,-0.054431487,-0.014019017,-0.07106546,-0.1578532,-0.09739223,-0.0531049,0.036533445,-0.061522648,0.043956622,-0.08424154,-0.08374055,-0.015856486,-0.034485575,0.011753222,0.22962025,0.047165364,-0.008329037,0.040192246,-0.16392782,0.059134625,0.05927956,-0.010557449,-0.047247693,-0.12968446,0.1480315,0.16480817,-0.041338906,-0.061722472,-0.103715315,-0.020634083,-0.031857718,0.11443614,0.101449974,-1.45E-05,0.06955795,-0.15613873,0.13847445,0.031329997,-0.12684841,-0.015730038,0.11416471,0.03368336,0.12921067,0.07390748,0.051755473,-0.11196166,-0.06856464,-0.02039172,0.03180164,-0.025671856,-0.034011833,0.041718915,0.1565121,0.04355345,-0.09110928,0.058419593,0.07802289,-0.04444768,0.03882348,-0.25774428,0.022978358,0.087956615"));
    featureList.append(QLatin1String("8,-0.044647772,-0.037053794,0.000271566,0.03136802,0.16183947,0.016837258,-0.102938265,-0.028036548,-0.21242589,-0.004769536,-0.059296586,0.01702952,0.06523539,0.12769446,-0.048142843,-0.056094576,-0.016441938,0.0945962,0.06794592,0.06863166,0.015836412,-0.057997175,0.18908854,-0.051386632,0.01739271,-0.077426,0.026621757,0.038432658,-0.003826757,-0.02857512,0.025983518,0.059191752,0.05505977,-0.06054558,0.1235231,-0.14393422,-0.086842336,0.078060746,0.105329745,-0.16617246,0.033486694,-0.019158432,0.2718777,0.05673541,-0.0414066,-0.08394019,-0.096718274,0.22832553,-0.061533943,-0.058021832,-0.026232352,0.011026247,0.08396872,-0.014794924,0.06937726,0.07089225,-0.06627181,0.03184934,-0.047052346,0.044006605,0.05421976,0.07053735,-0.06257787,-0.09996612,-0.1177336,0.060829226,-0.07759955,-0.055617888,-0.031892195,-0.11790963,-0.08774588,-0.03757087,0.06225615,-0.1138197,-0.005963614,-0.06234844,0.0426097,-0.0754129,-0.14374092,0.040516138,0.17543414,-0.036924873,0.056372456,-0.010172626,-0.14330094,0.095186405,0.09861543,-0.10319938,0.002003776,-0.12003546,0.16094564,0.1333446,-0.080246955,-0.058611553,-0.10725184,-0.006958262,-0.04676596,0.10262353,0.06154277,-0.106976315,0.20885402,-0.1642281,0.05644939,-0.006885355,-0.04324316,0.045553334,0.16631533,-0.051461242,0.09427954,0.15188405,0.06810144,-0.1276755,-0.019412313,-0.027312495,-0.035024803,0.034875847,0.049586494,0.026889184,0.057696495,0.021249356,-0.051058766,-0.04734746,0.102888115,0.038155645,0.003822632,-0.14734082,0.06246976,0.016474416"));
    featureList.append(QLatin1String("9,-0.017681358,-0.16868366,-0.016081972,0.027365854,-0.070767716,0.067421526,-0.07018366,-0.06331891,-0.09117793,-0.052545056,-0.08057285,-0.04625068,0.036981486,-0.050162967,0.20734286,0.115241595,0.009629858,0.03732771,0.041336156,0.03981951,-0.04507619,0.15087079,0.11131904,-0.058354035,-0.03328704,0.09973452,-0.1590581,-0.053806078,-0.026614124,-0.04450503,0.12906599,0.026840923,-0.021339832,0.07800164,-0.013637835,-0.09171261,0.093142174,-0.111008614,0.076475255,-0.11971257,-0.055310443,-0.14535876,0.03313832,0.029191827,-0.04631473,-0.028511442,-0.03442309,-0.036889624,0.023324348,-0.07857378,0.001978214,0.14783828,0.09955418,0.08625607,0.12042394,0.02989964,-0.081907146,-0.06403094,-0.07580734,-0.0332133,-0.043111913,0.13002437,-0.26989162,-0.090344705,0.064498164,0.14849256,0.025090083,-0.07528961,-0.084795535,0.050769,0.12737884,0.029851189,-0.079269916,-0.01990741,0.018759524,-0.18493366,0.04427317,-0.05635997,-0.20207725,-0.005779908,0.079479106,-0.000741414,-0.040004052,-0.039974768,-0.07829343,-0.14327367,-0.017355239,-0.027427388,-0.03611182,0.001046836,-0.028951136,0.072610274,0.053645816,0.110420175,-0.008048656,0.13890605,-0.061543915,-0.045664676,-0.15517364,-0.12474742,0.10130069,0.010362899,0.06753187,-0.18672821,-0.009484253,0.055663474,-0.17000896,0.10334214,0.027523834,0.118958086,0.03993654,0.025548628,-0.101113744,-0.050344896,-0.011338642,0.08810284,-0.03235122,-0.011917093,-0.03504287,-0.137802,-0.12337219,-0.15419784,0.06324434,-0.07675014,0.008289741,0.073048405,-0.09688015,0.11748877"));
    featureList.append(QLatin1String("9,0.044558257,-0.10232294,0.00892196,-0.040603954,-0.014965476,0.01703672,-0.030703934,-0.13669226,-0.12943932,-0.019427627,-0.04030575,-0.009214093,0.05484753,-0.006614885,0.22860526,0.1145371,0.004891997,-0.00051228,-0.09240304,0.065633185,-0.1305526,0.17897849,0.04292986,-0.12321446,-0.122759834,-0.017137809,-0.09247006,-0.121672995,-0.058203,-0.04765434,0.14034344,0.014433652,-0.021393484,0.055895325,-0.005459944,-0.06482968,0.10700752,-0.017435227,0.034043513,-0.07022434,-0.055777147,-0.13712116,0.021305762,-0.06241115,-0.046159636,-0.020800116,-0.059697483,-0.037708268,-0.025319804,-0.062116507,0.035187185,0.07048868,0.057391632,0.051152077,0.09749907,-0.009157382,-0.077591985,-0.07313439,-0.03686724,-0.03526953,0.030199392,0.150953,-0.2621008,-0.0447054,0.03716735,0.1399681,-0.014480962,-0.04707088,-0.090522304,0.055509206,0.11639238,-0.01963534,-0.04874197,-0.049866214,0.06285745,-0.21362668,0.025316577,-0.060915958,-0.18501559,0.036692735,0.056612194,-0.01615217,-0.12165957,-0.050842013,-0.03601334,-0.17287175,0.029578561,-0.08507604,-0.010041956,-0.058073465,-0.02338968,0.07422886,0.08698078,0.13068368,-0.06575216,0.07389823,-0.048592743,-0.05279978,-0.15136565,-0.113655105,0.115682214,-0.029270573,0.054505784,-0.27430305,0.009302946,0.09857854,-0.14122263,0.09009408,0.091730446,0.06646037,0.03443477,-0.071920164,-0.042637788,-0.08032708,0.022431852,0.017654918,-0.035130896,-0.025049035,-0.06742007,-0.09723586,-0.07371661,-0.19358882,0.078396186,-0.069154195,0.019652842,0.043627866,-0.04310256,0.1435108"));
    featureList.append(QLatin1String("9,-0.002798333,-0.12077292,-0.057333674,0.004352375,0.010664081,0.08128363,-0.06626368,-0.004224084,-0.06679357,-0.047867082,-0.010788084,-0.1363495,0.042007986,0.010919338,0.2634184,0.08526185,-0.02496736,-0.01228172,-0.014125453,0.078364946,-0.092821166,0.17904013,0.10344774,-0.051840946,-0.035585675,0.059792504,-0.12115714,-0.056497853,-0.06935432,-0.017863637,0.14095943,-0.015146501,0.070025794,0.07844537,0.056143314,-0.073822975,0.032029726,-0.0977899,0.08615149,-0.10947449,-0.008555544,-0.10448342,0.031996712,-0.041456442,-0.022386668,-0.104733996,-0.093209855,0.013465384,-0.007755754,-0.062253755,0.034652032,0.13798583,0.06226848,0.088102624,0.1316565,-0.058573958,-0.06518924,-0.074716866,-0.06228978,-0.06548026,0.003626978,0.08988243,-0.15495038,-0.07083579,0.08669721,0.020084878,-0.0202643,-0.02582753,-0.08053067,0.009371233,0.14680949,0.050047155,-0.13336648,-0.09990556,0.091193974,-0.18861747,0.10865693,-0.07061652,-0.15539013,0.020419424,0.11862183,0.028545141,-0.041066755,-0.02658817,-0.121049814,-0.102518655,-0.020257512,-0.10663253,-0.016213292,-0.001734696,-0.14072515,-0.008847591,0.04481888,0.07115676,0.005566929,0.1179932,-0.00258786,-0.054046772,-0.053654093,-0.12095767,0.0890545,-0.019185886,0.083763264,-0.26423824,-0.049473766,0.082385056,-0.15718384,0.03770085,0.055839956,0.094891034,0.04825284,-0.020642802,-0.13490589,-0.088712074,-0.033482354,0.052345864,-0.04677371,-0.05132773,0.011183696,-0.076642,-0.027969737,-0.25080627,0.047376156,-0.100084715,0.021733232,0.012150607,-0.10595859,0.1732722"));

    for (int i = 0; i < featureList.size(); ++i)
    {
        QStringList parts = featureList.at(i).split(QLatin1Char(','), Qt::SkipEmptyParts);

        if (parts.size() < 129)
        {
            continue;
        }

        cv::Mat features(1, 128, CV_32F);

        for (int j = 0; j < 128; ++j)
        {
            features.at<float>(0, j) = parts[j + 1].toFloat();
        }

        m_idProvider->addTraining(m_idProvider->identity(parts[0].toInt()), QString::number(i), features);
    }

    classifier->retrain();

    QTest::qWait(2000);

    QString target(QLatin1String("-0.033131853,-0.07562129,0.094871,0.14901902,0.015341093,-0.067466564,0.058864925,0.045275092,0.0323881,0.05091712,-0.10240302,0.19989395,-0.06241081,-0.11798663,0.016763724,-0.028603876,-0.12620921,0.1303773,-0.03530589,0.09929939,0.009072037,-0.13666216,-0.042646896,-0.044234294,0.09710265,-0.027400719,-0.029403822,0.10160872,0.018439105,-0.04638693,-0.18293802,0.19380355,0.03261224,-0.10567152,-0.014626429,-0.10011008,-0.03258871,0.068059996,0.0299799,0.07671056,0.016370911,0.065420635,0.0489549,-0.12299267,-0.04350005,-0.1663782,-0.06277769,-0.06890516,-0.012865229,0.23691796,-0.16800533,0.03299991,0.06303085,-0.08636596,0.04823346,0.006761336,0.023643654,-0.09836512,0.05002582,-0.03013158,-0.07199467,-0.0326686,0.058208134,-0.024371404,0.06284339,0.017114758,0.016073754,-0.000603331,0.04744236,-0.19955628,0.018515898,0.011568921,-0.026412513,-0.09919079,0.11714681,0.012179423,0.039082535,0.042947087,-0.06547415,-0.07371305,0.026597435,-0.005376318,-0.028710958,0.0413312,0.111582965,0.19080253,-0.12795049,-0.054398283,0.1316155,-0.017513053,-0.19880845,0.084753424,-0.015010008,0.16122478,-0.11810096,-0.008177375,-0.11818214,-0.005423737,0.06063142,0.09651744,-0.14992692,-0.1637755,0.017979505,-0.048526037,-0.09138023,0.059961386,-0.11750289,0.011654021,0.119147666,-0.01367995,-0.000895048,0.020841349,0.068358265,0.009349358,0.10202134,0.034986634,0.0401149,-0.08917925,-0.031993087,-0.03073732,-0.03426356,0.0816176,0.09034271,0.036180798,-0.013713819,0.22150229,-0.110783465,-0.10790335"));
    QStringList parts = target.split(QLatin1Char(','), Qt::SkipEmptyParts);
    cv::Mat targetFeatures(1, 128, CV_32F);

    for (int j = 0; j < 128; ++j)
    {
        targetFeatures.at<float>(0, j) = parts[j].toFloat();
    }

    QVERIFY2(FaceClassifier::instance()->predict(targetFeatures) == m_idProvider->identity(1).id(),
             "positive predict failed");

    m_idProvider->clearAllTraining();
    classifier->retrain();
    QTest::qWait(2000);
}

void FaceClassifierTest::testPredictNegative()
{
    FaceClassifier* classifier = FaceClassifier::instance();

    // FaceScanSettings parameters;
    // parameters.recognizeAccuracy = 7;
    // parameters.recognizeModel = FaceScanSettings::FaceRecognitionModel::SFace;
    // classifier->setParameters(parameters);
    classifier->setThreshold(0.5);

    QMultiMap<QString, QString> attributes;
    attributes.insert(QLatin1String("uuid"), QUuid::createUuid().toString());
    attributes.insert(QLatin1String("name"), QLatin1String("test_id_01"));
    Identity id01 = m_idProvider->addIdentity(attributes);

    QList<QString> featureList;
    featureList.append(QLatin1String("-0.047281403,-0.07474381,0.18678404,-0.05520716,0.010853133,-0.14348315,0.008160604,0.09736014,0.10964414,0.04066433,-0.026659422,0.100382514,-0.14132598,-0.07574784,-0.08431036,0.05169243,0.004651547,0.07322956,-0.25055045,0.034236744,-0.019558707,-0.11333167,0.000951085,-0.037991114,0.025830604,-0.009498599,0.015018532,0.006168048,0.082809515,-0.07373069,-0.19332403,0.21315591,-0.030255774,-0.023226283,-0.06836336,-0.1935418,-0.09257777,0.031265114,0.037895836,0.19380751,-0.043753523,0.16488996,-0.06359069,-0.081164,0.0806125,-0.0672546,-0.11743675,0.0202263,-0.048734594,0.088581495,-0.139474,0.08210075,-0.004360999,-0.000611505,0.027147964,0.03460771,-0.15236597,-0.051156193,0.028382896,0.061550345,-0.10292523,0.006337127,0.16700515,0.08835749,-0.011415936,-0.098105215,0.015307047,-0.041633777,0.023416625,-0.1344745,-0.008491004,0.05564281,-0.039523218,-0.08386901,0.11139244,0.003178574,0.003832626,0.03700139,-0.15400001,-0.012633625,0.05243851,-0.09997221,0.03865657,-0.00394618,0.13090804,0.22784027,-0.13002598,0.023789762,0.13102293,0.007242764,-0.088376895,-0.09132104,0.000236351,0.14750421,-0.042197328,-0.08691427,-0.10907041,0.11416536,0.0546626,0.11419356,-0.10883488,-0.11407062,0.071064055,-0.036179367,-0.037827253,0.006766912,-0.07194462,-0.02112297,0.117017455,-0.01176431,0.009570991,-0.059715625,0.026351372,-0.040147442,0.10661339,0.05411962,-0.005610226,-0.0595129,0.09074693,-0.001376847,-0.06547876,0.003314705,0.03970739,0.022410264,-0.05971824,0.08515292,-0.114376545,-0.105876766"));
    featureList.append(QLatin1String("-0.17861661,-0.040475924,0.090081684,0.08168912,0.042131767,-0.105440564,0.05213071,0.10890906,-0.005793085,-0.027744493,-0.06156276,0.09604177,-0.040716294,-0.10973724,-0.04968564,-0.006188312,-0.028958596,0.09010428,-0.09254466,0.085760765,0.06879532,-0.1420803,0.003392484,-0.001823023,0.008288479,0.012042236,-0.009894501,-0.041561715,0.124355294,0.005234093,-0.22946352,0.20422767,0.046762377,-0.099779636,-0.08162261,-0.08667685,-0.06419169,0.021214036,-0.02164374,0.09546424,-0.06975184,0.1641133,-0.037354812,-0.05630515,0.063229874,-0.1760392,-0.1212846,-0.0339956,-0.025867945,0.13633148,-0.12441383,0.07568663,-0.059415016,-0.015307453,0.082054466,0.003850304,-0.07530409,-0.12822503,0.058756106,-0.01664572,-0.07626505,0.02793208,0.14851916,0.0970666,-0.028266465,-0.0731352,0.001720438,-0.13753337,0.05200308,-0.14790036,-0.09750276,0.10381213,-0.027783908,-0.1125119,0.11904662,0.040635206,-0.002530122,0.07392372,-0.13502833,-0.03529188,-0.004829353,-0.04418042,-0.007376272,0.057115436,0.09940613,0.22096556,-0.054663345,0.049966995,0.115494154,-0.006208859,-0.05916577,0.083571315,-0.000248464,0.20099033,-0.07125319,-0.002540504,-0.1812666,0.057650816,0.09328636,0.1498713,-0.070790075,-0.05284402,0.07324293,-0.04722559,-0.086268336,0.019139389,-0.061146956,-0.07422941,0.11213745,0.015419872,0.037457317,-0.085921705,0.09004879,-0.044826407,0.12882109,0.073522426,0.061204065,-0.07676954,0.039913606,-0.112205125,-0.059481956,0.064906724,0.010922485,-0.022747723,0.02731935,0.11116442,-0.10254583,-0.13259776"));

    for (int i = 0; i < featureList.size(); ++i)
    {
        QStringList parts = featureList.at(i).split(QLatin1Char(','), Qt::SkipEmptyParts);

        if (parts.size() < 128)
        {
            continue;
        }

        cv::Mat features(1, 128, CV_32F);

        for (int j = 0; j < 128; ++j)
        {
            features.at<float>(0, j) = parts[j].toFloat();
        }

        m_idProvider->addTraining(id01, QString::number(i), features);
    }

    classifier->retrain();

    QTest::qWait(2000);

    QString target(QLatin1String("-0.14108169,-0.06561462,0.009795716,0.071816996,0.14977641,0.10411683,-0.03208957,-0.004286174,-0.008501582,0.18897426,-0.046909552,-0.02019653,-0.04317547,0.15274397,-0.003716695,-0.028482374,-0.020939734,0.004936259,-0.27139637,-0.061207697,0.03332451,0.048427474,0.07209493,-0.10948147,0.06330159,0.1785959,-0.045840677,0.07023985,0.00163812,-0.0877878,0.02441841,0.11810749,0.07067695,0.001932303,0.08060627,0.034438003,-0.0287355,0.03926437,-0.12168494,-0.030866085,0.015406848,-0.021104082,0.12414389,-0.05492533,0.10626195,0.149341,0.092160136,0.04386895,0.025270645,-0.042089704,-0.003659044,0.18496777,0.11782783,-0.1442408,-0.07795319,0.06540441,-0.15748623,-0.013344835,-0.14350589,0.07744578,-0.016284112,-0.045901105,-0.02999994,-0.004157805,0.040422987,-0.019417264,0.009126312,0.086644664,0.01985672,0.041636765,0.1242433,0.001259675,-0.0938781,-0.13900046,0.006132674,-0.14529677,0.005598851,0.011755991,-0.049416844,-0.077402875,-0.12591583,0.07332009,-0.031825338,0.057976797,0.15895668,-0.025286842,0.022197401,0.14661717,0.029736022,-0.0376908,0.1836274,0.004002347,0.15106769,0.011871814,0.061211545,-0.08930394,0.10613474,-0.026596006,0.114169866,-0.101550445,-0.000448593,-0.14716814,-0.002353969,-0.12671897,0.1041017,-0.010145188,-0.09487001,-0.24051222,-0.01843692,0.025201315,-0.13700871,-0.002218879,-0.03464856,-0.076138966,0.14355673,0.017813463,-0.042478893,0.05054796,0.027996344,-0.086326964,0.15646926,0.035591044,0.05337985,-0.04338531,0.008414595,0.008075785,-0.08128672,-0.005482423"));
    QStringList parts = target.split(QLatin1Char(','), Qt::SkipEmptyParts);
    cv::Mat targetFeatures(1, 128, CV_32F);

    for (int j = 0; j < 128; ++j)
    {
        targetFeatures.at<float>(0, j) = parts[j].toFloat();
    }

    QVERIFY2(FaceClassifier::instance()->predict(targetFeatures) == -1,
             "negative predict failed");

    m_idProvider->clearAllTraining();
    classifier->retrain();

    QTest::qWait(2000);
}

#include "moc_faceclassifier_utest.cpp"
