#!/usr/bin/env python

"""
Configuration management.

Copyright (C) 2017 Paul Boddie <paul@boddie.org.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
this program.  If not, see <http://www.gnu.org/licenses/>.
"""

from compiler import parseFile
from os.path import isfile, join, split
from compiler.ast import Assign, AssName, Const, Discard, Module, Name, Stmt

config_path = [".", "/etc/imip-agent", split(__file__)[0]]

builtins = {
    "False" : False,
    "None" : None,
    "True" : True,
    }

def get_config(filename="config.txt", path=config_path):

    "Obtain the configuration from 'filename', searching 'path' for the file."

    for dirname in path:
        pathname = join(dirname, filename)
        if isfile(pathname):
            module = parseFile(pathname)
            break
    else:
        return {}

    return get_config_data(module, {})

def get_config_data(module, d):

    "Interpret the parsed 'module', storing mappings in 'd'."

    name = None

    for node in module.getChildNodes():
        if isinstance(node, (Module, Stmt)):
            get_config_data(node, d)
        elif isinstance(node, Discard):
            pass
        elif isinstance(node, Assign):
            if len(node.nodes) == 1 and isinstance(node.nodes[0], AssName):
                name = node.nodes[0].name
                if isinstance(node.expr, Const):
                    d[name] = node.expr.value
                elif isinstance(node.expr, Name):
                    d[name] = d.get(node.expr.name, builtins.get(node.expr.name))

    return d

# Expose settings via a module-level name.

settings = get_config()

# vim: tabstop=4 expandtab shiftwidth=4
