#!/usr/bin/env python

"""
Interpretation of vCalendar content.

Copyright (C) 2014, 2015, 2016 Paul Boddie <paul@boddie.org.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
this program.  If not, see <http://www.gnu.org/licenses/>.
"""

from bisect import bisect_left
from datetime import date, datetime, timedelta
from email.mime.text import MIMEText
from imiptools.dates import format_datetime, get_datetime, \
                            get_datetime_item as get_item_from_datetime, \
                            get_datetime_tzid, \
                            get_duration, get_period, get_period_item, \
                            get_recurrence_start_point, \
                            get_time, get_timestamp, get_tzid, to_datetime, \
                            to_timezone, to_utc_datetime
from imiptools.period import FreeBusyPeriod, Period, RecurringPeriod
from vCalendar import iterwrite, parse, ParseError, to_dict, to_node
from vRecurrence import get_parameters, get_rule
import email.utils

try:
    from cStringIO import StringIO
except ImportError:
    from StringIO import StringIO

class Object:

    "Access to calendar structures."

    def __init__(self, fragment):

        """
        Initialise the object with the given 'fragment'. This must be a
        dictionary mapping an object type (such as "VEVENT") to a tuple
        containing the object details and attributes, each being a dictionary
        itself.

        The result of parse_object can be processed to obtain a fragment by
        obtaining a collection of records for an object type. For example:

        l = parse_object(f, encoding, "VCALENDAR")
        events = l["VEVENT"]
        event = events[0]

        Then, the specific object must be presented as follows:

        object = Object({"VEVENT" : event})

        A convienience function is also provided to initialise objects:

        object = new_object("VEVENT")
        """

        self.objtype, (self.details, self.attr) = fragment.items()[0]

    def get_uid(self):
        return self.get_value("UID")

    def get_recurrenceid(self):

        """
        Return the recurrence identifier, normalised to a UTC datetime if
        specified as a datetime or date with accompanying time zone information,
        maintained as a date or floating datetime otherwise. If no recurrence
        identifier is present, None is returned.

        Note that this normalised form of the identifier may well not be the
        same as the originally-specified identifier because that could have been
        specified using an accompanying TZID attribute, whereas the normalised
        form is effectively a converted datetime value.
        """

        if not self.has_key("RECURRENCE-ID"):
            return None
        dt, attr = self.get_datetime_item("RECURRENCE-ID")

        # Coerce any date to a UTC datetime if TZID was specified.

        tzid = attr.get("TZID")
        if tzid:
            dt = to_timezone(to_datetime(dt, tzid), "UTC")
        return format_datetime(dt)

    def get_recurrence_start_point(self, recurrenceid, tzid):

        """
        Return the start point corresponding to the given 'recurrenceid', using
        the fallback 'tzid' to define the specific point in time referenced by
        the recurrence identifier if the identifier has a date representation.

        If 'recurrenceid' is given as None, this object's recurrence identifier
        is used to obtain a start point, but if this object does not provide a
        recurrence, None is returned.

        A start point is typically used to match free/busy periods which are
        themselves defined in terms of UTC datetimes.
        """

        recurrenceid = recurrenceid or self.get_recurrenceid()
        if recurrenceid:
            return get_recurrence_start_point(recurrenceid, tzid)
        else:
            return None

    def get_recurrence_start_points(self, recurrenceids, tzid):
        return [self.get_recurrence_start_point(recurrenceid, tzid) for recurrenceid in recurrenceids]

    # Structure access.

    def add(self, obj):

        "Add 'obj' to the structure."

        name = obj.objtype
        if not self.details.has_key(name):
            l = self.details[name] = []
        else:
            l = self.details[name]
        l.append((obj.details, obj.attr))

    def copy(self):
        return Object(self.to_dict())

    def get_items(self, name, all=True):
        return get_items(self.details, name, all)

    def get_item(self, name):
        return get_item(self.details, name)

    def get_value_map(self, name):
        return get_value_map(self.details, name)

    def get_values(self, name, all=True):
        return get_values(self.details, name, all)

    def get_value(self, name):
        return get_value(self.details, name)

    def get_utc_datetime(self, name, date_tzid=None):
        return get_utc_datetime(self.details, name, date_tzid)

    def get_date_value_items(self, name, tzid=None):
        return get_date_value_items(self.details, name, tzid)

    def get_date_value_item_periods(self, name, tzid=None):
        return get_date_value_item_periods(self.details, name, self.get_main_period(tzid).get_duration(), tzid)

    def get_period_values(self, name, tzid=None):
        return get_period_values(self.details, name, tzid)

    def get_datetime(self, name):
        t = get_datetime_item(self.details, name)
        if not t: return None
        dt, attr = t
        return dt

    def get_datetime_item(self, name):
        return get_datetime_item(self.details, name)

    def get_duration(self, name):
        return get_duration(self.get_value(name))

    # Serialisation.

    def to_dict(self):
        return to_dict(self.to_node())

    def to_node(self):
        return to_node({self.objtype : [(self.details, self.attr)]})

    def to_part(self, method, encoding="utf-8", line_length=None):
        return to_part(method, [self.to_node()], encoding, line_length)

    def to_string(self, encoding="utf-8", line_length=None):
        return to_string(self.to_node(), encoding, line_length)

    # Direct access to the structure.

    def has_key(self, name):
        return self.details.has_key(name)

    def get(self, name):
        return self.details.get(name)

    def keys(self):
        return self.details.keys()

    def __getitem__(self, name):
        return self.details[name]

    def __setitem__(self, name, value):
        self.details[name] = value

    def __delitem__(self, name):
        del self.details[name]

    def remove(self, name):
        try:
            del self[name]
        except KeyError:
            pass

    def remove_all(self, names):
        for name in names:
            self.remove(name)

    def preserve(self, names):
        for name in self.keys():
            if not name in names:
                self.remove(name)

    # Computed results.

    def get_main_period(self, tzid=None):

        """
        Return a period object corresponding to the main start-end period for
        the object.
        """

        (dtstart, dtstart_attr), (dtend, dtend_attr) = self.get_main_period_items()
        tzid = tzid or get_tzid(dtstart_attr, dtend_attr)
        return RecurringPeriod(dtstart, dtend, tzid, "DTSTART", dtstart_attr, dtend_attr)

    def get_main_period_items(self):

        """
        Return two (value, attributes) items corresponding to the main start-end
        period for the object.
        """

        dtstart, dtstart_attr = self.get_datetime_item("DTSTART")

        if self.has_key("DTEND"):
            dtend, dtend_attr = self.get_datetime_item("DTEND")
        elif self.has_key("DURATION"):
            duration = self.get_duration("DURATION")
            dtend = dtstart + duration
            dtend_attr = dtstart_attr
        else:
            dtend, dtend_attr = dtstart, dtstart_attr

        return (dtstart, dtstart_attr), (dtend, dtend_attr)

    def get_periods(self, tzid, end=None, inclusive=False):

        """
        Return periods defined by this object, employing the given 'tzid' where
        no time zone information is defined, and limiting the collection to a
        window of time with the given 'end'.

        If 'end' is omitted, only explicit recurrences and recurrences from
        explicitly-terminated rules will be returned.

        If 'inclusive' is set to a true value, any period occurring at the 'end'
        will be included.
        """

        return get_periods(self, tzid, end, inclusive)

    def has_period(self, tzid, period):

        """
        Return whether this object, employing the given 'tzid' where no time
        zone information is defined, has the given 'period'.
        """

        return period in self.get_periods(tzid, period.get_start_point(), inclusive=True)

    def has_recurrence(self, tzid, recurrenceid):

        """
        Return whether this object, employing the given 'tzid' where no time
        zone information is defined, has the given 'recurrenceid'.
        """

        start_point = self.get_recurrence_start_point(recurrenceid, tzid)
        for p in self.get_periods(tzid, start_point, inclusive=True):
            if p.get_start_point() == start_point:
                return True
        return False

    def get_active_periods(self, recurrenceids, tzid, end=None):

        """
        Return all periods specified by this object that are not replaced by
        those defined by 'recurrenceids', using 'tzid' as a fallback time zone
        to convert floating dates and datetimes, and using 'end' to indicate the
        end of the time window within which periods are considered.
        """

        # Specific recurrences yield all specified periods.

        periods = self.get_periods(tzid, end)

        if self.get_recurrenceid():
            return periods

        # Parent objects need to have their periods tested against redefined
        # recurrences.

        active = []

        for p in periods:

            # Subtract any recurrences from the free/busy details of a
            # parent object.

            if not p.is_replaced(recurrenceids):
                active.append(p)

        return active

    def get_freebusy_period(self, period, only_organiser=False):

        """
        Return a free/busy period for the given 'period' provided by this
        object, using the 'only_organiser' status to produce a suitable
        transparency value.
        """

        return FreeBusyPeriod(
            period.get_start_point(),
            period.get_end_point(),
            self.get_value("UID"),
            only_organiser and "ORG" or self.get_value("TRANSP") or "OPAQUE",
            self.get_recurrenceid(),
            self.get_value("SUMMARY"),
            get_uri(self.get_value("ORGANIZER"))
            )

    def get_participation_status(self, participant):

        """
        Return the participation status of the given 'participant', with the
        special value "ORG" indicating organiser-only participation.
        """
    
        attendees = uri_dict(self.get_value_map("ATTENDEE"))
        organiser = get_uri(self.get_value("ORGANIZER"))

        attendee_attr = attendees.get(participant)
        if attendee_attr:
            return attendee_attr.get("PARTSTAT", "NEEDS-ACTION")
        elif organiser == participant:
            return "ORG"

        return None

    def get_participation(self, partstat, include_needs_action=False):

        """
        Return whether 'partstat' indicates some kind of participation in an
        event. If 'include_needs_action' is specified as a true value, events
        not yet responded to will be treated as events with tentative
        participation.
        """

        return not partstat in ("DECLINED", "DELEGATED", "NEEDS-ACTION") or \
               include_needs_action and partstat == "NEEDS-ACTION" or \
               partstat == "ORG"

    def get_tzid(self):

        """
        Return a time zone identifier used by the start or end datetimes,
        potentially suitable for converting dates to datetimes.
        """

        if not self.has_key("DTSTART"):
            return None
        dtstart, dtstart_attr = self.get_datetime_item("DTSTART")
        if self.has_key("DTEND"):
            dtend, dtend_attr = self.get_datetime_item("DTEND")
        else:
            dtend_attr = None
        return get_tzid(dtstart_attr, dtend_attr)

    def is_shared(self):

        """
        Return whether this object is shared based on the presence of a SEQUENCE
        property.
        """

        return self.get_value("SEQUENCE") is not None

    def possibly_active_from(self, dt, tzid):

        """
        Return whether the object is possibly active from or after the given
        datetime 'dt' using 'tzid' to convert any dates or floating datetimes.
        """

        dt = to_datetime(dt, tzid)
        periods = self.get_periods(tzid)

        for p in periods:
            if p.get_end_point() > dt:
                return True

        return self.possibly_recurring_indefinitely()

    def possibly_recurring_indefinitely(self):

        "Return whether this object may recur indefinitely."

        rrule = self.get_value("RRULE")
        parameters = rrule and get_parameters(rrule)
        until = parameters and parameters.get("UNTIL")
        count = parameters and parameters.get("COUNT")

        # Non-recurring periods or constrained recurrences.

        if not rrule or until or count:
            return False

        # Unconstrained recurring periods will always lie beyond any specified
        # datetime.

        else:
            return True

    # Modification methods.

    def set_datetime(self, name, dt, tzid=None):

        """
        Set a datetime for property 'name' using 'dt' and the optional fallback
        'tzid', returning whether an update has occurred.
        """

        if dt:
            old_value = self.get_value(name)
            self[name] = [get_item_from_datetime(dt, tzid)]
            return format_datetime(dt) != old_value

        return False

    def set_period(self, period):

        "Set the given 'period' as the main start and end."

        result = self.set_datetime("DTSTART", period.get_start())
        result = self.set_datetime("DTEND", period.get_end()) or result
        if self.has_key("DURATION"):
            del self["DURATION"]

        return result

    def set_periods(self, periods):

        """
        Set the given 'periods' as recurrence date properties, replacing the
        previous RDATE properties and ignoring any RRULE properties.
        """

        old_values = set(self.get_date_value_item_periods("RDATE") or [])
        new_rdates = []

        if self.has_key("RDATE"):
            del self["RDATE"]

        main_changed = False

        for p in periods:
            if p.origin == "RDATE" and p != self.get_main_period():
                new_rdates.append(get_period_item(p.get_start(), p.get_end()))
            elif p.origin == "DTSTART":
                main_changed = self.set_period(p)

        if new_rdates:
            self["RDATE"] = new_rdates

        return main_changed or old_values != set(self.get_date_value_item_periods("RDATE") or [])

    def set_rule(self, rule):

        """
        Set the given 'rule' in this object, replacing the previous RRULE
        property, returning whether the object has changed. The provided 'rule'
        must be an item.
        """

        if not rule:
            return False

        old_rrule = self.get_item("RRULE")
        self["RRULE"] = [rule]
        return old_rrule != rule

    def set_exceptions(self, exceptions):

        """
        Set the given 'exceptions' in this object, replacing the previous EXDATE
        properties, returning whether the object has changed. The provided
        'exceptions' must be a collection of items.
        """

        old_exdates = set(self.get_date_value_item_periods("EXDATE") or [])
        if exceptions:
            self["EXDATE"] = exceptions
            return old_exdates != set(self.get_date_value_item_periods("EXDATE") or [])
        elif old_exdates:
            del self["EXDATE"]
            return True
        else:
            return False

    def update_dtstamp(self):

        "Update the DTSTAMP in the object."

        dtstamp = self.get_utc_datetime("DTSTAMP")
        utcnow = get_time()
        dtstamp = format_datetime(dtstamp and dtstamp > utcnow and dtstamp or utcnow)
        self["DTSTAMP"] = [(dtstamp, {})]
        return dtstamp

    def update_sequence(self, increment=False):

        "Set or update the SEQUENCE in the object."

        sequence = self.get_value("SEQUENCE") or "0"
        self["SEQUENCE"] = [(str(int(sequence) + (increment and 1 or 0)), {})]
        return sequence

    def update_exceptions(self, excluded, asserted):

        """
        Update the exceptions to any rule by applying the list of 'excluded'
        periods. Where 'asserted' periods are provided, exceptions will be
        removed corresponding to those periods.
        """

        old_exdates = self.get_date_value_item_periods("EXDATE") or []
        new_exdates = set(old_exdates)
        new_exdates.update(excluded)
        new_exdates.difference_update(asserted)

        if not new_exdates and self.has_key("EXDATE"):
            del self["EXDATE"]
        else:
            self["EXDATE"] = []
            for p in new_exdates:
                self["EXDATE"].append(get_period_item(p.get_start(), p.get_end()))

        return set(old_exdates) != new_exdates

    def correct_object(self, tzid, permitted_values):

        """
        Correct the object's period details using the given 'tzid' and
        'permitted_values'.
        """

        corrected = set()
        rdates = []

        for period in self.get_periods(tzid):
            corrected_period = period.get_corrected(permitted_values)

            if corrected_period is period:
                if period.origin == "RDATE":
                    rdates.append(period)
                continue

            if period.origin == "DTSTART":
                self.set_period(corrected_period)
                corrected.add("DTSTART")
            elif period.origin == "RDATE":
                rdates.append(corrected_period)
                corrected.add("RDATE")

        if "RDATE" in corrected:
            self.set_periods(rdates)

        return corrected

# Construction and serialisation.

def make_calendar(nodes, method=None):

    """
    Return a complete calendar node wrapping the given 'nodes' and employing the
    given 'method', if indicated.
    """

    return ("VCALENDAR", {},
            (method and [("METHOD", {}, method)] or []) +
            [("VERSION", {}, "2.0")] +
            nodes
           )

def make_freebusy(freebusy, uid, organiser, organiser_attr=None, attendee=None,
                  attendee_attr=None, period=None):
    
    """
    Return a calendar node defining the free/busy details described in the given
    'freebusy' list, employing the given 'uid', for the given 'organiser' and
    optional 'organiser_attr', with the optional 'attendee' providing recipient
    details together with the optional 'attendee_attr'.

    The result will be constrained to the 'period' if specified.
    """
    
    record = []
    rwrite = record.append
    
    rwrite(("ORGANIZER", organiser_attr or {}, organiser))

    if attendee:
        rwrite(("ATTENDEE", attendee_attr or {}, attendee)) 

    rwrite(("UID", {}, uid))

    if freebusy:

        # Get a constrained view if start and end limits are specified.

        if period:
            periods = freebusy.get_overlapping([period])
        else:
            periods = freebusy

        # Write the limits of the resource.

        if periods:
            rwrite(("DTSTART", {"VALUE" : "DATE-TIME"}, format_datetime(periods[0].get_start_point())))
            rwrite(("DTEND", {"VALUE" : "DATE-TIME"}, format_datetime(periods[-1].get_end_point())))
        else:
            rwrite(("DTSTART", {"VALUE" : "DATE-TIME"}, format_datetime(period.get_start_point())))
            rwrite(("DTEND", {"VALUE" : "DATE-TIME"}, format_datetime(period.get_end_point())))

        for p in periods:
            if p.transp == "OPAQUE":
                rwrite(("FREEBUSY", {"FBTYPE" : "BUSY"}, "/".join(
                    map(format_datetime, [p.get_start_point(), p.get_end_point()])
                    )))

    return ("VFREEBUSY", {}, record)

def parse_object(f, encoding, objtype=None):

    """
    Parse the iTIP content from 'f' having the given 'encoding'. If 'objtype' is
    given, only objects of that type will be returned. Otherwise, the root of
    the content will be returned as a dictionary with a single key indicating
    the object type.

    Return None if the content was not readable or suitable.
    """

    try:
        try:
            doctype, attrs, elements = obj = parse(f, encoding=encoding)
            if objtype and doctype == objtype:
                return to_dict(obj)[objtype][0]
            elif not objtype:
                return to_dict(obj)
        finally:
            f.close()

    # NOTE: Handle parse errors properly.

    except (ParseError, ValueError):
        pass

    return None

def parse_string(s, encoding, objtype=None):

    """
    Parse the iTIP content from 's' having the given 'encoding'. If 'objtype' is
    given, only objects of that type will be returned. Otherwise, the root of
    the content will be returned as a dictionary with a single key indicating
    the object type.

    Return None if the content was not readable or suitable.
    """

    return parse_object(StringIO(s), encoding, objtype)

def to_part(method, fragments, encoding="utf-8", line_length=None):

    """
    Write using the given 'method', the given 'fragments' to a MIME
    text/calendar part.
    """

    out = StringIO()
    try:
        to_stream(out, make_calendar(fragments, method), encoding, line_length)
        part = MIMEText(out.getvalue(), "calendar", encoding)
        part.set_param("method", method)
        return part

    finally:
        out.close()

def to_stream(out, fragment, encoding="utf-8", line_length=None):

    "Write to the 'out' stream the given 'fragment'."

    iterwrite(out, encoding=encoding, line_length=line_length).append(fragment)

def to_string(fragment, encoding="utf-8", line_length=None):

    "Return a string encoding the given 'fragment'."

    out = StringIO()
    try:
        to_stream(out, fragment, encoding, line_length)
        return out.getvalue()

    finally:
        out.close()

def new_object(object_type):

    "Make a new object of the given 'object_type'."

    return Object({object_type : ({}, {})})

def make_uid(user):

    "Return a unique identifier for a new object by the given 'user'."

    utcnow = get_timestamp()
    return "imip-agent-%s-%s" % (utcnow, get_address(user))

# Structure access functions.

def get_items(d, name, all=True):

    """
    Get all items from 'd' for the given 'name', returning single items if
    'all' is specified and set to a false value and if only one value is
    present for the name. Return None if no items are found for the name or if
    many items are found but 'all' is set to a false value.
    """

    if d.has_key(name):
        items = [(value or None, attr) for value, attr in d[name]]
        if all:
            return items
        elif len(items) == 1:
            return items[0]
        else:
            return None
    else:
        return None

def get_item(d, name):
    return get_items(d, name, False)

def get_value_map(d, name):

    """
    Return a dictionary for all items in 'd' having the given 'name'. The
    dictionary will map values for the name to any attributes or qualifiers
    that may have been present.
    """

    items = get_items(d, name)
    if items:
        return dict(items)
    else:
        return {}

def values_from_items(items):
    return map(lambda x: x[0], items)

def get_values(d, name, all=True):
    if d.has_key(name):
        items = d[name]
        if not all and len(items) == 1:
            return items[0][0]
        else:
            return values_from_items(items)
    else:
        return None

def get_value(d, name):
    return get_values(d, name, False)

def get_date_value_items(d, name, tzid=None):

    """
    Obtain items from 'd' having the given 'name', where a single item yields
    potentially many values. Return a list of tuples of the form (value,
    attributes) where the attributes have been given for the property in 'd'.
    """

    items = get_items(d, name)
    if items:
        all_items = []
        for item in items:
            values, attr = item
            if not attr.has_key("TZID") and tzid:
                attr["TZID"] = tzid
            if not isinstance(values, list):
                values = [values]
            for value in values:
                all_items.append((get_datetime(value, attr) or get_period(value, attr), attr))
        return all_items
    else:
        return None

def get_date_value_item_periods(d, name, duration, tzid=None):

    """
    Obtain items from 'd' having the given 'name', where a single item yields
    potentially many values. The 'duration' must be provided to define the
    length of periods having only a start datetime. Return a list of periods
    corresponding to the property in 'd'.
    """

    items = get_date_value_items(d, name, tzid)
    if not items:
        return items

    periods = []

    for value, attr in items:
        if isinstance(value, tuple):
            periods.append(RecurringPeriod(value[0], value[1], tzid, name, attr))
        else:
            periods.append(RecurringPeriod(value, value + duration, tzid, name, attr))

    return periods

def get_period_values(d, name, tzid=None):

    """
    Return period values from 'd' for the given property 'name', using 'tzid'
    where specified to indicate the time zone.
    """

    values = []
    for value, attr in get_items(d, name) or []:
        if not attr.has_key("TZID") and tzid:
            attr["TZID"] = tzid
        start, end = get_period(value, attr)
        values.append(Period(start, end, tzid=tzid))
    return values

def get_utc_datetime(d, name, date_tzid=None):

    """
    Return the value provided by 'd' for 'name' as a datetime in the UTC zone
    or as a date, converting any date to a datetime if 'date_tzid' is specified.
    If no datetime or date is available, None is returned.
    """

    t = get_datetime_item(d, name)
    if not t:
        return None
    else:
        dt, attr = t
        return dt is not None and to_utc_datetime(dt, date_tzid) or None

def get_datetime_item(d, name):

    """
    Return the value provided by 'd' for 'name' as a datetime or as a date,
    together with the attributes describing it. Return None if no value exists
    for 'name' in 'd'.
    """

    t = get_item(d, name)
    if not t:
        return None
    else:
        value, attr = t
        dt = get_datetime(value, attr)
        tzid = get_datetime_tzid(dt)
        if tzid:
            attr["TZID"] = tzid
        return dt, attr

# Conversion functions.

def get_address_parts(values):

    "Return name and address tuples for each of the given 'values'."

    l = []
    for name, address in values and email.utils.getaddresses(values) or []:
        if is_mailto_uri(name):
            name = name[7:] # strip "mailto:"
        l.append((name, address))
    return l

def get_addresses(values):

    """
    Return only addresses from the given 'values' which may be of the form
    "Common Name <recipient@domain>", with the latter part being the address
    itself.
    """

    return [address for name, address in get_address_parts(values)]

def get_address(value):

    "Return an e-mail address from the given 'value'."

    if not value: return None
    return get_addresses([value])[0]

def get_verbose_address(value, attr=None):

    """
    Return a verbose e-mail address featuring any name from the given 'value'
    and any accompanying 'attr' dictionary.
    """

    l = get_address_parts([value])
    if not l:
        return value
    name, address = l[0]
    if not name:
        name = attr and attr.get("CN")
    if name and address:
        return "%s <%s>" % (name, address)
    else:
        return address

def is_mailto_uri(value):
    return value.lower().startswith("mailto:")

def get_uri(value):

    "Return a URI for the given 'value'."

    if not value: return None
    return is_mailto_uri(value) and ("mailto:%s" % value[7:]) or \
           ":" in value and value or \
           "mailto:%s" % get_address(value)

def uri_parts(values):

    "Return any common name plus the URI for each of the given 'values'."

    return [(name, get_uri(address)) for name, address in get_address_parts(values)]

uri_value = get_uri

def uri_values(values):
    return map(get_uri, values)

def uri_dict(d):
    return dict([(get_uri(key), value) for key, value in d.items()])

def uri_item(item):
    return get_uri(item[0]), item[1]

def uri_items(items):
    return [(get_uri(value), attr) for value, attr in items]

# Operations on structure data.

def is_new_object(old_sequence, new_sequence, old_dtstamp, new_dtstamp, ignore_dtstamp):

    """
    Return for the given 'old_sequence' and 'new_sequence', 'old_dtstamp' and
    'new_dtstamp', and the 'ignore_dtstamp' indication, whether the object
    providing the new information is really newer than the object providing the
    old information.
    """

    have_sequence = old_sequence is not None and new_sequence is not None
    is_same_sequence = have_sequence and int(new_sequence) == int(old_sequence)

    have_dtstamp = old_dtstamp and new_dtstamp
    is_old_dtstamp = have_dtstamp and new_dtstamp < old_dtstamp or old_dtstamp and not new_dtstamp

    is_old_sequence = have_sequence and (
        int(new_sequence) < int(old_sequence) or
        is_same_sequence and is_old_dtstamp
        )

    return is_same_sequence and ignore_dtstamp or not is_old_sequence

def check_delegation(attendee_map, attendee, attendee_attr):

    """
    Using the 'attendee_map', check the attributes for the given 'attendee'
    provided as 'attendee_attr', following the delegation chain back to the
    delegators and forward again to yield the delegate identities in each
    case. Pictorially...

    attendee -> DELEGATED-FROM -> delegator
           ? <-  DELEGATED-TO  <---

    Return whether 'attendee' was identified as a delegate by providing the
    identity of any delegators referencing the attendee.
    """

    delegators = []

    # The recipient should have a reference to the delegator.

    delegated_from = attendee_attr and attendee_attr.get("DELEGATED-FROM")
    if delegated_from:

        # Examine all delegators.

        for delegator in delegated_from:
            delegator_attr = attendee_map.get(delegator)

            # The delegator should have a reference to the recipient.

            delegated_to = delegator_attr and delegator_attr.get("DELEGATED-TO")
            if delegated_to and attendee in delegated_to:
                delegators.append(delegator)

    return delegators

def get_periods(obj, tzid, end=None, inclusive=False):

    """
    Return periods for the given object 'obj', employing the given 'tzid' where
    no time zone information is available (for whole day events, for example),
    confining materialised periods to before the given 'end' datetime.

    If 'end' is omitted, only explicit recurrences and recurrences from
    explicitly-terminated rules will be returned.

    If 'inclusive' is set to a true value, any period occurring at the 'end'
    will be included.
    """

    rrule = obj.get_value("RRULE")
    parameters = rrule and get_parameters(rrule)

    # Use localised datetimes.

    main_period = obj.get_main_period(tzid)

    dtstart = main_period.get_start()
    dtstart_attr = main_period.get_start_attr()

    # Attempt to get time zone details from the object, using the supplied zone
    # only as a fallback.

    obj_tzid = obj.get_tzid()

    if not rrule:
        periods = [main_period]

    elif end or parameters and parameters.has_key("UNTIL") or parameters.has_key("COUNT"):

        # Recurrence rules create multiple instances to be checked.
        # Conflicts may only be assessed within a period defined by policy
        # for the agent, with instances outside that period being considered
        # unchecked.

        selector = get_rule(dtstart, rrule)
        periods = []

        until = parameters.get("UNTIL")
        if until:
            until_dt = to_timezone(get_datetime(until, dtstart_attr), obj_tzid)
            end = end and min(until_dt, end) or until_dt
            inclusive = True

        for recurrence_start in selector.materialise(dtstart, end, parameters.get("COUNT"), parameters.get("BYSETPOS"), inclusive):
            create = len(recurrence_start) == 3 and date or datetime
            recurrence_start = to_timezone(create(*recurrence_start), obj_tzid)
            recurrence_end = recurrence_start + main_period.get_duration()
            periods.append(RecurringPeriod(recurrence_start, recurrence_end, tzid, "RRULE", dtstart_attr))

    else:
        periods = []

    # Add recurrence dates.

    rdates = obj.get_date_value_item_periods("RDATE", tzid)
    if rdates:
        periods += rdates

    # Return a sorted list of the periods.

    periods.sort()

    # Exclude exception dates.

    exdates = obj.get_date_value_item_periods("EXDATE", tzid)

    if exdates:
        for period in exdates:
            i = bisect_left(periods, period)
            while i < len(periods) and periods[i] == period:
                del periods[i]

    return periods

def get_sender_identities(mapping):

    """
    Return a mapping from actual senders to the identities for which they
    have provided data, extracting this information from the given
    'mapping'.
    """

    senders = {}

    for value, attr in mapping.items():
        sent_by = attr.get("SENT-BY")
        if sent_by:
            sender = get_uri(sent_by)
        else:
            sender = value

        if not senders.has_key(sender):
            senders[sender] = []

        senders[sender].append(value)

    return senders

def get_window_end(tzid, days=100):

    """
    Return a datetime in the time zone indicated by 'tzid' marking the end of a
    window of the given number of 'days'.
    """

    return to_timezone(datetime.now(), tzid) + timedelta(days)

# vim: tabstop=4 expandtab shiftwidth=4
