#!/usr/bin/env python

"""
Access-control-related scheduling functionality.

Copyright (C) 2016 Paul Boddie <paul@boddie.org.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
this program.  If not, see <http://www.gnu.org/licenses/>.
"""

from imiptools.data import get_address, get_addresses
from imiptools.text import parse_line
import codecs

def access_control_list(handler, args):

    """
    Attempt to schedule the current object of the given 'handler' using an
    access control list provided in the given 'args', applying it to the
    organiser.
    """

    # Obtain either a file from the user's preferences directory...

    if not args:
        acl = handler.get_preferences().get("acl")
        lines = acl.strip().split("\n")

    # Or obtain the contents of a specific file.

    else:
        try:
            f = codecs.open(args[0], encoding="utf-8")
        except IOError:
            return None
        try:
            lines = f.readlines()
        finally:
            f.close()

    # Use the current object's identities with the ACL rules.

    organiser = get_address(handler.obj.get_value("ORGANIZER"))
    attendees = get_addresses(handler.obj.get_values("ATTENDEE"))

    response = None

    for line in lines:
        parts = parse_line(line.strip())

        # Skip empty lines.

        if not parts:
            continue

        # Accept either a single word with an action or a rule.
        # NOTE: Should signal an error with the format.

        if len(parts) == 1:
            action = parts[0]
        elif len(parts) >= 3:
            action, role, identities = parts[0], parts[1], map(get_address, parts[2:])
        else:
            return None

        if action.lower() == "accept":
            result = "ACCEPTED"
        elif action.lower() in ["decline", "reject"]:
            result = "DECLINED"
        else:
            return None

        # With only an action, prepare a default response in case none of
        # the rules match.

        if len(parts) == 1:
            response = result
            continue

        # Where no default has been set, use an implicit default based on
        # the action appearing in a rule.

        elif not response:
            response = result == "ACCEPTED" and "DECLINED" or "ACCEPTED"

        # Interpret a rule, attempting to match identities to properties.

        if role.lower() in ["organiser", "organizer"]:
            match = organiser in identities
        elif role.lower() in ["attendee", "attendees"]:
            match = set(attendees).intersection(identities)
        else:
            return None

        # Use the result of any match.

        if match:
            response = result

    return standard_responses(handler, response)

def same_domain_only(handler, args):

    """
    Attempt to schedule the current object of the given 'handler' if the
    organiser employs an address in the same domain as the resource.
    """

    organiser = get_address(handler.obj.get_value("ORGANIZER"))
    user = get_address(handler.user)

    organiser_domain = organiser.rsplit("@", 1)[-1]
    user_domain = user.rsplit("@", 1)[-1]
    
    response = organiser_domain == user_domain and "ACCEPTED" or "DECLINED"
    return standard_responses(handler, response)

def standard_responses(handler, response):

    """
    Using 'handler' to translate descriptions, return a tuple containing the
    'response' and a suitable description.
    """

    _ = handler.get_translator()

    if response == "ACCEPTED":
        return response, _("The recipient has scheduled the requested period.")
    elif response == "DECLINED":
        return response, _("The recipient has refused to schedule the requested period.")
    else:
        return response, None

# Registry of scheduling functions.

scheduling_functions = {
    "access_control_list" : [access_control_list],
    "same_domain_only" : [same_domain_only],
    }

# Registries of locking and unlocking functions.

locking_functions = {}
unlocking_functions = {}

# Registries of listener functions.

confirmation_functions = {}
retraction_functions = {}

# vim: tabstop=4 expandtab shiftwidth=4
