ITK Release 5.0 Beta 1
======================

We are happy to announce the first beta release for ITK 5. This release features an additional, more Pythonic interface to filters. This interface is more familiar in Python and facilitates rapid prototyping. ITK 5.0 Beta 1 also includes many performance-related improvements, such as a pure C++11 thread pool by default, an HDF5 update to 1.10, new frequency domain iterators, new neighborhood range classes, and a new multi-dimensional, multi-component superpixel class.

For an overview of ITK 5 performance-related changes and the transition to modern C++, see the [ITK 5 Alpha 1: Modern C++](https://discourse.itk.org/t/itk-5-0-alpha-1-modern-c/843/7) and [ITK 5 Alpha 2: Performance](https://discourse.itk.org/t/itk-5-0-alpha-2-performance/959) release announcements.

Python Interface
-----------------------

The traditional, object-oriented interface to ITK's Python classes allows the composition of filters in a pipeline. After setting up a chain of filters, `Update()` is called at the end of the pipeline. This facilitates [streaming of very large images](https://itk.org/ITKSoftwareGuide/html/Book1/ITKSoftwareGuide-Book1ch8.html#x54-1360008.3) that may exceed the system's memory capacity. For example,

```python
import itk

input_image = itk.imread('input.nrrd')
ImageType = type(input_image)
radius = 2

median_filter = itk.MedianImageFilter[ImageType, ImageType].New()
median_filter.SetInput(input_image)
median_filter.SetRadius(radius)
median_filter.Update()
output_image = median_filter.GetOutput()

itk.imwrite(output_image, 'output.nrrd')
```

While powerful, this interface is verbose.

A procedural interface is now available with [PEP8](https://www.python.org/dev/peps/pep-0008/)-compliant snake case naming of all filters that inherit from `itk.ProcessObject`. For example,

```python
import itk

input_image = itk.imread('input.nrrd')
radius = 2

output_image = itk.median_image_filter(input_image, radius=radius)

itk.imwrite(output_image, 'output.nrrd')
```

In this case, the type of the filter is implicitly determined by the input image type. Immediately, the filter is executed, and the result is directly returned. Object parameters defined by `Set*` methods can be specified in the filter call with keyword arguments. The Pythonic, snake case form of these parameters can also be used.

PEP8 function naming has also been applied to the NumPy bridge.

```
array = itk.array_from_image(image)
````

```
array = itk.array_from_image(image)
````

Similar functions names are available when creating NumPy array views, `itk.Image`'s from NumPy arrays, and converting VNL vector and matrices to NumPy arrays.

To test the 5.0 Beta 1 Python packages, run

```sh
python -m pip install --upgrade pip
python -m pip install --upgrade --pre itk
```

Performance Improvements
---------------------------------------

Based on experiences since initial changes introduced in [ITK 5 Alpha 2](https://discourse.itk.org/t/itk-5-0-alpha-1-modern-c/843), many notable performance improvements have been made.

- The default thread pool is now fully backed by C++11 threading primitives.
- A new `itk::MultiThreaderBase::ParallelizeArray` function is available to quickly and easily dispatch [C++11 lamda's](http://en.cppreference.com/w/cpp/language/lambda) or [std::function's](http://en.cppreference.com/w/cpp/utility/functional/function) on linearly indexed containers, similar to `itk::MultiThreaderBase::ParallelizeImageRegion` for operating on N-dimensional image regions.
- A new `itk::MultiThreaderBase::ParallelizeImageRegionRestrictDirection` function is available to prevent splitting an image region over a given direction.
- `SetNumberOfThreads` has been renamed to `SetNumberOfWorkUnits` to more accurately reflect configuration of work unit granularity, especially in the context of the Threading Building Blocks (TBB) backend.
- Progress reporting with multiple parallel operations is enabled with a new `itk::ProgressTransformer` class.
- A large corpus of code has been transitioned ITK 5, and the process is now relatively straight forward and well documented. Please see the [ITK 5 Migration Guide](https://github.com/InsightSoftwareConsortium/ITK/blob/master/Documentation/ITK5MigrationGuide.md) for more information.

Style changes
--------------------

Classes in the C++11 standard library are now preferred over ITK's implementations. This includes atomic integers, mutex locks and related classes, and hash maps. For more information, please see [the migration guide](https://github.com/InsightSoftwareConsortium/ITK/blob/master/Documentation/ITK5MigrationGuide.md).

New Classes
--------------------------------

- A [set of new iterators and a band pass filter](https://github.com/InsightSoftwareConsortium/ITK/commit/e16d0bc4eec7b3ee7ab63c3469077d54390df5c6) are available in a new *ITKImageFrequency* module. These classes help when operating in the dual domain and with the output of FFT's. More details are available [in this Insight Journal article](http://www.insight-journal.org/browse/publication/986).
- New, experimental [ShapedImageNeighborhoodRange](https://itk.org/Doxygen/html/classitk_1_1Experimental_1_1ShapedImageNeighborhoodRange.html), [HyperrectangularImageNeighborhoodShape](https://itk.org/Doxygen/html/classitk_1_1Experimental_1_1HyperrectangularImageNeighborhoodShape.html) classes are available, providing modern, C++ range operations and high performance.
- A new [Simple Linear Iterative Clustering (SLIC) super-pixel segmentation filter](https://itk.org/Doxygen/html/classitk_1_1SLICImageFilter.html) is available. This implementation improves or the original algorithm by supporting arbitrary dimensions, multi-component images, and a parallized implementation. More details are available [in the Insight Journal article](http://hdl.handle.net/1926/3596).

![visiblesuperpixel](https://user-images.githubusercontent.com/25432/45373250-ed07f180-b5bc-11e8-89ca-b149e6fadbeb.png)
*Visible Human superpixel segmentation. A 2D slice of 3D superpixel. For more information, see the Insight Journal article, Scalable Simple Linear Iterative Clustering (SSLIC) Using a Generic and Parallel Approach, Lowekamp, B et al, http://hdl.handle.net/1926/3596*.

What's Next
--------------------

There are many more bug fixes and critical improvements not mentioned above. For more details, please see the change log below. Congratulations to everyone who contributed to this release.

Please discuss your experiences on [Discourse](https://discourse.itk.org). The community has been quite active in testing and improving these changes for the next major release of ITK. Another beta release is planned for the October-November time period.

```
Changes from v5.0a02 to v5.0b01
-----------------------------------------------

Bradley Lowekamp (47):
      BUG: Prevent concurrent read/write in output image
      BUG: Handle case where output image is zero sized
      BUG: Add missing dependencies in Module testing
      ENH: Remove  SPECIFIC_IMAGEIO_MODULE_TEST feature
      BUG: Fix name typo for ITK_GLOBAL_DEFAULT_THREADER
      BUG: Fix incorrect library variable for MeshIO test drivers
      ENH: Create TestKernel library
      COMP: Fix overflow in floating-point conversion warning
      ENH: Add SLIC segmentation to new Superpixel module
      COMP: Fix C++11 compatibility with assert in constexpr
      COMP: Use std::min over vnl_math_min
      BUG: Fix index type and and number of indexes
      BUG: Add alternate baseline itkSLICImageFilterTest1 test
      ENH: Update test output to include test name
      BUG: improve numeric consistency in linear algorithm
      COMP: Add missing overrride to virtual destructor
      ENH: TileImageFilter learns to use VectorImages
      COMP: Remove unused testing function and headers
      BUG: Finish updating to ScanlineIterator in nonlinear method
      ENH: Update SLIC filter to use new ITK threading model
      ENH: TileImageFilter learns to use VectorImages
      ENH: Update SimpleITKFilters remote modules
      COMP: remove duplicate declaration of TempImageType
      DOC: Add link to published Insight Journal paper
      ENH: Update CenteredTransformInitializer print self method
      PERF: Remove internal streaming in DiscreteGaussianImageFilter
      ENH: Update MinimumMaximumImageFilter to use dynamic threading
      ENH: Update StatisticsImageFiler to use DynamicThreadedGenerateData
      ENH: Add compensated summation to the StatisticsImageFitler
      BUG: Use pip to install cmake
      BUG: Use pip to install cmake for release
      PERF: Use the recursive Gaussian filter in registration method
      BUG: Add additional baseline for debug
      COMP: Length GTest discovery timeout
      BUG: Correct image used for debug SimpleImageRegistrationTest
      BUG: Add cxx suffix to specified source GTest code
      ENH: Consistently use default OutputWindow for messages
      BUG: Explicitly set sampled pointset in virtual domain
      ENH: Use std::enable_if and std::is_same in  filters
      PERF: Memory alloc reduction in ComputeJacobianWithRespect
      PERF: Improve MatrixOffset Jacobian computation performance
      COMP: Address signed to unsigned comparison warning
      STYLE: Cleanup development comments
      PERF: perform matrix multiplication in-place for composite Jacobian
      ENH: Provide explicitly instantiated vnl_svd_fixed
      PERF: Change type of Jacobian w.r. position to vnl_matrix_fixed
      PERF: Change the purpose of the cache jacobian argument

Brian Avants (1):
      PERF: Only throw exception if zero valid points

Chao Wu (2):
      ENH: Enable writing scl_slope and scl_inter in NIfTI header
      BUG: Missing SetCPUBufferPointer in GPUImage<...>::SetPixelContainer(...)

Dženan Zukić (45):
      STYLE: a more direct link (after PR has been merged)
      STYLE: update clang-format configuration
      ENH: adding ParallelizeArray method to MultiThreaderBase
      COMP: fixing warning in external module
      COMP: Address usage of deleted assignment operator
      ENH: more thorough removal of deprecated "multiple method"
      DOC: add compatibility warning to Barrier
      STYLE: initializing SingleMethod/SingleData in base class
      DOC: merging instructions from itkMultiThreader.h into migration guide
      ENH: rename NumberOfThreads into NumberOfWorkUnits in filters
      ENH: Add backwards compatibility for Get/SetNumberOfThreads
      ENH: renaming NumberOfThreads into NumberOfWorkUnits in MultiThreaderBase
      ENH: adding backwards compatibility for MultiThreaders
      ENH: default number of work units is greater than default number of threads
      COMP: fixing compile warning on GCC 4.8
      ENH: rewriting thread pool to take advantage of C++11
      ENH: implement Parallelize Array and ImageRegion in PoolMultiThreader
      COMP: SmoothingRecursiveYvvGaussianFilter compiles with new ITKv5 threading
      ENH: RGB/RGBA consistency, operator / and remove comments from implementation
      ENH: refactoring RecursiveSeparableImageFilter to use new threading
      ENH: updating MCI: adjusting to split between threads and work units in ITKv5
      ENH: compute spacing using n-1 instead 2-1 in ImageSeriesReader
      BUG: updating the HDF5 symbol mangling
      ENH: fix reading of oblique image series
      BUG: fixing crash in ThreadPool's destructor
      COMP: fixing warning about shadowed global
      COMP: avoid warning with CMake 3.12 and newer regarding policy CMP0075
      ENH: write format which is backwards-compatible with HDF5 version 1.8
      ENH: refactoring BinaryImageToLabelMapFilter to use the new threading model
      ENH: adding ProgressTransformer
      ENH: proper backwards compatibility to HDF5 1.8
      STYLE: fixing 'No new line at the end of file'
      ENH: adding short description of split between threads and work units
      ENH: rewriting BarrierTest to work with any MultiThreader
      ENH: cause compile error with HDF5 version 1.10.0 and 1.10.1
      STYLE: fixing 'No new line at the end of file'
      ENH: restore BeforeThreadedGenerateData method
      DOC: state that SetRecursive must be called before SetInputDirectory
      BUG: updating the symbol mangling list
      BUG: removing mangling of unversioned variant of versioned definitions
      ENH: update migration guide: prefer C++11 classes over ITK's
      ENH: cleaning up ProcessObject
      ENH: using progress transformer in recently refactored classes
      COMP: fixing LoggerBase to compile even when DEBUG is defined
      DOC: ProgressReporter should be replaced by ProgressTransformer

Francois Budin (3):
      BUG: Improvements to ITK NumPyBridge
      DOC: Improve exception message to precise what image sizes can be processed
      BUG: Improvements to ITK NumPyBridge

GCC-XML Upstream (1):
      ENH: pygccxml develop (reduced)

GDCM Upstream (1):
      GDCM 2018-05-23 (69cb71a4)

HDF5 Maintainers (3):
      HDF5 2016-05-10 (7453bbef)
      HDF5 2018-07-20 (bed5b207)
      HDF5 2018-08-22 (08d77e43)

Hans Johnson (6):
      ENH: HDF5 moved from svn to git
      ENH: Update HDF5 import script for 1.10.2
      ENH: Remove svn version of hdf5 files not added by import from git
      STYLE: Consistently name for ITK conventions
      BUG: Fixing the HDF5 internal path names tests
      COMP: Fix const assignment error in testing

Isaiah Norton (3):
      BUG: prevent segfault when transform reader fails to load .mat
      PERF: improve DCMTKFileReader::CanReadFile perf
      BUG: prevent segfault when transform reader fails to load .mat

Jean-Christophe Fillion-Robin (1):
      BUG: Refactor itkFloatingPointExceptions to fix macOS support

Jon Haitz Legarreta (1):
      ENH: User initialization lists over direct assignment.

Jon Haitz Legarreta Gorroño (4):
      ENH: Add the ITK BoneMorphometry module as a remote.
      COMP: Bump remote modules versions.
      ENH: Clean up itkGrayscaleMorphologyImageFilter tests.
      ENH: Increase itk::OtsuMultipleThresholdsImageFilter coverage.

KWSys Upstream (1):
      KWSys 2018-06-01 (8ef62b28)

Kwame Kutten (2):
      BUG: Removed quotes from fftwf and fftwd hashes
      ENH: Make StatisticsImageFilter returns Sum Of Squares

Marian Klymov (1):
      ENH: Mark copy constructor of ExceptionObject and derived classes noexcept.

Matthew McCormick (39):
      COMP: Reduce example line lengths for inclusion in the SG
      COMP: Remove duplicate InPlaceImageFilter wrapping for RGB -> UC
      BUG: Do not copy workbox files when BUILD_DOCUMENTATION is OFF
      COMP: Remove duplicate BinaryGeneratorImageFilter class wrapping
      BUG: Migrate midas3.kitware.com to data.kitware.com
      ENH: Remove itkzlib in preparation for subtree update
      ENH: Add ZLIB UpdateFromUpstream.sh
      BUG: Do not use P-threads with Emscripten
      COMP: Remove extra argument from HoughTransform3DCircles wrapping
      ENH: Update testing data content links
      BUG: Correct Windows and macOS CastXML hashes
      BUG: Fix SLICImageFilter wrapping
      BUG: Improve backwards compatibility of BoxUtilities functions
      BUG: Remove commented, unused Python wrapping configuration
      BUG: Remove deprecated GetPointer wrapping method
      ENH: Python snake case functions
      BUG: Remove deprecated itk.write function
      BUG: Remove unmaintained itk.show, itk.show2D
      ENH: Add snake case function aliases for NumPy conversions
      DOC: ITK 5 now requires CMake 3.10.2
      BUG: Bump SplitComponents remote module to 2018-07-18
      BUG: Remove TBBImageToImageFilter remote module
      BUG: Correct Windows and macOS CastXML hashes
      COMP: Set CMake project command VERSION
      COMP: Do not wrap itk::ThreadJob
      COMP: Set CMake project command VERSION
      ENH: Add ParallelizeImageRegionRestrictDirection
      COMP: Bump CastXML for Visual Studio 2017 support
      ENH: Bump pygccxml hash to 2018-08-07 develop branch
      ENH: Update pygccxml git subtree version
      BUG: Remove ArchiveTestingDataOnMidas.py
      BUG: Use manylinux to build CastXML linux executable
      BUG: Avoid symlink issues in UpdateThirdPartyFromUpstream.sh
      DOC: Additional information and updates on bug fix release process
      COMP: Prevent duplicate wrapping ouput file specification
      COMP: Support instantiation of a 1D TranslationTransform
      DOC: Update how to reference a GitHub issue in the commit-msg hook
      DOC: ITK 5 Migration Guide grammar tweaks
      ENH: PEP8 for itk.Extras arguments

MetaIO Maintainers (1):
      MetaIO 2018-07-17 (6328f544)

Niels Dekker (14):
      DOC: Fixed \code sections ShapedImageNeighborhoodRange and added \author
      STYLE: Renamed RelativeIndices to ShapeOffsets in ShapedImageNeighborhoodRange
      ENH: Added ConnectedImageNeighborhoodShape, for N-connected neighborhoods
      ENH: Added operator[] to ShapedImageNeighborhoodRange
      ENH: Documented + tested order offsets ConnectedImageNeighborhoodShape
      COMP: Fixed conversion to PDFValueType in MattesMutualInfo...Threader
      ENH: Added reverse iterators to ShapedImageNeighborhoodRange
      STYLE: Removed HoughTransform2DCircles default for TRadiusPixelType
      ENH: Changed type float data members HoughTransform filters to double
      ENH: Added Functor::CoLexicographicCompare
      ENH: Made Functor::LexicographicCompare more generic and easier to use
      BUG: Added default-constructor iterator ShapedImageNeighborhoodRange
      ENH: Added ImageNeighborhoodPixelAccessPolicy for custom border extrapolation
      STYLE: Removed LinearInterpolateImageFunction::m_Neighbors

Pablo Hernandez-Cerdan (11):
      COMP: Remove ignored const qualifier in template.
      COMP: Fix warnings -Wcatch-value
      ENH: Update isotropic wavelets module.
      ENH: Add frequency iterators and band pass filter.
      COMP: Update IsotropicWavelets, remove FrequencyIterators
      STYLE: Remove comment in ITKImageFrequency cmake
      BUG: Fixes generating floating point exceptions
      COMP: Fix warnings in cdash introduced after FPE.
      DOC: Add example using std::atomic for multithreading.
      STYLE: Mark CMAKE_DEBUG_POSTFIX as advanced.
      COMP: Avoid extra point copy in MetaTube

Sean McBride (2):
      COMP: Fixed warning about missing case label in public header
      COMP: Fixed three misc clang warnings

Simon Rit (1):
      ENH: allow wrapping of XML writers

VXL Maintainers (3):
      VNL 2018-07-26 (2a77e40b)
      VNL 2018-08-04 (3dafea87)
      VNL 2018-09-05 (fc0c9043)

Vladimir S. FONOV (1):
      MINC 2018-08-09 (ce4333ca)

Zlib Upstream (1):
      zlib 2018-06-11 (355d8648)
```

Enjoy ITK!
