/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   _lcmaps_credential.h
    \brief  This header contains the methods to access a lcmaps credential
    \author Martijn Steenbakkers for EGEE

    This header contains the methods to access a lcmaps credential. The lcmaps
    credential is a generic container for all kinds of credentials:
    DN, GSI (gss_cred_id_t), security context (gss_ctx_id_t), or parsed voms attributes (FQANs)
    -# lcmaps_credential_store_dn():
    -# lcmaps_credential_store_gss_cred_id_t():
    -# lcmaps_credential_store_gss_ctx_id_t():
    -# lcmaps_credential_store_pem_string():
    -# lcmaps_credential_store_x509():
    -# lcmaps_credential_store_fqan_list():
    -# lcmaps_credential_get_dn():
    -# lcmaps_credential_get_gss_cred_id_t():
    -# lcmaps_credential_get_gss_ctx_id_t():
    -# lcmaps_credential_get_pem_string():
    -# lcmaps_credential_get_x509():
    -# lcmaps_credential_get_fqan_list():
    -# lcmaps_credential_release():
*/

#ifndef _LCMAPS_CREDENTIAL_H
#define _LCMAPS_CREDENTIAL_H

/******************************************************************************
                             Include header files
******************************************************************************/
#ifdef LCMAPS_GSI_MODE
#   include <gssapi.h>
#   include <openssl/x509.h>
#endif
#include "lcmaps_account.h"
#include "lcmaps_types.h"
#include <grp.h>
#include <sys/types.h>

/******************************************************************************
                             Define constants
******************************************************************************/
#define     LCMAPS_CRED_SUCCESS           0x0
#define     LCMAPS_CRED_NO_GSS_CRED       0x1
#define     LCMAPS_CRED_NO_GSS_CTX        0x2
#define     LCMAPS_CRED_NO_PEM_STRING     0x4
#define     LCMAPS_CRED_NO_X509_CRED      0x8
#define     LCMAPS_CRED_NO_X509_CHAIN     0x16
#define     LCMAPS_CRED_NO_DN             0x32
#define     LCMAPS_CRED_NO_FQAN           0x64
#define     LCMAPS_CRED_NO_NFQAN          0x128
#define     LCMAPS_CRED_NO_MAPCOUNTER     0x256
#define     LCMAPS_CRED_INVOCATION_ERROR  0x512
#define     LCMAPS_CRED_ERROR             0x1024


/******************************************************************************
                               Type definitions
******************************************************************************/

/*!
    \struct lcmaps_cred_id_s
    \typedef lcmaps_cred_id_t
    \brief structure type representing an LCMAPS credential.
*/

typedef struct lcmaps_cred_id_s
{
#ifdef LCMAPS_GSI_MODE
    gss_cred_id_t        cred;              /*!< the original gss (globus) credential */
    gss_ctx_id_t         context;           /*!< the gss security context */
    /* It's in the planning to move this to the gsi-free */
    /* area, once we use the globus-free voms parser (openssl based) */
    char *               pem_string;       /*!< the proxy certificate as PEM-encoded string */
    X509 *               px509_cred;       /*!< the proxy certificate as X509 struct */
    STACK_OF(X509) *     px509_chain;      /*!< the chain of the proxy certificate as STACK_OF(X509) */
#endif
    lcmaps_vomsdata_t *  voms_data_list;   /*!< The VOMS Data structures which contain all known information per VOMS BLOB from a proxy, added for GUMS and the Job Repository */
    int                  nvoms_data;       /*!< The number of VOMS Data Structures, equal to the number of VOMS BLOBs in the proxy, added for GUMS and the Job Repository */
    char *               dn;               /*!< the user distinguished name (DN) */
    char **              fqan;             /*!< the list of FQANs */
    int                  nfqan;            /*!< the number of FQANs */
    int                  mapcounter;       /*!< the counter to be used for multiple account mappings */
    lcmaps_account_info_t
        requested_account;              /*!< the account requested by the calling application,
                                             that should be verified */
}
lcmaps_cred_id_t;


/******************************************************************************
 *                 Module definition
 *****************************************************************************/

/*! \name CREDENTIAL FUNCTIONS */
/*@{*/ /* Beginning of credential functions */

#ifdef LCMAPS_GSI_MODE


/* TODO function textual headers */
int lcmaps_credential_store_gss_cred_id_t_and_sub_elements(
        gss_cred_id_t gss_cred_id,
        lcmaps_cred_id_t * plcmaps_credential
);
int lcmaps_credential_store_pem_string_and_sub_elements(
        char * pem_string,
        lcmaps_cred_id_t * plcmaps_credential
);
int lcmaps_credential_store_x509_and_sub_elements(
        X509 * cert,
        STACK_OF(X509) * certchain,
        lcmaps_cred_id_t * plcmaps_credential
);


/******************************************************************************
Function:       lcmaps_credential_store_gss_cred_id_t()
Description:    Fill credential with gss_cred_id_t (internally gsi credential)
                If the lcmaps credential does not yet contain the DN,
                try to retrieve it from the gss credential.
                If the lcmaps credential does not yet contain VOMS
                attributes, try to retrieve them from the gss credential.
Parameters:
                gss_credential:     gss credential (internally gsi credential)
                plcmaps_credential: pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS: succes
                LCMAPS_CRED_NO_GSS_CRED:  The supplied credential is empty
                LCMAPS_CRED_ERROR:   error in retrieving VOMS attributes.
                LCMAPS_CRED_INVOCATION_ERROR: failure, because lcmaps_credential
                                     does not exist
******************************************************************************/
int lcmaps_credential_store_gss_cred_id_t(
        gss_cred_id_t      gss_credential,
        lcmaps_cred_id_t * plcmaps_credential
);

/******************************************************************************
Function:       lcmaps_credential_get_gss_cred_id_t()
Description:    returns gss credential (internally gsi credential)
Parameters:
                lcmaps_credential: lcmaps_credential
Returns:        gss credential
******************************************************************************/
gss_cred_id_t lcmaps_credential_get_gss_cred_id_t(
        lcmaps_cred_id_t lcmaps_credential
);
/******************************************************************************
Function:       lcmaps_credential_store_gss_ctx_id_t()
Description:    Fill credential with gss_ctx_id_t (gss security context)
Parameters:
                gss_context:        gss security context
                plcmaps_credential: pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS: succes
                LCMAPS_CRED_ERROR: failure
******************************************************************************/
int lcmaps_credential_store_gss_ctx_id_t(
        gss_ctx_id_t       gss_context,
        lcmaps_cred_id_t * plcmaps_credential
);

/******************************************************************************
Function:       lcmaps_credential_get_gss_ctx_id_t()
Description:    returns gss security context
Parameters:
                lcmaps_credential: lcmaps_credential
Returns:        gss security context
******************************************************************************/
gss_ctx_id_t lcmaps_credential_get_gss_ctx_id_t(
        lcmaps_cred_id_t lcmaps_credential
);

/******************************************************************************
Function:       lcmaps_credential_store_pem_string()
Description:    Fill credential with PEM-encoded string
                If the lcmaps credential does not yet contain the X509, try
                to retrieve it from the string
Parameters:
                pem_string:         PEM-encoded string
                plcmaps_credential: pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS:  success
                LCMAPS_CRED_NO_PEM_STRING: PEM-encoded string is empty
                LCMAPS_CRED_INVOCATION_ERROR: failure, because lcmaps_credential does not exist
******************************************************************************/
int lcmaps_credential_store_pem_string(
        char *             pem_string,
        lcmaps_cred_id_t * plcmaps_credential
);

/******************************************************************************
Function:       lcmaps_credential_get_pem_string()
Description:    returns PEM-encoded string
Parameters:
                lcmaps_credential: lcmaps_credential
Returns:        PEM-encoded string
******************************************************************************/
char * lcmaps_credential_get_pem_string(
        lcmaps_cred_id_t lcmaps_credential
);

/******************************************************************************
Function:       lcmaps_credential_store_stack_of_x509()
Description:    Fill credential with X.509 certificate information
Parameters:
                px509_chain:        x509 proxy chain
                plcmaps_credential: pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS: succes
                LCMAPS_CRED_NO_X509_CHAIN: The supplied credential chain is empty
                LCMAPS_CRED_INVOCATION_ERROR: failure, because lcmaps_credential
                                     does not exist
******************************************************************************/
int lcmaps_credential_store_stack_of_x509(
        STACK_OF(X509)   * px509_chain,
        lcmaps_cred_id_t * plcmaps_credential
);

/******************************************************************************
Function:       lcmaps_credential_store_x509()
Description:    Fill credential with PEM-encoded string
                If the lcmaps credential does not yet contain the X509, try
                to retrieve it from the string
Parameters:
                px509:              x509 proxy certificate
                plcmaps_credential: pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS: succes
                LCMAPS_CRED_NO_X509_CRED:  The supplied credential is empty
                LCMAPS_CRED_INVOCATION_ERROR: failure, because lcmaps_credential
                                     does not exist
******************************************************************************/
int lcmaps_credential_store_x509(
        X509 *             px509,
        lcmaps_cred_id_t * plcmaps_credential
);

/******************************************************************************
Function:       lcmaps_credential_get_x509_cred()
Description:    Return the X509 cert.
Parameters:
                lcmaps_credential: lcmaps_credential
Returns:        the X509 cert.
******************************************************************************/
X509 * lcmaps_credential_get_x509_cred(
        lcmaps_cred_id_t lcmaps_credential
);

/******************************************************************************
Function:       lcmaps_credential_get_x509_chain()
Description:    Return the X509 cert chain
Parameters:
                lcmaps_credential: lcmaps_credential
Returns:        the X509 cert chain
******************************************************************************/
STACK_OF(X509) * lcmaps_credential_get_x509_chain(
        lcmaps_cred_id_t lcmaps_credential
);

#endif /* LCMAPS_GSI_MODE */


/******************************************************************************
Function:       lcmaps_credential_store_dn()
Description:    Fill credential with dn
Parameters:
                dn:                 distinguished name
                plcmaps_credential: pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS: success
                LCMAPS_CRED_NO_DN:   no DN to fill
                LCMAPS_CRED_INVOCATION_ERROR: failure, because lcmaps_credential does not exist
******************************************************************************/
int lcmaps_credential_store_dn(
        char *             dn,
        lcmaps_cred_id_t * plcmaps_credential
);

/******************************************************************************
Function:       lcmaps_credential_get_dn()
Description:    returns user dn
Parameters:
                lcmaps_credential: lcmaps_credential
Returns:        user dn
******************************************************************************/
char * lcmaps_credential_get_dn(
        lcmaps_cred_id_t lcmaps_credential
);


/******************************************************************************
Function:       lcmaps_credential_store_lcmaps_vomsdata()
Description:    Fill credential with LCMAPS vomsdata list
Parameters:
                lcmaps_credential: lcmaps_credential
                lcmaps_vomsdata_t: LCMAPS vomsdata
Returns:        LCMAPS_CRED_SUCCESS: success
                LCMAPS_CRED_NO_FQAN: supplied fqan list is empty
                LCMAPS_CRED_ERROR: failure, because of a malloc error or a malformed fqan list
                LCMAPS_CRED_INVOCATION_ERROR: failure, because lcmaps_credential does not exist
******************************************************************************/
int lcmaps_credential_store_lcmaps_vomsdata (
        lcmaps_vomsdata_t * lcmaps_vomsdata,
        lcmaps_cred_id_t * plcmaps_credential
);

/******************************************************************************
Function:       lcmaps_credential_store_fqan_list()
Description:    Fill credential with list of FQANs
Parameters:
                nfqan:              number of elements of the FQAN list
                fqan_list:          FQAN list (list of strings)
                plcmaps_credential: pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS: success
                LCMAPS_CRED_NO_FQAN: supplied fqan list is empty
                LCMAPS_CRED_ERROR: failure, because of a malloc error or a malformed fqan list
                LCMAPS_CRED_INVOCATION_ERROR: failure, because lcmaps_credential does not exist
******************************************************************************/
int lcmaps_credential_store_fqan_list(
        int                nfqan,
        char **            fqan_list,
        lcmaps_cred_id_t * plcmaps_credential
);

/******************************************************************************
Function:       lcmaps_credential_get_fqan_list()
Description:    returns the list of FQANs and number of elements
Parameters:
                pnfqan:            pointer to the number of FQANs in the list
                lcmaps_credential: lcmaps_credential
Returns:        list of FQANs
******************************************************************************/
char ** lcmaps_credential_get_fqan_list(
        int *            pnfqan,
        lcmaps_cred_id_t lcmaps_credential
);

/******************************************************************************
Function:       lcmaps_credential_store_mapcounter()
Description:    stores mapcounter
Parameters:
                mapcounter:         the mapping counter
                plcmaps_credential: pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS: success
                LCMAPS_CRED_INVOCATION_ERROR: failure, because lcmaps_credential does not exist
******************************************************************************/
int lcmaps_credential_store_mapcounter(
        int mapcounter,
        lcmaps_cred_id_t * plcmaps_credential
);

/******************************************************************************
Function:       lcmaps_credential_get_mapcounter()
Description:    returns mapcounter
Parameters:
                lcmaps_credential: lcmaps_credential
Returns:        mapcounter
******************************************************************************/
int lcmaps_credential_get_mapcounter(
        lcmaps_cred_id_t lcmaps_credential
);

/******************************************************************************
Function:       lcmaps_credential_store_requested_account()
Description:    stores the requested account (the account that should be verified)
                in an lcmaps_account_info_t structure.
Parameters:
                puid:                pointer to the uid of the local account
                ppgid_list:          pointer to the list of primary gids
                pnpgid:              pointer to the number of primary gids found
                psgid_list:          pointer to the list of secondary gids
                pnsgid:              pointer to the number of secondary gids found
                ppoolindex:          pointer to the pool index
                plcmaps_credential:  pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS: success
                LCMAPS_CRED_ERROR:   failure, because error filling lcmaps_account_info_t
                LCMAPS_CRED_INVOCATION_ERROR: failure, because lcmaps_credential does not exist
******************************************************************************/
int lcmaps_credential_store_requested_account(
        uid_t *                 puid,
        gid_t **                ppgid_list,
        int *                   pnpgid,
        gid_t **                psgid_list,
        int *                   pnsgid,
        char **                 ppoolindex,
        lcmaps_cred_id_t *      plcmaps_credential
);

/******************************************************************************
Function:       lcmaps_credential_get_requested_account()
Description:    returns requested account in lcmaps_account_info_t format
Parameters:
                lcmaps_credential: lcmaps_credential
Returns:        requested account
******************************************************************************/
lcmaps_account_info_t lcmaps_credential_get_requested_account(
        lcmaps_cred_id_t lcmaps_credential
);

/******************************************************************************
Function:       lcmaps_clean_list_of_strings()
Description:    clean list of strings
Parameters:
                listlen: number of elements in the list of strings
                string_list: list of strings
Returns:        LCMAPS_CRED_SUCCESS
******************************************************************************/
int lcmaps_clean_list_of_strings(
    int listlen, char ** string_list
);

/******************************************************************************
Function:       lcmaps_credential_init()
Description:    initialize the lcmaps_cred_id_t: set the member values to
                the default values
Parameters:
                plcmaps_credential: pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS:  success
                LCMAPS_CRED_ERROR:  failed to set the member values
                LCMAPS_CRED_INVOCATION_ERROR: failure, because lcmaps_credential does not exist
******************************************************************************/
int lcmaps_credential_init(
        lcmaps_cred_id_t * plcmaps_credential
);

/******************************************************************************
Function:       lcmaps_release_cred()
Description:    release lcmaps credential
Parameters:
                plcmaps_credential: pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS: success
                LCMAPS_CRED_ERROR: failure
******************************************************************************/
int lcmaps_release_cred(
        lcmaps_cred_id_t * plcmaps_credential
);

/******************************************************************************
Function:       lcmaps_release_cred_leave_STACK_OF_X509()
Description:    release lcmaps credential, but leave the STACK_OF_X509
Parameters:
                plcmaps_credential: pointer to lcmaps_credential
Returns:        LCMAPS_CRED_SUCCESS: success
                LCMAPS_CRED_ERROR: failure
******************************************************************************/
int lcmaps_release_cred_leave_STACK_OF_X509(
        lcmaps_cred_id_t * plcmaps_credential
);

/******************************************************************************
Function:       lcmaps_clean_vomsdata()
Description:    release vomsdata
Parameters:
                vomsdata: pointer to lcmaps_vomsdata_t
Returns:        LCMAPS_CRED_SUCCESS: success
                LCMAPS_CRED_ERROR: failure
******************************************************************************/
int lcmaps_clean_vomsdata(
        lcmaps_vomsdata_t * vomsdata
);

/*@}*/ /* End of credential functions */

#endif /* _LCMAPS_CREDENTIAL_H */

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps/src/grid_credential_handling/_lcmaps_credential.h,v $
    $Date: 2015-02-02 11:46:42 +0100 (Mon, 02 Feb 2015) $
    $Revision: 18216 $
    $Author: msalle $
******************************************************************************/
