/*
 * Copyright 2009-2019 The VOTCA Development Team (http://www.votca.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
#define BOOST_TEST_MAIN

#define BOOST_TEST_MODULE sphere_lebedev_rule_test
#include "votca/xtp/orbitals.h"
#include <boost/test/unit_test.hpp>
#include <fstream>
#include <votca/xtp/sphere_lebedev_rule.h>
using namespace votca::xtp;
using namespace votca;

BOOST_AUTO_TEST_SUITE(sphere_lebedev_rule_test)

Eigen::VectorXd ReadVectorFromString(const std::string& matrix) {
  votca::tools::Tokenizer lines(matrix, "\n");

  std::vector<double> entries;
  for (auto line : lines) {
    if (line[0] == '#') {
      continue;
    }
    votca::tools::Tokenizer entries_tok(line, ", ");
    std::vector<std::string> temp = entries_tok.ToVector();

    for (const auto& s : temp) {
      entries.push_back(std::stod(s));
    }
  }

  return Eigen::Map<Eigen::VectorXd>(entries.data(), entries.size());
}

BOOST_AUTO_TEST_CASE(medium_test) {

  std::ofstream xyzfile("molecule.xyz");
  xyzfile << " 5" << std::endl;
  xyzfile << " methane" << std::endl;
  xyzfile << " C            .000000     .000000     .000000" << std::endl;
  xyzfile << " H            .629118     .629118     .629118" << std::endl;
  xyzfile << " H           -.629118    -.629118     .629118" << std::endl;
  xyzfile << " H            .629118    -.629118    -.629118" << std::endl;
  xyzfile << " H           -.629118     .629118    -.629118" << std::endl;
  xyzfile.close();

  QMMolecule mol("noname", 0);
  mol.LoadFromFile("molecule.xyz");

  LebedevGrid spheregrid;

  auto grid = spheregrid.CalculateSphericalGrids(mol, "medium");

  auto Hgrid = grid.at("H");
  auto Cgrid = grid.at("C");

  std::string C_phi_ref_string =
      "90,90,90,90,0,180,45,45,135,135,45,45,135,135,90,"
      "90,90,90,54.7356,54.7356,54.7356,54.7356,125.264,125.264,125.264,"
      "125.264,77.7225,77.7225,77.7225,77.7225,102.278,102.278,102.278,"
      "102.278,46.2959,46.2959,46.2959,46.2959,133.704,133.704,133.704,"
      "133.704,46.2959,46.2959,46.2959,46.2959,133.704,133.704,133.704,"
      "133.704,14.5367,14.5367,14.5367,14.5367,165.463,165.463,165.463,"
      "165.463,79.7768,79.7768,79.7768,79.7768,100.223,100.223,100.223,"
      "100.223,79.7768,79.7768,79.7768,79.7768,100.223,100.223,100.223,"
      "100.223,44.0267,44.0267,44.0267,44.0267,135.973,135.973,135.973,"
      "135.973,60.5651,60.5651,60.5651,60.5651,119.435,119.435,119.435,"
      "119.435,60.5651,60.5651,60.5651,60.5651,119.435,119.435,119.435,"
      "119.435,65.9388,65.9388,65.9388,65.9388,114.061,114.061,114.061,"
      "114.061,49.7843,49.7843,49.7843,49.7843,130.216,130.216,130.216,"
      "130.216,49.7843,49.7843,49.7843,49.7843,130.216,130.216,130.216,"
      "130.216,23.869,23.869,23.869,23.869,156.131,156.131,156.131,"
      "156.131,73.3737,73.3737,73.3737,73.3737,106.626,106.626,106.626,"
      "106.626,73.3737,73.3737,73.3737,73.3737,106.626,106.626,106.626,"
      "106.626,6.14407,6.14407,6.14407,6.14407,173.856,173.856,173.856,"
      "173.856,85.6597,85.6597,85.6597,85.6597,94.3403,94.3403,94.3403,"
      "94.3403,85.6597,85.6597,85.6597,85.6597,94.3403,94.3403,94.3403,"
      "94.3403,33.7382,33.7382,33.7382,33.7382,146.262,146.262,146.262,"
      "146.262,66.8758,66.8758,66.8758,66.8758,113.124,113.124,113.124,"
      "113.124,66.8758,66.8758,66.8758,66.8758,113.124,113.124,113.124,"
      "113.124,90,90,90,90,90,90,90,90,61.8619,61.8619,118.138,"
      "118.138,28.1381,28.1381,151.862,151.862,61.8619,61.8619,118.138,"
      "118.138,28.1381,28.1381,151.862,151.862,90,90,90,90,90,90,90,"
      "90,77.8617,77.8617,102.138,102.138,12.1383,12.1383,167.862,"
      "167.862,77.8617,77.8617,102.138,102.138,12.1383,12.1383,167.862,"
      "167.862,63.2414,63.2414,63.2414,63.2414,116.759,116.759,116.759,"
      "116.759,29.6636,29.6636,29.6636,29.6636,150.336,150.336,150.336,"
      "150.336,63.2414,63.2414,63.2414,63.2414,116.759,116.759,116.759,"
      "116.759,78.1423,78.1423,78.1423,78.1423,101.858,101.858,101.858,"
      "101.858,29.6636,29.6636,29.6636,29.6636,150.336,150.336,150.336,"
      "150.336,78.1423,78.1423,78.1423,78.1423,101.858,101.858,101.858,"
      "101.858,83.869,83.869,83.869,83.869,96.131,96.131,96.131,96.131,"
      "36.8768,36.8768,36.8768,36.8768,143.123,143.123,143.123,143.123,"
      "83.869,83.869,83.869,83.869,96.131,96.131,96.131,96.131,53.8064,"
      "53.8064,53.8064,53.8064,126.194,126.194,126.194,126.194,36.8768,"
      "36.8768,36.8768,36.8768,143.123,143.123,143.123,143.123,53.8064,"
      "53.8064,53.8064,53.8064,126.194,126.194,126.194,126.194,71.915,"
      "71.915,71.915,71.915,108.085,108.085,108.085,108.085,39.489,"
      "39.489,39.489,39.489,140.511,140.511,140.511,140.511,71.915,"
      "71.915,71.915,71.915,108.085,108.085,108.085,108.085,56.2882,"
      "56.2882,56.2882,56.2882,123.712,123.712,123.712,123.712,39.489,"
      "39.489,39.489,39.489,140.511,140.511,140.511,140.511,56.2882,"
      "56.2882,56.2882,56.2882,123.712,123.712,123.712,123.712,84.3059,"
      "84.3059,84.3059,84.3059,95.6941,95.6941,95.6941,95.6941,70.4617,"
      "70.4617,70.4617,70.4617,109.538,109.538,109.538,109.538,84.3059,"
      "84.3059,84.3059,84.3059,95.6941,95.6941,95.6941,95.6941,20.4166,"
      "20.4166,20.4166,20.4166,159.583,159.583,159.583,159.583,70.4617,"
      "70.4617,70.4617,70.4617,109.538,109.538,109.538,109.538,20.4166,"
      "20.4166,20.4166,20.4166,159.583,159.583,159.583,159.583";
  Eigen::VectorXd C_phi_ref = ReadVectorFromString(C_phi_ref_string);
  C_phi_ref *= votca::tools::conv::Pi / 180.0;

  std::string C_theta_ref_string =
      "0,180,90,-90,0,0,90,-90,90,-90,0,180,0,180,45,"
      "135,-45,-135,45,135,-45,-135,45,135,-45,-135,45,135,-45,"
      "-135,45,135,-45,-135,17.1066,162.893,-17.1066,-162.893,17.1066,"
      "162.893,-17.1066,-162.893,72.8934,107.107,-72.8934,-107.107,"
      "72.8934,107.107,-72.8934,-107.107,45,135,-45,-135,45,135,-45,"
      "-135,79.61,100.39,-79.61,-100.39,79.61,100.39,-79.61,-100.39,"
      "10.39,169.61,-10.39,-169.61,10.39,169.61,-10.39,-169.61,45,135,"
      "-45,-135,45,135,-45,-135,55.6481,124.352,-55.6481,-124.352,"
      "55.6481,124.352,-55.6481,-124.352,34.3519,145.648,-34.3519,"
      "-145.648,34.3519,145.648,-34.3519,-145.648,45,135,-45,-135,45,"
      "135,-45,-135,32.2708,147.729,-32.2708,-147.729,32.2708,147.729,"
      "-32.2708,-147.729,57.7292,122.271,-57.7292,-122.271,57.7292,"
      "122.271,-57.7292,-122.271,45,135,-45,-135,45,135,-45,-135,"
      "72.6256,107.374,-72.6256,-107.374,72.6256,107.374,-72.6256,"
      "-107.374,17.3744,162.626,-17.3744,-162.626,17.3744,162.626,"
      "-17.3744,-162.626,45,135,-45,-135,45,135,-45,-135,85.6471,"
      "94.3529,-85.6471,-94.3529,85.6471,94.3529,-85.6471,-94.3529,"
      "4.35285,175.647,-4.35285,-175.647,4.35285,175.647,-4.35285,"
      "-175.647,45,135,-45,-135,45,135,-45,-135,64.7204,115.28,"
      "-64.7204,-115.28,64.7204,115.28,-64.7204,-115.28,25.2796,154.72,"
      "-25.2796,-154.72,25.2796,154.72,-25.2796,-154.72,28.1381,151.862,"
      "-28.1381,-151.862,61.8619,118.138,-61.8619,-118.138,0,180,0,180,"
      "0,180,0,180,90,-90,90,-90,90,-90,90,-90,12.1383,167.862,"
      "-12.1383,-167.862,77.8617,102.138,-77.8617,-102.138,0,180,0,180,"
      "0,180,0,180,90,-90,90,-90,90,-90,90,-90,76.6955,103.305,"
      "-76.6955,-103.305,76.6955,103.305,-76.6955,-103.305,65.4685,"
      "114.532,-65.4685,-114.532,65.4685,114.532,-65.4685,-114.532,"
      "13.3045,166.695,-13.3045,-166.695,13.3045,166.695,-13.3045,"
      "-166.695,27.3903,152.61,-27.3903,-152.61,27.3903,152.61,-27.3903,"
      "-152.61,24.5315,155.468,-24.5315,-155.468,24.5315,155.468,-24.5315,"
      "-155.468,62.6097,117.39,-62.6097,-117.39,62.6097,117.39,-62.6097,"
      "-117.39,53.5648,126.435,-53.5648,-126.435,53.5648,126.435,-53.5648,"
      "-126.435,10.2518,169.748,-10.2518,-169.748,10.2518,169.748,"
      "-10.2518,-169.748,36.4352,143.565,-36.4352,-143.565,36.4352,"
      "143.565,-36.4352,-143.565,7.60483,172.395,-7.60483,-172.395,"
      "7.60483,172.395,-7.60483,-172.395,79.7482,100.252,-79.7482,"
      "-100.252,79.7482,100.252,-79.7482,-100.252,82.3952,97.6048,"
      "-82.3952,-97.6048,82.3952,97.6048,-82.3952,-97.6048,54.2775,"
      "125.722,-54.2775,-125.722,54.2775,125.722,-54.2775,-125.722,"
      "29.2189,150.781,-29.2189,-150.781,29.2189,150.781,-29.2189,"
      "-150.781,35.7225,144.278,-35.7225,-144.278,35.7225,144.278,"
      "-35.7225,-144.278,21.912,158.088,-21.912,-158.088,21.912,158.088,"
      "-21.912,-158.088,60.7811,119.219,-60.7811,-119.219,60.7811,119.219,"
      "-60.7811,-119.219,68.088,111.912,-68.088,-111.912,68.088,111.912,"
      "-68.088,-111.912,19.6391,160.361,-19.6391,-160.361,19.6391,160.361,"
      "-19.6391,-160.361,6.04329,173.957,-6.04329,-173.957,6.04329,"
      "173.957,-6.04329,-173.957,70.3609,109.639,-70.3609,-109.639,"
      "70.3609,109.639,-70.3609,-109.639,16.5241,163.476,-16.5241,"
      "-163.476,16.5241,163.476,-16.5241,-163.476,83.9567,96.0433,"
      "-83.9567,-96.0433,83.9567,96.0433,-83.9567,-96.0433,73.4759,"
      "106.524,-73.4759,-106.524,73.4759,106.524,-73.4759,-106.524";
  Eigen::VectorXd C_theta_ref = ReadVectorFromString(C_theta_ref_string);
  C_theta_ref *= votca::tools::conv::Pi / 180.0;

  std::string C_weight_ref_string =
      "0.00661732,0.00661732,0.00661732,0.00661732,0.00661732,"
      "0.00661732,0.0320219,0.0320219,0.0320219,0.0320219,0.0320219,"
      "0.0320219,0.0320219,0.0320219,0.0320219,0.0320219,0.0320219,"
      "0.0320219,0.0315707,0.0315707,0.0315707,0.0315707,0.0315707,"
      "0.0315707,0.0315707,0.0315707,0.031798,0.031798,0.031798,0.031798,"
      "0.031798,0.031798,0.031798,0.031798,0.031798,0.031798,0.031798,"
      "0.031798,0.031798,0.031798,0.031798,0.031798,0.031798,0.031798,"
      "0.031798,0.031798,0.031798,0.031798,0.031798,0.031798,0.0253122,"
      "0.0253122,0.0253122,0.0253122,0.0253122,0.0253122,0.0253122,"
      "0.0253122,0.0253122,0.0253122,0.0253122,0.0253122,0.0253122,"
      "0.0253122,0.0253122,0.0253122,0.0253122,0.0253122,0.0253122,"
      "0.0253122,0.0253122,0.0253122,0.0253122,0.0253122,0.0314376,"
      "0.0314376,0.0314376,0.0314376,0.0314376,0.0314376,0.0314376,"
      "0.0314376,0.0314376,0.0314376,0.0314376,0.0314376,0.0314376,"
      "0.0314376,0.0314376,0.0314376,0.0314376,0.0314376,0.0314376,"
      "0.0314376,0.0314376,0.0314376,0.0314376,0.0314376,0.0315826,"
      "0.0315826,0.0315826,0.0315826,0.0315826,0.0315826,0.0315826,"
      "0.0315826,0.0315826,0.0315826,0.0315826,0.0315826,0.0315826,"
      "0.0315826,0.0315826,0.0315826,0.0315826,0.0315826,0.0315826,"
      "0.0315826,0.0315826,0.0315826,0.0315826,0.0315826,0.0289365,"
      "0.0289365,0.0289365,0.0289365,0.0289365,0.0289365,0.0289365,"
      "0.0289365,0.0289365,0.0289365,0.0289365,0.0289365,0.0289365,"
      "0.0289365,0.0289365,0.0289365,0.0289365,0.0289365,0.0289365,"
      "0.0289365,0.0289365,0.0289365,0.0289365,0.0289365,0.0183783,"
      "0.0183783,0.0183783,0.0183783,0.0183783,0.0183783,0.0183783,"
      "0.0183783,0.0183783,0.0183783,0.0183783,0.0183783,0.0183783,"
      "0.0183783,0.0183783,0.0183783,0.0183783,0.0183783,0.0183783,"
      "0.0183783,0.0183783,0.0183783,0.0183783,0.0183783,0.0307295,"
      "0.0307295,0.0307295,0.0307295,0.0307295,0.0307295,0.0307295,"
      "0.0307295,0.0307295,0.0307295,0.0307295,0.0307295,0.0307295,"
      "0.0307295,0.0307295,0.0307295,0.0307295,0.0307295,0.0307295,"
      "0.0307295,0.0307295,0.0307295,0.0307295,0.0307295,0.0303785,"
      "0.0303785,0.0303785,0.0303785,0.0303785,0.0303785,0.0303785,"
      "0.0303785,0.0303785,0.0303785,0.0303785,0.0303785,0.0303785,"
      "0.0303785,0.0303785,0.0303785,0.0303785,0.0303785,0.0303785,"
      "0.0303785,0.0303785,0.0303785,0.0303785,0.0303785,0.0240137,"
      "0.0240137,0.0240137,0.0240137,0.0240137,0.0240137,0.0240137,"
      "0.0240137,0.0240137,0.0240137,0.0240137,0.0240137,0.0240137,"
      "0.0240137,0.0240137,0.0240137,0.0240137,0.0240137,0.0240137,"
      "0.0240137,0.0240137,0.0240137,0.0240137,0.0240137,0.030372,0.030372,"
      "0.030372,0.030372,0.030372,0.030372,0.030372,0.030372,0.030372,"
      "0.030372,0.030372,0.030372,0.030372,0.030372,0.030372,0.030372,"
      "0.030372,0.030372,0.030372,0.030372,0.030372,0.030372,0.030372,"
      "0.030372,0.030372,0.030372,0.030372,0.030372,0.030372,0.030372,"
      "0.030372,0.030372,0.030372,0.030372,0.030372,0.030372,0.030372,"
      "0.030372,0.030372,0.030372,0.030372,0.030372,0.030372,0.030372,"
      "0.030372,0.030372,0.030372,0.030372,0.0315697,0.0315697,0.0315697,"
      "0.0315697,0.0315697,0.0315697,0.0315697,0.0315697,0.0315697,"
      "0.0315697,0.0315697,0.0315697,0.0315697,0.0315697,0.0315697,"
      "0.0315697,0.0315697,0.0315697,0.0315697,0.0315697,0.0315697,"
      "0.0315697,0.0315697,0.0315697,0.0315697,0.0315697,0.0315697,"
      "0.0315697,0.0315697,0.0315697,0.0315697,0.0315697,0.0315697,"
      "0.0315697,0.0315697,0.0315697,0.0315697,0.0315697,0.0315697,"
      "0.0315697,0.0315697,0.0315697,0.0315697,0.0315697,0.0315697,"
      "0.0315697,0.0315697,0.0315697,0.0313738,0.0313738,0.0313738,"
      "0.0313738,0.0313738,0.0313738,0.0313738,0.0313738,0.0313738,"
      "0.0313738,0.0313738,0.0313738,0.0313738,0.0313738,0.0313738,"
      "0.0313738,0.0313738,0.0313738,0.0313738,0.0313738,0.0313738,"
      "0.0313738,0.0313738,0.0313738,0.0313738,0.0313738,0.0313738,"
      "0.0313738,0.0313738,0.0313738,0.0313738,0.0313738,0.0313738,"
      "0.0313738,0.0313738,0.0313738,0.0313738,0.0313738,0.0313738,"
      "0.0313738,0.0313738,0.0313738,0.0313738,0.0313738,0.0313738,"
      "0.0313738,0.0313738,0.0313738,0.028106,0.028106,0.028106,0.028106,"
      "0.028106,0.028106,0.028106,0.028106,0.028106,0.028106,0.028106,"
      "0.028106,0.028106,0.028106,0.028106,0.028106,0.028106,0.028106,"
      "0.028106,0.028106,0.028106,0.028106,0.028106,0.028106,0.028106,"
      "0.028106,0.028106,0.028106,0.028106,0.028106,0.028106,0.028106,"
      "0.028106,0.028106,0.028106,0.028106,0.028106,0.028106,0.028106,"
      "0.028106,0.028106,0.028106,0.028106,0.028106,0.028106,0.028106,"
      "0.028106,0.028106";
  Eigen::VectorXd C_weight_ref = ReadVectorFromString(C_weight_ref_string);
  BOOST_CHECK_EQUAL(Cgrid.phi.size(), C_phi_ref.size());
  BOOST_CHECK_EQUAL(Cgrid.theta.size(), C_theta_ref.size());
  BOOST_CHECK_EQUAL(Cgrid.weight.size(), C_weight_ref.size());
  BOOST_CHECK_EQUAL(Hgrid.phi.size(), C_phi_ref.size());
  BOOST_CHECK_EQUAL(Hgrid.theta.size(), C_theta_ref.size());
  BOOST_CHECK_EQUAL(Hgrid.weight.size(), C_weight_ref.size());

  bool Cphi = C_phi_ref.isApprox(Cgrid.phi, 0.001);
  bool Ctheta = C_theta_ref.isApprox(Cgrid.theta, 0.001);
  if (!Cphi || !Ctheta) {
    std::cout << "phi_ref : Phi_comp | theta_ref : theta_comp" << std::endl;
    for (Index i = 0; i < C_phi_ref.size(); i++) {
      std::cout << Cgrid.phi[i] << ":" << C_phi_ref[i] << " | "
                << Cgrid.theta[i] << ":" << C_theta_ref[i] << std::endl;
    }
  }

  bool Cweight = C_weight_ref.isApprox(Cgrid.weight, 0.0001);
  BOOST_CHECK_EQUAL(Cphi, true);
  BOOST_CHECK_EQUAL(Ctheta, true);
  BOOST_CHECK_EQUAL(Cweight, true);

  bool Hphi = C_phi_ref.isApprox(Hgrid.phi, 0.001);
  bool Htheta = C_theta_ref.isApprox(Hgrid.theta, 0.001);
  bool Hweight = C_weight_ref.isApprox(Hgrid.weight, 0.0001);
  BOOST_CHECK_EQUAL(Hphi, true);
  BOOST_CHECK_EQUAL(Htheta, true);
  BOOST_CHECK_EQUAL(Hweight, true);
}

BOOST_AUTO_TEST_CASE(fine_test) {

  std::ofstream xyzfile("molecule.xyz");
  xyzfile << " 5" << std::endl;
  xyzfile << " methane" << std::endl;
  xyzfile << " Ge            .000000     .000000     .000000" << std::endl;
  xyzfile << " H            .629118     .629118     .629118" << std::endl;
  xyzfile << " H           -.629118    -.629118     .629118" << std::endl;
  xyzfile << " H            .629118    -.629118    -.629118" << std::endl;
  xyzfile << " H           -.629118     .629118    -.629118" << std::endl;
  xyzfile.close();

  QMMolecule mol("noname", 0);
  mol.LoadFromFile("molecule.xyz");

  LebedevGrid spheregrid;

  auto grid = spheregrid.CalculateSphericalGrids(mol, "fine");

  auto Hgrid = grid.at("H");
  auto Gegrid = grid.at("Ge");
  std::string Ge_phi_ref_string =
      "1.5708,1.5708,1.5708,1.5708,0,3.14159,0.955317,0.955317,"
      "0.955317,0.955317,2.18628,2.18628,2.18628,2.18628,0.060749,"
      "0.060749,0.060749,0.060749,3.08084,3.08084,3.08084,3.08084,1.52785,"
      "1.52785,1.52785,1.52785,1.61374,1.61374,1.61374,1.61374,1.52785,"
      "1.52785,1.52785,1.52785,1.61374,1.61374,1.61374,1.61374,0.149248,"
      "0.149248,0.149248,0.149248,2.99234,2.99234,2.99234,2.99234,1.46546,"
      "1.46546,1.46546,1.46546,1.67613,1.67613,1.67613,1.67613,1.46546,"
      "1.46546,1.46546,1.46546,1.67613,1.67613,1.67613,1.67613,0.25009,"
      "0.25009,0.25009,0.25009,2.8915,2.8915,2.8915,2.8915,1.39489,"
      "1.39489,1.39489,1.39489,1.7467,1.7467,1.7467,1.7467,1.39489,"
      "1.39489,1.39489,1.39489,1.7467,1.7467,1.7467,1.7467,0.357991,"
      "0.357991,0.357991,0.357991,2.7836,2.7836,2.7836,2.7836,1.32042,"
      "1.32042,1.32042,1.32042,1.82117,1.82117,1.82117,1.82117,1.32042,"
      "1.32042,1.32042,1.32042,1.82117,1.82117,1.82117,1.82117,0.470663,"
      "0.470663,0.470663,0.470663,2.67093,2.67093,2.67093,2.67093,1.24437,"
      "1.24437,1.24437,1.24437,1.89722,1.89722,1.89722,1.89722,1.24437,"
      "1.24437,1.24437,1.24437,1.89722,1.89722,1.89722,1.89722,0.587017,"
      "0.587017,0.587017,0.587017,2.55458,2.55458,2.55458,2.55458,1.16837,"
      "1.16837,1.16837,1.16837,1.97322,1.97322,1.97322,1.97322,1.16837,"
      "1.16837,1.16837,1.16837,1.97322,1.97322,1.97322,1.97322,0.706586,"
      "0.706586,0.706586,0.706586,2.43501,2.43501,2.43501,2.43501,1.09383,"
      "1.09383,1.09383,1.09383,2.04776,2.04776,2.04776,2.04776,1.09383,"
      "1.09383,1.09383,1.09383,2.04776,2.04776,2.04776,2.04776,0.829288,"
      "0.829288,0.829288,0.829288,2.3123,2.3123,2.3123,2.3123,1.02224,"
      "1.02224,1.02224,1.02224,2.11935,2.11935,2.11935,2.11935,1.02224,"
      "1.02224,1.02224,1.02224,2.11935,2.11935,2.11935,2.11935,1.08506,"
      "1.08506,1.08506,1.08506,2.05653,2.05653,2.05653,2.05653,0.895259,"
      "0.895259,0.895259,0.895259,2.24633,2.24633,2.24633,2.24633,"
      "0.895259,0.895259,0.895259,0.895259,2.24633,2.24633,2.24633,"
      "2.24633,1.21897,1.21897,1.21897,1.21897,1.92262,1.92262,1.92262,"
      "1.92262,0.844918,0.844918,0.844918,0.844918,2.29667,2.29667,"
      "2.29667,2.29667,0.844918,0.844918,0.844918,0.844918,2.29667,"
      "2.29667,2.29667,2.29667,1.35722,1.35722,1.35722,1.35722,1.78437,"
      "1.78437,1.78437,1.78437,0.807868,0.807868,0.807868,0.807868,"
      "2.33372,2.33372,2.33372,2.33372,0.807868,0.807868,0.807868,"
      "0.807868,2.33372,2.33372,2.33372,2.33372,1.49911,1.49911,1.49911,"
      "1.49911,1.64248,1.64248,1.64248,1.64248,0.787963,0.787963,0.787963,"
      "0.787963,2.35363,2.35363,2.35363,2.35363,0.787963,0.787963,"
      "0.787963,0.787963,2.35363,2.35363,2.35363,2.35363,1.5708,1.5708,"
      "1.5708,1.5708,1.5708,1.5708,1.5708,1.5708,0.123986,0.123986,"
      "3.01761,3.01761,1.44681,1.44681,1.69478,1.69478,0.123986,0.123986,"
      "3.01761,3.01761,1.44681,1.44681,1.69478,1.69478,1.5708,1.5708,"
      "1.5708,1.5708,1.5708,1.5708,1.5708,1.5708,0.29849,0.29849,2.8431,"
      "2.8431,1.27231,1.27231,1.86929,1.86929,0.29849,0.29849,2.8431,"
      "2.8431,1.27231,1.27231,1.86929,1.86929,1.5708,1.5708,1.5708,"
      "1.5708,1.5708,1.5708,1.5708,1.5708,0.489036,0.489036,2.65256,"
      "2.65256,1.08176,1.08176,2.05983,2.05983,0.489036,0.489036,2.65256,"
      "2.65256,1.08176,1.08176,2.05983,2.05983,1.5708,1.5708,1.5708,"
      "1.5708,1.5708,1.5708,1.5708,1.5708,0.686012,0.686012,2.45558,"
      "2.45558,0.884784,0.884784,2.25681,2.25681,0.686012,0.686012,"
      "2.45558,2.45558,0.884784,0.884784,2.25681,2.25681,0.213134,"
      "0.213134,0.213134,0.213134,2.92846,2.92846,2.92846,2.92846,1.36646,"
      "1.36646,1.36646,1.36646,1.77513,1.77513,1.77513,1.77513,0.213134,"
      "0.213134,0.213134,0.213134,2.92846,2.92846,2.92846,2.92846,1.51102,"
      "1.51102,1.51102,1.51102,1.63057,1.63057,1.63057,1.63057,1.36646,"
      "1.36646,1.36646,1.36646,1.77513,1.77513,1.77513,1.77513,1.51102,"
      "1.51102,1.51102,1.51102,1.63057,1.63057,1.63057,1.63057,0.501092,"
      "0.501092,0.501092,0.501092,2.6405,2.6405,2.6405,2.6405,1.09251,"
      "1.09251,1.09251,1.09251,2.04909,2.04909,2.04909,2.04909,0.501092,"
      "0.501092,0.501092,0.501092,2.6405,2.6405,2.6405,2.6405,1.43278,"
      "1.43278,1.43278,1.43278,1.70881,1.70881,1.70881,1.70881,1.09251,"
      "1.09251,1.09251,1.09251,2.04909,2.04909,2.04909,2.04909,1.43278,"
      "1.43278,1.43278,1.43278,1.70881,1.70881,1.70881,1.70881,0.651884,"
      "0.651884,0.651884,0.651884,2.48971,2.48971,2.48971,2.48971,1.04365,"
      "1.04365,1.04365,1.04365,2.09794,2.09794,2.09794,2.09794,0.651884,"
      "0.651884,0.651884,0.651884,2.48971,2.48971,2.48971,2.48971,1.22483,"
      "1.22483,1.22483,1.22483,1.91676,1.91676,1.91676,1.91676,1.04365,"
      "1.04365,1.04365,1.04365,2.09794,2.09794,2.09794,2.09794,1.22483,"
      "1.22483,1.22483,1.22483,1.91676,1.91676,1.91676,1.91676,0.314277,"
      "0.314277,0.314277,0.314277,2.82732,2.82732,2.82732,2.82732,1.28517,"
      "1.28517,1.28517,1.28517,1.85642,1.85642,1.85642,1.85642,0.314277,"
      "0.314277,0.314277,0.314277,2.82732,2.82732,2.82732,2.82732,1.44328,"
      "1.44328,1.44328,1.44328,1.69831,1.69831,1.69831,1.69831,1.28517,"
      "1.28517,1.28517,1.28517,1.85642,1.85642,1.85642,1.85642,1.44328,"
      "1.44328,1.44328,1.44328,1.69831,1.69831,1.69831,1.69831,0.535245,"
      "0.535245,0.535245,0.535245,2.60635,2.60635,2.60635,2.60635,1.1228,"
      "1.1228,1.1228,1.1228,2.01879,2.01879,2.01879,2.01879,0.535245,"
      "0.535245,0.535245,0.535245,2.60635,2.60635,2.60635,2.60635,1.29812,"
      "1.29812,1.29812,1.29812,1.84347,1.84347,1.84347,1.84347,1.1228,"
      "1.1228,1.1228,1.1228,2.01879,2.01879,2.01879,2.01879,1.29812,"
      "1.29812,1.29812,1.29812,1.84347,1.84347,1.84347,1.84347,0.696485,"
      "0.696485,0.696485,0.696485,2.44511,2.44511,2.44511,2.44511,"
      "0.894874,0.894874,0.894874,0.894874,2.24672,2.24672,2.24672,"
      "2.24672,0.696485,0.696485,0.696485,0.696485,2.44511,2.44511,"
      "2.44511,2.44511,1.42834,1.42834,1.42834,1.42834,1.71326,1.71326,"
      "1.71326,1.71326,0.894874,0.894874,0.894874,0.894874,2.24672,"
      "2.24672,2.24672,2.24672,1.42834,1.42834,1.42834,1.42834,1.71326,"
      "1.71326,1.71326,1.71326,0.395988,0.395988,0.395988,0.395988,"
      "2.74561,2.74561,2.74561,2.74561,1.18117,1.18117,1.18117,1.18117,"
      "1.96042,1.96042,1.96042,1.96042,0.395988,0.395988,0.395988,"
      "0.395988,2.74561,2.74561,2.74561,2.74561,1.50365,1.50365,1.50365,"
      "1.50365,1.63794,1.63794,1.63794,1.63794,1.18117,1.18117,1.18117,"
      "1.18117,1.96042,1.96042,1.96042,1.96042,1.50365,1.50365,1.50365,"
      "1.50365,1.63794,1.63794,1.63794,1.63794,0.589862,0.589862,0.589862,"
      "0.589862,2.55173,2.55173,2.55173,2.55173,0.986335,0.986335,"
      "0.986335,0.986335,2.15526,2.15526,2.15526,2.15526,0.589862,"
      "0.589862,0.589862,0.589862,2.55173,2.55173,2.55173,2.55173,1.50016,"
      "1.50016,1.50016,1.50016,1.64143,1.64143,1.64143,1.64143,0.986335,"
      "0.986335,0.986335,0.986335,2.15526,2.15526,2.15526,2.15526,1.50016,"
      "1.50016,1.50016,1.50016,1.64143,1.64143,1.64143,1.64143,0.726324,"
      "0.726324,0.726324,0.726324,2.41527,2.41527,2.41527,2.41527,"
      "0.923588,0.923588,0.923588,0.923588,2.218,2.218,2.218,2.218,"
      "0.726324,0.726324,0.726324,0.726324,2.41527,2.41527,2.41527,"
      "2.41527,1.28868,1.28868,1.28868,1.28868,1.85291,1.85291,1.85291,"
      "1.85291,0.923588,0.923588,0.923588,0.923588,2.218,2.218,2.218,"
      "2.218,1.28868,1.28868,1.28868,1.28868,1.85291,1.85291,1.85291,"
      "1.85291,0.422374,0.422374,0.422374,0.422374,2.71922,2.71922,"
      "2.71922,2.71922,1.20364,1.20364,1.20364,1.20364,1.93795,1.93795,"
      "1.93795,1.93795,0.422374,0.422374,0.422374,0.422374,2.71922,"
      "2.71922,2.71922,2.71922,1.37152,1.37152,1.37152,1.37152,1.77007,"
      "1.77007,1.77007,1.77007,1.20364,1.20364,1.20364,1.20364,1.93795,"
      "1.93795,1.93795,1.93795,1.37152,1.37152,1.37152,1.37152,1.77007,"
      "1.77007,1.77007,1.77007,0.611568,0.611568,0.611568,0.611568,"
      "2.53002,2.53002,2.53002,2.53002,1.00645,1.00645,1.00645,1.00645,"
      "2.13515,2.13515,2.13515,2.13515,0.611568,0.611568,0.611568,"
      "0.611568,2.53002,2.53002,2.53002,2.53002,1.36052,1.36052,1.36052,"
      "1.36052,1.78107,1.78107,1.78107,1.78107,1.00645,1.00645,1.00645,"
      "1.00645,2.13515,2.13515,2.13515,2.13515,1.36052,1.36052,1.36052,"
      "1.36052,1.78107,1.78107,1.78107,1.78107,0.771893,0.771893,0.771893,"
      "0.771893,2.3697,2.3697,2.3697,2.3697,0.96733,0.96733,0.96733,"
      "0.96733,2.17426,2.17426,2.17426,2.17426,0.771893,0.771893,0.771893,"
      "0.771893,2.3697,2.3697,2.3697,2.3697,1.15326,1.15326,1.15326,"
      "1.15326,1.98834,1.98834,1.98834,1.98834,0.96733,0.96733,0.96733,"
      "0.96733,2.17426,2.17426,2.17426,2.17426,1.15326,1.15326,1.15326,"
      "1.15326,1.98834,1.98834,1.98834,1.98834";
  Eigen::VectorXd Ge_phi_ref = ReadVectorFromString(Ge_phi_ref_string);

  std::string Ge_theta_ref_string =
      "0,3.14159,1.5708,-1.5708,0,0,0.785398,2.35619,"
      "-0.785398,-2.35619,0.785398,2.35619,-0.785398,-2.35619,0.785398,"
      "2.35619,-0.785398,-2.35619,0.785398,2.35619,-0.785398,-2.35619,"
      "1.52781,1.61378,-1.52781,-1.61378,1.52781,1.61378,-1.52781,"
      "-1.61378,0.0429825,3.09861,-0.0429825,-3.09861,0.0429825,3.09861,"
      "-0.0429825,-3.09861,0.785398,2.35619,-0.785398,-2.35619,0.785398,"
      "2.35619,-0.785398,-2.35619,1.46487,1.67672,-1.46487,-1.67672,"
      "1.46487,1.67672,-1.46487,-1.67672,0.105927,3.03567,-0.105927,"
      "-3.03567,0.105927,3.03567,-0.105927,-3.03567,0.785398,2.35619,"
      "-0.785398,-2.35619,0.785398,2.35619,-0.785398,-2.35619,1.3921,"
      "1.74949,-1.3921,-1.74949,1.3921,1.74949,-1.3921,-1.74949,0.178695,"
      "2.9629,-0.178695,-2.9629,0.178695,2.9629,-0.178695,-2.9629,"
      "0.785398,2.35619,-0.785398,-2.35619,0.785398,2.35619,-0.785398,"
      "-2.35619,1.31218,1.82941,-1.31218,-1.82941,1.31218,1.82941,"
      "-1.31218,-1.82941,0.258612,2.88298,-0.258612,-2.88298,0.258612,"
      "2.88298,-0.258612,-2.88298,0.785398,2.35619,-0.785398,-2.35619,"
      "0.785398,2.35619,-0.785398,-2.35619,1.22544,1.91615,-1.22544,"
      "-1.91615,1.22544,1.91615,-1.22544,-1.91615,0.345357,2.79624,"
      "-0.345357,-2.79624,0.345357,2.79624,-0.345357,-2.79624,0.785398,"
      "2.35619,-0.785398,-2.35619,0.785398,2.35619,-0.785398,-2.35619,"
      "1.13111,2.01048,-1.13111,-2.01048,1.13111,2.01048,-1.13111,"
      "-2.01048,0.439687,2.70191,-0.439687,-2.70191,0.439687,2.70191,"
      "-0.439687,-2.70191,0.785398,2.35619,-0.785398,-2.35619,0.785398,"
      "2.35619,-0.785398,-2.35619,1.02774,2.11385,-1.02774,-2.11385,"
      "1.02774,2.11385,-1.02774,-2.11385,0.543057,2.59854,-0.543057,"
      "-2.59854,0.543057,2.59854,-0.543057,-2.59854,0.785398,2.35619,"
      "-0.785398,-2.35619,0.785398,2.35619,-0.785398,-2.35619,0.91332,"
      "2.22827,-0.91332,-2.22827,0.91332,2.22827,-0.91332,-2.22827,"
      "0.657477,2.48412,-0.657477,-2.48412,0.657477,2.48412,-0.657477,"
      "-2.48412,0.785398,2.35619,-0.785398,-2.35619,0.785398,2.35619,"
      "-0.785398,-2.35619,0.641319,2.50027,-0.641319,-2.50027,0.641319,"
      "2.50027,-0.641319,-2.50027,0.929478,2.21212,-0.929478,-2.21212,"
      "0.929478,2.21212,-0.929478,-2.21212,0.785398,2.35619,-0.785398,"
      "-2.35619,0.785398,2.35619,-0.785398,-2.35619,0.478857,2.66274,"
      "-0.478857,-2.66274,0.478857,2.66274,-0.478857,-2.66274,1.09194,"
      "2.04965,-1.09194,-2.04965,1.09194,2.04965,-1.09194,-2.04965,"
      "0.785398,2.35619,-0.785398,-2.35619,0.785398,2.35619,-0.785398,"
      "-2.35619,0.297608,2.84398,-0.297608,-2.84398,0.297608,2.84398,"
      "-0.297608,-2.84398,1.27319,1.8684,-1.27319,-1.8684,1.27319,1.8684,"
      "-1.27319,-1.8684,0.785398,2.35619,-0.785398,-2.35619,0.785398,"
      "2.35619,-0.785398,-2.35619,0.101206,3.04039,-0.101206,-3.04039,"
      "0.101206,3.04039,-0.101206,-3.04039,1.46959,1.672,-1.46959,-1.672,"
      "1.46959,1.672,-1.46959,-1.672,1.44681,1.69478,-1.44681,-1.69478,"
      "0.123986,3.01761,-0.123986,-3.01761,0,3.14159,0,3.14159,0,"
      "3.14159,0,3.14159,1.5708,-1.5708,1.5708,-1.5708,1.5708,-1.5708,"
      "1.5708,-1.5708,1.27231,1.86929,-1.27231,-1.86929,0.29849,2.8431,"
      "-0.29849,-2.8431,0,3.14159,0,3.14159,0,3.14159,0,3.14159,1.5708,"
      "-1.5708,1.5708,-1.5708,1.5708,-1.5708,1.5708,-1.5708,1.08176,"
      "2.05983,-1.08176,-2.05983,0.489036,2.65256,-0.489036,-2.65256,0,"
      "3.14159,0,3.14159,0,3.14159,0,3.14159,1.5708,-1.5708,1.5708,"
      "-1.5708,1.5708,-1.5708,1.5708,-1.5708,0.884784,2.25681,-0.884784,"
      "-2.25681,0.686012,2.45558,-0.686012,-2.45558,0,3.14159,0,3.14159,"
      "0,3.14159,0,3.14159,1.5708,-1.5708,1.5708,-1.5708,1.5708,-1.5708,"
      "1.5708,-1.5708,1.28447,1.85712,-1.28447,-1.85712,1.28447,1.85712,"
      "-1.28447,-1.85712,1.50975,1.63184,-1.50975,-1.63184,1.50975,"
      "1.63184,-1.50975,-1.63184,0.286325,2.85527,-0.286325,-2.85527,"
      "0.286325,2.85527,-0.286325,-2.85527,1.36609,1.7755,-1.36609,"
      "-1.7755,1.36609,1.7755,-1.36609,-1.7755,0.0610476,3.08055,"
      "-0.0610476,-3.08055,0.0610476,3.08055,-0.0610476,-3.08055,0.204703,"
      "2.93689,-0.204703,-2.93689,0.204703,2.93689,-0.204703,-2.93689,"
      "1.28034,1.86125,-1.28034,-1.86125,1.28034,1.86125,-1.28034,"
      "-1.86125,1.4152,1.72639,-1.4152,-1.72639,1.4152,1.72639,-1.4152,"
      "-1.72639,0.290455,2.85114,-0.290455,-2.85114,0.290455,2.85114,"
      "-0.290455,-2.85114,1.08752,2.05407,-1.08752,-2.05407,1.08752,"
      "2.05407,-1.08752,-2.05407,0.155593,2.986,-0.155593,-2.986,0.155593,"
      "2.986,-0.155593,-2.986,0.483274,2.65832,-0.483274,-2.65832,"
      "0.483274,2.65832,-0.483274,-2.65832,0.977687,2.16391,-0.977687,"
      "-2.16391,0.977687,2.16391,-0.977687,-2.16391,1.16759,1.974,"
      "-1.16759,-1.974,1.16759,1.974,-1.16759,-1.974,0.593109,2.54848,"
      "-0.593109,-2.54848,0.593109,2.54848,-0.593109,-2.54848,1.00658,"
      "2.13501,-1.00658,-2.13501,1.00658,2.13501,-1.00658,-2.13501,"
      "0.403203,2.73839,-0.403203,-2.73839,0.403203,2.73839,-0.403203,"
      "-2.73839,0.564214,2.57738,-0.564214,-2.57738,0.564214,2.57738,"
      "-0.564214,-2.57738,1.14684,1.99476,-1.14684,-1.99476,1.14684,"
      "1.99476,-1.14684,-1.99476,1.43787,1.70372,-1.43787,-1.70372,"
      "1.43787,1.70372,-1.43787,-1.70372,0.423961,2.71763,-0.423961,"
      "-2.71763,0.423961,2.71763,-0.423961,-2.71763,1.28276,1.85883,"
      "-1.28276,-1.85883,1.28276,1.85883,-1.28276,-1.85883,0.132928,"
      "3.00866,-0.132928,-3.00866,0.132928,3.00866,-0.132928,-3.00866,"
      "0.288033,2.85356,-0.288033,-2.85356,0.288033,2.85356,-0.288033,"
      "-2.85356,1.01454,2.12705,-1.01454,-2.12705,1.01454,2.12705,"
      "-1.01454,-2.12705,1.26736,1.87423,-1.26736,-1.87423,1.26736,"
      "1.87423,-1.26736,-1.87423,0.556254,2.58534,-0.556254,-2.58534,"
      "0.556254,2.58534,-0.556254,-2.58534,1.10428,2.03731,-1.10428,"
      "-2.03731,1.10428,2.03731,-1.10428,-2.03731,0.303433,2.83816,"
      "-0.303433,-2.83816,0.303433,2.83816,-0.303433,-2.83816,0.466512,"
      "2.67508,-0.466512,-2.67508,0.466512,2.67508,-0.466512,-2.67508,"
      "1.34763,1.79396,-1.34763,-1.79396,1.34763,1.79396,-1.34763,"
      "-1.79396,1.38778,1.75381,-1.38778,-1.75381,1.38778,1.75381,"
      "-1.38778,-1.75381,0.223162,2.91843,-0.223162,-2.91843,0.223162,"
      "2.91843,-0.223162,-2.91843,0.88664,2.25495,-0.88664,-2.25495,"
      "0.88664,2.25495,-0.88664,-2.25495,0.183013,2.95858,-0.183013,"
      "-2.95858,0.183013,2.95858,-0.183013,-2.95858,0.684156,2.45744,"
      "-0.684156,-2.45744,0.684156,2.45744,-0.684156,-2.45744,1.39596,"
      "1.74563,-1.39596,-1.74563,1.39596,1.74563,-1.39596,-1.74563,1.4982,"
      "1.64339,-1.4982,-1.64339,1.4982,1.64339,-1.4982,-1.64339,0.174831,"
      "2.96676,-0.174831,-2.96676,0.174831,2.96676,-0.174831,-2.96676,"
      "1.18025,1.96135,-1.18025,-1.96135,1.18025,1.96135,-1.18025,"
      "-1.96135,0.0725924,3.069,-0.0725924,-3.069,0.0725924,3.069,"
      "-0.0725924,-3.069,0.39055,2.75104,-0.39055,-2.75104,0.39055,"
      "2.75104,-0.39055,-2.75104,1.44357,1.69802,-1.44357,-1.69802,"
      "1.44357,1.69802,-1.44357,-1.69802,1.48607,1.65552,-1.48607,"
      "-1.65552,1.48607,1.65552,-1.48607,-1.65552,0.127224,3.01437,"
      "-0.127224,-3.01437,0.127224,3.01437,-0.127224,-3.01437,0.98468,"
      "2.15691,-0.98468,-2.15691,0.98468,2.15691,-0.98468,-2.15691,"
      "0.0847256,3.05687,-0.0847256,-3.05687,0.0847256,3.05687,-0.0847256,"
      "-3.05687,0.586117,2.55548,-0.586117,-2.55548,0.586117,2.55548,"
      "-0.586117,-2.55548,1.13825,2.00334,-1.13825,-2.00334,1.13825,"
      "2.00334,-1.13825,-2.00334,1.21434,1.92726,-1.21434,-1.92726,"
      "1.21434,1.92726,-1.21434,-1.92726,0.432543,2.70905,-0.432543,"
      "-2.70905,0.432543,2.70905,-0.432543,-2.70905,0.8921,2.24949,"
      "-0.8921,-2.24949,0.8921,2.24949,-0.8921,-2.24949,0.35646,2.78513,"
      "-0.35646,-2.78513,0.35646,2.78513,-0.35646,-2.78513,0.678697,"
      "2.4629,-0.678697,-2.4629,0.678697,2.4629,-0.678697,-2.4629,1.06682,"
      "2.07477,-1.06682,-2.07477,1.06682,2.07477,-1.06682,-2.07477,"
      "1.35708,1.78451,-1.35708,-1.78451,1.35708,1.78451,-1.35708,"
      "-1.78451,0.503975,2.63762,-0.503975,-2.63762,0.503975,2.63762,"
      "-0.503975,-2.63762,1.19587,1.94573,-1.19587,-1.94573,1.19587,"
      "1.94573,-1.19587,-1.94573,0.213717,2.92788,-0.213717,-2.92788,"
      "0.213717,2.92788,-0.213717,-2.92788,0.37493,2.76666,-0.37493,"
      "-2.76666,0.37493,2.76666,-0.37493,-2.76666,1.19872,1.94287,"
      "-1.19872,-1.94287,1.19872,1.94287,-1.19872,-1.94287,1.32117,"
      "1.82042,-1.32117,-1.82042,1.32117,1.82042,-1.32117,-1.82042,"
      "0.372071,2.76952,-0.372071,-2.76952,0.372071,2.76952,-0.372071,"
      "-2.76952,0.992123,2.14947,-0.992123,-2.14947,0.992123,2.14947,"
      "-0.992123,-2.14947,0.249621,2.89197,-0.249621,-2.89197,0.249621,"
      "2.89197,-0.249621,-2.89197,0.578673,2.56292,-0.578673,-2.56292,"
      "0.578673,2.56292,-0.578673,-2.56292,0.950366,2.19123,-0.950366,"
      "-2.19123,0.950366,2.19123,-0.950366,-2.19123,1.05584,2.08576,"
      "-1.05584,-2.08576,1.05584,2.08576,-1.05584,-2.08576,0.620431,"
      "2.52116,-0.620431,-2.52116,0.620431,2.52116,-0.620431,-2.52116,"
      "0.900988,2.2406,-0.900988,-2.2406,0.900988,2.2406,-0.900988,"
      "-2.2406,0.514961,2.62663,-0.514961,-2.62663,0.514961,2.62663,"
      "-0.514961,-2.62663,0.669809,2.47178,-0.669809,-2.47178,0.669809,"
      "2.47178,-0.669809,-2.47178";
  Eigen::VectorXd Ge_theta_ref = ReadVectorFromString(Ge_theta_ref_string);

  std::string Ge_weight_ref_string =
      "0.00180741,0.00180741,0.00180741,0.00180741,0.00180741,"
      "0.00180741,0.0141469,0.0141469,0.0141469,0.0141469,0.0141469,"
      "0.0141469,0.0141469,0.0141469,0.00621788,0.00621788,0.00621788,"
      "0.00621788,0.00621788,0.00621788,0.00621788,0.00621788,0.00621788,"
      "0.00621788,0.00621788,0.00621788,0.00621788,0.00621788,0.00621788,"
      "0.00621788,0.00621788,0.00621788,0.00621788,0.00621788,0.00621788,"
      "0.00621788,0.00621788,0.00621788,0.00924632,0.00924632,0.00924632,"
      "0.00924632,0.00924632,0.00924632,0.00924632,0.00924632,0.00924632,"
      "0.00924632,0.00924632,0.00924632,0.00924632,0.00924632,0.00924632,"
      "0.00924632,0.00924632,0.00924632,0.00924632,0.00924632,0.00924632,"
      "0.00924632,0.00924632,0.00924632,0.0111704,0.0111704,0.0111704,"
      "0.0111704,0.0111704,0.0111704,0.0111704,0.0111704,0.0111704,"
      "0.0111704,0.0111704,0.0111704,0.0111704,0.0111704,0.0111704,"
      "0.0111704,0.0111704,0.0111704,0.0111704,0.0111704,0.0111704,"
      "0.0111704,0.0111704,0.0111704,0.0124261,0.0124261,0.0124261,"
      "0.0124261,0.0124261,0.0124261,0.0124261,0.0124261,0.0124261,"
      "0.0124261,0.0124261,0.0124261,0.0124261,0.0124261,0.0124261,"
      "0.0124261,0.0124261,0.0124261,0.0124261,0.0124261,0.0124261,"
      "0.0124261,0.0124261,0.0124261,0.0132362,0.0132362,0.0132362,"
      "0.0132362,0.0132362,0.0132362,0.0132362,0.0132362,0.0132362,"
      "0.0132362,0.0132362,0.0132362,0.0132362,0.0132362,0.0132362,"
      "0.0132362,0.0132362,0.0132362,0.0132362,0.0132362,0.0132362,"
      "0.0132362,0.0132362,0.0132362,0.0137323,0.0137323,0.0137323,"
      "0.0137323,0.0137323,0.0137323,0.0137323,0.0137323,0.0137323,"
      "0.0137323,0.0137323,0.0137323,0.0137323,0.0137323,0.0137323,"
      "0.0137323,0.0137323,0.0137323,0.0137323,0.0137323,0.0137323,"
      "0.0137323,0.0137323,0.0137323,0.0140038,0.0140038,0.0140038,"
      "0.0140038,0.0140038,0.0140038,0.0140038,0.0140038,0.0140038,"
      "0.0140038,0.0140038,0.0140038,0.0140038,0.0140038,0.0140038,"
      "0.0140038,0.0140038,0.0140038,0.0140038,0.0140038,0.0140038,"
      "0.0140038,0.0140038,0.0140038,0.0141211,0.0141211,0.0141211,"
      "0.0141211,0.0141211,0.0141211,0.0141211,0.0141211,0.0141211,"
      "0.0141211,0.0141211,0.0141211,0.0141211,0.0141211,0.0141211,"
      "0.0141211,0.0141211,0.0141211,0.0141211,0.0141211,0.0141211,"
      "0.0141211,0.0141211,0.0141211,0.0141402,0.0141402,0.0141402,"
      "0.0141402,0.0141402,0.0141402,0.0141402,0.0141402,0.0141402,"
      "0.0141402,0.0141402,0.0141402,0.0141402,0.0141402,0.0141402,"
      "0.0141402,0.0141402,0.0141402,0.0141402,0.0141402,0.0141402,"
      "0.0141402,0.0141402,0.0141402,0.0141517,0.0141517,0.0141517,"
      "0.0141517,0.0141517,0.0141517,0.0141517,0.0141517,0.0141517,"
      "0.0141517,0.0141517,0.0141517,0.0141517,0.0141517,0.0141517,"
      "0.0141517,0.0141517,0.0141517,0.0141517,0.0141517,0.0141517,"
      "0.0141517,0.0141517,0.0141517,0.0142036,0.0142036,0.0142036,"
      "0.0142036,0.0142036,0.0142036,0.0142036,0.0142036,0.0142036,"
      "0.0142036,0.0142036,0.0142036,0.0142036,0.0142036,0.0142036,"
      "0.0142036,0.0142036,0.0142036,0.0142036,0.0142036,0.0142036,"
      "0.0142036,0.0142036,0.0142036,0.0142627,0.0142627,0.0142627,"
      "0.0142627,0.0142627,0.0142627,0.0142627,0.0142627,0.0142627,"
      "0.0142627,0.0142627,0.0142627,0.0142627,0.0142627,0.0142627,"
      "0.0142627,0.0142627,0.0142627,0.0142627,0.0142627,0.0142627,"
      "0.0142627,0.0142627,0.0142627,0.0085745,0.0085745,0.0085745,"
      "0.0085745,0.0085745,0.0085745,0.0085745,0.0085745,0.0085745,"
      "0.0085745,0.0085745,0.0085745,0.0085745,0.0085745,0.0085745,"
      "0.0085745,0.0085745,0.0085745,0.0085745,0.0085745,0.0085745,"
      "0.0085745,0.0085745,0.0085745,0.0118804,0.0118804,0.0118804,"
      "0.0118804,0.0118804,0.0118804,0.0118804,0.0118804,0.0118804,"
      "0.0118804,0.0118804,0.0118804,0.0118804,0.0118804,0.0118804,"
      "0.0118804,0.0118804,0.0118804,0.0118804,0.0118804,0.0118804,"
      "0.0118804,0.0118804,0.0118804,0.0135017,0.0135017,0.0135017,"
      "0.0135017,0.0135017,0.0135017,0.0135017,0.0135017,0.0135017,"
      "0.0135017,0.0135017,0.0135017,0.0135017,0.0135017,0.0135017,"
      "0.0135017,0.0135017,0.0135017,0.0135017,0.0135017,0.0135017,"
      "0.0135017,0.0135017,0.0135017,0.0141912,0.0141912,0.0141912,"
      "0.0141912,0.0141912,0.0141912,0.0141912,0.0141912,0.0141912,"
      "0.0141912,0.0141912,0.0141912,0.0141912,0.0141912,0.0141912,"
      "0.0141912,0.0141912,0.0141912,0.0141912,0.0141912,0.0141912,"
      "0.0141912,0.0141912,0.0141912,0.0106021,0.0106021,0.0106021,"
      "0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,"
      "0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,"
      "0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,"
      "0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,"
      "0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,"
      "0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,"
      "0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,"
      "0.0106021,0.0106021,0.0106021,0.0135121,0.0135121,0.0135121,"
      "0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,"
      "0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,"
      "0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,"
      "0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,"
      "0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,"
      "0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,"
      "0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,"
      "0.0135121,0.0135121,0.0135121,0.0139308,0.0139308,0.0139308,"
      "0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,"
      "0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,"
      "0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,"
      "0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,"
      "0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,"
      "0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,"
      "0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,"
      "0.0139308,0.0139308,0.0139308,0.0120216,0.0120216,0.0120216,"
      "0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,"
      "0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,"
      "0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,"
      "0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,"
      "0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,"
      "0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,"
      "0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,"
      "0.0120216,0.0120216,0.0120216,0.01358,0.01358,0.01358,0.01358,"
      "0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,"
      "0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,"
      "0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,"
      "0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,"
      "0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,"
      "0.01358,0.01358,0.01358,0.01358,0.0141598,0.0141598,0.0141598,"
      "0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,"
      "0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,"
      "0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,"
      "0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,"
      "0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,"
      "0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,"
      "0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,"
      "0.0141598,0.0141598,0.0141598,0.01285,0.01285,0.01285,0.01285,"
      "0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,"
      "0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,"
      "0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,"
      "0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,"
      "0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,"
      "0.01285,0.01285,0.01285,0.01285,0.0139356,0.0139356,0.0139356,"
      "0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,"
      "0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,"
      "0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,"
      "0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,"
      "0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,"
      "0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,"
      "0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,"
      "0.0139356,0.0139356,0.0139356,0.0141094,0.0141094,0.0141094,"
      "0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,"
      "0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,"
      "0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,"
      "0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,"
      "0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,"
      "0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,"
      "0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,"
      "0.0141094,0.0141094,0.0141094,0.0129735,0.0129735,0.0129735,"
      "0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,"
      "0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,"
      "0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,"
      "0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,"
      "0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,"
      "0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,"
      "0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,"
      "0.0129735,0.0129735,0.0129735,0.0139141,0.0139141,0.0139141,"
      "0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,"
      "0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,"
      "0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,"
      "0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,"
      "0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,"
      "0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,"
      "0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,"
      "0.0139141,0.0139141,0.0139141,0.0140967,0.0140967,0.0140967,"
      "0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,"
      "0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,"
      "0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,"
      "0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,"
      "0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,"
      "0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,"
      "0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,"
      "0.0140967,0.0140967,0.0140967";
  Eigen::VectorXd Ge_weight_ref = ReadVectorFromString(Ge_weight_ref_string);

  std::string H_phi_ref_string =
      "1.5708,1.5708,1.5708,1.5708,0,3.14159,0.955317,0.955317,"
      "0.955317,0.955317,2.18628,2.18628,2.18628,2.18628,1.47847,1.47847,"
      "1.47847,1.47847,1.66312,1.66312,1.66312,1.66312,0.789648,0.789648,"
      "0.789648,0.789648,2.35194,2.35194,2.35194,2.35194,0.789648,"
      "0.789648,0.789648,0.789648,2.35194,2.35194,2.35194,2.35194,1.29703,"
      "1.29703,1.29703,1.29703,1.84456,1.84456,1.84456,1.84456,0.821977,"
      "0.821977,0.821977,0.821977,2.31962,2.31962,2.31962,2.31962,"
      "0.821977,0.821977,0.821977,0.821977,2.31962,2.31962,2.31962,"
      "2.31962,1.12256,1.12256,1.12256,1.12256,2.01903,2.01903,2.01903,"
      "2.01903,0.879866,0.879866,0.879866,0.879866,2.26173,2.26173,"
      "2.26173,2.26173,0.879866,0.879866,0.879866,0.879866,2.26173,"
      "2.26173,2.26173,2.26173,0.794322,0.794322,0.794322,0.794322,"
      "2.34727,2.34727,2.34727,2.34727,1.04206,1.04206,1.04206,1.04206,"
      "2.09953,2.09953,2.09953,2.09953,1.04206,1.04206,1.04206,1.04206,"
      "2.09953,2.09953,2.09953,2.09953,0.638758,0.638758,0.638758,"
      "0.638758,2.50283,2.50283,2.50283,2.50283,1.13561,1.13561,1.13561,"
      "1.13561,2.00598,2.00598,2.00598,2.00598,1.13561,1.13561,1.13561,"
      "1.13561,2.00598,2.00598,2.00598,2.00598,0.488413,0.488413,0.488413,"
      "0.488413,2.65318,2.65318,2.65318,2.65318,1.23259,1.23259,1.23259,"
      "1.23259,1.909,1.909,1.909,1.909,1.23259,1.23259,1.23259,1.23259,"
      "1.909,1.909,1.909,1.909,0.343997,0.343997,0.343997,0.343997,"
      "2.7976,2.7976,2.7976,2.7976,1.33,1.33,1.33,1.33,1.81159,1.81159,"
      "1.81159,1.81159,1.33,1.33,1.33,1.33,1.81159,1.81159,1.81159,"
      "1.81159,0.207832,0.207832,0.207832,0.207832,2.93376,2.93376,"
      "2.93376,2.93376,1.42437,1.42437,1.42437,1.42437,1.71722,1.71722,"
      "1.71722,1.71722,1.42437,1.42437,1.42437,1.42437,1.71722,1.71722,"
      "1.71722,1.71722,0.0863039,0.0863039,0.0863039,0.0863039,3.05529,"
      "3.05529,3.05529,3.05529,1.50981,1.50981,1.50981,1.50981,1.63178,"
      "1.63178,1.63178,1.63178,1.50981,1.50981,1.50981,1.50981,1.63178,"
      "1.63178,1.63178,1.63178,1.5708,1.5708,1.5708,1.5708,1.5708,1.5708,"
      "1.5708,1.5708,0.658188,0.658188,2.4834,2.4834,0.912608,0.912608,"
      "2.22898,2.22898,0.658188,0.658188,2.4834,2.4834,0.912608,0.912608,"
      "2.22898,2.22898,1.5708,1.5708,1.5708,1.5708,1.5708,1.5708,1.5708,"
      "1.5708,0.407675,0.407675,2.73392,2.73392,1.16312,1.16312,1.97847,"
      "1.97847,0.407675,0.407675,2.73392,2.73392,1.16312,1.16312,1.97847,"
      "1.97847,1.5708,1.5708,1.5708,1.5708,1.5708,1.5708,1.5708,1.5708,"
      "0.173345,0.173345,2.96825,2.96825,1.39745,1.39745,1.74414,1.74414,"
      "0.173345,0.173345,2.96825,2.96825,1.39745,1.39745,1.74414,1.74414,"
      "0.723776,0.723776,0.723776,0.723776,2.41782,2.41782,2.41782,"
      "2.41782,1.21127,1.21127,1.21127,1.21127,1.93032,1.93032,1.93032,"
      "1.93032,0.723776,0.723776,0.723776,0.723776,2.41782,2.41782,"
      "2.41782,2.41782,0.975171,0.975171,0.975171,0.975171,2.16642,"
      "2.16642,2.16642,2.16642,1.21127,1.21127,1.21127,1.21127,1.93032,"
      "1.93032,1.93032,1.93032,0.975171,0.975171,0.975171,0.975171,"
      "2.16642,2.16642,2.16642,2.16642,0.573426,0.573426,0.573426,"
      "0.573426,2.56817,2.56817,2.56817,2.56817,1.30418,1.30418,1.30418,"
      "1.30418,1.83742,1.83742,1.83742,1.83742,0.573426,0.573426,0.573426,"
      "0.573426,2.56817,2.56817,2.56817,2.56817,1.0767,1.0767,1.0767,"
      "1.0767,2.0649,2.0649,2.0649,2.0649,1.30418,1.30418,1.30418,"
      "1.30418,1.83742,1.83742,1.83742,1.83742,1.0767,1.0767,1.0767,"
      "1.0767,2.0649,2.0649,2.0649,2.0649,0.675618,0.675618,0.675618,"
      "0.675618,2.46597,2.46597,2.46597,2.46597,1.38812,1.38812,1.38812,"
      "1.38812,1.75347,1.75347,1.75347,1.75347,0.675618,0.675618,0.675618,"
      "0.675618,2.46597,2.46597,2.46597,2.46597,0.929278,0.929278,"
      "0.929278,0.929278,2.21231,2.21231,2.21231,2.21231,1.38812,1.38812,"
      "1.38812,1.38812,1.75347,1.75347,1.75347,1.75347,0.929278,0.929278,"
      "0.929278,0.929278,2.21231,2.21231,2.21231,2.21231,0.429405,"
      "0.429405,0.429405,0.429405,2.71219,2.71219,2.71219,2.71219,1.39786,"
      "1.39786,1.39786,1.39786,1.74374,1.74374,1.74374,1.74374,0.429405,"
      "0.429405,0.429405,0.429405,2.71219,2.71219,2.71219,2.71219,1.18197,"
      "1.18197,1.18197,1.18197,1.95962,1.95962,1.95962,1.95962,1.39786,"
      "1.39786,1.39786,1.39786,1.74374,1.74374,1.74374,1.74374,1.18197,"
      "1.18197,1.18197,1.18197,1.95962,1.95962,1.95962,1.95962,0.293967,"
      "0.293967,0.293967,0.293967,2.84763,2.84763,2.84763,2.84763,1.48857,"
      "1.48857,1.48857,1.48857,1.65302,1.65302,1.65302,1.65302,0.293967,"
      "0.293967,0.293967,0.293967,2.84763,2.84763,2.84763,2.84763,1.28922,"
      "1.28922,1.28922,1.28922,1.85237,1.85237,1.85237,1.85237,1.48857,"
      "1.48857,1.48857,1.48857,1.65302,1.65302,1.65302,1.65302,1.28922,"
      "1.28922,1.28922,1.28922,1.85237,1.85237,1.85237,1.85237,0.536741,"
      "0.536741,0.536741,0.536741,2.60485,2.60485,2.60485,2.60485,1.48068,"
      "1.48068,1.48068,1.48068,1.66091,1.66091,1.66091,1.66091,0.536741,"
      "0.536741,0.536741,0.536741,2.60485,2.60485,2.60485,2.60485,1.04332,"
      "1.04332,1.04332,1.04332,2.09828,2.09828,2.09828,2.09828,1.48068,"
      "1.48068,1.48068,1.48068,1.66091,1.66091,1.66091,1.66091,1.04332,"
      "1.04332,1.04332,1.04332,2.09828,2.09828,2.09828,2.09828";
  Eigen::VectorXd H_phi_ref = ReadVectorFromString(H_phi_ref_string);
  std::string H_theta_ref_string =
      "0,3.14159,1.5708,-1.5708,0,0,0.785398,2.35619,"
      "-0.785398,-2.35619,0.785398,2.35619,-0.785398,-2.35619,0.785398,"
      "2.35619,-0.785398,-2.35619,0.785398,2.35619,-0.785398,-2.35619,"
      "0.130194,3.0114,-0.130194,-3.0114,0.130194,3.0114,-0.130194,"
      "-3.0114,1.4406,1.70099,-1.4406,-1.70099,1.4406,1.70099,-1.4406,"
      "-1.70099,0.785398,2.35619,-0.785398,-2.35619,0.785398,2.35619,"
      "-0.785398,-2.35619,0.37803,2.76356,-0.37803,-2.76356,0.37803,"
      "2.76356,-0.37803,-2.76356,1.19277,1.94883,-1.19277,-1.94883,"
      "1.19277,1.94883,-1.19277,-1.94883,0.785398,2.35619,-0.785398,"
      "-2.35619,0.785398,2.35619,-0.785398,-2.35619,0.59722,2.54437,"
      "-0.59722,-2.54437,0.59722,2.54437,-0.59722,-2.54437,0.973576,"
      "2.16802,-0.973576,-2.16802,0.973576,2.16802,-0.973576,-2.16802,"
      "0.785398,2.35619,-0.785398,-2.35619,0.785398,2.35619,-0.785398,"
      "-2.35619,0.946878,2.19471,-0.946878,-2.19471,0.946878,2.19471,"
      "-0.946878,-2.19471,0.623918,2.51767,-0.623918,-2.51767,0.623918,"
      "2.51767,-0.623918,-2.51767,0.785398,2.35619,-0.785398,-2.35619,"
      "0.785398,2.35619,-0.785398,-2.35619,1.08726,2.05433,-1.08726,"
      "-2.05433,1.08726,2.05433,-1.08726,-2.05433,0.483532,2.65806,"
      "-0.483532,-2.65806,0.483532,2.65806,-0.483532,-2.65806,0.785398,"
      "2.35619,-0.785398,-2.35619,0.785398,2.35619,-0.785398,-2.35619,"
      "1.21139,1.9302,-1.21139,-1.9302,1.21139,1.9302,-1.21139,-1.9302,"
      "0.359403,2.78219,-0.359403,-2.78219,0.359403,2.78219,-0.359403,"
      "-2.78219,0.785398,2.35619,-0.785398,-2.35619,0.785398,2.35619,"
      "-0.785398,-2.35619,1.3227,1.81889,-1.3227,-1.81889,1.3227,1.81889,"
      "-1.3227,-1.81889,0.248096,2.8935,-0.248096,-2.8935,0.248096,2.8935,"
      "-0.248096,-2.8935,0.785398,2.35619,-0.785398,-2.35619,0.785398,"
      "2.35619,-0.785398,-2.35619,1.42277,1.71882,-1.42277,-1.71882,"
      "1.42277,1.71882,-1.42277,-1.71882,0.148022,2.99357,-0.148022,"
      "-2.99357,0.148022,2.99357,-0.148022,-2.99357,0.785398,2.35619,"
      "-0.785398,-2.35619,0.785398,2.35619,-0.785398,-2.35619,1.50969,"
      "1.6319,-1.50969,-1.6319,1.50969,1.6319,-1.50969,-1.6319,0.0611019,"
      "3.08049,-0.0611019,-3.08049,0.0611019,3.08049,-0.0611019,-3.08049,"
      "0.912608,2.22898,-0.912608,-2.22898,0.658188,2.4834,-0.658188,"
      "-2.4834,0,3.14159,0,3.14159,0,3.14159,0,3.14159,1.5708,-1.5708,"
      "1.5708,-1.5708,1.5708,-1.5708,1.5708,-1.5708,1.16312,1.97847,"
      "-1.16312,-1.97847,0.407675,2.73392,-0.407675,-2.73392,0,3.14159,0,"
      "3.14159,0,3.14159,0,3.14159,1.5708,-1.5708,1.5708,-1.5708,1.5708,"
      "-1.5708,1.5708,-1.5708,1.39745,1.74414,-1.39745,-1.74414,0.173345,"
      "2.96825,-0.173345,-2.96825,0,3.14159,0,3.14159,0,3.14159,0,"
      "3.14159,1.5708,-1.5708,1.5708,-1.5708,1.5708,-1.5708,1.5708,"
      "-1.5708,0.560119,2.58147,-0.560119,-2.58147,0.560119,2.58147,"
      "-0.560119,-2.58147,0.928113,2.21348,-0.928113,-2.21348,0.928113,"
      "2.21348,-0.928113,-2.21348,1.01068,2.13092,-1.01068,-2.13092,"
      "1.01068,2.13092,-1.01068,-2.13092,1.13182,2.00978,-1.13182,"
      "-2.00978,1.13182,2.00978,-1.13182,-2.00978,0.642684,2.49891,"
      "-0.642684,-2.49891,0.642684,2.49891,-0.642684,-2.49891,0.438981,"
      "2.70261,-0.438981,-2.70261,0.438981,2.70261,-0.438981,-2.70261,"
      "0.507107,2.63449,-0.507107,-2.63449,0.507107,2.63449,-0.507107,"
      "-2.63449,1.05686,2.08473,-1.05686,-2.08473,1.05686,2.08473,"
      "-1.05686,-2.08473,1.06369,2.0779,-1.06369,-2.0779,1.06369,2.0779,"
      "-1.06369,-2.0779,1.26687,1.87472,-1.26687,-1.87472,1.26687,1.87472,"
      "-1.26687,-1.87472,0.513937,2.62766,-0.513937,-2.62766,0.513937,"
      "2.62766,-0.513937,-2.62766,0.303922,2.83767,-0.303922,-2.83767,"
      "0.303922,2.83767,-0.303922,-2.83767,0.294735,2.84686,-0.294735,"
      "-2.84686,0.294735,2.84686,-0.294735,-2.84686,0.916579,2.22501,"
      "-0.916579,-2.22501,0.916579,2.22501,-0.916579,-2.22501,1.27606,"
      "1.86553,-1.27606,-1.86553,1.27606,1.86553,-1.27606,-1.86553,"
      "1.34206,1.79953,-1.34206,-1.79953,1.34206,1.79953,-1.34206,"
      "-1.79953,0.654217,2.48738,-0.654217,-2.48738,0.654217,2.48738,"
      "-0.654217,-2.48738,0.228733,2.91286,-0.228733,-2.91286,0.228733,"
      "2.91286,-0.228733,-2.91286,0.426102,2.71549,-0.426102,-2.71549,"
      "0.426102,2.71549,-0.426102,-2.71549,1.17576,1.96584,-1.17576,"
      "-1.96584,1.17576,1.96584,-1.17576,-1.96584,1.14469,1.9969,-1.14469,"
      "-1.9969,1.14469,1.9969,-1.14469,-1.9969,1.38375,1.75785,-1.38375,"
      "-1.75785,1.38375,1.75785,-1.38375,-1.75785,0.395039,2.74655,"
      "-0.395039,-2.74655,0.395039,2.74655,-0.395039,-2.74655,0.18705,"
      "2.95454,-0.18705,-2.95454,0.18705,2.95454,-0.18705,-2.95454,"
      "0.287391,2.8542,-0.287391,-2.8542,0.287391,2.8542,-0.287391,"
      "-2.8542,1.28824,1.85335,-1.28824,-1.85335,1.28824,1.85335,-1.28824,"
      "-1.85335,1.28341,1.85819,-1.28341,-1.85819,1.28341,1.85819,"
      "-1.28341,-1.85819,1.48519,1.6564,-1.48519,-1.6564,1.48519,1.6564,"
      "-1.48519,-1.6564,0.282554,2.85904,-0.282554,-2.85904,0.282554,"
      "2.85904,-0.282554,-2.85904,0.0856016,3.05599,-0.0856016,-3.05599,"
      "0.0856016,3.05599,-0.0856016,-3.05599,0.176915,2.96468,-0.176915,"
      "-2.96468,0.176915,2.96468,-0.176915,-2.96468,1.04094,2.10065,"
      "-1.04094,-2.10065,1.04094,2.10065,-1.04094,-2.10065,1.39388,"
      "1.74771,-1.39388,-1.74771,1.39388,1.74771,-1.39388,-1.74771,"
      "1.46646,1.67513,-1.46646,-1.67513,1.46646,1.67513,-1.46646,"
      "-1.67513,0.529854,2.61174,-0.529854,-2.61174,0.529854,2.61174,"
      "-0.529854,-2.61174,0.104337,3.03726,-0.104337,-3.03726,0.104337,"
      "3.03726,-0.104337,-3.03726";
  Eigen::VectorXd H_theta_ref = ReadVectorFromString(H_theta_ref_string);
  std::string H_weight_ref_string =
      "0.00388944,0.00388944,0.00388944,0.00388944,0.00388944,"
      "0.00388944,0.0232777,0.0232777,0.0232777,0.0232777,0.0232777,"
      "0.0232777,0.0232777,0.0232777,0.0235216,0.0235216,0.0235216,"
      "0.0235216,0.0235216,0.0235216,0.0235216,0.0235216,0.0235216,"
      "0.0235216,0.0235216,0.0235216,0.0235216,0.0235216,0.0235216,"
      "0.0235216,0.0235216,0.0235216,0.0235216,0.0235216,0.0235216,"
      "0.0235216,0.0235216,0.0235216,0.0233585,0.0233585,0.0233585,"
      "0.0233585,0.0233585,0.0233585,0.0233585,0.0233585,0.0233585,"
      "0.0233585,0.0233585,0.0233585,0.0233585,0.0233585,0.0233585,"
      "0.0233585,0.0233585,0.0233585,0.0233585,0.0233585,0.0233585,"
      "0.0233585,0.0233585,0.0233585,0.0232733,0.0232733,0.0232733,"
      "0.0232733,0.0232733,0.0232733,0.0232733,0.0232733,0.0232733,"
      "0.0232733,0.0232733,0.0232733,0.0232733,0.0232733,0.0232733,"
      "0.0232733,0.0232733,0.0232733,0.0232733,0.0232733,0.0232733,"
      "0.0232733,0.0232733,0.0232733,0.0232065,0.0232065,0.0232065,"
      "0.0232065,0.0232065,0.0232065,0.0232065,0.0232065,0.0232065,"
      "0.0232065,0.0232065,0.0232065,0.0232065,0.0232065,0.0232065,"
      "0.0232065,0.0232065,0.0232065,0.0232065,0.0232065,0.0232065,"
      "0.0232065,0.0232065,0.0232065,0.0228516,0.0228516,0.0228516,"
      "0.0228516,0.0228516,0.0228516,0.0228516,0.0228516,0.0228516,"
      "0.0228516,0.0228516,0.0228516,0.0228516,0.0228516,0.0228516,"
      "0.0228516,0.0228516,0.0228516,0.0228516,0.0228516,0.0228516,"
      "0.0228516,0.0228516,0.0228516,0.0219857,0.0219857,0.0219857,"
      "0.0219857,0.0219857,0.0219857,0.0219857,0.0219857,0.0219857,"
      "0.0219857,0.0219857,0.0219857,0.0219857,0.0219857,0.0219857,"
      "0.0219857,0.0219857,0.0219857,0.0219857,0.0219857,0.0219857,"
      "0.0219857,0.0219857,0.0219857,0.0203225,0.0203225,0.0203225,"
      "0.0203225,0.0203225,0.0203225,0.0203225,0.0203225,0.0203225,"
      "0.0203225,0.0203225,0.0203225,0.0203225,0.0203225,0.0203225,"
      "0.0203225,0.0203225,0.0203225,0.0203225,0.0203225,0.0203225,"
      "0.0203225,0.0203225,0.0203225,0.0174011,0.0174011,0.0174011,"
      "0.0174011,0.0174011,0.0174011,0.0174011,0.0174011,0.0174011,"
      "0.0174011,0.0174011,0.0174011,0.0174011,0.0174011,0.0174011,"
      "0.0174011,0.0174011,0.0174011,0.0174011,0.0174011,0.0174011,"
      "0.0174011,0.0174011,0.0174011,0.0122702,0.0122702,0.0122702,"
      "0.0122702,0.0122702,0.0122702,0.0122702,0.0122702,0.0122702,"
      "0.0122702,0.0122702,0.0122702,0.0122702,0.0122702,0.0122702,"
      "0.0122702,0.0122702,0.0122702,0.0122702,0.0122702,0.0122702,"
      "0.0122702,0.0122702,0.0122702,0.0233378,0.0233378,0.0233378,"
      "0.0233378,0.0233378,0.0233378,0.0233378,0.0233378,0.0233378,"
      "0.0233378,0.0233378,0.0233378,0.0233378,0.0233378,0.0233378,"
      "0.0233378,0.0233378,0.0233378,0.0233378,0.0233378,0.0233378,"
      "0.0233378,0.0233378,0.0233378,0.0214276,0.0214276,0.0214276,"
      "0.0214276,0.0214276,0.0214276,0.0214276,0.0214276,0.0214276,"
      "0.0214276,0.0214276,0.0214276,0.0214276,0.0214276,0.0214276,"
      "0.0214276,0.0214276,0.0214276,0.0214276,0.0214276,0.0214276,"
      "0.0214276,0.0214276,0.0214276,0.0163403,0.0163403,0.0163403,"
      "0.0163403,0.0163403,0.0163403,0.0163403,0.0163403,0.0163403,"
      "0.0163403,0.0163403,0.0163403,0.0163403,0.0163403,0.0163403,"
      "0.0163403,0.0163403,0.0163403,0.0163403,0.0163403,0.0163403,"
      "0.0163403,0.0163403,0.0163403,0.0231581,0.0231581,0.0231581,"
      "0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,"
      "0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,"
      "0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,"
      "0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,"
      "0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,"
      "0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,"
      "0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,"
      "0.0231581,0.0231581,0.0231581,0.0226529,0.0226529,0.0226529,"
      "0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,"
      "0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,"
      "0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,"
      "0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,"
      "0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,"
      "0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,"
      "0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,"
      "0.0226529,0.0226529,0.0226529,0.0232457,0.0232457,0.0232457,"
      "0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,"
      "0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,"
      "0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,"
      "0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,"
      "0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,"
      "0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,"
      "0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,"
      "0.0232457,0.0232457,0.0232457,0.0215376,0.0215376,0.0215376,"
      "0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,"
      "0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,"
      "0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,"
      "0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,"
      "0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,"
      "0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,"
      "0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,"
      "0.0215376,0.0215376,0.0215376,0.0195434,0.0195434,0.0195434,"
      "0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,"
      "0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,"
      "0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,"
      "0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,"
      "0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,"
      "0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,"
      "0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,"
      "0.0195434,0.0195434,0.0195434,0.0226476,0.0226476,0.0226476,"
      "0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,"
      "0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,"
      "0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,"
      "0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,"
      "0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,"
      "0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,"
      "0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,"
      "0.0226476,0.0226476,0.0226476";
  Eigen::VectorXd H_weight_ref = ReadVectorFromString(H_weight_ref_string);

  BOOST_CHECK_EQUAL(Gegrid.phi.size(), Ge_phi_ref.size());
  BOOST_CHECK_EQUAL(Gegrid.theta.size(), Ge_theta_ref.size());
  BOOST_CHECK_EQUAL(Gegrid.weight.size(), Ge_weight_ref.size());
  BOOST_CHECK_EQUAL(Hgrid.phi.size(), H_phi_ref.size());
  BOOST_CHECK_EQUAL(Hgrid.theta.size(), H_theta_ref.size());
  BOOST_CHECK_EQUAL(Hgrid.weight.size(), H_weight_ref.size());

  bool Gephi = Ge_phi_ref.isApprox(Gegrid.phi, 0.001);
  bool Getheta = Ge_theta_ref.isApprox(Gegrid.theta, 0.001);
  if (!Gephi || !Getheta) {
    std::cout << "phi_ref : Phi_comp | theta_ref : theta_comp" << std::endl;
    for (Index i = 0; i < Ge_phi_ref.size(); i++) {
      std::cout << Gegrid.phi[i] << ":" << Ge_phi_ref[i] << " | "
                << Gegrid.theta[i] << ":" << Ge_theta_ref[i] << std::endl;
    }
  }

  bool Geweight = Ge_weight_ref.isApprox(Gegrid.weight, 0.0001);
  BOOST_CHECK_EQUAL(Gephi, true);
  BOOST_CHECK_EQUAL(Getheta, true);
  BOOST_CHECK_EQUAL(Geweight, true);

  bool Hphi = H_phi_ref.isApprox(Hgrid.phi, 0.001);
  bool Htheta = H_theta_ref.isApprox(Hgrid.theta, 0.001);
  bool Hweight = H_weight_ref.isApprox(Hgrid.weight, 0.0001);
  BOOST_CHECK_EQUAL(Hphi, true);
  BOOST_CHECK_EQUAL(Htheta, true);
  BOOST_CHECK_EQUAL(Hweight, true);
}
BOOST_AUTO_TEST_CASE(element_not_implemented) {

  std::ofstream xyzfile("molecule.xyz");
  xyzfile << " 1" << std::endl;
  xyzfile << " mercury" << std::endl;
  xyzfile << " Hg            .000000     .000000     .000000" << std::endl;
  xyzfile.close();

  QMMolecule mol("noname", 0);
  mol.LoadFromFile("molecule.xyz");

  LebedevGrid spheregrid;

  BOOST_REQUIRE_THROW(spheregrid.CalculateSphericalGrids(mol, "xfine"),
                      std::runtime_error);
}

BOOST_AUTO_TEST_CASE(xfine_test) {

  std::ofstream xyzfile("molecule.xyz");
  xyzfile << " 5" << std::endl;
  xyzfile << " methane" << std::endl;
  xyzfile << " Ge            .000000     .000000     .000000" << std::endl;
  xyzfile << " H            .629118     .629118     .629118" << std::endl;
  xyzfile << " H           -.629118    -.629118     .629118" << std::endl;
  xyzfile << " H            .629118    -.629118    -.629118" << std::endl;
  xyzfile << " H           -.629118     .629118    -.629118" << std::endl;
  xyzfile.close();

  QMMolecule mol("noname", 0);
  mol.LoadFromFile("molecule.xyz");

  LebedevGrid spheregrid;

  auto grid = spheregrid.CalculateSphericalGrids(mol, "xfine");

  auto Hgrid = grid.at("H");
  auto Gegrid = grid.at("Ge");

  Eigen::VectorXd Ge_phi_ref = Eigen::VectorXd::Zero(1454);
  Ge_phi_ref << 1.5708, 1.5708, 1.5708, 1.5708, 0, 3.14159, 0.955317, 0.955317,
      0.955317, 0.955317, 2.18628, 2.18628, 2.18628, 2.18628, 0.045685,
      0.045685, 0.045685, 0.045685, 3.09591, 3.09591, 3.09591, 3.09591, 1.5385,
      1.5385, 1.5385, 1.5385, 1.60309, 1.60309, 1.60309, 1.60309, 1.5385,
      1.5385, 1.5385, 1.5385, 1.60309, 1.60309, 1.60309, 1.60309, 0.113903,
      0.113903, 0.113903, 0.113903, 3.02769, 3.02769, 3.02769, 3.02769, 1.49034,
      1.49034, 1.49034, 1.49034, 1.65125, 1.65125, 1.65125, 1.65125, 1.49034,
      1.49034, 1.49034, 1.49034, 1.65125, 1.65125, 1.65125, 1.65125, 0.192716,
      0.192716, 0.192716, 0.192716, 2.94888, 2.94888, 2.94888, 2.94888, 1.43495,
      1.43495, 1.43495, 1.43495, 1.70664, 1.70664, 1.70664, 1.70664, 1.43495,
      1.43495, 1.43495, 1.43495, 1.70664, 1.70664, 1.70664, 1.70664, 0.277769,
      0.277769, 0.277769, 0.277769, 2.86382, 2.86382, 2.86382, 2.86382, 1.37566,
      1.37566, 1.37566, 1.37566, 1.76593, 1.76593, 1.76593, 1.76593, 1.37566,
      1.37566, 1.37566, 1.37566, 1.76593, 1.76593, 1.76593, 1.76593, 0.367019,
      0.367019, 0.367019, 0.367019, 2.77457, 2.77457, 2.77457, 2.77457, 1.31426,
      1.31426, 1.31426, 1.31426, 1.82734, 1.82734, 1.82734, 1.82734, 1.31426,
      1.31426, 1.31426, 1.31426, 1.82734, 1.82734, 1.82734, 1.82734, 0.459379,
      0.459379, 0.459379, 0.459379, 2.68221, 2.68221, 2.68221, 2.68221, 1.25189,
      1.25189, 1.25189, 1.25189, 1.8897, 1.8897, 1.8897, 1.8897, 1.25189,
      1.25189, 1.25189, 1.25189, 1.8897, 1.8897, 1.8897, 1.8897, 0.554252,
      0.554252, 0.554252, 0.554252, 2.58734, 2.58734, 2.58734, 2.58734, 1.18947,
      1.18947, 1.18947, 1.18947, 1.95213, 1.95213, 1.95213, 1.95213, 1.18947,
      1.18947, 1.18947, 1.18947, 1.95213, 1.95213, 1.95213, 1.95213, 0.651332,
      0.651332, 0.651332, 0.651332, 2.49026, 2.49026, 2.49026, 2.49026, 1.12776,
      1.12776, 1.12776, 1.12776, 2.01383, 2.01383, 2.01383, 2.01383, 1.12776,
      1.12776, 1.12776, 1.12776, 2.01383, 2.01383, 2.01383, 2.01383, 0.750502,
      0.750502, 0.750502, 0.750502, 2.39109, 2.39109, 2.39109, 2.39109, 1.06757,
      1.06757, 1.06757, 1.06757, 2.07402, 2.07402, 2.07402, 2.07402, 1.06757,
      1.06757, 1.06757, 1.06757, 2.07402, 2.07402, 2.07402, 2.07402, 0.851786,
      0.851786, 0.851786, 0.851786, 2.28981, 2.28981, 2.28981, 2.28981, 1.00976,
      1.00976, 1.00976, 1.00976, 2.13184, 2.13184, 2.13184, 2.13184, 1.00976,
      1.00976, 1.00976, 1.00976, 2.13184, 2.13184, 2.13184, 2.13184, 1.06131,
      1.06131, 1.06131, 1.06131, 2.08029, 2.08029, 2.08029, 2.08029, 0.90549,
      0.90549, 0.90549, 0.90549, 2.2361, 2.2361, 2.2361, 2.2361, 0.90549,
      0.90549, 0.90549, 0.90549, 2.2361, 2.2361, 2.2361, 2.2361, 1.17, 1.17,
      1.17, 1.17, 1.9716, 1.9716, 1.9716, 1.9716, 0.861806, 0.861806, 0.861806,
      0.861806, 2.27979, 2.27979, 2.27979, 2.27979, 0.861806, 0.861806,
      0.861806, 0.861806, 2.27979, 2.27979, 2.27979, 2.27979, 1.28154, 1.28154,
      1.28154, 1.28154, 1.86005, 1.86005, 1.86005, 1.86005, 0.826123, 0.826123,
      0.826123, 0.826123, 2.31547, 2.31547, 2.31547, 2.31547, 0.826123,
      0.826123, 0.826123, 0.826123, 2.31547, 2.31547, 2.31547, 2.31547, 1.39583,
      1.39583, 1.39583, 1.39583, 1.74576, 1.74576, 1.74576, 1.74576, 0.800552,
      0.800552, 0.800552, 0.800552, 2.34104, 2.34104, 2.34104, 2.34104,
      0.800552, 0.800552, 0.800552, 0.800552, 2.34104, 2.34104, 2.34104,
      2.34104, 1.51221, 1.51221, 1.51221, 1.51221, 1.62939, 1.62939, 1.62939,
      1.62939, 0.787113, 0.787113, 0.787113, 0.787113, 2.35448, 2.35448,
      2.35448, 2.35448, 0.787113, 0.787113, 0.787113, 0.787113, 2.35448,
      2.35448, 2.35448, 2.35448, 1.5708, 1.5708, 1.5708, 1.5708, 1.5708, 1.5708,
      1.5708, 1.5708, 1.47647, 1.47647, 1.66512, 1.66512, 0.0943258, 0.0943258,
      3.04727, 3.04727, 1.47647, 1.47647, 1.66512, 1.66512, 0.0943258,
      0.0943258, 3.04727, 3.04727, 1.5708, 1.5708, 1.5708, 1.5708, 1.5708,
      1.5708, 1.5708, 1.5708, 1.33968, 1.33968, 1.80191, 1.80191, 0.231115,
      0.231115, 2.91048, 2.91048, 1.33968, 1.33968, 1.80191, 1.80191, 0.231115,
      0.231115, 2.91048, 2.91048, 1.5708, 1.5708, 1.5708, 1.5708, 1.5708,
      1.5708, 1.5708, 1.5708, 1.18785, 1.18785, 1.95374, 1.95374, 0.382942,
      0.382942, 2.75865, 2.75865, 1.18785, 1.18785, 1.95374, 1.95374, 0.382942,
      0.382942, 2.75865, 2.75865, 1.5708, 1.5708, 1.5708, 1.5708, 1.5708,
      1.5708, 1.5708, 1.5708, 1.02904, 1.02904, 2.11256, 2.11256, 0.54176,
      0.54176, 2.59983, 2.59983, 1.02904, 1.02904, 2.11256, 2.11256, 0.54176,
      0.54176, 2.59983, 2.59983, 1.5708, 1.5708, 1.5708, 1.5708, 1.5708, 1.5708,
      1.5708, 1.5708, 0.866936, 0.866936, 2.27466, 2.27466, 0.70386, 0.70386,
      2.43773, 2.43773, 0.866936, 0.866936, 2.27466, 2.27466, 0.70386, 0.70386,
      2.43773, 2.43773, 0.803621, 0.803621, 0.803621, 0.803621, 2.33797,
      2.33797, 2.33797, 2.33797, 1.11664, 1.11664, 1.11664, 1.11664, 2.02495,
      2.02495, 2.02495, 2.02495, 0.803621, 0.803621, 0.803621, 0.803621,
      2.33797, 2.33797, 2.33797, 2.33797, 0.963375, 0.963375, 0.963375,
      0.963375, 2.17822, 2.17822, 2.17822, 2.17822, 1.11664, 1.11664, 1.11664,
      1.11664, 2.02495, 2.02495, 2.02495, 2.02495, 0.963375, 0.963375, 0.963375,
      0.963375, 2.17822, 2.17822, 2.17822, 2.17822, 0.703983, 0.703983,
      0.703983, 0.703983, 2.43761, 2.43761, 2.43761, 2.43761, 1.17462, 1.17462,
      1.17462, 1.17462, 1.96697, 1.96697, 1.96697, 1.96697, 0.703983, 0.703983,
      0.703983, 0.703983, 2.43761, 2.43761, 2.43761, 2.43761, 1.02436, 1.02436,
      1.02436, 1.02436, 2.11723, 2.11723, 2.11723, 2.11723, 1.17462, 1.17462,
      1.17462, 1.17462, 1.96697, 1.96697, 1.96697, 1.96697, 1.02436, 1.02436,
      1.02436, 1.02436, 2.11723, 2.11723, 2.11723, 2.11723, 0.606519, 0.606519,
      0.606519, 0.606519, 2.53507, 2.53507, 2.53507, 2.53507, 1.23429, 1.23429,
      1.23429, 1.23429, 1.90731, 1.90731, 1.90731, 1.90731, 0.606519, 0.606519,
      0.606519, 0.606519, 2.53507, 2.53507, 2.53507, 2.53507, 1.08758, 1.08758,
      1.08758, 1.08758, 2.05402, 2.05402, 2.05402, 2.05402, 1.23429, 1.23429,
      1.23429, 1.23429, 1.90731, 1.90731, 1.90731, 1.90731, 1.08758, 1.08758,
      1.08758, 1.08758, 2.05402, 2.05402, 2.05402, 2.05402, 0.511308, 0.511308,
      0.511308, 0.511308, 2.63028, 2.63028, 2.63028, 2.63028, 1.29476, 1.29476,
      1.29476, 1.29476, 1.84683, 1.84683, 1.84683, 1.84683, 0.511308, 0.511308,
      0.511308, 0.511308, 2.63028, 2.63028, 2.63028, 2.63028, 1.1523, 1.1523,
      1.1523, 1.1523, 1.9893, 1.9893, 1.9893, 1.9893, 1.29476, 1.29476, 1.29476,
      1.29476, 1.84683, 1.84683, 1.84683, 1.84683, 1.1523, 1.1523, 1.1523,
      1.1523, 1.9893, 1.9893, 1.9893, 1.9893, 0.418613, 0.418613, 0.418613,
      0.418613, 2.72298, 2.72298, 2.72298, 2.72298, 1.35518, 1.35518, 1.35518,
      1.35518, 1.78641, 1.78641, 1.78641, 1.78641, 0.418613, 0.418613, 0.418613,
      0.418613, 2.72298, 2.72298, 2.72298, 2.72298, 1.21788, 1.21788, 1.21788,
      1.21788, 1.92371, 1.92371, 1.92371, 1.92371, 1.35518, 1.35518, 1.35518,
      1.35518, 1.78641, 1.78641, 1.78641, 1.78641, 1.21788, 1.21788, 1.21788,
      1.21788, 1.92371, 1.92371, 1.92371, 1.92371, 0.328976, 0.328976, 0.328976,
      0.328976, 2.81262, 2.81262, 2.81262, 2.81262, 1.41457, 1.41457, 1.41457,
      1.41457, 1.72702, 1.72702, 1.72702, 1.72702, 0.328976, 0.328976, 0.328976,
      0.328976, 2.81262, 2.81262, 2.81262, 2.81262, 1.28373, 1.28373, 1.28373,
      1.28373, 1.85786, 1.85786, 1.85786, 1.85786, 1.41457, 1.41457, 1.41457,
      1.41457, 1.72702, 1.72702, 1.72702, 1.72702, 1.28373, 1.28373, 1.28373,
      1.28373, 1.85786, 1.85786, 1.85786, 1.85786, 0.243405, 0.243405, 0.243405,
      0.243405, 2.89819, 2.89819, 2.89819, 2.89819, 1.47171, 1.47171, 1.47171,
      1.47171, 1.66989, 1.66989, 1.66989, 1.66989, 0.243405, 0.243405, 0.243405,
      0.243405, 2.89819, 2.89819, 2.89819, 2.89819, 1.34922, 1.34922, 1.34922,
      1.34922, 1.79237, 1.79237, 1.79237, 1.79237, 1.47171, 1.47171, 1.47171,
      1.47171, 1.66989, 1.66989, 1.66989, 1.66989, 1.34922, 1.34922, 1.34922,
      1.34922, 1.79237, 1.79237, 1.79237, 1.79237, 0.163819, 0.163819, 0.163819,
      0.163819, 2.97777, 2.97777, 2.97777, 2.97777, 1.52479, 1.52479, 1.52479,
      1.52479, 1.6168, 1.6168, 1.6168, 1.6168, 0.163819, 0.163819, 0.163819,
      0.163819, 2.97777, 2.97777, 2.97777, 2.97777, 1.41368, 1.41368, 1.41368,
      1.41368, 1.72791, 1.72791, 1.72791, 1.72791, 1.52479, 1.52479, 1.52479,
      1.52479, 1.6168, 1.6168, 1.6168, 1.6168, 1.41368, 1.41368, 1.41368,
      1.41368, 1.72791, 1.72791, 1.72791, 1.72791, 0.762697, 0.762697, 0.762697,
      0.762697, 2.3789, 2.3789, 2.3789, 2.3789, 1.22636, 1.22636, 1.22636,
      1.22636, 1.91523, 1.91523, 1.91523, 1.91523, 0.762697, 0.762697, 0.762697,
      0.762697, 2.3789, 2.3789, 2.3789, 2.3789, 0.923871, 0.923871, 0.923871,
      0.923871, 2.21772, 2.21772, 2.21772, 2.21772, 1.22636, 1.22636, 1.22636,
      1.22636, 1.91523, 1.91523, 1.91523, 1.91523, 0.923871, 0.923871, 0.923871,
      0.923871, 2.21772, 2.21772, 2.21772, 2.21772, 0.665984, 0.665984,
      0.665984, 0.665984, 2.47561, 2.47561, 2.47561, 2.47561, 1.28468, 1.28468,
      1.28468, 1.28468, 1.85691, 1.85691, 1.85691, 1.85691, 0.665984, 0.665984,
      0.665984, 0.665984, 2.47561, 2.47561, 2.47561, 2.47561, 0.988907,
      0.988907, 0.988907, 0.988907, 2.15269, 2.15269, 2.15269, 2.15269, 1.28468,
      1.28468, 1.28468, 1.28468, 1.85691, 1.85691, 1.85691, 1.85691, 0.988907,
      0.988907, 0.988907, 0.988907, 2.15269, 2.15269, 2.15269, 2.15269,
      0.571752, 0.571752, 0.571752, 0.571752, 2.56984, 2.56984, 2.56984,
      2.56984, 1.34399, 1.34399, 1.34399, 1.34399, 1.7976, 1.7976, 1.7976,
      1.7976, 0.571752, 0.571752, 0.571752, 0.571752, 2.56984, 2.56984, 2.56984,
      2.56984, 1.05621, 1.05621, 1.05621, 1.05621, 2.08538, 2.08538, 2.08538,
      2.08538, 1.34399, 1.34399, 1.34399, 1.34399, 1.7976, 1.7976, 1.7976,
      1.7976, 1.05621, 1.05621, 1.05621, 1.05621, 2.08538, 2.08538, 2.08538,
      2.08538, 0.480286, 0.480286, 0.480286, 0.480286, 2.66131, 2.66131,
      2.66131, 2.66131, 1.40339, 1.40339, 1.40339, 1.40339, 1.7382, 1.7382,
      1.7382, 1.7382, 0.480286, 0.480286, 0.480286, 0.480286, 2.66131, 2.66131,
      2.66131, 2.66131, 1.12526, 1.12526, 1.12526, 1.12526, 2.01633, 2.01633,
      2.01633, 2.01633, 1.40339, 1.40339, 1.40339, 1.40339, 1.7382, 1.7382,
      1.7382, 1.7382, 1.12526, 1.12526, 1.12526, 1.12526, 2.01633, 2.01633,
      2.01633, 2.01633, 0.392169, 0.392169, 0.392169, 0.392169, 2.74942,
      2.74942, 2.74942, 2.74942, 1.46188, 1.46188, 1.46188, 1.46188, 1.67971,
      1.67971, 1.67971, 1.67971, 0.392169, 0.392169, 0.392169, 0.392169,
      2.74942, 2.74942, 2.74942, 2.74942, 1.19565, 1.19565, 1.19565, 1.19565,
      1.94594, 1.94594, 1.94594, 1.94594, 1.46188, 1.46188, 1.46188, 1.46188,
      1.67971, 1.67971, 1.67971, 1.67971, 1.19565, 1.19565, 1.19565, 1.19565,
      1.94594, 1.94594, 1.94594, 1.94594, 0.308465, 0.308465, 0.308465,
      0.308465, 2.83313, 2.83313, 2.83313, 2.83313, 1.51825, 1.51825, 1.51825,
      1.51825, 1.62334, 1.62334, 1.62334, 1.62334, 0.308465, 0.308465, 0.308465,
      0.308465, 2.83313, 2.83313, 2.83313, 2.83313, 1.26713, 1.26713, 1.26713,
      1.26713, 1.87446, 1.87446, 1.87446, 1.87446, 1.51825, 1.51825, 1.51825,
      1.51825, 1.62334, 1.62334, 1.62334, 1.62334, 1.26713, 1.26713, 1.26713,
      1.26713, 1.87446, 1.87446, 1.87446, 1.87446, 0.731046, 0.731046, 0.731046,
      0.731046, 2.41055, 2.41055, 2.41055, 2.41055, 1.33897, 1.33897, 1.33897,
      1.33897, 1.80262, 1.80262, 1.80262, 1.80262, 0.731046, 0.731046, 0.731046,
      0.731046, 2.41055, 2.41055, 2.41055, 2.41055, 0.893264, 0.893264,
      0.893264, 0.893264, 2.24833, 2.24833, 2.24833, 2.24833, 1.33897, 1.33897,
      1.33897, 1.33897, 1.80262, 1.80262, 1.80262, 1.80262, 0.893264, 0.893264,
      0.893264, 0.893264, 2.24833, 2.24833, 2.24833, 2.24833, 0.638729,
      0.638729, 0.638729, 0.638729, 2.50286, 2.50286, 2.50286, 2.50286, 1.39762,
      1.39762, 1.39762, 1.39762, 1.74397, 1.74397, 1.74397, 1.74397, 0.638729,
      0.638729, 0.638729, 0.638729, 2.50286, 2.50286, 2.50286, 2.50286,
      0.963399, 0.963399, 0.963399, 0.963399, 2.17819, 2.17819, 2.17819,
      2.17819, 1.39762, 1.39762, 1.39762, 1.39762, 1.74397, 1.74397, 1.74397,
      1.74397, 0.963399, 0.963399, 0.963399, 0.963399, 2.17819, 2.17819,
      2.17819, 2.17819, 0.549471, 0.549471, 0.549471, 0.549471, 2.59212,
      2.59212, 2.59212, 2.59212, 1.45652, 1.45652, 1.45652, 1.45652, 1.68507,
      1.68507, 1.68507, 1.68507, 0.549471, 0.549471, 0.549471, 0.549471,
      2.59212, 2.59212, 2.59212, 2.59212, 1.03603, 1.03603, 1.03603, 1.03603,
      2.10556, 2.10556, 2.10556, 2.10556, 1.45652, 1.45652, 1.45652, 1.45652,
      1.68507, 1.68507, 1.68507, 1.68507, 1.03603, 1.03603, 1.03603, 1.03603,
      2.10556, 2.10556, 2.10556, 2.10556, 0.463863, 0.463863, 0.463863,
      0.463863, 2.67773, 2.67773, 2.67773, 2.67773, 1.51465, 1.51465, 1.51465,
      1.51465, 1.62694, 1.62694, 1.62694, 1.62694, 0.463863, 0.463863, 0.463863,
      0.463863, 2.67773, 2.67773, 2.67773, 2.67773, 1.11088, 1.11088, 1.11088,
      1.11088, 2.03071, 2.03071, 2.03071, 2.03071, 1.51465, 1.51465, 1.51465,
      1.51465, 1.62694, 1.62694, 1.62694, 1.62694, 1.11088, 1.11088, 1.11088,
      1.11088, 2.03071, 2.03071, 2.03071, 2.03071, 0.710836, 0.710836, 0.710836,
      0.710836, 2.43076, 2.43076, 2.43076, 2.43076, 1.45411, 1.45411, 1.45411,
      1.45411, 1.68748, 1.68748, 1.68748, 1.68748, 0.710836, 0.710836, 0.710836,
      0.710836, 2.43076, 2.43076, 2.43076, 2.43076, 0.873695, 0.873695,
      0.873695, 0.873695, 2.2679, 2.2679, 2.2679, 2.2679, 1.45411, 1.45411,
      1.45411, 1.45411, 1.68748, 1.68748, 1.68748, 1.68748, 0.873695, 0.873695,
      0.873695, 0.873695, 2.2679, 2.2679, 2.2679, 2.2679, 0.624391, 0.624391,
      0.624391, 0.624391, 2.5172, 2.5172, 2.5172, 2.5172, 1.51279, 1.51279,
      1.51279, 1.51279, 1.6288, 1.6288, 1.6288, 1.6288, 0.624391, 0.624391,
      0.624391, 0.624391, 2.5172, 2.5172, 2.5172, 2.5172, 0.949952, 0.949952,
      0.949952, 0.949952, 2.19164, 2.19164, 2.19164, 2.19164, 1.51279, 1.51279,
      1.51279, 1.51279, 1.6288, 1.6288, 1.6288, 1.6288, 0.949952, 0.949952,
      0.949952, 0.949952, 2.19164, 2.19164, 2.19164, 2.19164;

  Eigen::VectorXd Ge_theta_ref = Eigen::VectorXd::Zero(1454);
  Ge_theta_ref << 0, 3.14159, 1.5708, -1.5708, 0, 0, 0.785398, 2.35619,
      -0.785398, -2.35619, 0.785398, 2.35619, -0.785398, -2.35619, 0.785398,
      2.35619, -0.785398, -2.35619, 0.785398, 2.35619, -0.785398, -2.35619,
      1.53848, 1.60311, -1.53848, -1.60311, 1.53848, 1.60311, -1.53848,
      -1.60311, 0.0323154, 3.10928, -0.0323154, -3.10928, 0.0323154, 3.10928,
      -0.0323154, -3.10928, 0.785398, 2.35619, -0.785398, -2.35619, 0.785398,
      2.35619, -0.785398, -2.35619, 1.49008, 1.65151, -1.49008, -1.65151,
      1.49008, 1.65151, -1.49008, -1.65151, 0.0807158, 3.06088, -0.0807158,
      -3.06088, 0.0807158, 3.06088, -0.0807158, -3.06088, 0.785398, 2.35619,
      -0.785398, -2.35619, 0.785398, 2.35619, -0.785398, -2.35619, 1.43368,
      1.70791, -1.43368, -1.70791, 1.43368, 1.70791, -1.43368, -1.70791,
      0.137118, 3.00448, -0.137118, -3.00448, 0.137118, 3.00448, -0.137118,
      -3.00448, 0.785398, 2.35619, -0.785398, -2.35619, 0.785398, 2.35619,
      -0.785398, -2.35619, 1.37184, 1.76975, -1.37184, -1.76975, 1.37184,
      1.76975, -1.37184, -1.76975, 0.198957, 2.94264, -0.198957, -2.94264,
      0.198957, 2.94264, -0.198957, -2.94264, 0.785398, 2.35619, -0.785398,
      -2.35619, 0.785398, 2.35619, -0.785398, -2.35619, 1.30537, 1.83622,
      -1.30537, -1.83622, 1.30537, 1.83622, -1.30537, -1.83622, 0.265425,
      2.87617, -0.265425, -2.87617, 0.265425, 2.87617, -0.265425, -2.87617,
      0.785398, 2.35619, -0.785398, -2.35619, 0.785398, 2.35619, -0.785398,
      -2.35619, 1.23431, 1.90728, -1.23431, -1.90728, 1.23431, 1.90728,
      -1.23431, -1.90728, 0.336486, 2.80511, -0.336486, -2.80511, 0.336486,
      2.80511, -0.336486, -2.80511, 0.785398, 2.35619, -0.785398, -2.35619,
      0.785398, 2.35619, -0.785398, -2.35619, 1.15824, 1.98336, -1.15824,
      -1.98336, 1.15824, 1.98336, -1.15824, -1.98336, 0.412561, 2.72903,
      -0.412561, -2.72903, 0.412561, 2.72903, -0.412561, -2.72903, 0.785398,
      2.35619, -0.785398, -2.35619, 0.785398, 2.35619, -0.785398, -2.35619,
      1.07641, 2.06518, -1.07641, -2.06518, 1.07641, 2.06518, -1.07641,
      -2.06518, 0.494385, 2.64721, -0.494385, -2.64721, 0.494385, 2.64721,
      -0.494385, -2.64721, 0.785398, 2.35619, -0.785398, -2.35619, 0.785398,
      2.35619, -0.785398, -2.35619, 0.98784, 2.15375, -0.98784, -2.15375,
      0.98784, 2.15375, -0.98784, -2.15375, 0.582956, 2.55864, -0.582956,
      -2.55864, 0.582956, 2.55864, -0.582956, -2.55864, 0.785398, 2.35619,
      -0.785398, -2.35619, 0.785398, 2.35619, -0.785398, -2.35619, 0.891301,
      2.25029, -0.891301, -2.25029, 0.891301, 2.25029, -0.891301, -2.25029,
      0.679496, 2.4621, -0.679496, -2.4621, 0.679496, 2.4621, -0.679496,
      -2.4621, 0.785398, 2.35619, -0.785398, -2.35619, 0.785398, 2.35619,
      -0.785398, -2.35619, 0.668678, 2.47291, -0.668678, -2.47291, 0.668678,
      2.47291, -0.668678, -2.47291, 0.902118, 2.23947, -0.902118, -2.23947,
      0.902118, 2.23947, -0.902118, -2.23947, 0.785398, 2.35619, -0.785398,
      -2.35619, 0.785398, 2.35619, -0.785398, -2.35619, 0.539871, 2.60172,
      -0.539871, -2.60172, 0.539871, 2.60172, -0.539871, -2.60172, 1.03093,
      2.11067, -1.03093, -2.11067, 1.03093, 2.11067, -1.03093, -2.11067,
      0.785398, 2.35619, -0.785398, -2.35619, 0.785398, 2.35619, -0.785398,
      -2.35619, 0.398367, 2.74323, -0.398367, -2.74323, 0.398367, 2.74323,
      -0.398367, -2.74323, 1.17243, 1.96916, -1.17243, -1.96916, 1.17243,
      1.96916, -1.17243, -1.96916, 0.785398, 2.35619, -0.785398, -2.35619,
      0.785398, 2.35619, -0.785398, -2.35619, 0.244975, 2.89662, -0.244975,
      -2.89662, 0.244975, 2.89662, -0.244975, -2.89662, 1.32582, 1.81577,
      -1.32582, -1.81577, 1.32582, 1.81577, -1.32582, -1.81577, 0.785398,
      2.35619, -0.785398, -2.35619, 0.785398, 2.35619, -0.785398, -2.35619,
      0.0827626, 3.05883, -0.0827626, -3.05883, 0.0827626, 3.05883, -0.0827626,
      -3.05883, 1.48803, 1.65356, -1.48803, -1.65356, 1.48803, 1.65356,
      -1.48803, -1.65356, 0.0943258, 3.04727, -0.0943258, -3.04727, 1.47647,
      1.66512, -1.47647, -1.66512, 0, 3.14159, 0, 3.14159, 0, 3.14159, 0,
      3.14159, 1.5708, -1.5708, 1.5708, -1.5708, 1.5708, -1.5708, 1.5708,
      -1.5708, 0.231115, 2.91048, -0.231115, -2.91048, 1.33968, 1.80191,
      -1.33968, -1.80191, 0, 3.14159, 0, 3.14159, 0, 3.14159, 0, 3.14159,
      1.5708, -1.5708, 1.5708, -1.5708, 1.5708, -1.5708, 1.5708, -1.5708,
      0.382942, 2.75865, -0.382942, -2.75865, 1.18785, 1.95374, -1.18785,
      -1.95374, 0, 3.14159, 0, 3.14159, 0, 3.14159, 0, 3.14159, 1.5708, -1.5708,
      1.5708, -1.5708, 1.5708, -1.5708, 1.5708, -1.5708, 0.54176, 2.59983,
      -0.54176, -2.59983, 1.02904, 2.11256, -1.02904, -2.11256, 0, 3.14159, 0,
      3.14159, 0, 3.14159, 0, 3.14159, 1.5708, -1.5708, 1.5708, -1.5708, 1.5708,
      -1.5708, 1.5708, -1.5708, 0.70386, 2.43773, -0.70386, -2.43773, 0.866936,
      2.27466, -0.866936, -2.27466, 0, 3.14159, 0, 3.14159, 0, 3.14159, 0,
      3.14159, 1.5708, -1.5708, 1.5708, -1.5708, 1.5708, -1.5708, 1.5708,
      -1.5708, 0.655324, 2.48627, -0.655324, -2.48627, 0.655324, 2.48627,
      -0.655324, -2.48627, 0.882614, 2.25898, -0.882614, -2.25898, 0.882614,
      2.25898, -0.882614, -2.25898, 0.915472, 2.22612, -0.915472, -2.22612,
      0.915472, 2.22612, -0.915472, -2.22612, 1.00715, 2.13444, -1.00715,
      -2.13444, 1.00715, 2.13444, -1.00715, -2.13444, 0.688183, 2.45341,
      -0.688183, -2.45341, 0.688183, 2.45341, -0.688183, -2.45341, 0.563647,
      2.57795, -0.563647, -2.57795, 0.563647, 2.57795, -0.563647, -2.57795,
      0.63875, 2.50284, -0.63875, -2.50284, 0.63875, 2.50284, -0.63875,
      -2.50284, 0.972452, 2.16914, -0.972452, -2.16914, 0.972452, 2.16914,
      -0.972452, -2.16914, 0.932046, 2.20955, -0.932046, -2.20955, 0.932046,
      2.20955, -0.932046, -2.20955, 1.10217, 2.03942, -1.10217, -2.03942,
      1.10217, 2.03942, -1.10217, -2.03942, 0.598344, 2.54325, -0.598344,
      -2.54325, 0.598344, 2.54325, -0.598344, -2.54325, 0.468626, 2.67297,
      -0.468626, -2.67297, 0.468626, 2.67297, -0.468626, -2.67297, 0.61784,
      2.52375, -0.61784, -2.52375, 0.61784, 2.52375, -0.61784, -2.52375,
      1.05613, 2.08546, -1.05613, -2.08546, 1.05613, 2.08546, -1.05613,
      -2.08546, 0.952956, 2.18864, -0.952956, -2.18864, 0.952956, 2.18864,
      -0.952956, -2.18864, 1.18868, 1.95292, -1.18868, -1.95292, 1.18868,
      1.95292, -1.18868, -1.95292, 0.514664, 2.62693, -0.514664, -2.62693,
      0.514664, 2.62693, -0.514664, -2.62693, 0.38212, 2.75947, -0.38212,
      -2.75947, 0.38212, 2.75947, -0.38212, -2.75947, 0.59075, 2.55084,
      -0.59075, -2.55084, 0.59075, 2.55084, -0.59075, -2.55084, 1.13473,
      2.00687, -1.13473, -2.00687, 1.13473, 2.00687, -1.13473, -2.00687,
      0.980047, 2.16155, -0.980047, -2.16155, 0.980047, 2.16155, -0.980047,
      -2.16155, 1.2679, 1.87369, -1.2679, -1.87369, 1.2679, 1.87369, -1.2679,
      -1.87369, 0.436069, 2.70552, -0.436069, -2.70552, 0.436069, 2.70552,
      -0.436069, -2.70552, 0.302895, 2.8387, -0.302895, -2.8387, 0.302895,
      2.8387, -0.302895, -2.8387, 0.554282, 2.58731, -0.554282, -2.58731,
      0.554282, 2.58731, -0.554282, -2.58731, 1.20914, 1.93245, -1.20914,
      -1.93245, 1.20914, 1.93245, -1.20914, -1.93245, 1.01651, 2.12508,
      -1.01651, -2.12508, 1.01651, 2.12508, -1.01651, -2.12508, 1.34077,
      1.80082, -1.34077, -1.80082, 1.34077, 1.80082, -1.34077, -1.80082,
      0.361659, 2.77993, -0.361659, -2.77993, 0.361659, 2.77993, -0.361659,
      -2.77993, 0.230026, 2.91157, -0.230026, -2.91157, 0.230026, 2.91157,
      -0.230026, -2.91157, 0.502478, 2.63911, -0.502478, -2.63911, 0.502478,
      2.63911, -0.502478, -2.63911, 1.28009, 1.8615, -1.28009, -1.8615, 1.28009,
      1.8615, -1.28009, -1.8615, 1.06832, 2.07327, -1.06832, -2.07327, 1.06832,
      2.07327, -1.06832, -2.07327, 1.40785, 1.73375, -1.40785, -1.73375,
      1.40785, 1.73375, -1.40785, -1.73375, 0.290708, 2.85088, -0.290708,
      -2.85088, 0.290708, 2.85088, -0.290708, -2.85088, 0.162951, 2.97864,
      -0.162951, -2.97864, 0.162951, 2.97864, -0.162951, -2.97864, 0.422979,
      2.71861, -0.422979, -2.71861, 0.422979, 2.71861, -0.422979, -2.71861,
      1.34811, 1.79348, -1.34811, -1.79348, 1.34811, 1.79348, -1.34811,
      -1.79348, 1.14782, 1.99378, -1.14782, -1.99378, 1.14782, 1.99378,
      -1.14782, -1.99378, 1.46921, 1.67238, -1.46921, -1.67238, 1.46921,
      1.67238, -1.46921, -1.67238, 0.222688, 2.91891, -0.222688, -2.91891,
      0.222688, 2.91891, -0.222688, -2.91891, 0.101583, 3.04001, -0.101583,
      -3.04001, 0.101583, 3.04001, -0.101583, -3.04001, 0.285851, 2.85574,
      -0.285851, -2.85574, 0.285851, 2.85574, -0.285851, -2.85574, 1.41351,
      1.72808, -1.41351, -1.72808, 1.41351, 1.72808, -1.41351, -1.72808,
      1.28495, 1.85665, -1.28495, -1.85665, 1.28495, 1.85665, -1.28495,
      -1.85665, 1.52422, 1.61737, -1.52422, -1.61737, 1.52422, 1.61737,
      -1.52422, -1.61737, 0.157283, 2.98431, -0.157283, -2.98431, 0.157283,
      2.98431, -0.157283, -2.98431, 0.0465768, 3.09502, -0.0465768, -3.09502,
      0.0465768, 3.09502, -0.0465768, -3.09502, 0.510653, 2.63094, -0.510653,
      -2.63094, 0.510653, 2.63094, -0.510653, -2.63094, 0.875849, 2.26574,
      -0.875849, -2.26574, 0.875849, 2.26574, -0.875849, -2.26574, 1.06014,
      2.08145, -1.06014, -2.08145, 1.06014, 2.08145, -1.06014, -2.08145,
      1.13386, 2.00773, -1.13386, -2.00773, 1.13386, 2.00773, -1.13386,
      -2.00773, 0.694947, 2.44665, -0.694947, -2.44665, 0.694947, 2.44665,
      -0.694947, -2.44665, 0.436938, 2.70465, -0.436938, -2.70465, 0.436938,
      2.70465, -0.436938, -2.70465, 0.474402, 2.66719, -0.474402, -2.66719,
      0.474402, 2.66719, -0.474402, -2.66719, 0.960765, 2.18083, -0.960765,
      -2.18083, 0.960765, 2.18083, -0.960765, -2.18083, 1.09639, 2.0452,
      -1.09639, -2.0452, 1.09639, 2.0452, -1.09639, -2.0452, 1.22619, 1.9154,
      -1.22619, -1.9154, 1.22619, 1.9154, -1.22619, -1.9154, 0.610031, 2.53156,
      -0.610031, -2.53156, 0.610031, 2.53156, -0.610031, -2.53156, 0.344608,
      2.79698, -0.344608, -2.79698, 0.344608, 2.79698, -0.344608, -2.79698,
      0.428561, 2.71303, -0.428561, -2.71303, 0.428561, 2.71303, -0.428561,
      -2.71303, 1.04129, 2.1003, -1.04129, -2.1003, 1.04129, 2.1003, -1.04129,
      -2.1003, 1.14224, 1.99936, -1.14224, -1.99936, 1.14224, 1.99936, -1.14224,
      -1.99936, 1.30952, 1.83207, -1.30952, -1.83207, 1.30952, 1.83207,
      -1.30952, -1.83207, 0.529505, 2.61209, -0.529505, -2.61209, 0.529505,
      2.61209, -0.529505, -2.61209, 0.261278, 2.88031, -0.261278, -2.88031,
      0.261278, 2.88031, -0.261278, -2.88031, 0.368942, 2.77265, -0.368942,
      -2.77265, 0.368942, 2.77265, -0.368942, -2.77265, 1.11848, 2.02311,
      -1.11848, -2.02311, 1.11848, 2.02311, -1.11848, -2.02311, 1.20185,
      1.93974, -1.20185, -1.93974, 1.20185, 1.93974, -1.20185, -1.93974,
      1.38508, 1.75651, -1.38508, -1.75651, 1.38508, 1.75651, -1.38508,
      -1.75651, 0.452318, 2.68927, -0.452318, -2.68927, 0.452318, 2.68927,
      -0.452318, -2.68927, 0.185714, 2.95588, -0.185714, -2.95588, 0.185714,
      2.95588, -0.185714, -2.95588, 0.288382, 2.85321, -0.288382, -2.85321,
      0.288382, 2.85321, -0.288382, -2.85321, 1.1933, 1.94829, -1.1933,
      -1.94829, 1.1933, 1.94829, -1.1933, -1.94829, 1.28241, 1.85918, -1.28241,
      -1.85918, 1.28241, 1.85918, -1.28241, -1.85918, 1.45371, 1.68788,
      -1.45371, -1.68788, 1.45371, 1.68788, -1.45371, -1.68788, 0.377497,
      2.7641, -0.377497, -2.7641, 0.377497, 2.7641, -0.377497, -2.7641,
      0.117088, 3.0245, -0.117088, -3.0245, 0.117088, 3.0245, -0.117088,
      -3.0245, 0.173867, 2.96773, -0.173867, -2.96773, 0.173867, 2.96773,
      -0.173867, -2.96773, 1.2667, 1.87489, -1.2667, -1.87489, 1.2667, 1.87489,
      -1.2667, -1.87489, 1.39693, 1.74466, -1.39693, -1.74466, 1.39693, 1.74466,
      -1.39693, -1.74466, 1.51573, 1.62586, -1.51573, -1.62586, 1.51573,
      1.62586, -1.51573, -1.62586, 0.304097, 2.8375, -0.304097, -2.8375,
      0.304097, 2.8375, -0.304097, -2.8375, 0.0550659, 3.08653, -0.0550659,
      -3.08653, 0.0550659, 3.08653, -0.0550659, -3.08653, 0.351303, 2.79029,
      -0.351303, -2.79029, 0.351303, 2.79029, -0.351303, -2.79029, 0.870947,
      2.27065, -0.870947, -2.27065, 0.870947, 2.27065, -0.870947, -2.27065,
      1.21949, 1.9221, -1.21949, -1.9221, 1.21949, 1.9221, -1.21949, -1.9221,
      1.27146, 1.87013, -1.27146, -1.87013, 1.27146, 1.87013, -1.27146,
      -1.87013, 0.69985, 2.44174, -0.69985, -2.44174, 0.69985, 2.44174,
      -0.69985, -2.44174, 0.299336, 2.84226, -0.299336, -2.84226, 0.299336,
      2.84226, -0.299336, -2.84226, 0.293205, 2.84839, -0.293205, -2.84839,
      0.293205, 2.84839, -0.293205, -2.84839, 0.952806, 2.18879, -0.952806,
      -2.18879, 0.952806, 2.18879, -0.952806, -2.18879, 1.27759, 1.864,
      -1.27759, -1.864, 1.27759, 1.864, -1.27759, -1.864, 1.35938, 1.78221,
      -1.35938, -1.78221, 1.35938, 1.78221, -1.35938, -1.78221, 0.61799, 2.5236,
      -0.61799, -2.5236, 0.61799, 2.5236, -0.61799, -2.5236, 0.211412, 2.93018,
      -0.211412, -2.93018, 0.211412, 2.93018, -0.211412, -2.93018, 0.220111,
      2.92148, -0.220111, -2.92148, 0.220111, 2.92148, -0.220111, -2.92148,
      1.03214, 2.10945, -1.03214, -2.10945, 1.03214, 2.10945, -1.03214,
      -2.10945, 1.35069, 1.79091, -1.35069, -1.79091, 1.35069, 1.79091,
      -1.35069, -1.79091, 1.43788, 1.70371, -1.43788, -1.70371, 1.43788,
      1.70371, -1.43788, -1.70371, 0.538655, 2.60294, -0.538655, -2.60294,
      0.538655, 2.60294, -0.538655, -2.60294, 0.132917, 3.00868, -0.132917,
      -3.00868, 0.132917, 3.00868, -0.132917, -3.00868, 0.125755, 3.01584,
      -0.125755, -3.01584, 0.125755, 3.01584, -0.125755, -3.01584, 1.1101,
      2.03149, -1.1101, -2.03149, 1.1101, 2.03149, -1.1101, -2.03149, 1.44504,
      1.69655, -1.44504, -1.69655, 1.44504, 1.69655, -1.44504, -1.69655,
      1.50813, 1.63346, -1.50813, -1.63346, 1.50813, 1.63346, -1.50813,
      -1.63346, 0.460698, 2.68089, -0.460698, -2.68089, 0.460698, 2.68089,
      -0.460698, -2.68089, 0.0626633, 3.07893, -0.0626633, -3.07893, 0.0626633,
      3.07893, -0.0626633, -3.07893, 0.179387, 2.96221, -0.179387, -2.96221,
      0.179387, 2.96221, -0.179387, -2.96221, 0.867949, 2.27364, -0.867949,
      -2.27364, 0.867949, 2.27364, -0.867949, -2.27364, 1.39141, 1.75018,
      -1.39141, -1.75018, 1.39141, 1.75018, -1.39141, -1.75018, 1.41837,
      1.72323, -1.41837, -1.72323, 1.41837, 1.72323, -1.41837, -1.72323,
      0.702848, 2.43874, -0.702848, -2.43874, 0.702848, 2.43874, -0.702848,
      -2.43874, 0.152431, 2.98916, -0.152431, -2.98916, 0.152431, 2.98916,
      -0.152431, -2.98916, 0.0993346, 3.04226, -0.0993346, -3.04226, 0.0993346,
      3.04226, -0.0993346, -3.04226, 0.948747, 2.19285, -0.948747, -2.19285,
      0.948747, 2.19285, -0.948747, -2.19285, 1.47146, 1.67013, -1.47146,
      -1.67013, 1.47146, 1.67013, -1.47146, -1.67013, 1.49946, 1.64213,
      -1.49946, -1.64213, 1.49946, 1.64213, -1.49946, -1.64213, 0.622049,
      2.51954, -0.622049, -2.51954, 0.622049, 2.51954, -0.622049, -2.51954,
      0.0713376, 3.07026, -0.0713376, -3.07026, 0.0713376, 3.07026, -0.0713376,
      -3.07026;

  Eigen::VectorXd Ge_weight_ref = Eigen::VectorXd::Zero(1454);
  Ge_weight_ref << 0.000977307, 0.000977307, 0.000977307, 0.000977307,
      0.000977307, 0.000977307, 0.00949722, 0.00949722, 0.00949722, 0.00949722,
      0.00949722, 0.00949722, 0.00949722, 0.00949722, 0.0035709, 0.0035709,
      0.0035709, 0.0035709, 0.0035709, 0.0035709, 0.0035709, 0.0035709,
      0.0035709, 0.0035709, 0.0035709, 0.0035709, 0.0035709, 0.0035709,
      0.0035709, 0.0035709, 0.0035709, 0.0035709, 0.0035709, 0.0035709,
      0.0035709, 0.0035709, 0.0035709, 0.0035709, 0.00549706, 0.00549706,
      0.00549706, 0.00549706, 0.00549706, 0.00549706, 0.00549706, 0.00549706,
      0.00549706, 0.00549706, 0.00549706, 0.00549706, 0.00549706, 0.00549706,
      0.00549706, 0.00549706, 0.00549706, 0.00549706, 0.00549706, 0.00549706,
      0.00549706, 0.00549706, 0.00549706, 0.00549706, 0.00680742, 0.00680742,
      0.00680742, 0.00680742, 0.00680742, 0.00680742, 0.00680742, 0.00680742,
      0.00680742, 0.00680742, 0.00680742, 0.00680742, 0.00680742, 0.00680742,
      0.00680742, 0.00680742, 0.00680742, 0.00680742, 0.00680742, 0.00680742,
      0.00680742, 0.00680742, 0.00680742, 0.00680742, 0.00772581, 0.00772581,
      0.00772581, 0.00772581, 0.00772581, 0.00772581, 0.00772581, 0.00772581,
      0.00772581, 0.00772581, 0.00772581, 0.00772581, 0.00772581, 0.00772581,
      0.00772581, 0.00772581, 0.00772581, 0.00772581, 0.00772581, 0.00772581,
      0.00772581, 0.00772581, 0.00772581, 0.00772581, 0.00837473, 0.00837473,
      0.00837473, 0.00837473, 0.00837473, 0.00837473, 0.00837473, 0.00837473,
      0.00837473, 0.00837473, 0.00837473, 0.00837473, 0.00837473, 0.00837473,
      0.00837473, 0.00837473, 0.00837473, 0.00837473, 0.00837473, 0.00837473,
      0.00837473, 0.00837473, 0.00837473, 0.00837473, 0.00882792, 0.00882792,
      0.00882792, 0.00882792, 0.00882792, 0.00882792, 0.00882792, 0.00882792,
      0.00882792, 0.00882792, 0.00882792, 0.00882792, 0.00882792, 0.00882792,
      0.00882792, 0.00882792, 0.00882792, 0.00882792, 0.00882792, 0.00882792,
      0.00882792, 0.00882792, 0.00882792, 0.00882792, 0.00913388, 0.00913388,
      0.00913388, 0.00913388, 0.00913388, 0.00913388, 0.00913388, 0.00913388,
      0.00913388, 0.00913388, 0.00913388, 0.00913388, 0.00913388, 0.00913388,
      0.00913388, 0.00913388, 0.00913388, 0.00913388, 0.00913388, 0.00913388,
      0.00913388, 0.00913388, 0.00913388, 0.00913388, 0.00932756, 0.00932756,
      0.00932756, 0.00932756, 0.00932756, 0.00932756, 0.00932756, 0.00932756,
      0.00932756, 0.00932756, 0.00932756, 0.00932756, 0.00932756, 0.00932756,
      0.00932756, 0.00932756, 0.00932756, 0.00932756, 0.00932756, 0.00932756,
      0.00932756, 0.00932756, 0.00932756, 0.00932756, 0.00943677, 0.00943677,
      0.00943677, 0.00943677, 0.00943677, 0.00943677, 0.00943677, 0.00943677,
      0.00943677, 0.00943677, 0.00943677, 0.00943677, 0.00943677, 0.00943677,
      0.00943677, 0.00943677, 0.00943677, 0.00943677, 0.00943677, 0.00943677,
      0.00943677, 0.00943677, 0.00943677, 0.00943677, 0.00948577, 0.00948577,
      0.00948577, 0.00948577, 0.00948577, 0.00948577, 0.00948577, 0.00948577,
      0.00948577, 0.00948577, 0.00948577, 0.00948577, 0.00948577, 0.00948577,
      0.00948577, 0.00948577, 0.00948577, 0.00948577, 0.00948577, 0.00948577,
      0.00948577, 0.00948577, 0.00948577, 0.00948577, 0.00949275, 0.00949275,
      0.00949275, 0.00949275, 0.00949275, 0.00949275, 0.00949275, 0.00949275,
      0.00949275, 0.00949275, 0.00949275, 0.00949275, 0.00949275, 0.00949275,
      0.00949275, 0.00949275, 0.00949275, 0.00949275, 0.00949275, 0.00949275,
      0.00949275, 0.00949275, 0.00949275, 0.00949275, 0.00949156, 0.00949156,
      0.00949156, 0.00949156, 0.00949156, 0.00949156, 0.00949156, 0.00949156,
      0.00949156, 0.00949156, 0.00949156, 0.00949156, 0.00949156, 0.00949156,
      0.00949156, 0.00949156, 0.00949156, 0.00949156, 0.00949156, 0.00949156,
      0.00949156, 0.00949156, 0.00949156, 0.00949156, 0.00950617, 0.00950617,
      0.00950617, 0.00950617, 0.00950617, 0.00950617, 0.00950617, 0.00950617,
      0.00950617, 0.00950617, 0.00950617, 0.00950617, 0.00950617, 0.00950617,
      0.00950617, 0.00950617, 0.00950617, 0.00950617, 0.00950617, 0.00950617,
      0.00950617, 0.00950617, 0.00950617, 0.00950617, 0.00953535, 0.00953535,
      0.00953535, 0.00953535, 0.00953535, 0.00953535, 0.00953535, 0.00953535,
      0.00953535, 0.00953535, 0.00953535, 0.00953535, 0.00953535, 0.00953535,
      0.00953535, 0.00953535, 0.00953535, 0.00953535, 0.00953535, 0.00953535,
      0.00953535, 0.00953535, 0.00953535, 0.00953535, 0.00956082, 0.00956082,
      0.00956082, 0.00956082, 0.00956082, 0.00956082, 0.00956082, 0.00956082,
      0.00956082, 0.00956082, 0.00956082, 0.00956082, 0.00956082, 0.00956082,
      0.00956082, 0.00956082, 0.00956082, 0.00956082, 0.00956082, 0.00956082,
      0.00956082, 0.00956082, 0.00956082, 0.00956082, 0.00505379, 0.00505379,
      0.00505379, 0.00505379, 0.00505379, 0.00505379, 0.00505379, 0.00505379,
      0.00505379, 0.00505379, 0.00505379, 0.00505379, 0.00505379, 0.00505379,
      0.00505379, 0.00505379, 0.00505379, 0.00505379, 0.00505379, 0.00505379,
      0.00505379, 0.00505379, 0.00505379, 0.00505379, 0.00729462, 0.00729462,
      0.00729462, 0.00729462, 0.00729462, 0.00729462, 0.00729462, 0.00729462,
      0.00729462, 0.00729462, 0.00729462, 0.00729462, 0.00729462, 0.00729462,
      0.00729462, 0.00729462, 0.00729462, 0.00729462, 0.00729462, 0.00729462,
      0.00729462, 0.00729462, 0.00729462, 0.00729462, 0.00853527, 0.00853527,
      0.00853527, 0.00853527, 0.00853527, 0.00853527, 0.00853527, 0.00853527,
      0.00853527, 0.00853527, 0.00853527, 0.00853527, 0.00853527, 0.00853527,
      0.00853527, 0.00853527, 0.00853527, 0.00853527, 0.00853527, 0.00853527,
      0.00853527, 0.00853527, 0.00853527, 0.00853527, 0.00921962, 0.00921962,
      0.00921962, 0.00921962, 0.00921962, 0.00921962, 0.00921962, 0.00921962,
      0.00921962, 0.00921962, 0.00921962, 0.00921962, 0.00921962, 0.00921962,
      0.00921962, 0.00921962, 0.00921962, 0.00921962, 0.00921962, 0.00921962,
      0.00921962, 0.00921962, 0.00921962, 0.00921962, 0.00952765, 0.00952765,
      0.00952765, 0.00952765, 0.00952765, 0.00952765, 0.00952765, 0.00952765,
      0.00952765, 0.00952765, 0.00952765, 0.00952765, 0.00952765, 0.00952765,
      0.00952765, 0.00952765, 0.00952765, 0.00952765, 0.00952765, 0.00952765,
      0.00952765, 0.00952765, 0.00952765, 0.00952765, 0.00947285, 0.00947285,
      0.00947285, 0.00947285, 0.00947285, 0.00947285, 0.00947285, 0.00947285,
      0.00947285, 0.00947285, 0.00947285, 0.00947285, 0.00947285, 0.00947285,
      0.00947285, 0.00947285, 0.00947285, 0.00947285, 0.00947285, 0.00947285,
      0.00947285, 0.00947285, 0.00947285, 0.00947285, 0.00947285, 0.00947285,
      0.00947285, 0.00947285, 0.00947285, 0.00947285, 0.00947285, 0.00947285,
      0.00947285, 0.00947285, 0.00947285, 0.00947285, 0.00947285, 0.00947285,
      0.00947285, 0.00947285, 0.00947285, 0.00947285, 0.00947285, 0.00947285,
      0.00947285, 0.00947285, 0.00947285, 0.00947285, 0.00940407, 0.00940407,
      0.00940407, 0.00940407, 0.00940407, 0.00940407, 0.00940407, 0.00940407,
      0.00940407, 0.00940407, 0.00940407, 0.00940407, 0.00940407, 0.00940407,
      0.00940407, 0.00940407, 0.00940407, 0.00940407, 0.00940407, 0.00940407,
      0.00940407, 0.00940407, 0.00940407, 0.00940407, 0.00940407, 0.00940407,
      0.00940407, 0.00940407, 0.00940407, 0.00940407, 0.00940407, 0.00940407,
      0.00940407, 0.00940407, 0.00940407, 0.00940407, 0.00940407, 0.00940407,
      0.00940407, 0.00940407, 0.00940407, 0.00940407, 0.00940407, 0.00940407,
      0.00940407, 0.00940407, 0.00940407, 0.00940407, 0.00926363, 0.00926363,
      0.00926363, 0.00926363, 0.00926363, 0.00926363, 0.00926363, 0.00926363,
      0.00926363, 0.00926363, 0.00926363, 0.00926363, 0.00926363, 0.00926363,
      0.00926363, 0.00926363, 0.00926363, 0.00926363, 0.00926363, 0.00926363,
      0.00926363, 0.00926363, 0.00926363, 0.00926363, 0.00926363, 0.00926363,
      0.00926363, 0.00926363, 0.00926363, 0.00926363, 0.00926363, 0.00926363,
      0.00926363, 0.00926363, 0.00926363, 0.00926363, 0.00926363, 0.00926363,
      0.00926363, 0.00926363, 0.00926363, 0.00926363, 0.00926363, 0.00926363,
      0.00926363, 0.00926363, 0.00926363, 0.00926363, 0.00902699, 0.00902699,
      0.00902699, 0.00902699, 0.00902699, 0.00902699, 0.00902699, 0.00902699,
      0.00902699, 0.00902699, 0.00902699, 0.00902699, 0.00902699, 0.00902699,
      0.00902699, 0.00902699, 0.00902699, 0.00902699, 0.00902699, 0.00902699,
      0.00902699, 0.00902699, 0.00902699, 0.00902699, 0.00902699, 0.00902699,
      0.00902699, 0.00902699, 0.00902699, 0.00902699, 0.00902699, 0.00902699,
      0.00902699, 0.00902699, 0.00902699, 0.00902699, 0.00902699, 0.00902699,
      0.00902699, 0.00902699, 0.00902699, 0.00902699, 0.00902699, 0.00902699,
      0.00902699, 0.00902699, 0.00902699, 0.00902699, 0.00866554, 0.00866554,
      0.00866554, 0.00866554, 0.00866554, 0.00866554, 0.00866554, 0.00866554,
      0.00866554, 0.00866554, 0.00866554, 0.00866554, 0.00866554, 0.00866554,
      0.00866554, 0.00866554, 0.00866554, 0.00866554, 0.00866554, 0.00866554,
      0.00866554, 0.00866554, 0.00866554, 0.00866554, 0.00866554, 0.00866554,
      0.00866554, 0.00866554, 0.00866554, 0.00866554, 0.00866554, 0.00866554,
      0.00866554, 0.00866554, 0.00866554, 0.00866554, 0.00866554, 0.00866554,
      0.00866554, 0.00866554, 0.00866554, 0.00866554, 0.00866554, 0.00866554,
      0.00866554, 0.00866554, 0.00866554, 0.00866554, 0.00814314, 0.00814314,
      0.00814314, 0.00814314, 0.00814314, 0.00814314, 0.00814314, 0.00814314,
      0.00814314, 0.00814314, 0.00814314, 0.00814314, 0.00814314, 0.00814314,
      0.00814314, 0.00814314, 0.00814314, 0.00814314, 0.00814314, 0.00814314,
      0.00814314, 0.00814314, 0.00814314, 0.00814314, 0.00814314, 0.00814314,
      0.00814314, 0.00814314, 0.00814314, 0.00814314, 0.00814314, 0.00814314,
      0.00814314, 0.00814314, 0.00814314, 0.00814314, 0.00814314, 0.00814314,
      0.00814314, 0.00814314, 0.00814314, 0.00814314, 0.00814314, 0.00814314,
      0.00814314, 0.00814314, 0.00814314, 0.00814314, 0.00741109, 0.00741109,
      0.00741109, 0.00741109, 0.00741109, 0.00741109, 0.00741109, 0.00741109,
      0.00741109, 0.00741109, 0.00741109, 0.00741109, 0.00741109, 0.00741109,
      0.00741109, 0.00741109, 0.00741109, 0.00741109, 0.00741109, 0.00741109,
      0.00741109, 0.00741109, 0.00741109, 0.00741109, 0.00741109, 0.00741109,
      0.00741109, 0.00741109, 0.00741109, 0.00741109, 0.00741109, 0.00741109,
      0.00741109, 0.00741109, 0.00741109, 0.00741109, 0.00741109, 0.00741109,
      0.00741109, 0.00741109, 0.00741109, 0.00741109, 0.00741109, 0.00741109,
      0.00741109, 0.00741109, 0.00741109, 0.00741109, 0.00640346, 0.00640346,
      0.00640346, 0.00640346, 0.00640346, 0.00640346, 0.00640346, 0.00640346,
      0.00640346, 0.00640346, 0.00640346, 0.00640346, 0.00640346, 0.00640346,
      0.00640346, 0.00640346, 0.00640346, 0.00640346, 0.00640346, 0.00640346,
      0.00640346, 0.00640346, 0.00640346, 0.00640346, 0.00640346, 0.00640346,
      0.00640346, 0.00640346, 0.00640346, 0.00640346, 0.00640346, 0.00640346,
      0.00640346, 0.00640346, 0.00640346, 0.00640346, 0.00640346, 0.00640346,
      0.00640346, 0.00640346, 0.00640346, 0.00640346, 0.00640346, 0.00640346,
      0.00640346, 0.00640346, 0.00640346, 0.00640346, 0.00947116, 0.00947116,
      0.00947116, 0.00947116, 0.00947116, 0.00947116, 0.00947116, 0.00947116,
      0.00947116, 0.00947116, 0.00947116, 0.00947116, 0.00947116, 0.00947116,
      0.00947116, 0.00947116, 0.00947116, 0.00947116, 0.00947116, 0.00947116,
      0.00947116, 0.00947116, 0.00947116, 0.00947116, 0.00947116, 0.00947116,
      0.00947116, 0.00947116, 0.00947116, 0.00947116, 0.00947116, 0.00947116,
      0.00947116, 0.00947116, 0.00947116, 0.00947116, 0.00947116, 0.00947116,
      0.00947116, 0.00947116, 0.00947116, 0.00947116, 0.00947116, 0.00947116,
      0.00947116, 0.00947116, 0.00947116, 0.00947116, 0.00939023, 0.00939023,
      0.00939023, 0.00939023, 0.00939023, 0.00939023, 0.00939023, 0.00939023,
      0.00939023, 0.00939023, 0.00939023, 0.00939023, 0.00939023, 0.00939023,
      0.00939023, 0.00939023, 0.00939023, 0.00939023, 0.00939023, 0.00939023,
      0.00939023, 0.00939023, 0.00939023, 0.00939023, 0.00939023, 0.00939023,
      0.00939023, 0.00939023, 0.00939023, 0.00939023, 0.00939023, 0.00939023,
      0.00939023, 0.00939023, 0.00939023, 0.00939023, 0.00939023, 0.00939023,
      0.00939023, 0.00939023, 0.00939023, 0.00939023, 0.00939023, 0.00939023,
      0.00939023, 0.00939023, 0.00939023, 0.00939023, 0.00922751, 0.00922751,
      0.00922751, 0.00922751, 0.00922751, 0.00922751, 0.00922751, 0.00922751,
      0.00922751, 0.00922751, 0.00922751, 0.00922751, 0.00922751, 0.00922751,
      0.00922751, 0.00922751, 0.00922751, 0.00922751, 0.00922751, 0.00922751,
      0.00922751, 0.00922751, 0.00922751, 0.00922751, 0.00922751, 0.00922751,
      0.00922751, 0.00922751, 0.00922751, 0.00922751, 0.00922751, 0.00922751,
      0.00922751, 0.00922751, 0.00922751, 0.00922751, 0.00922751, 0.00922751,
      0.00922751, 0.00922751, 0.00922751, 0.00922751, 0.00922751, 0.00922751,
      0.00922751, 0.00922751, 0.00922751, 0.00922751, 0.00896092, 0.00896092,
      0.00896092, 0.00896092, 0.00896092, 0.00896092, 0.00896092, 0.00896092,
      0.00896092, 0.00896092, 0.00896092, 0.00896092, 0.00896092, 0.00896092,
      0.00896092, 0.00896092, 0.00896092, 0.00896092, 0.00896092, 0.00896092,
      0.00896092, 0.00896092, 0.00896092, 0.00896092, 0.00896092, 0.00896092,
      0.00896092, 0.00896092, 0.00896092, 0.00896092, 0.00896092, 0.00896092,
      0.00896092, 0.00896092, 0.00896092, 0.00896092, 0.00896092, 0.00896092,
      0.00896092, 0.00896092, 0.00896092, 0.00896092, 0.00896092, 0.00896092,
      0.00896092, 0.00896092, 0.00896092, 0.00896092, 0.00856652, 0.00856652,
      0.00856652, 0.00856652, 0.00856652, 0.00856652, 0.00856652, 0.00856652,
      0.00856652, 0.00856652, 0.00856652, 0.00856652, 0.00856652, 0.00856652,
      0.00856652, 0.00856652, 0.00856652, 0.00856652, 0.00856652, 0.00856652,
      0.00856652, 0.00856652, 0.00856652, 0.00856652, 0.00856652, 0.00856652,
      0.00856652, 0.00856652, 0.00856652, 0.00856652, 0.00856652, 0.00856652,
      0.00856652, 0.00856652, 0.00856652, 0.00856652, 0.00856652, 0.00856652,
      0.00856652, 0.00856652, 0.00856652, 0.00856652, 0.00856652, 0.00856652,
      0.00856652, 0.00856652, 0.00856652, 0.00856652, 0.00801853, 0.00801853,
      0.00801853, 0.00801853, 0.00801853, 0.00801853, 0.00801853, 0.00801853,
      0.00801853, 0.00801853, 0.00801853, 0.00801853, 0.00801853, 0.00801853,
      0.00801853, 0.00801853, 0.00801853, 0.00801853, 0.00801853, 0.00801853,
      0.00801853, 0.00801853, 0.00801853, 0.00801853, 0.00801853, 0.00801853,
      0.00801853, 0.00801853, 0.00801853, 0.00801853, 0.00801853, 0.00801853,
      0.00801853, 0.00801853, 0.00801853, 0.00801853, 0.00801853, 0.00801853,
      0.00801853, 0.00801853, 0.00801853, 0.00801853, 0.00801853, 0.00801853,
      0.00801853, 0.00801853, 0.00801853, 0.00801853, 0.00948809, 0.00948809,
      0.00948809, 0.00948809, 0.00948809, 0.00948809, 0.00948809, 0.00948809,
      0.00948809, 0.00948809, 0.00948809, 0.00948809, 0.00948809, 0.00948809,
      0.00948809, 0.00948809, 0.00948809, 0.00948809, 0.00948809, 0.00948809,
      0.00948809, 0.00948809, 0.00948809, 0.00948809, 0.00948809, 0.00948809,
      0.00948809, 0.00948809, 0.00948809, 0.00948809, 0.00948809, 0.00948809,
      0.00948809, 0.00948809, 0.00948809, 0.00948809, 0.00948809, 0.00948809,
      0.00948809, 0.00948809, 0.00948809, 0.00948809, 0.00948809, 0.00948809,
      0.00948809, 0.00948809, 0.00948809, 0.00948809, 0.00939794, 0.00939794,
      0.00939794, 0.00939794, 0.00939794, 0.00939794, 0.00939794, 0.00939794,
      0.00939794, 0.00939794, 0.00939794, 0.00939794, 0.00939794, 0.00939794,
      0.00939794, 0.00939794, 0.00939794, 0.00939794, 0.00939794, 0.00939794,
      0.00939794, 0.00939794, 0.00939794, 0.00939794, 0.00939794, 0.00939794,
      0.00939794, 0.00939794, 0.00939794, 0.00939794, 0.00939794, 0.00939794,
      0.00939794, 0.00939794, 0.00939794, 0.00939794, 0.00939794, 0.00939794,
      0.00939794, 0.00939794, 0.00939794, 0.00939794, 0.00939794, 0.00939794,
      0.00939794, 0.00939794, 0.00939794, 0.00939794, 0.00921859, 0.00921859,
      0.00921859, 0.00921859, 0.00921859, 0.00921859, 0.00921859, 0.00921859,
      0.00921859, 0.00921859, 0.00921859, 0.00921859, 0.00921859, 0.00921859,
      0.00921859, 0.00921859, 0.00921859, 0.00921859, 0.00921859, 0.00921859,
      0.00921859, 0.00921859, 0.00921859, 0.00921859, 0.00921859, 0.00921859,
      0.00921859, 0.00921859, 0.00921859, 0.00921859, 0.00921859, 0.00921859,
      0.00921859, 0.00921859, 0.00921859, 0.00921859, 0.00921859, 0.00921859,
      0.00921859, 0.00921859, 0.00921859, 0.00921859, 0.00921859, 0.00921859,
      0.00921859, 0.00921859, 0.00921859, 0.00921859, 0.00893484, 0.00893484,
      0.00893484, 0.00893484, 0.00893484, 0.00893484, 0.00893484, 0.00893484,
      0.00893484, 0.00893484, 0.00893484, 0.00893484, 0.00893484, 0.00893484,
      0.00893484, 0.00893484, 0.00893484, 0.00893484, 0.00893484, 0.00893484,
      0.00893484, 0.00893484, 0.00893484, 0.00893484, 0.00893484, 0.00893484,
      0.00893484, 0.00893484, 0.00893484, 0.00893484, 0.00893484, 0.00893484,
      0.00893484, 0.00893484, 0.00893484, 0.00893484, 0.00893484, 0.00893484,
      0.00893484, 0.00893484, 0.00893484, 0.00893484, 0.00893484, 0.00893484,
      0.00893484, 0.00893484, 0.00893484, 0.00893484, 0.00951446, 0.00951446,
      0.00951446, 0.00951446, 0.00951446, 0.00951446, 0.00951446, 0.00951446,
      0.00951446, 0.00951446, 0.00951446, 0.00951446, 0.00951446, 0.00951446,
      0.00951446, 0.00951446, 0.00951446, 0.00951446, 0.00951446, 0.00951446,
      0.00951446, 0.00951446, 0.00951446, 0.00951446, 0.00951446, 0.00951446,
      0.00951446, 0.00951446, 0.00951446, 0.00951446, 0.00951446, 0.00951446,
      0.00951446, 0.00951446, 0.00951446, 0.00951446, 0.00951446, 0.00951446,
      0.00951446, 0.00951446, 0.00951446, 0.00951446, 0.00951446, 0.00951446,
      0.00951446, 0.00951446, 0.00951446, 0.00951446, 0.0094121, 0.0094121,
      0.0094121, 0.0094121, 0.0094121, 0.0094121, 0.0094121, 0.0094121,
      0.0094121, 0.0094121, 0.0094121, 0.0094121, 0.0094121, 0.0094121,
      0.0094121, 0.0094121, 0.0094121, 0.0094121, 0.0094121, 0.0094121,
      0.0094121, 0.0094121, 0.0094121, 0.0094121, 0.0094121, 0.0094121,
      0.0094121, 0.0094121, 0.0094121, 0.0094121, 0.0094121, 0.0094121,
      0.0094121, 0.0094121, 0.0094121, 0.0094121, 0.0094121, 0.0094121,
      0.0094121, 0.0094121, 0.0094121, 0.0094121, 0.0094121, 0.0094121,
      0.0094121, 0.0094121, 0.0094121, 0.0094121;

  Eigen::VectorXd H_phi_ref = Eigen::VectorXd::Zero(1202);
  H_phi_ref << 1.5708, 1.5708, 1.5708, 1.5708, 0, 3.14159, 0.785398, 0.785398,
      2.35619, 2.35619, 0.785398, 0.785398, 2.35619, 2.35619, 1.5708, 1.5708,
      1.5708, 1.5708, 0.955317, 0.955317, 0.955317, 0.955317, 2.18628, 2.18628,
      2.18628, 2.18628, 0.0525288, 0.0525288, 0.0525288, 0.0525288, 3.08906,
      3.08906, 3.08906, 3.08906, 1.53366, 1.53366, 1.53366, 1.53366, 1.60793,
      1.60793, 1.60793, 1.60793, 1.53366, 1.53366, 1.53366, 1.53366, 1.60793,
      1.60793, 1.60793, 1.60793, 0.129623, 0.129623, 0.129623, 0.129623,
      3.01197, 3.01197, 3.01197, 3.01197, 1.47927, 1.47927, 1.47927, 1.47927,
      1.66232, 1.66232, 1.66232, 1.66232, 1.47927, 1.47927, 1.47927, 1.47927,
      1.66232, 1.66232, 1.66232, 1.66232, 0.218256, 0.218256, 0.218256,
      0.218256, 2.92334, 2.92334, 2.92334, 2.92334, 1.41708, 1.41708, 1.41708,
      1.41708, 1.72451, 1.72451, 1.72451, 1.72451, 1.41708, 1.41708, 1.41708,
      1.41708, 1.72451, 1.72451, 1.72451, 1.72451, 0.313542, 0.313542, 0.313542,
      0.313542, 2.82805, 2.82805, 2.82805, 2.82805, 1.35094, 1.35094, 1.35094,
      1.35094, 1.79066, 1.79066, 1.79066, 1.79066, 1.35094, 1.35094, 1.35094,
      1.35094, 1.79066, 1.79066, 1.79066, 1.79066, 0.413284, 0.413284, 0.413284,
      0.413284, 2.72831, 2.72831, 2.72831, 2.72831, 1.28285, 1.28285, 1.28285,
      1.28285, 1.85875, 1.85875, 1.85875, 1.85875, 1.28285, 1.28285, 1.28285,
      1.28285, 1.85875, 1.85875, 1.85875, 1.85875, 0.51637, 0.51637, 0.51637,
      0.51637, 2.62522, 2.62522, 2.62522, 2.62522, 1.21417, 1.21417, 1.21417,
      1.21417, 1.92743, 1.92743, 1.92743, 1.92743, 1.21417, 1.21417, 1.21417,
      1.21417, 1.92743, 1.92743, 1.92743, 1.92743, 0.622242, 0.622242, 0.622242,
      0.622242, 2.51935, 2.51935, 2.51935, 2.51935, 1.14599, 1.14599, 1.14599,
      1.14599, 1.9956, 1.9956, 1.9956, 1.9956, 1.14599, 1.14599, 1.14599,
      1.14599, 1.9956, 1.9956, 1.9956, 1.9956, 0.730667, 0.730667, 0.730667,
      0.730667, 2.41093, 2.41093, 2.41093, 2.41093, 1.07935, 1.07935, 1.07935,
      1.07935, 2.06224, 2.06224, 2.06224, 2.06224, 1.07935, 1.07935, 1.07935,
      1.07935, 2.06224, 2.06224, 2.06224, 2.06224, 0.841638, 0.841638, 0.841638,
      0.841638, 2.29995, 2.29995, 2.29995, 2.29995, 1.01536, 1.01536, 1.01536,
      1.01536, 2.12623, 2.12623, 2.12623, 2.12623, 1.01536, 1.01536, 1.01536,
      1.01536, 2.12623, 2.12623, 2.12623, 2.12623, 1.07199, 1.07199, 1.07199,
      1.07199, 2.06961, 2.06961, 2.06961, 2.06961, 0.900845, 0.900845, 0.900845,
      0.900845, 2.24075, 2.24075, 2.24075, 2.24075, 0.900845, 0.900845,
      0.900845, 0.900845, 2.24075, 2.24075, 2.24075, 2.24075, 1.19197, 1.19197,
      1.19197, 1.19197, 1.94962, 1.94962, 1.94962, 1.94962, 0.854001, 0.854001,
      0.854001, 0.854001, 2.28759, 2.28759, 2.28759, 2.28759, 0.854001,
      0.854001, 0.854001, 0.854001, 2.28759, 2.28759, 2.28759, 2.28759, 1.31545,
      1.31545, 1.31545, 1.31545, 1.82615, 1.82615, 1.82615, 1.82615, 0.817319,
      0.817319, 0.817319, 0.817319, 2.32427, 2.32427, 2.32427, 2.32427,
      0.817319, 0.817319, 0.817319, 0.817319, 2.32427, 2.32427, 2.32427,
      2.32427, 1.44212, 1.44212, 1.44212, 1.44212, 1.69948, 1.69948, 1.69948,
      1.69948, 0.793632, 0.793632, 0.793632, 0.793632, 2.34796, 2.34796,
      2.34796, 2.34796, 0.793632, 0.793632, 0.793632, 0.793632, 2.34796,
      2.34796, 2.34796, 2.34796, 1.5708, 1.5708, 1.5708, 1.5708, 1.5708, 1.5708,
      1.5708, 1.5708, 0.107445, 0.107445, 3.03415, 3.03415, 1.46335, 1.46335,
      1.67824, 1.67824, 0.107445, 0.107445, 3.03415, 3.03415, 1.46335, 1.46335,
      1.67824, 1.67824, 1.5708, 1.5708, 1.5708, 1.5708, 1.5708, 1.5708, 1.5708,
      1.5708, 0.261166, 0.261166, 2.88043, 2.88043, 1.30963, 1.30963, 1.83196,
      1.83196, 0.261166, 0.261166, 2.88043, 2.88043, 1.30963, 1.30963, 1.83196,
      1.83196, 1.5708, 1.5708, 1.5708, 1.5708, 1.5708, 1.5708, 1.5708, 1.5708,
      0.430445, 0.430445, 2.71115, 2.71115, 1.14035, 1.14035, 2.00124, 2.00124,
      0.430445, 0.430445, 2.71115, 2.71115, 1.14035, 1.14035, 2.00124, 2.00124,
      1.5708, 1.5708, 1.5708, 1.5708, 1.5708, 1.5708, 1.5708, 1.5708, 0.606551,
      0.606551, 2.53504, 2.53504, 0.964246, 0.964246, 2.17735, 2.17735,
      0.606551, 0.606551, 2.53504, 2.53504, 0.964246, 0.964246, 2.17735,
      2.17735, 1.51867, 1.51867, 1.51867, 1.51867, 1.62293, 1.62293, 1.62293,
      1.62293, 1.39268, 1.39268, 1.39268, 1.39268, 1.74891, 1.74891, 1.74891,
      1.74891, 1.51867, 1.51867, 1.51867, 1.51867, 1.62293, 1.62293, 1.62293,
      1.62293, 0.185747, 0.185747, 0.185747, 0.185747, 2.95585, 2.95585,
      2.95585, 2.95585, 1.39268, 1.39268, 1.39268, 1.39268, 1.74891, 1.74891,
      1.74891, 1.74891, 0.185747, 0.185747, 0.185747, 0.185747, 2.95585,
      2.95585, 2.95585, 2.95585, 1.459, 1.459, 1.459, 1.459, 1.68259, 1.68259,
      1.68259, 1.68259, 1.32062, 1.32062, 1.32062, 1.32062, 1.82097, 1.82097,
      1.82097, 1.82097, 1.459, 1.459, 1.459, 1.459, 1.68259, 1.68259, 1.68259,
      1.68259, 0.275001, 0.275001, 0.275001, 0.275001, 2.86659, 2.86659,
      2.86659, 2.86659, 1.32062, 1.32062, 1.32062, 1.32062, 1.82097, 1.82097,
      1.82097, 1.82097, 0.275001, 0.275001, 0.275001, 0.275001, 2.86659,
      2.86659, 2.86659, 2.86659, 1.5117, 1.5117, 1.5117, 1.5117, 1.62989,
      1.62989, 1.62989, 1.62989, 1.2287, 1.2287, 1.2287, 1.2287, 1.91289,
      1.91289, 1.91289, 1.91289, 1.5117, 1.5117, 1.5117, 1.5117, 1.62989,
      1.62989, 1.62989, 1.62989, 0.347577, 0.347577, 0.347577, 0.347577,
      2.79402, 2.79402, 2.79402, 2.79402, 1.2287, 1.2287, 1.2287, 1.2287,
      1.91289, 1.91289, 1.91289, 1.91289, 0.347577, 0.347577, 0.347577,
      0.347577, 2.79402, 2.79402, 2.79402, 2.79402, 1.39524, 1.39524, 1.39524,
      1.39524, 1.74635, 1.74635, 1.74635, 1.74635, 1.24785, 1.24785, 1.24785,
      1.24785, 1.89374, 1.89374, 1.89374, 1.89374, 1.39524, 1.39524, 1.39524,
      1.39524, 1.74635, 1.74635, 1.74635, 1.74635, 0.370677, 0.370677, 0.370677,
      0.370677, 2.77092, 2.77092, 2.77092, 2.77092, 1.24785, 1.24785, 1.24785,
      1.24785, 1.89374, 1.89374, 1.89374, 1.89374, 0.370677, 0.370677, 0.370677,
      0.370677, 2.77092, 2.77092, 2.77092, 2.77092, 1.44877, 1.44877, 1.44877,
      1.44877, 1.69282, 1.69282, 1.69282, 1.69282, 1.14941, 1.14941, 1.14941,
      1.14941, 1.99218, 1.99218, 1.99218, 1.99218, 1.44877, 1.44877, 1.44877,
      1.44877, 1.69282, 1.69282, 1.69282, 1.69282, 0.440901, 0.440901, 0.440901,
      0.440901, 2.70069, 2.70069, 2.70069, 2.70069, 1.14941, 1.14941, 1.14941,
      1.14941, 1.99218, 1.99218, 1.99218, 1.99218, 0.440901, 0.440901, 0.440901,
      0.440901, 2.70069, 2.70069, 2.70069, 2.70069, 1.32943, 1.32943, 1.32943,
      1.32943, 1.81216, 1.81216, 1.81216, 1.81216, 1.17512, 1.17512, 1.17512,
      1.17512, 1.96647, 1.96647, 1.96647, 1.96647, 1.32943, 1.32943, 1.32943,
      1.32943, 1.81216, 1.81216, 1.81216, 1.81216, 0.470723, 0.470723, 0.470723,
      0.470723, 2.67087, 2.67087, 2.67087, 2.67087, 1.17512, 1.17512, 1.17512,
      1.17512, 1.96647, 1.96647, 1.96647, 1.96647, 0.470723, 0.470723, 0.470723,
      0.470723, 2.67087, 2.67087, 2.67087, 2.67087, 1.50809, 1.50809, 1.50809,
      1.50809, 1.6335, 1.6335, 1.6335, 1.6335, 1.05501, 1.05501, 1.05501,
      1.05501, 2.08659, 2.08659, 2.08659, 2.08659, 1.50809, 1.50809, 1.50809,
      1.50809, 1.6335, 1.6335, 1.6335, 1.6335, 0.520353, 0.520353, 0.520353,
      0.520353, 2.62124, 2.62124, 2.62124, 2.62124, 1.05501, 1.05501, 1.05501,
      1.05501, 2.08659, 2.08659, 2.08659, 2.08659, 0.520353, 0.520353, 0.520353,
      0.520353, 2.62124, 2.62124, 2.62124, 2.62124, 1.38396, 1.38396, 1.38396,
      1.38396, 1.75763, 1.75763, 1.75763, 1.75763, 1.07181, 1.07181, 1.07181,
      1.07181, 2.06978, 2.06978, 2.06978, 2.06978, 1.38396, 1.38396, 1.38396,
      1.38396, 1.75763, 1.75763, 1.75763, 1.75763, 0.539048, 0.539048, 0.539048,
      0.539048, 2.60255, 2.60255, 2.60255, 2.60255, 1.07181, 1.07181, 1.07181,
      1.07181, 2.06978, 2.06978, 2.06978, 2.06978, 0.539048, 0.539048, 0.539048,
      0.539048, 2.60255, 2.60255, 2.60255, 2.60255, 1.26301, 1.26301, 1.26301,
      1.26301, 1.87858, 1.87858, 1.87858, 1.87858, 1.1032, 1.1032, 1.1032,
      1.1032, 2.03839, 2.03839, 2.03839, 2.03839, 1.26301, 1.26301, 1.26301,
      1.26301, 1.87858, 1.87858, 1.87858, 1.87858, 0.574111, 0.574111, 0.574111,
      0.574111, 2.56748, 2.56748, 2.56748, 2.56748, 1.1032, 1.1032, 1.1032,
      1.1032, 2.03839, 2.03839, 2.03839, 2.03839, 0.574111, 0.574111, 0.574111,
      0.574111, 2.56748, 2.56748, 2.56748, 2.56748, 1.44368, 1.44368, 1.44368,
      1.44368, 1.69792, 1.69792, 1.69792, 1.69792, 0.972528, 0.972528, 0.972528,
      0.972528, 2.16906, 2.16906, 2.16906, 2.16906, 1.44368, 1.44368, 1.44368,
      1.44368, 1.69792, 1.69792, 1.69792, 1.69792, 0.615424, 0.615424, 0.615424,
      0.615424, 2.52617, 2.52617, 2.52617, 2.52617, 0.972528, 0.972528,
      0.972528, 0.972528, 2.16906, 2.16906, 2.16906, 2.16906, 0.615424,
      0.615424, 0.615424, 0.615424, 2.52617, 2.52617, 2.52617, 2.52617, 1.31872,
      1.31872, 1.31872, 1.31872, 1.82287, 1.82287, 1.82287, 1.82287, 0.996278,
      0.996278, 0.996278, 0.996278, 2.14531, 2.14531, 2.14531, 2.14531, 1.31872,
      1.31872, 1.31872, 1.31872, 1.82287, 1.82287, 1.82287, 1.82287, 0.640918,
      0.640918, 0.640918, 0.640918, 2.50067, 2.50067, 2.50067, 2.50067,
      0.996278, 0.996278, 0.996278, 0.996278, 2.14531, 2.14531, 2.14531,
      2.14531, 0.640918, 0.640918, 0.640918, 0.640918, 2.50067, 2.50067,
      2.50067, 2.50067, 1.19718, 1.19718, 1.19718, 1.19718, 1.94441, 1.94441,
      1.94441, 1.94441, 1.03288, 1.03288, 1.03288, 1.03288, 2.10872, 2.10872,
      2.10872, 2.10872, 1.19718, 1.19718, 1.19718, 1.19718, 1.94441, 1.94441,
      1.94441, 1.94441, 0.680344, 0.680344, 0.680344, 0.680344, 2.46125,
      2.46125, 2.46125, 2.46125, 1.03288, 1.03288, 1.03288, 1.03288, 2.10872,
      2.10872, 2.10872, 2.10872, 0.680344, 0.680344, 0.680344, 0.680344,
      2.46125, 2.46125, 2.46125, 2.46125, 1.50651, 1.50651, 1.50651, 1.50651,
      1.63509, 1.63509, 1.63509, 1.63509, 0.877042, 0.877042, 0.877042,
      0.877042, 2.26455, 2.26455, 2.26455, 2.26455, 1.50651, 1.50651, 1.50651,
      1.50651, 1.63509, 1.63509, 1.63509, 1.63509, 0.697949, 0.697949, 0.697949,
      0.697949, 2.44364, 2.44364, 2.44364, 2.44364, 0.877042, 0.877042,
      0.877042, 0.877042, 2.26455, 2.26455, 2.26455, 2.26455, 0.697949,
      0.697949, 0.697949, 0.697949, 2.44364, 2.44364, 2.44364, 2.44364, 1.37902,
      1.37902, 1.37902, 1.37902, 1.76257, 1.76257, 1.76257, 1.76257, 0.893125,
      0.893125, 0.893125, 0.893125, 2.24847, 2.24847, 2.24847, 2.24847, 1.37902,
      1.37902, 1.37902, 1.37902, 1.76257, 1.76257, 1.76257, 1.76257, 0.714595,
      0.714595, 0.714595, 0.714595, 2.427, 2.427, 2.427, 2.427, 0.893125,
      0.893125, 0.893125, 0.893125, 2.24847, 2.24847, 2.24847, 2.24847,
      0.714595, 0.714595, 0.714595, 0.714595, 2.427, 2.427, 2.427, 2.427,
      1.25431, 1.25431, 1.25431, 1.25431, 1.88728, 1.88728, 1.88728, 1.88728,
      0.923394, 0.923394, 0.923394, 0.923394, 2.2182, 2.2182, 2.2182, 2.2182,
      1.25431, 1.25431, 1.25431, 1.25431, 1.88728, 1.88728, 1.88728, 1.88728,
      0.745969, 0.745969, 0.745969, 0.745969, 2.39562, 2.39562, 2.39562,
      2.39562, 0.923394, 0.923394, 0.923394, 0.923394, 2.2182, 2.2182, 2.2182,
      2.2182, 0.745969, 0.745969, 0.745969, 0.745969, 2.39562, 2.39562, 2.39562,
      2.39562, 1.13309, 1.13309, 1.13309, 1.13309, 2.0085, 2.0085, 2.0085,
      2.0085, 0.965057, 0.965057, 0.965057, 0.965057, 2.17654, 2.17654, 2.17654,
      2.17654, 1.13309, 1.13309, 1.13309, 1.13309, 2.0085, 2.0085, 2.0085,
      2.0085, 0.789242, 0.789242, 0.789242, 0.789242, 2.35235, 2.35235, 2.35235,
      2.35235, 0.965057, 0.965057, 0.965057, 0.965057, 2.17654, 2.17654,
      2.17654, 2.17654, 0.789242, 0.789242, 0.789242, 0.789242, 2.35235,
      2.35235, 2.35235, 2.35235;
  Eigen::VectorXd H_theta_ref = Eigen::VectorXd::Zero(1202);
  H_theta_ref << 0, 3.14159, 1.5708, -1.5708, 0, 0, 1.5708, -1.5708, 1.5708,
      -1.5708, 0, 3.14159, 0, 3.14159, 0.785398, 2.35619, -0.785398, -2.35619,
      0.785398, 2.35619, -0.785398, -2.35619, 0.785398, 2.35619, -0.785398,
      -2.35619, 0.785398, 2.35619, -0.785398, -2.35619, 0.785398, 2.35619,
      -0.785398, -2.35619, 1.53364, 1.60796, -1.53364, -1.60796, 1.53364,
      1.60796, -1.53364, -1.60796, 0.0371605, 3.10443, -0.0371605, -3.10443,
      0.0371605, 3.10443, -0.0371605, -3.10443, 0.785398, 2.35619, -0.785398,
      -2.35619, 0.785398, 2.35619, -0.785398, -2.35619, 1.47888, 1.66271,
      -1.47888, -1.66271, 1.47888, 1.66271, -1.47888, -1.66271, 0.0919142,
      3.04968, -0.0919142, -3.04968, 0.0919142, 3.04968, -0.0919142, -3.04968,
      0.785398, 2.35619, -0.785398, -2.35619, 0.785398, 2.35619, -0.785398,
      -2.35619, 1.41524, 1.72636, -1.41524, -1.72636, 1.41524, 1.72636,
      -1.41524, -1.72636, 0.155561, 2.98603, -0.155561, -2.98603, 0.155561,
      2.98603, -0.155561, -2.98603, 0.785398, 2.35619, -0.785398, -2.35619,
      0.785398, 2.35619, -0.785398, -2.35619, 1.34542, 1.79617, -1.34542,
      -1.79617, 1.34542, 1.79617, -1.34542, -1.79617, 0.225375, 2.91622,
      -0.225375, -2.91622, 0.225375, 2.91622, -0.225375, -2.91622, 0.785398,
      2.35619, -0.785398, -2.35619, 0.785398, 2.35619, -0.785398, -2.35619,
      1.2701, 1.87149, -1.2701, -1.87149, 1.2701, 1.87149, -1.2701, -1.87149,
      0.300693, 2.8409, -0.300693, -2.8409, 0.300693, 2.8409, -0.300693,
      -2.8409, 0.785398, 2.35619, -0.785398, -2.35619, 0.785398, 2.35619,
      -0.785398, -2.35619, 1.18903, 1.95256, -1.18903, -1.95256, 1.18903,
      1.95256, -1.18903, -1.95256, 0.381766, 2.75983, -0.381766, -2.75983,
      0.381766, 2.75983, -0.381766, -2.75983, 0.785398, 2.35619, -0.785398,
      -2.35619, 0.785398, 2.35619, -0.785398, -2.35619, 1.1014, 2.04019,
      -1.1014, -2.04019, 1.1014, 2.04019, -1.1014, -2.04019, 0.469397, 2.6722,
      -0.469397, -2.6722, 0.469397, 2.6722, -0.469397, -2.6722, 0.785398,
      2.35619, -0.785398, -2.35619, 0.785398, 2.35619, -0.785398, -2.35619,
      1.006, 2.13559, -1.006, -2.13559, 1.006, 2.13559, -1.006, -2.13559,
      0.564797, 2.5768, -0.564797, -2.5768, 0.564797, 2.5768, -0.564797,
      -2.5768, 0.785398, 2.35619, -0.785398, -2.35619, 0.785398, 2.35619,
      -0.785398, -2.35619, 0.901275, 2.24032, -0.901275, -2.24032, 0.901275,
      2.24032, -0.901275, -2.24032, 0.669522, 2.47207, -0.669522, -2.47207,
      0.669522, 2.47207, -0.669522, -2.47207, 0.785398, 2.35619, -0.785398,
      -2.35619, 0.785398, 2.35619, -0.785398, -2.35619, 0.656433, 2.48516,
      -0.656433, -2.48516, 0.656433, 2.48516, -0.656433, -2.48516, 0.914364,
      2.22723, -0.914364, -2.22723, 0.914364, 2.22723, -0.914364, -2.22723,
      0.785398, 2.35619, -0.785398, -2.35619, 0.785398, 2.35619, -0.785398,
      -2.35619, 0.512718, 2.62887, -0.512718, -2.62887, 0.512718, 2.62887,
      -0.512718, -2.62887, 1.05808, 2.08351, -1.05808, -2.08351, 1.05808,
      2.08351, -1.05808, -2.08351, 0.785398, 2.35619, -0.785398, -2.35619,
      0.785398, 2.35619, -0.785398, -2.35619, 0.353657, 2.78794, -0.353657,
      -2.78794, 0.353657, 2.78794, -0.353657, -2.78794, 1.21714, 1.92445,
      -1.21714, -1.92445, 1.21714, 1.92445, -1.21714, -1.92445, 0.785398,
      2.35619, -0.785398, -2.35619, 0.785398, 2.35619, -0.785398, -2.35619,
      0.180991, 2.9606, -0.180991, -2.9606, 0.180991, 2.9606, -0.180991,
      -2.9606, 1.38981, 1.75179, -1.38981, -1.75179, 1.38981, 1.75179, -1.38981,
      -1.75179, 1.46335, 1.67824, -1.46335, -1.67824, 0.107445, 3.03415,
      -0.107445, -3.03415, 0, 3.14159, 0, 3.14159, 0, 3.14159, 0, 3.14159,
      1.5708, -1.5708, 1.5708, -1.5708, 1.5708, -1.5708, 1.5708, -1.5708,
      1.30963, 1.83196, -1.30963, -1.83196, 0.261166, 2.88043, -0.261166,
      -2.88043, 0, 3.14159, 0, 3.14159, 0, 3.14159, 0, 3.14159, 1.5708, -1.5708,
      1.5708, -1.5708, 1.5708, -1.5708, 1.5708, -1.5708, 1.14035, 2.00124,
      -1.14035, -2.00124, 0.430445, 2.71115, -0.430445, -2.71115, 0, 3.14159, 0,
      3.14159, 0, 3.14159, 0, 3.14159, 1.5708, -1.5708, 1.5708, -1.5708, 1.5708,
      -1.5708, 1.5708, -1.5708, 0.964246, 2.17735, -0.964246, -2.17735,
      0.606551, 2.53504, -0.606551, -2.53504, 0, 3.14159, 0, 3.14159, 0,
      3.14159, 0, 3.14159, 1.5708, -1.5708, 1.5708, -1.5708, 1.5708, -1.5708,
      1.5708, -1.5708, 0.178363, 2.96323, -0.178363, -2.96323, 0.178363,
      2.96323, -0.178363, -2.96323, 0.0529688, 3.08862, -0.0529688, -3.08862,
      0.0529688, 3.08862, -0.0529688, -3.08862, 1.39243, 1.74916, -1.39243,
      -1.74916, 1.39243, 1.74916, -1.39243, -1.74916, 0.286028, 2.85557,
      -0.286028, -2.85557, 0.286028, 2.85557, -0.286028, -2.85557, 1.51783,
      1.62377, -1.51783, -1.62377, 1.51783, 1.62377, -1.51783, -1.62377,
      1.28477, 1.85682, -1.28477, -1.85682, 1.28477, 1.85682, -1.28477,
      -1.85682, 0.251779, 2.88981, -0.251779, -2.88981, 0.251779, 2.88981,
      -0.251779, -2.88981, 0.115405, 3.02619, -0.115405, -3.02619, 0.115405,
      3.02619, -0.115405, -3.02619, 1.31902, 1.82257, -1.31902, -1.82257,
      1.31902, 1.82257, -1.31902, -1.82257, 0.423381, 2.71821, -0.423381,
      -2.71821, 0.423381, 2.71821, -0.423381, -2.71821, 1.45539, 1.6862,
      -1.45539, -1.6862, 1.45539, 1.6862, -1.45539, -1.6862, 1.14742, 1.99418,
      -1.14742, -1.99418, 1.14742, 1.99418, -1.14742, -1.99418, 0.342718,
      2.79887, -0.342718, -2.79887, 0.342718, 2.79887, -0.342718, -2.79887,
      0.0627328, 3.07886, -0.0627328, -3.07886, 0.0627328, 3.07886, -0.0627328,
      -3.07886, 1.22808, 1.91351, -1.22808, -1.91351, 1.22808, 1.91351,
      -1.22808, -1.91351, 0.174267, 2.96733, -0.174267, -2.96733, 0.174267,
      2.96733, -0.174267, -2.96733, 1.50806, 1.63353, -1.50806, -1.63353,
      1.50806, 1.63353, -1.50806, -1.63353, 1.39653, 1.74506, -1.39653,
      -1.74506, 1.39653, 1.74506, -1.39653, -1.74506, 0.328175, 2.81342,
      -0.328175, -2.81342, 0.328175, 2.81342, -0.328175, -2.81342, 0.185234,
      2.95636, -0.185234, -2.95636, 0.185234, 2.95636, -0.185234, -2.95636,
      1.24262, 1.89897, -1.24262, -1.89897, 1.24262, 1.89897, -1.24262,
      -1.89897, 0.5031, 2.63849, -0.5031, -2.63849, 0.5031, 2.63849, -0.5031,
      -2.63849, 1.38556, 1.75603, -1.38556, -1.75603, 1.38556, 1.75603,
      -1.38556, -1.75603, 1.0677, 2.0739, -1.0677, -2.0739, 1.0677, 2.0739,
      -1.0677, -2.0739, 0.424748, 2.71684, -0.424748, -2.71684, 0.424748,
      2.71684, -0.424748, -2.71684, 0.133791, 3.0078, -0.133791, -3.0078,
      0.133791, 3.0078, -0.133791, -3.0078, 1.14605, 1.99554, -1.14605,
      -1.99554, 1.14605, 1.99554, -1.14605, -1.99554, 0.289247, 2.85235,
      -0.289247, -2.85235, 0.289247, 2.85235, -0.289247, -2.85235, 1.43701,
      1.70459, -1.43701, -1.70459, 1.43701, 1.70459, -1.43701, -1.70459,
      1.28155, 1.86004, -1.28155, -1.86004, 1.28155, 1.86004, -1.28155,
      -1.86004, 0.408175, 2.73342, -0.408175, -2.73342, 0.408175, 2.73342,
      -0.408175, -2.73342, 0.26203, 2.87956, -0.26203, -2.87956, 0.26203,
      2.87956, -0.26203, -2.87956, 1.16262, 1.97897, -1.16262, -1.97897,
      1.16262, 1.97897, -1.16262, -1.97897, 0.555112, 2.58648, -0.555112,
      -2.58648, 0.555112, 2.58648, -0.555112, -2.58648, 1.30877, 1.83283,
      -1.30877, -1.83283, 1.30877, 1.83283, -1.30877, -1.83283, 1.01568,
      2.12591, -1.01568, -2.12591, 1.01568, 2.12591, -1.01568, -2.12591,
      0.516907, 2.62469, -0.516907, -2.62469, 0.516907, 2.62469, -0.516907,
      -2.62469, 0.0720963, 3.0695, -0.0720963, -3.0695, 0.0720963, 3.0695,
      -0.0720963, -3.0695, 1.05389, 2.0877, -1.05389, -2.0877, 1.05389, 2.0877,
      -1.05389, -2.0877, 0.12637, 3.01522, -0.12637, -3.01522, 0.12637, 3.01522,
      -0.12637, -3.01522, 1.4987, 1.64289, -1.4987, -1.64289, 1.4987, 1.64289,
      -1.4987, -1.64289, 1.44443, 1.69717, -1.44443, -1.69717, 1.44443, 1.69717,
      -1.44443, -1.69717, 0.50866, 2.63293, -0.50866, -2.63293, 0.50866,
      2.63293, -0.50866, -2.63293, 0.213155, 2.92844, -0.213155, -2.92844,
      0.213155, 2.92844, -0.213155, -2.92844, 1.06214, 2.07946, -1.06214,
      -2.07946, 1.06214, 2.07946, -1.06214, -2.07946, 0.370264, 2.77133,
      -0.370264, -2.77133, 0.370264, 2.77133, -0.370264, -2.77133, 1.35764,
      1.78395, -1.35764, -1.78395, 1.35764, 1.78395, -1.35764, -1.78395,
      1.20053, 1.94106, -1.20053, -1.94106, 1.20053, 1.94106, -1.20053,
      -1.94106, 0.492657, 2.64894, -0.492657, -2.64894, 0.492657, 2.64894,
      -0.492657, -2.64894, 0.346255, 2.79534, -0.346255, -2.79534, 0.346255,
      2.79534, -0.346255, -2.79534, 1.07814, 2.06345, -1.07814, -2.06345,
      1.07814, 2.06345, -1.07814, -2.06345, 0.591759, 2.54983, -0.591759,
      -2.54983, 0.591759, 2.54983, -0.591759, -2.54983, 1.22454, 1.91705,
      -1.22454, -1.91705, 1.22454, 1.91705, -1.22454, -1.91705, 0.979037,
      2.16256, -0.979037, -2.16256, 0.979037, 2.16256, -0.979037, -2.16256,
      0.603823, 2.53777, -0.603823, -2.53777, 0.603823, 2.53777, -0.603823,
      -2.53777, 0.154034, 2.98756, -0.154034, -2.98756, 0.154034, 2.98756,
      -0.154034, -2.98756, 0.966973, 2.17462, -0.966973, -2.17462, 0.966973,
      2.17462, -0.966973, -2.17462, 0.221407, 2.92019, -0.221407, -2.92019,
      0.221407, 2.92019, -0.221407, -2.92019, 1.41676, 1.72483, -1.41676,
      -1.72483, 1.41676, 1.72483, -1.41676, -1.72483, 1.34939, 1.7922, -1.34939,
      -1.7922, 1.34939, 1.7922, -1.34939, -1.7922, 0.595792, 2.5458, -0.595792,
      -2.5458, 0.595792, 2.5458, -0.595792, -2.5458, 0.301666, 2.83993,
      -0.301666, -2.83993, 0.301666, 2.83993, -0.301666, -2.83993, 0.975004,
      2.16659, -0.975004, -2.16659, 0.975004, 2.16659, -0.975004, -2.16659,
      0.430278, 2.71132, -0.430278, -2.71132, 0.430278, 2.71132, -0.430278,
      -2.71132, 1.26913, 1.87246, -1.26913, -1.87246, 1.26913, 1.87246,
      -1.26913, -1.87246, 1.14052, 2.00107, -1.14052, -2.00107, 1.14052,
      2.00107, -1.14052, -2.00107, 0.582743, 2.55885, -0.582743, -2.55885,
      0.582743, 2.55885, -0.582743, -2.55885, 0.438966, 2.70263, -0.438966,
      -2.70263, 0.438966, 2.70263, -0.438966, -2.70263, 0.988054, 2.15354,
      -0.988054, -2.15354, 0.988054, 2.15354, -0.988054, -2.15354, 0.618979,
      2.52261, -0.618979, -2.52261, 0.618979, 2.52261, -0.618979, -2.52261,
      1.13183, 2.00976, -1.13183, -2.00976, 1.13183, 2.00976, -1.13183,
      -2.00976, 0.951818, 2.18978, -0.951818, -2.18978, 0.951818, 2.18978,
      -0.951818, -2.18978, 0.695477, 2.44612, -0.695477, -2.44612, 0.695477,
      2.44612, -0.695477, -2.44612, 0.0836579, 3.05793, -0.0836579, -3.05793,
      0.0836579, 3.05793, -0.0836579, -3.05793, 0.875319, 2.26627, -0.875319,
      -2.26627, 0.875319, 2.26627, -0.875319, -2.26627, 0.100137, 3.04146,
      -0.100137, -3.04146, 0.100137, 3.04146, -0.100137, -3.04146, 1.48714,
      1.65445, -1.48714, -1.65445, 1.48714, 1.65445, -1.48714, -1.65445,
      1.47066, 1.67093, -1.47066, -1.67093, 1.47066, 1.67093, -1.47066,
      -1.67093, 0.692794, 2.4488, -0.692794, -2.4488, 0.692794, 2.4488,
      -0.692794, -2.4488, 0.247173, 2.89442, -0.247173, -2.89442, 0.247173,
      2.89442, -0.247173, -2.89442, 0.878003, 2.26359, -0.878003, -2.26359,
      0.878003, 2.26359, -0.878003, -2.26359, 0.295122, 2.84647, -0.295122,
      -2.84647, 0.295122, 2.84647, -0.295122, -2.84647, 1.32362, 1.81797,
      -1.32362, -1.81797, 1.32362, 1.81797, -1.32362, -1.81797, 1.27567,
      1.86592, -1.27567, -1.86592, 1.27567, 1.86592, -1.27567, -1.86592,
      0.687536, 2.45406, -0.687536, -2.45406, 0.687536, 2.45406, -0.687536,
      -2.45406, 0.400826, 2.74077, -0.400826, -2.74077, 0.400826, 2.74077,
      -0.400826, -2.74077, 0.88326, 2.25833, -0.88326, -2.25833, 0.88326,
      2.25833, -0.88326, -2.25833, 0.476391, 2.6652, -0.476391, -2.6652,
      0.476391, 2.6652, -0.476391, -2.6652, 1.16997, 1.97162, -1.16997,
      -1.97162, 1.16997, 1.97162, -1.16997, -1.97162, 1.09441, 2.04719,
      -1.09441, -2.04719, 1.09441, 2.04719, -1.09441, -2.04719, 0.679796,
      2.4618, -0.679796, -2.4618, 0.679796, 2.4618, -0.679796, -2.4618,
      0.541707, 2.59989, -0.541707, -2.59989, 0.541707, 2.59989, -0.541707,
      -2.59989, 0.891, 2.25059, -0.891, -2.25059, 0.891, 2.25059, -0.891,
      -2.25059, 0.639936, 2.50166, -0.639936, -2.50166, 0.639936, 2.50166,
      -0.639936, -2.50166, 1.02909, 2.1125, -1.02909, -2.1125, 1.02909, 2.1125,
      -1.02909, -2.1125, 0.93086, 2.21073, -0.93086, -2.21073, 0.93086, 2.21073,
      -0.93086, -2.21073;
  Eigen::VectorXd H_weight_ref = Eigen::VectorXd::Zero(1202);
  H_weight_ref << 0.00138882, 0.00138882, 0.00138882, 0.00138882, 0.00138882,
      0.00138882, 0.0115676, 0.0115676, 0.0115676, 0.0115676, 0.0115676,
      0.0115676, 0.0115676, 0.0115676, 0.0115676, 0.0115676, 0.0115676,
      0.0115676, 0.0114771, 0.0114771, 0.0114771, 0.0114771, 0.0114771,
      0.0114771, 0.0114771, 0.0114771, 0.00463752, 0.00463752, 0.00463752,
      0.00463752, 0.00463752, 0.00463752, 0.00463752, 0.00463752, 0.00463752,
      0.00463752, 0.00463752, 0.00463752, 0.00463752, 0.00463752, 0.00463752,
      0.00463752, 0.00463752, 0.00463752, 0.00463752, 0.00463752, 0.00463752,
      0.00463752, 0.00463752, 0.00463752, 0.00704218, 0.00704218, 0.00704218,
      0.00704218, 0.00704218, 0.00704218, 0.00704218, 0.00704218, 0.00704218,
      0.00704218, 0.00704218, 0.00704218, 0.00704218, 0.00704218, 0.00704218,
      0.00704218, 0.00704218, 0.00704218, 0.00704218, 0.00704218, 0.00704218,
      0.00704218, 0.00704218, 0.00704218, 0.00862719, 0.00862719, 0.00862719,
      0.00862719, 0.00862719, 0.00862719, 0.00862719, 0.00862719, 0.00862719,
      0.00862719, 0.00862719, 0.00862719, 0.00862719, 0.00862719, 0.00862719,
      0.00862719, 0.00862719, 0.00862719, 0.00862719, 0.00862719, 0.00862719,
      0.00862719, 0.00862719, 0.00862719, 0.00970166, 0.00970166, 0.00970166,
      0.00970166, 0.00970166, 0.00970166, 0.00970166, 0.00970166, 0.00970166,
      0.00970166, 0.00970166, 0.00970166, 0.00970166, 0.00970166, 0.00970166,
      0.00970166, 0.00970166, 0.00970166, 0.00970166, 0.00970166, 0.00970166,
      0.00970166, 0.00970166, 0.00970166, 0.010432, 0.010432, 0.010432,
      0.010432, 0.010432, 0.010432, 0.010432, 0.010432, 0.010432, 0.010432,
      0.010432, 0.010432, 0.010432, 0.010432, 0.010432, 0.010432, 0.010432,
      0.010432, 0.010432, 0.010432, 0.010432, 0.010432, 0.010432, 0.010432,
      0.010916, 0.010916, 0.010916, 0.010916, 0.010916, 0.010916, 0.010916,
      0.010916, 0.010916, 0.010916, 0.010916, 0.010916, 0.010916, 0.010916,
      0.010916, 0.010916, 0.010916, 0.010916, 0.010916, 0.010916, 0.010916,
      0.010916, 0.010916, 0.010916, 0.0112181, 0.0112181, 0.0112181, 0.0112181,
      0.0112181, 0.0112181, 0.0112181, 0.0112181, 0.0112181, 0.0112181,
      0.0112181, 0.0112181, 0.0112181, 0.0112181, 0.0112181, 0.0112181,
      0.0112181, 0.0112181, 0.0112181, 0.0112181, 0.0112181, 0.0112181,
      0.0112181, 0.0112181, 0.0113862, 0.0113862, 0.0113862, 0.0113862,
      0.0113862, 0.0113862, 0.0113862, 0.0113862, 0.0113862, 0.0113862,
      0.0113862, 0.0113862, 0.0113862, 0.0113862, 0.0113862, 0.0113862,
      0.0113862, 0.0113862, 0.0113862, 0.0113862, 0.0113862, 0.0113862,
      0.0113862, 0.0113862, 0.0114603, 0.0114603, 0.0114603, 0.0114603,
      0.0114603, 0.0114603, 0.0114603, 0.0114603, 0.0114603, 0.0114603,
      0.0114603, 0.0114603, 0.0114603, 0.0114603, 0.0114603, 0.0114603,
      0.0114603, 0.0114603, 0.0114603, 0.0114603, 0.0114603, 0.0114603,
      0.0114603, 0.0114603, 0.0114715, 0.0114715, 0.0114715, 0.0114715,
      0.0114715, 0.0114715, 0.0114715, 0.0114715, 0.0114715, 0.0114715,
      0.0114715, 0.0114715, 0.0114715, 0.0114715, 0.0114715, 0.0114715,
      0.0114715, 0.0114715, 0.0114715, 0.0114715, 0.0114715, 0.0114715,
      0.0114715, 0.0114715, 0.011474, 0.011474, 0.011474, 0.011474, 0.011474,
      0.011474, 0.011474, 0.011474, 0.011474, 0.011474, 0.011474, 0.011474,
      0.011474, 0.011474, 0.011474, 0.011474, 0.011474, 0.011474, 0.011474,
      0.011474, 0.011474, 0.011474, 0.011474, 0.011474, 0.0115018, 0.0115018,
      0.0115018, 0.0115018, 0.0115018, 0.0115018, 0.0115018, 0.0115018,
      0.0115018, 0.0115018, 0.0115018, 0.0115018, 0.0115018, 0.0115018,
      0.0115018, 0.0115018, 0.0115018, 0.0115018, 0.0115018, 0.0115018,
      0.0115018, 0.0115018, 0.0115018, 0.0115018, 0.0115453, 0.0115453,
      0.0115453, 0.0115453, 0.0115453, 0.0115453, 0.0115453, 0.0115453,
      0.0115453, 0.0115453, 0.0115453, 0.0115453, 0.0115453, 0.0115453,
      0.0115453, 0.0115453, 0.0115453, 0.0115453, 0.0115453, 0.0115453,
      0.0115453, 0.0115453, 0.0115453, 0.0115453, 0.00650558, 0.00650558,
      0.00650558, 0.00650558, 0.00650558, 0.00650558, 0.00650558, 0.00650558,
      0.00650558, 0.00650558, 0.00650558, 0.00650558, 0.00650558, 0.00650558,
      0.00650558, 0.00650558, 0.00650558, 0.00650558, 0.00650558, 0.00650558,
      0.00650558, 0.00650558, 0.00650558, 0.00650558, 0.00921259, 0.00921259,
      0.00921259, 0.00921259, 0.00921259, 0.00921259, 0.00921259, 0.00921259,
      0.00921259, 0.00921259, 0.00921259, 0.00921259, 0.00921259, 0.00921259,
      0.00921259, 0.00921259, 0.00921259, 0.00921259, 0.00921259, 0.00921259,
      0.00921259, 0.00921259, 0.00921259, 0.00921259, 0.0106352, 0.0106352,
      0.0106352, 0.0106352, 0.0106352, 0.0106352, 0.0106352, 0.0106352,
      0.0106352, 0.0106352, 0.0106352, 0.0106352, 0.0106352, 0.0106352,
      0.0106352, 0.0106352, 0.0106352, 0.0106352, 0.0106352, 0.0106352,
      0.0106352, 0.0106352, 0.0106352, 0.0106352, 0.0113488, 0.0113488,
      0.0113488, 0.0113488, 0.0113488, 0.0113488, 0.0113488, 0.0113488,
      0.0113488, 0.0113488, 0.0113488, 0.0113488, 0.0113488, 0.0113488,
      0.0113488, 0.0113488, 0.0113488, 0.0113488, 0.0113488, 0.0113488,
      0.0113488, 0.0113488, 0.0113488, 0.0113488, 0.00815027, 0.00815027,
      0.00815027, 0.00815027, 0.00815027, 0.00815027, 0.00815027, 0.00815027,
      0.00815027, 0.00815027, 0.00815027, 0.00815027, 0.00815027, 0.00815027,
      0.00815027, 0.00815027, 0.00815027, 0.00815027, 0.00815027, 0.00815027,
      0.00815027, 0.00815027, 0.00815027, 0.00815027, 0.00815027, 0.00815027,
      0.00815027, 0.00815027, 0.00815027, 0.00815027, 0.00815027, 0.00815027,
      0.00815027, 0.00815027, 0.00815027, 0.00815027, 0.00815027, 0.00815027,
      0.00815027, 0.00815027, 0.00815027, 0.00815027, 0.00815027, 0.00815027,
      0.00815027, 0.00815027, 0.00815027, 0.00815027, 0.00934314, 0.00934314,
      0.00934314, 0.00934314, 0.00934314, 0.00934314, 0.00934314, 0.00934314,
      0.00934314, 0.00934314, 0.00934314, 0.00934314, 0.00934314, 0.00934314,
      0.00934314, 0.00934314, 0.00934314, 0.00934314, 0.00934314, 0.00934314,
      0.00934314, 0.00934314, 0.00934314, 0.00934314, 0.00934314, 0.00934314,
      0.00934314, 0.00934314, 0.00934314, 0.00934314, 0.00934314, 0.00934314,
      0.00934314, 0.00934314, 0.00934314, 0.00934314, 0.00934314, 0.00934314,
      0.00934314, 0.00934314, 0.00934314, 0.00934314, 0.00934314, 0.00934314,
      0.00934314, 0.00934314, 0.00934314, 0.00934314, 0.0100512, 0.0100512,
      0.0100512, 0.0100512, 0.0100512, 0.0100512, 0.0100512, 0.0100512,
      0.0100512, 0.0100512, 0.0100512, 0.0100512, 0.0100512, 0.0100512,
      0.0100512, 0.0100512, 0.0100512, 0.0100512, 0.0100512, 0.0100512,
      0.0100512, 0.0100512, 0.0100512, 0.0100512, 0.0100512, 0.0100512,
      0.0100512, 0.0100512, 0.0100512, 0.0100512, 0.0100512, 0.0100512,
      0.0100512, 0.0100512, 0.0100512, 0.0100512, 0.0100512, 0.0100512,
      0.0100512, 0.0100512, 0.0100512, 0.0100512, 0.0100512, 0.0100512,
      0.0100512, 0.0100512, 0.0100512, 0.0100512, 0.0101809, 0.0101809,
      0.0101809, 0.0101809, 0.0101809, 0.0101809, 0.0101809, 0.0101809,
      0.0101809, 0.0101809, 0.0101809, 0.0101809, 0.0101809, 0.0101809,
      0.0101809, 0.0101809, 0.0101809, 0.0101809, 0.0101809, 0.0101809,
      0.0101809, 0.0101809, 0.0101809, 0.0101809, 0.0101809, 0.0101809,
      0.0101809, 0.0101809, 0.0101809, 0.0101809, 0.0101809, 0.0101809,
      0.0101809, 0.0101809, 0.0101809, 0.0101809, 0.0101809, 0.0101809,
      0.0101809, 0.0101809, 0.0101809, 0.0101809, 0.0101809, 0.0101809,
      0.0101809, 0.0101809, 0.0101809, 0.0101809, 0.0106605, 0.0106605,
      0.0106605, 0.0106605, 0.0106605, 0.0106605, 0.0106605, 0.0106605,
      0.0106605, 0.0106605, 0.0106605, 0.0106605, 0.0106605, 0.0106605,
      0.0106605, 0.0106605, 0.0106605, 0.0106605, 0.0106605, 0.0106605,
      0.0106605, 0.0106605, 0.0106605, 0.0106605, 0.0106605, 0.0106605,
      0.0106605, 0.0106605, 0.0106605, 0.0106605, 0.0106605, 0.0106605,
      0.0106605, 0.0106605, 0.0106605, 0.0106605, 0.0106605, 0.0106605,
      0.0106605, 0.0106605, 0.0106605, 0.0106605, 0.0106605, 0.0106605,
      0.0106605, 0.0106605, 0.0106605, 0.0106605, 0.0107522, 0.0107522,
      0.0107522, 0.0107522, 0.0107522, 0.0107522, 0.0107522, 0.0107522,
      0.0107522, 0.0107522, 0.0107522, 0.0107522, 0.0107522, 0.0107522,
      0.0107522, 0.0107522, 0.0107522, 0.0107522, 0.0107522, 0.0107522,
      0.0107522, 0.0107522, 0.0107522, 0.0107522, 0.0107522, 0.0107522,
      0.0107522, 0.0107522, 0.0107522, 0.0107522, 0.0107522, 0.0107522,
      0.0107522, 0.0107522, 0.0107522, 0.0107522, 0.0107522, 0.0107522,
      0.0107522, 0.0107522, 0.0107522, 0.0107522, 0.0107522, 0.0107522,
      0.0107522, 0.0107522, 0.0107522, 0.0107522, 0.0110624, 0.0110624,
      0.0110624, 0.0110624, 0.0110624, 0.0110624, 0.0110624, 0.0110624,
      0.0110624, 0.0110624, 0.0110624, 0.0110624, 0.0110624, 0.0110624,
      0.0110624, 0.0110624, 0.0110624, 0.0110624, 0.0110624, 0.0110624,
      0.0110624, 0.0110624, 0.0110624, 0.0110624, 0.0110624, 0.0110624,
      0.0110624, 0.0110624, 0.0110624, 0.0110624, 0.0110624, 0.0110624,
      0.0110624, 0.0110624, 0.0110624, 0.0110624, 0.0110624, 0.0110624,
      0.0110624, 0.0110624, 0.0110624, 0.0110624, 0.0110624, 0.0110624,
      0.0110624, 0.0110624, 0.0110624, 0.0110624, 0.0110723, 0.0110723,
      0.0110723, 0.0110723, 0.0110723, 0.0110723, 0.0110723, 0.0110723,
      0.0110723, 0.0110723, 0.0110723, 0.0110723, 0.0110723, 0.0110723,
      0.0110723, 0.0110723, 0.0110723, 0.0110723, 0.0110723, 0.0110723,
      0.0110723, 0.0110723, 0.0110723, 0.0110723, 0.0110723, 0.0110723,
      0.0110723, 0.0110723, 0.0110723, 0.0110723, 0.0110723, 0.0110723,
      0.0110723, 0.0110723, 0.0110723, 0.0110723, 0.0110723, 0.0110723,
      0.0110723, 0.0110723, 0.0110723, 0.0110723, 0.0110723, 0.0110723,
      0.0110723, 0.0110723, 0.0110723, 0.0110723, 0.0111216, 0.0111216,
      0.0111216, 0.0111216, 0.0111216, 0.0111216, 0.0111216, 0.0111216,
      0.0111216, 0.0111216, 0.0111216, 0.0111216, 0.0111216, 0.0111216,
      0.0111216, 0.0111216, 0.0111216, 0.0111216, 0.0111216, 0.0111216,
      0.0111216, 0.0111216, 0.0111216, 0.0111216, 0.0111216, 0.0111216,
      0.0111216, 0.0111216, 0.0111216, 0.0111216, 0.0111216, 0.0111216,
      0.0111216, 0.0111216, 0.0111216, 0.0111216, 0.0111216, 0.0111216,
      0.0111216, 0.0111216, 0.0111216, 0.0111216, 0.0111216, 0.0111216,
      0.0111216, 0.0111216, 0.0111216, 0.0111216, 0.0113366, 0.0113366,
      0.0113366, 0.0113366, 0.0113366, 0.0113366, 0.0113366, 0.0113366,
      0.0113366, 0.0113366, 0.0113366, 0.0113366, 0.0113366, 0.0113366,
      0.0113366, 0.0113366, 0.0113366, 0.0113366, 0.0113366, 0.0113366,
      0.0113366, 0.0113366, 0.0113366, 0.0113366, 0.0113366, 0.0113366,
      0.0113366, 0.0113366, 0.0113366, 0.0113366, 0.0113366, 0.0113366,
      0.0113366, 0.0113366, 0.0113366, 0.0113366, 0.0113366, 0.0113366,
      0.0113366, 0.0113366, 0.0113366, 0.0113366, 0.0113366, 0.0113366,
      0.0113366, 0.0113366, 0.0113366, 0.0113366, 0.0113224, 0.0113224,
      0.0113224, 0.0113224, 0.0113224, 0.0113224, 0.0113224, 0.0113224,
      0.0113224, 0.0113224, 0.0113224, 0.0113224, 0.0113224, 0.0113224,
      0.0113224, 0.0113224, 0.0113224, 0.0113224, 0.0113224, 0.0113224,
      0.0113224, 0.0113224, 0.0113224, 0.0113224, 0.0113224, 0.0113224,
      0.0113224, 0.0113224, 0.0113224, 0.0113224, 0.0113224, 0.0113224,
      0.0113224, 0.0113224, 0.0113224, 0.0113224, 0.0113224, 0.0113224,
      0.0113224, 0.0113224, 0.0113224, 0.0113224, 0.0113224, 0.0113224,
      0.0113224, 0.0113224, 0.0113224, 0.0113224, 0.0113383, 0.0113383,
      0.0113383, 0.0113383, 0.0113383, 0.0113383, 0.0113383, 0.0113383,
      0.0113383, 0.0113383, 0.0113383, 0.0113383, 0.0113383, 0.0113383,
      0.0113383, 0.0113383, 0.0113383, 0.0113383, 0.0113383, 0.0113383,
      0.0113383, 0.0113383, 0.0113383, 0.0113383, 0.0113383, 0.0113383,
      0.0113383, 0.0113383, 0.0113383, 0.0113383, 0.0113383, 0.0113383,
      0.0113383, 0.0113383, 0.0113383, 0.0113383, 0.0113383, 0.0113383,
      0.0113383, 0.0113383, 0.0113383, 0.0113383, 0.0113383, 0.0113383,
      0.0113383, 0.0113383, 0.0113383, 0.0113383, 0.0115083, 0.0115083,
      0.0115083, 0.0115083, 0.0115083, 0.0115083, 0.0115083, 0.0115083,
      0.0115083, 0.0115083, 0.0115083, 0.0115083, 0.0115083, 0.0115083,
      0.0115083, 0.0115083, 0.0115083, 0.0115083, 0.0115083, 0.0115083,
      0.0115083, 0.0115083, 0.0115083, 0.0115083, 0.0115083, 0.0115083,
      0.0115083, 0.0115083, 0.0115083, 0.0115083, 0.0115083, 0.0115083,
      0.0115083, 0.0115083, 0.0115083, 0.0115083, 0.0115083, 0.0115083,
      0.0115083, 0.0115083, 0.0115083, 0.0115083, 0.0115083, 0.0115083,
      0.0115083, 0.0115083, 0.0115083, 0.0115083, 0.0114751, 0.0114751,
      0.0114751, 0.0114751, 0.0114751, 0.0114751, 0.0114751, 0.0114751,
      0.0114751, 0.0114751, 0.0114751, 0.0114751, 0.0114751, 0.0114751,
      0.0114751, 0.0114751, 0.0114751, 0.0114751, 0.0114751, 0.0114751,
      0.0114751, 0.0114751, 0.0114751, 0.0114751, 0.0114751, 0.0114751,
      0.0114751, 0.0114751, 0.0114751, 0.0114751, 0.0114751, 0.0114751,
      0.0114751, 0.0114751, 0.0114751, 0.0114751, 0.0114751, 0.0114751,
      0.0114751, 0.0114751, 0.0114751, 0.0114751, 0.0114751, 0.0114751,
      0.0114751, 0.0114751, 0.0114751, 0.0114751, 0.0114452, 0.0114452,
      0.0114452, 0.0114452, 0.0114452, 0.0114452, 0.0114452, 0.0114452,
      0.0114452, 0.0114452, 0.0114452, 0.0114452, 0.0114452, 0.0114452,
      0.0114452, 0.0114452, 0.0114452, 0.0114452, 0.0114452, 0.0114452,
      0.0114452, 0.0114452, 0.0114452, 0.0114452, 0.0114452, 0.0114452,
      0.0114452, 0.0114452, 0.0114452, 0.0114452, 0.0114452, 0.0114452,
      0.0114452, 0.0114452, 0.0114452, 0.0114452, 0.0114452, 0.0114452,
      0.0114452, 0.0114452, 0.0114452, 0.0114452, 0.0114452, 0.0114452,
      0.0114452, 0.0114452, 0.0114452, 0.0114452, 0.0114426, 0.0114426,
      0.0114426, 0.0114426, 0.0114426, 0.0114426, 0.0114426, 0.0114426,
      0.0114426, 0.0114426, 0.0114426, 0.0114426, 0.0114426, 0.0114426,
      0.0114426, 0.0114426, 0.0114426, 0.0114426, 0.0114426, 0.0114426,
      0.0114426, 0.0114426, 0.0114426, 0.0114426, 0.0114426, 0.0114426,
      0.0114426, 0.0114426, 0.0114426, 0.0114426, 0.0114426, 0.0114426,
      0.0114426, 0.0114426, 0.0114426, 0.0114426, 0.0114426, 0.0114426,
      0.0114426, 0.0114426, 0.0114426, 0.0114426, 0.0114426, 0.0114426,
      0.0114426, 0.0114426, 0.0114426, 0.0114426;
  BOOST_CHECK_EQUAL(Gegrid.phi.size(), Ge_phi_ref.size());
  BOOST_CHECK_EQUAL(Gegrid.theta.size(), Ge_theta_ref.size());
  BOOST_CHECK_EQUAL(Gegrid.weight.size(), Ge_weight_ref.size());
  BOOST_CHECK_EQUAL(Hgrid.phi.size(), H_phi_ref.size());
  BOOST_CHECK_EQUAL(Hgrid.theta.size(), H_theta_ref.size());
  BOOST_CHECK_EQUAL(Hgrid.weight.size(), H_weight_ref.size());

  bool Gephi = Ge_phi_ref.isApprox(Gegrid.phi, 0.001);
  bool Getheta = Ge_theta_ref.isApprox(Gegrid.theta, 0.001);
  if (!Gephi || !Getheta) {
    std::cout << "phi_ref : Phi_comp | theta_ref : theta_comp" << std::endl;
    for (Index i = 0; i < Ge_phi_ref.size(); i++) {
      std::cout << Gegrid.phi[i] << ":" << Ge_phi_ref[i] << " | "
                << Gegrid.theta[i] << ":" << Ge_theta_ref[i] << std::endl;
    }
  }

  bool Geweight = Ge_weight_ref.isApprox(Gegrid.weight, 0.0001);
  BOOST_CHECK_EQUAL(Gephi, true);
  BOOST_CHECK_EQUAL(Getheta, true);
  BOOST_CHECK_EQUAL(Geweight, true);

  bool Hphi = H_phi_ref.isApprox(Hgrid.phi, 0.001);
  bool Htheta = H_theta_ref.isApprox(Hgrid.theta, 0.001);
  bool Hweight = H_weight_ref.isApprox(Hgrid.weight, 0.0001);
  BOOST_CHECK_EQUAL(Hphi, true);
  BOOST_CHECK_EQUAL(Htheta, true);
  BOOST_CHECK_EQUAL(Hweight, true);
}

BOOST_AUTO_TEST_SUITE_END()
