/*****************************************************************************
 *
 *  Copyright (C) 2003 Cdric Brgardis <cedric.bregardis@free.fr>
 *
 *  This file is part of BRIQUOLO
 *
 *  BRIQUOLO is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  BRIQUOLO is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with BRIQUOLO; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *****************************************************************************/
#ifndef MOGL_FENETRE
#define MOGL_FENETRE


#ifdef _WIN32
#include <windows.h>
#include <windowsx.h>
#endif

#include <GL/gl.h>
#include <GL/glu.h>
#include <SDL/SDL.h>
#include <iostream>
#include <string>
#include <map>
#include <set>
#include "MOGL_Interface.h"
#include "MOGL_Signal.h"

using namespace std;

/**
   Permet de grer une fentre destine  recevoir l'affichage.
   La rsolution par dfaut est 640x480 avec des couleurs codes sur 16 bits, en mode fentr.
*/
class MOGL_Fenetre
{
    // **** Types locaux  la classe ****
  public:
    struct MOGL_Resolution;
  private:
    struct CompareResolution
    {
        bool operator() (const MOGL_Resolution & x, const MOGL_Resolution & y)
        {
          return x.w < y.w;
        }
    };

    typedef set<MOGL_Interface *> MOGL_Set_Interface;
    typedef MOGL_Set_Interface::iterator MOGL_ItSet_Interface;

    typedef map<unsigned int, MOGL_Set_Interface> MOGL_Map_Interface;
    typedef MOGL_Map_Interface::iterator MOGL_ItMap_Interface;
  public:
    struct MOGL_Resolution
    {
        MOGL_Resolution(unsigned int pW, unsigned int pH): w(pW), h(pH)
        {
        }
        unsigned int w, h;
    };
    typedef set<MOGL_Resolution, CompareResolution> MOGL_Set_Resolution;
    typedef MOGL_Set_Resolution::iterator MOGL_ItSet_Resolution;


    // **********************************

  protected :
    char * _Titre;
    int _ResolutionX, _ResolutionY;
    int _MemResolutionX, _MemResolutionY;
    unsigned int _NbBitParPixel;
    bool _FinBoucle;
    SDL_Surface * _SurfaceSDL;
    bool _Fullscreen;
    bool _GrabCurseur;
    int _RepetitionDelai;
    int _RepetitionIntervalle;

    MOGL_Map_Interface _EnsembleInterface;

    unsigned int _NumeroImage;
    bool _ExisteFenetre;
    bool _CacheSouris;

    MOGL_Set_Resolution _SetResolution;

    // Donnes pour les joysticks
    unsigned int _NombreJoysticks;
    SDL_Joystick ** _Joysticks;
    bool _bManualResize;

    // Les Signaux
  public:
    MOGL_Signal0 Idle;
    MOGL_Signal1<SDL_keysym *> KeyDown;
    MOGL_Signal1<SDL_keysym *> KeyUp;
    MOGL_Signal1<SDL_MouseMotionEvent *> MouseMove;
    MOGL_Signal1<SDL_MouseButtonEvent *> MouseButtonDown;
    MOGL_Signal1<SDL_MouseButtonEvent *> MouseButtonUp;
    MOGL_Signal3<int,int,int> ChangementMode;
    MOGL_Signal0 Destruction;
    MOGL_Signal3<unsigned int, unsigned int, float> MouvementAxeJoystick; // Joystick index, Axe index, valeur
    MOGL_Signal2<unsigned int, unsigned int> BoutonJoystickDown;          // Joystick index, Bouton index
    MOGL_Signal2<unsigned int, unsigned int> BoutonJoystickUp;            // Joystick index, Bouton index

  protected:
    void _MouvementAxeJoystick(SDL_JoyAxisEvent * p_JoyAxisEvent);
    void _BoutonJoystickDown(SDL_JoyButtonEvent * p_JoyButtonEvent);
    void _BoutonJoystickUp(SDL_JoyButtonEvent * p_JoyButtonEvent);
    bool _InitGL();
    void _GestionEvenement();
    void _Reshape (int w,int h);
    void _ManualResize(int p_ResolutionX, int p_ResolutionY);

    SDL_Surface * _CreerFenetre(unsigned int & p_Bpp, unsigned int p_Flags);

  public :
    MOGL_Fenetre(const char * p_Titre=NULL);
    ~MOGL_Fenetre();

    /**
       Permet de dfinir la rsolution de la fentre
       @param p_ResolutionX : rsolution en X de la fentre
       @param p_ResolutionY : rsolution en Y de la fentre
       @param p_Fullscreen : indique si la fentre doit tre plein cran ou non.
       @return <i>true</i> si il n'y a pas eu de problme, <i>false</i> sinon. Un problme peut intervenir seuleument s'il
               s'agit d'un changement de mode alors que la fen^tre existe
    */
    bool SetMode(int p_ResolutionX, int p_ResolutionY, bool p_Fullscreen);

    MOGL_Set_Resolution GetAvailableResolution() const;

    /**
       Permet de dfinir la rsolution de la fentre. Le nombre de bit par pixel reste inchang.
       @param p_ResolutionX : rsolution en X de la fentre
       @param p_ResolutionY : rsolution en Y de la fentre
    */
    bool SetMode(int p_ResolutionX, int p_ResolutionY);

    /**
       Permet de rcuprer la rsolution X de la fentre.
       @return La rsolution X.
     */
    int GetResolutionX() const;

    /**
       Permet de rcuprer la rsolution Y de la fentre.
       @return La rsolution Y.
     */
    int GetResolutionY() const;

    /**
       Permet de rcuprer le nombre de bits par pixel de la fentre.
       @return Le nombre de bits par pixels.
     */
    int GetBPP() const;

    /**
       Permet de dfinir le titre de la fentre. Le titre n'est affich que lorsque la fenetre n'est pas en plein cran
       (@sa SetFullscreen()].
       @param p_Titre : titre de la fentre
    */
    void SetTitre(char * p_Titre);

    /**
       Permet de mettre la fentre en plein cran.
       p_Fullscreen : true si plein cran, false sinon.
    */
    void SetFullscreen(bool p_Fullscreen);

    /**
       Indique si la fentre est en plein cran.
       @return <i>true</i> si a fentre est en plein cran, <i>false</i> sinon.
     */
    bool GetFullscreen() const;

    /**
       Permet d'initialiser la fentre d'affichage ainsi que tout l'environnement MOGL.
       Cette initialisation doit tre effectue avant la plupart des oprations pouvant tre effectues par MOGL
       (chargement de texture...)
       @return <i>true</i> s'il n'y a pas eu de problme, et <i>false</i> sinon.
    */
    bool Initialiser();

    /**
       Permet de Lancer la boucle d'vnement de MOGL. La mthode ne redonne la main que lorsque la boucle d'vnement est termine,
       ce qui signifie que l'excution de MOGL est acheve.
       @return <i>true</i> s'il n'y a pas eu de problme, <i>false</i> sinon.
    */
    bool LancerBoucle();

    /**
       Permet d'arreter la boucle init par LancerBoucle()
     */
    void Arreter();

    /**
       Permet d'ajouter une interface  la fentre. Les interface reprsente des <i>contenus</i> lis  la fentre
       comme par exemple MOGL_Univers qui reprsente un univers 3D ou encore MOGL_Facade qui reprsente un affichage 2D.
    */
    void AjouterInterface(MOGL_Interface * p_Interface);

    /**
       Permet de retirer une interface.
       @sa AjouterInterface().
    */
    bool RetirerInterface(MOGL_Interface * p_Interface);


    /**
       Permet d'infiquer  la fentre si elle peut garder le curseur.
       @param p_Val : <i>true</i> pour l'attraper, <i>false</i> sinon.
    */
    void SetGrabCurseur(bool p_Val);

    /**
       Permet de cacher le pointeur de souris. Par dfaut, le pointeur est cach.
       param p_Val : <i>true</i> pour le cacher, <i>false</i> sinon.
     */
    void SetCacherCurseur(bool p_Val);

    /**
       Permet de dfinir la rptition des touches.
       @todo  ammliorer.
    */
    void DefinirRepetitionTouche(int p_Delai, int p_Intervalle);

    /**
     * Permet de prendre un screenshot au format BMP.
     *
     * @param p_NomFichier : nom du fichier de sauvegarde.
     *
     * @return <i>true</i> s'il n'y a pas eu d'erreur, <i>false</i> sinon.
     */
    bool ScreenshotBMP(char * p_NomFichier);

    /**
     * Permet de prendre un screenshot au format BMP.
     * Le nom du fichier est ici dtermin de manire automatique,  partir du prefix fourni.
     * Le nom correspond au prfix indiqu qui est suivi d'un numro (le premier disponible,
     * de manire  ce que le fichier n'existe pas sur le disque), et de l'extension <i>.bmp</i>
     *
     * @param p_Prefix : prfix du nom du fichier
     *
     * @return <i>true</i> s'il n'y a pas eu d'erreur, <i>false</i> sinon.
     */
    bool ScreenshotPrefixBMP(char * p_Prefix);

    unsigned int GetNombreJoystick() const;
    bool OuvrirJoystick(unsigned int p_IndiceJoystick);
    bool FermerJoystick(unsigned int p_IndiceJoystick);
};

#endif

/** @mainpage Moteur Open GL - MOGL
        @section intro Introduction
         @subsection quoi Qu'est-ce que c'est ?
         MOGL est un <i>moteur 3D temps rel</i>.
         Cela signifie qu'il permet de grer l'affichage d'un univers en trois dimension de manire suffisamment rapide pour
         tre interactif.
         Ainsi MOGL propose un ensemble de classes C++ disponibles sous la forme d'une bibliothque permettant la ralisation aise
         d'une application, en proposant tous les services lies  la mise en place d'un univers 3D : Cration d'une fentre d'affichage,
         affichage d'objets, animations, interraction...

         @subsection specif Spcificits de MOGL
         Ce genre d'outil existe sous de trs nombreuses formes. Par exemple <i>3D Master Suite</i> dvelopp  l'origine par SGI est une
         bibliotque reposant sur OpenGL, qui est utilis dans le monde professionel qui propose la ralisation d'un affichage 3D
         (pour la ralisation de logiciel de CAO par exemple). De mme, Id Software s'est dmarqu par la cration de moteurs 3D
         utiliss dans le jeux video (<i>Quake 3</i> par exemple).
         L'initiative de base est semblable, cependant le but tant diffrents ces deux outils n'ont pas les mme contraintes,
         et ne sont pas interchangables.
         @par
         De la mme manire MOGL a t conu dans un but prcis : c'est un moteur 3D de type jeux video (ainsi il n'est peut pas conu
         pour s'intgrer dans une IHM complexe), simple d'utilisation et multiplateforme.

         @subsection fonctionnalites Fonctionnalits
          @li Affichage de triangles et de strips
          @li Gestion des lments de l'affichage par l'intermdiaire d'un arbre, de manipulation aise
          @li Moteur de particules permettant la cration de nouvelles particules facilement
          @li Affichage de Lens Flare
          @li Animation aise des objets prsents dans l'arbre par l'intermdiare de classes spcialiss
          @li Elements 2D plac au dessus de l'affichage 3D pour la ralisation aise d'IHM.

        @section install Installation
*/
