/*
 * Copyright (c) 2008, intarsys consulting GmbH
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 *
 * - Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 *
 * - Neither the name of intarsys nor the names of its contributors may be used
 *   to endorse or promote products derived from this software without specific
 *   prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
package de.intarsys.tools.installresource;

import java.io.File;
import java.io.IOException;
import java.net.URL;

import de.intarsys.tools.file.TempTools;

/**
 * An abstraction to access a file to be deployed along with the application via
 * the class loader.
 * <p>
 * The file is searched in the class loader with getResourceAsStream(x) where x
 * is one of
 * <ul>
 * <li> [path]/[os.name]-[os.arch]/[filename] </li>
 * <li> [path]/[os.short name]-[os.arch]/[filename] </li>
 * <li> [path]/default/[filename] </li>
 * </ul>
 * 
 * where
 * 
 * <ul>
 * <li> [path] is a path prefix defined upon object creation </li>
 * <li> [os.name] is the System property os.name in lowercase</li>
 * <li> [os.short name] is derived from the System property os.name by using the
 * beginning up to the first whitespace in lowercase</li>
 * <li> [os.arch] is the System property os.arch in lowercase</li>
 * <li> [filename] is the name defined upon object creation</li>
 * </ul>
 * 
 * example, with NativeCode("foo", "bar.dll") on a Windows Vista machine you
 * will search for:
 * 
 * <ul>
 * <li>foo/windows vista-x86/bar.dll</li>
 * <li>foo/windows-x86/bar.dll</li>
 * <li>foo/default/bar.dll</li>
 * </ul>
 */
public class InstallFile extends Install {

	public InstallFile(String path, String name, boolean platformDependent) {
		super(path, name, platformDependent);
	}

	@Override
	protected File loadURL(URL url) throws IOException {
		File file = TempTools.createTempFile("file", getName());
		copy(url, file);
		return file;
	}

}
